#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/IOCTL.H>
#include <SPAD/VMDEF.H>
#include <ARCH/BSF.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <SPAD/CONSOLE.H>

#include "SIOREG.H"

#define SIO_BUFFER_SIZE		4096		/* must be power of 2 */
#define SIO_RECV_THRESHOLD	(SIO_BUFFER_SIZE * 7 / 8)
#define SIO_SEND_THRESHOLD	(SIO_BUFFER_SIZE * 7 / 8)
#define SIO_DEFAULT_TRIGGER	8
#define SIO_RECV_LIVELOCK	256
#define SIO_TIMEOUT_LIVELOCK	TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 10)

#define CRASHDUMP_TIMEOUT	1000000
#define CRASHDUMP_TIMEOUT_FC	15000000

#define CHIP_UNKNOWN	0
#define CHIP_8250	1
#define CHIP_16450	2
#define CHIP_16550	3
#define CHIP_16550A	4
#define CHIP_16650	5

static __const__ char * __const__ chipnames[] = {
	"UNKNOWN",
	"8250",
	"16450",
	"16550",
	"16550A",
	"16650",
};

#define UART_CLOCK	115200

struct sio {
	io_t io;
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	__u8 xmit_fifo_size;
	__u8 mcr;
	__u8 msr;
	char drop_flag;

	char *rcv_buffer;
	char *xmit_buffer;

	unsigned rcv_buffer_start;
	unsigned rcv_buffer_end;
	WQ rcv_wait;
	unsigned xmit_buffer_start;
	unsigned xmit_buffer_end;
	WQ xmit_wait;

	unsigned long current_flags;
	unsigned long current_flags2;

	char errorlevel;
	__u8 chiptype;
	char crashdump_reboot;

	void *lnte;
	void *dlrq;
	IO_RANGE range;
	char dev_name[__MAX_STR_LEN];
};

void *sio_lookup(HANDLE *h, char *str, int open_flags);
extern IO_STUB sio_read;
extern IO_STUB sio_write;
extern IO_STUB sio_ioctl;

static __const__ HANDLE_OPERATIONS sio_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	sio_lookup,		/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	sio_read,		/* read */
	sio_write,		/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	sio_ioctl,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static __finline__ __u8 sio_in(struct sio *sio, unsigned port)
{
	return io_inb(sio->io + port);
}

static __finline__ void sio_out(struct sio *sio, unsigned port, __u8 val)
{
	io_outb(sio->io + port, val);
}

/* ^BAUD=1234567890/^WORD=5678/^STOP=12/^PARITY=NONE,ODD,EVEN */

		/* don't change --- corresponds with LCR register bits */
#define FLAGS_DIV		0x0000ffffUL
#define FLAGS_WORD		0x00030000UL
#define FLAGS_STOPBIT		0x00040000UL
#define FLAGS_PARITY		0x00080000UL
#define FLAGS_PARITY_EVEN	0x00100000UL
#define FLAGS_TRIGGER		0xff000000UL
#define FLAGS_PORTCONTROL	0xff1fffffUL

#define FLAGS2_RTS		0x00000003UL
#define  FLAGS2_RTS_LO		 0x00000000UL
#define  FLAGS2_RTS_HI		 0x00000001UL
#define  FLAGS2_RTS_REQ_TO_SEND	 0x00000002UL
#define  FLAGS2_RTS_CAN_RECV	 0x00000003UL
#define FLAGS2_DTR		0x0000000cUL
#define  FLAGS2_DTR_LO		 0x00000000UL
#define  FLAGS2_DTR_HI		 0x00000004UL
#define  FLAGS2_DTR_CAN_RECV	 0x00000008UL
#define FLAGS2_CTS		0x00000030UL
#define  FLAGS2_CTS_IGN		 0x00000000UL
#define  FLAGS2_CTS_CAN_SEND	 0x00000010UL
#define FLAGS2_DSR		0x000000c0UL
#define  FLAGS2_DSR_IGN		 0x00000000UL
#define  FLAGS2_DSR_CAN_SEND	 0x00000040UL
#define  FLAGS2_DSR_ERR		 0x00000080UL
#define FLAGS2_CD		0x00000300UL
#define  FLAGS2_CD_IGN		 0x00000000UL
#define  FLAGS2_CD_ERR		 0x00000100UL
#define FLAGS2_NONBLOCK		0x80000000UL

#define INIT_FLAGS ((((UART_CLOCK + (9600 >> 1)) / 9600) << __BSF_CONST(FLAGS_DIV)) | (3 << __BSF_CONST(FLAGS_WORD)) | (SIO_DEFAULT_TRIGGER << __BSF_CONST(FLAGS_TRIGGER)))
#define INIT_FLAGS2 (FLAGS2_RTS_REQ_TO_SEND | FLAGS2_DTR_HI | FLAGS2_CTS_IGN | FLAGS2_DSR_IGN | FLAGS2_CD_IGN)

void *sio_lookup(HANDLE *h, char *str, int open_flags)
{
	struct sio *sio = h->fnode;
	char *e = strchr(str, '=');
	if (__unlikely(!e)) {

		if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
			h->flags2 |= FLAGS2_NONBLOCK;

		} else {
			return __ERR_PTR(-EBADMOD);

		}
		return NULL;
	}
	e++;

	if (!__strcasexcmp("^BAUD=", str, e)) {
		long baud, real_baud, div;
		if (__unlikely(__get_number(e, e + strlen(e), 0, &baud)) || __unlikely(baud <= 0)) return __ERR_PTR(-EBADMOD);
		div = (UART_CLOCK + (baud >> 1)) / baud;
		if (__unlikely(!div) || __unlikely(div >= 0x10000)) return __ERR_PTR(-EINVAL);
		real_baud = (UART_CLOCK + (div >> 1)) / div;
		if (__unlikely(baud != real_baud)) return __ERR_PTR(-EINVAL);
		if (__unlikely(sio->chiptype != CHIP_16550A) && __unlikely(div < 12)) return __ERR_PTR(-ENXIO);
		h->flags = (h->flags & ~FLAGS_DIV) | (div << __BSF_CONST(FLAGS_DIV));

	} else if (!__strcasexcmp("^WORD=", str, e)) {
		if (__unlikely(e[0] < '5') || __unlikely(e[0] > '8') || __unlikely(e[1])) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~FLAGS_WORD) | ((e[0] - '5') << __BSF_CONST(FLAGS_WORD));

	} else if (!__strcasexcmp("^STOP=", str, e)) {
		if (__unlikely(e[0] < '1') || __unlikely(e[0] > '2') || __unlikely(e[1])) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~FLAGS_STOPBIT) | ((e[0] - '1') << __BSF_CONST(FLAGS_STOPBIT));

	} else if (__likely(!__strcasexcmp("^PARITY=", str, e))) {
		if (!_strcasecmp(e, "NONE")) h->flags &= ~(FLAGS_PARITY | FLAGS_PARITY_EVEN);
		else if (!_strcasecmp(e, "ODD")) h->flags = (h->flags | FLAGS_PARITY) & ~FLAGS_PARITY_EVEN;
		else if (__likely(!_strcasecmp(e, "EVEN"))) h->flags |= FLAGS_PARITY | FLAGS_PARITY_EVEN;
		else return __ERR_PTR(-EBADMOD);

	} else if (!__strcasexcmp("^TRIGGER=", str, e)) {
		long trigger;
		if (__unlikely(__get_number(e, e + strlen(e), 0, &trigger))) return __ERR_PTR(-EBADMOD);
		if (__unlikely((unsigned long)trigger > FLAGS_TRIGGER >> __BSF_CONST(FLAGS_TRIGGER))) return __ERR_PTR(-EINVAL);
		h->flags = (h->flags & ~FLAGS_TRIGGER) | (trigger << __BSF_CONST(FLAGS_TRIGGER));

	} else if (!__strcasexcmp("^RTS=", str, e)) {
		unsigned long rts;
		if (!_strcasecmp(e, "LO")) rts = FLAGS2_RTS_LO;
		else if (!_strcasecmp(e, "HI")) rts = FLAGS2_RTS_HI;
		else if (__likely(!_strcasecmp(e, "RTS"))) rts = FLAGS2_RTS_REQ_TO_SEND;
		else if (__likely(!_strcasecmp(e, "RECV_FLOW"))) rts = FLAGS2_RTS_CAN_RECV;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~FLAGS2_RTS) | rts;

	} else if (__likely(!__strcasexcmp("^DTR=", str, e))) {
		unsigned long dtr;
		if (!_strcasecmp(e, "LO")) dtr = FLAGS2_DTR_LO;
		else if (!_strcasecmp(e, "HI")) dtr = FLAGS2_DTR_HI;
		else if (__likely(!_strcasecmp(e, "RECV_FLOW"))) dtr = FLAGS2_DTR_CAN_RECV;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~FLAGS2_DTR) | dtr;

	} else if (__likely(!__strcasexcmp("^CTS=", str, e))) {
		unsigned long cts;
		if (!_strcasecmp(e, "IGNORE")) cts = FLAGS2_CTS_IGN;
		else if (__likely(!_strcasecmp(e, "SEND_FLOW"))) cts = FLAGS2_CTS_CAN_SEND;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~FLAGS2_CTS) | cts;

	} else if (__likely(!__strcasexcmp("^DSR=", str, e))) {
		unsigned long dsr;
		if (!_strcasecmp(e, "IGNORE")) dsr = FLAGS2_DSR_IGN;
		else if (__likely(!_strcasecmp(e, "SEND_FLOW"))) dsr = FLAGS2_DSR_CAN_SEND;
		else if (__likely(!_strcasecmp(e, "ERROR"))) dsr = FLAGS2_DSR_ERR;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~FLAGS2_DSR) | dsr;

	} else if (__likely(!__strcasexcmp("^CD=", str, e))) {
		unsigned long cd;
		if (!_strcasecmp(e, "IGNORE")) cd = FLAGS2_CD_IGN;
		else if (__likely(!_strcasecmp(e, "ERROR"))) cd = FLAGS2_CD_ERR;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~FLAGS2_CD) | cd;

	} else if (__likely(!__strcasexcmp("^FLOW=", str, e))) {
		unsigned long flow;
		if (!_strcasecmp(e, "NONE")) flow = FLAGS2_RTS_HI | FLAGS2_DTR_HI | FLAGS2_CTS_IGN | FLAGS2_DSR_IGN | FLAGS2_CD_IGN;
		else if (!_strcasecmp(e, "MODEM")) flow = FLAGS2_RTS_REQ_TO_SEND | FLAGS2_DTR_HI | FLAGS2_CTS_CAN_SEND | FLAGS2_CD_IGN;
		else if (!_strcasecmp(e, "DTRDSR")) flow = FLAGS2_RTS_HI | FLAGS2_DTR_CAN_RECV | FLAGS2_CTS_IGN | FLAGS2_DSR_CAN_SEND | FLAGS2_CD_IGN;
		else if (!_strcasecmp(e, "RTSCTS")) flow = FLAGS2_RTS_CAN_RECV | FLAGS2_DTR_HI | FLAGS2_CTS_CAN_SEND | FLAGS2_DSR_IGN | FLAGS2_CD_IGN;
		else return __ERR_PTR(-EBADMOD);
		h->flags2 = (h->flags2 & ~(FLAGS2_RTS | FLAGS2_DTR | FLAGS2_CTS | FLAGS2_DSR | FLAGS2_CD)) | flow;

	} else return __ERR_PTR(-EBADMOD);
	return NULL;
}

static __u8 read_lsr_error(struct sio *sio, __u8 lsr)
{
	if (lsr & LSR_OE && sio->errorlevel >= 2)
		KERNEL$SYSLOG(__SYSLOG_NET_WARNING, sio->dev_name, "HARDWARE BUFFER OVERFLOW");
	if (lsr & LSR_PE && sio->errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sio->dev_name, "PARITY ERROR");
	if (lsr & LSR_FE && sio->errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, sio->dev_name, "FRAMING ERROR");
	return lsr;
}

static __finline__ __u8 read_lsr(struct sio *sio)
{
	__u8 lsr = sio_in(sio, SIO_LSR);
	if (__unlikely(lsr & (LSR_OE | LSR_PE | LSR_FE))) return read_lsr_error(sio, lsr);
	return lsr;
}

static void update_mcr(struct sio *sio)
{
	unsigned mcr = MCR_OUT2;
	unsigned long rts, dtr;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SIO)))
		KERNEL$SUICIDE("update_mcr AT SPL %08X", KERNEL$SPL);
#endif
	rts = sio->current_flags2 & FLAGS2_RTS;
	if (rts == FLAGS2_RTS_HI) {
		mcr |= MCR_RTS;
	} else if (rts == FLAGS2_RTS_REQ_TO_SEND) {
		if (sio->xmit_buffer_start != sio->xmit_buffer_end) mcr |= MCR_RTS;
	} else if (rts == FLAGS2_RTS_CAN_RECV) {
		if (__likely(((sio->rcv_buffer_end - sio->rcv_buffer_start) & (SIO_BUFFER_SIZE - 1)) < SIO_RECV_THRESHOLD)) mcr |= MCR_RTS;
	}

	dtr = sio->current_flags2 & FLAGS2_DTR;
	if (dtr == FLAGS2_DTR_HI) {
		mcr |= MCR_DTR;
	} else if (dtr == FLAGS2_DTR_CAN_RECV) {
		if (__likely(((sio->rcv_buffer_end - sio->rcv_buffer_start) & (SIO_BUFFER_SIZE - 1)) < SIO_RECV_THRESHOLD)) mcr |= MCR_DTR;
	}

	if (__unlikely(mcr != sio->mcr)) {
		sio_out(sio, SIO_MCR, mcr);
		sio->mcr = mcr;
	}
}

static void set_sio(struct sio *sio, unsigned long flags, unsigned long flags2)
{
	unsigned lcr;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SIO)))
		KERNEL$SUICIDE("set_sio AT SPL %08X", KERNEL$SPL);
#endif
	if ((flags & FLAGS_PORTCONTROL) == (sio->current_flags & FLAGS_PORTCONTROL)) goto set_cs;
	lcr = (flags >> __BSF_CONST(FLAGS_WORD)) & 0x1f;
	sio_out(sio, SIO_LCR, lcr | LCR_DLAB);
	sio_out(sio, SIO_DLL, flags >> __BSF_CONST(FLAGS_DIV));
	sio_out(sio, SIO_DLH, flags >> (__BSF_CONST(FLAGS_DIV) + 8));
	sio_out(sio, SIO_LCR, lcr);
	if (__likely(sio->chiptype == CHIP_16550A)) {
		unsigned trigger;
		sio_out(sio, SIO_FCR, 0);
		__slowdown_io();
		__slowdown_io();
		__slowdown_io();
		trigger = flags >> __BSF_CONST(FLAGS_TRIGGER);
		sio_out(sio, SIO_FCR, FCR_ENABLE_FIFO | FCR_CLEAR_RCVR | FCR_CLEAR_XMIT | FCR_DMA_SELECT | (trigger >= 14 ? FCR_TRIGGER_14 : trigger >= 8 ? FCR_TRIGGER_8 : trigger >= 4 ? FCR_TRIGGER_4 : FCR_TRIGGER_1));
	} else {
		sio_out(sio, SIO_FCR, 0);
	}
	sio->mcr = -1;
	sio_in(sio, SIO_IIR);
	sio_in(sio, SIO_LSR);
	sio_in(sio, SIO_MSR);

	sio->rcv_buffer_start = 0;
	sio->rcv_buffer_end = 0;
	sio->xmit_buffer_start = 0;
	sio->xmit_buffer_end = 0;

	set_cs:
	sio->current_flags = flags;
	sio->current_flags2 = flags2;

	update_mcr(sio);
}

static void sio_try_xmit(struct sio *sio)
{
	unsigned msr;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SIO)))
		KERNEL$SUICIDE("sio_try_xmit AT SPL %08X", KERNEL$SPL);
#endif
	msr = sio_in(sio, SIO_MSR);      /* needed to clear an interrupt */
	while (sio->xmit_buffer_start != sio->xmit_buffer_end && read_lsr(sio) & LSR_THRE && ((sio->current_flags2 & FLAGS2_CTS) == FLAGS2_CTS_IGN || msr & MSR_CTS) && ((sio->current_flags2 & FLAGS2_DSR) == FLAGS2_DSR_IGN || msr & MSR_DSR)) {
		sio_out(sio, SIO_TX, sio->xmit_buffer[sio->xmit_buffer_start]);
		__slowdown_io();
		sio->xmit_buffer_start = (sio->xmit_buffer_start + 1) & (SIO_BUFFER_SIZE - 1);
		if (__likely(((sio->xmit_buffer_end - sio->xmit_buffer_start) & (SIO_BUFFER_SIZE - 1)) < SIO_SEND_THRESHOLD)) WQ_WAKE_ALL_PL(&sio->xmit_wait);
	}
	if (__unlikely(sio->msr & ~msr & (MSR_DSR | MSR_DCD))) {
		WQ_WAKE_ALL(&sio->rcv_wait);
		WQ_WAKE_ALL(&sio->xmit_wait);
	}
	sio->msr = msr;
	update_mcr(sio);
}

static void sio_try_receive(struct sio *sio)
{
	unsigned count;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SIO)))
		KERNEL$SUICIDE("sio_try_xmit AT SPL %08X", KERNEL$SPL);
#endif
		/* needed to read LSR to clear an interrupt */
	count = 0;
	while (1) {
		if (__unlikely(++count > SIO_RECV_LIVELOCK)) {
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, sio->dev_name, "RECEIVE LIVELOCK");
			break;
		}
		if (!(read_lsr(sio) & LSR_DR)) break;
		sio->rcv_buffer[sio->rcv_buffer_end] = sio_in(sio, SIO_RX);
		__slowdown_io();
		if (__unlikely(sio->rcv_buffer_start == ((sio->rcv_buffer_end + 1) & (SIO_BUFFER_SIZE - 1)))) {
			if (sio->errorlevel >= 2 && !sio->drop_flag) {
				KERNEL$SYSLOG(__SYSLOG_NET_WARNING, sio->dev_name, "SOFTWARE BUFFER OVERFLOW");
				sio->drop_flag = 1;
			}
			continue;	/* have to drop byte :-( */
		}
		sio->drop_flag = 0;
		sio->rcv_buffer_end = (sio->rcv_buffer_end + 1) & (SIO_BUFFER_SIZE - 1);
		WQ_WAKE_ALL_PL(&sio->rcv_wait);
	}
	update_mcr(sio);
}

static int test_sio_error(struct sio *sio, unsigned long flags2, int w)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SIO)))
		KERNEL$SUICIDE("test_sio_error AT SPL %08X", KERNEL$SPL);
#endif
	if (__likely((flags2 & FLAGS2_DSR) != FLAGS2_DSR_ERR) &&
	    __likely((flags2 & FLAGS2_CD) != FLAGS2_CD_ERR)) {
		return 0;
	}
	sio_try_xmit(sio);	/* update sio->msr */
	if ((flags2 & FLAGS2_DSR) == FLAGS2_DSR_ERR && !(sio->msr & MSR_DSR)) {
		return w ? -EPIPE : 1;
	}
	if ((flags2 & FLAGS2_CD) == FLAGS2_CD_ERR && !(sio->msr & MSR_DCD)) {
		return -ENOLINK;
	}
	return 0;
}

DECL_IOCALL(sio_read, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	struct sio *sio = h->fnode;
	VBUF vbuf;
	long s;
	int r;
	if (__unlikely(h->op != &sio_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(((sio->current_flags ^ h->flags) | (sio->current_flags2 ^ h->flags2)) != 0)) {
		RAISE_SPL(SPL_SIO);
		if (__unlikely(r = test_sio_error(sio, h->flags2, 0))) goto sio_err;
		set_sio(sio, h->flags, h->flags2);
		LOWER_SPL(SPL_DEV);
	}
	if (__unlikely(!RQ->v.len)) goto ret;
	again:
	RAISE_SPL(SPL_SIO);
	if (__unlikely(r = test_sio_error(sio, h->flags2, 0))) {
		sio_err:
		LOWER_SPL(SPL_DEV);
		if (r == 1) goto ret;
		RQ->status = r;
		RETURN_AST(RQ);
	}
	if (sio->rcv_buffer_end >= sio->rcv_buffer_start) vbuf.len = sio->rcv_buffer_end - sio->rcv_buffer_start;
	else vbuf.len = SIO_BUFFER_SIZE - sio->rcv_buffer_start;
	vbuf.ptr = &sio->rcv_buffer[sio->rcv_buffer_start];
	vbuf.spl = SPL_X(SPL_DEV);
	if (__unlikely(!vbuf.len)) {
		if (RQ->progress) {
			LOWER_SPL(SPL_DEV);
			goto ret;
		}
		if (h->flags2 & FLAGS2_NONBLOCK) {
			LOWER_SPL(SPL_DEV);
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&sio->rcv_wait, RQ);
		LOWER_SPL(SPL_DEV);
		RETURN;
	}
	LOWER_SPL(SPL_DEV);
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf);
	if (__unlikely(!s)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	RQ->progress += s;
	RAISE_SPL(SPL_SIO);
	sio->rcv_buffer_start = (sio->rcv_buffer_start + s) & (SIO_BUFFER_SIZE - 1);
	update_mcr(sio);
	LOWER_SPL(SPL_DEV);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (__unlikely(RQ->v.len != 0)) goto again;

	ret:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

DECL_IOCALL(sio_write, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	struct sio *sio = h->fnode;
	VBUF vbuf;
	long s;
	int r;
	if (__unlikely(h->op != &sio_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(((sio->current_flags ^ h->flags) | (sio->current_flags2 ^ h->flags2)) != 0)) {
		RAISE_SPL(SPL_SIO);
		if (__unlikely(r = test_sio_error(sio, h->flags2, 1))) goto sio_err;
		set_sio(sio, h->flags, h->flags2);
		LOWER_SPL(SPL_DEV);
	}
	if (__unlikely(!RQ->v.len)) goto ret;
	again:
	RAISE_SPL(SPL_SIO);
	if (__unlikely(r = test_sio_error(sio, h->flags2, 1))) {
		sio_err:
		LOWER_SPL(SPL_DEV);
		if (r == 1) goto ret;
		RQ->status = r;
		RETURN_AST(RQ);
	}
	if (sio->xmit_buffer_start > sio->xmit_buffer_end) {
		vbuf.len = sio->xmit_buffer_start - sio->xmit_buffer_end - 1;
	} else {
		vbuf.len = SIO_BUFFER_SIZE - sio->xmit_buffer_end - !sio->xmit_buffer_start;
	}
	vbuf.ptr = &sio->xmit_buffer[sio->xmit_buffer_end];
	vbuf.spl = SPL_X(SPL_DEV);
	if (__unlikely(!vbuf.len)) {
		if (RQ->progress) {
			LOWER_SPL(SPL_DEV);
			goto ret;
		}
		if (h->flags2 & FLAGS2_NONBLOCK) {
			LOWER_SPL(SPL_DEV);
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&sio->xmit_wait, RQ);
		LOWER_SPL(SPL_DEV);
		RETURN;
	}
	LOWER_SPL(SPL_DEV);
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_get(&RQ->v, &vbuf);
	if (__unlikely(!s)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
	RQ->progress += s;
	/*__debug_printf("got: %d / %d -> %d,%d (%d)\n", s, vbuf.len, sio->xmit_buffer_start, sio->xmit_buffer_end, RQ->progress);*/
	RAISE_SPL(SPL_SIO);
	sio->xmit_buffer_end = (sio->xmit_buffer_end + s) & (SIO_BUFFER_SIZE - 1);
	if (((sio->xmit_buffer_end + 1) & (SIO_BUFFER_SIZE - 1)) != sio->xmit_buffer_start) WQ_WAKE_ALL(&sio->xmit_wait);
	sio_try_xmit(sio);
	LOWER_SPL(SPL_DEV);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (__unlikely(RQ->v.len != 0)) goto again;
	
	ret:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

DECL_IOCALL(sio_ioctl, SPL_DEV, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	struct sio *sio = h->fnode;
	if (__unlikely(h->op != &sio_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_SELECT_READ: {
			RAISE_SPL(SPL_SIO);
			if (test_sio_error(sio, h->flags2, 0)) goto select_read_ok;
			set_sio(sio, h->flags, h->flags2);
			if (sio->rcv_buffer_start == sio->rcv_buffer_end) {
				if (RQ->param) {
					WQ_WAIT(&sio->rcv_wait, RQ, KERNEL$WAKE_IOCTL);
					LOWER_SPL(SPL_DEV);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
			} else {
				select_read_ok:
				RQ->status = 0;
			}
			LOWER_SPL(SPL_DEV);
			RETURN_AST(RQ);
		}
		case IOCTL_SELECT_WRITE: {
			RAISE_SPL(SPL_SIO);
			if (test_sio_error(sio, h->flags2, 1)) goto select_write_ok;
			set_sio(sio, h->flags, h->flags2);
			if (((sio->xmit_buffer_end + 1) & (SIO_BUFFER_SIZE - 1)) == sio->xmit_buffer_start) {
				if (RQ->param) {
					WQ_WAIT(&sio->xmit_wait, RQ, KERNEL$WAKE_IOCTL);
					LOWER_SPL(SPL_DEV);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
			} else {
				select_write_ok:
				RQ->status = 0;
			}
			LOWER_SPL(SPL_DEV);
			RETURN_AST(RQ);
		}
		case IOCTL_NREAD: {
			RAISE_SPL(SPL_SIO);
			RQ->status = (sio->rcv_buffer_end - sio->rcv_buffer_start) & (SIO_BUFFER_SIZE - 1);
			LOWER_SPL(SPL_DEV);
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

DECL_AST(sio_irq, SPL_SIO, AST)
{
	struct sio *sio = GET_STRUCT(RQ, struct sio, irq_ast);
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	unsigned iir = sio_in(sio, SIO_IIR);
	if (__unlikely(iir & IIR_NO_INT)) goto eoi;
	next_int:
	sio_try_receive(sio);
	sio_try_xmit(sio);
	/* if UART is not on PCI, it does not have level-triggered interrupt
	   --- we must recheck interrupt bit here or we will deadlock */
	iir = sio_in(sio, SIO_IIR);
	if (__unlikely(!(iir & IIR_NO_INT))) {
		if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > SIO_TIMEOUT_LIVELOCK)) {
		/* TODO: use IRQ throttling when it will be implemented */
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, sio->dev_name, "INTERRUPT LIVELOCK, IIR %02X", sio_in(sio, SIO_IIR));
			goto eoi;
		}
		goto next_int;
	}
	eoi:
	sio->irq_ctrl.eoi();
	RETURN;
}

static void sio_init_root(HANDLE *h, void *p)
{
	struct sio *sio = p;
	h->flags = INIT_FLAGS;
	h->flags2 = INIT_FLAGS2;
	h->fnode = sio;
	h->op = &sio_operations;
}

static int sio_dctl(void *p, void **release, char *argv[]);
static int sio_unload(void *p, void **release, char *argv[]);
static int sio_dump(void);

int main(int argc, char *argv[])
{
	long io = -1;
	int irq = -1;
	int com = -1;
	char scan_ports = 0;
	int errorlevel = 0;
	struct __param_table params[] = {
		"IO", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 6, NULL,
		"IRQ", __PARAM_INT, 0, IRQ_LIMIT + 1, NULL,
		"COM", __PARAM_INT, 1, BIOS_COM_PORTS + 1, NULL,
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg;
	int state;
	struct sio *sio;
	int r;
	char *e;
	MALLOC_REQUEST mrq;
	DEVICE_REQUEST devrq;
	unsigned scratch, scratch2;
	params[0].__p = &io;
	params[1].__p = &irq;
	params[2].__p = &com;
	params[3].__p = &errorlevel;
	params[4].__p = &errorlevel;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	mrq.size = sizeof(struct sio);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: CAN'T ALLOC MEMORY");
		goto ret0;
	}
	sio = mrq.ptr;
	memset(sio, 0, sizeof(struct sio));
	WQ_INIT(&sio->rcv_wait, "SIO$RCV_WAIT");
	WQ_INIT(&sio->xmit_wait, "SIO$XMIT_WAIT");
	if (io == -1) {
		if (com == -1) {
			com = 1;
			scan_ports = 1;
		}
		get_port:
		io = *(__u16 *)(KERNEL$ZERO_BANK + 0x400 + (com - 1) * 2);
		if (!io) {
			if (scan_ports) {
				scan_next:
				com++;
				if (com > BIOS_COM_PORTS) {
					_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: NO LPT PORTS FOUND");
					r = -ENODEV;
					goto ret1;
				}
				goto get_port;
			}
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: PORT COM%d NOT ENABLED IN BIOS", com);
			r = -ENODEV;
			goto ret1;
		}
	} else if (com != -1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: CAN'T SPECIFY BOTH IO PORT AND COM NUMBER");
		r = -EBADSYN;
		goto ret1;
	}

	if (com != -1) {
		_snprintf(sio->dev_name, sizeof sio->dev_name, "SIO$COM%d", com);
	} else {
		_snprintf(sio->dev_name, sizeof sio->dev_name, "SIO$COM@" IO_ID, sio->io);
	}
	sio->io = io;

	sio->range.start = io;
	sio->range.len = SIO_REGSPACE;
	sio->range.name = sio->dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&sio->range))) {
		sio->range.name = NULL;
		if (scan_ports) goto scan_next;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", sio->dev_name, sio->range.start, sio->range.start + sio->range.len - 1, strerror(-r));
		goto ret1;
	}

	if (irq == -1) {
		if (io == 0x3f8 || io == 0x3e8) irq = 4;
		else if (io == 0x2f8 || io == 0x2e8) irq = 3;
		else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: IRQ NOT SPECIFIED AND CAN'T BE DETERMINED AUTOMATICALLY");
			r = -ENODEV;
			goto ret1;
		}
	}
	sio->errorlevel = errorlevel;
	mrq.size = SIO_BUFFER_SIZE;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: CAN'T ALLOC MEMORY");
		goto ret1;
	}
	sio->rcv_buffer = mrq.ptr;
	mrq.size = SIO_BUFFER_SIZE;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: CAN'T ALLOC MEMORY");
		goto ret2;
	}
	sio->xmit_buffer = mrq.ptr;
	scratch = sio_in(sio, SIO_IER);
	sio_out(sio, SIO_IER, 0);
	scratch2 = sio_in(sio, SIO_IER);
	if (scratch2) {
		sio_out(sio, SIO_IER, scratch);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SERIAL CONTROLLER NOT FOUND", sio->dev_name);
		r = -ENODEV;
		goto ret3;
	}

	scratch2 = sio_in(sio, SIO_LCR) & ~LCR_DLAB;
	sio_out(sio, SIO_LCR, scratch2 | LCR_DLAB);
	sio_out(sio, SIO_EFR, 0);
	sio_out(sio, SIO_LCR, scratch2);
	sio_out(sio, SIO_FCR, FCR_ENABLE_FIFO);
	scratch = sio_in(sio, SIO_IIR) >> 6;
	sio->xmit_fifo_size = 1;
	switch (scratch) {
		case 0:
			sio->chiptype = CHIP_16450;
			break;
		case 1:
			sio->chiptype = CHIP_UNKNOWN;
			break;
		case 2:
			sio->chiptype = CHIP_16550;
			break;
		case 3:
			sio_out(sio, SIO_LCR, scratch2 | LCR_DLAB);
			if (!sio_in(sio, SIO_EFR)) {
				sio->chiptype = CHIP_16650;
				sio->xmit_fifo_size = 32;
			} else {
				sio->chiptype = CHIP_16550A;
				sio->xmit_fifo_size = 16;
			}
			sio_out(sio, SIO_LCR, scratch2);
			break;
	}
	if (sio->chiptype == CHIP_16450) {
		unsigned status1, status2;
		scratch = sio_in(sio, SIO_SCR);
		sio_out(sio, SIO_SCR, 0x5a);
		status1 = sio_in(sio, SIO_SCR);
		sio_out(sio, SIO_SCR, 0xa5);
		status2 = sio_in(sio, SIO_SCR);
		sio_out(sio, SIO_SCR, scratch);
		if (!(status1 == 0x5a && status2 == 0xa5))
			sio->chiptype = CHIP_8250;
	}
	sio_out(sio, SIO_MCR, 0);
	sio_out(sio, SIO_FCR, FCR_CLEAR_RCVR | FCR_CLEAR_XMIT);
	sio_out(sio, SIO_IER, 0);
	sio_in(sio, SIO_RX);
	sio_in(sio, SIO_IIR);
	sio_in(sio, SIO_MSR);
	sio->irq_ast.fn = sio_irq;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | IRQ_SHARED | IRQ_MASK, NULL, &sio->irq_ast, &sio->irq_ctrl, sio->dev_name))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, sio->dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", sio->dev_name);
		r = -EINVAL;
		goto ret3;
	}
	sio_out(sio, SIO_IER, IER_RDI | IER_THRI | IER_RLSI | IER_MSI);
	RAISE_SPL(SPL_SIO);
	set_sio(sio, INIT_FLAGS, INIT_FLAGS2);
	LOWER_SPL(SPL_ZERO);
	sio->irq_ctrl.eoi();

	_printf("%s: %s\n", sio->dev_name, chipnames[sio->chiptype]);
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", sio->dev_name, sio->io, irq);

	devrq.name = sio->dev_name;
	devrq.driver_name = "SIO.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = sio_init_root;
	devrq.dev_ptr = sio;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = sio_dctl;
	devrq.unload = sio_unload;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", sio->dev_name);
		r = devrq.status;
		goto ret4;
	}
	sio->lnte = devrq.lnte;
	sio->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), sio->dev_name);
	return 0;

	r = 0;

	ret4:
	sio->irq_ctrl.mask();
	sio_out(sio, SIO_IER, 0);
	sio_out(sio, SIO_MCR, 0);
	KERNEL$RELEASE_IRQ(&sio->irq_ctrl, IRQ_RELEASE_MASKED);
	ret3:
	KERNEL$UNIVERSAL_FREE(sio->xmit_buffer);
	ret2:
	KERNEL$UNIVERSAL_FREE(sio->rcv_buffer);
	ret1:
	if (sio->range.name) KERNEL$UNREGISTER_IO_RANGE(&sio->range);
	KERNEL$UNIVERSAL_FREE(sio);
	ret0:
	return r;
}

static int sio_dctl(void *p, void **release, char *argv[])
{
	struct sio *sio = p;
	int reboot = 0;
	if (!argv[1] || _strcasecmp(argv[1], "CRASHDUMP")) {
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "DCTL %s: SYNTAX ERROR", sio->dev_name);
		return -EBADSYN;
	}
	if (argv[2]) {
		if (!_strcasecmp(argv[2], "/REBOOT")) reboot = 1;
		else goto bads;
	}
	RAISE_SPL(SPL_DEV);
	if (KERNEL$SUICIDE_DUMP) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "DCTL %s: CRASHDUMP ALREADY INSTALLED", sio->dev_name);
		return -EEXIST;
	}
	sio->crashdump_reboot = reboot;
	KERNEL$SUICIDE_DUMP_PARAM = sio;
	__barrier();
	KERNEL$SUICIDE_DUMP = sio_dump;
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static int sio_unload(void *p, void **release, char *argv[])
{
	int r;
	struct sio *sio = p;
	if ((r = KERNEL$DEVICE_UNLOAD(sio->lnte, argv))) return r;
	RAISE_SPL(SPL_TOP);
	if (KERNEL$SUICIDE_DUMP == sio_dump && sio == KERNEL$SUICIDE_DUMP_PARAM) {
		KERNEL$SUICIDE_DUMP = NULL;
		__barrier();
		KERNEL$SUICIDE_DUMP_PARAM = NULL;
	}
	LOWER_SPL(SPL_ZERO);
	sio->irq_ctrl.mask();
	sio_out(sio, SIO_IER, 0);
	sio_out(sio, SIO_MCR, 0);
	KERNEL$RELEASE_IRQ(&sio->irq_ctrl, IRQ_RELEASE_MASKED);
	*release = sio->dlrq;
	WQ_WAKE_ALL(&sio->xmit_wait);
	WQ_WAKE_ALL(&sio->rcv_wait);
	KERNEL$UNIVERSAL_FREE(sio->xmit_buffer);
	KERNEL$UNIVERSAL_FREE(sio->rcv_buffer);
	KERNEL$UNREGISTER_IO_RANGE(&sio->range);
	KERNEL$UNIVERSAL_FREE(sio);
	return 0;
}

static int sio_dump(void)
{
	unsigned start, len;
	struct sio *sio = KERNEL$SUICIDE_DUMP_PARAM;
	int r;
	KERNEL$PPL = 0;
	KERNEL$SPL = SPL_X(SPL_SIO);
	if ((r = test_sio_error(sio, sio->current_flags2, 1))) {
		if (r > 0) r = -EPIPE;
		__critical_printf("%s: LINK NOT READY: %s\n", sio->dev_name, strerror(-r));
		return r;
	}
	if (KERNEL$CONSOLE_BUFFER_PTR < KERNEL$CONSOLE_BUFFER_SIZE) {
		start = 0;
		len = KERNEL$CONSOLE_BUFFER_PTR;
	} else {
		start = KERNEL$CONSOLE_BUFFER_PTR & (KERNEL$CONSOLE_BUFFER_SIZE - 1);
		len = KERNEL$CONSOLE_BUFFER_SIZE;
	}
	while (len--) {
		unsigned t;
		sio->xmit_buffer[0] = KERNEL$CONSOLE_BUFFER[start++];
		start &= KERNEL$CONSOLE_BUFFER_SIZE - 1;
		sio->xmit_buffer_start = 0;
		sio->xmit_buffer_end = 1;
		if (sio->xmit_buffer[0] == '\n') {
			sio->xmit_buffer[0] = '\r';
			sio->xmit_buffer[1] = '\n';
			sio->xmit_buffer_end = 2;
		}
		t = 1;
		while (1) {
			sio_try_xmit(sio);
			if (sio->xmit_buffer_start == sio->xmit_buffer_end) break;
			if (__unlikely(t > ((sio->current_flags2 & FLAGS2_CTS) == FLAGS2_CTS_CAN_SEND || (sio->current_flags2 & FLAGS2_DSR) == FLAGS2_DSR_CAN_SEND ? CRASHDUMP_TIMEOUT_FC : CRASHDUMP_TIMEOUT))) {
				__critical_printf("%s: CRASH DUMP TIMEOUT\n", sio->dev_name);
				return -EIO;
			}
			KERNEL$UDELAY(t);
			t *= 2;
		}
	}
	__critical_printf("%s: LOG DUMPED OK\n", sio->dev_name);
	return !sio->crashdump_reboot;
}
