#include <SPAD/PCI.H>
#include <SPAD/PCI_IDS.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/IO.H>
#include <SPAD/TIMER.H>
#include <SYS/SOUNDCARD.H>
#include <ARCH/IRQ.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <VALUES.H>

#include "HDA.H"
#include "HDACODEC.H"
#include "HDAREG.H"

static __const__ struct pci_id_s devices[] = {
	{0x8086, 0x2668, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH6", 0 },
	{0x8086, 0x27d8, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "INTEL ICH7", 0 },
	{0x10b9, 0x5461, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ALI", 0 },
	{0},
};

static pci_id_t pci_id;
static __u8 *mem;
char dev_name[__MAX_STR_LEN];
static char *chip_name;

static AST AZ_IRQ_AST;
static IRQ_CONTROL AZ_IRQ_CTRL;

#define AZ_READ_8(reg)		mmio_inb(mem + (reg))
#define AZ_READ_16(reg)		__16LE2CPU(mmio_inw(mem + (reg)))
#define AZ_READ_24(reg)		(AZ_READ_16(reg) + (AZ_READ_8((reg) + 2) << 16))
#define AZ_READ_32(reg)		__32LE2CPU(mmio_inl(mem + (reg)))
#define AZ_WRITE_8(reg, val)	mmio_outb(mem + (reg), val)
#define AZ_WRITE_16(reg, val)	mmio_outw(mem + (reg), __16CPU2LE(val))
#define AZ_WRITE_24(reg, val)	(AZ_WRITE_8((reg) + 2, (val) >> 16), AZ_WRITE_16(reg, val))
#define AZ_WRITE_32(reg, val)	mmio_outl(mem + (reg), __32CPU2LE(val))

struct az_stream {
	struct bm_entry frag[MAX_DESCS];
};

struct az_desc {
	struct az_stream stream[2];
	__u32 corb[256];
	struct rirb_entry rirb[256];
};

#define DESC_CLUSTERS ((sizeof(struct az_desc) + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS)

char OSS_DMA_USE_64;
static __u32 az_gcap;
static struct az_desc *desc;
static __u64 desc_dmaaddr;
static vspace_dmaunlock_t *desc_unlock_32;
static vspace_dma64unlock_t *desc_unlock_64;
static unsigned corb_size;
static unsigned rirb_size;
static unsigned corb_write;
static unsigned rirb_read;
static __u32 intctl;
static unsigned codec;

static TIMER HDA_PLAYBACK_TIMEOUT;
static TIMER HDA_RECORD_TIMEOUT;

__const__ int RATELIST[129] = {
	48000, 24000, 16000, -1, 9600, 8000, -1, 6000,
	96000, -1, 32000, -1, -1, -1, -1, -1,
	144000, -1, -1, -1, -1, -1, -1, -1,
	192000, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	44100, 22050, -1, 11025, -1, -1, -1, -1,
	88200, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	176400, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	-1, -1, -1, -1, -1, -1, -1, -1,
	0,
};

unsigned long PLAYBACK_RATES[(128 + 8*sizeof(unsigned long)-1) / 8 / sizeof(unsigned long)];
unsigned long RECORD_RATES[(128 + 8*sizeof(unsigned long)-1) / 8 / sizeof(unsigned long)];

__const__ HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_OPEN,		/* open */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};


static void HDA_STREAM_INTERRUPT(int record);

static int AZ_RESET(void)
{
	int i;
	AZ_WRITE_16(AZ_STATESTS, 0xffff);
	AZ_WRITE_32(AZ_GCTL, 0);
	i = 0;
	while (KERNEL$UDELAY(10), AZ_READ_32(AZ_GCTL) & AZ_GCTL_RESET) {
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO SET RESET STATE");
			return -1;
		}
	}
	KERNEL$UDELAY(300);
	AZ_WRITE_32(AZ_GCTL, AZ_GCTL_RESET);
	KERNEL$UDELAY(10000);
	i = 0;
	while (KERNEL$UDELAY(10), !(AZ_READ_32(AZ_GCTL) & AZ_GCTL_RESET)) {
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO CLEAR RESET STATE");
			return -1;
		}
	}
	KERNEL$UDELAY(1000);
	return 0;
}

static void AZ_SETUP_BUFFERS(void)
{
	int i;
	AZ_WRITE_8(AZ_CORBCTL, 0);
	AZ_WRITE_16(AZ_CORBRP, AZ_CORBRP_RESET);
	i = 0;
	while (!(AZ_READ_16(AZ_CORBRP) & AZ_CORBRP_RESET)) {
		KERNEL$UDELAY(10);
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO SET CORB RESET STATE");
			break;
		}
	}
	AZ_WRITE_16(AZ_CORBRP, 0);
	i = 0;
	while (AZ_READ_16(AZ_CORBRP) & AZ_CORBRP_RESET) {
		KERNEL$UDELAY(10);
		if (++i >= 100) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "FAILED TO CLEAR CORB RESET STATE");
			break;
		}
	}
	KERNEL$UDELAY(10);
	AZ_WRITE_32(AZ_CORBLBASE, desc_dmaaddr + (int)((struct az_desc *)NULL)->corb);
	AZ_WRITE_32(AZ_CORBUBASE, desc_dmaaddr >> 32);
	AZ_WRITE_16(AZ_CORBWP, 0);
	AZ_WRITE_8(AZ_CORBCTL, AZ_CORBCTL_ENABLE | AZ_CORBCTL_CMEI);
	corb_write = 0;
	AZ_WRITE_8(AZ_RIRBCTL, 0);
	AZ_WRITE_16(AZ_RIRBWP, AZ_RIRBWP_RESET);
	AZ_WRITE_32(AZ_RIRBLBASE, desc_dmaaddr + (int)((struct az_desc *)NULL)->rirb);
	AZ_WRITE_32(AZ_RIRBUBASE, desc_dmaaddr >> 32);
	AZ_WRITE_16(AZ_RINTCNT, rirb_size / 4);
	AZ_WRITE_8(AZ_RIRBCTL, AZ_RIRBCTL_OVER_INT | AZ_RIRBCTL_ENABLE | AZ_RIRBCTL_ENABLE_INT);
	rirb_read = 0;
}

DECL_AST(AZ_IRQ, SPL_SND, AST)
{
	__u32 intstat = AZ_READ_32(AZ_INTSTS) & intctl;
	/*if (intstat) __debug_printf("irq...: %x\n", intstat);*/
	if (__unlikely(intstat & AZ_INTSTS_CIS)) {
		unsigned corbst, rirbsts;
		unsigned statests = AZ_READ_16(AZ_STATESTS);
		AZ_WRITE_16(AZ_STATESTS, statests);
	
		corbst = AZ_READ_8(AZ_CORBST);
		if (__unlikely(corbst & AZ_CORBST_CMEI)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CORB MEMORY ERROR");
			AZ_WRITE_8(AZ_CORBST, AZ_CORBST_CMEI);
			AZ_SETUP_BUFFERS();
			goto skip_buffer_int;
		}
		rirbsts = AZ_READ_8(AZ_RIRBSTS);
		if (__unlikely(rirbsts & AZ_RIRBSTS_OVER_INT))
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "RESPONSE INPUT BUFFER OVERFLOW");
		if (__likely(rirbsts & (AZ_RIRBSTS_OVER_INT | AZ_RIRBSTS_INT))) {
			unsigned wp = AZ_READ_16(AZ_RIRBWP);
			unsigned i = rirb_read;
			while (1) {
				unsigned ex = __32LE2CPU(desc->rirb[i].extd);
				if (__unlikely((ex & RIRB_CODEC_MASK) != codec)) goto skip_resp;
				HDA_CODEC_RESPONSE(ex & RIRB_CODEC_MASK, __32LE2CPU(desc->rirb[i].resp), ex & RIRB_UNSOLICITED);
				skip_resp:
				if (__likely(i == wp)) break;
				i = (i + 1) & (rirb_size - 1);
			}
			rirb_read = (i + 1) & (rirb_size - 1);
			AZ_WRITE_8(AZ_RIRBSTS, rirbsts & (AZ_RIRBSTS_OVER_INT | AZ_RIRBSTS_INT));
		}
		skip_buffer_int:;
	}
	if (__likely(intstat & AZ_INTSTS_OUTPUT0)) {
		HDA_STREAM_INTERRUPT(0);
	}
	if (__unlikely(intstat & AZ_INTSTS_INPUT0)) {
		HDA_STREAM_INTERRUPT(1);
	}
	AZ_IRQ_CTRL.eoi();
	RETURN;
}

int HDA_CODEC_REQUEST(unsigned codec_id, unsigned node, unsigned verb)
{
	unsigned rp;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("HDA_CODEC_REQUEST AT SPL %08X", KERNEL$SPL);
	rp = (AZ_READ_16(AZ_CORBRP) - 1) & (corb_size - 1);
	if (__unlikely(corb_write == rp)) return -1;
	corb_write = (corb_write + 1) & (corb_size - 1);
	desc->corb[corb_write] = __32CPU2LE((codec_id << CORB_CODEC_SHIFT) + (node << CORB_NODE_SHIFT) + (verb << CORB_VERB_SHIFT));
	AZ_WRITE_16(AZ_CORBWP, corb_write);
	return 0;
}

unsigned play_frag;
unsigned rec_frag;

void SND_START(int record)
{
	unsigned frags, fragsize_bits, format, rate;
	unsigned base, stream_id, int_id;
	unsigned i, cnt, c_fm;
	struct az_stream *stream;
	struct page_desc *pages;
	SND_STOP(record);
	if (__likely(!record)) {
		frags = oss_playback_frags;
		fragsize_bits = oss_playback_fragsize_bits;
		format = oss_playback_format;
		rate = oss_playback_rate;
		base = AZ_OUTPUT0;
		int_id = AZ_INTCTL_OUTPUT0;
		stream = &desc->stream[0];
		stream_id = PLAYBACK_STREAM_ID;
		pages = PAGES;
	} else {
		frags = oss_record_frags;
		fragsize_bits = oss_record_fragsize_bits;
		format = oss_record_format;
		rate = oss_record_rate;
		base = AZ_INPUT0;
		int_id = AZ_INTCTL_INPUT0;
		stream = &desc->stream[1];
		stream_id = RECORD_STREAM_ID;
		pages = CAP_PAGES;
	}
	c_fm = (rate << AZ_SDFMT_RATE_SHIFT) | AZ_SDFMT_BITS_16 | (1 << AZ_SDFMT_CHAN_SHIFT);
	HDA_CODEC_SET_FORMAT(record, c_fm);
	memset(stream, 0, MAX_DESCS * sizeof(struct bm_entry));
	if (__unlikely(frags << fragsize_bits >> PAGESIZE_BITS > N_PAGES))
		KERNEL$SUICIDE("SND_START: %s: OUT OF PAGES (%u >= %u)", dev_name, frags << fragsize_bits >> PAGESIZE_BITS, N_PAGES);
	for (i = 0; i < frags; i++) {
		stream->frag[i].addr = __64CPU2LE(pages[i << fragsize_bits >> PAGESIZE_BITS].device + ((i << fragsize_bits) & ((1 << PAGESIZE_BITS) - 1)));
		stream->frag[i].len = 1 << fragsize_bits;
		stream->frag[i].flags = BM_IOC;
	}
	AZ_WRITE_24(base + AZ_SDCTL, 0);
	KERNEL$UDELAY(10);
	AZ_WRITE_24(base + AZ_SDCTL, AZ_SDCTL_SRST);
	cnt = 0;
	while (KERNEL$UDELAY(3), !(AZ_READ_24(base + AZ_SDCTL) & AZ_SDCTL_SRST)) {
		if (__unlikely(++cnt >= 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO SET RESET STATE", record);
			return;
		}
	}
	AZ_WRITE_24(base + AZ_SDCTL, 0);
	cnt = 0;
	while (KERNEL$UDELAY(3), AZ_READ_24(base + AZ_SDCTL) & AZ_SDCTL_SRST) {
		if (__unlikely(++cnt >= 1000)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO CLEAR RESET STATE", record);
			return;
		}
	}
	KERNEL$UDELAY(10);
	AZ_WRITE_8(base + AZ_SDSTS, AZ_READ_8(base + AZ_SDSTS));
	AZ_WRITE_24(base + AZ_SDCTL, (stream_id << AZ_SDCTL_NUMBER_SHIFT) | AZ_SDCTL_PRIO | AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL);
	AZ_WRITE_32(base + AZ_SDCBL, frags << fragsize_bits);
	AZ_WRITE_16(base + AZ_SDFMT, c_fm);
	AZ_WRITE_16(base + AZ_SDLVI, frags - 1);
	AZ_WRITE_32(base + AZ_SDBPL, desc_dmaaddr + ((__u8 *)stream - (__u8 *)desc));
	AZ_WRITE_32(base + AZ_SDBPU, desc_dmaaddr >> 32);
	/*__debug_printf("%016Lx\n", desc_dmaaddr);*/
	KERNEL$UDELAY(10);
	intctl |= int_id;
	AZ_WRITE_32(AZ_INTCTL, intctl);
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL | AZ_SDCTL_RUN);
	if (__likely(!record)) {
		play_frag = 0;
		OSS_FLAGS |= OSS_PLAYBACK;
		KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &HDA_PLAYBACK_TIMEOUT);
	} else {
		rec_frag = 0;
		OSS_FLAGS |= OSS_RECORD;
		KERNEL$DEL_TIMER(&HDA_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &HDA_RECORD_TIMEOUT);
	}
}

void SND_STOP(int record)
{
	unsigned base, int_id, cnt;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
		int_id = AZ_INTCTL_OUTPUT0;
		if (OSS_FLAGS & OSS_PLAYBACK) OSS_FLAGS &= ~(OSS_PLAYBACK | OSS_PLAYBACK_STOP | OSS_LOOP_PLAYBACK);
		KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&HDA_PLAYBACK_TIMEOUT.list);
	} else {
		base = AZ_INPUT0;
		int_id = AZ_INTCTL_INPUT0;
		OSS_FLAGS &= ~OSS_RECORD;
		KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&HDA_PLAYBACK_TIMEOUT.list);
	}
	AZ_WRITE_16(base + AZ_SDCTL, 0);
	cnt = 0;
	while (AZ_READ_16(base + AZ_SDCTL) & AZ_SDCTL_RUN) {
		KERNEL$UDELAY(1);
		if (++cnt >= 1000) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "STREAM %d FAILED TO STOP", record);
			AZ_WRITE_24(base + AZ_SDCTL, AZ_SDCTL_SRST);
			break;
		}
	}
	intctl &= ~int_id;
	AZ_WRITE_32(AZ_INTCTL, intctl);
}

void SND_PAUSE(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
		KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&HDA_PLAYBACK_TIMEOUT.list);
	} else {
		base = AZ_INPUT0;
		KERNEL$DEL_TIMER(&HDA_RECORD_TIMEOUT);
		VOID_LIST_ENTRY(&HDA_RECORD_TIMEOUT.list);
	}
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL);
}

void SND_RESUME(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
		KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &HDA_PLAYBACK_TIMEOUT);
	} else {
		base = AZ_INPUT0;
		KERNEL$DEL_TIMER(&HDA_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &HDA_RECORD_TIMEOUT);
	}
	AZ_WRITE_16(base + AZ_SDCTL, AZ_SDCTL_EN_INT_ERR | AZ_SDCTL_EN_INT_COMPL | AZ_SDCTL_RUN);
}

unsigned SND_GET_PTR(int record)
{
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	return AZ_READ_32(base + AZ_SDLPIB);
}

static void HDA_STREAM_INTERRUPT(int record)
{
	unsigned stat;
	unsigned base;
	if (__likely(!record)) {
		base = AZ_OUTPUT0;
	} else {
		base = AZ_INPUT0;
	}
	stat = AZ_READ_8(base + AZ_SDSTS);
	if (__unlikely(stat & AZ_SDSTS_ERR)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "MEMORY ERROR ON STREAM %d", record);
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_ERR);
		OSS_STOP(record);
		return;
	}
	if (__unlikely(stat & AZ_SDSTS_FIFO_ERR)) {
		KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "FIFO UNDER/OVERFLOW ON STREAM %d", record);
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_FIFO_ERR);
	}
	if (__likely(stat & AZ_SDSTS_COMPL)) {
		unsigned pos;
		unsigned cnt;
		AZ_WRITE_8(base + AZ_SDSTS, AZ_SDSTS_COMPL);
		pos = SND_GET_PTR(record);
		if (__likely(!record)) {
			pos >>= oss_playback_fragsize_bits;
			cnt = oss_playback_frags;
			if (__likely(!(OSS_FLAGS & OSS_BLOCK_PLAYBACK))) {
				KERNEL$DEL_TIMER(&HDA_PLAYBACK_TIMEOUT);
				KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &HDA_PLAYBACK_TIMEOUT);
			}
			while (play_frag != pos) {
				if (__unlikely(OSS_PLAYBACK_INTR())) return;
				if (__unlikely(++play_frag == oss_playback_frags)) play_frag = 0;
				if (__unlikely(!cnt--)) {
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "BAD PLAYBACK POINTER RETURNED (%u); FRAGS %u, FRAGSIZE_BITS %u", pos, oss_playback_frags, oss_playback_fragsize_bits);
					break;
				}
			}
		} else {
			pos >>= oss_record_fragsize_bits;
			cnt = oss_record_frags;
			if (__likely(!(OSS_FLAGS & OSS_BLOCK_RECORD))) {
				KERNEL$DEL_TIMER(&HDA_RECORD_TIMEOUT);
				KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &HDA_RECORD_TIMEOUT);
			}
			while (rec_frag != pos) {
				if (__unlikely(OSS_RECORD_INTR())) return;
				if (__unlikely(++rec_frag == oss_record_frags)) rec_frag = 0;
				if (__unlikely(!cnt--)) {
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "BAD RECORD POINTER RETURNED (%u); FRAGS %u, FRAGSIZE_BITS %u", pos, oss_record_frags, oss_record_fragsize_bits);
					break;
				}
			}
		}
	}
}

static void HDA_PLAYBACK_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	VOID_LIST_ENTRY(&t->list);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "PLAYBACK INTERRUPT TIMEOUT");
	OSS_STOP(0);
}

static void HDA_RECORD_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	VOID_LIST_ENTRY(&t->list);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "RECORD INTERRUPT TIMEOUT");
	OSS_STOP(1);
}

static void *lnte, *dlrq;
static int HDA_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	unsigned memory = DEFAULT_MEMORY;
	struct __param_table params[] = {
		"MEMORY", __PARAM_UNSIGNED_INT, 1, MAXINT, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	int irq;
	char *opt, *optend, *str;
	int r;
	CONTIG_AREA_REQUEST car;
	DEVICE_REQUEST devrq;
	VDESC v;
	unsigned statests;
	int i;
	char *m;
	unsigned pages, page_size;
	char gstr[__MAX_STR_LEN];
	params[0].__p = &memory;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (memory < __PAGE_CLUSTER_SIZE || memory & (memory - 1) || memory > MAX_MEMORY_SIZE) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: BAD MEMORY SIZE (MUST BE POWER OF 2 AND BETWEEN %u AND %u)", (unsigned)__PAGE_CLUSTER_SIZE, (unsigned)MAX_MEMORY_SIZE);
		r = -EBADSYN;
		goto ret0;
	}
	if (PCI$FIND_DEVICE(devices, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, NULL, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "HDA: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	pci_id = id;
	_snprintf(dev_name, sizeof(dev_name), "SND$HDA@" PCI_ID_FORMAT, pci_id);
	/*__debug_printf("command: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_COMMAND));*/
	PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER | PCI_COMMAND_INT_DIS);
	/*__debug_printf("command: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_COMMAND));
	__debug_printf("status: %04x\n", PCI$READ_CONFIG_WORD(pci_id, PCI_STATUS));
	__debug_printf("bist: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_BIST));
	__debug_printf("header type: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_HEADER_TYPE));
	__debug_printf("latency timer: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_LATENCY_TIMER));
	__debug_printf("cache size: %02x\n", PCI$READ_CONFIG_BYTE(pci_id, PCI_CACHE_LINE_SIZE));*/
	irq = PCI$READ_INTERRUPT_LINE(pci_id);
	mem = PCI$MAP_MEM_RESOURCE(pci_id, 0, AZ_REGSPACE);
	if (!mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURC", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
		r = __PTR_ERR(mem);
		goto ret1;
	}
	if (AZ_RESET()) {
		cant_init:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CONTROLLER FAILED TO INITIALIZE", dev_name);
		r = -ENXIO;
		goto ret2;
	}
	az_gcap = AZ_READ_32(AZ_GCAP);
	OSS_DMA_USE_64 = !!(AZ_GCAP_64 & AZ_GCAP_64);
	statests = AZ_READ_16(AZ_STATESTS);
	if (!statests) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "NO CODEC ON HDA LINK");
		r = -ENXIO;
		goto ret2;
	}

	for (i = 0; i < 2; i++) {
		unsigned reg = !i ? AZ_CORBSIZE : AZ_RIRBSIZE;
		unsigned sizereg = AZ_READ_8(reg);
		unsigned size = 0;	/* warning, go away */
		if ((sizereg & AZ_CORBSIZE_MASK) != AZ_CORBSIZE_256) {
			if (size & AZ_CORBSIZE_CAP_256)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_256);
			else if (size & AZ_CORBSIZE_CAP_16)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_16);
			else if (size & AZ_CORBSIZE_CAP_2)
				AZ_WRITE_8(reg, (sizereg & ~AZ_CORBSIZE_MASK) | AZ_CORBSIZE_2);
			sizereg = AZ_READ_8(reg);
		}
		if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_256)
			size = 256;
		else if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_16)
			size = 16;
		else if ((sizereg & AZ_CORBSIZE_MASK) == AZ_CORBSIZE_2)
			size = 2;
		else {
			KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "UNSUPPORTED %s SIZE (REG=%02X)", !i ? "CORB" : "RIRB", sizereg);
			goto cant_init;
		}
		if (!i) corb_size = size;
		else rirb_size = size;
	}

	car.nclusters = DESC_CLUSTERS;
	car.flags = CARF_PHYSCONTIG | CARF_DATA | (OSS_DMA_USE_64 ? 0 : CARF_PCIDMA);
	car.align = 0;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		if (car.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE COMMAND BUFFER", dev_name);
		r = car.status;
		goto ret2;
	}
	desc = car.ptr;

	if ((r = KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), (__p_addr)DESC_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_UC))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T SET UNCACHED MEMORY MODE", dev_name);
		goto ret25;
	}

	v.ptr = (unsigned long)desc;
	v.len = sizeof(struct az_desc);
	v.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	if (!OSS_DMA_USE_64) {
		VDMA dma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &desc_unlock_32);
		desc_dmaaddr = dma.ptr;
	} else {
		VDMA64 dma;
		KERNEL$VIRTUAL.op->vspace_dma64lock(&v, PF_READ | PF_WRITE, &dma, &desc_unlock_64);
		desc_dmaaddr = dma.ptr;
	}
	LOWER_SPL(SPL_ZERO);
	
	AZ_SETUP_BUFFERS();

	intctl = 0;
	AZ_IRQ_AST.fn = AZ_IRQ;
	if ((m = KERNEL$REQUEST_IRQ(irq, IRQ_SHARED | IRQ_AST_HANDLER, NULL, &AZ_IRQ_AST, &AZ_IRQ_CTRL, dev_name))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IRQ %d: %s", dev_name, irq, m);
		r = -EBUSY;
		goto ret3;
	}
	RAISE_SPL(SPL_SND);
	intctl = AZ_INTCTL_GIE | AZ_INTCTL_CIE;
	AZ_WRITE_32(AZ_INTCTL, intctl);
	LOWER_SPL(SPL_ZERO);

	KERNEL$SLEEP(JIFFIES_PER_SECOND / 50 + 1);	/* ICH 7 needs this */

	r = 0;
	/* it is already known that statests != 0 */
	do {
		int rr;
		codec = __BSF(statests);
		RAISE_SPL(SPL_SND);
		rr = HDA_CODEC_INIT(codec);
		LOWER_SPL(SPL_ZERO);
		if (!rr) goto codec_ok;
		if (!r) r = rr;
	} while (statests &= ~(1 << codec));
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INIT HDA CODEC", dev_name);
	goto ret4;

	codec_ok:

	OSS_INIT(1);
	pages = MAX_DESCS;
	page_size = memory / MAX_DESCS;
	while (page_size < __PAGE_CLUSTER_SIZE) {
		page_size <<= 1;
		pages >>= 1;
	}
	if ((r = OSS_ALLOC_DMAPAGES(pages, page_size))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", dev_name, strerror(-r));
		goto ret5;
	}

	VOID_LIST_ENTRY(&HDA_PLAYBACK_TIMEOUT.list);
	HDA_PLAYBACK_TIMEOUT.fn = HDA_PLAYBACK_TIMEOUT_FN;
	VOID_LIST_ENTRY(&HDA_RECORD_TIMEOUT.list);
	HDA_RECORD_TIMEOUT.fn = HDA_RECORD_TIMEOUT_FN;

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
	_printf("%s: MEM %"__64_format"X, IRQ %d\n", dev_name, PCI$READ_MEM_RESOURCE(pci_id, 0, NULL), irq);

	devrq.name = dev_name;
	devrq.driver_name = "HDA.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = OSS_SET_ROOT;
	devrq.dev_ptr = NULL;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = HDA_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret6;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, dev_name);
	lnte = devrq.lnte;
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret6:
	OSS_FREE_DMAPAGES();
	ret5:
	HDA_CODEC_DESTROY();
	ret4:
	AZ_IRQ_CTRL.mask();
	AZ_WRITE_32(AZ_INTCTL, 0);
	KERNEL$RELEASE_IRQ(&AZ_IRQ_CTRL, IRQ_RELEASE_MASKED);
	ret3:
	RAISE_SPL(SPL_VSPACE);
	if (!OSS_DMA_USE_64) desc_unlock_32(desc_dmaaddr);
	else desc_unlock_64(desc_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	ret25:
	KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), (__p_addr)DESC_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_WB);
	KERNEL$VM_RELEASE_CONTIG_AREA(desc, DESC_CLUSTERS);
	ret2:
	PCI$UNMAP_MEM_RESOURCE(pci_id, mem, AZ_REGSPACE);
	ret1:
	PCI$FREE_DEVICE(pci_id);
	ret0:
	return r;
}

static int HDA_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	RAISE_SPL(SPL_SND);
	OSS_STOP(0);
	OSS_STOP(1);
	LOWER_SPL(SPL_ZERO);
	OSS_FREE_DMAPAGES();
	HDA_CODEC_DESTROY();
	AZ_IRQ_CTRL.mask();
	AZ_WRITE_32(AZ_INTCTL, 0);
	KERNEL$RELEASE_IRQ(&AZ_IRQ_CTRL, IRQ_RELEASE_MASKED);
	RAISE_SPL(SPL_VSPACE);
	if (!OSS_DMA_USE_64) desc_unlock_32(desc_dmaaddr);
	else desc_unlock_64(desc_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	KERNEL$SET_MEMORY_PAT(KERNEL$VIRT_2_PHYS(desc), (__p_addr)DESC_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_WB);
	KERNEL$VM_RELEASE_CONTIG_AREA(desc, DESC_CLUSTERS);
	PCI$UNMAP_MEM_RESOURCE(pci_id, mem, AZ_REGSPACE);
	PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
