#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/TIMER.H>
#include <SPAD/WQ.H>
#include <SPAD/SYSLOG.H>
#include <STDLIB.H>

#include "HDA.H"
#include "HDACODEC.H"

#define CODEC_READY_TIMEOUT	JIFFIES_PER_SECOND
#define CODEC_RESPONSE_TIMEOUT	(JIFFIES_PER_SECOND / 10)
#define CODEC_ASYNC_WAIT	(JIFFIES_PER_SECOND / 2)
#define ASYNC_REQUEST_BUFFER	1024

static int codec;
static int block_async_requests = 0;
static int codec_error;
static int was_response;
static __u32 response;

static DECL_TIMER(response_timeout);
static WQ_DECL(response_wait, "HDA$RESPONSE_WAIT");

static __u32 async_request_buffer[ASYNC_REQUEST_BUFFER];
static unsigned async_request_buffer_size;
static u_jiffies_lo_t last_async_rq = 0;

#define VERB_MASK	0xfffff
#define NODE_SHIFT	20

static void HDA_FLUSH_ASYNC_REQUESTS(void)
{
	unsigned i;
	for (i = 0; i < async_request_buffer_size; i++) {
		if (HDA_CODEC_REQUEST(codec, async_request_buffer[i] >> NODE_SHIFT, async_request_buffer[i] & VERB_MASK) < 0) break;
		last_async_rq = KERNEL$GET_JIFFIES_LO();
	}
	memmove(async_request_buffer, async_request_buffer + i, (async_request_buffer_size -= i) * sizeof(*async_request_buffer));
}

void HDA_CODEC_RESPONSE(unsigned codec_id, __u32 resp, int unsol)
{
	if (__unlikely(codec_id != codec)) return;
	if (!WQ_EMPTY(&response_wait) && __likely(!unsol)) {
		response = resp;
		was_response = 1;
		WQ_WAKE_ALL_PL(&response_wait);
	} else {
	}
	if (__unlikely(async_request_buffer_size)) HDA_FLUSH_ASYNC_REQUESTS();
	/*__debug_printf("response: codec %d, response %08x, unsol %d\n", codec, resp, unsol);*/
}

static void HDA_CODEC_RESPONSE_TIMEOUT(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	VOID_LIST_ENTRY(&t->list);
	WQ_WAKE_ALL_PL(&response_wait);
}

static __u32 HDA_CODEC_REQUEST_SYNC(unsigned node, unsigned verb)
{
	unsigned t = 0;
	block_async_requests++;
	while (__unlikely(KERNEL$GET_JIFFIES_LO() - last_async_rq <= TIMEOUT_JIFFIES(CODEC_ASYNC_WAIT))) {
		KERNEL$SLEEP(1);
	}
	while (__unlikely(HDA_CODEC_REQUEST(codec, node, verb) < 0)) {
		KERNEL$SLEEP(1);
		if (__unlikely(++t >= CODEC_READY_TIMEOUT)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "TIMEOUT WAITING FOR SPACE IN REQUEST QUEUE");
			block_async_requests--;
			codec_error = 1;
			return 0;
		}
	}
	block_async_requests--;
	was_response = 0;
	response_timeout.fn = HDA_CODEC_RESPONSE_TIMEOUT;
	KERNEL$SET_TIMER(CODEC_RESPONSE_TIMEOUT, &response_timeout);
	WQ_WAIT_SYNC(&response_wait);
	KERNEL$DEL_TIMER(&response_timeout);
	if (__unlikely(!was_response)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "TIMEOUT WAITING FOR CODEC RESPONSE");
		codec_error = 1;
		return 0;
	}
	return response;
}

static void HDA_CODEC_REQUEST_ASYNC(unsigned node, unsigned verb, unsigned verb_param_mask)
{
	unsigned i;
	if (__unlikely(!block_async_requests) && __likely(HDA_CODEC_REQUEST(codec, node, verb) >= 0)) {
		last_async_rq = KERNEL$GET_JIFFIES_LO();
		return;
	}
	verb |= node << NODE_SHIFT;
	for (i = 0; i < async_request_buffer_size; i++) {
		if ((async_request_buffer[i] & ~verb_param_mask) == (verb & ~verb_param_mask)) {
			async_request_buffer[i] = verb;
			return;
		}
	}
	if (__likely(async_request_buffer_size < ASYNC_REQUEST_BUFFER)) {
		async_request_buffer[async_request_buffer_size++] = verb;
		return;
	}
	memmove(async_request_buffer, async_request_buffer + 1, sizeof(async_request_buffer) - sizeof(*async_request_buffer));
	async_request_buffer[ASYNC_REQUEST_BUFFER - 1] = verb;
}

static __u32 vendor_id, revision_id;
static unsigned func_node;
static unsigned out_da_node, in_ad_node;

static struct widget {
	int present;
	__u32 type;
	__u32 cap;
	unsigned delay;
	unsigned n_conn_list;
	unsigned *conn_list;
	unsigned n_rev_conn_list;
	unsigned *rev_conn_list;

	union {
		struct {
			__u32 sz_rate;
			__u32 fmt;
		} io;
		struct {
			__u32 cap;
			__u32 config;
		} pin;
	} u;
	__u32 in_amp_cap;
	__u32 out_amp_cap;

	int ff_dist;
	unsigned origin;
	int used;
} *widgets;

static unsigned max_widgets;

static void HDA_GET_SUB_NODES(unsigned node, unsigned *start, unsigned *count)
{
	__u32 sub_nodes = HDA_CODEC_REQUEST_SYNC(node, HDA_GET_PARAMETER | HDA_SUB_NODE_COUNT);
	*count = (sub_nodes & HDA_SUB_NODE_COUNT_NODES_MASK) >> HDA_SUB_NODE_COUNT_NODES_SHIFT;
	*start = (sub_nodes & HDA_SUB_NODE_COUNT_START_MASK) >> HDA_SUB_NODE_COUNT_START_SHIFT;
}

static void HDA_CONNECTION_LIST_ADD(unsigned node, unsigned connection, unsigned range)
{
	if (range) {
		unsigned start;
		if (!widgets[node].n_conn_list) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "RANGE SPECIFIED ON FIRST ENTRY OF CONNECTION LIST");
			goto no_range;
		}
		for (start = widgets[node].conn_list[widgets[node].n_conn_list - 1]; start < connection; start++) HDA_CONNECTION_LIST_ADD(node, start, 0);
	}
	no_range:
	if (connection >= max_widgets) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "BAD CONNECTION WIDGET NUMBER %u >= %u", connection, max_widgets);
		return;
	}

	if (__alloc_size(widgets[node].conn_list) < (widgets[node].n_conn_list + 1) * sizeof(unsigned)) {
		unsigned *c = __sync_realloc(widgets[node].conn_list, (widgets[node].n_conn_list + 1) * sizeof(unsigned));
		if (!c) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, dev_name, "OUT OF MEMORY FOR CONNECTION LIST");
			return;
		}
		widgets[node].conn_list = c;
	}
	widgets[node].conn_list[widgets[node].n_conn_list++] = connection;

	if (__alloc_size(widgets[connection].rev_conn_list) < (widgets[connection].n_rev_conn_list + 1) * sizeof(unsigned)) {
		unsigned *c = __sync_realloc(widgets[connection].rev_conn_list, (widgets[connection].n_rev_conn_list + 1) * sizeof(unsigned));
		if (!c) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, dev_name, "OUT OF MEMORY FOR CONNECTION LIST");
			return;
		}
		widgets[connection].rev_conn_list = c;
	}
	widgets[connection].rev_conn_list[widgets[connection].n_rev_conn_list++] = node;
#ifdef HDA_DUMP
	_printf("%d, ", connection);
#endif
}

static char comp_ad;
static char comp_input;

static int CAPABLE(struct widget *w)
{
	if (w->used) return 0;
	if (!comp_ad) {
		if (w->type != HDA_AUD_WIDGET_CAP_TYPE_PIN_COMPLEX) return 0;
		if (!comp_input) {
			if (!(w->u.pin.cap & HDA_PIN_CAP_OUTPUT)) return 0;
		} else {
			if (!(w->u.pin.cap & HDA_PIN_CAP_INPUT)) return 0;
		}
	} else {
		if (w->ff_dist <= 0) return 0;
		if (!comp_input) {
			if (w->type != HDA_AUD_WIDGET_CAP_TYPE_AUDIO_OUTPUT) return 0;
		} else {
			if (w->type != HDA_AUD_WIDGET_CAP_TYPE_AUDIO_INPUT) return 0;
		}
		if (!(w->u.io.sz_rate & HDA_SAMPLE_SZ_RATE_CAP_16B)) return 0;
		if (!(w->u.io.fmt & HDA_STREAM_FMT_PCM)) return 0;
	}
	if (!(w->cap & HDA_AUD_WIDGET_CAP_STEREO)) return 0;
	if (w->cap & HDA_AUD_WIDGET_CAP_DIGITAL) return 0;
	return 1;
}

static int COMPARE_WIDGET(__const__ void *p1, __const__ void *p2)
{
	struct widget *w1 = *(struct widget **)p1;
	struct widget *w2 = *(struct widget **)p2;
	if (!CAPABLE(w1)) {
		if (!CAPABLE(w2)) return 0;
		else return 1;
	}
	if (!CAPABLE(w2)) {
		return -1;
	}
	if (!comp_ad) {
		unsigned device1 = w1->u.pin.config & HDA_CONFIG_DEFAULT_DEVICE_MASK;
		unsigned device2 = w2->u.pin.config & HDA_CONFIG_DEFAULT_DEVICE_MASK;
		unsigned location1 = w1->u.pin.config & HDA_CONFIG_LOCATION_HI_MASK;
		unsigned location2 = w2->u.pin.config & HDA_CONFIG_LOCATION_HI_MASK;
		unsigned color1 = w1->u.pin.config & HDA_CONFIG_COLOR_MASK;
		unsigned color2 = w1->u.pin.config & HDA_CONFIG_COLOR_MASK;
		if (!comp_input) {
			if (device1 == HDA_CONFIG_DEFAULT_DEVICE_LINE_OUT && device2 != HDA_CONFIG_DEFAULT_DEVICE_LINE_OUT) return -1;
			if (device1 != HDA_CONFIG_DEFAULT_DEVICE_LINE_OUT && device2 == HDA_CONFIG_DEFAULT_DEVICE_LINE_OUT) return 1;
		} else {
			if (device1 == HDA_CONFIG_DEFAULT_DEVICE_LINE_IN && device2 != HDA_CONFIG_DEFAULT_DEVICE_LINE_IN) return -1;
			if (device1 != HDA_CONFIG_DEFAULT_DEVICE_LINE_IN && device2 == HDA_CONFIG_DEFAULT_DEVICE_LINE_IN) return 1;
		}
		if (location1 == HDA_CONFIG_LOCATION_HI_PRIMARY	&& location2 != HDA_CONFIG_LOCATION_HI_PRIMARY) return -1;
		if (location1 != HDA_CONFIG_LOCATION_HI_PRIMARY	&& location2 == HDA_CONFIG_LOCATION_HI_PRIMARY) return 1;
		if (!comp_input) {
			if (color1 == HDA_CONFIG_COLOR_GREEN && color2 != HDA_CONFIG_COLOR_GREEN) return -1;
			if (color1 != HDA_CONFIG_COLOR_GREEN && color2 == HDA_CONFIG_COLOR_GREEN) return -1;
		} else {
			if (color1 == HDA_CONFIG_COLOR_BLUE && color2 != HDA_CONFIG_COLOR_BLUE) return -1;
			if (color1 != HDA_CONFIG_COLOR_BLUE && color2 == HDA_CONFIG_COLOR_BLUE) return -1;
		}
	} else {
		unsigned sz_rate1 = w1->u.io.sz_rate;
		unsigned sz_rate2 = w2->u.io.sz_rate;
		static __const__ unsigned cap_order[] = {
			HDA_SAMPLE_SZ_RATE_CAP_44100,
			HDA_SAMPLE_SZ_RATE_CAP_48000,
			HDA_SAMPLE_SZ_RATE_CAP_22050,
			HDA_SAMPLE_SZ_RATE_CAP_384000,
			HDA_SAMPLE_SZ_RATE_CAP_192000,
			HDA_SAMPLE_SZ_RATE_CAP_176400,
			HDA_SAMPLE_SZ_RATE_CAP_96000,
			HDA_SAMPLE_SZ_RATE_CAP_88200,
			HDA_SAMPLE_SZ_RATE_CAP_11025,
			HDA_SAMPLE_SZ_RATE_CAP_32000,
			HDA_SAMPLE_SZ_RATE_CAP_16000,
			HDA_SAMPLE_SZ_RATE_CAP_8000,
			HDA_SAMPLE_SZ_RATE_CAP_32B,
			HDA_SAMPLE_SZ_RATE_CAP_24B,
			HDA_SAMPLE_SZ_RATE_CAP_20B,
			HDA_SAMPLE_SZ_RATE_CAP_8B,
			0,
		};
		__const__ unsigned *c;
		for (c = cap_order; *c; c++) {
			if (sz_rate1 & *c && !(sz_rate2 & *c)) return -1;
			if (!(sz_rate1 & *c) && sz_rate2 & *c) return 1;
		}
	}
	return w1 - w2;
}

static unsigned SORT_WIDGETS(struct widget **w)
{
	unsigned i;
	for (i = 0; i < max_widgets; i++) w[i] = &widgets[i];
	qsort(w, max_widgets, sizeof(struct widget *), COMPARE_WIDGET);
	for (i = 0; i < max_widgets; i++) if (!CAPABLE(w[i])) break;
	return i;
}

static void FLOOD_FILL(unsigned from, int rev)
{
	unsigned i;
	int k, new;
	for (i = 0; i < max_widgets; i++) widgets[i].ff_dist = -1;
	widgets[from].ff_dist = 0;
	widgets[from].origin = from;
	k = 0;
	do {
		new = 0;
		for (i = 0; i < max_widgets; i++) if (widgets[i].ff_dist == k) {
			unsigned n, *l;
			if (!rev) n = widgets[i].n_conn_list, l = widgets[i].conn_list;
			else n = widgets[i].n_rev_conn_list, l = widgets[i].rev_conn_list;
			if (k && widgets[i].type != HDA_AUD_WIDGET_CAP_TYPE_AUDIO_MIXER	&& widgets[i].type != HDA_AUD_WIDGET_CAP_TYPE_AUDIO_SELECTOR) continue;
			for (; n; l++, n--) {
				if (widgets[*l].ff_dist == -1 && !widgets[*l].used) {
					widgets[*l].ff_dist = k + 1;
					widgets[*l].origin = i;
					new = 1;
				}
			}
		}
		k++;
	} while (new);
}

static unsigned out_amp, out_amp_idx, out_amp_out;
static unsigned in_amp, in_amp_idx, in_amp_out;

static void HDA_MUTE_AMP(unsigned node, unsigned idx, int out)
{
	unsigned cmd;
	if (!out) {
		if (!(widgets[node].cap & HDA_AUD_WIDGET_CAP_IN_AMP_PRESENT)) return;
		cmd = HDA_SET_AMPLIFIER_GAIN_SET_INPUT;
		if (widgets[node].in_amp_cap & HDA_AMP_CAP_MUTE) cmd = HDA_SET_AMPLIFIER_GAIN_MUTE;
	} else {
		if (!(widgets[node].cap & HDA_AUD_WIDGET_CAP_OUT_AMP_PRESENT)) return;
		cmd = HDA_SET_AMPLIFIER_GAIN_SET_OUTPUT;
		if (widgets[node].out_amp_cap & HDA_AMP_CAP_MUTE) cmd = HDA_SET_AMPLIFIER_GAIN_MUTE;
	}
	cmd |= (idx & (HDA_SET_AMPLIFIER_GAIN_INDEX_MASK >> HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT)) << HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT;
	cmd |= HDA_SET_AMPLIFIER_GAIN_SET_RIGHT | HDA_SET_AMPLIFIER_GAIN_SET_LEFT;
	HDA_CODEC_REQUEST_SYNC(node, HDA_SET_AMPLIFIER_GAIN | cmd);
}

static void HDA_INIT_AMP(unsigned node, unsigned idx, int out, int rev)
{
	unsigned cmd;
	__u32 amp_cap;
	if (!out) {
		if (!(widgets[node].cap & HDA_AUD_WIDGET_CAP_IN_AMP_PRESENT)) return;
		cmd = HDA_SET_AMPLIFIER_GAIN_SET_INPUT;
		amp_cap = widgets[node].in_amp_cap;
	} else {
		if (!(widgets[node].cap & HDA_AUD_WIDGET_CAP_OUT_AMP_PRESENT)) return;
		cmd = HDA_SET_AMPLIFIER_GAIN_SET_OUTPUT;
		amp_cap = widgets[node].out_amp_cap;
	}
	cmd |= (amp_cap & HDA_AMP_CAP_OFFSET_MASK) >> HDA_AMP_CAP_OFFSET_SHIFT;
	cmd |= (idx & (HDA_SET_AMPLIFIER_GAIN_INDEX_MASK >> HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT)) << HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT;
	cmd |= HDA_SET_AMPLIFIER_GAIN_SET_RIGHT | HDA_SET_AMPLIFIER_GAIN_SET_LEFT;
	HDA_CODEC_REQUEST_SYNC(node, HDA_SET_AMPLIFIER_GAIN | cmd);
	if (amp_cap & HDA_AMP_CAP_STEPS_MASK) {
		if (!rev) {
			if (!out_amp) {
				out_amp = node;
				out_amp_idx = idx;
				out_amp_out = out;
			}
		} else {
			in_amp = node;
			in_amp_idx = idx;
			in_amp_out = out;
		}
	}
}

static void HDA_CONNECT(unsigned from, unsigned to, int rev)
{
	unsigned idx;
	unsigned i;
	if (!rev) HDA_INIT_AMP(from, 0, 1, rev);
	for (i = 0; i < widgets[to].n_conn_list; i++) if (widgets[to].conn_list[i] == from) {
		idx = i;
		goto idx_ok;
	}
	KERNEL$SUICIDE("HDA_CONNECT: %s: NO PATH FROM %u TO %u", dev_name, from, to);
	idx_ok:
	if (widgets[to].type == HDA_AUD_WIDGET_CAP_TYPE_AUDIO_MIXER) {
		for (i = 0; i < widgets[to].n_conn_list; i++) if (i != idx) {
			HDA_MUTE_AMP(to, i, 0);
			if (!(widgets[to].cap & HDA_AUD_WIDGET_CAP_IN_AMP_PRESENT) || !(widgets[to].in_amp_cap & HDA_AMP_CAP_MUTE)) {
				if (widgets[to].conn_list[idx] != from)
					HDA_MUTE_AMP(widgets[to].conn_list[idx], 0, 1);
			}
		}
	} else {
		if (widgets[to].n_conn_list > 1) HDA_CODEC_REQUEST_SYNC(to, HDA_SET_CONNECTION_SELECT | from);
	}
	HDA_INIT_AMP(to, idx, 0, rev);
	if (rev) HDA_INIT_AMP(from, 0, 1, rev);
}

static void HDA_SETUP_PATH(unsigned from, unsigned to, int rev)
{
	unsigned i;
	if (!rev) {
		HDA_CODEC_REQUEST_SYNC(to, HDA_SET_PIN_WIDGET_CONTROL | HDA_SET_PIN_WIDGET_CONTROL_OUT);
		HDA_CODEC_REQUEST_SYNC(from, HDA_SET_CHANNEL_STREAM | (PLAYBACK_STREAM_ID << HDA_SET_STREAM_SHIFT) | (0 << HDA_SET_CHANNEL_SHIFT));
		i = from;
	} else {
		HDA_CODEC_REQUEST_SYNC(from, HDA_SET_PIN_WIDGET_CONTROL | HDA_SET_PIN_WIDGET_CONTROL_IN);
		HDA_CODEC_REQUEST_SYNC(to, HDA_SET_CHANNEL_STREAM | (RECORD_STREAM_ID << HDA_SET_STREAM_SHIFT) | (0 << HDA_SET_CHANNEL_SHIFT));
		i = to;
	}
	while (1) {
		widgets[i].used = 1;
		if (!rev) {
			if (i == to) break;
		} else {
			if (i == from) break;
		}
		if (i == widgets[i].origin) {
			KERNEL$SUICIDE("HDA_SETUP_PATH: %s: WIDGET LOOP", dev_name);
		}
		if (!rev) {
			HDA_CONNECT(i, widgets[i].origin, rev);
		} else {
			HDA_CONNECT(widgets[i].origin, i, rev);
		}
		i = widgets[i].origin;
	}
	if (!rev) {
		HDA_INIT_AMP(to, 0, 1, rev);
	} else {
		HDA_INIT_AMP(from, 0, 0, rev);
	}
}

static void HDA_RESTRICT_RATES(unsigned ad, int rec)
{
	int i;
	void *ratebits = !rec ? PLAYBACK_RATES : RECORD_RATES;
	__u32 cap = widgets[ad].u.io.sz_rate;
	for (i = 0; RATELIST[i]; i++) {
		__BS(ratebits, i);
		switch (RATELIST[i]) {
		case 8000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_8000) continue;
			break;
		case 11025:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_11025) continue;
			break;
		case 16000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_16000) continue;
			break;
		case 22050:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_22050) continue;
			break;
		case 32000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_32000) continue;
			break;
		case 44100:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_44100) continue;
	/* dirty hack to allow 44100 on codecs that know only 48000 */
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_48000) continue;
			break;
		case 48000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_48000) continue;
			break;
		case 88200:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_88200) continue;
			break;
		case 96000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_96000) continue;
			break;
		case 176400:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_176400) continue;
			break;
		case 192000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_192000) continue;
			break;
		case 384000:
			if (cap & HDA_SAMPLE_SZ_RATE_CAP_384000) continue;
			break;
		}
		__BR(ratebits, i);
	}
	/*for (i = 0; RATELIST[i]; i++) {
		if (__BT(ratebits, i)) __debug_printf("rate(%d): %d\n", rec, RATELIST[i]);
	}*/
}

void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
	int best, want_rate;
	int i;
	open_flags &= _O_KRNL_READ | _O_KRNL_WRITE;
	best = -1;
	want_rate = RATELIST[FORMAT_RATE(h)];
	for (i = 0; RATELIST[i]; i++) {
		if (__likely(RATELIST[i] < 0)) continue;
		if (best < 0 || abs(RATELIST[i] - want_rate) < abs(RATELIST[best] - want_rate)) {
			switch (open_flags) {
				case 0:
					if (__BT(PLAYBACK_RATES, i) | __BT(RECORD_RATES, i)) break;
					continue;
				case _O_KRNL_READ:
					if (__BT(RECORD_RATES, i)) break;
					continue;
				case _O_KRNL_WRITE:
					if (__BT(PLAYBACK_RATES, i)) break;
					continue;
				case _O_KRNL_READ | _O_KRNL_WRITE:
					if (__BT(PLAYBACK_RATES, i) & __BT(RECORD_RATES, i)) break;
					continue;
				default:
					KERNEL$SUICIDE("SND_FIXUP_PARAMS: BAD OPEN FLAGS: 0%o", open_flags);
			}
			best = i;
		}
	}
	if (best < 0) best = 0;
	FORMAT_SETRATE(h, best);
}

int HDA_CODEC_INIT(unsigned codec_id)
{
	struct widget **w;
	unsigned s, n, i, j, o, p;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("HDA_CODEC_INIT AT SPL %08X", KERNEL$SPL);
	async_request_buffer_size = 0;
	out_amp = in_amp = 0;
	codec = codec_id;
	codec_error = 0;
	vendor_id = HDA_CODEC_REQUEST_SYNC(0, HDA_GET_PARAMETER | HDA_VID);
	if (codec_error) return -EIO;
	revision_id = HDA_CODEC_REQUEST_SYNC(0, HDA_GET_PARAMETER | HDA_RID);
	if (codec_error) return -EIO;
	HDA_GET_SUB_NODES(0, &s, &n);
	if (codec_error) return -EIO;
	for (; n; s++, n--) {
		__u32 func_type = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_FUNC_GRP_TYPE);
		if (codec_error) return -EIO;
		if ((func_type & HDA_FUNC_GRP_TYPE_MASK) == HDA_FUNC_GRP_TYPE_AUDIO) goto found_fg;

	}
	return -ENODEV;
	found_fg:
	func_node = s;
	HDA_GET_SUB_NODES(func_node, &s, &n);
	if (codec_error) return -EIO;
	max_widgets = s + n;
	widgets = __sync_calloc(max_widgets, sizeof(struct widget));
	if (!widgets) return -errno;
	for (; n; s++, n--) {
		__u32 cap = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_AUD_WIDGET_CAP);
		if (codec_error) {
			codec_error = 0;
			continue;
		}
		if (s >= max_widgets) continue;
		widgets[s].present = 1;
		widgets[s].type = cap & HDA_AUD_WIDGET_CAP_TYPE_MASK;
		widgets[s].cap = cap;
		widgets[s].delay = (cap & HDA_AUD_WIDGET_CAP_DELAY_MASK) >> HDA_AUD_WIDGET_CAP_DELAY_SHIFT;
#ifdef HDA_DUMP
		_printf("%02d: %08x", s, cap);
#endif
		if (widgets[s].type == HDA_AUD_WIDGET_CAP_TYPE_AUDIO_OUTPUT ||
		    widgets[s].type == HDA_AUD_WIDGET_CAP_TYPE_AUDIO_INPUT) {
			__u32 sz_rate, fmt;
			sz_rate = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_SAMPLE_SZ_RATE_CAP);
			if (!sz_rate) sz_rate = HDA_CODEC_REQUEST_SYNC(func_node, HDA_GET_PARAMETER | HDA_SAMPLE_SZ_RATE_CAP);
			fmt = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_STREAM_FMT);
			if (!fmt) fmt = HDA_CODEC_REQUEST_SYNC(func_node, HDA_GET_PARAMETER | HDA_STREAM_FMT);
			widgets[s].u.io.sz_rate = sz_rate;
			widgets[s].u.io.fmt = fmt;
#ifdef HDA_DUMP
			_printf("  output cap: %08x  stream: %08x", sz_rate, fmt);
#endif
		}
		if (widgets[s].cap & HDA_AUD_WIDGET_CAP_IN_AMP_PRESENT) {
			__u32 amp_cap;
			if (widgets[s].cap & HDA_AUD_WIDGET_CAP_AMP_PARAM_OVERRIDE) {
				amp_cap = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_IN_AMP_CAP);
			} else {
				amp_cap = HDA_CODEC_REQUEST_SYNC(func_node, HDA_GET_PARAMETER | HDA_IN_AMP_CAP);
			}
			widgets[s].in_amp_cap = amp_cap;
#ifdef HDA_DUMP
			_printf("  in amp: %08x", amp_cap);
#endif
		}
		if (widgets[s].cap & HDA_AUD_WIDGET_CAP_OUT_AMP_PRESENT) {
			__u32 amp_cap;
			if (widgets[s].cap & HDA_AUD_WIDGET_CAP_AMP_PARAM_OVERRIDE) {
				amp_cap = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_OUT_AMP_CAP);
			} else {
				amp_cap = HDA_CODEC_REQUEST_SYNC(func_node, HDA_GET_PARAMETER | HDA_OUT_AMP_CAP);
			}
			widgets[s].out_amp_cap = amp_cap;
#ifdef HDA_DUMP
			_printf("  out amp: %08x", amp_cap);
#endif
		}
		if (widgets[s].type == HDA_AUD_WIDGET_CAP_TYPE_PIN_COMPLEX) {
			widgets[s].u.pin.cap = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_PIN_CAP);
			widgets[s].u.pin.config = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_CONFIG_DEFAULT_BYTES);
#ifdef HDA_DUMP
			_printf("  pin cap: %08x  cfg: %08x", widgets[s].u.pin.cap, widgets[s].u.pin.config);
#endif
		}
		if (cap & HDA_AUD_WIDGET_CAP_CONN_LIST) {
			__u32 conn_ll = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_PARAMETER | HDA_CONNECTION_LIST_LENGTH);
			int lng = conn_ll & HDA_CONNECTION_LIST_LENGTH_LONG;
			int i;
#ifdef HDA_DUMP
			_printf("  con ll: ");
#endif
			conn_ll &= HDA_CONNECTION_LIST_LENGTH_MASK;
			for (i = 0; i < conn_ll; i += (lng ? 2 : 4)) {
				__u32 conn = HDA_CODEC_REQUEST_SYNC(s, HDA_GET_CONNECTION_LIST_ENTRY | i);
				if (!conn) continue;
				if (!lng) {
					HDA_CONNECTION_LIST_ADD(s, conn & 0x7f, conn & 0x80);
					if (i + 1 < conn_ll) HDA_CONNECTION_LIST_ADD(s, (conn >> 8) & 0x7f, conn & 0x8000);
					if (i + 2 < conn_ll) HDA_CONNECTION_LIST_ADD(s, (conn >> 16) & 0x7f, conn & 0x800000);
					if (i + 3 < conn_ll) HDA_CONNECTION_LIST_ADD(s, (conn >> 24) & 0x7f, conn & 0x80000000);
				} else {
					HDA_CONNECTION_LIST_ADD(s, conn & 0x7fff, conn & 0x8000);
					if (i + 1 < conn_ll) HDA_CONNECTION_LIST_ADD(s, (conn >> 16) & 0x7fff, conn & 0x80000000);
				}
			}
		}
#ifdef HDA_DUMP
		_printf("\n");
#endif
	}
	w = __sync_malloc(sizeof(struct widget *) * max_widgets);
	if (!w) {
		HDA_CODEC_DESTROY();
		return -errno;
	}
	comp_ad = 0;
	comp_input = 0;
	n = SORT_WIDGETS(w);
	comp_ad = 1;
	for (i = 0; i < n; i++) {
		o = w[i] - widgets;
		FLOOD_FILL(o, 0);
		for (j = 0; j < max_widgets; j++)
			if (CAPABLE(&widgets[j])) goto find_out_path;
	}
	__slow_free(w);
	HDA_CODEC_DESTROY();
	return -ENXIO;

	find_out_path:
	n = SORT_WIDGETS(w);
	p = *w - widgets;
	HDA_RESTRICT_RATES(p, 0);
	out_da_node = p;
	HDA_SETUP_PATH(p, o, 0);

	comp_ad = 0;
	comp_input = 1;
	n = SORT_WIDGETS(w);
	comp_ad = 1;
	for (i = 0; i < n; i++) {
		o = w[i] - widgets;
		FLOOD_FILL(o, 1);
		for (j = 0; j < max_widgets; j++)
			if (CAPABLE(&widgets[j])) goto find_in_path;
	}
	__slow_free(w);
	HDA_CODEC_DESTROY();
	return -ENXIO;

	find_in_path:
	n = SORT_WIDGETS(w);
	p = *w - widgets;
	HDA_RESTRICT_RATES(p, 1);
	in_ad_node = p;
	HDA_SETUP_PATH(o, p, 1);
	free(w);

	OSS_MIXER_DEVMASK = 0;
	if (out_amp) {
		OSS_MIXER_DEVMASK |= SOUND_MASK_PCM;
		SND_MIXER_WRITE(SOUND_MIXER_PCM, (100 << 8) + 100);
	}
	if (in_amp) {
		OSS_MIXER_DEVMASK |= SOUND_MASK_LINE;
		SND_MIXER_WRITE(SOUND_MIXER_LINE, (100 << 8) + 100);
	}
	OSS_MIXER_STEREODEVS = OSS_MIXER_DEVMASK;
	OSS_MIXER_RECMASK = SOUND_MASK_LINE;
	OSS_MIXER_OUTMASK = 0;

	return 0;
}

void HDA_CODEC_DESTROY(void)
{
	unsigned i;
	for (i = 0; i < max_widgets; i++) {
		if (widgets[i].conn_list) free(widgets[i].conn_list);
		if (widgets[i].rev_conn_list) free(widgets[i].rev_conn_list);
	}
	free(widgets);
}

void HDA_CODEC_SET_FORMAT(int record, unsigned fmt)
{
	unsigned node = __likely(!record) ? out_da_node : in_ad_node;
	/* dirty hack to allow 44100 on codecs that know only 48000 */
	if (fmt >> 8 == 64 && !(widgets[node].u.io.sz_rate & HDA_SAMPLE_SZ_RATE_CAP_44100)) fmt &= ~(64 << 8);
	HDA_CODEC_REQUEST_ASYNC(node, HDA_SET_STREAM_FORMAT | fmt, HDA_PARAM_LONG_MASK);
}

unsigned OSS_MIXER_DEVMASK;
unsigned OSS_MIXER_STEREODEVS;
unsigned OSS_MIXER_RECMASK;
unsigned OSS_MIXER_OUTMASK;

unsigned vol_pcm;
unsigned vol_line;

long SND_MIXER_READ(unsigned mix)
{
	if (__likely(mix == SOUND_MIXER_PCM)) {
		return vol_pcm;
	} else if (__likely(mix == SOUND_MIXER_LINE)) {
		return vol_line;
	} else {
		return -EINVAL;
	}
}

void SND_MIXER_WRITE(unsigned mix, unsigned long val)
{
	unsigned left, right;
	unsigned amp, idx, out;
	unsigned cap, steps;
	unsigned cmd_left, cmd_right;
	if (__likely(mix == SOUND_MIXER_PCM)) {
		amp = out_amp;
		idx = out_amp_idx;
		out = out_amp_out;
	} else if (__likely(mix == SOUND_MIXER_LINE)) {
		amp = in_amp;
		idx = in_amp_idx;
		out = in_amp_out;
	} else {
		return;
	}
	if (__unlikely(!amp)) return;
	if (!out) cap = widgets[amp].in_amp_cap;
	else cap = widgets[amp].out_amp_cap;

	left = val & 0xff;
	right = (val >> 8) & 0xff;
	if (__unlikely(left > 100)) left = 100;
	if (__unlikely(right > 100)) right = 100;

	steps = !!(cap & HDA_AMP_CAP_MUTE) + ((cap & HDA_AMP_CAP_STEPS_MASK) >> HDA_AMP_CAP_STEPS_SHIFT) + 1;
	/*__debug_printf("cap: %x\n", cap);
	__debug_printf("steps: %d\n", steps);*/
	left = ((left * (steps - 1) + 49) / 100);
	right = ((right * (steps - 1) + 49) / 100);
	if (cap & HDA_AMP_CAP_MUTE) {
		if (!left) cmd_left = HDA_SET_AMPLIFIER_GAIN_MUTE;
		else cmd_left = (left - 1) & HDA_SET_AMPLIFIER_GAIN_MASK;
		if (!right) cmd_right = HDA_SET_AMPLIFIER_GAIN_MUTE;
		else cmd_right = (right - 1) & HDA_SET_AMPLIFIER_GAIN_MASK;
	} else {
		cmd_left = left & HDA_SET_AMPLIFIER_GAIN_MASK;
		cmd_right = right & HDA_SET_AMPLIFIER_GAIN_MASK;
	}
	cmd_left |= (idx << HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT) | HDA_SET_AMPLIFIER_GAIN_SET_LEFT;
	cmd_right |= (idx << HDA_SET_AMPLIFIER_GAIN_INDEX_SHIFT) | HDA_SET_AMPLIFIER_GAIN_SET_RIGHT;
	if (!out) {
		cmd_left |= HDA_SET_AMPLIFIER_GAIN_SET_INPUT;
		cmd_right |= HDA_SET_AMPLIFIER_GAIN_SET_INPUT;
	} else {
		cmd_left |= HDA_SET_AMPLIFIER_GAIN_SET_OUTPUT;
		cmd_right |= HDA_SET_AMPLIFIER_GAIN_SET_OUTPUT;
	}
	/*__debug_printf("%x %x\n", HDA_SET_AMPLIFIER_GAIN | cmd_left, HDA_SET_AMPLIFIER_GAIN | cmd_right);*/
	HDA_CODEC_REQUEST_ASYNC(amp, HDA_SET_AMPLIFIER_GAIN | cmd_left, HDA_PARAM_AMP_GAIN_MASK);
	HDA_CODEC_REQUEST_ASYNC(amp, HDA_SET_AMPLIFIER_GAIN | cmd_right, HDA_PARAM_AMP_GAIN_MASK);
	/*HDA_CODEC_REQUEST_ASYNC(amp, HDA_GET_AMPLIFIER_GAIN | HDA_GET_AMPLIFIER_GAIN_LEFT | HDA_GET_AMPLIFIER_GAIN_OUTPUT, HDA_PARAM_AMP_GAIN_MASK);*/
	left = ((left * 100) / (steps - 1));
	right = ((right * 100) / (steps - 1));
	val = left + (right << 8);
	if (__likely(mix == SOUND_MIXER_PCM)) {
		vol_pcm = val;
	} else if (__likely(mix == SOUND_MIXER_LINE)) {
		vol_line = val;
	}
}

long SND_MIXER_READ_RECSRC(void)
{
	return SOUND_MASK_LINE;
}

void SND_MIXER_WRITE_RECSRC(unsigned long mask)
{
}

long SND_MIXER_READ_OUTSRC(void)
{
	return 0;
}

void SND_MIXER_WRITE_OUTSRC(unsigned long mask)
{
}

