#include "OSS.H"

#include <ERRNO.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <SYS/TYPES.H>
#include <SYS/PARAM.H>
#include <ENDIAN.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <VALUES.H>

#include <SYS/SOUNDCARD.H>

#define O_LOOP_FLAG	_O_AUX1

/*
 * ^RATE=12345
 * ^CHANNELS=1, 2
 * ^FORMAT=0 - 14
 * ^FRAGSIZE=4 - 16
 * ^FRAGS=2 - 4096
 */

static void *SET_RATE(HANDLE *h, unsigned rate, int file_flags)
{
#ifdef OSS_FIXED_RATES
	int i, j = -1, mindiff = MAXINT;
	if (__unlikely(rate > FORMAT_MAXRATE)) rate = FORMAT_MAXRATE;
	for (i = 0; RATELIST[i]; i++) if (__likely(RATELIST[i] > 0) && __likely(RATELIST[i] <= FORMAT_MAXRATE)) {
		int diff = abs(rate - RATELIST[i]);
		if (__likely(diff < mindiff)) j = i, mindiff = diff;
	}
	if (__unlikely(j == -1)) return __ERR_PTR(-EINVAL);
	FORMAT_SETRATE(h, j);
#else
	if (__unlikely(rate < MIN_RATE)) rate = MIN_RATE;
	if (__unlikely(rate > MIN(MAX_RATE,FORMAT_MAXRATE))) rate = MIN(MAX_RATE,FORMAT_MAXRATE);
	FORMAT_SETRATE(h, rate);
#endif
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_CHANNELS(HANDLE *h, unsigned ch, int file_flags)
{
	if (__unlikely(ch < MIN_CHANNELS)) ch = MIN_CHANNELS;
	if (__unlikely(ch > MIN(MAX_CHANNELS,FORMAT_MAXCHANNELS))) ch = MIN(MAX_CHANNELS,FORMAT_MAXCHANNELS);
	FORMAT_SETCHANNELS(h, ch);
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_FORMAT(HANDLE *h, unsigned f, int file_flags)
{
	if (__unlikely(f > MIN(_FMT_MAX,FORMAT_MAXFORMAT)) || __unlikely(!FORMATS[f])) {
		if (f == _FMT_MU_LAW || f == _FMT_A_LAW || f == _FMT_S8 || f == _FMT_IMA_ADPCM) f = _FMT_U8;
		else f = _FMT_S16_LE;
		if (__unlikely(!FORMATS[f])) f = DEFAULT_FORMAT;
	}
	FORMAT_SETFORMAT(h, f);
	SND_FIXUP_PARAMS(h, file_flags);
	return NULL;
}

static void *SET_FRAGSIZE(HANDLE *h, unsigned fragsize_bits)
{
	if (__unlikely(fragsize_bits >= sizeof(int) * 8 - 1)) fragsize_bits = sizeof(int) * 8 - 2;
	if (__unlikely((1 << fragsize_bits) < MIN_FRAGSIZE)) fragsize_bits = __BSR(MIN_FRAGSIZE);
	if (__unlikely(fragsize_bits > PAGESIZE_BITS)) fragsize_bits = PAGESIZE_BITS;
	if (N_PAGES == 1 && __unlikely(fragsize_bits > PAGESIZE_BITS - 1)) fragsize_bits = PAGESIZE_BITS - 1;
	if (__unlikely(fragsize_bits > FORMAT_MAXFRAGSIZE)) fragsize_bits = FORMAT_MAXFRAGSIZE;
	FORMAT_SETFRAGSIZE(h, fragsize_bits);
	return NULL;
}

static void *SET_FRAGS(HANDLE *h, unsigned frags)
{
	unsigned fragsize_bits, frags1;
	if (__unlikely(!FORMAT_FRAGSIZE(h))) SET_FRAGSIZE(h, PAGESIZE_BITS);
	fragsize_bits = FORMAT_FRAGSIZE(h);
	if (__unlikely(frags < MIN_FRAGS)) frags = MIN_FRAGS;
	if (__unlikely(frags > MIN(MAX_FRAGS,FORMAT_MAXFRAGS))) frags = MIN(MAX_FRAGS,FORMAT_MAXFRAGS);
	if (__unlikely(frags != frags << fragsize_bits >> fragsize_bits)) frags = MAXINT >> fragsize_bits;
	frags1 = frags;
	set_frags_again:
	if (__unlikely(frags << fragsize_bits > N_PAGES << PAGESIZE_BITS))
		frags = N_PAGES << (PAGESIZE_BITS - fragsize_bits);
	if (__unlikely(frags < MIN_FRAGS) && (1 << fragsize_bits) > MIN_FRAGSIZE) {
		frags = frags1;
		fragsize_bits--;
		FORMAT_SETFRAGSIZE(h, fragsize_bits);
		goto set_frags_again;
	}
#ifdef NEED_VALIDATE_FRAGS
	VALIDATE_FRAGS(&frags);
#endif
	FORMAT_SETFRAGS(h, frags);
	return NULL;
}

static void *SET_SUBDIVIDE(HANDLE *h, unsigned s)
{
	if (__unlikely(s > FORMAT_MAXSUBDIVIDE)) s = FORMAT_MAXSUBDIVIDE;
	FORMAT_SETSUBDIVIDE(h, s);
	return NULL;
}

void *OSS_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	char *s;
	unsigned long n;
	vspace_unmap_t *unmap;
	void *r;
	FFILE *f;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_LOOKUP AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_VSPACE);
	f = KERNEL$MAP_FILE_STRUCT(h, NULL, &unmap);
	LOWER_SPL(SPL_OSS);
	if (__unlikely(!f))
		return (void *)1;
	s = strchr(str, '=');
	if (__unlikely(!s)) {
		if (__likely(!_strcasecmp(str, "^NONBLOCK"))) {
			FORMAT_SETNONBLOCK(h, 1);
			r = NULL;
			goto ret;
		}
		r = __ERR_PTR(-EBADMOD);
		goto ret;
	}
	if (__unlikely(__get_number(s + 1, s + strlen(s), 0, (long *)&n)) || __unlikely(n > MAXINT)) {
		if (!__strcasexcmp("^FORMAT", str, s)) {
			if (!_strcasecmp(s + 1, "MU-LAW")) n = _FMT_MU_LAW;
			else if (!_strcasecmp(s + 1, "A-LAW")) n = _FMT_A_LAW;
			else if (!_strcasecmp(s + 1, "S8")) n = _FMT_S8;
			else if (!_strcasecmp(s + 1, "U8")) n = _FMT_U8;
			else if (!_strcasecmp(s + 1, "S16")) n = _FMT_S16_LE;
			else if (!_strcasecmp(s + 1, "S16LE")) n = _FMT_S16_LE;
			else if (!_strcasecmp(s + 1, "S16BE")) n = _FMT_S16_BE;
			else if (!_strcasecmp(s + 1, "U16LE")) n = _FMT_U16_LE;
			else if (!_strcasecmp(s + 1, "U16BE")) n = _FMT_U16_BE;
			else if (!_strcasecmp(s + 1, "S32LE")) n = _FMT_S32_LE;
			else if (!_strcasecmp(s + 1, "S32BE")) n = _FMT_S32_BE;
			else if (!_strcasecmp(s + 1, "U32LE")) n = _FMT_U32_LE;
			else if (!_strcasecmp(s + 1, "U32BE")) n = _FMT_U32_BE;
			else if (!_strcasecmp(s + 1, "ADPCM")) n = _FMT_IMA_ADPCM;
			else if (!_strcasecmp(s + 1, "MPEG")) n = _FMT_MPEG;
			else if (!_strcasecmp(s + 1, "AC3")) n = _FMT_AC3;
			else {
				r = __ERR_PTR(-EINVAL);
				goto ret;
			}
			SET_FORMAT(h, n, f->flags);
			if (__unlikely(FORMAT_FORMAT(h) != n)) {
				r = __ERR_PTR(-EPROTO);
				goto ret;
			}
			r = NULL;
			goto ret;
		}
		r = __ERR_PTR(-EINVAL);
		goto ret;
	}
	if (!__strcasexcmp("^RATE", str, s)) {
		r = SET_RATE(h, n, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^CHANNELS", str, s)) {
		r = SET_CHANNELS(h, __likely(n != 0) ? n - 1 : 0, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^FORMAT", str, s)) {
		r = SET_FORMAT(h, n, f->flags);
		goto ret;
	}
	if (!__strcasexcmp("^FRAGSIZE", str, s)) {
		r = SET_FRAGSIZE(h, n);
		goto ret;
	}
	if (!__strcasexcmp("^FRAGS", str, s)) {
		r = SET_FRAGS(h, n);
		goto ret;
	}
	if (!__strcasexcmp("^SUBDIVIDE", str, s)) {
		r = SET_SUBDIVIDE(h, n);
		goto ret;
	}
	r = __ERR_PTR(-EBADMOD);
	goto ret;

	ret:
	RAISE_SPL(SPL_VSPACE);
	unmap(f);
	LOWER_SPL(SPL_OSS);
	return r;
}

void OSS_LOOKUP_IO(HANDLE *h, char *str, IORQ *rq, int open_flags)
{
	vspace_unmap_t *unmap;
	FFILE *f;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_LOOKUP_IO AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_VSPACE);
	f = KERNEL$MAP_FILE_STRUCT(h, rq, &unmap);
	LOWER_SPL(SPL_OSS);
	if (__likely(!f)) return;
	RAISE_SPL(SPL_VSPACE);
	unmap(f);
	LOWER_SPL(SPL_OSS);
	CALL_IORQ_EXPR(rq, (IO_STUB *)rq->tmp1);
}

void OSS_OPEN(HANDLE *h, int open_flags)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_OPEN AT SPL %08X", KERNEL$SPL);
	if (__likely(open_flags & _O_KRNL_WRITE)) {
		RAISE_SPL(SPL_SND);
		oss_playback_stat_underruns = 0;
		if (__unlikely(OSS_FLAGS & (OSS_BLOCK_PLAYBACK | OSS_LOOP_PLAYBACK))) {
			OSS_FLAGS &= ~(OSS_BLOCK_PLAYBACK | OSS_LOOP_PLAYBACK);
			OSS_STOP(0);
		}
		LOWER_SPL(SPL_OSS);
	}
	if (__unlikely(open_flags & _O_KRNL_READ)) {
		RAISE_SPL(SPL_SND);
		oss_record_stat_overruns = 0;
		OSS_FLAGS &= ~OSS_BLOCK_RECORD;
		LOWER_SPL(SPL_OSS);
	}
}

int OSS_CLOSE(HANDLE *h, IORQ *RQ)
{
	FFILE *file;
	int r;
	vspace_unmap_t *funmap;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_OSS)))
		KERNEL$SUICIDE("OSS_CLOSE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_VSPACE);
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	LOWER_SPL(SPL_OSS);
	if (__unlikely(!file)) return 1;
	r = 0;
	if (__unlikely(!(file->flags & _O_KRNL_WRITE))) goto retz;
	if (__unlikely(file->l_refcount > 1) && __likely(RQ->tmp1 != (unsigned long)KERNEL$WAKE_IOCTL)) goto retz;
	RAISE_SPL(SPL_SND);
	if (__unlikely(OSS_FLAGS & (OSS_LOOP_PLAYBACK | OSS_BLOCK_PLAYBACK))) {
		OSS_STOP(0);
		retzs:
		LOWER_SPL(SPL_OSS);
		goto retz;
	}
	if (__unlikely((off_t)(file->pos - (oss_playback_file_pos - (1 << oss_playback_fragsize_bits))) < 0)) {
		goto retzs;
	}
	if (__likely((off_t)(file->pos - (oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits))) > 0)) {
		goto retzs;
	}
	if (__unlikely(!(OSS_FLAGS & OSS_PLAYBACK))) {
		TRIGGER_PLAY();
		if (__unlikely(!(OSS_FLAGS & OSS_PLAYBACK))) goto retzs;
	}
	WQ_WAIT_F(&OSS_PLAYBACK_WAIT, RQ);
	LOWER_SPL(SPL_OSS);
	r = 1;
	retz:
	RAISE_SPL(SPL_VSPACE);
	funmap(file);
	LOWER_SPL(SPL_OSS);
	return r;
}

void OSS_SET_ROOT(HANDLE *h, void *f)
{
	h->op = &SOUND_OPERATIONS;
	h->flags = 0;
	h->flags2 = 0;
	SET_RATE(h, 8000, _O_KRNL_READ);
	SET_CHANNELS(h, 0, _O_KRNL_READ);
	SET_FORMAT(h, _FMT_U8, _O_KRNL_READ);
}

static void CLEAR(__u8 *ptr, unsigned size, int format)
{
	switch (format) {
		default:
		case _FMT_S8:
		case _FMT_S16_LE:
		case _FMT_S16_BE:
		case _FMT_S32_LE:
		case _FMT_S32_BE:
			memset(ptr, 0, size);
			break;
		case _FMT_U8:
			memset(ptr, 0x80, size);
			break;
		case _FMT_U16_LE:
			while (size--) *ptr = !((unsigned long)ptr & 1) ? 0x00 : 0x80, ptr++;
			break;
		case _FMT_U16_BE:
			while (size--) *ptr = !((unsigned long)ptr & 1) ? 0x80 : 0x00, ptr++;
			break;
		case _FMT_U32_LE:
			while (size--) *ptr = ((unsigned long)ptr & 3) != 3 ? 0x00 : 0x80, ptr++;
			break;
		case _FMT_U32_BE:
			while (size--) *ptr = !((unsigned long)ptr & 3) ? 0x80 : 0x00, ptr++;
			break;
	}
}

static void MIXIN(__u8 *dst, __u8 *src, unsigned size, int format)
{
	switch (format) {
		default:
			memcpy(dst, src, size);
			break;
		case _FMT_U8:
			while (size) {
				int val = *dst + *src - 0x80;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x100)) val = 0xff;
				*dst = val;
				src++, dst++, size--;
			}
			break;
		case _FMT_S8:
			while (size) {
				int val = (__s8)*dst + (__s8)*src;
				if (__unlikely(val < -0x80)) val = -0x80;
				if (__unlikely(val >= 0x80)) val = 0x7f;
				*dst = val;
				src++, dst++, size--;
			}
			break;
		case _FMT_U16_LE:
			while (size > 1) {
				int val = __16LE2CPU(*(__u16 *)dst) + __16LE2CPU(*(__u16 *)src) - 0x8000;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x10000)) val = 0xffff;
				*(__u16 *)dst = __16CPU2LE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_U16_BE:
			while (size > 1) {
				int val = __16BE2CPU(*(__u16 *)dst) + __16BE2CPU(*(__u16 *)src) - 0x8000;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val >= 0x10000)) val = 0xffff;
				*(__u16 *)dst = __16CPU2BE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_S16_LE:
			while (size > 1) {
				int val = __16LE2CPU(*(__s16 *)dst) + __16LE2CPU(*(__s16 *)src);
				if (__unlikely(val < -0x8000)) val = -0x8000;
				if (__unlikely(val >= 0x8000)) val = 0x7fff;
				*(__s16 *)dst = __16CPU2LE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_S16_BE:
			while (size > 1) {
				int val = __16BE2CPU(*(__s16 *)dst) + __16BE2CPU(*(__s16 *)src);
				if (__unlikely(val < -0x8000)) val = -0x8000;
				if (__unlikely(val >= 0x8000)) val = 0x7fff;
				*(__s16 *)dst = __16CPU2BE(val);
				src += 2, dst += 2, size -= 2;
			}
			break;
		case _FMT_U32_LE:
			while (size > 3) {
				__s64 val = (__s64)__32LE2CPU(*(__u32 *)dst) + (__s64)__32LE2CPU(*(__u32 *)src) - 0x80000000U;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val > 0xffffffffU)) val = 0xffffffffU;
				*(__u32 *)dst = __32CPU2LE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_U32_BE:
			while (size > 3) {
				__s64 val = (__s64)__32BE2CPU(*(__u32 *)dst) + (__s64)__32BE2CPU(*(__u32 *)src) - 0x80000000U;
				if (__unlikely(val < 0)) val = 0;
				if (__unlikely(val > 0xffffffffU)) val = 0xffffffffU;
				*(__u32 *)dst = __32CPU2BE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_S32_LE:
			while (size > 3) {
				__s64 val = (__s64)__32LE2CPU(*(__s32 *)dst) + (__s64)__32LE2CPU(*(__s32 *)src);
				if (__unlikely(val < -(__s64)0x80000000U)) val = -(__s64)0x80000000U;
				if (__unlikely(val >= (__s64)0x80000000U)) val = (__s64)0x80000000U;
				*(__s32 *)dst = __32CPU2LE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
		case _FMT_S32_BE:
			while (size > 3) {
				__s64 val = (__s64)__32BE2CPU(*(__s32 *)dst) + (__s64)__32BE2CPU(*(__s32 *)src);
				if (__unlikely(val < -(__s64)0x80000000U)) val = -(__s64)0x80000000U;
				if (__unlikely(val >= (__s64)0x80000000U)) val = (__s64)0x80000000U;
				*(__s32 *)dst = __32CPU2BE(val);
				src += 4, dst += 4, size -= 4;
			}
			break;
	}
}

static __u8 *playback_frag_data(unsigned frag)
{
	unsigned page;
	while (__unlikely(frag >= oss_playback_frags)) frag -= oss_playback_frags;
	page = frag >> (PAGESIZE_BITS - oss_playback_fragsize_bits);
	if (__unlikely(page >= N_PAGES)) KERNEL$SUICIDE("playback_frag_data: OUT OF PAGES, REQUESTED FRAG %d, FRAGS %d, FRAGSIZE_BITS %d, PAGES %d, PAGESIZE_BITS %d", frag, oss_playback_frags, oss_playback_fragsize_bits, N_PAGES, PAGESIZE_BITS);
	return (__u8 *)PAGES[page].virtual + ((frag << oss_playback_fragsize_bits) & ((1 << PAGESIZE_BITS) - 1));
}

static __u8 *record_frag_data(unsigned frag)
{
	unsigned page;
	while (__unlikely(frag >= oss_record_frags)) frag -= oss_record_frags;
	page = frag >> (PAGESIZE_BITS - oss_record_fragsize_bits);
	if (__unlikely(page >= N_PAGES)) KERNEL$SUICIDE("record_frag_data: OUT OF PAGES, REQUESTED FRAG %d, FRAGS %d, FRAGSIZE_BITS %d, PAGES %d, PAGESIZE_BITS %d", frag, oss_record_frags, oss_record_fragsize_bits, N_PAGES, PAGESIZE_BITS);
	return (__u8 *)(OSS_FLAGS & OSS_DUPLEX ? CAP_PAGES : PAGES)[page].virtual + ((frag << oss_record_fragsize_bits) & ((1 << PAGESIZE_BITS) - 1));
}

static void default_frags(HANDLE *h)
{
	int fragsize = FORMAT_FRAGSIZE(h);
	int frags = FORMAT_FRAGS(h);
	int div;
	if (__unlikely(!frags) && __unlikely(!fragsize) && __unlikely(div = FORMAT_SUBDIVIDE(h))) {
		unsigned long s, s2 = MAXLONG;
		int fmt = FORMAT_FORMAT(h), chan = FORMAT_CHANNELS(h), rate = OSS_GET_RATE(FORMAT_RATE(h));
		rate /= div;
		fragsize = PAGESIZE_BITS;
		frags = FORMAT_MAXFRAGS;
		again:
		SET_FRAGSIZE(h, fragsize);
		SET_FRAGS(h, frags);
		fragsize = FORMAT_FRAGSIZE(h);
		frags = FORMAT_FRAGS(h);
		if (__likely((s = _FMT_DIV(frags << fragsize, fmt, chan)) > rate) && __likely(s < s2)) {
			s2 = s;
			if (frags >> 1 >= MIN_FRAGS) frags >>= 1;
			else if (1 << fragsize > MIN_FRAGSIZE) fragsize--;
			else goto ss;
			goto again;
		}
		ss:
		return;
	}
	if (__unlikely(!fragsize)) SET_FRAGSIZE(h, PAGESIZE_BITS);
	if (__unlikely(!frags)) SET_FRAGS(h, FORMAT_MAXFRAGS);
}

int OSS_FLAGS;

int oss_playback_frags, oss_playback_fragsize_bits;
int oss_playback_fragment, oss_playback_buf_fragments;
int oss_playback_rate, oss_playback_channels, oss_playback_format;
off_t oss_playback_file_pos;

int oss_playback_stat_bytes, oss_playback_stat_frags;
unsigned long oss_playback_stat_bytes_drop;
int oss_playback_stat_underruns;

WQ_DECL(OSS_PLAYBACK_WAIT, "OSS$PLAYBACK_WAIT");

int oss_record_frags, oss_record_fragsize_bits;
int oss_record_fragment, oss_record_buf_fragments;
int oss_record_buf_pos;
int oss_record_rate, oss_record_channels, oss_record_format;

int oss_record_stat_bytes, oss_record_stat_frags;
unsigned long oss_record_stat_bytes_drop;
int oss_record_stat_overruns;

WQ_DECL(OSS_RECORD_WAIT, "OSS$RECORD_WAIT");

static void TRIGGER_REC(HANDLE *h)
{
	default_frags(h);
	RAISE_SPL(SPL_SND);
	if (__unlikely((OSS_FLAGS & (OSS_DUPLEX | OSS_PLAYBACK)) == OSS_PLAYBACK)) {
		OSS_STOP(0);
	}
	if (__unlikely(!(OSS_FLAGS & OSS_RECORD))) {
		restart:
		SND_FIXUP_PARAMS(h, _O_KRNL_READ);
		oss_record_frags = FORMAT_FRAGS(h);
		oss_record_fragsize_bits = FORMAT_FRAGSIZE(h);
		oss_record_rate = FORMAT_RATE(h);
		oss_record_channels = FORMAT_CHANNELS(h);
		oss_record_format = FORMAT_FORMAT(h);
		oss_record_buf_pos = 0;
		oss_record_fragment = 0;
		oss_record_buf_fragments = 0;
		SND_START(1);
	}
	if (__unlikely(oss_record_rate != FORMAT_RATE(h)) ||
	    __unlikely(oss_record_channels != FORMAT_CHANNELS(h)) ||
	    __unlikely(oss_record_format != FORMAT_FORMAT(h)) ||
	    __unlikely(oss_record_frags != FORMAT_FRAGS(h)) ||
	    __unlikely(oss_record_fragsize_bits != FORMAT_FRAGSIZE(h))) {
		OSS_STOP(1);
		goto restart;
	}
	LOWER_SPL(SPL_OSS);
}

DECL_IOCALL(OSS_READ, SPL_OSS, SIORQ)
{
	HANDLE *h = RQ->handle;
	VBUF vbuf;
	long s;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	if (!RQ->v.len) goto zlen;
	again:
	if (__unlikely(OSS_FLAGS & OSS_BLOCK_RECORD)) {
		RQ->status = -EAGAIN;
		RETURN_AST(RQ);
	}
	TRIGGER_REC(h);
	RAISE_SPL(SPL_SND);
	if (__unlikely(!oss_record_buf_fragments)) {
		if (FORMAT_NONBLOCK(h)) {
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		WQ_WAIT_F(&OSS_RECORD_WAIT, RQ);
		LOWER_SPL(SPL_OSS);
		RETURN;
	}
	LOWER_SPL(SPL_OSS);
	vbuf.ptr = record_frag_data(oss_record_fragment) + oss_record_buf_pos;
	vbuf.len = (1 << oss_record_fragsize_bits) - oss_record_buf_pos;
	vbuf.spl = SPL_X(SPL_OSS);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) {
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	}
	RQ->progress += s;
	if (oss_record_buf_pos + s == (1 << oss_record_fragsize_bits)) {
		RAISE_SPL(SPL_SND);
		if (__likely(OSS_FLAGS & OSS_RECORD)) SND_SUBMIT_FRAGMENT(1, oss_record_fragment);
		oss_record_buf_pos = 0;
		oss_record_fragment++;
		if (__unlikely(oss_record_fragment == oss_record_frags)) oss_record_fragment = 0;
		oss_record_buf_fragments--;
		LOWER_SPL(SPL_OSS);
	} else oss_record_buf_pos += s;
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (RQ->v.len) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static __finline__ void OSS_CLR_PLAYBACK(void)
{
	oss_playback_file_pos = 0;
	oss_playback_fragment = -1;
	oss_playback_buf_fragments = 0;
	oss_playback_stat_bytes = 0;
	oss_playback_stat_frags = 0;
	oss_playback_stat_bytes_drop = 0;
}

static __finline__ void OSS_CLR_RECORD(void)
{
	oss_record_buf_pos = 0;
	oss_record_fragment = 0;
	oss_record_buf_fragments = 0;
	oss_record_stat_bytes = 0;
	oss_record_stat_frags = 0;
	oss_record_stat_bytes_drop = 0;
}

void OSS_STOP(int record)
{
	if (__likely(!record)) {
		OSS_CLR_PLAYBACK();
		oss_playback_stat_underruns++;
		SND_STOP(0);
		WQ_WAKE_ALL(&OSS_PLAYBACK_WAIT);
	} else {
		OSS_CLR_RECORD();
		oss_record_stat_overruns++;
		SND_STOP(1);
		WQ_WAKE_ALL(&OSS_RECORD_WAIT);
	}
}

void TRIGGER_PLAY(void)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("TRIGGER PLAY AT SPL %08X", KERNEL$SPL);
	if (__likely(!(OSS_FLAGS & OSS_PLAYBACK))) {
		if (__unlikely(!oss_playback_buf_fragments) && !(OSS_FLAGS & OSS_LOOP_PLAYBACK)) return;
		if (__likely(oss_playback_buf_fragments)) oss_playback_buf_fragments--;
		oss_playback_fragment++;
		if (__unlikely(oss_playback_fragment >= oss_playback_frags)) oss_playback_fragment = 0;
		oss_playback_file_pos += 1 << oss_playback_fragsize_bits;
		SND_START(0);
	}
}

#define READ_BUFFER_SIZE	4096

static __u8 read_buffer[READ_BUFFER_SIZE];

DECL_IOCALL(OSS_WRITE, SPL_OSS, SIORQ)
{
	HANDLE *h = RQ->handle;
	VBUF vbuf;
	FFILE *file;
	vspace_unmap_t *funmap;
	int x;
	off_t fpos;
	long dpos;
	long s;
	int pf;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	if (__unlikely(!RQ->v.len)) goto zlen;
	RAISE_SPL(SPL_SND);
	if (__unlikely((OSS_FLAGS & (OSS_DUPLEX | OSS_RECORD)) == OSS_RECORD)) {
		OSS_STOP(1);
	}
	LOWER_SPL(SPL_OSS);
	default_frags(h);
	if (__unlikely(FORMAT_FRAGS(h) != oss_playback_frags) ||
	    __unlikely(FORMAT_FRAGSIZE(h) != oss_playback_fragsize_bits)) {
		if (__likely(oss_playback_fragment == -1) && __likely(!oss_playback_buf_fragments)) {
			oss_playback_frags = FORMAT_FRAGS(h);
			oss_playback_fragsize_bits = FORMAT_FRAGSIZE(h);
		}
	}
	RAISE_SPL(SPL_SND);
	if (__unlikely(FORMAT_RATE(h) != oss_playback_rate) ||
	    __unlikely(FORMAT_CHANNELS(h) != oss_playback_channels) ||
	    __unlikely(FORMAT_FORMAT(h) != oss_playback_format)) {
		if (__likely(oss_playback_fragment == -1) && __likely(!oss_playback_buf_fragments)) {
			SND_FIXUP_PARAMS(h, _O_KRNL_WRITE);
			oss_playback_rate = FORMAT_RATE(h);
			oss_playback_channels = FORMAT_CHANNELS(h);
			oss_playback_format = FORMAT_FORMAT(h);
		} else {
			OSS_FLAGS |= OSS_PLAYBACK_STOP;
			wfs:
			if (__unlikely(OSS_FLAGS & OSS_BLOCK_PLAYBACK)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = -EAGAIN;
				RETURN_AST(RQ);
			}
			if (FORMAT_NONBLOCK(h)) {
				TRIGGER_PLAY();
				LOWER_SPL(SPL_OSS);
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&OSS_PLAYBACK_WAIT, RQ);
			TRIGGER_PLAY();
			LOWER_SPL(SPL_OSS);
			RETURN;
		}
	}
	if (__unlikely(OSS_FLAGS & OSS_PLAYBACK_STOP)) goto wfs;
	LOWER_SPL(SPL_OSS);
	RAISE_SPL(SPL_VSPACE);
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	LOWER_SPL(SPL_OSS);
	if (__unlikely(!file)) RETURN;
	if (__unlikely(!(file->flags & _O_KRNL_WRITE))) {
		RAISE_SPL(SPL_VSPACE);
		funmap(file);
		LOWER_SPL(SPL_OSS);
		RQ->status = -EPERM;
		RETURN_AST(RQ);
	}
	again:
	if (__unlikely(x = (int)(fpos = file->pos) & (_FMT_SIZE(oss_playback_format) - 1))) {
		*(__u32 *)read_buffer = *(__u32 *)&file->aux;
	}
	fpos -= x;
	vbuf.ptr = read_buffer + x;
	vbuf.len = (1 << oss_playback_fragsize_bits) - (fpos & ((1 << oss_playback_fragsize_bits) - 1)) - x;
	if (vbuf.len + x > READ_BUFFER_SIZE) vbuf.len = READ_BUFFER_SIZE - x;
	vbuf.spl = SPL_X(SPL_OSS);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(s = RQ->v.vspace->op->vspace_get(&RQ->v, &vbuf)))) {
		RAISE_SPL(SPL_VSPACE);
		funmap(file);
		LOWER_SPL(SPL_OSS);
		DO_PAGEIN(RQ, &RQ->v, PF_READ);
	}
	RQ->progress += s;
	s += x;
	RAISE_SPL(SPL_SND);
	repeat:
	dpos = fpos - oss_playback_file_pos;
	if ((off_t)dpos < 0) fpos = oss_playback_file_pos, dpos = 0;
	if ((off_t)dpos > (oss_playback_buf_fragments << oss_playback_fragsize_bits)) fpos = oss_playback_file_pos + (dpos = oss_playback_buf_fragments << oss_playback_fragsize_bits);
	if (dpos >= oss_playback_buf_fragments << oss_playback_fragsize_bits) {
		if (__unlikely(oss_playback_buf_fragments >= oss_playback_frags - (oss_playback_fragment != -1))) {
			s -= x;
			RQ->v.ptr -= s;
			RQ->v.len += s;
			RQ->progress -= s;
			WQ_WAIT_F(&OSS_PLAYBACK_WAIT, RQ);
			LOWER_SPL(SPL_OSS);
			unmap_return:
			RAISE_SPL(SPL_VSPACE);
			funmap(file);
			LOWER_SPL(SPL_OSS);
			RETURN;
		}
		pf = oss_playback_fragment + 1 + oss_playback_buf_fragments;
		LOWER_SPL(SPL_OSS);
/*__debug_printf("clear(%d)", pf);*/
		CLEAR(playback_frag_data(pf), 1 << oss_playback_fragsize_bits, oss_playback_format);
#ifdef OSS_CANT_STOP
		if (__likely(!(OSS_FLAGS & OSS_LOOP_PLAYBACK))) if (oss_playback_buf_fragments < oss_playback_frags - (oss_playback_fragment != -1) - 1) CLEAR(playback_frag_data(pf + 1), 1 << oss_playback_fragsize_bits, oss_playback_format);
#endif
		RAISE_SPL(SPL_SND);
		if (__unlikely(pf != oss_playback_fragment + 1 + oss_playback_buf_fragments)) goto repeat;
		oss_playback_buf_fragments++;
		/*if (__likely((OSS_FLAGS & (OSS_PLAYBACK | OSS_LOOP_PLAYBACK)) == OSS_PLAYBACK) && __likely(oss_playback_buf_fragments != 1)) SND_SUBMIT_FRAGMENT(0);*/
		goto repeat;
	}
	pf = oss_playback_fragment + 1 + (dpos >> oss_playback_fragsize_bits);
	LOWER_SPL(SPL_OSS);
	if (__unlikely((dpos & ((1 << oss_playback_fragsize_bits) - 1)) + s > 1 << oss_playback_fragsize_bits))
		KERNEL$SUICIDE("OSS_WRITE: MIXING OUT OF FRAGMENT: DPOS %ld, FRAGSIZE_BITS %d, S %ld", dpos, oss_playback_fragsize_bits, s);
/*__debug_printf("mixin(%d,%d)", pf, (dpos & ((1 << oss_playback_fragsize_bits) - 1)));*/
	MIXIN(playback_frag_data(pf) + (dpos & ((1 << oss_playback_fragsize_bits) - 1)), read_buffer, s & ~(_FMT_SIZE(oss_playback_format) - 1), oss_playback_format);
	if (__unlikely(x = s & (_FMT_SIZE(oss_playback_format) - 1))) {
		memcpy(&file->aux, read_buffer + (s & ~(_FMT_SIZE(oss_playback_format) - 1)), x);
	}
	file->pos = fpos + s;
	if (__unlikely(oss_playback_fragment == -1) && __likely(oss_playback_buf_fragments >= 2) && __likely(!(OSS_FLAGS & OSS_BLOCK_PLAYBACK))) {
		RAISE_SPL(SPL_SND);
		TRIGGER_PLAY();
		LOWER_SPL(SPL_OSS);
	}
	if (fpos + s == oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits)) {
		if (__likely((OSS_FLAGS & (OSS_PLAYBACK | OSS_LOOP_PLAYBACK)) == OSS_PLAYBACK)) SND_SUBMIT_FRAGMENT(0, oss_playback_fragment + oss_playback_buf_fragments);
	}
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_OSS));
	TEST_LOCKUP_LOOP(RQ, goto unmap_return;);
	if (RQ->v.len) goto again;
	RAISE_SPL(SPL_VSPACE);
	funmap(file);
	LOWER_SPL(SPL_OSS);
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

int OSS_PLAYBACK_INTR(void)
{
	if (__unlikely(!(OSS_FLAGS & OSS_PLAYBACK))) return 1;
	/*if (__unlikely(OSS_FLAGS & OSS_BLOCK_PLAYBACK)) {
		SND_STOP(0);
		WQ_WAKE_ALL(&OSS_PLAYBACK_WAIT);
		return 1;
	}*/
	if (!oss_playback_buf_fragments && __likely((OSS_FLAGS & (OSS_LOOP_PLAYBACK | OSS_PLAYBACK_STOP)) != OSS_LOOP_PLAYBACK)) {
		OSS_STOP(0);
		return 1;
	}
#ifdef OSS_CANT_STOP
	if (__likely(!(OSS_FLAGS & OSS_LOOP_PLAYBACK))) CLEAR(playback_frag_data(oss_playback_fragment + 1 + oss_playback_buf_fragments), 1 << oss_playback_fragsize_bits, oss_playback_format);
#endif
	oss_playback_fragment++;
	if (__unlikely(oss_playback_fragment >= oss_playback_frags)) oss_playback_fragment = 0;
	if (__unlikely(oss_playback_buf_fragments == 1) && __likely(!(OSS_FLAGS & OSS_LOOP_PLAYBACK))) SND_SUBMIT_FRAGMENT(0, oss_playback_fragment);
	if (__unlikely(oss_playback_fragment >> (PAGESIZE_BITS - oss_playback_fragsize_bits) >= N_PAGES)) KERNEL$SUICIDE("OSS_PLAYBACK_INTR: OUT OF PAGE: FRAGMENT %d, PAGESIZE_BITS %d, FRAGSIZE_BITS %d, N_PAGES %d", oss_playback_fragment, PAGESIZE_BITS, oss_playback_fragsize_bits, N_PAGES);
	if (__likely(oss_playback_buf_fragments)) oss_playback_buf_fragments--;
	oss_playback_file_pos += 1 << oss_playback_fragsize_bits;
	oss_playback_stat_bytes += 1 << oss_playback_fragsize_bits;
	oss_playback_stat_frags++;
	if (__unlikely(oss_playback_stat_bytes >= MAXINT >> 1)) {
		oss_playback_stat_bytes -= MAXINT >> 1;
		oss_playback_stat_bytes_drop += MAXINT >> 1;
	}
	WQ_WAKE_ALL_PL(&OSS_PLAYBACK_WAIT);
	return 0;
}

int OSS_RECORD_INTR(void)
{
	if (__unlikely(!(OSS_FLAGS & OSS_RECORD))) return 1;
	oss_record_buf_fragments++;
	if (__unlikely(oss_record_buf_fragments >= oss_record_frags)) {
		OSS_STOP(1);
		return 1;
	}
	oss_record_stat_bytes += 1 << oss_record_fragsize_bits;
	oss_record_stat_frags++;
	if (__unlikely(oss_record_stat_bytes >= MAXINT >> 1)) {
		oss_record_stat_bytes -= MAXINT >> 1;
		oss_record_stat_bytes_drop += MAXINT >> 1;
	}
	WQ_WAKE_ALL_PL(&OSS_RECORD_WAIT);
	return 0;
}

static __finline__ int bytes_to_read(FFILE *f)
{
	if (__unlikely(!(OSS_FLAGS & OSS_RECORD))) return 0;
	return (oss_record_frags << oss_record_fragsize_bits) - oss_record_buf_pos;
}

static __finline__ int bytes_to_write(HANDLE *h, FFILE *f)
{
	int a;
	off_t fpos;
	int x;
	fpos = f->pos;
	x = (int)fpos & (_FMT_SIZE(oss_playback_format) - 1);
	if (__unlikely(!(OSS_FLAGS & OSS_PLAYBACK))) return (FORMAT_FRAGS(h) << FORMAT_FRAGSIZE(h)) - x;
	if (__unlikely(fpos < oss_playback_file_pos)) fpos = oss_playback_file_pos | x;
	if (__unlikely(fpos > oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits))) fpos = (oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits)) | x;
	a = oss_playback_file_pos + ((oss_playback_frags - ((OSS_FLAGS & OSS_PLAYBACK) != 0)) << oss_playback_fragsize_bits) - fpos;
	if (__unlikely(a < 0)) a = 0;
	return a;
}

static __finline__ int bytes_written(FFILE *f)
{
	off_t fpos = f->pos;
	if (__unlikely(fpos < oss_playback_file_pos)) fpos = oss_playback_file_pos;
	if (__unlikely(fpos > oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits))) fpos = (oss_playback_file_pos + (oss_playback_buf_fragments << oss_playback_fragsize_bits));
#ifdef OSS_PRECISE_TIMING
	if (__likely(OSS_FLAGS & OSS_PLAYBACK)) {
		int p = SND_GET_PTR(0) - (oss_playback_fragment << oss_playback_fragsize_bits);
		if (__unlikely(p < 0)) p += oss_playback_frags << oss_playback_fragsize_bits;
		return fpos - oss_playback_file_pos + (1 << oss_playback_fragsize_bits) - p;
	}
#endif
	return fpos - oss_playback_file_pos;
}

DECL_IOCALL(OSS_IOCTL, SPL_OSS, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	vspace_unmap_t *funmap;
	FFILE *file;
	if (__unlikely(h->op != &SOUND_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_OSS));
	RAISE_SPL(SPL_VSPACE);
	file = KERNEL$MAP_FILE_STRUCT(h, (IORQ *)RQ, &funmap);
	LOWER_SPL(SPL_OSS);
	if (__unlikely(!file)) RETURN;
	switch (RQ->ioctl) {
		case IOCTL_SELECT_READ: {
			RAISE_SPL(SPL_SND);
			if (bytes_to_read(file)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = 0;
				break;
			}
			if (RQ->param) {
				WQ_WAIT_F(&OSS_RECORD_WAIT, RQ);
				LOWER_SPL(SPL_OSS);
				RAISE_SPL(SPL_VSPACE);
				funmap(file);
				LOWER_SPL(SPL_OSS);
				RETURN;
			}
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			break;
		}
		case IOCTL_SELECT_WRITE: {
			default_frags(h);
			RAISE_SPL(SPL_SND);
			if (bytes_to_write(h, file)) {
				LOWER_SPL(SPL_OSS);
				RQ->status = 0;
				break;
			}
			if (RQ->param) {
				WQ_WAIT_F(&OSS_PLAYBACK_WAIT, RQ);
				LOWER_SPL(SPL_OSS);
				RAISE_SPL(SPL_VSPACE);
				funmap(file);
				LOWER_SPL(SPL_OSS);
				RETURN;
			}
			LOWER_SPL(SPL_OSS);
			RQ->status = -EWOULDBLOCK;
			break;
		}
		case IOCTL_NREAD: {
			RAISE_SPL(SPL_SND);
			RQ->status = bytes_to_read(file);
			LOWER_SPL(SPL_OSS);
			break;
		}
		case IOCTL_SND_SYNC: {
			switch (RQ->param) {
				case PARAM_SND_SYNC_RESET: {
					if (file->flags & _O_KRNL_READ) {
						RAISE_SPL(SPL_SND);
						OSS_STOP(1);
						LOWER_SPL(SPL_OSS);
					}
					if (file->flags & _O_KRNL_WRITE) {
						RAISE_SPL(SPL_SND);
						OSS_STOP(0);
						LOWER_SPL(SPL_OSS);
					}
					RQ->status = 0;
					break;
				}
				case PARAM_SND_SYNC_SYNC: {
					RAISE_SPL(SPL_VSPACE);
					funmap(file);
					LOWER_SPL(SPL_OSS);
					if (OSS_CLOSE(h, (IORQ *)RQ)) RETURN;
					
					/*if (file->flags & _O_KRNL_WRITE) {
						RAISE_SPL(SPL_SND);
						TRIGGER_PLAY();
						LOWER_SPL(SPL_OSS);
					}*/
					RQ->status = 0;
					RETURN_AST(RQ);
				}
				case PARAM_SND_SYNC_POST: {
					if (file->flags & _O_KRNL_WRITE) {
						RAISE_SPL(SPL_SND);
						TRIGGER_PLAY();
						LOWER_SPL(SPL_OSS);
					}
					RQ->status = 0;
					break;
				}
				default: {
					RQ->status = -EINVAL;
					break;
				}
			}
			break;
		}
		case IOCTL_SND_GETRATE: {
			RQ->status = OSS_GET_RATE(FORMAT_RATE(h));
			break;
		}
		case IOCTL_SND_GETCHANNELS: {
			RQ->status = FORMAT_CHANNELS(h) + 1;
			break;
		}
		case IOCTL_SND_GETFORMAT: {
			RQ->status = FORMAT_FORMAT(h);
			break;
		}
		case IOCTL_SND_GETFORMATMASK: {
			int i;
			long ret = 0;
			for (i = 0; i <= _FMT_MAX; i++) if (__unlikely(FORMATS[i])) ret |= 1 << i;
			if (__unlikely(ret < 0)) ret &= ~__LONG_SGN_BIT;
			RQ->status = ret;
			break;
		}
		case IOCTL_SND_GETSPACE: {
			int frags, fragsize_bits;
			int r;
			audio_buf_info info;
			memset(&info, 0, sizeof info);
			default_frags(h);
			if (__likely(RQ->param == PARAM_SND_GETSPACE_PLAYBACK)) {
				gets_playback:
				RAISE_SPL(SPL_SND);
				info.bytes = bytes_to_write(h, file);
				if (OSS_FLAGS & OSS_PLAYBACK) {
					frags = oss_playback_frags;
					fragsize_bits = oss_playback_fragsize_bits;
					LOWER_SPL(SPL_OSS);
				} else {
					LOWER_SPL(SPL_OSS);
					frags = FORMAT_FRAGS(h);
					fragsize_bits = FORMAT_FRAGSIZE(h);
				}
			} else if (__likely(RQ->param == PARAM_SND_GETSPACE_RECORD)) {
				gets_record:
				RAISE_SPL(SPL_SND);
				info.bytes = bytes_to_read(file);
				if (OSS_FLAGS & OSS_RECORD) {
					frags = oss_record_frags;
					fragsize_bits = oss_record_fragsize_bits;
					LOWER_SPL(SPL_OSS);
				} else {
					LOWER_SPL(SPL_OSS);
					frags = FORMAT_FRAGS(h);
					fragsize_bits = FORMAT_FRAGSIZE(h);
				}
			} else if (__likely(RQ->param == PARAM_SND_GETSPACE_CURRENT)) {
				if (__likely(file->flags & _O_KRNL_WRITE)) goto gets_playback;
				else if (__likely(file->flags & _O_KRNL_READ)) goto gets_record;
				else goto gets_playback;
			} else {
				RQ->status = -EINVAL;
				break;
			}
			info.fragstotal = frags;
			info.fragsize = 1 << fragsize_bits;
			info.fragments = info.bytes >> fragsize_bits;
			/*__debug_printf("buf_fragments: %d\n", oss_playback_buf_fragments);
			__debug_printf("ospace: bytes %d, fragstotal %d, fragsize %d, fragments %d", info.bytes, info.fragstotal, info.fragsize, info.fragments);*/
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &info, sizeof info)) == 1)) {
				RAISE_SPL(SPL_VSPACE);
				funmap(file);
				LOWER_SPL(SPL_OSS);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_SND_GETCAPS: {
			long caps = 1 | DSP_CAP_TRIGGER | DSP_CAP_MMAP;
#ifdef OSS_PRECISE_TIMING
			caps |= DSP_CAP_REALTIME;
#endif
			if (OSS_FLAGS & OSS_DUPLEX) caps |= DSP_CAP_DUPLEX;
			RQ->status = caps;
			break;
		}
		case IOCTL_SND_GETTRIGGER: {
			long trig = 0;
			if (file->flags & _O_KRNL_READ && !(OSS_FLAGS & OSS_BLOCK_RECORD)) trig |= PCM_ENABLE_INPUT;
			if (file->flags & _O_KRNL_WRITE && !(OSS_FLAGS & OSS_BLOCK_PLAYBACK)) trig |= PCM_ENABLE_OUTPUT;
			RQ->status = trig;
			break;
		}
		case IOCTL_SND_SETTRIGGER: {
			if (__unlikely(RQ->param & ~(PCM_ENABLE_INPUT | PCM_ENABLE_OUTPUT))) {
				RQ->status = -EINVAL;
				break;
			}
			default_frags(h);
			if (file->flags & _O_KRNL_READ) {
				RAISE_SPL(SPL_SND);
				if (RQ->param & PCM_ENABLE_INPUT) {
					OSS_FLAGS &= ~OSS_BLOCK_RECORD;
					if (OSS_FLAGS & OSS_RECORD) SND_RESUME(1);
					else TRIGGER_REC(h);
				} else {
					OSS_FLAGS |= OSS_BLOCK_RECORD;
					if (OSS_FLAGS & OSS_RECORD) SND_PAUSE(1);
				}
				LOWER_SPL(SPL_OSS);
			}
			if (file->flags & _O_KRNL_WRITE) {
				RAISE_SPL(SPL_SND);
				if (RQ->param & PCM_ENABLE_OUTPUT) {
					if (file->flags & O_LOOP_FLAG && !(OSS_FLAGS & OSS_LOOP_PLAYBACK)) {
						if (__unlikely(OSS_FLAGS & OSS_PLAYBACK)) OSS_STOP(0);
						SND_FIXUP_PARAMS(h, _O_KRNL_WRITE);
						oss_playback_frags = FORMAT_FRAGS(h);
						oss_playback_fragsize_bits = FORMAT_FRAGSIZE(h);
						oss_playback_rate = FORMAT_RATE(h);
						oss_playback_channels = FORMAT_CHANNELS(h);
						oss_playback_format = FORMAT_FORMAT(h);
						OSS_FLAGS |= OSS_LOOP_PLAYBACK;
					}
					OSS_FLAGS &= ~OSS_BLOCK_PLAYBACK;
					if (OSS_FLAGS & OSS_PLAYBACK) SND_RESUME(0);
					else TRIGGER_PLAY();
				} else {
					OSS_FLAGS |= OSS_BLOCK_PLAYBACK;
					if (OSS_FLAGS & OSS_PLAYBACK) SND_PAUSE(0);
				}
				LOWER_SPL(SPL_OSS);
			}
			RQ->status = 0;
			break;
		}
		case IOCTL_SND_GETPTR: {
			int r;
			count_info info;
			memset(&info, 0, sizeof info);
			if (__likely(RQ->param == PARAM_SND_GETPTR_PLAYBACK)) {
				RAISE_SPL(SPL_SND);
				if (__likely(OSS_FLAGS & OSS_PLAYBACK)) {
#ifdef OSS_PRECISE_TIMING
					int p;
					info.ptr = SND_GET_PTR(0);
					p = info.ptr - (oss_playback_fragment << oss_playback_fragsize_bits);
					if (__unlikely(p < 0)) p += oss_playback_frags << oss_playback_fragsize_bits;
					info.bytes = oss_playback_stat_bytes + p;
#else
					info.ptr = oss_playback_fragment << oss_playback_fragsize_bits;
					info.bytes = oss_playback_stat_bytes;
#endif
					info.blocks = oss_playback_stat_frags;
				}
				LOWER_SPL(SPL_OSS);
			} else if (__likely(RQ->param == PARAM_SND_GETPTR_RECORD)) {
				RAISE_SPL(SPL_SND);
				if (__likely(OSS_FLAGS & OSS_RECORD)) {
#ifdef OSS_PRECISE_TIMING
					int p;
#endif
					int frag = oss_record_fragment + oss_record_buf_fragments;
					if (__unlikely(frag >= oss_record_frags)) frag -= oss_record_frags;
#ifdef OSS_PRECISE_TIMING
					info.ptr = SND_GET_PTR(1);
					p = info.ptr - (frag << oss_record_fragsize_bits);
					if (__unlikely(p < 0)) p += oss_record_frags << oss_record_fragsize_bits;
					info.bytes = oss_record_stat_bytes + p;
#else
					info.ptr = frag << oss_playback_fragsize_bits;
					info.bytes = oss_record_stat_bytes;
#endif
					info.blocks = oss_record_stat_frags;
				}
				LOWER_SPL(SPL_OSS);
			} else {
				RQ->status = -EINVAL;
				break;
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &info, sizeof info)) == 1)) {
				RAISE_SPL(SPL_VSPACE);
				funmap(file);
				LOWER_SPL(SPL_OSS);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			if (__likely(!r)) {
				RAISE_SPL(SPL_SND);
				
				if (__likely(RQ->param == PARAM_SND_GETPTR_PLAYBACK)) oss_playback_stat_frags -= info.blocks;
				else oss_record_stat_frags -= info.blocks;
				LOWER_SPL(SPL_OSS);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_SND_GETODELAY: {
			RAISE_SPL(SPL_SND);
			if (__likely(OSS_FLAGS & OSS_PLAYBACK)) RQ->status = bytes_written(file);
			else RQ->status = 0;
			LOWER_SPL(SPL_OSS);
			break;
		}
		case IOCTL_SND_GETERROR: {
			int r;
			audio_errinfo ae;
			memset(&ae, 0, sizeof ae);
			RAISE_SPL(SPL_SND);
			ae.play_ptradjust = oss_playback_stat_bytes_drop;
			ae.rec_ptradjust = oss_record_stat_bytes_drop;
			ae.play_underruns = oss_playback_stat_underruns;
			ae.rec_overruns = oss_record_stat_overruns;
			LOWER_SPL(SPL_OSS);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &ae, sizeof ae)) == 1)) {
				RAISE_SPL(SPL_VSPACE);
				funmap(file);
				LOWER_SPL(SPL_OSS);
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			if (__likely(!r)) {
				RAISE_SPL(SPL_SND);
				
				oss_playback_stat_bytes_drop -= ae.play_ptradjust;
				oss_record_stat_bytes_drop -= ae.rec_ptradjust;
				oss_playback_stat_underruns -= ae.play_underruns;
				oss_record_stat_overruns -= ae.rec_overruns;
				LOWER_SPL(SPL_OSS);
			}
			RQ->status = r;
			break;
		}
		case IOCTL_CAN_MMAP: {
			file->flags |= O_LOOP_FLAG;
			RQ->status = 0;
			break;
		}
#ifdef OSS_MIXER
		case IOCTL_SND_MIXWRITE: {
			unsigned mix = RQ->param & PARAM_MIX_IDX_MASK;
			unsigned long mix_data = RQ->param >> PARAM_MIX_DATA_SHIFT;
			RAISE_SPL(SPL_SND);
			if (mix == SOUND_MIXER_RECSRC) SND_MIXER_WRITE_RECSRC(mix_data);
			else if (mix == SOUND_MIXER_OUTSRC) SND_MIXER_WRITE_OUTSRC(mix_data);
			else if (__likely(mix < SOUND_MIXER_NRDEVICES) && __likely(OSS_MIXER_DEVMASK & (1 << mix))) SND_MIXER_WRITE(mix, mix_data);
		}
			/* fall through */
		case IOCTL_SND_MIXREAD: {
			unsigned mix = RQ->param & PARAM_MIX_IDX_MASK;
			RAISE_SPL(SPL_SND);
			if (mix == SOUND_MIXER_CAPS) RQ->status = OSS_MIXER_CAPS;
			else if (mix == SOUND_MIXER_DEVMASK) RQ->status = OSS_MIXER_DEVMASK;
			else if (mix == SOUND_MIXER_STEREODEVS) RQ->status = OSS_MIXER_STEREODEVS;
			else if (mix == SOUND_MIXER_RECSRC) RQ->status = SND_MIXER_READ_RECSRC();
			else if (mix == SOUND_MIXER_RECMASK) RQ->status = OSS_MIXER_RECMASK;
			else if (mix == SOUND_MIXER_OUTSRC) RQ->status = SND_MIXER_READ_OUTSRC();
			else if (mix == SOUND_MIXER_OUTMASK) RQ->status = OSS_MIXER_OUTMASK;
			else if (__likely(mix < SOUND_MIXER_NRDEVICES) && __likely(OSS_MIXER_DEVMASK & (1 << mix))) RQ->status = SND_MIXER_READ(mix);
			else RQ->status = -EINVAL;
			LOWER_SPL(SPL_OSS);
			break;
		}
#endif
		default: {
			RQ->status = -ENOOP;
			break;
		}
	}
	RAISE_SPL(SPL_VSPACE);
	funmap(file);
	LOWER_SPL(SPL_OSS);
	RETURN_AST(RQ);
}

PAGE *OSS_GET_PAGE(HANDLE *h, __v_off idx, int wr)
{
	if (__unlikely(wr & ~PF_RW)) return __ERR_PTR(-EINVAL);
	if (__unlikely(idx >= N_PAGES << PAGESIZE_BITS)) return __ERR_PTR(-ERANGE);
	return KERNEL$PHYS_2_PAGE(PAGES[(unsigned)idx >> PAGESIZE_BITS].physical + (idx & ((1 << PAGESIZE_BITS) - 1)));
}

#if defined(OSS_ISA_DMA) || defined(OSS_PCI_DMA)

unsigned N_PAGES, PAGESIZE_BITS;

struct page_desc *PAGES;
struct page_desc *CAP_PAGES;

unsigned PAGE_CLUSTERS;

int OSS_ALLOC_DMAPAGES(unsigned n_pages, unsigned page_size)
{
#ifdef OSS_PCI_DMA
	VDMA dma;
#ifdef OSS_DMA_CAPABLE_64
	VDMA64 dma64;
#endif
	VDESC desc;
	int spl = KERNEL$SPL;
#endif
	MALLOC_REQUEST mrq;
	CONTIG_AREA_REQUEST car;
	int i, n;
	PAGESIZE_BITS = __BSR(page_size);
	if (1 << PAGESIZE_BITS != page_size || !page_size) KERNEL$SUICIDE("OSS_ALLOC_ISA_DMAPAGES: BOGUS PAGE SIZE %X", page_size);
	if (page_size < __PAGE_CLUSTER_SIZE) page_size = __PAGE_CLUSTER_SIZE;
	mrq.size = n_pages * sizeof(struct page_desc);
	if (OSS_FLAGS & OSS_DUPLEX) mrq.size *= 2;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (mrq.status < 0) return mrq.status;
	PAGES = mrq.ptr;
	if (OSS_FLAGS & OSS_DUPLEX) {
		CAP_PAGES = PAGES + n_pages;
	}
	PAGE_CLUSTERS = page_size >> __PAGE_CLUSTER_BITS;
	for (i = 0; i < n_pages; i++) {
		car.nclusters = PAGE_CLUSTERS;
		car.align = page_size - 1;
#ifdef OSS_ISA_DMA
		car.flags = CARF_ISADMA | CARF_PHYSCONTIG;
#else
#ifdef OSS_DMA_CAPABLE_64
		if (OSS_DMA_USE_64) car.flags = CARF_DATA | CARF_PHYSCONTIG; else
#endif
		car.flags = CARF_PCIDMA | CARF_DATA | CARF_PHYSCONTIG;
#endif
		SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
		if (car.status < 0) {
#ifdef OSS_CACHEMODE
			err1:
#endif
			if (!i) {
				KERNEL$UNIVERSAL_FREE(PAGES);
				PAGES = CAP_PAGES = NULL;
				return car.status;
			}
			break;
		}
		memset(car.ptr, 0, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS);
		PAGES[i].virtual = car.ptr;
		PAGES[i].physical = KERNEL$VIRT_2_PHYS(car.ptr);
#ifdef OSS_CACHEMODE
		if (__unlikely(car.status = KERNEL$SET_MEMORY_PAT(PAGES[i].physical, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS, OSS_CACHEMODE))) {
			KERNEL$VM_RELEASE_CONTIG_AREA(PAGES[i].virtual, PAGE_CLUSTERS);
			goto err1;
		}
#endif
#ifdef OSS_PCI_DMA
		desc.ptr = PAGES[i].physical;
		desc.len = PAGE_CLUSTERS << __PAGE_CLUSTER_BITS;
		desc.vspace = &KERNEL$PHYSICAL;
#ifdef OSS_DMA_CAPABLE_64
		if (!OSS_DMA_USE_64) {
#endif
			RAISE_SPL(SPL_VSPACE);
			dma = KERNEL$PHYSICAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &PAGES[i].u.unlock32);
			LOWER_SPLX(spl);
			if (__unlikely(dma.len != (PAGE_CLUSTERS << __PAGE_CLUSTER_BITS))) {
				KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
			}
			PAGES[i].device = dma.ptr;
#ifdef OSS_DMA_CAPABLE_64
		} else {
			RAISE_SPL(SPL_VSPACE);
			KERNEL$PHYSICAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &dma64, &PAGES[i].u.unlock64);
			LOWER_SPLX(spl);
			if (__unlikely(dma64.len != (PAGE_CLUSTERS << __PAGE_CLUSTER_BITS))) {
				KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
			}
			PAGES[i].device = dma64.ptr;
		}
#endif
#endif
		for (n = 0; n < PAGE_CLUSTERS; n++) {
			KERNEL$VM_PREPARE_PAGE_FOR_MMAP(KERNEL$PHYS_2_PAGE(PAGES[i].physical + (n << __PAGE_CLUSTER_BITS)));
		}
		if (OSS_FLAGS & OSS_DUPLEX) {
			car.nclusters = PAGE_CLUSTERS;
			car.align = page_size - 1;
#ifdef OSS_ISA_DMA
			car.flags = CARF_ISADMA | CARF_PHYSCONTIG;
#else
#ifdef OSS_DMA_CAPABLE_64
			if (OSS_DMA_USE_64) car.flags = CARF_DATA | CARF_PHYSCONTIG; else
#endif
			car.flags = CARF_PCIDMA | CARF_DATA | CARF_PHYSCONTIG;
#endif
			SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
			if (car.status < 0) {
#ifdef OSS_CACHEMODE
				err2:
#endif
#ifdef OSS_PCI_DMA
#ifdef OSS_DMA_CAPABLE_64
				if (OSS_DMA_USE_64) PAGES[i].u.unlock64(PAGES[i].device); else
#endif
				PAGES[i].u.unlock32(PAGES[i].device);
#endif
#ifdef OSS_CACHEMODE
				KERNEL$SET_MEMORY_PAT(PAGES[i].physical, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_WB);
#endif
				KERNEL$VM_RELEASE_CONTIG_AREA(PAGES[i].virtual, PAGE_CLUSTERS);
				if (!i) {
					KERNEL$UNIVERSAL_FREE(PAGES);
					PAGES = CAP_PAGES = NULL;
					return car.status;
				}
				break;
			}
			memset(car.ptr, 0, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS);
			CAP_PAGES[i].virtual = car.ptr;
			CAP_PAGES[i].physical = KERNEL$VIRT_2_PHYS(car.ptr);
#ifdef OSS_CACHEMODE
			if (__unlikely(car.status = KERNEL$SET_MEMORY_PAT(CAP_PAGES[i].physical, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS, OSS_CACHEMODE))) {
				KERNEL$VM_RELEASE_CONTIG_AREA(CAP_PAGES[i].virtual, PAGE_CLUSTERS);
				goto err2;
			}
#endif
#ifdef OSS_PCI_DMA
			desc.ptr = CAP_PAGES[i].physical;
			desc.len = PAGE_CLUSTERS << __PAGE_CLUSTER_BITS;
			desc.vspace = &KERNEL$PHYSICAL;
#ifdef OSS_DMA_CAPABLE_64
			if (!OSS_DMA_USE_64) {
#endif
				RAISE_SPL(SPL_VSPACE);
				dma = KERNEL$PHYSICAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &CAP_PAGES[i].u.unlock32);
				LOWER_SPLX(spl);
				if (__unlikely(dma.len != PAGE_CLUSTERS << __PAGE_CLUSTER_BITS)) {
					KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
				}
				CAP_PAGES[i].device = dma.ptr;
#ifdef OSS_DMA_CAPABLE_64
			} else {
				RAISE_SPL(SPL_VSPACE);
				KERNEL$PHYSICAL.op->vspace_dma64lock(&desc, PF_READ | PF_WRITE, &dma64, &CAP_PAGES[i].u.unlock64);
				LOWER_SPLX(spl);
				if (__unlikely(dma64.len != PAGE_CLUSTERS << __PAGE_CLUSTER_BITS)) {
					KERNEL$SUICIDE("OSS_ALLOC_DMAPAGES: %s: CAN'T LOCK DMA", dev_name);
				}
				CAP_PAGES[i].device = dma64.ptr;
			}
#endif
#endif
		}
	}
	N_PAGES = i;
	if (__unlikely(N_PAGES != n_pages)) {
		OSS_FREE_DMAPAGES();
		return car.status;
	}
	return 0;
}

void OSS_FREE_DMAPAGES(void)
{
#ifdef OSS_PCI_DMA
	int spl = KERNEL$SPL;
#endif
	int i, n;
	for (i = 0; i < N_PAGES; i++) {
		for (n = 0; n < PAGE_CLUSTERS; n++) {
			WQ *wq;
			PAGE *p = KERNEL$PHYS_2_PAGE(PAGES[i].physical + (n << __PAGE_CLUSTER_BITS));
			while (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
				WQ_WAIT_SYNC(wq);
				LOWER_SPL(SPL_ZERO);
			}
			LOWER_SPL(SPL_ZERO);
		}
#ifdef OSS_PCI_DMA
		RAISE_SPL(SPL_VSPACE);
#ifdef OSS_DMA_CAPABLE_64
		if (OSS_DMA_USE_64) PAGES[i].u.unlock64(PAGES[i].device); else
#endif
		PAGES[i].u.unlock32(PAGES[i].device);
		LOWER_SPLX(spl);
#endif
#ifdef OSS_CACHEMODE
		KERNEL$SET_MEMORY_PAT(PAGES[i].physical, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_WB);
#endif
		KERNEL$VM_RELEASE_CONTIG_AREA(PAGES[i].virtual, PAGE_CLUSTERS);
		if (OSS_FLAGS & OSS_DUPLEX) {
#ifdef OSS_PCI_DMA
			RAISE_SPL(SPL_VSPACE);
#ifdef OSS_DMA_CAPABLE_64
			if (OSS_DMA_USE_64) CAP_PAGES[i].u.unlock64(PAGES[i].device); else
#endif
			CAP_PAGES[i].u.unlock32(CAP_PAGES[i].device);
			LOWER_SPLX(spl);
#endif
#ifdef OSS_CACHEMODE
			KERNEL$SET_MEMORY_PAT(CAP_PAGES[i].physical, PAGE_CLUSTERS << __PAGE_CLUSTER_BITS, PAT_WB);
#endif
			KERNEL$VM_RELEASE_CONTIG_AREA(CAP_PAGES[i].virtual, PAGE_CLUSTERS);
		}
	}
	KERNEL$UNIVERSAL_FREE(PAGES);
}

#endif

void OSS_INIT(int duplex)
{
	OSS_FLAGS = duplex ? OSS_DUPLEX : 0;
	OSS_CLR_PLAYBACK();
	OSS_CLR_RECORD();
	oss_playback_stat_underruns = 0;
	oss_record_stat_overruns = 0;
}

