#include <SPAD/PCI.H>
#include <SPAD/PCI_IDS.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/IO.H>
#include <SPAD/TIMER.H>
#include <SYS/SOUNDCARD.H>
#include <ARCH/IRQ.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/BSF.H>
#include <SYS/PARAM.H>

#include "AC97.H"
#include "AC97REG.H"
#include "OSS.H"

#include "VIA82REG.H"

#define MAX_DMADESCS	128
#define MAX_RUN		MIN(PG_SIZE * PG_BANK, 0x00400000)
#define DEFAULT_MEMORY	(1024 * 1024)
#define MAX_MEMORY	MIN((__u64)PG_SIZE * PG_BANK * MAX_DMADESCS, (__u64)0x80000000)

#define AC97_TRIES	3
#define AC97_TIMEOUT	1024
#define AC97_CODEC_READY_TIMEOUT TIMEOUT_JIFFIES(JIFFIES_PER_SECOND * 3 / 4)
#define STOP_CHANNEL_TIMEOUT	TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 100)

#define CODEC	0

#define SPL_SND		SPL_AC97
#define OSS_PCI_DMA
#define OSS_PRECISE_TIMING
#define MIN_CHANNELS	0
#define MAX_CHANNELS	1
#define MIN_RATE	1
#define MAX_RATE	48000
#define MIN_FRAGS	2
#define MAX_FRAGS	MAX_DMADESCS
static __const__ char FORMATS[_FMT_MAX + 1] = { 0, 0, 0, 1, 1, 0, 0, 0,
						0, 0, 0, 0, 0, 0, 0, 0 };
#define DEFAULT_FORMAT	_FMT_S16_LE
#define MIN_FRAGSIZE	128

#define FLAG_8233		0x80
#define FLAG_DXS_MASK		0x0f
#define FLAG_DXS_NO		0x01
#define FLAG_DXS_PREFER_NO	0x02
#define FLAG_DXS_PREFER_YES	0x04
#define FLAG_DXS_FORCE		0x08

static __const__ struct pci_id_s devices[] = {
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x10, "VIA 82C686A", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x11, "VIA 82C686B", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x12, "VIA 82C686C", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x13, "VIA 82C686D", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x14, "VIA 82C686E", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x20, "VIA 82C686H", FLAG_DXS_NO },
	{0x1106, 0x3058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA 82C686 - unknown", FLAG_DXS_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x10, "VIA 8233-Pre", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x20, "VIA 8233C", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x30, "VIA 8233", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x40, "VIA 8233A", FLAG_8233 | FLAG_DXS_NO },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x50, "VIA 8235", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x60, "VIA 8237", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0xff, 0x70, "VIA 8251", FLAG_8233 | FLAG_DXS_PREFER_YES },
	{0x1106, 0x3059, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA 8233 - unknown", FLAG_8233 | FLAG_DXS_PREFER_NO },
	{0},
};

static char dev_name[__MAX_STR_LEN];
static char *chip_name;
static unsigned long via_flags;

static pci_id_t pci_id;
static IO_RANGE range;
static io_t io;

static char play_desc_[sizeof(VIA_DMADESC) * (MAX_DMADESCS + 1) - 1];
static char rec_desc_[sizeof(VIA_DMADESC) * (MAX_DMADESCS + 1) - 1];
static VIA_DMADESC *PLAY_DESC;
static VIA_DMADESC *REC_DESC;
static __u32 PLAY_DEVADDR;
static __u32 REC_DEVADDR;
static vspace_dmaunlock_t *PLAY_UNMAP = NULL;
static vspace_dmaunlock_t *REC_UNMAP = NULL;
static unsigned PLAY_FRAG;
static unsigned REC_FRAG;

static DECL_TIMER(VIA_PLAYBACK_TIMEOUT);
static DECL_TIMER(VIA_RECORD_TIMEOUT);

static AST VIA_IRQ_AST;
static IRQ_CONTROL VIA_IRQ_CTRL;

static __finline__ __u8 READ_8(unsigned reg)
{
	return io_inb(io + reg);
}

static __finline__ __u16 READ_16(unsigned reg)
{
	return io_inw(io + reg);
}

static __finline__ __u32 READ_32(unsigned reg)
{
	return io_inl(io + reg);
}

static __finline__ void WRITE_8(unsigned reg, __u8 val)
{
	io_outb(io + reg, val);
}

static __finline__ void WRITE_16(unsigned reg, __u16 val)
{
	io_outw(io + reg, val);
}

static __finline__ void WRITE_32(unsigned reg, __u32 val)
{
	io_outl(io + reg, val);
}

static __u8 playback_channel = VIA_OUT_CHANNEL_BASE;

#define VIA_CHANNEL(rec)	(__likely(!(rec)) ? playback_channel : __unlikely(!(via_flags & FLAG_8233)) ? VIA_IN_CHANNEL_BASE : VIA_NEW_IN0_CHANNEL_BASE)
#define VIA_CHANNEL_MODE(ch)	((__unlikely(!(via_flags & FLAG_8233))) ? 0 : (ch) == VIA_NEW_3D_CHANNEL_BASE ? 2 : 1)

static int USE_DXS(int channels, int rate)
{
	if (via_flags & FLAG_DXS_NO) return 0;
	if (via_flags & FLAG_DXS_FORCE) return 1;
	if (via_flags & FLAG_DXS_PREFER_YES) return channels <= 1;
	return channels <= 1 && !(ac97_xtd_id & AC_XTD_ID_VRA) && rate != 48000;
}

void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
	int dxs;
	open_flags &= _O_KRNL_READ | _O_KRNL_WRITE;
	dxs = USE_DXS(FORMAT_CHANNELS(h), FORMAT_RATE(h));
	if (__likely(dxs)) {
		if (__unlikely(FORMAT_CHANNELS(h) > 1)) {
			FORMAT_SETCHANNELS(h, 1);
		}
		if (__unlikely(!open_flags) || __likely(open_flags == _O_KRNL_WRITE)) {
			return;
		}
		open_flags &= ~_O_KRNL_WRITE;
	}
	AC97_FIXUP_RATE(h, open_flags);
}

static void STOP_CHANNEL(unsigned ch)
{
	u_jiffies_lo_t j, jj;
	__u8 flag;
	if (READ_8(ch + VIA_STATUS) & VIA_STATUS_ACTIVE) {
		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_SGD_TERMINATE);
		j = KERNEL$GET_JIFFIES_LO();
		while (jj = KERNEL$GET_JIFFIES_LO(), READ_8(ch + VIA_STATUS) & VIA_STATUS_ACTIVE) {
			if (jj - j > STOP_CHANNEL_TIMEOUT) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNABLE TO STOP CHANNEL %02X, STATUS %02X", ch, READ_8(ch + VIA_STATUS));
				break;
			}
		}
	} else {
		WRITE_8(ch + VIA_CONTROL, 0);
	}
	if (ch < VIA_NEW_3D_CHANNEL_BASE && via_flags & FLAG_8233) {
		WRITE_8(ch + VIA_LEFTVOL, 0x3f);
		WRITE_8(ch + VIA_RIGHTVOL, 0x3f);
	} else {
		WRITE_8(ch + VIA_TYPE, 0x00);
	}
	flag = READ_8(ch + VIA_STATUS);
	if (flag) WRITE_8(ch + VIA_STATUS, flag);
	KERNEL$UDELAY(20);
}

void SND_START(int record)
{
	unsigned frags, fragsize_bits;
	unsigned rate, channels, format;
	unsigned i;
	VIA_DMADESC *d;
	struct page_desc *pages;
	VDESC v;
	VDMA dma;
	unsigned ch;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_SND)))
		KERNEL$SUICIDE("SND_START AT SPL %08X", KERNEL$SPL);
	SND_STOP(record);
	if (__likely(!record)) {
		if (!(via_flags & FLAG_8233)) {
			playback_channel = VIA_OUT_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(oss_playback_rate);
		} else if (!USE_DXS(oss_playback_channels, oss_playback_rate)) {
			playback_channel = VIA_NEW_3D_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(oss_playback_rate);
		} else {
			playback_channel = VIA_NEW_DXS0_CHANNEL_BASE;
			AC97_SET_PLAYBACK_RATE(48000);
		}
		frags = oss_playback_frags;
		fragsize_bits = oss_playback_fragsize_bits;
		rate = oss_playback_rate;
		channels = oss_playback_channels;
		format = oss_playback_format;
		d = PLAY_DESC;
		pages = PAGES;
	} else {
		AC97_SET_RECORD_RATE(oss_record_rate);
		frags = oss_record_frags;
		fragsize_bits = oss_record_fragsize_bits;
		rate = oss_record_rate;
		channels = oss_record_channels;
		format = oss_record_format;
		d = REC_DESC;
		pages = CAP_PAGES;
	}
	/*__debug_printf("pages: %p\n", pages);*/
	for (i = 0; i < frags; i++, d++) {
		/*__debug_printf("%x : %08x\n", i, pages[i << fragsize_bits >> PAGESIZE_BITS].device);*/
		d->base = __32CPU2LE(pages[i << fragsize_bits >> PAGESIZE_BITS].device + ((i << fragsize_bits) & ((1 << PAGESIZE_BITS) - 1)));
		d->len_flags = __32CPU2LE((1 << fragsize_bits) | (__unlikely(i == frags - 1) ? DESC_LEN_EOL : DESC_LEN_FLAG));
	}
	if (__likely(!record)) {
		v.ptr = (unsigned long)PLAY_DESC;
		v.len = sizeof(VIA_DMADESC) * frags;
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		dma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ, &PLAY_UNMAP);
		LOWER_SPL(SPL_SND);
		if (__unlikely(dma.len != v.len))
			KERNEL$SUICIDE("SND_START: CAN'T DMALOCK BUFFER LIST");
		PLAY_DEVADDR = dma.ptr;
	} else {
		v.ptr = (unsigned long)REC_DESC;
		v.len = sizeof(VIA_DMADESC) * frags;
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		dma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ, &REC_UNMAP);
		LOWER_SPL(SPL_SND);
		if (__unlikely(dma.len != v.len))
			KERNEL$SUICIDE("SND_START: CAN'T DMALOCK BUFFER LIST");
		REC_DEVADDR = dma.ptr;
	}
	ch = VIA_CHANNEL(record);
	WRITE_32(ch + VIA_ADDR, dma.ptr);

	if (__unlikely(!VIA_CHANNEL_MODE(ch))) {
		__u8 type = VIA_TYPE_INT_FLAG | VIA_TYPE_INT_EOL;
		if (__likely(!record)) type |= VIA_TYPE_INT_LSAMPLE;
		else type |= VIA_TYPE_INT_LLINE | VIA_TYPE_REC_FIFO;
		if (__likely(format == _FMT_S16_LE)) type |= VIA_TYPE_16_BIT;
		type |= channels << __BSF_CONST(VIA_TYPE_STEREO);
		WRITE_8(ch + VIA_TYPE, type);
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_SGD_TRIGGER);
	} else if (__likely(VIA_CHANNEL_MODE(ch) == 1)) {
		__u32 stoprate = 0;
		if (__likely(!record)) {
			stoprate = (__u64)rate * 0x100000 / 48000;
			if (stoprate > VIA_STOPRATE_SAMPLE_RATE) stoprate = VIA_STOPRATE_SAMPLE_RATE;
		}
		if (__likely(format == _FMT_S16_LE)) stoprate |= VIA_STOPRATE_16_BIT;
		stoprate |= channels << __BSF_CONST(VIA_STOPRATE_STEREO);
		stoprate |= VIA_STOPRATE_STOP_IDX;
		WRITE_32(ch + VIA_STOPRATE, stoprate);

		if (record) {
			WRITE_8(ch + VIA_TYPE, VIA_TYPE_REC_FIFO);
		}
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_INT_ON_FLAG | VIA_CONTROL_INT_ON_EOL | VIA_CONTROL_AUTO_RESTART | VIA_CONTROL_SGD_TRIGGER);

		if (!record) {
			WRITE_8(ch + VIA_LEFTVOL, 0x00);
			WRITE_8(ch + VIA_RIGHTVOL, 0x00);
		}
	} else {
		__u8 type;
		__u32 stoprate;
		stoprate = VIA_STOPRATE_STOP_IDX;
		switch (channels + 1) {
			case 1: stoprate |= (1<<0) | (1<<4); break;
			case 2: stoprate |= (1<<0) | (2<<4); break;
			case 3: stoprate |= (1<<0) | (2<<4) | (5<<8); break;
			case 4: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12); break;
			case 5: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12) | (5<<16); break;
			case 6: stoprate |= (1<<0) | (2<<4) | (3<<8) | (4<<12) | (5<<16) | (6<<20); break;
			default: KERNEL$SUICIDE("SND_START: BAD CHANNEL NUMBER: %d", channels);
		}
		WRITE_32(ch + VIA_STOPRATE, stoprate);
		type = (channels + 1) << __BSF_CONST(VIA_CH4_TYPE_CHANNELS);
		if (__likely(format == _FMT_S16_LE)) type |= VIA_CH4_TYPE_16BIT;
		WRITE_8(ch + VIA_TYPE, type);
		KERNEL$UDELAY(20);

		WRITE_8(ch + VIA_CONTROL, VIA_CONTROL_INT_ON_FLAG | VIA_CONTROL_INT_ON_EOL | VIA_CONTROL_AUTO_RESTART | VIA_CONTROL_SGD_TRIGGER);
	}
	if (!record) {
		PLAY_FRAG = 0;
		OSS_FLAGS |= OSS_PLAYBACK;
		KERNEL$DEL_TIMER(&VIA_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &VIA_PLAYBACK_TIMEOUT);
	} else {
		REC_FRAG = 0;
		OSS_FLAGS |= OSS_RECORD;
		KERNEL$DEL_TIMER(&VIA_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &VIA_RECORD_TIMEOUT);
	}
}

void SND_STOP(int record)
{
	STOP_CHANNEL(VIA_CHANNEL(record));
	if (!record) {
		RAISE_SPL(SPL_VSPACE);
		if (__likely(PLAY_UNMAP != NULL)) PLAY_UNMAP(PLAY_DEVADDR), PLAY_UNMAP = NULL;
		LOWER_SPL(SPL_SND);
		if (OSS_FLAGS & OSS_PLAYBACK) {
			OSS_FLAGS &= ~(OSS_PLAYBACK | OSS_PLAYBACK_STOP | OSS_LOOP_PLAYBACK);
		}
		KERNEL$DEL_TIMER(&VIA_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&VIA_PLAYBACK_TIMEOUT.list);
	} else {
		RAISE_SPL(SPL_VSPACE);
		if (__likely(REC_UNMAP != NULL)) REC_UNMAP(REC_DEVADDR), REC_UNMAP = NULL;
		LOWER_SPL(SPL_SND);
		OSS_FLAGS &= ~OSS_RECORD;
		KERNEL$DEL_TIMER(&VIA_RECORD_TIMEOUT);
		VOID_LIST_ENTRY(&VIA_RECORD_TIMEOUT.list);
	}
}

void SND_PAUSE(int record)
{
	unsigned ch;
	__u8 ctrl;
	if (!record) {
		KERNEL$DEL_TIMER(&VIA_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&VIA_PLAYBACK_TIMEOUT.list);
	} else {
		KERNEL$DEL_TIMER(&VIA_RECORD_TIMEOUT);
		VOID_LIST_ENTRY(&VIA_RECORD_TIMEOUT.list);
	}
	ch = VIA_CHANNEL(record);
	ctrl = READ_8(ch + VIA_CONTROL);
	WRITE_8(ch + VIA_CONTROL, (ctrl & ~(VIA_CONTROL_SGD_TERMINATE | VIA_CONTROL_SGD_TRIGGER)) | VIA_CONTROL_SGD_PAUSE);
}

void SND_RESUME(int record)
{
	unsigned ch;
	__u8 ctrl;
	if (!record) {
		KERNEL$DEL_TIMER(&VIA_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &VIA_PLAYBACK_TIMEOUT);
	} else {
		KERNEL$DEL_TIMER(&VIA_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &VIA_RECORD_TIMEOUT);
	}
	ch = VIA_CHANNEL(record);
	ctrl = READ_8(ch + VIA_CONTROL);
	WRITE_8(ch + VIA_CONTROL, ctrl & ~(VIA_CONTROL_SGD_TERMINATE | VIA_CONTROL_SGD_TRIGGER | VIA_CONTROL_SGD_PAUSE));
}

unsigned SND_GET_PTR(int record)
{
	unsigned val;
	unsigned ch;
	unsigned fragsize_bits, frags;
	__u32 ptr;
	if (!record) {
		frags = oss_playback_frags;
		fragsize_bits = oss_playback_fragsize_bits;
	} else {
		frags = oss_record_frags;
		fragsize_bits = oss_record_fragsize_bits;
	}
	ch = VIA_CHANNEL(record);
	ptr = READ_32(ch + VIA_BLOCK_COUNT);
	val = (ptr >> (__BSF_CONST(VIA_BLOCK_COUNT_INDEX) - fragsize_bits)) + (1 << fragsize_bits) - (ptr & VIA_BLOCK_COUNT_OFFSET);
	if (__unlikely(val >= frags << fragsize_bits)) val = 0;
	return val;
}

__finline__ void SND_SUBMIT_FRAGMENT(int record, int frag)
{
}

static void INTERRUPT(int record);

DECL_AST(VIA_IRQ, SPL_SND, AST)
{
	INTERRUPT(0);
	INTERRUPT(1);
	VIA_IRQ_CTRL.eoi();
	RETURN;
}

static void INTERRUPT(int record)
{
	__u32 pos;
	__u8 frag;
	unsigned frags;
	unsigned ch = VIA_CHANNEL(record);
	__u8 status = READ_8(ch + VIA_STATUS);
	if (__likely(!(status & (VIA_STATUS_EOL | VIA_STATUS_FLAG | VIA_STATUS_STOPPED)))) return;
	WRITE_8(ch + VIA_STATUS, status & (VIA_STATUS_EOL | VIA_STATUS_FLAG | VIA_STATUS_STOPPED));
	pos = READ_32(ch + VIA_BLOCK_COUNT);
	frag = (pos >> __BSF_CONST(VIA_BLOCK_COUNT_INDEX)) + !(pos & VIA_BLOCK_COUNT_OFFSET);
	if (__likely(!record)) {
		if (!(OSS_FLAGS & OSS_PLAYBACK)) return;
		frags = (oss_playback_frags + frag - PLAY_FRAG) % oss_playback_frags;
		PLAY_FRAG = frag;
		frags = 1;
		if (__likely(frags)) {
			if (__likely(!(OSS_FLAGS & OSS_BLOCK_PLAYBACK))) {
				KERNEL$DEL_TIMER(&VIA_PLAYBACK_TIMEOUT);
				KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &VIA_PLAYBACK_TIMEOUT);
			}
			do {
				if (__unlikely(OSS_PLAYBACK_INTR())) return;
			} while (__unlikely(--frags));
		}
	} else {
		if (!(OSS_FLAGS & OSS_RECORD)) return;
		frags = (oss_record_frags + frag - REC_FRAG) % oss_record_frags;
		REC_FRAG = frag;
		if (__likely(frags)) {
			if (__likely(!(OSS_FLAGS & OSS_BLOCK_RECORD))) {
				KERNEL$DEL_TIMER(&VIA_RECORD_TIMEOUT);
				KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &VIA_RECORD_TIMEOUT);
			}
			do {
				if (__unlikely(OSS_RECORD_INTR())) return;
			} while (__unlikely(--frags));
		}
	}
}

static void VIA_PLAYBACK_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	VOID_LIST_ENTRY(&t->list);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "PLAYBACK INTERRUPT TIMEOUT");
	OSS_STOP(0);
}

static void VIA_RECORD_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_SND);
	VOID_LIST_ENTRY(&t->list);
	/*__debug_printf("%02x %02x %02x %08x %08x\n", READ_8(VIA_CHANNEL(1) + VIA_STATUS), READ_8(VIA_CHANNEL(1) + VIA_CONTROL), READ_8(VIA_CHANNEL(1) + VIA_TYPE), READ_32(VIA_CHANNEL(1) + VIA_STOPRATE), READ_32(VIA_CHANNEL(1) + VIA_BLOCK_COUNT));*/
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "RECORD INTERRUPT TIMEOUT");
	OSS_STOP(1);
}

static int read_ac97(unsigned reg)
{
	unsigned i;
	__u32 x;
	for (i = 0; i < AC97_TRIES * AC97_TIMEOUT; i++) {
		if (__likely(!(i % AC97_TIMEOUT))) {
			x = VIA_AC97_CTRL_READ | (reg << __BSF_CONST(VIA_AC97_CTRL_ADDRESS));
			if (!CODEC) x |= VIA_AC97_CTRL_1ST_CODEC_VALID;
			else x |= VIA_AC97_CTRL_2ND_CODEC | VIA_AC97_CTRL_2ND_CODEC_VALID;
			WRITE_32(VIA_AC97_CTRL, x);
			KERNEL$UDELAY(20);
		}
		KERNEL$UDELAY(1);
		x = READ_32(VIA_AC97_CTRL);
		if (x & VIA_AC97_CTRL_BUSY) continue;
		if (!(x & (!CODEC ? VIA_AC97_CTRL_1ST_CODEC_VALID : VIA_AC97_CTRL_2ND_CODEC_VALID))) continue;
		goto got_it;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d READ TIMED OUT, REGISTER %02X, STATUS %08X", CODEC, reg, x);
	return -1;

	got_it:
	KERNEL$UDELAY(25);
	x = READ_32(VIA_AC97_CTRL);
	if (x & VIA_AC97_CTRL_BUSY ||
	    !(x & (!CODEC ? VIA_AC97_CTRL_1ST_CODEC_VALID : VIA_AC97_CTRL_2ND_CODEC_VALID)) ||
	    (x & VIA_AC97_CTRL_ADDRESS) >> __BSF_CONST(VIA_AC97_CTRL_ADDRESS) != reg) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d HAS INVALID STATUS WHEN READING REGISTER %02X: %08X", CODEC, reg, x);
		return -1;
	}
	return x & VIA_AC97_CTRL_VALUE;
}

static void write_ac97(unsigned reg, int val)
{
	unsigned i;
	__u32 x = (reg << __BSF_CONST(VIA_AC97_CTRL_ADDRESS)) | val;
	if (CODEC) x |= VIA_AC97_CTRL_2ND_CODEC;
	WRITE_32(VIA_AC97_CTRL, x);
	KERNEL$UDELAY(10);
	for (i = 0; i < AC97_TIMEOUT; i++) {
		KERNEL$UDELAY(1);
		if (!(READ_32(VIA_AC97_CTRL) & VIA_AC97_CTRL_BUSY)) return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "CODEC %d WRITE TIMED OUT, REGISTER %02X, VALUE %04X", CODEC, reg, val);
}

static __const__ HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_OPEN,		/* open */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

#include "OSS.I"

#include "AC97.I"

static int RESET_AC97(void)
{
	u_jiffies_lo_t j, jj;
	__u32 ac97_ctrl;
	__u8 ac_stat = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_STATUS);
	if (!(ac_stat & (!CODEC ? VIA_PCI_AC_STATUS_C00_READY : VIA_PCI_AC_STATUS_C01_READY))) {
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_SYNC);
		KERNEL$UDELAY(100);
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, 0);
		KERNEL$UDELAY(100);
		enable_it:
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL, VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_VRA | VIA_PCI_AC_CTRL_PCM);
		KERNEL$UDELAY(100);
	} else {
		__u8 ac_ctrl = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL);
		if (ac_ctrl != (VIA_PCI_AC_CTRL_ENABLE | VIA_PCI_AC_CTRL_RESET | VIA_PCI_AC_CTRL_VRA | VIA_PCI_AC_CTRL_PCM)) goto enable_it;
	}
	j = KERNEL$GET_JIFFIES_LO();
	while (jj = KERNEL$GET_JIFFIES_LO(), !((ac_stat = PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_STATUS)) & (!CODEC ? VIA_PCI_AC_STATUS_C00_READY : VIA_PCI_AC_STATUS_C01_READY))) {
		if (jj - j > AC97_CODEC_READY_TIMEOUT) {
			break;
		}
		KERNEL$SLEEP(1);
	}
	ac97_ctrl = READ_32(VIA_AC97_CTRL);
	if (ac97_ctrl & VIA_AC97_CTRL_BUSY) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "CODEC NOT READY AFTER RESET, STATUS %02X, CTRL %02X, AC97_CTRL %08X", ac_stat, PCI$READ_CONFIG_BYTE(pci_id, VIA_PCI_AC_CTRL), ac97_ctrl);
		return -ENXIO;
	}
	if (!(via_flags & FLAG_8233)) {
		PCI$WRITE_CONFIG_BYTE(pci_id, VIA_PCI_FN_NMI_CTRL, VIA_PCI_FN_NMI_CTRL_IRQ_SMI);
		WRITE_32(VIA_GPI_INT_ENABLE, 0);
	}
	return 0;
}

static void PCM_RESET(void)
{
	unsigned ch;
	for (ch = 0; ch < (via_flags & FLAG_8233 ? 0x80 : 0x30); ch++) {
		if (ch != VIA_NEW_FM_CHANNEL_BASE) STOP_CHANNEL(ch);
	}
}

static void *lnte, *dlrq;
static int VIA_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	int memory = -1;
	int dxs = -1;
	unsigned pages, page_size;
	int irq;
	struct __param_table params[] = {
		"MEMORY", __PARAM_INT, 0, MAX_DMADESCS * MAX_RUN + 1, NULL,
		"DXS", __PARAM_EXTD, 0, 0, NULL,
		NULL, 0, 0, 0, NULL
	};
	struct __param_table dxs_params[] = {
		"NO", __PARAM_BOOL, ~0, FLAG_DXS_NO, NULL,
		"PREFER_NO", __PARAM_BOOL, ~0, FLAG_DXS_PREFER_NO, NULL,
		"PREFER_YES", __PARAM_BOOL, ~0, FLAG_DXS_PREFER_YES, NULL,
		"FORCE", __PARAM_BOOL, ~0, FLAG_DXS_FORCE, NULL,
		NULL, 0, 0, 0, NULL
	};
	char **arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char *opt, *optend, *str;
	int r;
	char *m;
	char gstr[__MAX_STR_LEN];
	DEVICE_REQUEST devrq;
	params[0].__min = __PAGE_CLUSTER_SIZE;
	params[0].__p = &memory;
	params[1].__p = dxs_params;
	dxs_params[0].__p = &dxs;
	dxs_params[1].__p = &dxs;
	dxs_params[2].__p = &dxs;
	dxs_params[3].__p = &dxs;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (memory < 0) {
		memory = MAX_DMADESCS * __PAGE_CLUSTER_SIZE;
		if (memory > DEFAULT_MEMORY) memory = DEFAULT_MEMORY;
	} else {
		if (memory < __PAGE_CLUSTER_SIZE || (memory - 1) || memory > MAX_MEMORY) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: BAD MEMORY SIZE (MUST BE POWER OF 2 AND BETWEEN %u AND %u)", (unsigned)__PAGE_CLUSTER_SIZE, (unsigned)MAX_MEMORY);
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(devices, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &via_flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "VIA82: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	pci_id = id;
	_snprintf(dev_name, sizeof(dev_name), "SND$VIA82@" PCI_ID_FORMAT, id);
	if (dxs >= 0) via_flags = (via_flags & ~FLAG_DXS_MASK) | dxs;
	if (!(via_flags & FLAG_8233) && via_flags & FLAG_DXS_MASK & ~FLAG_DXS_NO) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCAPABILITY, dev_name, "VIA82C686 DOESN'T HAVE DXS");
		via_flags = (via_flags & ~FLAG_DXS_MASK) | FLAG_DXS_NO;
	}
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	io = PCI$READ_IO_RESOURCE(id, 0);
	irq = PCI$READ_INTERRUPT_LINE(id);
	if (!io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	range.start = io;
	range.len = VIA82_REGSPACE;
	range.name = dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, range.start, range.start + range.len - 1, strerror(-r));
		goto ret1;
	}
	r = RESET_AC97();
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T RESET AC97 LINK", dev_name);
		goto ret15;
	}
	r = AC97_INIT();
	if (r) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T INITIALIZE AC97 CODEC", dev_name);
		goto ret15;
	}
	PCM_RESET();
	OSS_INIT(1);
	pages = MAX_DMADESCS;
	page_size = memory / MAX_DMADESCS;
	while (page_size < __PAGE_CLUSTER_SIZE) {
		page_size <<= 1;
		pages >>= 1;
	}
	if ((r = OSS_ALLOC_DMAPAGES(pages, page_size))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC DMA PAGES: %s", dev_name, strerror(-r));
		goto ret15;
	}

	PLAY_DESC = (VIA_DMADESC *)(((unsigned long)play_desc_ + 7) & ~7);
	REC_DESC = (VIA_DMADESC *)(((unsigned long)rec_desc_ + 7) & ~7);

	VOID_LIST_ENTRY(&VIA_PLAYBACK_TIMEOUT.list);
	VIA_PLAYBACK_TIMEOUT.fn = VIA_PLAYBACK_TIMEOUT_FN;
	VOID_LIST_ENTRY(&VIA_RECORD_TIMEOUT.list);
	VIA_RECORD_TIMEOUT.fn = VIA_RECORD_TIMEOUT_FN;

	VIA_IRQ_AST.fn = VIA_IRQ;
	if ((m = KERNEL$REQUEST_IRQ(irq, IRQ_SHARED | IRQ_AST_HANDLER, NULL, &VIA_IRQ_AST, &VIA_IRQ_CTRL, dev_name))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IRQ %d: %s", dev_name, irq, m);
		r = -EBUSY;
		goto ret2;
	}

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", dev_name, io, irq);

	devrq.name = dev_name;
	devrq.driver_name = "VIA82.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = OSS_SET_ROOT;
	devrq.dev_ptr = NULL;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = VIA_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret3;
	}

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, dev_name);
	lnte = devrq.lnte;
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	ret3:
	KERNEL$RELEASE_IRQ(&VIA_IRQ_CTRL, 0);
	ret2:
	OSS_FREE_DMAPAGES();
	ret15:
	KERNEL$UNREGISTER_IO_RANGE(&range);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int VIA_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	VIA_IRQ_CTRL.mask();
	RAISE_SPL(SPL_SND);
	OSS_STOP(0);
	OSS_STOP(1);
	LOWER_SPL(SPL_ZERO);
	KERNEL$RELEASE_IRQ(&VIA_IRQ_CTRL, IRQ_RELEASE_MASKED);
	OSS_FREE_DMAPAGES();
	KERNEL$UNREGISTER_IO_RANGE(&range);
	PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
