#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV_KRNL.H>
#include <SYS/SOUNDCARD.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <SPAD/ISADMA.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>

#include "WSSREG.H"

#define OSS_ISA_DMA
#define OSS_FIXED_RATES
#define OSS_PRECISE_TIMING
#define OSS_MIXER
#define OSS_MIXER_CAPS	0
static unsigned OSS_MIXER_DEVMASK = SOUND_MASK_PCM | SOUND_MASK_LINE | SOUND_MASK_CD;
#define OSS_MIXER_STEREODEVS (OSS_MIXER_DEVMASK & ~SOUND_MIXER_SPEAKER)
static unsigned OSS_MIXER_RECMASK = SOUND_MASK_LINE | SOUND_MASK_CD;
static unsigned OSS_MIXER_OUTMASK = SOUND_MASK_LINE | SOUND_MASK_CD;
#define SPL_SND		SPL_WSS

#include "OSS.H"

static __const__ int MODE1_RATELIST[17] = {
		8000, 5510, 16000, 11025,
		27420, 18900, 32000, 22050,
		-1, 37800, -1, 44100,
		48000, 33075, 9600, 6620,
		0 };

/* rate2.c:

#include <stdio.h>

void main(void)
{
	int i;
	for (i = 0; i < 512; i++) {
		int cs2 = i & 1;
		int n = (i >> 1) & 63;
		int osm = i >> 7;
		int m = osm == 0 ? 128 : osm == 1 ? 64 : osm == 2 ? 256 : 0;
		int xt = !cs2 ? 24576000 : 16934400;
		int f;
		if (osm == 3) {
			n:
			printf("-1, ");
			continue;
		}
		if (!cs2) if (n < 16 || n > 49) goto n;
		if (cs2) if (n < 12 || n > 33) goto n;
		f = (2 * xt + m * n / 2) / (m * n);
		printf("%d, ", f);
	}
}
*/

static __const__ int MODE2_RATELIST[] = {
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, 22050, -1, 20354, -1, 18900, -1, 17640,
		24000, 16538, 22588, 15565, 21333, 14700, 20211, 13926,
		19200, 13230, 18286, 12600, 17455, 12027, 16696, 11504,
		16000, 11025, 15360, 10584, 14769, 10177, 14222, 9800,
		13714, 9450, 13241, 9124, 12800, 8820, 12387, 8535,
		12000, 8269, 11636, 8018, 11294, -1, 10971, -1,
		10667, -1, 10378, -1, 10105, -1, 9846, -1,
		9600, -1, 9366, -1, 9143, -1, 8930, -1,
		8727, -1, 8533, -1, 8348, -1, 8170, -1,
		8000, -1, 7837, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, 44100, -1, 40708, -1, 37800, -1, 35280,
		48000, 33075, 45176, 31129, 42667, 29400, 40421, 27853,
		38400, 26460, 36571, 25200, 34909, 24055, 33391, 23009,
		32000, 22050, 30720, 21168, 29538, 20354, 28444, 19600,
		27429, 18900, 26483, 18248, 25600, 17640, 24774, 17071,
		24000, 16538, 23273, 16036, 22588, -1, 21943, -1,
		21333, -1, 20757, -1, 20211, -1, 19692, -1,
		19200, -1, 18732, -1, 18286, -1, 17860, -1,
		17455, -1, 17067, -1, 16696, -1, 16340, -1,
		16000, -1, 15673, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, 11025, -1, 10177, -1, 9450, -1, 8820,
		12000, 8269, 11294, 7782, 10667, 7350, 10105, 6963,
		9600, 6615, 9143, 6300, 8727, 6014, 8348, 5752,
		8000, 5513, 7680, 5292, 7385, 5088, 7111, 4900,
		6857, 4725, 6621, 4562, 6400, 4410, 6194, 4268,
		6000, 4134, 5818, 4009, 5647, -1, 5486, -1,
		5333, -1, 5189, -1, 5053, -1, 4923, -1,
		4800, -1, 4683, -1, 4571, -1, 4465, -1,
		4364, -1, 4267, -1, 4174, -1, 4085, -1,
		4000, -1, 3918, /*-1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, -1, -1, */
		0 };

/* rate3.c:

#include <stdio.h>

#define F 16934400

void main(void)
{
	int i;
	for (i = 0; i < 193; i++) {
		int d;
		switch (i) {
			case 0: d = 16 * 21; break;
			case 1: d = 353; break;
			case 2: d = 529; break;
			case 3: d = 617; break;
			case 4: d = 1058; break;
			case 5: d = 1764; break;
			case 6: d = 2117; break;
			case 7: d = 2558; break;
			default:
				if (i < 22) {
					printf("-1, ");
					continue;
				}
				d = 16 * i;
		}
		printf("%d, ", (F + (d / 2)) / d);
	}
}
*/

static __const__ int MODE3_RATELIST[194] = {
		/*50400, 47973, 32012, 27446, 16006, 9600, 7999, 6620,*/
		50400, 48000, 32000, 27420, 16000, 9600, 8000, 6620,
		-1, -1, -1, -1, -1, -1, -1, -1,
		-1, -1, -1, -1, -1, -1, 48109, 46017,
		44100, 42336, 40708, 39200, 37800, 36497, 35280, 34142,
		33075, 32073, 31129, 30240, 29400, 28605, 27853, 27138,
		26460, 25815, 25200, 24614, /*24055*/24000, 23520, 23009, 22519,
		22050, 21600, 21168, 20753, 20354, 19970, 19600, 19244,
		18900, 18568, 18248, 17939, 17640, 17351, 17071, 16800,
		16538, 16283, 16036, 15797, 15565, 15339, 15120, 14907,
		14700, 14499, 14303, 14112, 13926, 13745, 13569, 13397,
		13230, 13067, 12907, 12752, 12600, 12452, 12307, 12166,
		12027, 11892, 11760, 11631, 11504, 11381, 11260, 11141,
		11025, 10911, 10800, 10691, 10584, 10479, 10376, 10276,
		10177, 10080, 9985, 9892, 9800, 9710, 9622, 9535,
		9450, 9366, 9284, 9203, 9124, 9046, 8969, 8894,
		8820, 8747, 8675, 8605, 8535, 8467, 8400, 8334,
		8269, 8205, 8142, 8079, 8018, 7958, 7899, 7840,
		7782, 7726, 7670, 7614, 7560, 7506, 7454, 7401,
		7350, 7299, 7249, 7200, 7151, 7103, 7056, 7009,
		6963, 6918, 6873, 6828, 6785, 6741, 6699, 6657,
		6615, 6574, 6533, 6493, 6454, 6415, 6376, 6338,
		6300, 6263, 6226, 6189, 6153, 6118, 6083, 6048,
		6014, 5980, 5946, 5913, 5880, 5848, 5815, 5784,
		5752, 5721, 5690, 5660, 5630, 5600, 5571, 5541,
		5513,
		0 };

static __const__ int *RATELIST;

#define MIN_CHANNELS	0
#define MAX_CHANNELS	1
#define MIN_FRAGS	2
#define MAX_FRAGS	MAXUINT

static __const__ char FORMATS[_FMT_MAX + 1] = { 0, 0, 0, 1, 1, 0, 0, 0,
						0, 0, 0, 0, 0, 0, 0, 0 };

#define DEFAULT_FORMAT	_FMT_S16_LE

#define MIN_FRAGSIZE	16

static char DEV_NAME[__MAX_STR_LEN];
static char *CHIP;

static void *LNTE, *DLRQ;

static AST WSS_IRQ_AST;
static IRQ_CONTROL WSS_IRQ_CTRL;
extern AST_STUB WSS_IRQ;

static io_t IO;
static int IRQ;
static int DMA1;
static int DMA2;

static int MODE;

static int R0;
static int I10;
static int I9;

#define IN_R(n)		(io_inb(IO + (n)))
#define OUT_R(n, v)	(io_outb(IO + (n), v))

#define IN_I(n)		(OUT_R(0, (n) | R0), IN_R(1))
#define OUT_I(n, v)	(OUT_R(0, (n) | R0), OUT_R(1, v))

#define IN_X(n)		(OUT_I(23, ((n & 0xf) << 4) | ((n & 0x10) >> 2) | 0x8), IN_R(1))
#define OUT_X(n, v)	(OUT_I(23, ((n & 0xf) << 4) | ((n & 0x10) >> 2) | 0x8), OUT_R(1, v))

static int playback_trd;
static int record_trd;

static DECL_TIMER(WSS_PLAYBACK_TIMEOUT);
static DECL_TIMER(WSS_RECORD_TIMEOUT);

static IO_RANGE range;

static int WAIT_FOR_READY(int syn)
{
	u_jiffies_t j = KERNEL$GET_JIFFIES_LO(), jj = j;
	while (__likely(IN_R(0) & R0_INIT)) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 10))) return 1;
		if (__unlikely(syn)) KERNEL$SLEEP(1);
		jj = KERNEL$GET_JIFFIES_LO();
	}
	return 0;
}

static int WAIT_FOR_CALIBRATION(int syn)
{
	u_jiffies_t j = KERNEL$GET_JIFFIES_LO(), jj;
	while (jj = KERNEL$GET_JIFFIES_LO(), __likely(IN_R(0) & R0_INIT)) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 10))) return 1;
		if (__unlikely(syn)) KERNEL$SLEEP(1);
	}
	while (jj = KERNEL$GET_JIFFIES_LO(), __likely(IN_I(11) & I11_ACI)) {
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 10))) return 1;
		if (__unlikely(syn)) KERNEL$SLEEP(1);
	}
	return 0;
}

/* enable TRD unconditionally
- disabled TRD may cause scratching with small fragments and it may cause
  corruption of card state when stopping transfer and full duplex is enabled (if
  one operation changes TRD while the other is holding interrupt the will
  terminate it).
- SPAD has very good interrupt latency, so it can service high SPL interrupt
  before card buffer empties. PCI DMA bursts cause much more problems than
  interrupt latency.
- with enabled TRD it plays almost OK with 16-byte fragments on Pentium 200.
  With disabled TRD, it scratched.
*/

static __finline__ void SET_TRD(int record, int trd)
{
	/*
	if (__likely(!record)) playback_trd = trd;
	else record_trd = trd;
	if (playback_trd | record_trd) R0 |= R0_TRD;
	else R0 &= ~R0_TRD;
	OUT_R(0, R0);
	*/
}

void SND_STOP(int record)
{
	u_jiffies_lo_t j, jj;
	/*__debug_printf("snd_stop.\n");*/
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_WSS)))
		KERNEL$SUICIDE("SND_STOP AT SPL %08X", KERNEL$SPL);
	/*SET_TRD(record, 1);*/	/* If this line is not here, I get playback
		interrupt timeout on subsequent playback. I don't know reason
		for this. */
		/* ... maybe wait for low prdy below does make this unnecessary,
		but i leave it here ... */
		/* TRD changing here on the other hand causes timeouts
		--- probably when the card is DMA transmitting with interrupt
		pending and TRD is turned ON. It can still be turned on by
		full-duplex operation */
		/* I enabled TRD unconditionally --- it fixes all problems,
		now I hope this driver is finally BugFree(tm) */
	if (__likely(!record)) {
		/* stopping the card is nasty ... if I just reset I9_PEN and
		   card is in the middle of reading sample, it will see swapped
		   hi/lo left/right bytes producing noise and playback timeout.
		   */
		if (__likely(OSS_FLAGS & OSS_PLAYBACK)) {
			if (__unlikely(!(I9 & I9_PEN))) SND_RESUME(0);
			j = KERNEL$GET_JIFFIES_LO();
			while (1) {
				jj = KERNEL$GET_JIFFIES_LO();
				I9 &= ~I9_PEN;
				OUT_I(9, I9);
				WAIT_FOR_READY(0);
				if (__likely(!(IN_I(11) & I11_DRS)) && __likely(!(IN_R(2) & R2_PRDY))) break;
				if (__unlikely(jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 100))) {
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "TIMED OUT WHEN WAITING FOR LOW DRS FOR PLAYBACK STOPPING");
					break;
				}
				I9 |= I9_PEN;
				OUT_I(9, I9);
				WAIT_FOR_READY(0);
			}
			ISADMA$END_TRANSFER(DMA1);
			OSS_FLAGS &= ~(OSS_PLAYBACK | OSS_PLAYBACK_STOP | OSS_LOOP_PLAYBACK);
		}
		OUT_I(24, IN_I(24) & ~I24_PI);
		KERNEL$DEL_TIMER(&WSS_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&WSS_PLAYBACK_TIMEOUT.list);
		/*__debug_printf("PRDY %d, PLR %d, PUR %d\n", (IN_R(2) / R2_PRDY) & 1, (IN_R(2) / R2_PLR) & 1, (IN_R(2) / R2_PUR) & 1);*/
	} else {
		if (__unlikely(OSS_FLAGS & OSS_RECORD)) {
			if (__unlikely(!(I9 & I9_CEN))) SND_RESUME(1);
			j = KERNEL$GET_JIFFIES_LO();
			while (1) {
				jj = KERNEL$GET_JIFFIES_LO();
				I9 &= ~I9_CEN;
				OUT_I(9, I9);
				WAIT_FOR_READY(0);
				if (__likely(!(IN_I(11) & I11_DRS)) && __likely(!(IN_R(2) & R2_CRDY))) break;
				if (__unlikely(jj - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 100))) {
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "TIMED OUT WHEN WAITING FOR LOW DRS FOR RECORD STOPPING");
					break;
				}
				I9 |= I9_CEN;
				OUT_I(9, I9);
				WAIT_FOR_READY(0);
				KERNEL$UDELAY(1);
			}
			ISADMA$END_TRANSFER(__likely(DMA2 != -1) ? DMA2 : DMA1);
			OSS_FLAGS &= ~OSS_RECORD;
		}
		OUT_I(24, IN_I(24) & ~I24_CI);
		KERNEL$DEL_TIMER(&WSS_RECORD_TIMEOUT);
		VOID_LIST_ENTRY(&WSS_RECORD_TIMEOUT.list);
	}
	SET_TRD(record, 0);
}

void SND_START(int record)
{
	int frags, fragsize_bits;
	int rate, channels, format;
	int dma, s;
	__p_addr pp;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_WSS)))
		KERNEL$SUICIDE("SND_START AT SPL %08X", KERNEL$SPL);
	SND_STOP(record);
	if (DMA2 == -1) OSS_STOP(1 - record);
	if (!record) {
		frags = oss_playback_frags;
		fragsize_bits = oss_playback_fragsize_bits;
		rate = oss_playback_rate;
		channels = oss_playback_channels;
		format = oss_playback_format;
	} else {
		frags = oss_record_frags;
		fragsize_bits = oss_record_fragsize_bits;
		rate = oss_record_rate;
		channels = oss_record_channels;
		format = oss_record_format;
	}
	if (__likely(!record) || __unlikely(DMA2 == -1)) {
		dma = DMA1;
		pp = PAGES[0].physical;
	} else {
		dma = DMA2;
		pp = CAP_PAGES[0].physical;
	}
	if (__unlikely(frags << fragsize_bits <= 65536)) {
		ISADMA$SETUP_TRANSFER(dma, 1 - record, 1, pp, frags << fragsize_bits);
		/*ISADMA$SETUP_AUTOINIT(dma, pp, frags << fragsize_bits);*/
		SET_TRD(record, 0);
	} else {
		ISADMA$SETUP_TRANSFER(dma, 1 - record, 0, pp, 65536);
		if (fragsize_bits < 16) SET_TRD(record, 0);
		else SET_TRD(record, 1);
	}
	if (__unlikely(MODE == 1)) {
		R0 |= R0_MCE;
		OUT_R(0, R0);
		OUT_I(8, (rate & 15) | (channels << 4) | (format == _FMT_U8 ? 0 : I8_16B));
		if (__unlikely(WAIT_FOR_READY(0))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "CODEC FAILED TO SET PARAMETERS");
			R0 &= ~R0_MCE;
			OUT_R(0, R0);
			WAIT_FOR_CALIBRATION(0);
			SND_STOP(record);
			return;
		}
		R0 &= ~R0_MCE;
		OUT_R(0, R0);
		if (__unlikely(WAIT_FOR_CALIBRATION(0))) {
			no_cal:
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "CODEC FAILED TO CALIBRATE");
			SND_STOP(record);
			return;
		}
	} else if (__unlikely(MODE == 2)) {
		int osm = (rate >> 7) & 3;
		int asf = rate | I22_SRE;
		if (__unlikely(asf != IN_I(22)) || __unlikely(osm != ((IN_I(10) >> 4) & 3))) {
			OSS_STOP(1 - record);
			R0 |= R0_MCE;
			OUT_R(0, R0);
			OUT_I(22, asf);
			OUT_I(10, (I10 & 0xcf) | (osm << 4));
			R0 &= ~R0_MCE;
			OUT_R(0, R0);
			if (__unlikely(WAIT_FOR_CALIBRATION(0))) goto no_cal;
		}
		if (__likely(!record)) {
			OUT_I(16, I16_PMCE);
			OUT_I(8, (channels << 4) | (format == _FMT_U8 ? 0 : I8_16B));
		} else {
			OUT_I(16, I16_CMCE);
			OUT_I(28, (channels << 4) | (format == _FMT_U8 ? 0 : I28_16B));
		}
		OUT_I(16, 0);
		if (__unlikely(WAIT_FOR_CALIBRATION(0))) goto no_cal;
	} else if (__likely(MODE == 3)) {
		if (__likely(!record)) {
			OUT_I(16, I16_PMCE);
			OUT_I(8, (channels << 4) | (format == _FMT_U8 ? 0 : I8_16B));
			OUT_X(13, rate);
		} else {
			OUT_I(16, I16_CMCE);
			OUT_I(28, (channels << 4) | (format == _FMT_U8 ? 0 : I28_16B));
			OUT_X(12, rate);
		}
		OUT_I(16, 0);
		if (__unlikely(WAIT_FOR_CALIBRATION(0))) goto no_cal;
	} else KERNEL$SUICIDE("SND_START: MODE %d", MODE);
	s = _FMT_DIV(1 << fragsize_bits, format, channels) - 1;
	if (__likely(!record) || __unlikely(MODE == 1)) {
		OUT_I(15, s & 0xff);
		OUT_I(14, s >> 8);
	} else {
		OUT_I(31, s & 0xff);
		OUT_I(30, s >> 8);
	}
	I9 |= 1 << record;
	OUT_I(9, I9);
	WAIT_FOR_READY(0);
	if (!record) {
		OSS_FLAGS |= OSS_PLAYBACK;
		KERNEL$DEL_TIMER(&WSS_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &WSS_PLAYBACK_TIMEOUT);
		/*__debug_printf("frags %d, fragsize_bits %d, format %d, channels %d, rate %d (%d), timeout %d\n", oss_playback_frags, oss_playback_fragsize_bits, oss_playback_format, oss_playback_channels, oss_playback_rate, OSS_GET_RATE(oss_playback_rate), OSS_PLAYBACK_TIMEOUT);*/
	} else {
		OSS_FLAGS |= OSS_RECORD;
		KERNEL$DEL_TIMER(&WSS_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &WSS_RECORD_TIMEOUT);
		/*__debug_printf("frags %d, fragsize_bits %d, format %d, channels %d, rate %d (%d), timeout %d\n", oss_record_frags, oss_record_fragsize_bits, oss_record_format, oss_record_channels, oss_record_rate, OSS_GET_RATE(oss_record_rate), OSS_RECORD_TIMEOUT);*/
	}
}

void SND_PAUSE(int record)
{
	if (__likely(!record)) {
		KERNEL$DEL_TIMER(&WSS_PLAYBACK_TIMEOUT);
		VOID_LIST_ENTRY(&WSS_PLAYBACK_TIMEOUT.list);
		I9 &= ~I9_PEN;
	} else {
		KERNEL$DEL_TIMER(&WSS_RECORD_TIMEOUT);
		VOID_LIST_ENTRY(&WSS_RECORD_TIMEOUT.list);
		I9 &= ~I9_CEN;
	}
	OUT_I(9, I9);
}

void SND_RESUME(int record)
{
	if (__likely(!record)) {
		KERNEL$DEL_TIMER(&WSS_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &WSS_PLAYBACK_TIMEOUT);
		I9 |= I9_PEN;
	} else {
		KERNEL$DEL_TIMER(&WSS_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &WSS_RECORD_TIMEOUT);
		I9 |= I9_CEN;
	}
	OUT_I(9, I9);
}

__finline__ void SND_SUBMIT_FRAGMENT(int record, int frag)
{
}

__finline__ void SND_FIXUP_PARAMS(HANDLE *h, int open_flags)
{
}

static void WSS_PLAYBACK_INTR(void)
{
	/*__debug_printf("playback intr(%lx).\n", ISADMA$GET_ADDRESS(DMA1));*/
	if (__unlikely(OSS_PLAYBACK_INTR())) return;
	if (__unlikely(!(oss_playback_fragment & ((1 << (16 - oss_playback_fragsize_bits)) - 1))) && __likely(oss_playback_frags << oss_playback_fragsize_bits > 65536)) {
		int res = (oss_playback_frags - oss_playback_fragment) << oss_playback_fragsize_bits;
		if (__likely(res >= 65536)) res = 65536;
		ISADMA$END_TRANSFER(DMA1);
		ISADMA$SETUP_TRANSFER(DMA1, 1, 0, PAGES[oss_playback_fragment >> (16 - oss_playback_fragsize_bits)].physical, res);
	}
	if (__likely(oss_playback_frags << oss_playback_fragsize_bits > 65536)) SET_TRD(0, oss_playback_fragment + 1 == oss_playback_frags || !((oss_playback_fragment + 1) & ((1 << (16 - oss_playback_fragsize_bits)) - 1)));
	if (__likely(!(OSS_FLAGS & OSS_BLOCK_PLAYBACK))) {
		KERNEL$DEL_TIMER(&WSS_PLAYBACK_TIMEOUT);
		KERNEL$SET_TIMER(OSS_PLAYBACK_TIMEOUT, &WSS_PLAYBACK_TIMEOUT);
	}
}

static void WSS_PLAYBACK_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_WSS);
	/*__debug_printf("timeout(%lx).\n", ISADMA$GET_ADDRESS(DMA1));*/
	VOID_LIST_ENTRY(&t->list);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "PLAYBACK INTERRUPT TIMEOUT");
	OSS_STOP(0);
}

static void WSS_RECORD_INTR(void)
{
	int frag;
	int dma;
	/*__debug_printf("record intr(%lx).\n", ISADMA$GET_ADDRESS(DMA2 != -1 ? DMA2 : DMA1));*/
	if (__unlikely(OSS_RECORD_INTR())) return;
	frag = oss_record_fragment + oss_record_buf_fragments;
	if (__unlikely(frag >= oss_record_frags)) frag -= oss_record_frags;
	if (__unlikely(frag >= oss_record_frags))
		KERNEL$SUICIDE("WSS_RECORD_INTR: BAD FRAGMENT. FRAGMENT %d, BUF FRAGMENTS %d", oss_record_fragment, oss_record_buf_fragments);
	if (__unlikely(!(frag & ((1 << (16 - oss_record_fragsize_bits)) - 1))) && __likely(oss_record_frags << oss_record_fragsize_bits > 65536)) {
		int res = (oss_record_frags - frag) << oss_record_fragsize_bits;
		if (__likely(res >= 65536)) res = 65536;
		dma = __likely(DMA2 != -1) ? DMA2 : DMA1;
		ISADMA$END_TRANSFER(dma);
		ISADMA$SETUP_TRANSFER(dma, 0, 0, (OSS_FLAGS & OSS_DUPLEX ? CAP_PAGES : PAGES)[frag >> (16 - oss_record_fragsize_bits)].physical, res);
	/*__debug_printf("da(%lx,%d - %lx).\n", (OSS_FLAGS & OSS_DUPLEX ? PHYSICAL_CAP_PAGES : PHYSICAL_PAGES)[frag >> (16 - oss_record_fragsize_bits)], res, ISADMA$GET_ADDRESS(DMA2 != -1 ? DMA2 : DMA1));*/
	}
	if (__likely(oss_record_frags << oss_record_fragsize_bits > 65536)) SET_TRD(0, frag + 1 == oss_record_frags || !((frag + 1) & ((1 << (16 - oss_record_fragsize_bits)) - 1)));
	if (__likely(!(OSS_FLAGS & OSS_BLOCK_RECORD))) {
		KERNEL$DEL_TIMER(&WSS_RECORD_TIMEOUT);
		KERNEL$SET_TIMER(OSS_RECORD_TIMEOUT, &WSS_RECORD_TIMEOUT);
	}
}

static void WSS_RECORD_TIMEOUT_FN(TIMER *t)
{
	LOWER_SPL(SPL_WSS);
	/*__debug_printf("timeout(%lx).\n", ISADMA$GET_ADDRESS(DMA2 != -1 ? DMA2 : DMA1));*/
	VOID_LIST_ENTRY(&t->list);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "RECORD INTERRUPT TIMEOUT");
	OSS_STOP(1);
}

unsigned SND_GET_PTR(int record)
{
		/* !!! FIXME: stop DMA on VIA chipset --- just like Linux does */
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_WSS)))
		KERNEL$SUICIDE("SND_GET_PTR AT SPL %08X", KERNEL$SPL);
	if (!record) {
		unsigned addr;
		addr = ISADMA$GET_ADDRESS(DMA1) & 0xffff;
		if (__unlikely(!addr) && __likely(__unlikely(MODE == 1) ? __likely(IN_R(2) & R2_INT) : __likely(IN_I(24) & I24_PI))) addr = 0x10000;
		return (oss_playback_fragment >> (16 - oss_playback_fragsize_bits) << 16) + addr;
	} else {
		unsigned addr;
		int frag;
		frag = oss_record_fragment + oss_record_buf_fragments;
		if (__unlikely(frag >= oss_record_frags)) frag -= oss_record_frags;
		addr = ISADMA$GET_ADDRESS(__likely(DMA2 != -1) ? DMA2 : DMA1) & 0xffff;
		if (record_trd && __unlikely(!addr) && __likely(__unlikely(MODE == 1) ? __likely(IN_R(2) & R2_INT) : __likely(IN_I(24) & I24_CI))) addr = 0x10000;
		return (frag >> (16 - oss_record_fragsize_bits) << 16) + addr;
	}
}

DECL_AST(WSS_IRQ, SPL_WSS, AST)
{
	/*__debug_printf("intr.\n");*/
	if (__unlikely(MODE == 1)) {
		int status = IN_R(2);
		if (__unlikely(!(status & R2_INT))) goto eoi;
		if (__likely(OSS_FLAGS & OSS_PLAYBACK)) WSS_PLAYBACK_INTR();
		else if (OSS_FLAGS & OSS_RECORD) WSS_RECORD_INTR();
		status &= ~R2_INT;
		OUT_R(2, status);
	} else if (__unlikely(MODE == 2) || __likely(MODE == 3)) {
		int altstatus = IN_I(24);
		if (__likely(altstatus & I24_PI)) {
			altstatus &= ~I24_PI;
			if (__likely(OSS_FLAGS & OSS_PLAYBACK)) WSS_PLAYBACK_INTR();
			else KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "UNEXPECTED PLAYBACK INTERRUPT");
		}
		if (__unlikely(altstatus & I24_CI)) {
			altstatus &= ~I24_CI;
			if (__likely(OSS_FLAGS & OSS_RECORD)) WSS_RECORD_INTR();
			else KERNEL$SYSLOG(__SYSLOG_HW_BUG, DEV_NAME, "UNEXPECTED RECORD INTERRUPT");
		}
		OUT_I(24, altstatus);
		/*OUT_R(2, IN_R(2) & ~R2_INT);*/
	} else KERNEL$SUICIDE("WSS_IRQ: MODE %d", MODE);
	eoi:
	WSS_IRQ_CTRL.eoi();
	RETURN;
}

void SND_MIXER_WRITE_RECSRC(unsigned long mask)
{
	if (!(mask & SOUND_MASK_MIC)) {
		if (__likely(MODE >= 3)) {
			OUT_X(2, IN_X(2) | X2_LMIM);
			OUT_X(3, IN_X(3) | X3_RMIM);
		}
	} else {
		if (__likely(MODE >= 3)) {
			OUT_X(2, IN_X(2) & ~X2_LMIM);
			OUT_X(3, IN_X(3) & ~X3_RMIM);
		}
	}
	if (!(mask & SOUND_MASK_LINE)) {
		OUT_I(2, IN_I(2) | I2_LX1IM);
		OUT_I(3, IN_I(3) | I3_RX1IM);
	} else {
		OUT_I(2, IN_I(2) & ~I2_LX1IM);
		OUT_I(3, IN_I(3) & ~I3_RX1IM);
	}
	if (!(mask & SOUND_MASK_CD)) {
		OUT_I(4, IN_I(4) | I4_LX2IM);
		OUT_I(5, IN_I(5) | I5_RX2IM);
	} else {
		OUT_I(4, IN_I(4) & ~I4_LX2IM);
		OUT_I(5, IN_I(5) & ~I5_RX2IM);
	}
}

long SND_MIXER_READ_RECSRC(void)
{
	long m = 0;
	if (__likely(MODE >= 3)) {
		if (!(IN_X(2) & X2_LMIM) || !(IN_X(3) & X3_RMIM)) m |= SOUND_MASK_MIC;
	}
	if (!(IN_I(2) & I2_LX1IM) || !(IN_I(3) & I3_RX1IM)) m |= SOUND_MASK_LINE;
	if (!(IN_I(4) & I4_LX2IM) || !(IN_I(5) & I5_RX2IM)) m |= SOUND_MASK_LINE;
	return m;
}

void SND_MIXER_WRITE_OUTSRC(unsigned long mask)
{
	if (!(mask & SOUND_MASK_MIC)) {
		if (__likely(MODE >= 3)) {
			OUT_X(2, IN_X(2) | X2_LMOM);
			OUT_X(3, IN_X(3) | X3_RMOM);
		}
	} else {
		if (__likely(MODE >= 3)) {
			OUT_X(2, IN_X(2) & ~X2_LMOM);
			OUT_X(3, IN_X(3) & ~X3_RMOM);
		}
	}
	if (!(mask & SOUND_MASK_LINE)) {
		OUT_I(2, IN_I(2) | I2_LX1OM);
		OUT_I(3, IN_I(3) | I3_RX1OM);
	} else {
		OUT_I(2, IN_I(2) & ~I2_LX1OM);
		OUT_I(3, IN_I(3) & ~I3_RX1OM);
	}
	if (!(mask & SOUND_MASK_CD)) {
		OUT_I(4, IN_I(4) | I4_LX2OM);
		OUT_I(5, IN_I(5) | I5_RX2OM);
	} else {
		OUT_I(4, IN_I(4) & ~I4_LX2OM);
		OUT_I(5, IN_I(5) & ~I5_RX2OM);
	}
}

long SND_MIXER_READ_OUTSRC(void)
{
	long m = 0;
	if (__likely(MODE >= 3)) {
		if (!(IN_X(2) & X2_LMOM) || !(IN_X(3) & X3_RMOM)) m |= SOUND_MASK_MIC;
	}
	if (!(IN_I(2) & I2_LX1OM) || !(IN_I(3) & I3_RX1OM)) m |= SOUND_MASK_LINE;
	if (!(IN_I(4) & I4_LX2OM) || !(IN_I(5) & I5_RX2OM)) m |= SOUND_MASK_CD;
	return m;
}

static unsigned SCALE(unsigned val, unsigned s_from, unsigned s_to)
{
	return (val * s_to + (s_from >> 1)) / s_from;
}

void SND_MIXER_WRITE(unsigned mix, unsigned long val)
{
	unsigned left = val & 0xff;
	unsigned right = (val >> 8) & 0xff;
	if (__unlikely(left > 100)) left = 100;
	if (__unlikely(right > 100)) right = 100;
	if (mix == SOUND_MIXER_PCM) {
		if (__unlikely(!left)) OUT_I(6, I6_LD1OM | 0x3f);
		else OUT_I(6, 0x3f - SCALE(left - 1, 99, 0x3f));
		if (__unlikely(!right)) OUT_I(7, I7_RD1OM | 0x3f);
		else OUT_I(7, 0x3f - SCALE(right - 1, 99, 0x3f));
	} else if (mix == SOUND_MIXER_LINE) {
		OUT_I(2, (IN_I(2) & 0xe0) | (0x1f - SCALE(left, 100, 0x1f)));
		OUT_I(3, (IN_I(3) & 0xe0) | (0x1f - SCALE(right, 100, 0x1f)));
	} else if (mix == SOUND_MIXER_CD) {
		OUT_I(4, (IN_I(4) & 0xe0) | (0x1f - SCALE(left, 100, 0x1f)));
		OUT_I(5, (IN_I(5) & 0xe0) | (0x1f - SCALE(right, 100, 0x1f)));
	} else if (mix == SOUND_MIXER_VOLUME) {
		unsigned i27, i29;
		i27 = IN_I(27) & (I27_LOS0 | I27_LOS1);
		if (__unlikely(!left)) OUT_I(27, i27 | I27_LOM | 0x1f);
		else OUT_I(27, i27 | (0x1f - SCALE(left - 1, 99, 0x1f)));
		i29 = IN_I(29) & (I29_ROS0 | I29_ROS1);
		if (__unlikely(!right)) OUT_I(29, i29 | I29_ROM | 0x1f);
		else OUT_I(29, i29 | (0x1f - SCALE(right - 1, 99, 0x1f)));
	} else if (mix == SOUND_MIXER_SPEAKER) {
		if (__unlikely(!left)) {
			OUT_I(26, I26_MIM | 0xf);
			if (MODE >= 3) OUT_X(4, IN_X(4) | X4_MIMR);
		} else {
			OUT_I(26, 0xf - SCALE(left - 1, 99, 0xf));
			if (MODE >= 3) OUT_X(4, IN_X(4) & ~X4_MIMR);
		}
	} else if (mix == SOUND_MIXER_MIC) {
		OUT_X(2, (IN_X(2) & 0xe0) | (0x1f - SCALE(left, 100, 0x1f)));
		OUT_X(3, (IN_X(3) & 0xe0) | (0x1f - SCALE(right, 100, 0x1f)));
	} else if (mix == SOUND_MIXER_RECLEV) {
		OUT_X(4, (IN_X(4) & 0x9f) | ((0x3 - SCALE(left, 100, 3)) << 5));
		OUT_X(5, (IN_X(5) & 0x9f) | ((0x3 - SCALE(right, 100, 3)) << 5));
	}
}

long SND_MIXER_READ(unsigned mix)
{
	unsigned left, right;
	if (mix == SOUND_MIXER_PCM) {
		unsigned i6, i7;
		i6 = IN_I(6);
		if (__unlikely(i6 & I6_LD1OM)) left = 0;
		else left = SCALE(0x3f - (i6 & 0x3f), 0x3f, 99) + 1;
		i7 = IN_I(7);
		if (__unlikely(i7 & I7_RD1OM)) right = 0;
		else right = SCALE(0x3f - (i7 & 0x3f), 0x3f, 99) + 1;
	} else if (mix == SOUND_MIXER_LINE) {
		left = SCALE(0x1f - (IN_I(2) & 0x1f), 0x1f, 100);
		right = SCALE(0x1f - (IN_I(3) & 0x1f), 0x1f, 100);
	} else if (mix == SOUND_MIXER_CD) {
		left = SCALE(0x1f - (IN_I(4) & 0x1f), 0x1f, 100);
		right = SCALE(0x1f - (IN_I(5) & 0x1f), 0x1f, 100);
	} else if (mix == SOUND_MIXER_VOLUME) {
		unsigned i27, i29;
		i27 = IN_I(27);
		if (__unlikely(i27 & I27_LOM)) left = 0;
		else left = SCALE(0x1f - (i27 & 0x1f), 0x1f, 99) + 1;
		i29 = IN_I(29);
		if (__unlikely(i29 & I29_ROM)) right = 0;
		else right = SCALE(0x1f - (i29 & 0x1f), 0x1f, 99) + 1;
	} else if (mix == SOUND_MIXER_SPEAKER) {
		unsigned i26 = IN_I(26);
		if (__unlikely(i26 & I26_MIM) && (__unlikely(MODE < 3) || __likely(IN_X(4) & X4_MIMR))) left = 0;
		else left = SCALE(0xf - (i26 & 0xf), 0xf, 99) + 1;
		right = left;
	} else if (mix == SOUND_MIXER_MIC) {
		left = SCALE(0x1f - (IN_X(2) & 0x1f), 0x1f, 100);
		right = SCALE(0x1f - (IN_X(3) & 0x1f), 0x1f, 100);
	} else if (mix == SOUND_MIXER_RECLEV) {
		unsigned x4, x5;
		x4 = (IN_X(4) >> 5) & 3;
		left = SCALE(3 - x4, 0x3, 100);
		x5 = (IN_X(5) >> 5) & 3;
		right = SCALE(3 - x5, 0x3, 100);
	} else return -EINVAL;
	return left + (right << 8);
}

static __const__ HANDLE_OPERATIONS SOUND_OPERATIONS = {
	SPL_X(SPL_OSS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	OSS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	OSS_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	OSS_LOOKUP_IO,		/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	OSS_OPEN,		/* open */
	OSS_CLOSE,		/* close */
	OSS_READ,		/* READ */
	OSS_WRITE,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	OSS_IOCTL,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

#include "OSS.I"

static int WSS_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	int pages = 65536 * 8;
	char **arg;
	int state;
	int r;
	int smode = MAXINT;
	DEVICE_REQUEST devrq;
	char *m;
	unsigned long io;
	struct __param_table params[] = {
		"IO", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 2, NULL,
		"IRQ", __PARAM_INT, 0, IRQ_LIMIT + 1, NULL,
		"DMA", __PARAM_INT, DMA_8BIT_MIN, DMA_8BIT_MAX, NULL,
		"DMA2", __PARAM_INT, DMA_8BIT_MIN, DMA_8BIT_MAX, NULL,
		"MEMORY", __PARAM_INT, 65536, 4194304+1, NULL,
		"MODE", __PARAM_INT, 1, 3+1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &io;
	params[1].__p = &IRQ;
	params[2].__p = &DMA1;
	params[3].__p = &DMA2;
	params[4].__p = &pages;
	params[5].__p = &smode;
	io = 0x534;
	IRQ = -1;
	DMA1 = -1;
	DMA2 = -1;

	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	IO = io;

	_snprintf(DEV_NAME, __MAX_STR_LEN, "SND$WSS@" IO_ID, IO);

	if (pages & 65535) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: BAD MEMORY SIZE (NOT MULTIPLE OF 65536)");
		r = -EBADSYN;
		goto ret0;
	}
	pages >>= 16;
	if (__unlikely(IRQ == -1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: IRQ NOT SPECIFIED");
		r = -EBADSYN;
		goto ret0;
	}
	if (DMA1 == -1) {
		if (DMA2 != -1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: DMA2 SPECIFIED AND DMA1 NOT");
			r = -EBADSYN;
			goto ret0;
		}
		DMA1 = 1;
		DMA2 = 3;
	}

	range.start = IO;
	range.len = WSS_REGSPACE;
	range.name = DEV_NAME;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", DEV_NAME, range.start, range.start + range.len - 1, strerror(-r));
		goto ret0;
	}

	if (WAIT_FOR_CALIBRATION(1)) {
		notf:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CODEC NOT FOUND AT IO "IO_FORMAT, IO);
		r = -ENODEV;
		goto ret05;
	}
	R0 = R0_MCE | R0_TRD;
	OUT_I(14, 0x12);
	OUT_I(15, 0x34);
	if (IN_I(14) != 0x12 || IN_I(15) != 0x34) goto notf;
	OUT_I(14, 0x56);
	OUT_I(15, 0x78);
	if (IN_I(14) != 0x56 || IN_I(15) != 0x78) goto notf;
	if (smode < 3) goto skip_mode_3;
	OUT_I(12, 0xEA);
	if ((IN_I(12) & 0xEF) == 0xEA) {
		OUT_X(11, X11_LD1IM | X11_RD1IM | X11_IFSE);
		OUT_X(12, 0x9a);
		OUT_X(13, 0xbc);
		if (IN_X(12) != 0x9a || IN_X(13) != 0xbc) goto skip_mode_3;
		OUT_X(12, 0xde);
		OUT_X(13, 0xf0);
		if (IN_X(12) != 0xde || IN_X(13) != 0xf0) goto skip_mode_3;
		MODE = 3;
		goto done_mode;
	}
	skip_mode_3:
	if (smode < 2) goto skip_mode_2;
	OUT_I(12, 0xCA);
	if ((IN_I(12) & 0xEF) == 0xCA) {
		OUT_I(30, 0x01);
		OUT_I(31, 0x23);
		OUT_I(14, 0x45);
		OUT_I(15, 0x67);
		if (IN_I(30) != 0x01 || IN_I(31) != 0x23 || IN_I(14) != 0x45 || IN_I(15) != 0x67) goto skip_mode_2;
		MODE = 2;
		goto done_mode;
	}
	skip_mode_2:
	OUT_I(12, 0x8A);
	MODE = 1;
	done_mode:
	if (smode != MAXINT && MODE != smode) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: MODE %d NOT SUPPORTED BY CODEC (MAXIMUM SUPPORTED MODE %d)", smode, MODE);
		r = -ENXIO;
		goto ret05;
	}
	if (MODE == 1) DMA2 = -1;

	CHIP = "CS423x-COMPATIBLE";

	OUT_R(2, 0);
	OUT_I(0, I0_MGE);
	OUT_I(1, I1_MGE);
	OUT_I(2, I2_LX1OM);
	OUT_I(3, I3_RX1OM);
	OUT_I(4, 0);
	OUT_I(5, 0);
	OUT_I(6, 0);
	OUT_I(7, 0);
	I9 = I9_CAL0 | I9_CAL1 | (DMA2 == -1 ? I9_SDC : 0);
	OUT_I(9, I9);
	WAIT_FOR_READY(1);
	OUT_I(10, I10 = IN_I(10) | I10_IEN);

	if (MODE == 1) {
		RATELIST = MODE1_RATELIST;
		CHIP = "CS4248";
	}
	if (MODE >= 2) {
		__u8 c;
		OSS_MIXER_DEVMASK |= SOUND_MASK_VOLUME | SOUND_MASK_SPEAKER;
		RATELIST = MODE2_RATELIST;
		OUT_I(16, I16_PMCE | I16_CMCE);
		OUT_I(22, 0);
		OUT_I(24, 0);
		c = IN_I(25);
		if ((c & 0x1f) == 0x00) CHIP = "CS4231";
		if ((c & 0x1f) == 0x02) CHIP = "CS4232";
		if ((c & 0x1f) == 0x03) CHIP = "CS4236";
		OUT_I(27, I27_LOS0);
		OUT_I(29, I29_ROS0);
	}
	if (MODE >= 3) {
		__u8 c;
		RATELIST = MODE3_RATELIST;
		OSS_MIXER_DEVMASK |= SOUND_MASK_MIC | SOUND_MASK_RECLEV;
		OSS_MIXER_RECMASK |= SOUND_MASK_MIC;
		OSS_MIXER_OUTMASK |= SOUND_MASK_MIC;
		OUT_X(2, X2_MBST | X2_LMOM);
		OUT_X(3, X3_MBST | X3_RMOM);
		OUT_X(4, IN_X(4) & (X4_IFM | X4_MIMR));
		OUT_X(5, 0);
		OUT_X(11, X11_IFSE | X11_RD1IM | X11_LD1IM);
		OUT_X(18, X18_PSH | (IN_X(18) & (X18_3DEN | X18_DSPD1)));
		OUT_X(24, IN_X(24) & ~X24_VCIE);
		c = IN_X(25);
		if ((c & 0x1f) == 0x1d) {
			CHIP = "CS4235";
			if ((c & 0xe0) == 0x80) CHIP = "CS4235A";
			if ((c & 0xe0) == 0xa0) CHIP = "CS4235B";
			if ((c & 0xe0) == 0xc0) CHIP = "CS4235C";
		} else CHIP = "CS4235-COMPATIBLE";
	}
	R0 &= ~R0_MCE;
	OUT_R(0, R0);
	if (WAIT_FOR_CALIBRATION(1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CODEC FAILED TO CALIBRATE");
		r = -ENODEV;
		goto ret05;
	}

	OSS_INIT(DMA2 != -1);

	if ((r = OSS_ALLOC_DMAPAGES(pages, 65536))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CAN'T ALLOC DMA PAGES: %s", strerror(-r));
		goto ret05;
	}

	if ((m = ISADMA$REQUEST_CHANNEL(DMA1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CAN'T REGISTER DMA %d: %s", DMA1, m);
		r = -EBUSY;
		goto ret1;
	}
	if (DMA2 != -1) if ((m = ISADMA$REQUEST_CHANNEL(DMA2))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CAN'T REGISTER SECONDARY DMA %d: %s", DMA2, m);
		r = -EBUSY;
		goto ret2;
	}
	WSS_IRQ_AST.fn = WSS_IRQ;
	if ((m = KERNEL$REQUEST_IRQ(IRQ, IRQ_AST_HANDLER | IRQ_EXCLUSIVE | IRQ_MASK, NULL, &WSS_IRQ_AST, &WSS_IRQ_CTRL, DEV_NAME))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "WSS: CAN'T REGISTER IRQ %d: %s", IRQ, m);
		r = -EBUSY;
		goto ret3;
	}

	_printf("%s: %s\n", DEV_NAME, CHIP);
	_printf("%s: IO "IO_FORMAT", IRQ %d, DMA %d", DEV_NAME, IO, IRQ, DMA1);
	if (DMA2 != -1) _printf(", DMA2 %d", DMA2);
	_printf("\n");

	devrq.name = DEV_NAME;
	devrq.driver_name = "WSS.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = OSS_SET_ROOT;
	devrq.dev_ptr = NULL;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = WSS_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", DEV_NAME);
		r = devrq.status;
		goto ret4;
	}

	playback_trd = record_trd = 0;
	VOID_LIST_ENTRY(&WSS_PLAYBACK_TIMEOUT.list);
	WSS_PLAYBACK_TIMEOUT.fn = WSS_PLAYBACK_TIMEOUT_FN;
	VOID_LIST_ENTRY(&WSS_RECORD_TIMEOUT.list);
	WSS_RECORD_TIMEOUT.fn = WSS_RECORD_TIMEOUT_FN;

	WSS_IRQ_CTRL.eoi();

	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, DEV_NAME);

	LNTE = devrq.lnte;
	DLRQ = KERNEL$TSR_IMAGE();
	return 0;

	ret4:
	KERNEL$RELEASE_IRQ(&WSS_IRQ_CTRL, IRQ_RELEASE_MASKED);
	ret3:
	if (DMA2 != -1) ISADMA$RELEASE_CHANNEL(DMA2);
	ret2:
	ISADMA$RELEASE_CHANNEL(DMA1);
	ret1:
	OSS_FREE_DMAPAGES();
	ret05:
	KERNEL$UNREGISTER_IO_RANGE(&range);
	ret0:
	return r;
}

static int WSS_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(LNTE, argv))) return r;
	RAISE_SPL(SPL_WSS);
	OSS_STOP(0);
	OSS_STOP(1);
	LOWER_SPL(SPL_ZERO);
	KERNEL$RELEASE_IRQ(&WSS_IRQ_CTRL, 0);
	OSS_FREE_DMAPAGES();
	if (DMA2 != -1) ISADMA$RELEASE_CHANNEL(DMA2);
	ISADMA$RELEASE_CHANNEL(DMA1);
	KERNEL$UNREGISTER_IO_RANGE(&range);
	*release = DLRQ;
	return 0;
}
