#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>

#include "SWAPPER.H"
#include "SWAPMAP.H"

#define LD_HASH_SIZE		8192

static void LDCACHE_WRITE(VMENTITY *e, PROC *p, int trashing);
static int LDCACHE_SWAPOUT(VMENTITY *e);
static int LDCACHE_CHECKMAP(VMENTITY *e);

static __const__ VMENTITY_T ldcache_calls = { LDCACHE_CHECKMAP, LDCACHE_WRITE, LDCACHE_SWAPOUT, "LDCACHE" };

int ldcache_vm_entity;

static XLIST_HEAD LD_HASH[LD_HASH_SIZE];

#define isroot_1(x)	1

static void DELETE_LDCACHE(LDCACHE *ld);

void REFERENCE_LDCACHE(SWAPNODE *sn, unsigned long account, long diff, PROC *owner)
{
	LDREF *ldr;
	LDCACHE *ld;
	ASSERT_LOCKED();
#if __DEBUG >= 1
	if (__unlikely(!ACCT_IS_LDCACHE(account)))
		KERNEL$SUICIDE("REFERENCE_LDCACHE: NOT LDCACHE: %lX", account);
#endif
	ld = ACCT_LDCACHE(account);
	sn = sn->ldrefsptr;
	XLIST_FOR_EACH(ldr, &sn->ldrefs, LDREF, list) {
		if (__likely(ldr->ldcache == ld)) {
			if (__unlikely(ldr != LIST_STRUCT(sn->ldrefs.next, LDREF, list))) {
				DEL_FROM_LIST(&ldr->list);
				ADD_TO_XLIST(&sn->ldrefs, &ldr->list);
			}
			goto found;
		}
	}
	KERNEL$SUICIDE("REFERENCE_LDCACHE: LDREF NOT FOUND");
	found:
	ldr->nrefs += diff;
	if (__unlikely(!ldr->nrefs)) {
#if __DEBUG >= 1
		if (__unlikely(ld->pageq.q_node_usage != ld->pageq.q_subtree_usage)
		 || __unlikely(ld->unpageq.q_node_usage != ld->unpageq.q_subtree_usage))
			KERNEL$SUICIDE("REFERENCE_LDCACHE: QUOTA SHOULDN'T HAVE ANY SUBTREES, PAGEQ (NODE %ld, SUBTREE %ld), UNPAGEQ (NODE %ld, SUBTREE %ld)", ld->pageq.q_node_usage, ld->pageq.q_subtree_usage, ld->unpageq.q_node_usage, ld->unpageq.q_subtree_usage);
#endif
		QFREE(&sn->pageq, ld->pageq.q_node_usage, pageq_isroot, pageq_parent, Q_NULL_CALL);
		QFREE(&sn->unpageq, ld->unpageq.q_node_usage, unpageq_isroot, unpageq_parent, Q_NULL_CALL);
		DEL_FROM_LIST(&ldr->list);
		__slfree(ldr);
		if (__unlikely(!ld->ldref_count))
			KERNEL$SUICIDE("REFERENCE_LDCACHE: LDREF_COUNT UNDERFLOW");
		ld->ldref_count--;
		if (__unlikely(ld->state != LD_ESTABLISHED)) {
			if (__unlikely(ld->ldref_count != 0))
				KERNEL$SUICIDE("REFERENCE_LDCACHE: NON-ESTABLISHED ENTRY %u HAS REFCOUNT %lu", (unsigned)ld->state, ld->ldref_count);
			DELETE_LDCACHE(ld);
		} else if (__unlikely(!ld->vme_used)) {
			ld->vme_used = 1;
			RAISE_SPL(SPL_CACHE);
			KERNEL$CACHE_INSERT_VM_ENTITY(&ld->vme, owner, 0);
			LOWER_SPL(SPL_FS);
		}
	}
}

#define DO_LDCACHE
#include "SWAPPND.I"

int SWAPLD_PREPARE(SWAPNODE *sn, LD_PREPARE *ldp, IOCTLRQ *rq)
{
#define CLEANUP						\
	do {						\
		if (!XLIST_EMPTY(&tmp_list)) {		\
			REFERENCE_LDCACHE(sn, ACCT_FROM_LDCACHE(ld), 0, rq->handle->name_addrspace);					\
		}					\
		SWAP_LOCK(SW_UNLOCK);			\
	} while (0)
	PAGENODE *pn;
	PAGE *p;
	LDCACHE *ld;
	LDREF *ldr;
	PAGEDIR **n;
	DECL_XLIST(tmp_list);
	QUOTA *pzap;
	WQ *wq;
	check_quota(ACCT_FROM_SWAPNODE(sn), "SWAPLD_PREPARE 1");
	if (__unlikely((ldp->from | ldp->to) & (PAGE_CLUSTER_SIZE - 1)) || __unlikely(ldp->from >= ldp->to)) {
		return -EINVAL;
	}
	p = SWAP_CREATE_PAGE(ACCT_FROM_SWAPNODE(sn), ldp->from, PF_READ, (IORQ *)rq, rq->handle->name_addrspace);
	if (__unlikely(!p)) return 1;
	pn = p->fnode;
	if (__likely(!ACCT_IS_LDCACHE(pn->account))) {
		ld = __slalloc(&ldcaches);
		if (__unlikely(!ld)) {
			nomem_unpaged:
			if (UNPAGED_ALLOC_FAILED()) goto freemem_wait;
			goto retry;
		}
		init_ldcache(ld);
		ld->ldref_count = 1;
		ld->state = LD_CREATING;
		memset(&ld->link_hash, 0, sizeof ld->link_hash);
		ld->from = ldp->from;
		ld->to = ld->from;
		ldr = __slalloc(&ldrefs);
		if (__unlikely(!ldr)) {
			__slow_slfree(ld);
			goto nomem_unpaged;
		}
		ldr->ldcache = ld;
		ldr->nrefs = 0;
		QALLOC(&sn->ldrefsptr->unpageq, sizeof(LDCACHE) + sizeof(LDREF), unpageq_isroot, unpageq_parent, Q_NULL_CALL, pzap, {
			__slow_slfree(ld);
			__slow_slfree(ldr);
			OUT_OF_UNPAGED(pzap);
			goto retry;
		});
		QALLOC(&ld->unpageq, sizeof(LDCACHE) + sizeof(LDREF), isroot_1, unpageq_parent, Q_NULL_CALL, pzap, {
			KERNEL$SUICIDE("SWAPLD_PREPARE: QALLOC ON UNPAGEQ FAILED");
		});
		ADD_TO_XLIST(&tmp_list, &ld->hash);
		ADD_TO_XLIST(&sn->ldrefsptr->ldrefs, &ldr->list);
		ld->vme_used = 0;
	} else {
		ld = ACCT_LDCACHE(pn->account);
		XLIST_FOR_EACH(ldr, &sn->ldrefsptr->ldrefs, LDREF, list) {
			if (__likely(ldr->ldcache == ld)) goto have_ld;
		}
		KERNEL$SUICIDE("SWAPLD_PREPARE: LDREF NOT FOUND");
		have_ld:
		check_quota(ACCT_FROM_LDCACHE(ld), "SWAPLD_PREPARE 2");
		if (__unlikely(ld->from != ldp->from) || __unlikely(ld->to > ldp->to) || __unlikely(ld->state != LD_CREATING)) {
			einval:
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "INVALID PREPARE CALL");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
		if (__unlikely(ld->to == ldp->to))
			return 0;
		again:
		p = SWAP_CREATE_PAGE(ACCT_FROM_SWAPNODE(sn), ld->to, PF_READ, (IORQ *)rq, rq->handle->name_addrspace);
		if (__unlikely(!p))
			return 1;
		pn = p->fnode;
		if (__unlikely(ACCT_IS_LDCACHE(pn->account)))
			goto einval;
	}
	SWAP_LOCK(SW_LOCK);
	if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
		WQ_WAIT_F(wq, rq);
		SWAPPAGE_ADD_TO_WANTFREE(p, wq);
		LOWER_SPL(SPL_FS);
		CLEANUP;
		return 1;
	}
	LOWER_SPL(SPL_FS);
	n = SWAPLD_RETURN_PAGENODE(ld, ld->to, sn->ldrefsptr);
	if (__unlikely((unsigned long)n <= 2)) {
		if (__unlikely(!n)) {
			KERNEL$SUICIDE("SWAPLD_PREPARE: SWAPLD_RETURN_PAGENODE RETURNED NULL");
		} else if (__unlikely(n == (void *)1)) {
			CLEANUP;
			goto retry;
		} else {
			CLEANUP;
			goto freemem_wait;
		}
	}
	if (__unlikely(__IS_ERR(n))) {
		CLEANUP;
		return __PTR_ERR(n);
	}
	/*if (random() & 1) { CLEANUP; goto lockup; }*/
	if (__unlikely(*n != NULL)) KERNEL$SUICIDE("SWAPLD_PREPARE: PAGENODE EXISTS OUT OF BOUNDS: %lX, %lX", ld->from, ld->to);
	*n = (PAGEDIR *)pn;
	QXFER(&ACCT_SWAPNODE(pn->account)->pageq, &sn->ldrefsptr->pageq, 1, pageq_parent);
	QALLOC(&ld->pageq, 1, isroot_1, pageq_parent, Q_NULL_CALL, pzap, {
		KERNEL$SUICIDE("SWAPLD_PREPARE: QALLOC ON PAGEQ FAILED");
	});
	pn->account = ACCT_FROM_LDCACHE(ld);
	pn->refcount++;
	ldr->nrefs++;

	SWAP_LOCK(SW_UNLOCK);

	if (__unlikely(!XLIST_EMPTY(&tmp_list))) {
		DEL_FROM_LIST(&ld->hash);
		ADD_TO_XLIST(&LD_HASH[0], &ld->hash);
	}

	ld->to += PAGE_CLUSTER_SIZE;

	if (__unlikely(ld->to < ldp->to)) {
		int r;
		check_quota(ACCT_FROM_SWAPNODE(sn), "SWAPLD_PREPARE 3");
		check_quota(ACCT_FROM_LDCACHE(ld), "SWAPLD_PREPARE 4");
		if (__unlikely(r = KERNEL$LOCKUP_WAIT_LOOP((IORQ *)rq, rq->handle))) return r;
		goto again;
	}
	check_quota(ACCT_FROM_SWAPNODE(sn), ld->from == ld->to - PAGE_CLUSTER_SIZE ? "SWAPLD_PREPARE 5/1" : "SWAPLD_PREPARE 5/2");
	check_quota(ACCT_FROM_LDCACHE(ld), ld->from == ld->to - PAGE_CLUSTER_SIZE ? "SWAPLD_PREPARE 6/1" : "SWAPLD_PREPARE 6/2");
	return 0;

	retry:
	CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
	return 1;
	freemem_wait:
	WQ_WAIT_F(&KERNEL$FREEMEM_WAIT, rq);
	return 1;
#undef CLEANUP
}

static __finline__ unsigned HASH_LINKHASH(LD_FINAL_HASH *hash)
{
	/* the input comes from cryptographic hash functions, so it's random
	   --- we do not have to do anything smart at all --- just take few
	   bits as an index to hash table */
	return (*(__u32 *)&hash->hash[0] + *(__u32 *)&hash->hash[LINK_HASH_LENGTH]) & (LD_HASH_SIZE - 1);
}


void SWAPLD_SET_ESTABLISHED(LDCACHE *ld)
{
	unsigned hash;
	unsigned long addr = ld->from;
	/*{
		unsigned i;
		__debug_printf("insert: ");
		for (i = 0; i < sizeof(LD_FINAL_HASH); i++) __debug_printf("%02x", ld->link_hash.hash[i]);
		__debug_printf("\n");
	}*/
	if (__unlikely(!SWAPLD_QUERY(&ld->link_hash, &addr)) && __unlikely(addr == ld->from)) {
		ld->state = LD_ESTABLISHED_NONSHARED;
		return;
	}
	hash = HASH_LINKHASH(&ld->link_hash);
	DEL_FROM_LIST(&ld->hash);
	ADD_TO_XLIST(&LD_HASH[hash], &ld->hash);
	ld->state = LD_ESTABLISHED;
}

int SWAPLD_QUERY(LD_FINAL_HASH *h, unsigned long *addr)
{
	LDCACHE *ld;
	unsigned hash = HASH_LINKHASH(h);
	unsigned long result = MAXULONG;
	/*{
		unsigned i;
		__debug_printf("lookup: ");
		for (i = 0; i < sizeof(LD_FINAL_HASH); i++) __debug_printf("%02x", h->hash[i]);
		__debug_printf("\n");
	}*/
	XLIST_FOR_EACH(ld, &LD_HASH[hash], LDCACHE, hash) {
		if (__likely(!memcmp(&ld->link_hash, h, sizeof(LD_FINAL_HASH))) && __likely(ld->state == LD_ESTABLISHED)) {
			if (__likely(ld->from >= *addr) && __unlikely(ld->from < result)) {
				result = ld->from;
			}
		}
	}
	if (__likely(result != MAXULONG)) {
		*addr = result;
		return 0;
	}
	return -ENOENT;
}

int SWAPLD_ATTACH(SWAPNODE *s, LD_FINAL_HASH *h, unsigned long addr, IOCTLRQ *rq)
{
	LDCACHE *ld;
	unsigned hash = HASH_LINKHASH(h);
	/*{
		unsigned i;
		__debug_printf("lookup: ");
		for (i = 0; i < sizeof(LD_FINAL_HASH); i++) __debug_printf("%02x", h->hash[i]);
	}*/
	XLIST_FOR_EACH(ld, &LD_HASH[hash], LDCACHE, hash) {
		if (__likely(!memcmp(&ld->link_hash, h, sizeof(LD_FINAL_HASH))) && __likely(ld->state == LD_ESTABLISHED) && __likely(ld->from == addr)) {
			unsigned long u;
			SWAP_LOCK(SW_LOCK);
			for (u = ld->from; u < ld->to; u += PAGE_CLUSTER_SIZE) {
				QUOTA *pzap;
				PAGENODE *pn = SWAP_SIMPLE_RETURN_PAGENODE((SWAPNODE *)ld, u);
				PAGEDIR **pnp;
				int wr;
				if (__unlikely(!pn))
					KERNEL$SUICIDE("SWAPLD_ATTACH: PAGENODE NOT PRESENT AT OFFSET %lX (BOUNDS %lX - %lX)", u, ld->from, ld->to);
				wr = PF_WRITE;
				pnp = SWAP_RETURN_PAGENODE(s, u, &wr);
				if (__unlikely((unsigned long)pnp <= 2)) {
					if (__unlikely(!pnp))
						KERNEL$SUICIDE("SWAPLD_ATTACH: SWAP_RETURN_PAGENODE RETURNED NULL");
					SWAP_LOCK(SW_UNLOCK);
					return (long)pnp;
				}
				if (__IS_ERR(pnp)) {
					SWAP_LOCK(SW_UNLOCK);
					return __PTR_ERR(pnp);
				}
				if (__unlikely(*pnp != NULL)) {
					PAGENODE *q = (PAGENODE *)*pnp;
					if (__unlikely(ACCT_LDCACHE(q->account) != ld)) {
						SWAP_LOCK(SW_UNLOCK);
						return -EINVAL;
					}
					continue;
				}
	/* if this is the first pagenode we are going to attach, create ldref */
				if (__unlikely(u == ld->from)) {
					LDREF *ldr = __slalloc(&ldrefs);
					if (__unlikely(!ldr)) {
						SWAP_LOCK(SW_UNLOCK);
						return UNPAGED_ALLOC_FAILED() + 1;
					}
					QALLOC(&s->ldrefsptr->pageq, ld->pageq.q_node_usage, pageq_isroot, pageq_parent, Q_NULL_CALL, pzap, {
						OUT_OF_PAGED(pzap);
						__slow_slfree(ldr);
						SWAP_LOCK(SW_UNLOCK);
						return 1;
					});
					QALLOC(&s->ldrefsptr->unpageq, ld->unpageq.q_node_usage, unpageq_isroot, unpageq_parent, Q_NULL_CALL, pzap, {
						QFREE(&s->ldrefsptr->pageq, ld->pageq.q_node_usage, pageq_isroot, pageq_parent, Q_NULL_CALL);
						__slow_slfree(ldr);
						OUT_OF_UNPAGED(pzap);
						SWAP_LOCK(SW_UNLOCK);
						return 1;
					});
					ldr->ldcache = ld;
					ldr->nrefs = 0;
					ADD_TO_XLIST(&s->ldrefsptr->ldrefs, &ldr->list);
					ld->ldref_count++;
					if (__likely(ld->vme_used)) {
						RAISE_SPL(SPL_CACHE);
						KERNEL$CACHE_TOUCH_VM_ENTITY(&ld->vme, rq->handle->name_addrspace);
						LOWER_SPL(SPL_FS);
					}
				}
				*pnp = (PAGEDIR *)pn;
				pn->refcount++;
				REFERENCE_LDCACHE(s, pn->account, 1, rq->handle->name_addrspace);
			}
			SWAP_LOCK(SW_UNLOCK);
			memcpy(h->hash + LINK_HASH_LENGTH, ld->exported_hash, REL_DIGEST_LENGTH);
			return 0;
		}
	}
	return -ENOENT;
}

static void DELETE_LDCACHE(LDCACHE *ld)
{
	unsigned i;
	ASSERT_LOCKED();
	if (__likely(ld->vme_used)) {
		RAISE_SPL(SPL_CACHE);
		KERNEL$CACHE_REMOVE_VM_ENTITY(&ld->vme);
		LOWER_SPL(SPL_FS);
	}
	for (i = 0; i < TOP_PGDIR_SIZE; i++) if (ld->pagedir[i]) {
		if (__unlikely(SWAP_ZAP(&ld->pagedir[i], PGDIR_DEPTH, 0, (1 << (PGDIR_DEPTH * PGDIR_SIZE_BITS)) - 1, ACCT_FROM_LDCACHE(ld), &KERNEL$PROC_KERNEL) != NULL))
			KERNEL$SUICIDE("DELETE_LDCACHE: SWAP_ZAP RETURNED NONZERO");
		if (__unlikely(ld->pagedir[i] != NULL))
			KERNEL$SUICIDE("DELETE_LDCACHE: CAN'T ZAP PAGEDIR");
	}
	QFREE(&ld->unpageq, sizeof(LDCACHE) + sizeof(LDREF), isroot_1, unpageq_parent, Q_NULL_CALL);
	QDONE2(&ld->pageq, "PAGEQ", &ld->unpageq, "UNPAGEQ");
	DEL_FROM_LIST(&ld->hash);
	__slfree(ld);
}

static void LDCACHE_WRITE(VMENTITY *e, PROC *p, int trashing)
{
	LOWER_SPL(SPL_FS);
}

static int LDCACHE_SWAPOUT(VMENTITY *e)
{
	LDCACHE *ld;
	LOWER_SPL(SPL_FS);
	ld = LIST_STRUCT(e, LDCACHE, vme);
	if (__unlikely(ld->ldref_count)) return 2;
	SWAP_LOCK(SW_LOCK);
	DELETE_LDCACHE(ld);
	SWAP_LOCK(SW_UNLOCK);
	return 0;
}

static int LDCACHE_CHECKMAP(VMENTITY *e)
{
	LDCACHE *ld;
	LOWER_SPL(SPL_FS);
	ld = LIST_STRUCT(e, LDCACHE, vme);
	return !!ld->ldref_count;
}

int PRUNE_LDCACHE(void)
{
	int pruned = 0;
	unsigned i;
	LDCACHE *ld;
	for (i = 0; i < LD_HASH_SIZE; i++) XLIST_FOR_EACH(ld, &LD_HASH[i], LDCACHE, hash) {
		if (!ld->ldref_count) {
			LDCACHE *ldd = LIST_STRUCT(ld->hash.prev, LDCACHE, hash);
			SWAP_LOCK(SW_LOCK);
			DELETE_LDCACHE(ld);
			SWAP_LOCK(SW_UNLOCK);
			ld = ldd;
			pruned = 1;
		}
	}
	return pruned;
}

int TEST_PRUNE_LDCACHE(void)
{
	unsigned i;
	LDCACHE *ld;
	for (i = 0; i < LD_HASH_SIZE; i++) XLIST_FOR_EACH(ld, &LD_HASH[i], LDCACHE, hash) {
		if (!ld->ldref_count) return 1;
	}
	return 0;
}

int LDCACHE_INIT(void)
{
	unsigned i;
	for (i = 0; i < LD_HASH_SIZE; i++) INIT_XLIST(&LD_HASH[i]);
	return KERNEL$CACHE_REGISTER_VM_TYPE(&ldcache_vm_entity, &ldcache_calls);;
}

void LDCACHE_DONE(void)
{
	unsigned i;
	PRUNE_LDCACHE();
	for (i = 0; i < LD_HASH_SIZE; i++) {
		if (__unlikely(!XLIST_EMPTY(&LD_HASH[i])))
			KERNEL$SUICIDE("LDCACHE_DONE: ENTRY LEAKED");
	}
	KERNEL$CACHE_UNREGISTER_VM_TYPE(ldcache_vm_entity);
}

