#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DEV_KRNL.H>

#include "SWAPPER.H"
#include "SWAPMAP.H"

#define PREFERRED_HASH_STEP	4096

#define HASH_STEP	(PAGE_CLUSTER_SIZE < PREFERRED_HASH_STEP ? PAGE_CLUSTER_SIZE : PREFERRED_HASH_STEP)

/* we can't use more than 3 cached because there can be at most 4 pages mapped
   with KERNEL$MAP_PHYSICAL_PAGE for 2 SPLs we have */

static unsigned long cache_addr[2] = { VMSPACE, VMSPACE };
static char *cache_ptr[2] = { NULL, NULL };
static int cache_repl = 0;

static __finline__ void CACHE_DONE_ENTRY(int idx)
{
	if (cache_ptr[idx]) {
		cache_addr[idx] = VMSPACE;
		KERNEL$UNMAP_PHYSICAL_BANK(cache_ptr[idx]);
		cache_ptr[idx] = 0;
	}
}

static void CACHE_DONE(void)
{
	CACHE_DONE_ENTRY(0);
	CACHE_DONE_ENTRY(1);
	cache_repl = 0;
}

#define DO_SIMPLE
#include "SWAPPND.I"

static LDCACHE *SWAPLD_FIND(SWAPNODE *sn, unsigned long addr)
{
	PAGENODE *pn;
	LDCACHE *ld;
	if (__unlikely(addr >= VMSPACE)) return NULL;
	pn = SWAP_SIMPLE_RETURN_PAGENODE(sn, addr);
	if (__unlikely(!pn)) return NULL;
	if (__unlikely(!ACCT_IS_LDCACHE(pn->account))) return NULL;
	ld = ACCT_LDCACHE(pn->account);
	if (__unlikely(ld->from != addr)) return NULL;
	return ld;
}

/* It is required that two successive calls to SWAPLD_MAP return two pointers
   that are both valid after the calls --- see SWAPLD_MAP2 */

static void *SWAPLD_MAP(LDCACHE *ld, unsigned long addr, IOCTLRQ *rq)
{
	PAGE *p;
	char *ret;
	if (__likely(!((addr ^ cache_addr[0]) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1)))) {
		cache_repl = 1;
		return cache_ptr[0] + (addr & (PAGE_CLUSTER_SIZE - 1));
	}
	if (__likely(!((addr ^ cache_addr[1]) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1)))) {
		cache_repl = 0;
		return cache_ptr[1] + (addr & (PAGE_CLUSTER_SIZE - 1));
	}
	CACHE_DONE_ENTRY(cache_repl);
	if (__unlikely(addr < ld->from) || __unlikely(addr >= ld->to)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "ACCESS OUT OF RANGE (%lX, RANGE %lX - %lX)", addr, ld->from, ld->to);
		ld->state = LD_ERROR;
		rq->status = -EFTYPE;
		CALL_AST(rq);
		return NULL;
	}
	p = SWAP_CREATE_PAGE(ACCT_FROM_LDCACHE(ld), addr, PF_WRITE | PF_SHARE | PF_SWAPPER, (IORQ *)rq, rq->handle->name_addrspace);
	if (__unlikely(!p)) return NULL;
	cache_addr[cache_repl] = addr & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	cache_ptr[cache_repl] = KERNEL$MAP_PHYSICAL_PAGE(p);
	ret = cache_ptr[cache_repl] + (addr & (PAGE_CLUSTER_SIZE - 1));
	cache_repl ^= 1;
	return ret;
}

static int SWAPLD_MAP2(LDCACHE *ld, unsigned long addr1, void **p1, unsigned long addr2, void **p2, IOCTLRQ *rq)
{
	if (__unlikely(!(*p1 = SWAPLD_MAP(ld, addr1, rq)))) return 1;
	if (__unlikely(!(*p2 = SWAPLD_MAP(ld, addr2, rq)))) return 1;
	return 0;
}

static int SWAPLD_RW_SPLIT(LDCACHE *ld, unsigned long addr, void *val, int write, unsigned len, IOCTLRQ *rq)
{
	int r;
	void *ptr1, *ptr2;
	unsigned split = PAGE_CLUSTER_SIZE - (addr & (PAGE_CLUSTER_SIZE - 1));
	if (__unlikely(r = SWAPLD_MAP2(ld, addr, &ptr1, addr + split, &ptr2, rq))) return r;
	if (!write) {
		memcpy(val, ptr1, split);
		memcpy((char *)val + split, ptr2, len - split);
	} else {
		memcpy(ptr1, val, split);
		memcpy(ptr2, (char *)val + split, len - split);
	}
	return 0;
}

static __finline__ int SWAPLD_RW(LDCACHE *ld, unsigned long addr, void *val, int write, unsigned len, IOCTLRQ *rq)
{
	void *ptr;
	if (len > 1 && __unlikely((addr & (PAGE_CLUSTER_SIZE - 1)) + len > PAGE_CLUSTER_SIZE)) return SWAPLD_RW_SPLIT(ld, addr, val, write, len, rq);
	ptr = SWAPLD_MAP(ld, addr, rq);
	if (__unlikely(!ptr)) return 1;
	if (!write) memcpy(val, ptr, len);
	else memcpy(ptr, val, len);
	return 0;
}

static int SWAPLD_READ(LDCACHE *ld, unsigned long addr, void *val, unsigned len, IOCTLRQ *rq)
{
	do {
		void *ptr;
		unsigned l = PAGE_CLUSTER_SIZE - (addr & (PAGE_CLUSTER_SIZE - 1));
		if (__likely(l > len)) l = len;
		ptr = SWAPLD_MAP(ld, addr, rq);
		if (__unlikely(!ptr)) return 1;
		memcpy(val, ptr, l);
		ptr = (char *)ptr + l;
		len -= l;
	} while (len);
	return 0;
}

static int SWAPLD_READ_F_OFF(LDCACHE *ld, unsigned long addr, __f_off *val, IOCTLRQ *rq)
{
	return SWAPLD_RW(ld, addr, val, 0, sizeof(__f_off), rq);
}

static int SWAPLD_WRITE_F_OFF(LDCACHE *ld, unsigned long addr, __f_off *val, IOCTLRQ *rq)
{
	return SWAPLD_RW(ld, addr, val, 1, sizeof(__f_off), rq);
}

static int SWAPLD_READ_UNSIGNED(LDCACHE *ld, unsigned long addr, unsigned *val, IOCTLRQ *rq)
{
	return SWAPLD_RW(ld, addr, val, 0, sizeof(unsigned), rq);
}

static int SWAPLD_READ_BYTE(LDCACHE *ld, unsigned long addr, __u8 *val, IOCTLRQ *rq)
{
	return SWAPLD_RW(ld, addr, val, 0, sizeof(__u8), rq);
}

static int SWAPLD_READ_STRING_SPLIT(LDCACHE *ld, unsigned long addr, IOCTLRQ *rq);
static char split_string[LINK_DYNAMIC_SYMBOL_NAME_LEN];

static int SWAPLD_READ_STRING(LDCACHE *ld, unsigned long addr, char **s, IOCTLRQ *rq)
{
	__f_off ptr;
	unsigned len;
	unsigned maxlen;
	int r;
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, addr, &ptr, rq))) return r;
	ptr += ld->from;
	*s = SWAPLD_MAP(ld, ptr, rq);
	if (__unlikely(!*s)) {
		return 1;
	}
	maxlen = PAGE_CLUSTER_SIZE - (ptr & (PAGE_CLUSTER_SIZE - 1));
	len = strnlen(*s, maxlen);
	if (__unlikely(len >= LINK_DYNAMIC_SYMBOL_NAME_LEN)) {
		return -EFTYPE;
	}
	if (__likely(len < maxlen)) return 0;
	*s = split_string;
	return SWAPLD_READ_STRING_SPLIT(ld, ptr, rq);
}

static int SWAPLD_READ_STRING_SPLIT(LDCACHE *ld, unsigned long ptr, IOCTLRQ *rq)
{
	int r;
	unsigned p;
	for (p = 0; p < LINK_DYNAMIC_SYMBOL_NAME_LEN; p++) {
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ptr + p, &split_string[p], rq))) return r;
		if (!split_string[p]) return 0;
	}
	return -EFTYPE;
}

static int unknown_hash(unsigned hash_type)
{
	return	hash_type != LINK_HASH_MD5 >> __BSF_CONST(LINK_HASH_MASK) &&
		hash_type != LINK_HASH_SHA1 >> __BSF_CONST(LINK_HASH_MASK) &&
		hash_type != LINK_HASH_RIPEMD160 >> __BSF_CONST(LINK_HASH_MASK) &&
		hash_type != LINK_HASH_SHA256 >> __BSF_CONST(LINK_HASH_MASK);
}

static void init_hash(LDCACHE *ld)
{
	switch (ld->hash_type) {
		case LINK_HASH_MD5 >> __BSF_CONST(LINK_HASH_MASK):
			MD5Init(&ld->u.chs.h.md5_ctx);
			break;
		case LINK_HASH_SHA1 >> __BSF_CONST(LINK_HASH_MASK):
			SHA1_Init(&ld->u.chs.h.sha1_ctx);
			break;
		case LINK_HASH_RIPEMD160 >> __BSF_CONST(LINK_HASH_MASK):
			RIPEMD160_Init(&ld->u.chs.h.ripemd160_ctx);
			break;
		case LINK_HASH_SHA256 >> __BSF_CONST(LINK_HASH_MASK):
			SHA256_Init(&ld->u.chs.h.sha256_ctx);
			break;
		default:
			KERNEL$SUICIDE("init_hash: %x", ld->hash_type);
	}
}

static void add_hash(LDCACHE *ld, void *ptr, size_t len)
{
	switch (ld->hash_type) {
		case LINK_HASH_MD5 >> __BSF_CONST(LINK_HASH_MASK):
			MD5Update(&ld->u.chs.h.md5_ctx, ptr, len);
			break;
		case LINK_HASH_SHA1 >> __BSF_CONST(LINK_HASH_MASK):
			SHA1_Update(&ld->u.chs.h.sha1_ctx, ptr, len);
			break;
		case LINK_HASH_RIPEMD160 >> __BSF_CONST(LINK_HASH_MASK):
			RIPEMD160_Update(&ld->u.chs.h.ripemd160_ctx, ptr, len);
			break;
		case LINK_HASH_SHA256 >> __BSF_CONST(LINK_HASH_MASK):
			SHA256_Update(&ld->u.chs.h.sha256_ctx, ptr, len);
			break;
		default:
			KERNEL$SUICIDE("add_hash: %x", ld->hash_type);
	}
}

static void get_hash(__u8 *bytes, LDCACHE *ld)
{
	switch (ld->hash_type) {
		case LINK_HASH_MD5 >> __BSF_CONST(LINK_HASH_MASK):
			MD5Final(bytes, &ld->u.chs.h.md5_ctx);
			break;
		case LINK_HASH_SHA1 >> __BSF_CONST(LINK_HASH_MASK):
			SHA1_Final(bytes, &ld->u.chs.h.sha1_ctx);
			break;
		case LINK_HASH_RIPEMD160 >> __BSF_CONST(LINK_HASH_MASK):
			RIPEMD160_Final(bytes, &ld->u.chs.h.ripemd160_ctx);
			break;
		case LINK_HASH_SHA256 >> __BSF_CONST(LINK_HASH_MASK):
			SHA256_Final(bytes, &ld->u.chs.h.sha256_ctx);
			break;
		default:
			KERNEL$SUICIDE("get_hash: %x", ld->hash_type);
	}
}

static char *hash_name(LDCACHE *ld)
{
	switch (ld->hash_type) {
		case LINK_HASH_MD5 >> __BSF_CONST(LINK_HASH_MASK):
			return "MD5";
		case LINK_HASH_SHA1 >> __BSF_CONST(LINK_HASH_MASK):
			return "SHA1";
		case LINK_HASH_RIPEMD160 >> __BSF_CONST(LINK_HASH_MASK):
			return "RIPEMD160";
		case LINK_HASH_SHA256 >> __BSF_CONST(LINK_HASH_MASK):
			return "SHA256";
		default:
			KERNEL$SUICIDE("hash_name: %x", ld->hash_type);
	}
}

static int TEST_ZERO(char *p)
{
	while (__unlikely(((unsigned long)p & (sizeof(unsigned long) - 1)) != 0)) {
		if (__unlikely(*p)) return -EFTYPE;
		p++;
	}
	while (__likely(((unsigned long)p & (PAGE_CLUSTER_SIZE - 1)) != 0)) {
		if (__unlikely(*(unsigned long *)p)) return -EFTYPE;
		p += sizeof(unsigned long);
	}
	return 0;
}

static int SWAPLD_CHECK_INIT(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	struct link_header lh;
	struct section secs[5];
	unsigned long size;
	if (__unlikely(r = SWAPLD_READ(ld, ld->from, &lh, sizeof(struct link_header), rq))) return r;
	if (__unlikely(lh.lnk_id != LNK_ID)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "MAGIC DOES NOT MATCH");
		eftype:
		ld->state = LD_ERROR;
		return -EFTYPE;
	}
	if (__unlikely(lh.lnk_ver != LNK_VER)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "UNKNOWN VERSION %u", lh.lnk_ver);
		goto eftype;
	}
	if (__unlikely(lh.n_sections < 5)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "TOO FEW SECTIONS");
		goto eftype;
	}
	if (__unlikely((lh.flags & LINK_COMPRESSION_MASK) != 0)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "COMPRESSED EXECUTABLE");
		goto eftype;
	}
	ld->hash_type = (lh.flags & LINK_HASH_MASK) >> __BSF_CONST(LINK_HASH_MASK);
	if (__unlikely(unknown_hash(ld->hash_type))) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "UNKNOWN HASH METHOD %X", ld->hash_type);
		goto eftype;
	}
	if (__unlikely(r = SWAPLD_READ(ld, ld->from + lh.sections, &secs, sizeof(secs), rq))) return r;
	size = secs[SECTION_DATA].offset + secs[SECTION_DATA].len;

	if (__unlikely(ld->to - ld->from != ((size + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1)))) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "SIZE DOES NOT MATCH");
		goto eftype;
	}
	ld->u.chs.data_end = ld->from + size;

	if (__likely((size & (PAGE_CLUSTER_SIZE - 1)) != 0)) {
		char *p = SWAPLD_MAP(ld, ld->from + size, rq);
		if (__unlikely(!p)) return 1;
		if (__unlikely(r = TEST_ZERO(p))) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "BSS NOT CLEARED");
			ld->state = LD_ERROR;
			return r;
		}
	}

	init_hash(ld);
	add_hash(ld, &lh, __offsetof(struct link_header, hash));
	ld->u.chs.idx = ld->from + __offsetof(struct link_header, hash) + LINK_HASH_LENGTH;
	ld->state = LD_CHECKSUMMING;
	return 0;
}

static int SWAPLD_CHECK_FINAL(LDCACHE *ld, IOCTLRQ *rq);

static int SWAPLD_CHECK(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	unsigned len;
	char *data;
	if (ld->state == LD_CREATING) {
		int r;
		if ((r = SWAPLD_CHECK_INIT(ld, rq))) return r;
	} else {
		if (__unlikely(ld->state != LD_CHECKSUMMING)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "LDCACHE_CHECK IN INVALID STATE");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
	}
	again:
	if (__unlikely(ld->u.chs.idx >= ld->u.chs.data_end)) {
		return SWAPLD_CHECK_FINAL(ld, rq);
	}
	data = SWAPLD_MAP(ld, ld->u.chs.idx, rq);
	if (__unlikely(!data)) return 1;
	len = HASH_STEP - (ld->u.chs.idx & (HASH_STEP - 1));
	ld->u.chs.idx += len;
	if (__unlikely(ld->u.chs.idx > ld->u.chs.data_end)) {
		len -= ld->u.chs.idx - ld->u.chs.data_end;
	}
	add_hash(ld, data, len);
	if (__unlikely(r = KERNEL$LOCKUP_WAIT_LOOP((IORQ *)rq, rq->handle))) return r;
	goto again;
}

static int SWAPLD_CHECK_FINAL(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	static __u8 file_hash[LINK_HASH_LENGTH];
	if (__unlikely(r = SWAPLD_READ(ld, ld->from + __offsetof(struct link_header, hash), file_hash, LINK_HASH_LENGTH, rq))) return r;
	get_hash(ld->link_hash.hash, ld);
	if (__unlikely(memcmp(ld->link_hash.hash, file_hash, LINK_HASH_LENGTH))) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "HASH %s DOESN'T MATCH", hash_name(ld));
		ld->state = LD_ERROR;
		return -EIO;
	}
	ld->state = LD_CHECKSUMMED;
	return 0;
}

static int SWAPLD_GET_SECTIONS(LDCACHE *ld, IOCTLRQ *rq)
{
	unsigned i;
	int r;
	__f_off secs;
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->from + __offsetof(struct link_header, sections), &secs, rq))) return r;
	secs += ld->from;
	for (i = 0; i < 5; i++) {
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, secs + i * sizeof(struct section) + __offsetof(struct section, offset), &ld->u.rel.sec_base[i], rq))) return r;
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, secs + i * sizeof(struct section) + __offsetof(struct section, len), &ld->u.rel.sec_len[i], rq))) return r;
	}
	return 0;
}

static int SWAPLD_SETUP_RELOC(LDCACHE *ld, unsigned long relptr, __f_off val, IOCTLRQ *rq)
{
/* in case of page fault, this must not change anything */
	int r;
	__f_off diff, reloc_ptr;
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, relptr + __offsetof(struct relocs, last_val), &diff, rq))) return r;
	diff = val - diff;
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, relptr + __offsetof(struct relocs, rel), &reloc_ptr, rq))) return r;
	if (__unlikely(r = SWAPLD_WRITE_F_OFF(ld, relptr + __offsetof(struct relocs, last_val), &val, rq))) return r;
/* we've written last_val --- point of no return */
	ld->u.rel.reloc_ptr = reloc_ptr;
	ld->u.rel.diff = diff;
	ld->u.rel.cur_section = REL_DEFAULT_SECTION;
	ld->u.rel.cur_type = REL_DEFAULT_TYPE;
	ld->u.rel.cur_pos = 0;
	ld->u.rel.rle_count = 0;
	return 0;
}

static int SWAPLD_DO_RELOC_STEPS(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	__u8 b0, b1, b2;
	__f_off cur_pos, reloc_ptr;
	if (__unlikely(!ld->u.rel.diff)) return 0;
	next_reloc:
	if (__unlikely(ld->u.rel.rle_count)) {
		cur_pos = ld->u.rel.cur_pos + ld->u.rel.rle_step;
		reloc_ptr = ld->u.rel.reloc_ptr;
		goto do_reloc;
	}
	if (__unlikely(ld->u.rel.reloc_ptr >= ld->u.rel.sec_len[0])) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "RELOCATIONS RAN OUT OF SECTION .HEAD");
		eftype:
		ld->state = LD_ERROR;
		return -EFTYPE;
	}
	if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + ld->u.rel.reloc_ptr, &b0, rq))) return r;
	if (__likely(b0 < REL_SPECIAL)) {
		unsigned long addr;
		__f_off val;
		/* ld->u.rel. must not be modified until write is committed */
		cur_pos = ld->u.rel.cur_pos + b0;
		reloc_ptr = ld->u.rel.reloc_ptr + 1;

		do_reloc:
		/*__debug_printf("reloc @ %x (%x)\n", cur_pos, ld->u.rel.reloc_ptr);*/
		if (__unlikely(cur_pos + sizeof(__f_off) > ld->u.rel.sec_len[ld->u.rel.cur_section])) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "RELOCATION POINTING OUT OF SECTION");
			goto eftype;
		}
		addr = ld->from + ld->u.rel.sec_base[ld->u.rel.cur_section] + cur_pos;
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, addr, &val, rq))) return r;
		if (__likely(ld->u.rel.cur_type == REL_SPECIAL_TYPE_ADD))
			val += ld->u.rel.diff;
		else
			val -= ld->u.rel.diff;
		if (__unlikely(r = SWAPLD_WRITE_F_OFF(ld, addr, &val, rq))) return r;

		ld->u.rel.cur_pos = cur_pos;
		ld->u.rel.reloc_ptr = reloc_ptr;
		if (__unlikely(ld->u.rel.rle_count)) ld->u.rel.rle_count--;

	} else if (__likely(b0 & REL_SPECIAL_LONG1)) {
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + ld->u.rel.reloc_ptr + 1, &b1, rq))) return r;
		cur_pos = ld->u.rel.cur_pos + ((b0 & REL_SPECIAL_LONG1_VAL) << 8) + b1;
		reloc_ptr = ld->u.rel.reloc_ptr + 2;
		goto do_reloc;
	} else if (b0 & REL_SPECIAL_RLE) {
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + ld->u.rel.reloc_ptr + 1, &b1, rq))) return r;
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + ld->u.rel.reloc_ptr + 2, &b2, rq))) return r;
		ld->u.rel.reloc_ptr += 2;
		ld->u.rel.rle_step = b1;
		ld->u.rel.rle_count = (b0 & REL_SPECIAL_RLE_REPEAT_MINUS_3) + 3;
		if (__unlikely(ld->u.rel.rle_count == REL_SPECIAL_RLE_REPEAT_MINUS_3 + 3)) ld->u.rel.rle_count += b2, ld->u.rel.reloc_ptr++;
	} else if (b0 & REL_SPECIAL_LONG2) {
		int i = b0 & REL_SPECIAL_LONG2_BYTES_MINUS_1;
		reloc_ptr = ld->u.rel.reloc_ptr + 1;
		cur_pos = ld->u.rel.cur_pos;
		do {
			if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + reloc_ptr, &b1, rq))) return r;
			cur_pos += b1 << (8 * i);
			reloc_ptr++;
		} while (--i >= 0);
		goto do_reloc;
	} else {
		if ((b0 & REL_SPECIAL_TYPE_MASK) == REL_SPECIAL_TYPE_END) return 0;
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + ld->u.rel.reloc_ptr + 1, &b1, rq))) return r;
		ld->u.rel.cur_type = b0 & REL_SPECIAL_TYPE_MASK;
		if (__unlikely((b0 & REL_SPECIAL_TYPE_MASK) >= REL_SPECIAL_TYPE_INVALID)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "INVALID RELOCATION TYPE");
			goto eftype;
		}
		if (__unlikely(b1 >= 4)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "INVALID RELOCATION SECTION");
			goto eftype;
		}
		ld->u.rel.cur_type = b0 & REL_SPECIAL_TYPE_MASK;
		ld->u.rel.cur_section = b1;
		ld->u.rel.reloc_ptr += 2;
		ld->u.rel.cur_pos = 0;
	}
	if (__unlikely(r = KERNEL$LOCKUP_WAIT_LOOP((IORQ *)rq, rq->handle))) return r;
	goto next_reloc;
}

static void SET_LD_RELOCATED(LDCACHE *ld);

static int SWAPLD_RELOC(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	if (ld->state == LD_CHECKSUMMED) {
		if (__unlikely(r = SWAPLD_GET_SECTIONS(ld, rq))) return r;
		ld->state = LD_RELOCATING;
		ld->u.rel.section_to_reloc = -1;
	} else {
		if (__unlikely(ld->state != LD_RELOCATING)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "LDCACHE_RELOC IN INVALID STATE");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
	}

	if (__likely(ld->u.rel.section_to_reloc < 0)) {
		__f_off secs;
		unsigned secn;
		__f_off ptr;

		reloc_next_section:
		secn = ld->u.rel.section_to_reloc + 1;
		if (__unlikely(secn == 5)) {
			SET_LD_RELOCATED(ld);
			return 0;
		}
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->from + __offsetof(struct link_header, sections), &secs, rq))) return r;
		ptr = ld->from + ld->u.rel.sec_base[secn];
		secs += ld->from;
		if (__unlikely(r = SWAPLD_WRITE_F_OFF(ld, secs + secn * sizeof(struct section) + __offsetof(struct section, ptr), &ptr, rq))) return r;
		if (__unlikely(r = SWAPLD_SETUP_RELOC(ld, secs + secn * sizeof(struct section) + __offsetof(struct section, relocs), ptr, rq))) return r;
		ld->u.rel.section_to_reloc = secn;
	}

	if (__unlikely(r = SWAPLD_DO_RELOC_STEPS(ld, rq))) return r;

	goto reloc_next_section;
}

static void SET_LD_RELOCATED(LDCACHE *ld)
{
	ld->state = LD_IFACE_SELECTING;
	ld->u.rel.current_iface = MAXUINT;
}

static int SWAPLD_FINISH_IFACE(LDCACHE *ld, IOCTLRQ *rq);

static int SWAPLD_SET_IFACE(LDCACHE *ld, LD_SET_IFACE *ldi, IOCTLRQ *rq)
{
	int r;
	unsigned n_libs;
	__f_off libs;
	char *s;
	if (__unlikely(ld->state != LD_IFACE_SELECTING)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "LDCACHE_SET_IFACE IN INVALID STATE");
		einval:
		ld->state = LD_ERROR;
		return -EINVAL;
	}
	if (__unlikely(r = SWAPLD_FINISH_IFACE(ld, rq))) return r;
	if (__unlikely(ldi->iface < ld->u.rel.current_iface + 1)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "SETTING INTERFACE BACKWARD");
		goto einval;
	}
	if (__unlikely(r = SWAPLD_READ_UNSIGNED(ld, ld->from + __offsetof(struct link_header, n_libs), &n_libs, rq))) return r;
	if (__unlikely(ldi->iface >= n_libs)) {
		_snprintf(ld->u.err.error, sizeof ld->u.err.error, "INVALID INTERFACE");
		goto einval;
	}
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->from + __offsetof(struct link_header, libs), &libs, rq))) return r;
	libs += ld->from + ldi->iface * sizeof(struct library);
	if (__unlikely(r = SWAPLD_READ_F_OFF(ld, libs + __offsetof(struct library, syms), &ld->u.rel.symbols, rq))) return r;
	ld->u.rel.symbols += ld->from;
	if (__unlikely(r = SWAPLD_READ_UNSIGNED(ld, libs + __offsetof(struct library, n_syms), &ld->u.rel.n_symbols, rq))) return r;
	if (__unlikely(r = SWAPLD_READ_STRING(ld, libs + __offsetof(struct library, lib_name), &s, rq))) return r;

	REL_Init(&ld->u.rel.rel_ctx);
	REL_Update(&ld->u.rel.rel_ctx, s, strlen(s) + 1);

	ld->state = LD_IFACE_SELECTING;
	ld->u.rel.current_iface = ldi->iface;
	return 0;
}

static int SWAPLD_RELOC_SYMBOLS(LDCACHE *ld, LD_SET_SYMBOLS *lds, IOCTLRQ *rq)
{
	int r;
	reloc_again:
	if (__likely(ld->state != LD_IFACE_RELOCATING)) {
		unsigned st, nst, namelen;
		int c;
		LD_SET_SYMBOLS_RECORD *lr;
		char *s;
		static char name[LINK_DYNAMIC_SYMBOL_NAME_LEN];
		__f_off val;
		if (__unlikely(ld->state != LD_IFACE_SELECTING)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "LDCACHE_RELOC_SYM IN INVALID STATE");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
		st = lds->start;
		if (__unlikely(st > PAGE_CLUSTER_SIZE - LD_SET_SYMBOLS_RECORD_LEN(0))) {
			out_of_page:
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "BAD START VALUE");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
		lr = (LD_SET_SYMBOLS_RECORD *)((char *)lds + st);
		namelen = lr->namelen;
		if (__unlikely(namelen >= LINK_DYNAMIC_SYMBOL_NAME_LEN)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "TOO LONG SYMBOL");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
		if (__unlikely(!namelen)) {
			return 0;
		}
		nst = st + LD_SET_SYMBOLS_RECORD_LEN(namelen);
		if (__unlikely(nst > PAGE_CLUSTER_SIZE - LD_SET_SYMBOLS_RECORD_LEN(0))) goto out_of_page;
		*(char *)mempcpy(name, lr->name, namelen) = 0;
		val = lr->value;
		/*__debug_printf("u1: \"%s\", %x\n", name, val);*/
		REL_Update(&ld->u.rel.rel_ctx, name, namelen + 1);
		REL_Update(&ld->u.rel.rel_ctx, &val, sizeof(__f_off));
		if (!ld->u.rel.n_symbols) {
			next_sym:
			lds->start = nst;
			goto test_again;
		}
		if (__unlikely(r = SWAPLD_READ_STRING(ld, ld->u.rel.symbols + __offsetof(struct lib_sym, name), &s, rq))) return r;
		c = strcmp(s, name);
		if (__likely(c > 0)) {
			goto next_sym;
		}
		if (__unlikely(c < 0)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "SYMBOL NOT FOUND: `%s'", s);
			ld->state = LD_ERROR;
			return -ENOENT;
		}
		if (__unlikely(r = SWAPLD_SETUP_RELOC(ld, ld->u.rel.symbols + __offsetof(struct lib_sym, reloc), val, rq))) return r;
		ld->state = LD_IFACE_RELOCATING;
		lds->start = nst;
	}
	if (__unlikely(r = SWAPLD_DO_RELOC_STEPS(ld, rq))) return r;
	ld->state = LD_IFACE_SELECTING;
	ld->u.rel.symbols += sizeof(struct lib_sym);
	ld->u.rel.n_symbols--;
	test_again:
	if (__unlikely(r = KERNEL$LOCKUP_WAIT_LOOP((IORQ *)rq, rq->handle))) return r;
	goto reloc_again;
}

static int SWAPLD_FINISH_IFACE(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	if (__likely(ld->u.rel.current_iface != MAXUINT)) {
		char *s;
		__u8 digest[REL_DIGEST_LENGTH];
		unsigned i;
		if (__unlikely(ld->u.rel.n_symbols)) {
			if (__unlikely(r = SWAPLD_READ_STRING(ld, ld->u.rel.symbols + __offsetof(struct lib_sym, name), &s, rq))) return r;
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "SYMBOL NOT FOUND: `%s'", s);
			ld->state = LD_ERROR;
			return -ENOENT;
		}
		REL_Final(digest, &ld->u.rel.rel_ctx);
		for (i = 0; i < REL_DIGEST_LENGTH; i++)
			ld->link_hash.hash[LINK_HASH_LENGTH + i] ^= digest[i];
	}
	return 0;
}

static int SWAPLD_FINALIZE(LDCACHE *ld, IOCTLRQ *rq)
{
	int r;
	char *s;
	if (ld->state != LD_FINALIZING) {
		if (__unlikely(ld->state != LD_IFACE_SELECTING)) {
			_snprintf(ld->u.err.error, sizeof ld->u.err.error, "LDCACHE_FINALIZE IN INVALID STATE");
			ld->state = LD_ERROR;
			return -EINVAL;
		}
		if (__unlikely(r = SWAPLD_FINISH_IFACE(ld, rq))) return r;

		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->from + __offsetof(struct link_header, syms), &ld->u.rel.symbols, rq))) return r;
		ld->u.rel.symbols += ld->from;
		if (__unlikely(r = SWAPLD_READ_UNSIGNED(ld, ld->from + __offsetof(struct link_header, n_exported_syms), &ld->u.rel.n_symbols, rq))) return r;
		if (__unlikely(r = SWAPLD_READ_STRING(ld, ld->from + __offsetof(struct link_header, lib_name), &s, rq))) return r;
		REL_Init(&ld->u.rel.rel_ctx);
		REL_Update(&ld->u.rel.rel_ctx, s, strlen(s) + 1);
		ld->state = LD_FINALIZING;
	}
	while (ld->u.rel.n_symbols) {
		__f_off value;
		__u8 section;
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->u.rel.symbols + __offsetof(struct sym, name), &value, rq))) return r;
		if (__unlikely(r = SWAPLD_READ_BYTE(ld, ld->from + value - 1, &section, rq))) return r;
		if (__unlikely(r = SWAPLD_READ_F_OFF(ld, ld->u.rel.symbols + __offsetof(struct sym, value), &value, rq))) return r;
		if (__likely(section != 0xff)) {
			if (__unlikely(section >= 5)) {
				_snprintf(ld->u.err.error, sizeof ld->u.err.error, "INVALID EXPORTED SYMBOL SECTION");
				ld->state = LD_ERROR;
				return -EFTYPE;
			}
			value += ld->from + ld->u.rel.sec_base[section];
		}
		if (__unlikely(r = SWAPLD_READ_STRING(ld, ld->u.rel.symbols + __offsetof(struct sym, name), &s, rq))) return r;
		/*__debug_printf("u2: \"%s\", %x (%d/%x)\n", s, value, section, section != 255 ? ld->u.rel.sec_base[section] : -1);*/
		REL_Update(&ld->u.rel.rel_ctx, s, strlen(s) + 1);
		REL_Update(&ld->u.rel.rel_ctx, &value, sizeof(__f_off));
		ld->u.rel.symbols += sizeof(struct sym);
		ld->u.rel.n_symbols--;
		if (__unlikely(r = KERNEL$LOCKUP_WAIT_LOOP((IORQ *)rq, rq->handle))) return r;
	}
	REL_Final(ld->exported_hash, &ld->u.rel.rel_ctx);

	if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(rq, ld->exported_hash, sizeof ld->exported_hash))) {
		if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);	
		return r;
	}

	SWAPLD_SET_ESTABLISHED(ld);
	return 0;
}

int LDCACHE_IOCTL(SWAPNODE *s, IOCTLRQ *rq)
{
	union {
		LD_PREPARE ldp;
		LD_SET_IFACE ldi;
		LD_QUERY ldq;
	} u;
	static LD_GET_ERROR ldg;
	int r;
	LDCACHE *ld;
	switch (rq->ioctl) {
		case IOCTL_SWAPPER_LDCACHE_GET_ERROR: {
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) goto notld;
			if (__unlikely(ld->state != LD_ERROR)) {
				r = -EINVAL;
				break;
			}
			memset(&ldg, 0, sizeof ldg);
			strlcpy(ldg.error, ld->u.err.error, sizeof ldg.error);
			if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(rq, &ldg, sizeof ldg))) {
				if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);	
				break;
			}
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_PREPARE: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &u.ldp, sizeof u.ldp))) {
				if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_READ);	
				break;
			}
			r = SWAPLD_PREPARE(s, &u.ldp, rq);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_CHECK: {
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) {
				notld:
				r = -ENOTBLK;
				break;
			}
			r = SWAPLD_CHECK(ld, rq);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_RELOC: {
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) goto notld;
			r = SWAPLD_RELOC(ld, rq);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_SET_IFACE: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &u.ldi, sizeof u.ldi))) {
				if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_READ);	
				break;
			}
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) goto notld;
			r = SWAPLD_SET_IFACE(ld, &u.ldi, rq);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_RELOC_SYM: {
			LD_SET_SYMBOLS *lds;
			vspace_unmap_t *unmap;
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) goto notld;
			if (__unlikely(rq->v.len != PAGE_CLUSTER_SIZE)) {
				r = -EINVAL;
				break;
			}
			RAISE_SPL(SPL_VSPACE);
			lds = rq->v.vspace->op->vspace_map(&rq->v, PF_RW, &unmap);
			LOWER_SPL(SPL_FS);
			if (__unlikely(!lds)) {
				DO_PAGEIN_NORET(rq, &rq->v, PF_RW);
				r = 1;
				break;
			}
			if (__unlikely(__IS_ERR(lds))) {
				r = __PTR_ERR(lds);
				break;
			}
			r = SWAPLD_RELOC_SYMBOLS(ld, lds, rq);
			RAISE_SPL(SPL_VSPACE);
			unmap(lds);
			LOWER_SPL(SPL_FS);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_FINALIZE: {
			ld = SWAPLD_FIND(s, rq->param);
			if (__unlikely(!ld)) goto notld;
			r = SWAPLD_FINALIZE(ld, rq);
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_QUERY: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &u.ldq, sizeof u.ldq))) {
				if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_READ);	
				break;
			}
			r = SWAPLD_QUERY(&u.ldq.h, &u.ldq.addr);
			if (__likely(!r)) {
				if (__likely(r = KERNEL$PUT_IOCTL_STRUCT(rq, &u.ldq, sizeof u.ldq))) {
					if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);	
					break;
				}
			}
			break;
		}
		case IOCTL_SWAPPER_LDCACHE_ATTACH: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &u.ldq, sizeof u.ldq))) {
				if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_READ);
				break;
			}
			r = SWAPLD_ATTACH(s, &u.ldq.h, u.ldq.addr, rq);
			if (__unlikely(r > 0)) {
				if (r == 2) {
					WQ_WAIT_F(&KERNEL$FREEMEM_WAIT, rq);
				} else {
					CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
				}
				break;
			}
			if (__likely(!r)) {
				if (__likely(r = KERNEL$PUT_IOCTL_STRUCT(rq, &u.ldq, sizeof u.ldq))) {
					if (__likely(r == 1)) DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);	
					break;
				}
			}
			break;
		}
		default: {
			r = -ENOOP;
			break;
		}
	}
	CACHE_DONE();
	return r;
}

