#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/LIBPROC.H>
#include <ARCH/BITOPS.H>
#include <SYS/STATFS.H>

#include "SWAPPER.H"

static void *SWAP_CLONE(HANDLE *hp, HANDLE *h, int open_flags);
static void *SWAP_LOOKUP(HANDLE *h, char *str, int open_flags);
static void *SWAP_CREATE(HANDLE *h, char *str, int open_flags);
static void *SWAP_DELETE(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp);
extern IO_STUB SWAP_IOCTL;

__const__ HANDLE_OPERATIONS SWAP_OPERATIONS = {
	SPL_X(SPL_FS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	SWAP_GET_PAGEIN_RQ,
	SWAP_GET_PAGE,
	SWAP_SWAP_OP,
	SWAP_CLONE,
	SWAP_LOOKUP,
	SWAP_CREATE,
	SWAP_DELETE,
	NULL,			/* rename */
	NULL,			/* lookup_io */
	SWAP_INSTANTIATE,
	NULL,			/* leave */
	SWAP_DETACH,
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	SWAP_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static __finline__ int str2jobname(char *str, __u64 *jobname)
{
#define j(x)	(((char *)jobname)[x])
	char a;
	unsigned char h;
	*jobname = 0;
	h = 0;
	a = str[0];
	if (__unlikely(!a)) return -1;
	h += a;
	j(0) = a;
	a = str[1];
	if (!a) goto ret;
	h += a;
	j(1) = a;
	a = str[2];
	if (!a) goto ret;
	h += a;
	j(2) = a;
	a = str[3];
	if (!a) goto ret;
	h += a;
	j(3) = a;
	a = str[4];
	if (!a) goto ret;
	h += a;
	j(4) = a;
	a = str[5];
	if (!a) goto ret;
	h += a;
	j(5) = a;
	a = str[6];
	if (!a) goto ret;
	h += a;
	j(6) = a;
	a = str[7];
	if (!a) goto ret;
	h += a;
	j(7) = a;
	a = str[8];
	if (__unlikely(a)) return -1;
	ret:
	h &= CHILDHASH_SIZE - 1;
	return h;
}

void *SWAP_INSTANTIATE(HANDLE *h, IORQ *rq, int open_flags)
{
	SWAPNODE *s = h->fnode;
	ADD_TO_XLIST(&s->handles, &h->fnode_entry);
	return NULL;
}

void SWAP_DETACH(HANDLE *h)
{
	DEL_FROM_LIST(&h->fnode_entry);
#if __DEBUG >= 1
	h->fnode = NULL;
#endif
}

__finline__ SWAPNODE *SWAP_FIND_NODE(SWAPNODE *s, char *str)
{
	SWAPNODE *ss;
	__u64 j;
	int x = str2jobname(str, &j);
	if (__unlikely(x < 0)) return __ERR_PTR(-ENAMETOOLONG);
	XLIST_FOR_EACH(ss, &s->childhash[x], SWAPNODE, hash)
		if (__likely(ss->jobname == j)) return ss;
	return __ERR_PTR(-ENOENT);
}

static void *SWAP_CLONE(HANDLE *hp, HANDLE *h, int open_flags)
{
	char *str;
	unsigned depth = 0;
	SWAPNODE *n;
	h->op = &SWAP_OPERATIONS;
	n = root;
	while ((str = KERNEL$PROC_PATH(h->file_addrspace, &depth))) {
		n = SWAP_FIND_NODE(n, str);
		if (__unlikely(__IS_ERR(n))) {
			return n;
		}
	}
	h->fnode = n;
	return NULL;
}

static void *SWAP_LOOKUP(HANDLE *h, char *str, int open_flags)
{
	SWAPNODE *p;
	if (__unlikely(str[0] == '^')) goto mod;
	p = SWAP_FIND_NODE(h->fnode, str);
	if (__likely(!__IS_ERR(p))) {
		h->fnode = p;
		return NULL;
	}
	return p;

	mod:
	if (!_strcasecmp(str, "^TTYP")) return SWAPTTYP_LOOKUP(h);
	if (!_strcasecmp(str, "^PTYP")) return SWAPPTY_LOOKUP(h);
	return __ERR_PTR(-EBADMOD);
}

static void *SWAP_CREATE(HANDLE *h, char *str, int open_flags)
{
	QUOTA *zap;
	SWAPNODE *s, *ss;
	__u64 j;
	int x = str2jobname(str, &j);
	if (__unlikely(x < 0)) return __ERR_PTR(-ENAMETOOLONG);
	s = h->fnode;
	if (__unlikely(s->depth == MAX_PROC_DEPTH - 1)) {
		return __ERR_PTR(-EDQUOT);
	}
	if (__unlikely(!(ss = __slalloc(&swapnodes)))) {
		if (!UNPAGED_ALLOC_FAILED()) return (void *)2;
		return &KERNEL$FREEMEM_WAIT;
	}
	init_swapnode(ss);
	QALLOC(&s->unpageq, sizeof(SWAPNODE), unpageq_isroot, unpageq_parent, Q_NULL_CALL, zap, {
		__slow_slfree(ss);
		OUT_OF_UNPAGED(zap);
		return (void *)2;
	});
#if __DEBUG >= 2
	{
		int i;
		for (i = 0; i < TOP_PGDIR_SIZE; i++) if (__unlikely(ss->pagedir[i] != NULL))
			KERNEL$SUICIDE("SWAP_CREATE: NOT-CONSTRUCTED PAGEDIR");
	}
#endif
	ss->jobname = j;
	s = h->fnode;
	ss->parent = s;
	ss->depth = s->depth + 1;
	ADD_TO_XLIST(&s->childhash[x], &ss->hash);
	h->fnode = ss;
	return NULL;
}

static void *SWAP_DELETE(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp)
{
	if (__unlikely(h->fnode == root)) return __ERR_PTR(-EBUSY);
	SWAP_LOCK(SW_LOCK);
	SWAP_DESTROY_SWAPNODE_AND_DATA(h->fnode);
	SWAP_LOCK(SW_UNLOCK);
	return NULL;
}

static void SWAP_STATFS(IOCTLRQ *RQ)
{
	int r;
	static struct statfs stat;
	memset(&stat, 0, sizeof stat);
	strncpy((char *)&stat.f_fsid, "SWAPPER", sizeof stat.f_fsid);
	strcpy(stat.f_fsmntonname, "SYS$SWAPPER:/");
	strcpy(stat.f_fsmntfromname, !swapper_device[0] ? "MEMORY" : swapper_device);
	strcpy(stat.f_fstypename, "SWAPPER");
	memcpy(&stat.f_type, stat.f_fstypename, sizeof stat.f_type);
	stat.f_bsize = PAGE_CLUSTER_SIZE;
	stat.f_iosize = PAGE_CLUSTER_SIZE << SWP_MAP_BITS_L2;
	stat.f_flags = MNT_NOATIME | MNT_ASYNC;

	stat.f_blocks = swppages;
	stat.f_bfree = swppages_free;
	stat.f_bavail = stat.f_bfree;
	stat.f_files = swppages ? swppages - 1 : 0;
	stat.f_ffree = swppages_free;
	stat.f_bavail = stat.f_ffree;
	stat.f_namelen = 8;

	if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &stat, sizeof stat)) == 1)) {
		DO_PAGEIN_NORET(RQ, &RQ->v, PF_WRITE);
		return;
	}
	RQ->status = r;
	CALL_AST(RQ);
}

/* Trick:
	There may be at most 2 mapping done via KERNEL$MAP_PHYSICAL_PAGE per SPL
	We need three mappings at LDCACHE_IOCTL --- so we start at SPL lower by
	one and raise SPL immediatelly --- this way we can do 4 mappings
*/

DECL_IOCALL(SWAP_IOCTL, SPL_FS - 1, IOCTLRQ)
{
	HANDLE *h;
	SWAPNODE *s;
	RAISE_SPL(SPL_FS);
	h = RQ->handle;
	if (__unlikely(h->op != &SWAP_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_FS));
	if (__unlikely((s = h->fnode) == root)) {
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	}
	switch (RQ->ioctl) {
		case IOCTL_SWAPPER_ISUSED: {
			int i;
			for (i = 0; i < TOP_PGDIR_SIZE; i++) if (__unlikely(s->pagedir[i] != NULL)) {
				RQ->status = 1;
				RETURN_AST(RQ);
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_SWAPPER_PUTPAGE: {
			PAGE *p;
			char *v;
			vspace_unmap_t *unmap;
			char *map;
			if (__unlikely((unsigned long)RQ->param >= VMSPACE) || __unlikely(RQ->v.len > PAGE_CLUSTER_SIZE)) {
				oor:
				RQ->status = -ERANGE;
				RETURN_AST(RQ);
			}
			if (__unlikely((RQ->param & (PAGE_CLUSTER_SIZE - 1)) + RQ->v.len > PAGE_CLUSTER_SIZE)) goto oor;
			if (__unlikely(((unsigned long)RQ->v.ptr & (PAGE_CLUSTER_SIZE - 1)) + RQ->v.len > PAGE_CLUSTER_SIZE)) goto oor;
			RAISE_SPL(SPL_VSPACE);
			map = RQ->v.vspace->op->vspace_map(&RQ->v, PF_READ, &unmap);
			LOWER_SPL(SPL_FS);
			if (__unlikely(!map)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(__IS_ERR(map))) {
				RQ->status = __PTR_ERR(map);
				RETURN_AST(RQ);
			}
			check_quota(ACCT_FROM_SWAPNODE(h->fnode), "PUTPAGE 1");
			p = SWAP_CREATE_PAGE(ACCT_FROM_SWAPNODE(h->fnode), RQ->param, PF_WRITE, (IORQ *)RQ, h->name_addrspace);
			check_quota(ACCT_FROM_SWAPNODE(h->fnode), "PUTPAGE 2");
			if (__unlikely(!p)) {
				RAISE_SPL(SPL_VSPACE);
				unmap(map);
				LOWER_SPL(SPL_FS);
				RETURN;
			}
			RAISE_SPL(SPL_CACHE);
			KERNEL$CACHE_TOUCH_VM_ENTITY(&p->e, RQ->handle->name_addrspace);
			LOWER_SPL(SPL_FS);
			v = KERNEL$MAP_PHYSICAL_PAGE(p);
			memcpy(v + (RQ->param & (PAGE_CLUSTER_SIZE - 1)), map, RQ->v.len);
			KERNEL$UNMAP_PHYSICAL_BANK(v);
			RAISE_SPL(SPL_VSPACE);
			unmap(map);
			LOWER_SPL(SPL_FS);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_CAN_MMAP: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_STATFS: {
			SWAP_STATFS(RQ);
			RETURN;
		}
		default: {
			int r;
			if (__unlikely(r = LDCACHE_IOCTL(s, RQ))) {
				if (__likely(r == 1)) RETURN;
			}
			RQ->status = r;
			RETURN_AST(RQ);
		}
	}
}
