#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/BIO.H>
#include <SPAD/IOCTL.H>
#include <ARCH/LINUXPG.H>
#include <SPAD/VM.H>
#include <SPAD/TIMER.H>

#include "SWAPMAP.H"
#include "SWAPPER.H"
#include "STRUCT.H"

#if PAGE_CLUSTER_SIZE <= LINUX_PAGE_SIZE
too small page size
#endif

unsigned long total_pages;

PAGEZONE pagezone;
WQ freemem;
LIST_HEAD all_pages;

swp_map_t *swpalloc;
unsigned swpalloc_clusters;
unsigned swpalloc_min;
unsigned swpalloc_min_bit;

int swphndl;
unsigned swppages;
unsigned swppages_free;
unsigned optimal_sectors;

struct __slhead pagenodes;
struct __slhead pagedirs;
struct __slhead swapnodes;
#ifdef SWAP_TTY
struct __slhead swapttys;
#endif
struct __slhead ldcaches;
struct __slhead ldrefs;
struct __slhead swaprqs;
WQ swaprqs_free;

SWAPNODE *root;

static SWAPRQ *dummy_swaprq = NULL;

char swapper_device[__MAX_STR_LEN];

static void *dlrq;
static void *lnte;

static char read_buffer_[LINUX_PAGE_SIZE + 2 * BIO_SECTOR_SIZE];
static union swap_header* read_buffer;

static void PAGEDIR_CTOR(void *g, void *o);
static void SWAPNODE_CTOR(void *g, void *o);
static void LDCACHE_CTOR(void *g, void *o);
static void SWAP_INIT_ROOT(HANDLE *h, void *null);
static int SWAPPER_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	char *swpdev;
	int r, i;
	char **arg;
	int state;
	PAGE *p;
	OPENRQ openrq;
	CHHRQ ch;
	BIORQ biorq;
	BIODESC desc;
	IOCTLRQ ioctlrq;
	CONTIG_AREA_REQUEST car;
	DEVICE_REQUEST devrq;
	struct __param_table params[] = {
		"", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &swpdev; swpdev = NULL;
	if (__unlikely(PAGE_CLUSTER_SIZE != __PAGE_CLUSTER_SIZE)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER COMPILED WITH PAGE SIZE %u, KERNEL COMPILED WITH PAGE SIZE %u", (unsigned)PAGE_CLUSTER_SIZE, (unsigned)__PAGE_CLUSTER_SIZE);
		return -EINVAL;
	}
	total_pages = KERNEL$GET_MEMORY_SIZE(VM_TYPE_USER_UNMAPPED) >> (PG_SIZE_BITS + PG_CLUSTER_BITS);
	swphndl = -1;
	read_buffer = (union swap_header *)((unsigned long)(read_buffer_ + BIO_SECTOR_SIZE - 1) & ~(unsigned long)(BIO_SECTOR_SIZE - 1));
	WQ_INIT(&freemem, "SWAPPER$FREEMEM");
	WQ_INIT(&swaprqs_free, "SWAPPER$SWAPRQS_FREE");
	INIT_LIST(&all_pages);
	VFS$ZINIT(&pagezone, VM_TYPE_USER_UNMAPPED, "SWAPPER PAGES");
	if (SWAPDATA_INIT()) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T REFISTER SWAP VM ENTITY");
		r = -ENFILE;
		goto err_2;
	}
	if (__unlikely(LDCACHE_INIT())) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T REFISTER LDCACHE VM ENTITY");
		r = -ENFILE;
		goto err_1;
	}
	KERNEL$SLAB_INIT(&pagenodes, sizeof(PAGENODE), 0, VM_TYPE_USER_MAPPED, NULL, NULL, NULL, "SWAPPER$PAGENODE");
	KERNEL$SLAB_INIT(&pagedirs, sizeof(PAGEDIR), 0, VM_TYPE_USER_MAPPED, PAGEDIR_CTOR, NULL, NULL, "SWAPPER$PAGEDIR");
	KERNEL$SLAB_INIT(&swapnodes, sizeof(SWAPNODE), 0, VM_TYPE_USER_MAPPED, SWAPNODE_CTOR, NULL, NULL, "SWAPPER$SWAPNODE");
#ifdef SWAP_TTY
	KERNEL$SLAB_INIT(&swapttys, SIZEOF_SWAPTTY, 0, VM_TYPE_USER_MAPPED, NULL, NULL, NULL, "SWAPPER$SWAPTTY");
#endif
	KERNEL$SLAB_INIT(&ldcaches, sizeof(LDCACHE), 0, VM_TYPE_USER_MAPPED, LDCACHE_CTOR, NULL, NULL, "SWAPPER$LDCACHE");
	KERNEL$SLAB_INIT(&ldrefs, sizeof(LDREF), 0, VM_TYPE_USER_MAPPED, NULL, NULL, NULL, "SWAPPER$LDREF");
	KERNEL$SLAB_INIT(&swaprqs, sizeof(SWAPRQ), 0, VM_TYPE_USER_MAPPED, SWAPRQ_CTOR, NULL, NULL, "SWAPPER$SWAPRQ");

	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: SYNTAX ERROR");
		r = -EBADSYN;
		goto err0;
	}
	swppages_free = 0;
	if (!swpdev) {
		swapper_device[0] = 0;
		swppages = 0;
		goto skip_swap;
	}
	strcpy(swapper_device, swpdev);
	openrq.flags = O_RDWR | O_DIRECT;
	openrq.path = swpdev;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: ERROR OPENING DEVICE %s: %s", swpdev, strerror(-openrq.status));
		r = openrq.status;
		goto err0;
	}
	swphndl = openrq.status;
	ch.h = swphndl;
	ch.option = "DIRECT";
	ch.value = "";
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	ioctlrq.h = swphndl;
	ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_WRITE;
	ioctlrq.v.ptr = 0;
	ioctlrq.v.len = 0;
	ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&ioctlrq, KERNEL$IOCTL);
	optimal_sectors = ioctlrq.status >= 0 ? ioctlrq.status : 0;
	biorq.h = swphndl;
	biorq.sec = 0;
	biorq.nsec = (LINUX_PAGE_SIZE + BIO_SECTOR_SIZE - 1) >> BIO_SECTOR_SIZE_BITS;
	biorq.flags = BIO_READ;
	biorq.desc = &desc;
	biorq.proc = &KERNEL$PROC_KERNEL;
	biorq.fault_sec = -1;
	desc.v.ptr = (unsigned long)read_buffer;
	desc.v.len = biorq.nsec << BIO_SECTOR_SIZE_BITS;
	desc.v.vspace = &KERNEL$VIRTUAL;
	desc.next = NULL;
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (biorq.status < 0) {
		r = biorq.status;
		goto err1;
	}
	if (!memcmp(read_buffer->magic.magic, SWAP_LINUX_V1_MAGIC, 10)) {
		int i;
		for (i = 1; i < (LINUX_PAGE_SIZE - 10) * 8; i++) {
			if (!(read_buffer->magic.reserved[i >> 3] & (1 << (i & 7)))) break;
		}
		swppages = i / (PAGE_CLUSTER_SIZE / LINUX_PAGE_SIZE);
		for (; i < (LINUX_PAGE_SIZE - 10) * 8; i++) {
			if (read_buffer->magic.reserved[i >> 3] & (1 << (i & 7))) {
				badpg:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE HAS BAD PAGES; NOT SUPPORTED", swpdev);
				r = -EINVAL;
				goto err1;
			}
		}
	} else if (!memcmp(read_buffer->magic.magic, SWAP_LINUX_V2_MAGIC, 10)) {
		if (read_buffer->info.version != 1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE HAS VERSION %d; NOT SUPPORTED", swpdev, read_buffer->info.version);
			r = -EINVAL;
			goto err1;
		}
		if (read_buffer->info.nr_badpages) goto badpg;
		swppages = ((__u64)read_buffer->info.last_page + 1) / (PAGE_CLUSTER_SIZE / LINUX_PAGE_SIZE);
	} else {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: SWAP FILE SIGNATURE NOT FOUND", swpdev);
		r = -EINVAL;
		goto err1;
	}
	if (swppages <= 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: %s: NO PAGES IN SWAP FILE", swpdev);
		r = -EINVAL;
		goto err1;
	}
	if (swppages > MAXINT) swppages = MAXINT;

	skip_swap:

	for (i = 0; i < RESERVED_PAGES; i++) {
		while (__unlikely(!(p = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED)))) {
			if (KERNEL$OOM(VM_TYPE_WIRED_UNMAPPED)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: OUT OF MEMORY FOR RESERVED PAGES");
				r = -ENOMEM;
				goto err1;
			}
			if ((r = WQ_WAIT_SYNC_CANCELABLE(&KERNEL$FREEMEM_WAIT))) goto err1;
		}
		VFS$ZRESERVE(&pagezone, p);
	}

	for (i = 0; i < 6; i++) {
		if (__unlikely(r = KERNEL$SLAB_RESERVE(i == 0 ? &pagenodes : i == 1 ? &pagedirs : i == 2 ? &swapnodes : i == 3 ? &ldcaches : i == 4 ? &ldrefs : i == 5 ? &swaprqs : NULL, i == 5 ? 2 : 1))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: OUT OF MEMORY FOR SLABS");
			goto err1;
		}
	}

	dummy_swaprq = __slalloc(&swaprqs);
	if (__unlikely(!dummy_swaprq))
		KERNEL$SUICIDE("SWAPPER: CAN'T ALLOCATE DUMMY SWAPRQ");

	if (swppages) {
		car.flags = CARF_DATA;
		car.nclusters = swpalloc_clusters = (swppages + 8 * PAGE_CLUSTER_SIZE - 1) / (8 * PAGE_CLUSTER_SIZE);
		car.align = 0;
		SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
		if (car.status < 0) {
			r = car.status;
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: CAN'T ALLOCATE SWAP ALLOC MAP: %s", strerror(-r));
			goto err1;
		}
		swpalloc = car.ptr;
		memset(swpalloc, 0xff, swpalloc_clusters * PAGE_CLUSTER_SIZE);
		for (i = 1; i < swppages; i++) CLEAR_SWPALLOC_BIT(i);
		swpalloc_min = 0;
		swpalloc_min_bit = 0;
	} else swpalloc = NULL;

	if (!(root = __slalloc(&swapnodes)))
		KERNEL$SUICIDE("SWAPPER: CAN'T ALLOCATE ROOT SWAPNODE");
	init_swapnode(root);
	
	VOID_LIST_ENTRY(&root->hash);
	root->jobname = 0;
	root->parent = NULL;
	QINIT(&root->pageq);
	root->pageq.q_limit = MAXINT;
	QINIT2(&root->pageq);
	QINIT(&root->unpageq);
	{
		__u64 mem;
		mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_USER_MAPPED);
		mem >>= 3;
		if (mem > MAXLONG) mem = MAXINT;
		root->unpageq.q_limit = mem;
	}
	QINIT2(&root->unpageq);
	root->depth = 0;

	devrq.name = "SYS$SWAPPER";
	devrq.driver_name = "SWAPPER.SYS";
	devrq.flags = LNTE_PUBLIC;
	devrq.init_root_handle = SWAP_INIT_ROOT;
	devrq.dev_ptr = NULL;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = SWAPPER_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		r = devrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SWAPPER: ERROR REGISTERING DEVICE");
		goto err3;
	}
	lnte = devrq.lnte;
	strlcpy(KERNEL$ERROR_MSG(), devrq.name, __MAX_STR_LEN);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;

	err3:
	RAISE_SPL(SPL_FS);
	LDCACHE_DONE();
	LOWER_SPL(SPL_ZERO);
	__slow_slfree(root);
	if (swpalloc)
		KERNEL$VM_RELEASE_CONTIG_AREA(swpalloc, swpalloc_clusters);
	err1:
	if (swphndl != -1) KERNEL$FAST_CLOSE(swphndl);
	err0:
	LDCACHE_DONE();
	KERNEL$SLAB_DESTROY(&pagenodes);
	KERNEL$SLAB_DESTROY(&pagedirs);
	KERNEL$SLAB_DESTROY(&ldcaches);
	KERNEL$SLAB_DESTROY(&ldrefs);
	KERNEL$SLAB_DESTROY(&swapnodes);
#ifdef SWAP_TTY
	KERNEL$SLAB_DESTROY(&swapttys);
#endif
	if (dummy_swaprq != NULL) __slow_slfree(dummy_swaprq);
	KERNEL$SLAB_DESTROY(&swaprqs);
	err_1:
	SWAPDATA_DONE();
	err_2:
	VFS$ZDONE(&pagezone);
	return r;
}

static void PAGEDIR_CTOR(void *g, void *o)
{
	memset(o, 0, sizeof(PAGEDIR));
}

static void SWAPNODE_CTOR(void *g, void *o)
{
	SWAPNODE *s = o;
	int i;
	INIT_XLIST(&s->handles);
	INIT_XLIST(&s->ldrefs);
	for (i = 0; i < CHILDHASH_SIZE; i++) INIT_XLIST(&s->childhash[i]);
	memset(s->pagedir, 0, sizeof s->pagedir);
}

static void LDCACHE_CTOR(void *g, void *o)
{
	LDCACHE *s = o;
	memset(&s->pagedir, 0, sizeof s->pagedir);
	s->parent = NULL;
	CACHE_CONSTRUCT_VM_ENTITY(&s->vme);
	s->vme.type = ldcache_vm_entity;
}

static void SWAP_INIT_ROOT(HANDLE *h, void *null)
{
	h->fnode = root;
	h->op = &SWAP_OPERATIONS;
	/* if flags will be used, they must be also added to SWAP_CLONE */
}

static int SWAPPER_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	RAISE_SPL(SPL_FS);
	SWAP_LOCK(SW_LOCK);
	SWAP_DESTROY_SWAPNODE_AND_DATA(root);
	SWAP_LOCK(SW_UNLOCK);
	LDCACHE_DONE();
	while (__unlikely(!LIST_EMPTY(&all_pages))) KERNEL$SLEEP(1);
	LOWER_SPL(SPL_ZERO);
	if (swppages && __unlikely(swppages_free != swppages - 1))
		KERNEL$SUICIDE("SWAPPER_UNLOAD: SWPPAGES_FREE LEAKED: %u, SWPPAGES %u", swppages_free, swppages);
	if (swpalloc)
		KERNEL$VM_RELEASE_CONTIG_AREA(swpalloc, swpalloc_clusters);
	if (swphndl != -1) KERNEL$FAST_CLOSE(swphndl);
	SWAPDATA_DONE();
	VFS$ZDONE(&pagezone);
	KERNEL$SLAB_DESTROY(&pagenodes);
	KERNEL$SLAB_DESTROY(&pagedirs);
	KERNEL$SLAB_DESTROY(&ldcaches);
	KERNEL$SLAB_DESTROY(&ldrefs);
	KERNEL$SLAB_DESTROY(&swapnodes);
#ifdef SWAP_TTY
	KERNEL$SLAB_DESTROY(&swapttys);
#endif
	if (dummy_swaprq != NULL) __slow_slfree(dummy_swaprq);
	KERNEL$SLAB_DESTROY(&swaprqs);
	*release = dlrq;
	return 0;
}



