#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV_KRNL.H>
#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/RANDOM.H>
#include <ARCH/TIME.H>
#include <ARCH/BSF.H>
#include <ARCH/ROL.H>
#include <VALUES.H>
#include <TIME.H>
#include <SYS/TIME.H>
#include <SPAD/IOCTL.H>
#include <STDLIB.H>

#define USE_RIPEMD160

#ifdef USE_RIPEMD160
#include <RIPEMD.H>
#define HASH_CTX RIPEMD160_CTX
#define HASH_DIGEST_LENGTH RIPEMD160_DIGEST_LENGTH
#define HASH_Init RIPEMD160_Init
#define HASH_Update RIPEMD160_Update
#define HASH_Final RIPEMD160_Final
#endif
#ifdef USE_SHA1
#include <SHA.H>
#define HASH_CTX SHA_CTX
#define HASH_DIGEST_LENGTH SHA_DIGEST_LENGTH
#define HASH_Init SHA1_Init
#define HASH_Update SHA1_Update
#define HASH_Final SHA1_Final
#endif

#define POOL_SIZE	512

#define POOL_SWAP_RATIO	16

#define POOL_BITS	(POOL_SIZE * 8)
#define POOL_MAXBITS	(POOL_BITS * 2)
#define POOL_MINBITS	(HASH_DIGEST_LENGTH * 8)
#define POOL_WORDS	(POOL_SIZE / 4)

/* there's no locking --- race contitions can only add randomness :) */

static struct random_pool {
	int random_bits;
	__u32 data[POOL_WORDS];
	unsigned rotate;
	unsigned pos;
	unsigned rotate2;
	unsigned pos2;
} pools[2];

static WQ_DECL(pool_wait, "RANDOM$POOL_WAIT");

static void add_to_pool(struct random_pool *pool, void *ptr, int len)
{
	if (__likely(!((unsigned long)ptr & 3))) while (len >= 4) {
		
		if (__unlikely(++pool->pos2 >= POOL_WORDS)) pool->rotate2 += 17, pool->pos2 = 0;
		pool->rotate2 = (pool->rotate2 + 7) & 31;
		pool->data[pool->pos2 & (POOL_WORDS - 1)] ^= __ROL32(*(__u32 *)ptr, pool->rotate2), ptr = (__u8 *)ptr + 4, len -= 4;
	}
	while (__unlikely(len)) {
		if (__unlikely(++pool->pos2 >= POOL_WORDS)) pool->rotate2 += 17, pool->pos2 = 0;
		pool->rotate2 = (pool->rotate2 + 7) & 31;
		pool->data[pool->pos2 & (POOL_WORDS - 1)] ^= __ROL32(*(__u8 *)ptr, pool->rotate2), ptr = (__u8 *)ptr + 1, len--;
	}
}

static void input_random(RANDOM_CTX *ctx, void *ptr, int len)
{
	struct random_pool *pool;
	static unsigned long last_time;
	unsigned long tim;
	static struct random_pool *lastpool = NULL;
	static int lastpool_n = 0;
	if (__unlikely(pools[0].random_bits < POOL_MAXBITS)) pool = &pools[0];
	else if (__unlikely(pools[1].random_bits < POOL_MAXBITS)) pool = &pools[1];
	else if (__unlikely((tim = (unsigned long)time(NULL)) != last_time)) pool = &pools[0], last_time = tim;
	else return;
	if (__likely(pool == lastpool)) {
		if (__unlikely(++lastpool_n >= POOL_SWAP_RATIO)) {
			pool = (struct random_pool *)((char *)pools + (((char *)pool - (char *)pools) ^ sizeof(struct random_pool)));
			lastpool_n = 0;
		}
	} else {
		lastpool = pool;
		lastpool_n = 0;
	}
	if (__likely(ctx != NULL)) {
		long delta, delta2;
		unsigned long tm = TIMER_RANDOMNESS();
		delta = tm - ctx->last_time;
		ctx->last_time = tm;
		if (__unlikely(++pool->pos >= POOL_WORDS)) pool->rotate += 17, pool->pos = 0;
		pool->rotate = (pool->rotate + 7) & 31;
		pool->data[pool->pos & (POOL_WORDS - 1)] ^= __ROL32(tm, pool->rotate);
		delta2 = delta - ctx->last_delta;
		ctx->last_delta = delta;
		if (__likely(delta < 0)) delta = -delta;
		if (__likely(delta2 < 0)) delta2 = -delta2;
		if (__likely(delta2 < delta)) delta = delta2;
		if (delta >= 1 << 16) pool->random_bits += 16;
		else if (delta) pool->random_bits += __BSR(delta) + 1;
		if (__likely(pool[0].random_bits >= POOL_MINBITS))
			WQ_WAKE_ALL(&pool_wait);
	}
	if (__unlikely(len)) {
		add_to_pool(pool, ptr, len);
	}
}

#define RANDOM_NONBLOCK	1
#define RANDOM_STRONG	2

static void *random_lookup(HANDLE *h, char *str, int open_flags);
extern IO_STUB random_read;
extern IO_STUB random_write;
extern IO_STUB random_ioctl;

static __const__ HANDLE_OPERATIONS random_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	random_lookup,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	random_read,		/* READ */
	random_write,		/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	random_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static void *random_lookup(HANDLE *h, char *str, int open_flags)
{
	if (!_strcasecmp(str, "STRONG")) h->flags |= RANDOM_STRONG;
	else if (!_strcasecmp(str, "^NONBLOCK")) h->flags |= RANDOM_NONBLOCK;
	else return __ERR_PTR(-ENOENT);
	return NULL;
}

static unsigned char digest[HASH_DIGEST_LENGTH];

static VBUF rvbuf = { digest, HASH_DIGEST_LENGTH, SPL_X(SPL_DEV) };

static HASH_CTX ctx;

DECL_IOCALL(random_read, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	long s;
	struct random_pool *pool;
	if (__unlikely(h->op != &random_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(!RQ->v.len)) goto zlen;
	again:
	if (__unlikely((h->flags & RANDOM_STRONG) != 0)) {
		if (__unlikely(pools[0].random_bits < POOL_MINBITS)) {
			if (__likely(pools[0].random_bits + pools[1].random_bits >= POOL_MINBITS)) {
				add_to_pool(&pools[0], pools[1].data, POOL_SIZE);
				pools[0].random_bits += pools[1].random_bits;
				pools[1].random_bits = 0;
				goto ok;
			}
			if (RQ->progress != 0) goto zlen;
			if (h->flags & RANDOM_NONBLOCK) {
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&pool_wait, RQ);
			RETURN;
		}
		ok:
		pool = &pools[0];
	} else {
		pool = &pools[1];
	}
	HASH_Init(&ctx);
	HASH_Update(&ctx, (char *)pool->data, POOL_SIZE);
	HASH_Final(digest, &ctx);
	add_to_pool(pool, digest, sizeof digest);
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_put(&RQ->v, &rvbuf);
	if (__unlikely(!s)) goto pf;
	RQ->progress += s;
	s <<= 3;
	if (__unlikely(pool->random_bits > POOL_BITS)) pool->random_bits = POOL_BITS;
	if (__likely(pool->random_bits > s)) pool->random_bits -= s;
	else pool->random_bits = 0;
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (__unlikely(RQ->v.len != 0)) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);

	pf:
	DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
}

#define BUF_SIZE	4096

static unsigned char buffer[BUF_SIZE];

static __const__ VBUF vbuf = { buffer, BUF_SIZE, SPL_X(SPL_DEV) };

DECL_IOCALL(random_write, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	long s;
	if (__unlikely(h->op != &random_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(!RQ->v.len)) goto zlen;
	again:
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_get(&RQ->v, &vbuf);
	if (__unlikely(!s)) goto pf;
	RQ->progress += s;
	input_random(NULL, buffer, s);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	TEST_LOCKUP_LOOP(RQ, RETURN);
	if (__unlikely(RQ->v.len != 0)) goto again;
	zlen:
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);

	pf:
	DO_PAGEIN(RQ, &RQ->v, PF_READ);
}

DECL_IOCALL(random_ioctl, SPL_DEV, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &random_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_SELECT_READ: {
			if (__unlikely((h->flags & RANDOM_STRONG) != 0) && __unlikely(pools[0].random_bits + pools[1].random_bits < POOL_MINBITS)) {
				if (RQ->param) {
					WQ_WAIT_F(&pool_wait, RQ);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_SELECT_WRITE: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_NREAD: {
			if (!(h->flags & RANDOM_STRONG)) RQ->status = MAXINT;
			else {
				int nbits = pools[0].random_bits + pools[1].random_bits - POOL_MINBITS;
				if (nbits < 0) RQ->status = 0;
				else if (nbits > POOL_BITS - POOL_MINBITS) RQ->status = (POOL_BITS - POOL_MINBITS) >> 3;
				else RQ->status = nbits >> 3;
			}
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void init_root(HANDLE *h, void *data)
{
	h->op = &random_operations;
	h->flags = 0;
}

static int unload(void *p, void **release, char *argv[]);

static void *lnte, *dlrq;

int main(int argc, char *argv[])
{
	DEVICE_REQUEST drq;
	RANDOM_CTX init_ctx;
	struct timeval tv;
	unsigned u[2];
	__u64 tm;
	int r;
	if (__unlikely(argc > 1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RANDOM: SYNTAX ERROR");
		return -EBADSYN;
	}
	memset(&pools, 0, sizeof pools);

	gettimeofday(&tv, NULL);
	u[0] = tv.tv_sec & 0x55555555;
	u[1] = tv.tv_usec & 0x55555555;
	add_to_pool(&pools[0], &u, sizeof u);
	u[0] = tv.tv_sec & 0xaaaaaaaa;
	u[1] = tv.tv_usec & 0xaaaaaaaa;
	add_to_pool(&pools[1], &u, sizeof u);
	tm = TIMER_RANDOMNESS();
	u[0] = tm & 0x55555555;
	u[1] = (tm >> 32) & 0x55555555;
	add_to_pool(&pools[0], &u, sizeof u);
	u[0] = tm & 0xaaaaaaaa;
	u[1] = (tm >> 32) & 0xaaaaaaaa;
	add_to_pool(&pools[1], &u, sizeof u);

	memset(&init_ctx, 0, sizeof init_ctx);
	input_random(&init_ctx, &tv, sizeof tv);

	if (__unlikely(r = KERNEL$REGISTER_RANDOM_HOOK(input_random))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RANDOM: CAN'T REGISTER RANDOM HOOK");
		return r;
	}
	drq.init_root_handle = init_root;
	drq.name = "SYS$RANDOM";
	drq.driver_name = "RANDOM.SYS";
	drq.flags = LNTE_PUBLIC;
	drq.dcall = NULL;
	drq.dcall_type = NULL;
	drq.dctl = NULL;
	drq.unload = unload;
	SYNC_IO_CANCELABLE(&drq, KERNEL$REGISTER_DEVICE);
	if (__unlikely(drq.status < 0)) {
		if (drq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SYSCALLS: COULD NOT REGISTER DEVICE %s: %s", drq.name, strerror(-drq.status));
		KERNEL$REGISTER_RANDOM_HOOK(NULL);
		return drq.status;
	}
	arc4random_stir();
	lnte = drq.lnte;
	strlcpy(KERNEL$ERROR_MSG(), drq.name, __MAX_STR_LEN);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static int unload(void *p, void **release, char *argv[])
{
	int r;
	arc4random_stir();
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	KERNEL$REGISTER_RANDOM_HOOK(NULL);
	WQ_WAKE_ALL(&pool_wait);
	*release = dlrq;
	return 0;
}

