#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TTY.H>
#include <SYS/TYPES.H>
#include <VALUES.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/IOCTL.H>
#include <TERMIOS.H>
#include <STDLIB.H>

#define IN_BUF		16

#define OUT_BUF		2048

struct tty_area {
	TTY tty;
	int kbd_h;
	int scr_h;
	SIORQ kbd_rq;
	SIORQ scr_rq;
	int state;
	char in_buf[IN_BUF];
	char name[__MAX_STR_LEN];
	void *lnte;
	void *dlrq;
};

#define TTY_KBD_PENDING		1
#define TTY_SCR_PENDING		2

static int unload(void *p, void **release, char *argv[]);
static void TTY_WRITEOUT(TTY *tty);

extern AST_STUB TTY_IN;
extern AST_STUB TTY_OUT;

int main(int argc, char *argv[])
{
	int lines = -1, columns = -1;
	int xpixels = 0, ypixels = 0;
	char *kbd = NULL, *scr = NULL, *kbdscr = NULL, *name = NULL;
	char **arg;
	int state;
	IOCTLRQ io;
	struct tty_window_size ws;
	struct __param_table params[] = {
		"KBD", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"SCR", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"KBDSCR", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"TTY", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"BUFSIZE", __PARAM_INT, __MAX_STR_LEN, MAXINT, NULL,
		"LINES", __PARAM_INT, 1, 1024 + 1, NULL,
		"COLUMNS", __PARAM_INT, 1, 1024 + 1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	struct tty_area *tty;
	MALLOC_REQUEST mrq;
	OPENRQ openrq;
	DEVICE_REQUEST drq;
	int r;
	char *xname, *xx;
	mrq.size = sizeof(struct tty_area);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (mrq.status < 0) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT ALLOCATE TTY STRUCTURE: %s", strerror(-mrq.status));
		err0:
		return r;
	}
	tty = mrq.ptr;
	memset(tty, 0, sizeof(struct tty_area));
	tty->tty.buf_size = OUT_BUF;
	params[0].__p = &kbd;
	params[1].__p = &scr;
	params[2].__p = &kbdscr;
	params[3].__p = &name;
	params[4].__p = &tty->tty.buf_size;
	params[5].__p = &lines;
	params[6].__p = &columns;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: SYNTAX ERROR");
		r = -EBADSYN;
		err1:
		KERNEL$UNIVERSAL_FREE(tty);
		goto err0;
	}
	if (kbdscr) {
		if (kbd || scr) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: CAN'T SET BOTH /KBDSCR AND /KBD OR /SCR");
			r = -EBADSYN;
			goto err1;
		}
		kbd = scr = kbdscr;
	}
	if (!kbd || !scr || !name) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: /KBD, /SCR AND /TTY MUST BE SET");
		r = -EBADSYN;
		goto err1;
	}
	strcpy(tty->name, name);
	if (tty->tty.buf_size < __MAX_STR_LEN || tty->tty.buf_size > __PAGE_CLUSTER_SIZE || (tty->tty.buf_size & (tty->tty.buf_size - 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: BAD BUFFER SIZE (ALLOWED POWER OF 2 FROM: %u TO %u)", __MAX_STR_LEN, (unsigned)__PAGE_CLUSTER_SIZE);
		r = -EINVAL;
		goto err1;
	}
	mrq.size = tty->tty.buf_size;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (mrq.status < 0) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT ALLOCATE TTY BUFFER: %s", strerror(mrq.status));
		r = mrq.status;
		goto err1;
	}
	tty->tty.buf = mrq.ptr;
	tty->tty.writeout = TTY_WRITEOUT;
	tty->tty.dtty = NULL;
	if ((r = TTY$CREATE(&tty->tty))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT CREATE TTY: %s", strerror(-r));
		err2:
		KERNEL$UNIVERSAL_FREE(tty->tty.buf);
		goto err1;
	}
	openrq.flags = O_RDONLY;
	openrq.path = kbd;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT OPEN %s: %s", openrq.path, strerror(-openrq.status));
		r = openrq.status;
		goto err2;
	}
	tty->kbd_h = openrq.status;
	openrq.flags = O_WRONLY;
	openrq.path = scr;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT OPEN %s: %s", openrq.path, strerror(-openrq.status));
		r = openrq.status;
		err3:
		KERNEL$FAST_CLOSE(tty->kbd_h);
		goto err2;
	}
	tty->scr_h = openrq.status;
	if (lines == -1 || columns == -1) {
		io.h = tty->scr_h;
		io.ioctl = IOCTL_TTY_GET_WINSIZE;
		io.param = 0;
		io.v.ptr = (unsigned long)&ws;
		io.v.len = sizeof ws;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
		if (io.status >= 0) {
			if (lines == -1) lines = ws.ws_row, ypixels = ws.ws_ypixel;
			if (columns == -1) columns = ws.ws_col, xpixels = ws.ws_xpixel;
		}
	}
	if (columns != -1) tty->tty.xsize = columns;
	if (lines != -1) tty->tty.ysize = lines;
	if (columns != -1 && lines != -1) tty->tty.bits |= TTY_LOCK_WINDOW_SIZE;
	if (xpixels) tty->tty.xpixels = xpixels;
	if (ypixels) tty->tty.ypixels = ypixels;

	io.h = tty->scr_h;
	io.ioctl = IOCTL_CONSOLE_VC;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
	if (io.status < 0) goto skip_dctl;
	tty->tty.vc = io.status;

	xname = alloca(strlen(scr) + 1);
	strcpy(xname, scr);
	xx = strchr(xname, ':');
	if (__unlikely(!xx)) goto skip_dctl;
	*xx = 0;
	if ((r = KERNEL$DCALL(xname, "CONSOLE", CONSOLE_CMD_GET + tty->tty.vc, &tty->tty.vmode))) goto skip_dctl;

	skip_dctl:

	io.ioctl = IOCTL_CONSOLE_VC;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
	if (io.status < 0) goto skip_kdctl;
	tty->tty.kc = io.status;

	xname = alloca(strlen(kbd) + 1);
	strcpy(xname, kbd);
	xx = strchr(xname, ':');
	if (__unlikely(!xx)) goto skip_kdctl;
	*xx = 0;
	if ((r = KERNEL$DCALL(xname, "KEYBOARD", KEYBOARD_CMD_GET + tty->tty.vc, &tty->tty.kmode))) goto skip_kdctl;

	skip_kdctl:

	drq.init_root_handle = TTY$INIT_ROOT;
	drq.dev_ptr = &tty->tty;
	drq.name = tty->name;
	drq.driver_name = "CTTY.SYS";
	drq.flags = 0;
	drq.dcall = NULL;
	drq.dcall_type = NULL;
	drq.dctl = NULL;
	drq.unload = unload;
	SYNC_IO_CANCELABLE(&drq, KERNEL$REGISTER_DEVICE);
	if (drq.status < 0) {
		if (drq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT REGISTER DEVICE %s: %s", tty->name, strerror(-drq.status));
		r = drq.status;
		if (tty->tty.vmode) tty->tty.vmode(VIDEOMODE_RELEASE, tty->tty.vc);
		if (tty->tty.kmode) tty->tty.kmode(KBD_RELEASE, tty->tty.kc);
		KERNEL$FAST_CLOSE(tty->scr_h);
		goto err3;
	}

	tty->lnte = drq.lnte;

	tty->scr_rq.fn = &TTY_OUT;
	tty->scr_rq.h = tty->scr_h;
	tty->scr_rq.v.vspace = &KERNEL$VIRTUAL;

	tty->kbd_rq.fn = &TTY_IN;
	tty->kbd_rq.h = tty->kbd_h;
	tty->kbd_rq.v.ptr = (unsigned long)tty->in_buf;
	tty->kbd_rq.v.vspace = &KERNEL$VIRTUAL;
	tty->kbd_rq.v.len = IN_BUF;
	tty->kbd_rq.progress = 0;
	tty->state |= TTY_KBD_PENDING;
	CALL_IORQ(&tty->kbd_rq, KERNEL$READ);

	strlcpy(KERNEL$ERROR_MSG(), tty->name, __MAX_STR_LEN);
	tty->dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static void TTY_WRITEOUT(TTY *ttyp)
{
	struct tty_area *tty = GET_STRUCT(ttyp, struct tty_area, tty);
	tty->scr_rq.v.ptr = (unsigned long)ttyp->buffer_to_write;
	tty->scr_rq.v.len = (unsigned long)ttyp->buffer_to_write_len;
	if (__unlikely(!ttyp->buffer_to_write_len))
		KERNEL$SUICIDE("TTY_WRITEOUT: ZERO BYTES");
	tty->scr_rq.progress = 0;
	if (__unlikely(tty->state & TTY_SCR_PENDING))
		KERNEL$SUICIDE("TTY_WRITEOUT: CALLED TWICE");
	tty->state |= TTY_SCR_PENDING;
	CALL_IORQ(&tty->scr_rq, KERNEL$WRITE);
	return;
}

DECL_AST(TTY_OUT, SPL_TTY, SIORQ)
{
	struct tty_area *tty = GET_STRUCT(RQ, struct tty_area, scr_rq);
	if (__likely(RQ->status > 0) && __unlikely(tty->scr_rq.v.len)) {
		tty->scr_rq.progress = 0;
		RETURN_IORQ(&tty->scr_rq, KERNEL$WRITE);
	}
	tty->state &= ~TTY_SCR_PENDING;
	TTY$WRITE_DONE(&tty->tty);
	RETURN;
}

DECL_AST(TTY_IN, SPL_TTY, SIORQ)
{
	struct tty_area *tty = GET_STRUCT(RQ, struct tty_area, kbd_rq);
	if (RQ->status < 0) {
		/*__critical_printf("CTTY %s: %s\n", tty->name, strerror(-RQ->status));*/
		tty->state &= ~TTY_KBD_PENDING;
		RETURN;
	}
	/*{
		int i;
		__debug_printf("TTY_IN(%p -> %p): ", &tty->kbd_rq, tty->kbd_rq.fn);
		for (i = 0; i < RQ->status; i++) __debug_printf("%c", tty->in_buf[i]);
		__debug_printf("\n");
	}*/
	TTY$IN(&tty->tty, tty->in_buf, RQ->status);
	/*__debug_printf("done: %p->%p\n", &tty->kbd_rq, tty->kbd_rq.fn);*/
	tty->kbd_rq.v.ptr = (unsigned long)tty->in_buf;
	tty->kbd_rq.v.len = IN_BUF;
	tty->kbd_rq.progress = 0;
	RETURN_IORQ(&tty->kbd_rq, KERNEL$READ);
}

static int unload(void *p, void **release, char *argv[])
{
	int r;
	struct tty_area *tty = p;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(tty->lnte, argv))) return r;
	RAISE_SPL(SPL_TTY);
	while (tty->state & TTY_KBD_PENDING) {
		KERNEL$CIO((IORQ *)&tty->kbd_rq);
		KERNEL$SLEEP(1);
	}
	while (tty->state & TTY_SCR_PENDING) {
		KERNEL$CIO((IORQ *)&tty->scr_rq);
		KERNEL$SLEEP(1);
	}
	TTY$DESTROY(&tty->tty);
	LOWER_SPL(SPL_ZERO);
	if (tty->tty.vmode) tty->tty.vmode(VIDEOMODE_RELEASE, tty->tty.vc);
	if (tty->tty.kmode) tty->tty.vmode(KBD_RELEASE, tty->tty.kc);
	*release = tty->dlrq;
	KERNEL$UNIVERSAL_FREE(tty->tty.buf);
	KERNEL$UNIVERSAL_FREE(tty);
	return 0;
}

