/* !!! FIXME: handle lost TDs */

#include <SPAD/DEV_KRNL.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/TIMER.H>
#include <ARCH/BSF.H>
#include <ARCH/MOV.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <SPAD/PCI.H>
#include <VALUES.H>

#include "EHCIBUST.H"

#include <SPAD/USBHC.H>
#include <SPAD/USBHUB.H>

#include "OHCIREG.H"

typedef struct {
	__u8 *mem;
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	unsigned flags;
	int errorlevel;
	XLIST_HEAD bulk;
	XLIST_HEAD ctrl;
	XLIST_HEAD intr_iso;
	XLIST_HEAD active_tds;
	__u32 hc_control;
	TIMER sf_timer;
	XLIST_HEAD all_eds;
	USB_ENDPOINT root_ctrl;
	USB_ENDPOINT root_intr;
	unsigned n_ports;
	USB_REQUEST *rh_intr;
	TIMER rhsc_timer;
	__u32 hcca_dmaaddr;
	vspace_dmaunlock_t *hcca_dmaunlock;
	char dev_name[__MAX_STR_LEN];
	pci_id_t id;
	void *lnte;
	void *dlrq;
} OHCI;

#define FLAG_AMD_BUG		1
#define FLAG_SUPERIO_BUG	2
#define FLAG_SF_ENABLED		4
#define FLAG_DEAD		8

typedef struct __tdpage TDPAGE;
typedef struct __td TD;

struct __tdpage {
	TDPAGE *next;
};

typedef struct {
	HW_ED ed;
	unsigned n_bounce_requests;
	unsigned bounce_request_size;
	unsigned n_tds;
	__u32 desc_dma_addr;
	vspace_dmaunlock_t *desc_dma_unlock;
	TDPAGE *pages;
	LIST_ENTRY all_eds_list;
	u_jiffies_lo_t cleanup_time;
	__u16 cleanup_frame_le;
	unsigned pkt_size;
	LIST_ENTRY list;
	TD *queue;
	TD *queue_end;
	TD *freelist;
	char state;
	char type;
	__u8 cmask;
	unsigned char stride;
	USB_ENDPOINT ep;
} ED;

#define EDS_INACTIVE		1
#define EDS_ACTIVE		2
#define EDS_PAUSING		3
#define EDS_UNLINKING		4
#define EDS_UNLINKING_FREEING	5

#define ED2OHCI(ed)	USB2OHCI((ed)->ep.usb)

#define ED_ADDR(e)	((__32LE2CPU((e)->ed.flags) & ED_FA) >> __BSF_CONST(ED_FA))
#define ED_PIPE(e)	((__32LE2CPU((e)->ed.flags) & ED_EN) >> __BSF_CONST(ED_EN))

struct __td {
	vspace_dmaunlock_t *desc_dma_unlock;
	LIST_ENTRY active_tds_list;
	TD *next;
	USB_REQUEST *brq;
	ED *ed;
	__u32 data_dma_addr_1;
	vspace_dmaunlock_t *data_dma_unlock_1;
	__u32 data_dma_addr_2;
	vspace_dmaunlock_t *data_dma_unlock_2;
	__u32 desc_dma_addr;
	HW_TD_ISO td;
};

static __finline__ __u32 TD_SIZE(TD *td)
{
	__u32 len = __32LE2CPU(td->td.end) - __32LE2CPU(td->td.ptr) + 1;
	if (__unlikely(len > OHCI_PAGE_SIZE)) len = OHCI_PAGE_SIZE + 1 + (__32LE2CPU(td->td.end) & (OHCI_PAGE_SIZE - 1)) - (__32LE2CPU(td->td.ptr) & (OHCI_PAGE_SIZE - 1));
	return len;
}

#define OHCI2USB(o)		((USB *)((char *)(o) + sizeof(OHCI)))
#define USB2OHCI(u)		((OHCI *)((char *)(u) - sizeof(OHCI)))
#define OHCI2HCCA(o)		((HCCA *)((char *)(o) - sizeof(HCCA)))

#define RHSC_INT_TIMER		(JIFFIES_PER_SECOND / 10)
#define ED_CLEANUP_TIMEOUT	(JIFFIES_PER_SECOND / 10)
#define CBSR			3
#define FSPM(fi)		((6 * ((fi) - 210)) / 7)
#define MAX_PERIODIC_BANDWIDTH	(USB_HIGH_SPEED / USB_FPS * 17 / 20)
#define OHCI_MAXPORTS		16

#define N_CLUSTERS		((unsigned)(sizeof(HCCA) + sizeof(OHCI) + USB$SIZEOF_USB + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS)

#define OHCI_READ(o, i)		__32LE2CPU(mmio_inl((o)->mem + (i)))
#define OHCI_WRITE(o, i, v)	mmio_outl((o)->mem + (i), __32CPU2LE(v))

#define OHCI_READ_HUB_A(o)		(__likely(!((o)->flags & FLAG_AMD_BUG)) ? OHCI_READ(o, HcRhDescriptorA) : OHCI_READ_BUG_WA(o, HcRhDescriptorA, 0xfc0fe000))
#define OHCI_WRITE_HUB_A(o, v)		OHCI_WRITE(o, HcRhDescriptorA, v)
#define OHCI_READ_HUB_B(o)		OHCI_READ(o, HcRhDescriptorB)
#define OHCI_WRITE_HUB_B(o, v)		OHCI_WRITE(o, HcRhDescriptorB, v)
#define OHCI_READ_HUB_STATUS(o)		OHCI_READ(o, HcRhStatus)
#define OHCI_WRITE_HUB_STATUS(o, v)	OHCI_WRITE(o, HcRhStatus, v)
#define OHCI_READ_HUB_PORT(o, i)	(__likely(!((o)->flags & FLAG_AMD_BUG)) ? OHCI_READ(o, HcRhPortStatus + ((i) << 2)) : OHCI_READ_BUG_WA(o, HcRhPortStatus + ((i) << 2), 0xffe0fce0))
#define OHCI_WRITE_HUB_PORT(o, i, v)	OHCI_WRITE(o, HcRhPortStatus + ((i) << 2), v)

static __u32 OHCI_READ_BUG_WA(OHCI *o, unsigned reg, __u32 invalid)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	__u32 v;
	again:
	v = OHCI_READ(o, reg);
	if (__unlikely(v & invalid)) {
		if (__unlikely(v == 0xffffffff)) return 0;
		if (__likely(KERNEL$GET_JIFFIES_LO() - j <= KERNEL$JIFFIES_STEP)) goto again;
	}
	return v;
}

static __finline__ void OHCI_ENABLE_SF_IRQ(OHCI *ohci)
{
	OHCI_WRITE(ohci, HcInterruptEnable, HcInterruptEnable_SF);
	if (!(ohci->flags & FLAG_SF_ENABLED)) {
		ohci->flags |= FLAG_SF_ENABLED;
		KERNEL$SET_TIMER(ED_CLEANUP_TIMEOUT, &ohci->sf_timer);
	}
}

static __finline__ void OHCI_DISABLE_SF_IRQ(OHCI *ohci)
{
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_SF);
	if (__likely(ohci->flags & FLAG_SF_ENABLED)) {
		ohci->flags &= ~FLAG_SF_ENABLED;
		KERNEL$DEL_TIMER(&ohci->sf_timer);
	}
}

static __finline__ void OHCI_REFRESH_SF_TIMEOUT(OHCI *ohci)
{
	if (__likely(ohci->flags & FLAG_SF_ENABLED)) {
		KERNEL$DEL_TIMER(&ohci->sf_timer);
		KERNEL$SET_TIMER(ED_CLEANUP_TIMEOUT, &ohci->sf_timer);
	}
}

static USB_ENDPOINT *OHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size);
static void OHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt);
static void OHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt);

static void OHCI_HARD_ERROR(OHCI *ohci);

static __const__ struct pci_id_s ohci_pci[] = {
	{ PCI_VENDOR_ID_AMD, 0x740c, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x10 << 8), 0xFFFFFF00U, "OHCI AMD 756", FLAG_AMD_BUG },
	{ PCI_VENDOR_ID_NS, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x10 << 8), 0xFFFFFF00U, "OHCI NSC 87560.", FLAG_SUPERIO_BUG },
	{ PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x10 << 8), 0xFFFFFF00U, "OHCI", 0 },
	{ 0 },
};

static __const__ HANDLE_OPERATIONS OHCI_HANDLE_OPERATIONS = {
	SPL_X(SPL_USB),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	KERNEL$NO_OPERATION,	/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static __const__ USB_HC_OPERATIONS OHCI_HC_OPERATIONS = {
	0,
	OHCI_ALLOC_ENDPOINT,
	OHCI_DISABLE_ENDPOINT,
	OHCI_FREE_ENDPOINT,
};

#define ERR_NONE	0
#define ERR_CABLE	1
#define ERR_DEV		2
#define ERR_INTERNAL	3

typedef struct {
	int severity;
	int priority;
	int code;
	char *msg;
} ERROR;

static __const__ ERROR ERRORS[16] = {
	ERR_NONE,	-1,	0,		"NO ERROR",
	ERR_CABLE,	2,	-EIO,		"CRC ERROR",
	ERR_CABLE,	2,	-EIO,		"BIT STUFFING ERROR",
	ERR_DEV,	7,	-EILSEQ,	"DATA TOGGLE MISMATCH",
	ERR_DEV,	4,	-EPIPE,		"STALL",
	ERR_CABLE,	3,	-ENOLINK,	"DEVICE NOT RESPONDING",
	ERR_CABLE,	2,	-EIO,		"PID CHECK FAILURE",
	ERR_DEV,	8,	-EBADMSG,	"UNEXPECTED PID",
	ERR_DEV,	6,	-EOVERFLOW,	"DATA OVERRUN",
	ERR_DEV,	5,	-ENODATA,	"SHORT PACKET",
	ERR_INTERNAL,	9,	-EFTYPE,	"UNKNOWN ERROR 10",
	ERR_INTERNAL,	9,	-EFTYPE,	"UNKNOWN ERROR 11",
	ERR_CABLE,	1,	-ENOBUFS,	"BUFFER OVERRUN",
	ERR_CABLE,	1,	-ENOBUFS,	"BUFFER UNDERRUN",
	ERR_INTERNAL,	9,	-EFTYPE,	"UNKNOWN ERROR 14",
	ERR_INTERNAL,	9,	-EFTYPE,	"ERROR CODE NOT WRITTEN",
};

static int ERROR_PRI(int code)
{
	unsigned i;
	if (__likely(code >= 0)) return -1;
	for (i = 0; i < sizeof(ERRORS) / sizeof(*ERRORS); i++)
		if (ERRORS[i].code == code) return ERRORS[i].priority;
	return 0;
}

static void OHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq);

static void OHCI_INIT_ROOT(HANDLE *h, void *data)
{
	OHCI *ohci = USB2OHCI(data);
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = ohci;
	h->op = &OHCI_HANDLE_OPERATIONS;
}

static int OHCI_UNLOAD(void *p, void **release, char *argv[]);

static int OHCI_RESET(OHCI *ohci, int init)
{
	__u32 ctrl, interval, rha;
	__u32 x_fi, x_hps;
	unsigned timeout;
	int initreset_bug;
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);

	interval = OHCI_READ(ohci, HcFmInterval) & HcFmInterval_FI;
	interval |= (FSPM(interval) << __BSF_CONST(HcFmInterval_FSMPS)) & HcFmInterval_FSMPS;

	ctrl = OHCI_READ(ohci, HcControl);

	if (!init) goto do_reset;
	switch (ctrl & HcControl_HCFS) {
		case HcControl_HCFS_RESET:
			do_reset:
			ctrl &= HcControl_RWC;
			ctrl |= HcControl_HCFS_RESET;
			OHCI_WRITE(ohci, HcControl, ctrl);
			if (init) KERNEL$SLEEP(50 * JIFFIES_PER_SECOND / 1000 + 1);
			else KERNEL$UDELAY(50000);
			break;
		case HcControl_HCFS_RESUME:
		case HcControl_HCFS_SUSPEND:
			ctrl &= HcControl_RWC;
			ctrl |= HcControl_HCFS_RESUME;
			OHCI_WRITE(ohci, HcControl, ctrl);
			if (init) KERNEL$SLEEP(10 * JIFFIES_PER_SECOND / 1000 + 1);
			else KERNEL$UDELAY(10000);
			break;
	}

	RAISE_SPL(SPL_ALMOST_TOP);
	initreset_bug = 0;
	reset_retry:
	OHCI_WRITE(ohci, HcCommandStatus, HcCommandStatus_HCR);
	timeout = 30;
	while (OHCI_READ(ohci, HcCommandStatus) & HcCommandStatus_HCR) {
		if (!--timeout) {
			LOWER_SPL(SPL_USB);
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ohci->dev_name, "RESET FAILED");
			return -EIO;
		}
		KERNEL$UDELAY(1);
	}
	if (initreset_bug) {
		OHCI_WRITE(ohci, HcControl, ctrl);
		KERNEL$UDELAY(10);	/* really needed :-/ */
	}
	memset(OHCI2HCCA(ohci), 0, sizeof(HCCA));
	OHCI_WRITE(ohci, HcFmInterval, interval);
	OHCI_WRITE(ohci, HcPeriodicCurrentED, 0);
	OHCI_WRITE(ohci, HcControlCurrentED, 0);
	OHCI_WRITE(ohci, HcControlHeadED, 0);
	OHCI_WRITE(ohci, HcBulkCurrentED, 0);
	OHCI_WRITE(ohci, HcBulkHeadED, 0);
	OHCI_WRITE(ohci, HcDoneHead, 0);
	OHCI_WRITE(ohci, HcHCCA, ohci->hcca_dmaaddr);
	OHCI_WRITE(ohci, HcPeriodicStart, (interval & HcFmInterval_FI) * 9 / 10);
	x_fi = OHCI_READ(ohci, HcFmInterval);
	x_hps = OHCI_READ(ohci, HcPeriodicStart);
	if (!(x_fi & HcFmInterval_FSMPS) || !x_hps) {
		if (initreset_bug) {
			LOWER_SPL(SPL_USB);
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ohci->dev_name, "UNABLE TO INITIALIZE REGISTERS: FRAMEINTERVAL %08X, PERIODICSTART %08X", x_fi, x_hps);
			return -EIO;
		}
		initreset_bug = 1;
		goto reset_retry;
	}
	ctrl &= HcControl_RWC;
	ctrl |= HcControl_HCFS_OPERATIONAL | (CBSR << __BSF_CONST(HcControl_CBSR));
	OHCI_WRITE(ohci, HcControl, ctrl);
	ctrl |= HcControl_BLE | HcControl_CLE | HcControl_IE;
	OHCI_WRITE(ohci, HcControl, ctrl);
	ohci->hc_control = ctrl;
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);

	rha = OHCI_READ_HUB_A(ohci);
	rha &= ~(HcRhDescriptorA_PSM | HcRhDescriptorA_OCPM);
	if (ohci->flags & FLAG_SUPERIO_BUG) {
		rha |= HcRhDescriptorA_NOCP;
		rha &= ~(HcRhDescriptorA_NPS | HcRhDescriptorA_POTPGT);
	} else {
		rha |= HcRhDescriptorA_NPS;
	}
	OHCI_WRITE_HUB_A(ohci, rha);
	OHCI_WRITE_HUB_STATUS(ohci, HcRhStatus_LPSC | HcRhStatus_OCIC | HcRhStatus_CRWE);
	OHCI_WRITE_HUB_B(ohci, rha & HcRhDescriptorA_NPS ? 0 : HcRhDescriptorB_PPCM);
	LOWER_SPL(SPL_USB);
	if (init) {
		ohci->n_ports = (rha & HcRhDescriptorA_NDP) >> __BSF_CONST(HcRhDescriptorA_NDP);
		if (__unlikely(!ohci->n_ports) || __unlikely(ohci->n_ports > OHCI_MAXPORTS)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ohci->dev_name, "HUB HAS BAD NUMBER OF PORTS: %d", ohci->n_ports);
			return -EIO;
		}
	}
	if (init) KERNEL$SLEEP(((rha & HcRhDescriptorA_POTPGT) >> __BSF_CONST(HcRhDescriptorA_POTPGT)) * JIFFIES_PER_SECOND / 500 + 1);
	else KERNEL$UDELAY(((rha & HcRhDescriptorA_POTPGT) >> __BSF_CONST(HcRhDescriptorA_POTPGT)) * 2000);
	return 0;
}

static void OHCI_CONTINUE_RESET(OHCI *ohci)
{
	OHCI_WRITE(ohci, HcInterruptEnable, HcInterruptEnable_SO | HcInterruptEnable_WDH | HcInterruptEnable_UE | HcInterruptEnable_RHSC | HcInterruptEnable_MIE);
}

static int OHCI_ROOT_PREPARE_CTRL(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	OHCI *ohci = GET_STRUCT(endpt, OHCI, root_ctrl);
	HC_CHECK_SPL("OHCI_ROOT_PREPARE_CTRL");
	brq->internal1 = (unsigned long)ohci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("OHCI_ROOT_PREPARE_CTRL: INVALID VSPACE");
	}
	return 0;
}

static void OHCI_ROOT_FREE(USB_REQUEST *rq)
{
	HC_CHECK_SPL("OHCI_ROOT_FREE");
}

static void OHCI_ROOT_POST_CTRL(USB_REQUEST *rq)
{
	USB_CTRL_REQUEST *urq = (USB_CTRL_REQUEST *)rq;
	OHCI *ohci = (OHCI *)urq->internal1;
	HC_CHECK_SPL("OHCI_ROOT_POST_CTRL");
	urq->status = 0;
	switch (__16LE2CPU(urq->setup.request)) {
		case USBHUB_REQ_CLEAR_HUB_FEATURE: {
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_HUB_FEAT_C_OVER_CURRENT:
					OHCI_WRITE_HUB_STATUS(ohci, HcRhStatus_OCIC);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_SET_PORT_FEATURE: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= ohci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_SUSPEND:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PSS);
					break;
				case USB_PORT_FEAT_POWER:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PPS);
					break;
				case USB_PORT_FEAT_RESET:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PRS);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_CLEAR_PORT_FEATURE: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= ohci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_ENABLE:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_CCS);
					break;
				case USB_PORT_FEAT_C_ENABLE:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PESC);
					break;
				case USB_PORT_FEAT_SUSPEND:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_POCI);
					break;
				case USB_PORT_FEAT_C_SUSPEND:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PSSC);
					break;
				case USB_PORT_FEAT_POWER:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_LSDA);
					break;
				case USB_PORT_FEAT_C_CONNECTION:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_CSC);
					break;
				case USB_PORT_FEAT_C_OVER_CURRENT:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_OCIC);
					break;
				case USB_PORT_FEAT_C_RESET:
					OHCI_WRITE_HUB_PORT(ohci, idx, HcRhPortStatus_PRSC);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_GET_HUB_STATUS: {
			__u32 stat;
			if (__unlikely(urq->v.len != sizeof(__u32))) {
				urq->status = -EINVAL;
				break;
			}
			stat = OHCI_READ_HUB_STATUS(ohci);
			stat &= ~(HcRhStatus_DRWE | HcRhStatus_CRWE);
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(stat);
			urq->status = sizeof(__u32);
			break;
		}
		case USBHUB_REQ_GET_PORT_STATUS: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			__u32 stat;
			if (__unlikely(idx >= ohci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			if (__unlikely(urq->v.len != sizeof(__u32))) {
				urq->status = -EINVAL;
				break;
			}
			stat = OHCI_READ_HUB_PORT(ohci, idx);
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(stat);
			urq->status = sizeof(__u32);
			break;
		}
		case USBHUB_REQ_GET_HUB_DESCRIPTOR: {
			USB_DESCRIPTOR_HUB *desc;
			__u32 rha, rhb;
			if (__unlikely(urq->v.len != USB_DESCRIPTOR_HUB_SIZE)) {
				urq->status = -EINVAL;
				break;
			}
			desc = (USB_DESCRIPTOR_HUB *)(unsigned long)urq->v.ptr;
			memset(desc, 0xff, USB_DESCRIPTOR_HUB_SIZE);
			desc->bDescLength = 7 + 2 * ((ohci->n_ports + 1 + 7) / 8);
			desc->bDescriptorType = USB_DT_HUB;
			desc->bNbrPorts = ohci->n_ports;
			rha = OHCI_READ_HUB_A(ohci);
			desc->wHubCharacteristics =
	(rha & HcRhDescriptorA_NPS ? USB_DESCRIPTOR_HUB_CHAR_POWER_NONE : 0) |
	(rha & HcRhDescriptorA_PSM ? USB_DESCRIPTOR_HUB_CHAR_POWER_INDIVIDUAL : 0) |
	(rha & HcRhDescriptorA_NOCP ? USB_DESCRIPTOR_HUB_CHAR_OC_NONE : 0) |
	(rha & HcRhDescriptorA_OCPM ? USB_DESCRIPTOR_HUB_CHAR_OC_INDIVIDUAL : 0);
			desc->wHubCharacteristics2 = 0;
			desc->bPwrOn2PwrGood = (rha & HcRhDescriptorA_POTPGT) >> __BSF_CONST(HcRhDescriptorA_POTPGT);
			desc->bHubContrCurrent = 0;
			rhb = OHCI_READ_HUB_B(ohci);
			desc->DeviceRemovable[0] = rhb & 0xfe;
			if (ohci->n_ports >= 8) desc->DeviceRemovable[1] = rhb >> 8;

			urq->status = desc->bDescLength;
			break;
		}
		default:
		dflt: {
			USB$HC_GENERIC_ROOT_CTRL(OHCI2USB(ohci), urq);
			return;
		}
	}
	CALL_AST(urq);
}

static void OHCI_ROOT_CANCEL_CTRL(USB_REQUEST *rq)
{
	HC_CHECK_SPL("OHCI_ROOT_CANCEL_CTRL");
}

static int OHCI_ROOT_PREPARE_INTR(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	OHCI *ohci = GET_STRUCT(endpt, OHCI, root_intr);
	HC_CHECK_SPL("OHCI_ROOT_PREPARE_INTR");
	brq->internal1 = (unsigned long)ohci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("OHCI_ROOT_PREPARE_INTR: INVALID VSPACE");
	}
	return 0;
}


static void OHCI_ROOT_POST_INTR(USB_REQUEST *urq)
{
	OHCI *ohci = (OHCI *)urq->internal1;
	__u8 *ptr;
	int set;
	__u32 rhs;
	unsigned i;
	HC_CHECK_SPL("OHCI_ROOT_POST_INTR");
	if (__unlikely(ohci->rh_intr != NULL))
		KERNEL$SUICIDE("OHCI_ROOT_POST_INTR: ANOTHER REQUEST ALREADY PENDING");
	if (__unlikely(urq->v.len < ((ohci->n_ports + 1 + 7) >> 3))) {
		urq->status = -EINVAL;
		goto call;
	}
	urq->v.len = ((ohci->n_ports + 1 + 7) >> 3);
	urq->status = urq->v.len;
	ptr = (__u8 *)(unsigned long)urq->v.ptr;
	memset(ptr, 0, urq->v.len);
	set = 0;
	rhs = OHCI_READ_HUB_STATUS(ohci);
	if (__unlikely(rhs & HcRhStatus_OCIC)) ptr[0] |= 1, set = 1;
	for (i = 1; i <= ohci->n_ports; i++) {
		__u32 ps = OHCI_READ_HUB_PORT(ohci, i - 1);
		if (ps & (HcRhPortStatus_CSC | HcRhPortStatus_PESC | HcRhPortStatus_PSSC | HcRhPortStatus_OCIC | HcRhPortStatus_PRSC)) ptr[i / 8] |= 1 << (i & 7), set = 1;
	}
	if (!set) {
		urq->status = RQS_CANCELABLE;
		ohci->rh_intr = urq;
		return;
	}
	call:
	CALL_AST(urq);
	return;
}

static void OHCI_ROOT_CANCEL_INTR(USB_REQUEST *urq)
{
	OHCI *ohci;
	HC_CHECK_SPL("OHCI_ROOT_CANCEL_INTR");
	if (__unlikely(urq->status != RQS_CANCELABLE)) return;
	ohci = (OHCI *)urq->internal1;
	if (__likely(urq == ohci->rh_intr)) {
		ohci->rh_intr = NULL;
		urq->status = -EINTR;
		CALL_AST(urq);
	}
}

static void OHCI_ROOT_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	HC_CHECK_SPL("OHCI_ROOT_CLEAR_STALL");
}

#if __DEBUG >= 1
static __NORET_ATTR__ void OHCI_GET_TD_SUICIDE(void)
{
	KERNEL$SUICIDE("OHCI_GET_TD_SUICIDE: NO FREE TDS");
}
#endif

static __finline__ TD *OHCI_GET_TD(ED *ed)
{
	TD *td = ed->freelist;
#if __DEBUG >= 1
	if (__unlikely(!td)) OHCI_GET_TD_SUICIDE();
#endif
	ed->freelist = td->next;
	ADD_TO_XLIST(&ED2OHCI(ed)->active_tds, &td->active_tds_list);
	return td;
}

/* td must be still accessible after this function before OHCI_GET_TD */
static __finline__ void OHCI_FREE_TD(ED *ed, TD *td)
{
	DEL_FROM_LIST(&td->active_tds_list);
	td->next = ed->freelist;
	ed->freelist = td;
}

static int OHCI_PREPARE_BULK_INTR_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	TD *head;
	TD *td;
	VDMA vdma;
	unsigned len;
	TD **ptr;
	__u32 *da;
	__u32 data_dma_addr = 0;	/* warning, go away */
	vspace_dmaunlock_t *data_dma_unlock;
	HC_CHECK_SPL("OHCI_PREPARE_BULK_INTR_REQUEST");
	__PREFETCHT0(&ed->state);
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) if (__likely(!(ed->pkt_size & (ed->pkt_size - 1))) ? brq->v.len & (ed->pkt_size - 1) : brq->v.len % ed->pkt_size) brq->flags &= ~USBRQ_TERMINATE_SHORT;
	ptr = &head;
	da = (__u32 *)(void *)&KERNEL$LIST_END;
	vdma.len = 0;
	brq->status = brq->v.len;
	brq->internal2 = 0;
	next_td:
	brq->internal2++;
	td = OHCI_GET_TD(ed);
	*ptr = td;
	ptr = &td->next;
	*da = __32CPU2LE(td->desc_dma_addr);
	da = &td->td.next_td;
	td->next = NULL;
	td->brq = brq;
	td->td.flags =
		__32CPU2LE(TD_CC) |
		(brq->flags & USBRQ_ALLOW_SHORT ? __32CPU2LE(TD_R) : __32CPU2LE(0)) |
		(brq->flags & USBRQ_DELAY_INTERRUPT ? __32CPU2LE(TD_DI_MAX) : __32CPU2LE(TD_DI_0));
	td->data_dma_addr_1 = 0;
	td->data_dma_unlock_1 = KERNEL$NULL_VSPACE_DMAUNLOCK;
	td->data_dma_addr_2 = 0;
	td->data_dma_unlock_2 = KERNEL$NULL_VSPACE_DMAUNLOCK;
	if (!vdma.len) {
		if (__unlikely(!brq->v.len)) {
			td->td.ptr = __32CPU2LE(0);
			td->td.end = __32CPU2LE(0);
			goto ret;
		}
		RAISE_SPL(SPL_VSPACE);
		vdma = brq->v.vspace->op->vspace_dmalock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &data_dma_unlock);
		LOWER_SPL(SPL_USB);
		if (__unlikely(!vdma.ptr)) {
			brq->internal1 = (unsigned long)td;
			td->next = head;
			OHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
			return -EVSPACEFAULT;
		}
		data_dma_addr = vdma.ptr;
	}
	td->td.ptr = __32CPU2LE(vdma.ptr);
	len = 2 * OHCI_PAGE_SIZE - (vdma.ptr & (OHCI_PAGE_SIZE - 1));
	if (len > vdma.len) len = vdma.len;
	td->td.end = __32CPU2LE(vdma.ptr + len - 1);
	if (len != brq->v.len) {
		unsigned tdlen, rem;
		brq->v.ptr += len;
		brq->v.len -= len;
		vdma.ptr += len;
		vdma.len -= len;
		tdlen = len;
		if (__unlikely(((vdma.ptr - len) & (OHCI_PAGE_SIZE - 1)) + len <= OHCI_PAGE_SIZE)) {
			if (!vdma.len) {
				td->data_dma_addr_2 = data_dma_addr;
				td->data_dma_unlock_2 = data_dma_unlock;
				RAISE_SPL(SPL_VSPACE);
				vdma = brq->v.vspace->op->vspace_dmalock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &data_dma_unlock);
				LOWER_SPL(SPL_USB);
				if (__unlikely(!vdma.ptr)) {
					brq->internal1 = (unsigned long)td;
					td->next = head;
					OHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
					return -EVSPACEFAULT;
				}
				data_dma_addr = vdma.ptr;
				if (__unlikely(vdma.ptr & (OHCI_PAGE_SIZE - 1))) {
					RAISE_SPL(SPL_VSPACE);
					data_dma_unlock(vdma.ptr);
					LOWER_SPL(SPL_USB);
					data_dma_addr = td->data_dma_addr_2;
					data_dma_unlock = td->data_dma_unlock_2;
					td->data_dma_addr_2 = 0;
					td->data_dma_unlock_2 = KERNEL$NULL_VSPACE_DMAUNLOCK;
					vdma.ptr = __32LE2CPU(td->td.end) + 1;
					vdma.len = 0;
					goto no_2vs;
				}
			}
			len = vdma.len;
			if (len > OHCI_PAGE_SIZE) len = OHCI_PAGE_SIZE;
			td->td.end = __32CPU2LE(vdma.ptr + len - 1);
			if (len == brq->v.len) goto done;
			brq->v.ptr += len;
			brq->v.len -= len;
			vdma.ptr += len;
			vdma.len -= len;
			tdlen += len;
		}
		no_2vs:
		if (__unlikely(rem = __likely(!(ed->pkt_size & (ed->pkt_size - 1))) ? tdlen & (ed->pkt_size - 1) : tdlen % ed->pkt_size)) {
			__u32 st, en;
			if (__unlikely(rem == tdlen)) {
				if (__unlikely(brq->v.vspace == &KERNEL$VIRTUAL))
					KERNEL$SUICIDE("OHCI_PREPARE_BULK_INTR_REQUEST: ALIGNMENT ERROR ON KERNEL$VIRTUAL VSPACE");
				if (__unlikely(brq->v.vspace == &KERNEL$PHYSICAL))
					KERNEL$SUICIDE("OHCI_PREPARE_BULK_INTR_REQUEST: ALIGNMENT ERROR ON KERNEL$PHYSICAL VSPACE");
				td->data_dma_addr_1 = data_dma_addr;
				td->data_dma_unlock_1 = data_dma_unlock;
				brq->internal1 = (unsigned long)td;
				td->next = head;
				OHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
				return -EINVAL;
			}
			st = __32LE2CPU(td->td.ptr);
			en = __32LE2CPU(td->td.end);
			if ((st & ~(OHCI_PAGE_SIZE - 1)) == (en & ~(OHCI_PAGE_SIZE - 1)) || (en & (OHCI_PAGE_SIZE - 1)) >= rem) {
				en -= rem;
				td->td.end = __32CPU2LE(en);
				vpa:
				vdma.ptr -= rem;
				vdma.len += rem;
			} else {
				en -= rem;
				td->td.end = __32CPU2LE((st & ~(OHCI_PAGE_SIZE - 1)) | (en & (OHCI_PAGE_SIZE - 1)));
				if (!td->data_dma_addr_2) goto vpa;
				RAISE_SPL(SPL_VSPACE);
				data_dma_unlock(data_dma_addr);
				LOWER_SPL(SPL_USB);
				data_dma_addr = td->data_dma_addr_2;
				data_dma_unlock = td->data_dma_unlock_2;
				td->data_dma_addr_2 = 0;
				td->data_dma_unlock_2 = KERNEL$NULL_VSPACE_DMAUNLOCK;
				vdma.ptr = en + 1;
				vdma.len = OHCI_PAGE_SIZE - ((en + 1) & (OHCI_PAGE_SIZE - 1));
			}
		}
		next_tdd:
		td->td.flags |= __32CPU2LE(TD_DI_NONE);
		td->td.flags &= ~__32CPU2LE(TD_R);
		if (__unlikely(!vdma.len)) {
			td->data_dma_addr_1 = data_dma_addr;
			td->data_dma_unlock_1 = data_dma_unlock;
		}
		goto next_td;
	}
	done:
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) {
		brq->v.len = 0;
		vdma.len = 0;
		goto next_tdd;
	}
	td->data_dma_addr_1 = data_dma_addr;
	td->data_dma_unlock_1 = data_dma_unlock;
	ret:
	brq->internal1 = (unsigned long)td;
	td->next = head;
	td->td.next_td = __32CPU2LE(head->desc_dma_addr);
	return 0;
}

static void OHCI_ACTIVATE_ED(ED *ed);

static void OHCI_POST_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	ED *ed;
	TD *td, *ltd, *pl;
	HC_CHECK_SPL("OHCI_POST_BULK_INTR_CTRL_REQUEST");
	brq->v.len = (unsigned long)-1;
	ltd = (TD *)brq->internal1;
	td = ltd->next;
	ed = ltd->ed;
	pl = ed->queue_end;
	pl->next = td->next;
	pl->brq = td->brq;
	pl->data_dma_addr_1 = td->data_dma_addr_1;
	pl->data_dma_unlock_1 = td->data_dma_unlock_1;
	pl->data_dma_addr_2 = td->data_dma_addr_2;
	pl->data_dma_unlock_2 = td->data_dma_unlock_2;
	pl->td.flags = td->td.flags;
	pl->td.ptr = td->td.ptr;
	pl->td.next_td = td->td.next_td;
	pl->td.end = td->td.end;
	td->next = NULL;
	ed->queue_end = td;
	__barrier();
	ed->ed.tail = __32CPU2LE(td->desc_dma_addr);
	if (__unlikely(ed->state == EDS_INACTIVE)) {
		OHCI_ACTIVATE_ED(ed);
		return;
	}
	__barrier();
	if (__likely(ed->cmask)) OHCI_WRITE(ED2OHCI(ed), HcCommandStatus, ed->cmask);
}

static void OHCI_ACTIVATE_INTR_ISO_ED(ED *ed);

static void OHCI_ACTIVATE_ED(ED *ed)
{
	OHCI *ohci = ED2OHCI(ed);
	if (__likely(USB_IS_BULK(ed->type))) {
		ed->state = EDS_ACTIVE;
		ed->ed.next_ed = OHCI_READ(ohci, HcBulkHeadED);
		ADD_TO_XLIST(&ohci->bulk, &ed->list);
		__barrier();
		OHCI_WRITE(ohci, HcBulkHeadED, ed->desc_dma_addr);
		OHCI_WRITE(ohci, HcCommandStatus, HcCommandStatus_BLF);
	} else if (__likely(USB_IS_CTRL(ed->type))) {
		ed->state = EDS_ACTIVE;
		ed->ed.next_ed = OHCI_READ(ohci, HcControlHeadED);
		ADD_TO_XLIST(&ohci->ctrl, &ed->list);
		__barrier();
		OHCI_WRITE(ohci, HcControlHeadED, ed->desc_dma_addr);
		OHCI_WRITE(ohci, HcCommandStatus, HcCommandStatus_CLF);
	} else {
		OHCI_ACTIVATE_INTR_ISO_ED(ed);
	}
}

static unsigned ENDPOINT_TIME(ED *ed);
static void REMOVE_ERRORNEOUS_TDS(ED *ed);

static void OHCI_ACTIVATE_INTR_ISO_ED(ED *ed)
{
	OHCI *ohci = ED2OHCI(ed);
	unsigned i, j;
	unsigned mintim, mini;
#if __DEBUG >= 1
	if (__unlikely((unsigned)(ed->stride - 1) >= HccaInterruptTableSize) || __unlikely(ed->stride & (ed->stride - 1)))
		KERNEL$SUICIDE("OHCI_ACTIVATE_INTR_ISO_ED: INVALID STRIDE %u", (unsigned)ed->stride);
#endif
	mintim = MAXINT, mini = 0;
	for (i = 0; i < ed->stride; i++) {
		unsigned maxtim = 0;
		for (j = i; j < HccaInterruptTableSize; j += ed->stride) {
			__u32 eh = __32LE2CPU(OHCI2HCCA(ohci)->HccaInterruptTable[j]);
			unsigned tim = 0;
			while (eh) {
				ED *e = GET_STRUCT(KERNEL$DMA_2_VIRT(eh), ED, ed);
				tim += ENDPOINT_TIME(e);
				eh = __32LE2CPU(e->ed.next_ed);
			}
			if (__likely(tim >= maxtim)) maxtim = tim;
		}
		if (maxtim < mintim) mintim = maxtim, mini = i;
	}
	if (__unlikely(mintim + ENDPOINT_TIME(ed) > MAX_PERIODIC_BANDWIDTH)) {
		TD *td;
		if (ohci->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, ohci->dev_name, "CAN'T ALLOCATE PERIODIC BANDWIDTH");
		for (td = ed->queue; td->next; td = td->next) {
			td->brq->status = -ENOSR;
		}
		REMOVE_ERRORNEOUS_TDS(ed);
		return;
	}
	for (j = mini; j < HccaInterruptTableSize; j += ed->stride) {
		__u32 *ehp_le = &OHCI2HCCA(ohci)->HccaInterruptTable[j];
		while (*ehp_le) {
			ED *e = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*ehp_le)), ED, ed);
			if (e == ed) goto already_have;
			if (e->stride < ed->stride) break;
			ehp_le = &e->ed.next_ed;
		}
		if (__unlikely(j == mini)) ed->ed.next_ed = *ehp_le;
		else if (__unlikely(ed->ed.next_ed != *ehp_le))
			KERNEL$SUICIDE("OHCI_ACTIVATE_INTR_ISO_ED: INTERRUPT TREE CORRUPT; %08X != %08X", __32LE2CPU(ed->ed.next_ed), __32LE2CPU(*ehp_le));
		__barrier();
		*ehp_le = __32CPU2LE(ed->desc_dma_addr);
		already_have:;
	}
	ed->state = EDS_ACTIVE;
	ADD_TO_XLIST(&ohci->intr_iso, &ed->list);
	__barrier();
	if (__unlikely(!(ohci->hc_control & HcControl_PLE)))
		OHCI_WRITE(ohci, HcControl, ohci->hc_control |= HcControl_PLE);
}

static unsigned ENDPOINT_TIME(ED *ed)
{
	int bytes = USB$HC_ENDPOINT_BANDWIDTH(ed->type, ed->ed.flags & __32CPU2LE(ED_S) ? USB_LOW_SPEED : USB_FULL_SPEED, ed->pkt_size);
	if (__unlikely(bytes < 0)) KERNEL$SUICIDE("ENDPOINT_TIME: COULD NOT DETERMINE TIME FOR ALLOCATED ENDPOINT: %d", bytes);
	return bytes;
}

static void OHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	TD *td;
	HC_CHECK_SPL("OHCI_FREE_BULK_INTR_CTRL_REQUEST");
	td = (TD *)brq->internal1;
	do {
		TD *tdc;
		RAISE_SPL(SPL_VSPACE);
		td->data_dma_unlock_1(td->data_dma_addr_1);
		td->data_dma_unlock_2(td->data_dma_addr_2);
		LOWER_SPL(SPL_USB);
		tdc = td->next;
		OHCI_FREE_TD(td->ed, td);
		td = tdc;
		brq->internal2--;
	} while (td != (TD *)brq->internal1);
#if __DEBUG >= 1
	if (__unlikely(brq->internal2 != 0))
		KERNEL$SUICIDE("OHCI_FREE_SUICIDE: NUMBER OF TDS MISCOUNTED, LEAK %ld", brq->internal2);
#endif
}

static void OHCI_POST_DISABLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("OHCI_POST_DISABLED");
	OHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -ECONNRESET;
	CALL_AST(brq);
}

static void OHCI_POST_STALLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("OHCI_POST_STALLED");
	OHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -EPIPE;
	CALL_AST(brq);
}

static void OHCI_CANCEL_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	TD *td;
	ED *ed;
	HC_CHECK_SPL("OHCI_CANCEL_BULK_INTR_CTRL_REQUEST");
	if (__likely(!brq->internal2)) return;
	td = (TD *)brq->internal1;
	if (__likely(brq->status >= 0)) brq->status = -EINTR;
	ed = td->ed;
	if (ed->state == EDS_ACTIVE) {
		ed->ed.flags |= __32CPU2LE(ED_K);
		__barrier();
		ed->state = EDS_PAUSING;
		ed->cleanup_frame_le = OHCI2HCCA(ED2OHCI(ed))->HccaFrameNumber;
		ed->cleanup_time = KERNEL$GET_JIFFIES_LO();
		OHCI_ENABLE_SF_IRQ(ED2OHCI(ed));
	}
}

static void OHCI_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	HC_CHECK_SPL("OHCI_CLEAR_STALL");
	if (__unlikely((ed->ed.head & __32CPU2LE(ED_QH_ADDR)) != ed->ed.tail))
		KERNEL$SUICIDE("OHCI_CLEAR_STALL: CLEAR STALL ON ACTIVE ENDPOINT");
	ed->ed.head &= ~__32CPU2LE(ED_QH_C);
	if (__likely(ed->ep.post == OHCI_POST_STALLED))
		ed->ep.post = OHCI_POST_BULK_INTR_CTRL_REQUEST;
}

static void OHCI_SCHEDULE_ED_UNLINK(ED *ed)
{
	ED *edd;
	XLIST_HEAD *head;
	OHCI *ohci = ED2OHCI(ed);
	if (ed->state == EDS_INACTIVE || ed->state == EDS_UNLINKING || ed->state == EDS_UNLINKING_FREEING) return;
	if (__likely(USB_IS_BULK(ed->type))) {
		head = &ohci->bulk;
		if (__likely(ohci->hc_control & HcControl_BLE))
			OHCI_WRITE(ohci, HcControl, ohci->hc_control &= ~HcControl_BLE);
	} else if (__likely(USB_IS_CTRL(ed->type))) {
		head = &ohci->ctrl;
		if (__likely(ohci->hc_control & HcControl_CLE))
			OHCI_WRITE(ohci, HcControl, ohci->hc_control &= ~HcControl_CLE);
	} else {
		unsigned i;
		ED *e;
		XLIST_FOR_EACH(e, &ohci->intr_iso, ED, list) {
			if (__unlikely(e->ed.next_ed == __32CPU2LE(ed->desc_dma_addr))) e->ed.next_ed = ed->ed.next_ed;
		}
		for (i = 0; i < HccaInterruptTableSize; i++) if (OHCI2HCCA(ohci)->HccaInterruptTable[i] == __32CPU2LE(ed->desc_dma_addr)) OHCI2HCCA(ohci)->HccaInterruptTable[i] = ed->ed.next_ed;
		head = NULL;	/* warning, go away */
	}
	__barrier();
	ed->state = EDS_UNLINKING;
	ed->cleanup_frame_le = OHCI2HCCA(ED2OHCI(ed))->HccaFrameNumber;
	ed->cleanup_time = KERNEL$GET_JIFFIES_LO();
	OHCI_ENABLE_SF_IRQ(ED2OHCI(ed));
	if (__unlikely(USB_IS_INTR(ed->type)) || __unlikely(USB_IS_ISO(ed->type))) return;
	edd = ed;
	while (edd->list.prev != (LIST_ENTRY *)head) {
		edd = LIST_STRUCT(edd->list.prev, ED, list);
		if (__likely(edd->state != EDS_UNLINKING) && __likely(edd->state != EDS_UNLINKING_FREEING)) {
			if (__unlikely(edd->state != EDS_ACTIVE) && __unlikely(edd->state != EDS_PAUSING)) KERNEL$SUICIDE("OHCI_SCHEDULE_ED_UNLINK: INVALID STATE ON QUEUE: %d", (int)edd->state);
			edd->ed.next_ed = ed->ed.next_ed;
			return;
		}
	}
	if (__likely(USB_IS_BULK(ed->type))) {
		OHCI_WRITE(ohci, HcBulkHeadED, ed->ed.next_ed);
	} else /*if (__likely(USB_IS_CTRL(ed->type)))*/ {
		OHCI_WRITE(ohci, HcControlHeadED, ed->ed.next_ed);
	}
}

static __NORET_ATTR__ void OHCI_PREPARE_CTRL_SUICIDE_ERR(int r);
static __NORET_ATTR__ void OHCI_PREPARE_CTRL_SUICIDE_PKT(void *p);

static int OHCI_PREPARE_CTRL_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *rq)
{
	int r;
	USB_CTRL_REQUEST *brq = (USB_CTRL_REQUEST *)rq;
	ED *ed = GET_STRUCT(endpt, ED, ep);
	USB_REQUEST setup;
	TD *setup_td;
	TD *status_td;
	HC_CHECK_SPL("OHCI_PREPARE_CTRL_REQUEST");
	brq->status = 0;
	brq->internal2 = 2;
	if (brq->v.len) {
		TD *td;
		unsigned phase;
		if (brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN)) ed->type = 0;
		r = OHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, (USB_REQUEST *)brq);
		ed->type = USB_EP_CTRL;
		if (__unlikely(r)) return r;
		brq->internal2 += 2;
		td = (TD *)brq->internal1;
		phase = 1;
		do {
			__u32 len;
			td = td->next;
			td->td.flags = td->td.flags | (__32CPU2LE(TD_DI_MAX | TD_T_DATA_SEL) | (phase * TD_T_DATA_BIT) | (brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN) ? __32CPU2LE(TD_DP_IN) : __32CPU2LE(TD_DP_OUT)));
			len = TD_SIZE(td);
			len += ed->pkt_size - 1;
			phase ^= (__likely(!(ed->pkt_size & (ed->pkt_size - 1))) ? len >> __BSF(ed->pkt_size) : len / ed->pkt_size) & 1;
		} while (td != (TD *)brq->internal1);
		brq->v.len = 1;
	}
	setup.flags = USBRQ_DELAY_INTERRUPT;
	setup.v.ptr = (unsigned long)&brq->setup;
	setup.v.len = sizeof brq->setup;
	setup.v.vspace = &KERNEL$VIRTUAL;
	if (__unlikely(r = OHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, &setup)))
		OHCI_PREPARE_CTRL_SUICIDE_ERR(r);
	setup_td = (TD *)setup.internal1;
	if (__unlikely(setup_td->next != setup_td))
		OHCI_PREPARE_CTRL_SUICIDE_PKT(brq);
	setup_td->td.flags = (setup_td->td.flags & ~__32CPU2LE(TD_DP | TD_T)) | __32CPU2LE(TD_DI_MAX | TD_DP_SETUP | TD_T_DATA0);
	setup_td->brq = (USB_REQUEST *)brq;
	status_td = OHCI_GET_TD(ed);
	status_td->next = setup_td;
	status_td->td.next_td = __32CPU2LE(setup_td->desc_dma_addr);
	status_td->brq = (USB_REQUEST *)brq;
	status_td->td.flags = __32CPU2LE(TD_CC | TD_T_DATA1) |
		(brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN) && brq->v.len ? __32CPU2LE(TD_DP_OUT) : __32CPU2LE(TD_DP_IN)) |
		(brq->flags & USBRQ_DELAY_INTERRUPT ? __32CPU2LE(TD_DI_MAX) : __32CPU2LE(TD_DI_0));
	status_td->td.ptr = __32CPU2LE(0);
	status_td->td.end = __32CPU2LE(0);
	status_td->data_dma_unlock_1 = KERNEL$NULL_VSPACE_DMAUNLOCK;
	status_td->data_dma_unlock_2 = KERNEL$NULL_VSPACE_DMAUNLOCK;
	if (brq->v.len) {
		TD *last = (TD *)brq->internal1;
		TD *first = last->next;
		setup_td->next = first->next;
		setup_td->td.next_td = __32CPU2LE(first->desc_dma_addr);
		last->next = status_td;
		last->td.next_td = __32CPU2LE(status_td->desc_dma_addr);
	} else {
		setup_td->next = status_td;
		setup_td->td.next_td = __32CPU2LE(status_td->desc_dma_addr);
	}
	brq->internal1 = (unsigned long)status_td;
	return 0;
}

static __NORET_ATTR__ void OHCI_PREPARE_CTRL_SUICIDE_ERR(int r)
{
	KERNEL$SUICIDE("OHCI_PREPARE_CTRL_SUICIDE_ERR: OHCI_PREPARE_BULK_INTR_REQUEST RETURNED ERROR %s", strerror(-r));
}

static __NORET_ATTR__ void OHCI_PREPARE_CTRL_SUICIDE_PKT(void *p)
{
	KERNEL$SUICIDE("OHCI_PREPARE_CTRL_SUICIDE_PKT: SETUP PACKET IS SPLIT, REQEUST %p", p);
}

static USB_ENDPOINT *OHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size)
{
	OHCI *ohci;
	int error = 0;
	CONTIG_AREA_REQUEST car;
	VDESC v;
	VDMA vdma;
	ED *ed;
	unsigned long p; /* really it is a pointer, but unsigned long allows arithmetics */
	TD **pp;
	TDPAGE **pg;
	unsigned long align, size, tx_size;
	unsigned n_tds;
	TD *td;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("OHCI_ALLOC_ENDPOINT AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!n_rq) || __unlikely(!rq_size))
		KERNEL$SUICIDE("OHCI_ALLOC_ENDPOINT: REQUEST_RESERVATION NOT SET (N_RQ %u, RQ_SIZE %u", n_rq, rq_size);
	ohci = USB2OHCI(usb);
	if (__unlikely(ohci->flags & FLAG_DEAD)) return __ERR_PTR(-EIO);
	if (__unlikely(addr == USB_ROOT_HUB_ADDR)) {
		switch (pipe) {
			case 0:
				if (type != USB_EP_CTRL) goto invl;
				ohci->root_ctrl.usb = usb;
				return &ohci->root_ctrl;
			case 1:
				if (type != USB_EP_INTR_IN) goto invl;
				ohci->root_intr.usb = usb;
				return &ohci->root_intr;
			default:
			invl:
				KERNEL$SUICIDE("OHCI_ALLOC_ENDPOINT: INVALID ROOT HUB ENDPOINT: TYPE %d, PIPE %d", type, pipe);
		}
	}
	align = __likely(!USB_IS_ISO(type)) ? sizeof(HW_TD) : sizeof(HW_TD_ISO);
	size = __likely(!USB_IS_ISO(type)) ? sizeof(TD) - sizeof(HW_TD_ISO) + sizeof(HW_TD) : sizeof(TD);
	tx_size = __PAGE_CLUSTER_SIZE < OHCI_PAGE_SIZE ? __PAGE_CLUSTER_SIZE : OHCI_PAGE_SIZE;
	n_tds = (((rq_size + tx_size) / tx_size) + USB_IS_CTRL(type) * 2) * n_rq + 1;
	car.nclusters = 1;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (__unlikely(car.status < 0)) return __ERR_PTR(car.status);
	ed = car.ptr;
	ADD_TO_XLIST(&ohci->all_eds, &ed->all_eds_list);
	ed->ed.flags = __32CPU2LE(
		(pkt_size << __BSF_CONST(ED_MPS)) |
		(addr << __BSF_CONST(ED_FA)) |
		(pipe << __BSF_CONST(ED_EN)) |
		(USB_IS_CTRL(type) ? ED_D_TD : USB_IS_OUT(type) ? ED_D_OUT : ED_D_IN) |
		(speed == USB_LOW_SPEED ? ED_S : 0) |
		(__unlikely(USB_IS_ISO(type)) ? ED_F : 0));
	ed->pkt_size = pkt_size;
	ed->freelist = NULL;
	ed->n_tds = 0;
	ed->type = type;
	ed->state = EDS_INACTIVE;
	ed->cmask = USB_IS_CTRL(type) ? HcCommandStatus_CLF : USB_IS_BULK(type) ? HcCommandStatus_BLF : 0;
	if (__unlikely(interval > HccaInterruptTableSize)) interval = HccaInterruptTableSize;
	if (__unlikely(interval <= 0)) interval = 1;
	interval = 1 << __BSR(interval);
	ed->stride = interval;
	ed->ep.usb = usb;

	pg = &ed->pages;
	ed->pages = NULL;
	pp = &ed->freelist;
	p = (unsigned long)(ed + 1);

	while (n_tds--) {
		if (__unlikely(!(p & __PAGE_CLUSTER_SIZE_MINUS_1))) goto new_page;
		align_again:
#define off	((unsigned)(unsigned long)&((TD *)NULL)->td)
		p = ((p + off + align - 1) & ~(align - 1)) - off;
#undef off
		if (__unlikely((p & __PAGE_CLUSTER_SIZE_MINUS_1) + size > __PAGE_CLUSTER_SIZE)) {
			TDPAGE *page;
			new_page:
			car.nclusters = 1;
			car.align = 0;
			car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
			SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
			if (__unlikely(car.status < 0)) {
				error = car.status;
				break;
			}
			page = car.ptr;
			page->next = NULL;
			*pg = page;
			pg = &page->next;
			p = (unsigned long)(page + 1);
			goto align_again;
		}
#define tdx	((TD *)p)
		v.ptr = (unsigned long)&tdx->td;
		v.len = USB_IS_ISO(type) ? sizeof(HW_TD_ISO) : sizeof(HW_TD);
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &tdx->desc_dma_unlock);
		LOWER_SPL(SPL_USB);
		tdx->desc_dma_addr = vdma.ptr;
		tdx->ed = ed;
		tdx->next = NULL;
		*pp = tdx;
		pp = &tdx->next;
#undef tdx
		p += size;
		ed->n_tds++;
	}
	v.ptr = (unsigned long)&ed->ed;
	v.len = sizeof(HW_ED);
	v.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &ed->desc_dma_unlock);
	LOWER_SPL(SPL_USB);
	ed->desc_dma_addr = vdma.ptr;
	
	td = OHCI_GET_TD(ed);
	ed->ed.head = __32CPU2LE(td->desc_dma_addr);
	ed->ed.tail = __32CPU2LE(td->desc_dma_addr);
	ed->queue = td;
	ed->queue_end = td;
	td->next = NULL;
	
	switch (type) {
		case USB_EP_CTRL:
			ed->ep.prepare = OHCI_PREPARE_CTRL_REQUEST;
			ed->ep.post = OHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = OHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = OHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = OHCI_CLEAR_STALL;
			break;
		case USB_EP_BULK_IN:
		case USB_EP_BULK_OUT:
		case USB_EP_INTR_IN:
		case USB_EP_INTR_OUT:
			ed->ep.prepare = OHCI_PREPARE_BULK_INTR_REQUEST;
			ed->ep.post = OHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = OHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = OHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = OHCI_CLEAR_STALL;
			break;
		case USB_EP_ISO_IN:
		case USB_EP_ISO_OUT:
			/* unimplemented */
			error = -ENOSYS;
			break;
		default:
			KERNEL$SUICIDE("OHCI_ALLOC_ENDPOINT: TYPE %d", type);
	}

	ed->n_bounce_requests = n_rq + (type == USB_EP_CTRL);
	ed->bounce_request_size = rq_size;
	if (__unlikely(error) || __unlikely(error = KERNEL$RESERVE_BOUNCE(ohci->dev_name, ed->n_bounce_requests, ed->bounce_request_size))) {
		ed->n_bounce_requests = 0;
		ed->bounce_request_size = 0;
	}

	if (__unlikely(error)) {
		OHCI_FREE_ENDPOINT(&ed->ep);
		return __ERR_PTR(error);
	}
	return &ed->ep;
}

static void OHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt)
{
	TD *td;
	USB *usb;
	OHCI *ohci;
	ED *ed;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("OHCI_DISABLE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	usb = endpt->usb;
	ohci = USB2OHCI(usb);
	if (__unlikely(endpt == &ohci->root_ctrl) || __unlikely(endpt == &ohci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);
	ed->ep.post = OHCI_POST_DISABLED;
	for (td = ed->queue; td != ed->queue_end; td = td->next) {
		if (__likely(td->brq->status >= 0)) td->brq->status = -ECONNRESET;
	}
	OHCI_SCHEDULE_ED_UNLINK(ed);
}

static void OHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt)
{
	TDPAGE *page;
	TD *td;
	ED *ed;
	unsigned i;
	USB *usb;
	OHCI *ohci;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("OHCI_FREE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	usb = endpt->usb;
	ohci = USB2OHCI(usb);
	if (__unlikely(endpt == &ohci->root_ctrl) || __unlikely(endpt == &ohci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);
	if (__unlikely(ed->queue != ed->queue_end))
		KERNEL$SUICIDE("OHCI_FREE_ENDPOINT: FREEING %s ENDPOINT %u:%u WHILE SOME REQUESTS ARE PENDING", USB$ENDPOINT_TYPE_STRING(ed->type), ED_ADDR(ed), ED_PIPE(ed));

	if (ed->state != EDS_INACTIVE) {
		OHCI_SCHEDULE_ED_UNLINK(ed);
		ed->state = EDS_UNLINKING_FREEING;
		return;
	}

	RAISE_SPL(SPL_VSPACE);
	ed->desc_dma_unlock(ed->desc_dma_addr);
	LOWER_SPL(SPL_USB);
	td = ed->queue;
	if (__unlikely(td->next != NULL)) KERNEL$SUICIDE("OHCI_FREE_ENDPOINT: THERE ARE STILL ACTIVE DESCRIPTORS");
	if (__unlikely(td != ed->queue_end)) KERNEL$SUICIDE("OHCI_FREE_ENDPOINT: TD QUEUE CORRUPTED");
	OHCI_FREE_TD(ed, td);
	for (i = 0; i < ed->n_tds; i++) {
		td = OHCI_GET_TD(ed);
		DEL_FROM_LIST(&td->active_tds_list);
		RAISE_SPL(SPL_VSPACE);
		td->desc_dma_unlock(td->desc_dma_addr);
		LOWER_SPL(SPL_USB);
	}
	if (__unlikely(ed->freelist != NULL)) KERNEL$SUICIDE("OHCI_FREE_ENDPOINT: N_TDS MISCOUNTED");

	page = ed->pages;
	while (__unlikely(page != NULL)) {
		TDPAGE *np = page->next;
		KERNEL$VM_RELEASE_CONTIG_AREA(page, 1);
		page = np;
	}
	DEL_FROM_LIST(&ed->all_eds_list);
	if (__likely(ed->bounce_request_size != 0)) {
		KERNEL$UNRESERVE_BOUNCE(ohci->dev_name, ed->n_bounce_requests, ed->bounce_request_size);
	}
	KERNEL$VM_RELEASE_CONTIG_AREA(ed, 1);
}

static void CANCEL_ED(ED *ed, int error)
{
	ed->ed.head = (ed->ed.head & __32CPU2LE(ED_QH_C | ED_QH_H)) | (ed->ed.tail & __32CPU2LE(ED_QH_ADDR));
	while (ed->queue != ed->queue_end) {
		TD *td = ed->queue;
		USB_REQUEST *brq;
		ed->queue = td->next;
		RAISE_SPL(SPL_VSPACE);
		td->data_dma_unlock_1(td->data_dma_addr_1);
		td->data_dma_unlock_2(td->data_dma_addr_2);
		LOWER_SPL(SPL_USB);
		if (__likely(td->brq->status >= 0)) td->brq->status = error;
		brq = td->brq;
#if __DEBUG >= 1
		if (__unlikely((long)brq->internal2 <= 0))
			KERNEL$SUICIDE("CANCEL_ED: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
		if (!--brq->internal2) CALL_AST(brq);
		OHCI_FREE_TD(ed, td);
	}
}


static void OHCI_SPECIAL_TD(TD *td);
static int OHCI_SPECIAL_IRQ(OHCI *ohci);
static ED *PROCESS_QUEUE_SOF(ED *ed, int *wtd, int *unlinking);
static void OHCI_RHSC_ENABLE(TIMER *t);

#define SIRQ_SF		1
#define SIRQ_RESET	2

DECL_AST(OHCI_IRQ, SPL_USB, AST)
{
	OHCI *ohci = GET_STRUCT(RQ, OHCI, irq_ast);
	__u32 dh;
	/*{
		static u_jiffies_lo_t j = 0;
		if (j && KERNEL$GET_JIFFIES_LO() - j > JIFFIES_PER_SECOND * 20) {
			j = KERNEL$GET_JIFFIES_LO();
			__debug_printf("resetting...\n");
			OHCI_HARD_ERROR(ohci);
			goto eoi;
		}
		if (!j) j = KERNEL$GET_JIFFIES_LO();
	}*/
	/*__debug_printf("irq: %08x\n", OHCI_READ(ohci, HcInterruptStatus));*/
	if (__unlikely(!(dh = __32LE2CPU(OHCI2HCCA(ohci)->HccaDoneHead)))) {
		OHCI_SPECIAL_IRQ(ohci);
		goto eoi;
	}
	if (__unlikely(dh & 1)) {
		if (__unlikely(OHCI_SPECIAL_IRQ(ohci) & SIRQ_RESET)) goto eoi;
		dh &= ~1;
	}
	do {
		TD *td = GET_STRUCT(KERNEL$DMA_2_VIRT(dh), TD, td);
		ED *ed = td->ed;
		ed->queue = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(ed->ed.head) & ED_QH_ADDR), TD, td);
		/* !!! FIXME: prefetch next td */
		RAISE_SPL(SPL_VSPACE);
		td->data_dma_unlock_1(td->data_dma_addr_1);
		td->data_dma_unlock_2(td->data_dma_addr_2);
		LOWER_SPL(SPL_USB);
		if (__likely(!(td->td.flags & __32CPU2LE(TD_CC | TD_ISO)))) {
			USB_REQUEST *brq = td->brq;
			if (__unlikely(td->td.ptr) != __32CPU2LE(0)) {
				if (__likely(brq->status > 0)) brq->status -= TD_SIZE(td);
			}
#if __DEBUG >= 1
			if (__unlikely((long)brq->internal2 <= 0))
				KERNEL$SUICIDE("OHCI_IRQ: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
			if (!--brq->internal2) {
				/* !!! FIXME: prefetch data */
				CALL_AST(brq);
			}
		} else {
			OHCI_SPECIAL_TD(td);
		}
		OHCI_FREE_TD(td->ed, td);
		dh = __32LE2CPU(td->td.next_td);
	} while (dh);
	OHCI2HCCA(ohci)->HccaDoneHead = __32CPU2LE(0);
	__barrier();
	OHCI_WRITE(ohci, HcInterruptStatus, HcInterruptStatus_WDH);
	eoi:
	ohci->irq_ctrl.eoi();
	RETURN;
}

static void OHCI_SPECIAL_TD(TD *td)
{
	__const__ ERROR *e;
	ED *ed;
	USB_REQUEST *brq;
	if (__likely(!(td->td.flags & __32CPU2LE(TD_CC)))) {
		/* iso - ok ... unimplemented */
		brq = td->brq;
#if __DEBUG >= 1
		if (__unlikely((long)brq->internal2 <= 0))
			KERNEL$SUICIDE("OHCI_SPECIAL_TD 0: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
		if (!--brq->internal2) CALL_AST(brq);
		return;
	}
	e = ERRORS + ((__32LE2CPU(td->td.flags) >> __BSF_CONST(TD_CC)) & 15);
	if (e->code == -ENODATA && !(td->td.flags & __32CPU2LE(TD_ISO | TD_R)) && td->brq->flags & USBRQ_ALLOW_SHORT && __likely(td->ed->ed.head & __32CPU2LE(ED_QH_H)) && __likely(td->td.ptr != __32CPU2LE(0))) {
		TD *tdd;
		ED *ed;
		ed = td->ed;
		brq = td->brq;
		if (__likely(brq->status > 0)) brq->status -= TD_SIZE(td);
#if __DEBUG >= 1
		if (__unlikely((long)brq->internal2 <= 0))
			KERNEL$SUICIDE("OHCI_SPECIAL_TD 1: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
		if (__unlikely(!--brq->internal2)) CALL_AST(brq);
		ed->queue = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(ed->ed.head) & ED_QH_ADDR), TD, td);
		while ((tdd = ed->queue)->next && tdd->brq == brq && (__likely(!(tdd->td.flags & __32CPU2LE(TD_T_DATA1))) || (tdd->next->next && tdd->next->brq == brq))) {
			ed->queue = tdd->next;
			if (__likely(brq->status > 0)) brq->status -= TD_SIZE(tdd);
#if __DEBUG >= 1
			if (__unlikely((long)brq->internal2 <= 0))
				KERNEL$SUICIDE("OHCI_SPECIAL_TD 2: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
			if (!--brq->internal2) CALL_AST(brq);
			RAISE_SPL(SPL_VSPACE);
			tdd->data_dma_unlock_1(tdd->data_dma_addr_1);
			tdd->data_dma_unlock_2(tdd->data_dma_addr_2);
			LOWER_SPL(SPL_USB);
			OHCI_FREE_TD(ed, tdd);
		}
		ed->ed.head = (ed->ed.head & __32CPU2LE(ED_QH_C)) | __32CPU2LE(tdd->desc_dma_addr);
		return;
	}
	brq = td->brq;
	if (__unlikely(e->code != -EPIPE) || __unlikely(!(brq->flags & (USBRQ_QUIET_ERROR | USBRQ_QUIET_STALL))))
		if (__unlikely(ERR_INTERNAL - e->severity <= (brq->flags & USBRQ_QUIET_ERROR ? 0 : ED2OHCI(td->ed)->errorlevel))) {
			KERNEL$SYSLOG(e->severity == ERR_CABLE ? __SYSLOG_NET_WARNING : e->severity == ERR_DEV ? __SYSLOG_NET_ERROR : __SYSLOG_HW_BUG, ED2OHCI(td->ed)->dev_name, "DEVICE %u, %s PIPE %u: %s", (unsigned)((__32LE2CPU(td->ed->ed.flags) & ED_FA) >> __BSF_CONST(ED_FA)), USB$ENDPOINT_TYPE_STRING(td->ed->type), (unsigned)((__32LE2CPU(td->ed->ed.flags) & ED_EN) >> __BSF_CONST(ED_EN)), e->msg);
	}
	if (__likely(ERROR_PRI(e->code) >= ERROR_PRI(brq->status))) brq->status = e->code;
#if __DEBUG >= 1
	if (__unlikely((long)brq->internal2 <= 0))
		KERNEL$SUICIDE("OHCI_SPECIAL_TD 3: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
	if (!--brq->internal2) CALL_AST(brq);
	ed = td->ed;
	if (__unlikely(!(ed->ed.head & __32CPU2LE(ED_QH_H)))) return;
	ed->queue = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(ed->ed.head) & ED_QH_ADDR), TD, td);
	if (USB_IS_CTRL(ed->type)) {
		TD *tdd;
		while ((tdd = ed->queue)->next && tdd->brq == brq) {
			ed->queue = tdd->next;
#if __DEBUG >= 1
			if (__unlikely((long)brq->internal2 <= 0))
				KERNEL$SUICIDE("OHCI_SPECIAL_TD 4: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
			if (!--brq->internal2) CALL_AST(brq);
			RAISE_SPL(SPL_VSPACE);
			tdd->data_dma_unlock_1(tdd->data_dma_addr_1);
			tdd->data_dma_unlock_2(tdd->data_dma_addr_2);
			LOWER_SPL(SPL_USB);
			OHCI_FREE_TD(ed, tdd);
		}
		ed->ed.head = (ed->ed.head & __32CPU2LE(ED_QH_C)) | __32CPU2LE(tdd->desc_dma_addr);
		return;
	}
	CANCEL_ED(ed, -EPIPE);
	ed->ed.head = ed->ed.head & ~__32CPU2LE(ED_QH_H);
	if ((__likely(USB_IS_BULK(ed->type)) || USB_IS_INTR(ed->type)) && __likely(ed->ep.post != OHCI_POST_DISABLED)) ed->ep.post = OHCI_POST_STALLED;
}

static int OHCI_SPECIAL_IRQ(OHCI *ohci)
{
	__u32 mask = OHCI_READ(ohci, HcInterruptStatus);
	int ret = 0;
	if (__unlikely(mask == 0xffffffff)) {
		ohci->flags |= FLAG_DEAD;
		OHCI_HARD_ERROR(ohci);
		return SIRQ_RESET;
	}
	mask &= ~HcInterruptStatus_WDH;
	OHCI_WRITE(ohci, HcInterruptStatus, mask);
	if (__unlikely(mask & HcInterruptStatus_SO)) {
		if (ohci->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, ohci->dev_name, "SCHEDULING OVERRUN");
	}
	if (__unlikely(mask & HcInterruptStatus_UE)) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, ohci->dev_name, "UNRECOVERABLE ERROR");
		need_reset:
		OHCI_HARD_ERROR(ohci);
		return SIRQ_RESET;
	}
	if (__likely(mask & HcInterruptStatus_SF)) {
		ED *ed;
		int wtd = 0, unlinking_c = 0, unlinking_b = 0;
		XLIST_FOR_EACH(ed, &ohci->bulk, ED, list) if (__unlikely(!(ed = PROCESS_QUEUE_SOF(ed, &wtd, &unlinking_b)))) goto need_reset;
		XLIST_FOR_EACH(ed, &ohci->ctrl, ED, list) if (__unlikely(!(ed = PROCESS_QUEUE_SOF(ed, &wtd, &unlinking_c)))) goto need_reset;
		XLIST_FOR_EACH(ed, &ohci->intr_iso, ED, list) if (__unlikely(!(ed = PROCESS_QUEUE_SOF(ed, &wtd, (int *)(void *)&KERNEL$LIST_END)))) goto need_reset;
		if (__likely(!unlinking_c) && !(ohci->hc_control & HcControl_CLE)) {
			OHCI_WRITE(ohci, HcControlCurrentED, 0);
			OHCI_WRITE(ohci, HcControl, ohci->hc_control |= HcControl_CLE);
		}
		if (__likely(!unlinking_b) && !(ohci->hc_control & HcControl_BLE)) {
			OHCI_WRITE(ohci, HcBulkCurrentED, 0);
			OHCI_WRITE(ohci, HcControl, ohci->hc_control |= HcControl_BLE);
		}
		if (XLIST_EMPTY(&ohci->intr_iso) && __unlikely(ohci->hc_control & HcControl_PLE)) {
			OHCI_WRITE(ohci, HcControl, ohci->hc_control &= ~HcControl_PLE);
		}
		if (__likely(!wtd)) OHCI_DISABLE_SF_IRQ(ohci);
		else OHCI_REFRESH_SF_TIMEOUT(ohci);
		OHCI_WRITE(ohci, HcCommandStatus, HcCommandStatus_CLF | HcCommandStatus_BLF);
		ret |= SIRQ_SF;
	}
	if (mask & HcInterruptStatus_RHSC && !ohci->rhsc_timer.fn) {
		USB_REQUEST *urq = ohci->rh_intr;
		if (__likely(urq != NULL)) {
			ohci->rh_intr = NULL;
			OHCI_ROOT_POST_INTR(urq);
		}
		OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_RHSC);
		ohci->rhsc_timer.fn = OHCI_RHSC_ENABLE;
		KERNEL$SET_TIMER(RHSC_INT_TIMER, &ohci->rhsc_timer);
	}
	return ret;
}

static void OHCI_RHSC_ENABLE(TIMER *t)
{
	OHCI *ohci;
	LOWER_SPL(SPL_USB);
	ohci = GET_STRUCT(t, OHCI, rhsc_timer);
	OHCI_WRITE(ohci, HcInterruptEnable, HcInterruptEnable_RHSC);
	ohci->rhsc_timer.fn = NULL;
}

static void REMOVE_ERRORNEOUS_TDS(ED *ed)
{
	TD **tt = &ed->queue;
	__u32 *aa = &ed->ed.head;
	while ((*tt)->next) {
		TD *td = *tt;
		USB_REQUEST *brq = td->brq;
		if (brq->status < 0) {
			*tt = td->next;
			*aa = (*aa & __32CPU2LE(ED_QH_H | ED_QH_C)) | __32CPU2LE((*tt)->desc_dma_addr);
#if __DEBUG >= 1
			if (__unlikely((long)brq->internal2 <= 0))
				KERNEL$SUICIDE("REMOVE_ERRORNEOUS_TDS: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
			if (!--brq->internal2) CALL_AST(brq);
			RAISE_SPL(SPL_VSPACE);
			td->data_dma_unlock_1(td->data_dma_addr_1);
			td->data_dma_unlock_2(td->data_dma_addr_2);
			LOWER_SPL(SPL_USB);
			OHCI_FREE_TD(ed, td);
		} else {
			tt = &td->next;
			aa = &td->td.next_td;
		}
	}
}


static ED *PROCESS_QUEUE_SOF(ED *ed, int *wtd, int *unlinking)
{
	u_jiffies_lo_t jj;
	if (ed->state != EDS_PAUSING && ed->state != EDS_UNLINKING && ed->state != EDS_UNLINKING_FREEING) {
		if (__unlikely(ed->state != EDS_ACTIVE))
			KERNEL$SUICIDE("PROCESS_QUEUE_SOF: INACTIVE QUEUE ON LIST");
		return ed;
	}
	jj = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(ed->cleanup_frame_le == OHCI2HCCA(ED2OHCI(ed))->HccaFrameNumber)) {
		if (__unlikely(jj - ed->cleanup_time > TIMEOUT_JIFFIES(ED_CLEANUP_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ED2OHCI(ed)->dev_name, "FRAME NUMBER DIDN'T ADVANCE TO CLEAN UP ENDPOINTS");
			return NULL;
		}
		if (ed->state != EDS_PAUSING) *unlinking = 1;
		*wtd = 1;
		return ed;
	}
	ed->queue = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(ed->ed.head) & ED_QH_ADDR), TD, td);
	REMOVE_ERRORNEOUS_TDS(ed);
	__barrier();
	ed->ed.flags &= ~__32CPU2LE(ED_K);
	if (__unlikely(ed->state != EDS_PAUSING)) {
		char st;
		ED *ret_ed;
		ret_ed = GET_STRUCT(ed->list.prev, ED, list);
		DEL_FROM_LIST(&ed->list);
		st = ed->state;
		ed->state = EDS_INACTIVE;
		if (st == EDS_UNLINKING_FREEING) {
			OHCI_FREE_ENDPOINT(&ed->ep);
			return ret_ed;
		}
		if (__unlikely((ed->ed.head & __32CPU2LE(ED_QH_ADDR)) != ed->ed.tail)) {
			OHCI_ACTIVATE_ED(ed);
		}
		return ret_ed;
	}
	ed->state = EDS_ACTIVE;
	return ed;
}

static void FORCE_FREE_ENDPOINT(ED *ed)
{
	DEL_FROM_LIST(&ed->list);
	ed->state = EDS_INACTIVE;
	OHCI_FREE_ENDPOINT(&ed->ep);
}

static void OHCI_SF_TIMEOUT(TIMER *t)
{
	OHCI *ohci;
	int sirq;
	LOWER_SPL(SPL_USB);
	ohci = GET_STRUCT(t, OHCI, sf_timer);
	ohci->flags &= ~FLAG_SF_ENABLED;
/* it may happen that someone has masked the interrupt ... if SF bit is set in
   interrupt status, it does not mean controller error */
	sirq = OHCI_SPECIAL_IRQ(ohci);
	if (sirq & (SIRQ_SF | SIRQ_RESET)) return;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ohci->dev_name, "CONTROLLER DIDN'T GENERATE START-OF-FRAME INTERRUPT");
	OHCI_HARD_ERROR(ohci);
}

static void OHCI_HARD_ERROR(OHCI *ohci)
{
	__u32 ctrl;
	TD *td, *tdd;
	ED *ed;
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);
	ctrl = OHCI_READ(ohci, HcControl);
	ctrl &= HcControl_RWC;
	ctrl |= HcControl_HCFS_RESET;
	OHCI_WRITE(ohci, HcControl, ctrl);
	KERNEL$UDELAY(50000);
	XLIST_FOR_EACH(td, &ohci->active_tds, TD, active_tds_list) {
		USB_REQUEST *brq;
		if (td == td->ed->queue_end) continue;
		brq = td->brq;
		if (__unlikely(brq->v.len != (unsigned long)-1)) continue;
		RAISE_SPL(SPL_VSPACE);
		td->data_dma_unlock_1(td->data_dma_addr_1);
		td->data_dma_unlock_2(td->data_dma_addr_2);
		LOWER_SPL(SPL_USB);
		brq->status = -EIO;
#if __DEBUG >= 1
		if (__unlikely((long)brq->internal2 <= 0))
			KERNEL$SUICIDE("OHCI_HARD_ERROR: USE COUNT UNDERFLOW: %ld", brq->internal2);
#endif
		if (!--brq->internal2) CALL_AST(brq);
		tdd = LIST_STRUCT(td->active_tds_list.prev, TD, active_tds_list);
		OHCI_FREE_TD(td->ed, td);
	}
	XLIST_FOR_EACH(ed, &ohci->all_eds, ED, all_eds_list) {
		int st;
		ed->queue = ed->queue_end;
		ed->ed.head = ed->ed.tail;
		st = ed->state;
		if (st != EDS_INACTIVE) DEL_FROM_LIST(&ed->list);
		ed->state = EDS_INACTIVE;
		if (st == EDS_UNLINKING_FREEING) {
			ED *edd = LIST_STRUCT(ed->all_eds_list.prev, ED, all_eds_list);
			OHCI_FREE_ENDPOINT(&ed->ep);
			ed = edd;
			continue;
		}
		OHCI_DISABLE_ENDPOINT(&ed->ep);
	}
	if (ohci->flags & FLAG_DEAD) return;
	if (OHCI_RESET(ohci, 0)) {
		ohci->flags |= FLAG_DEAD;
		return;
	}
	OHCI_CONTINUE_RESET(ohci);
}

static void WAIT_FOR_LISTS(OHCI *ohci)
{
	ED *ed;
	int cnt;
	char *str = "";
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("WAIT_FOR_LISTS AT SPL %08X", KERNEL$SPL);
	cnt = 0;
	while (__unlikely(!XLIST_EMPTY(&ohci->bulk)) || __unlikely(!XLIST_EMPTY(&ohci->ctrl)) || __unlikely(!XLIST_EMPTY(&ohci->intr_iso))) {
		XLIST_FOR_EACH_UNLIKELY(ed, &ohci->bulk, ED, list) if (__unlikely(ed->state != EDS_UNLINKING_FREEING)) sc:KERNEL$SUICIDE("WAIT_FOR_LISTS: LEAKED %s ENDPOINT%s %u:%u (STATE %d)", USB$ENDPOINT_TYPE_STRING(ed->type), str, ED_ADDR(ed), ED_PIPE(ed), ed->state);
		XLIST_FOR_EACH_UNLIKELY(ed, &ohci->ctrl, ED, list) if (__unlikely(ed->state != EDS_UNLINKING_FREEING)) goto sc;
		XLIST_FOR_EACH_UNLIKELY(ed, &ohci->intr_iso, ED, list) if (__unlikely(ed->state != EDS_UNLINKING_FREEING)) goto sc;
		if (__unlikely(++cnt > ED_CLEANUP_TIMEOUT)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ohci->dev_name, "FRAME NUMBER DIDN'T ADVANCE TO CLEAN UP ENDPOINTS");
			while (!XLIST_EMPTY(&ohci->bulk)) FORCE_FREE_ENDPOINT(LIST_STRUCT(ohci->bulk.next, ED, list));
			while (!XLIST_EMPTY(&ohci->ctrl)) FORCE_FREE_ENDPOINT(LIST_STRUCT(ohci->ctrl.next, ED, list));
			while (!XLIST_EMPTY(&ohci->intr_iso)) FORCE_FREE_ENDPOINT(LIST_STRUCT(ohci->intr_iso.next, ED, list));
			OHCI_DISABLE_SF_IRQ(ohci);
			goto ok;
		}
		KERNEL$SLEEP(1);
		OHCI_SPECIAL_IRQ(ohci);	/* in case interrupt was blocked by some other driver */
	}
	ok:
	XLIST_FOR_EACH_UNLIKELY(ed, &ohci->all_eds, ED, all_eds_list) {
		str = " IN ALL_EDS";
		goto sc;
	}
	if (__unlikely(!XLIST_EMPTY(&ohci->active_tds)))
		KERNEL$SUICIDE("WAIT_FOR_LISTS: ACTIVE TRANSMIT DESCRIPTORS LEAKED");
}

int main(int argc, char *argv[])
{
	char **arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char dev_name[__MAX_STR_LEN];
	char *opt, *optend, *str;
	char *e;
	int r, rr, reg = 0;
	unsigned long flags;
	char *chip_name;
	char gstr[__MAX_STR_LEN];
	int irq;
	__u8 *mem;
	OHCI *ohci;
	CONTIG_AREA_REQUEST car;
	DEVICE_REQUEST devrq;
	VDESC desc;
	VDMA vdma;
	__u32 rev, ctrl;
	unsigned timeout;
	int errorlevel = 0;
	char *plug = USB$HC_DEFAULT_PLUG_SCRIPT;
	struct __param_table params[] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		"PLUG", __PARAM_STRING, 0, __MAX_STR_LEN, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &errorlevel;
	params[1].__p = &errorlevel;
	params[2].__p = &plug;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "OHCI: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(ohci_pci, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "OHCI: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	BUST_EHCI();
	_snprintf(dev_name, sizeof(dev_name), "USB$OHCI@" PCI_ID_FORMAT, id);
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	mem = PCI$MAP_MEM_RESOURCE(id, 0, OHCI_REGSPACE);
	if (!mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
		r = __PTR_ERR(mem);
		goto ret1;
	}
	irq = PCI$READ_INTERRUPT_LINE(id);
	car.nclusters = N_CLUSTERS;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		if (car.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-car.status));
		r = car.status;
		goto ret2;
	}
	memset(car.ptr, 0, sizeof(HCCA) + sizeof(OHCI));
	ohci = (OHCI *)((char *)car.ptr + sizeof(HCCA));
	strcpy(ohci->dev_name, dev_name);
	ohci->id = id;
	ohci->mem = mem;
	ohci->flags = flags;
	ohci->errorlevel = errorlevel;

	INIT_XLIST(&ohci->bulk);
	INIT_XLIST(&ohci->ctrl);
	INIT_XLIST(&ohci->intr_iso);
	INIT_XLIST(&ohci->active_tds);
	INIT_XLIST(&ohci->all_eds);
	INIT_TIMER(&ohci->rhsc_timer);
	INIT_TIMER(&ohci->sf_timer);
	ohci->sf_timer.fn = OHCI_SF_TIMEOUT;

	ohci->root_ctrl.prepare = OHCI_ROOT_PREPARE_CTRL;
	ohci->root_ctrl.post = OHCI_ROOT_POST_CTRL;
	ohci->root_ctrl.free = OHCI_ROOT_FREE;
	ohci->root_ctrl.cancel = OHCI_ROOT_CANCEL_CTRL;
	ohci->root_ctrl.clear_stall = OHCI_ROOT_CLEAR_STALL;
	ohci->root_intr.prepare = OHCI_ROOT_PREPARE_INTR;
	ohci->root_intr.post = OHCI_ROOT_POST_INTR;
	ohci->root_intr.free = OHCI_ROOT_FREE;
	ohci->root_intr.cancel = OHCI_ROOT_CANCEL_INTR;
	ohci->root_intr.clear_stall = OHCI_ROOT_CLEAR_STALL;

	desc.ptr = (unsigned long)OHCI2HCCA(ohci);
	desc.len = sizeof(HCCA);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &ohci->hcca_dmaunlock);
	LOWER_SPL(SPL_ZERO);
	ohci->hcca_dmaaddr = vdma.ptr;

	rev = OHCI_READ(ohci, HcRevision);
	if ((rev & HcRevision_REV) != HcRevision_REV_val) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, dev_name, "BAD OHCI VERSION: %08X", (unsigned)rev);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD VERSION", dev_name);
		r = -ENXIO;
		goto ret3;
	}
	if ((ctrl = OHCI_READ(ohci, HcControl)) & HcControl_IR) {
		OHCI_WRITE(ohci, HcInterruptEnable, HcInterruptEnable_OC);
		OHCI_WRITE(ohci, HcCommandStatus, HcCommandStatus_OCR);
		timeout = JIFFIES_PER_SECOND * 5;
		while ((ctrl = OHCI_READ(ohci, HcControl)) & HcControl_IR) {
			if (!--timeout) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "TAKEOVER FAILED");
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INITIALIZE CONTROLLER", dev_name);
				r = -EIO;
				goto ret3;
			}
			KERNEL$SLEEP(1);
		}
		ctrl &= HcControl_RWC;
		ctrl |= HcControl_HCFS_RESET;
		OHCI_WRITE(ohci, HcControl, ctrl);
	}

	RAISE_SPL(SPL_USB);
	if (__unlikely(r = OHCI_RESET(ohci, 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INITIALIZE CONTROLLER", dev_name);
		goto ret4;
	}
	ohci->irq_ast.fn = OHCI_IRQ;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &ohci->irq_ast, &ohci->irq_ctrl, dev_name))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		r = -EINVAL;
		goto ret4;
	}
	OHCI_CONTINUE_RESET(ohci);
	LOWER_SPL(SPL_ZERO);

	if (__unlikely(r = USB$HC_REGISTER(OHCI2USB(ohci), &OHCI_HC_OPERATIONS, ohci->dev_name, errorlevel, plug))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER USB HOST CONTROLLER: %s", dev_name, strerror(-r));
		goto ret5;
	}
	reg = 1;

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, id));
	_printf("%s: MEM %"__64_format"X, IRQ %d\n", dev_name, PCI$READ_MEM_RESOURCE(id, 0, NULL), irq);

	devrq.name = dev_name;
	devrq.driver_name = "OHCI.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = OHCI_INIT_ROOT;
	devrq.dev_ptr = OHCI2USB(ohci);
	devrq.dcall = USB$HC_DCALL;
	devrq.dcall_type = USB$HC_DCALL_TYPE;
	devrq.dctl = USB$HC_DCTL;
	devrq.unload = OHCI_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret5;
	}
	ohci->lnte = devrq.lnte;
	ohci->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret5:
	ohci->irq_ctrl.mask();
	RAISE_SPL(SPL_USB);
	OHCI_DISABLE_SF_IRQ(ohci);
	if (ohci->rhsc_timer.fn) KERNEL$DEL_TIMER(&ohci->rhsc_timer);
	WAIT_FOR_LISTS(ohci);
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);
	KERNEL$RELEASE_IRQ(&ohci->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	if (reg) {
		if ((rr = USB$HC_PREPARE_UNREGISTER(OHCI2USB(ohci))))
			KERNEL$SUICIDE("OHCI: FAILED TO UNREGISTER OHCI CONTROLLER: %s", strerror(-rr));
		USB$HC_UNREGISTER(OHCI2USB(ohci));
	}
	RAISE_SPL(SPL_USB);
	ret4:
	ctrl = OHCI_READ(ohci, HcControl);
	ctrl &= HcControl_RWC;
	ctrl |= HcControl_HCFS_RESET;
	OHCI_WRITE(ohci, HcControl, ctrl);
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);
	LOWER_SPL(SPL_ZERO);
	KERNEL$SLEEP(50 * JIFFIES_PER_SECOND / 1000 + 1);
	ret3:
	RAISE_SPL(SPL_VSPACE);
	ohci->hcca_dmaunlock(ohci->hcca_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	KERNEL$VM_RELEASE_CONTIG_AREA(OHCI2HCCA(ohci), N_CLUSTERS);
	ret2:
	PCI$UNMAP_MEM_RESOURCE(id, mem, OHCI_REGSPACE);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int OHCI_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	OHCI *ohci = USB2OHCI(p);
	__u32 ctrl;
	if (argv[1]) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", ohci->dev_name);
		return -EBADSYN;
	}
	if ((r = USB$HC_PREPARE_UNREGISTER(OHCI2USB(ohci)))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: UNABLE TO REMOVE HOST CONTROLLER: %s", ohci->dev_name, strerror(-r));
		return r;
	}
	KERNEL$UNREGISTER_DEVICE(ohci->lnte, 0);
	ohci->irq_ctrl.mask();
	RAISE_SPL(SPL_USB);
	OHCI_DISABLE_SF_IRQ(ohci);
	if (ohci->rhsc_timer.fn) KERNEL$DEL_TIMER(&ohci->rhsc_timer);
	WAIT_FOR_LISTS(ohci);
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);
	KERNEL$RELEASE_IRQ(&ohci->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	USB$HC_UNREGISTER(OHCI2USB(ohci));
	ctrl = OHCI_READ(ohci, HcControl);
	ctrl &= HcControl_RWC;
	ctrl |= HcControl_HCFS_RESET;
	OHCI_WRITE(ohci, HcControl, ctrl);
	OHCI_WRITE(ohci, HcInterruptDisable, HcInterruptDisable_MIE);
	KERNEL$SLEEP(50 * JIFFIES_PER_SECOND / 1000 + 1);
	RAISE_SPL(SPL_VSPACE);
	ohci->hcca_dmaunlock(ohci->hcca_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	PCI$UNMAP_MEM_RESOURCE(ohci->id, ohci->mem, OHCI_REGSPACE);
	PCI$FREE_DEVICE(ohci->id);
	*release = ohci->dlrq;
	KERNEL$VM_RELEASE_CONTIG_AREA(OHCI2HCCA(ohci), N_CLUSTERS);
	return 0;
}

