#include <SPAD/DEV_KRNL.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/TIMER.H>
#include <ARCH/BSF.H>
#include <ARCH/BITOPS.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <SPAD/PCI.H>
#include <STDLIB.H>
#include <VALUES.H>

#include "EHCIBUST.H"

#include <SPAD/USBHC.H>
#include <SPAD/USBHUB.H>

#include "UHCIREG.H"

#define QHS_INACTIVE		1
#define QHS_ACTIVE		2
#define QHS_UNLINKING		3
#define QHS_UNLINKING_FREEING	4

typedef struct {
	io_t io;
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	unsigned flags;
	char errorlevel;
	__u32 *last_head_link;
	XLIST_HEAD intr_qh;
	TIMER fsbr_timer;
	u_jiffies_lo_t fsbr_time;
	unsigned n_ports;
	TIMER hub_timer;
	XLIST_HEAD unlinking;
	USB_ENDPOINT root_ctrl;
	USB_ENDPOINT root_intr;
	USB_REQUEST *rh_intr;
	XLIST_HEAD endpoints;
	__u32 area_dmaaddr;
	vspace_dmaunlock_t *area_dmaunlock;
	char dev_name[__MAX_STR_LEN];
	pci_id_t id;
	void *lnte;
	void *dlrq;
} UHCI;

#define FLAG_REVERT_OC		1
#define FLAG_DEAD		2

typedef struct __bounce_buffer BOUNCE_BUFFER;
typedef struct __ed_page ED_PAGE;
typedef struct __ed ED;

struct __bounce_buffer {
	BOUNCE_BUFFER *list;
	ED *ed;
	VPHYS vphys1;
	vspace_physunlock_t *physunlock1;
	VPHYS vphys2;
	vspace_physunlock_t *physunlock2;
	__u32 prev_dma_addr;
	vspace_dmaunlock_t *prev_dma_unlock;
	__u32 data_dma_addr;
	vspace_dmaunlock_t *data_dma_unlock;
};

typedef struct {
	HW_TD td;
	USB_REQUEST *brq;
	vspace_dmaunlock_t *data_dma_unlock;
	__u32 data_dma_addr;
	__u32 desc_dma_addr;
} TD;

typedef struct {
	LIST_ENTRY list;
	vspace_dmaunlock_t *desc_dma_unlock;
	__u32 desc_dma_addr;
} TD_DMALOCK;

typedef struct {
	HW_QH qh;
	__u32 last_element_link;
	TD *td_first;
	__u32 *link_last;
	UHCI *uhci;
	__u32 desc_dma_addr;
	char state;
	char type;
	unsigned short stride;
	__u16 cleanup_frame;
	u_jiffies_lo_t unlink_start;
	LIST_ENTRY list;
} QH;

struct __ed_page {
	ED_PAGE *next;
};

struct __ed {
	QH qh_default;
	vspace_dmaunlock_t *desc_dma_unlock;
	unsigned n_bounce_requests;
	unsigned bounce_request_size;
	XLIST_HEAD dmalock_list;
	unsigned n_tds;
	unsigned n_bounces;
	char addr;
	char pipe;
	char type;
	QH *qh;
	unsigned pkt_size;
	__u32 td_flags_1_le;
	__u32 td_flags_2_le;
	__u32 d_le;
	USB_ENDPOINT ep;
	TD *freelist;
	BOUNCE_BUFFER *bounce_freelist;
	ED_PAGE *pages;
	LIST_ENTRY endpoints_list;
};

typedef struct {
	FRAME_LIST fl;
	HW_TD sof_irq;
	HW_QH ctrl_start;
/* VIA must have queue heads aligned to 16 bytes. Intel accepts 8 bytes. */
	char pad1[8];
	HW_QH loop_start;
	char pad2[8];
	QH low_ctrl;
} AREA;

#define UHCI_SOF_IRQ_Q_LE(uhci)		__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, sof_irq) + UHCI_ADDR_Q)
#define UHCI_CTRL_START_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, ctrl_start) + UHCI_ADDR_Q)
#define UHCI_LOOP_START_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, loop_start) + UHCI_ADDR_Q)
#define UHCI_LOW_CTRL_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, low_ctrl.qh) + UHCI_ADDR_Q)

#define UHCI2USB(u)		((USB *)((char *)(u) + sizeof(UHCI)))
#define USB2UHCI(u)		((UHCI *)((char *)(u) - sizeof(UHCI)))
#define UHCI2AREA(u)		((AREA *)((char *)(u) - sizeof(AREA)))

#define UHCI_RESET_STEP		10
#define UHCI_RESET_TIMEOUT	100000
#define UHCI_HUB_CHECK_TIME	(JIFFIES_PER_SECOND / 2)
#define UHCI_HUB_RESET_TIME	((JIFFIES_PER_SECOND + 19) / 20)
#define QH_CLEANUP_TIMEOUT	(JIFFIES_PER_SECOND / 10 + 1)
#define DEFAULT_FSBR_TIME	20	/* ms */
		/* 63 --- bytes per low-speed control frame */
#define MAX_PERIODIC_BANDWIDTH	(USB_HIGH_SPEED / USB_FPS * 19 / 20 - 63 * (USB_HIGH_SPEED / USB_LOW_SPEED))

#define UHCI_READ_8(u, i)		io_inb((u)->io + (i))
#define UHCI_READ_16(u, i)		io_inw((u)->io + (i))
#define UHCI_READ_32(u, i)		io_inl((u)->io + (i))
#define UHCI_WRITE_8(u, i, v)		io_outb((u)->io + (i), (v))
#define UHCI_WRITE_16(u, i, v)		io_outw((u)->io + (i), (v))
#define UHCI_WRITE_32(u, i, v)		io_outl((u)->io + (i), (v))
#define UHCI_READ_PORT(u, p)		UHCI_READ_16(u, UHCI_PORTSC + (unsigned)(p) * UHCI_PORTSC_NEXT)
#define UHCI_WRITE_PORT(u, p, v)	UHCI_WRITE_16(u, UHCI_PORTSC + (unsigned)(p) * UHCI_PORTSC_NEXT, (v))
#define UHCI_READ_FRNUM(u)		(UHCI_READ_16(u, UHCI_FRNUM) & UHCI_FRNUM_MASK)

static __finline__ void UHCI_SET_PORT_BIT(UHCI *uhci, unsigned port, unsigned bit)
{
	__u16 val = UHCI_READ_PORT(uhci, port);
	val &= ~(UHCI_PORTSC_RWC | UHCI_PORTSC_WZ);
	val |= bit;
	UHCI_WRITE_PORT(uhci, port, val);
}

static __finline__ void UHCI_CLEAR_PORT_BIT(UHCI *uhci, unsigned port, unsigned bit)
{
	__u16 val = UHCI_READ_PORT(uhci, port);
	val &= ~(UHCI_PORTSC_RWC | UHCI_PORTSC_WZ);
	val &= ~bit;
	val |= bit & UHCI_PORTSC_RWC;
	UHCI_WRITE_PORT(uhci, port, val);
}

#define N_CLUSTERS	((unsigned)(sizeof(AREA) + sizeof(UHCI) + USB$SIZEOF_USB + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS)

static USB_ENDPOINT *UHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size);
static void UHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt);
static void UHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt);

static __const__ struct pci_id_s uhci_pci[] = {
	{ PCI_VENDOR_ID_VIA, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x00 << 8), 0xFFFFFF00U, "UHCI (VIA)", FLAG_REVERT_OC },
	{ PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x00 << 8), 0xFFFFFF00U, "UHCI (INTEL)", 0 },
	{ 0 },
};

static __const__ HANDLE_OPERATIONS UHCI_HANDLE_OPERATIONS = {
	SPL_X(SPL_USB),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	KERNEL$NO_OPERATION,	/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static __const__ USB_HC_OPERATIONS UHCI_HC_OPERATIONS = {
	0,
	UHCI_ALLOC_ENDPOINT,
	UHCI_DISABLE_ENDPOINT,
	UHCI_FREE_ENDPOINT,
};

static __finline__ void UHCI_ENABLE_SOF_IRQ(UHCI *uhci)
{
	UHCI2AREA(uhci)->ctrl_start.element_link = UHCI_SOF_IRQ_Q_LE(uhci);
}

static __finline__ void UHCI_DISABLE_SOF_IRQ(UHCI *uhci)
{
	UHCI2AREA(uhci)->ctrl_start.element_link = __32CPU2LE(UHCI_ADDR_T);
}

static __finline__ void UHCI_SET_FSBR(UHCI *uhci)
{
	*uhci->last_head_link = UHCI2AREA(uhci)->loop_start.head_link;
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	KERNEL$SET_TIMER(uhci->fsbr_time, &uhci->fsbr_timer);
}

static __finline__ void UHCI_CLEAR_FSBR(UHCI *uhci)
{
	*uhci->last_head_link = __32CPU2LE(UHCI_ADDR_T);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	VOID_LIST_ENTRY(&uhci->fsbr_timer.list);
}

#if __DEBUG >= 1
static __NORET_ATTR__ void UHCI_GET_TD_SUICIDE(void)
{
	KERNEL$SUICIDE("UHCI_GET_TD_SUICIDE: NO FREE TDS");
}
#endif

static __finline__ TD *UHCI_GET_TD(ED *ed)
{
	TD *td = ed->freelist;
#if __DEBUG >= 1
	if (__unlikely(!td)) UHCI_GET_TD_SUICIDE();
#endif
	ed->freelist = (TD *)td->brq;
	return td;
}

static __finline__ void UHCI_FREE_TD(ED *ed, TD *td)
{
	td->brq = (void *)ed->freelist;
	ed->freelist = td;
}

#if __DEBUG >= 1
static __NORET_ATTR__ void UHCI_GET_BB_SUICIDE(void)
{
	KERNEL$SUICIDE("UHCI_GET_BB_SUICIDE: NO FREE BOUNCE BUFFERS");
}
#endif

static __finline__ BOUNCE_BUFFER *UHCI_GET_BB(ED *ed)
{
	BOUNCE_BUFFER *bb = ed->bounce_freelist;
#if __DEBUG >= 1
	if (__unlikely(!bb)) UHCI_GET_BB_SUICIDE();
#endif
	ed->bounce_freelist = bb->list;
	return bb;
}

static __finline__ void UHCI_FREE_BB(ED *ed, BOUNCE_BUFFER *bb)
{
	bb->list = ed->bounce_freelist;
	ed->bounce_freelist = bb;
}

static void UHCI_UNBOUNCE_COMMON(BOUNCE_BUFFER *bb);

static void UHCI_UNBOUNCE_COPY(__u32 dma)
{
	void *p1;
	BOUNCE_BUFFER *bb = (BOUNCE_BUFFER *)KERNEL$DMA_2_VIRT(dma) - 1;
	bb->prev_dma_unlock(bb->prev_dma_addr);
	p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys1.ptr);
	memcpy(p1, bb + 1, bb->vphys1.len);
	KERNEL$UNMAP_PHYSICAL_BANK(p1);
	p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys2.ptr);
	memcpy(p1, (char *)(bb + 1) + bb->vphys1.len, bb->vphys2.len);
	KERNEL$UNMAP_PHYSICAL_BANK(p1);
	UHCI_UNBOUNCE_COMMON(bb);
}

static void UHCI_UNBOUNCE(__u32 dma)
{
	BOUNCE_BUFFER *bb = (BOUNCE_BUFFER *)KERNEL$DMA_2_VIRT(dma) - 1;
	bb->prev_dma_unlock(bb->prev_dma_addr);
	UHCI_UNBOUNCE_COMMON(bb);
}

static void UHCI_UNBOUNCE_COMMON(BOUNCE_BUFFER *bb)
{
	bb->physunlock1(bb->vphys1.ptr);
	bb->physunlock2(bb->vphys2.ptr);
	UHCI_FREE_BB(bb->ed, bb);
}

static void UHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq);
static void UHCI_SET_VERTICAL(USB_REQUEST *brq);

static int UHCI_PREPARE_BULK_INTR_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	TD *td;
	VDMA vdma;
	__u32 data_dma_addr;
	vspace_dmaunlock_t *data_dma_unlock;
	__u32 *next_le = (__u32 *)&brq->internal1;
	int r;
	brq->internal2 = (unsigned long)ed;
	HC_CHECK_SPL("UHCI_PREPARE_BULK_INTR_REQUEST");
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) if (__likely(!(ed->pkt_size & (ed->pkt_size - 1))) ? brq->v.len & (ed->pkt_size - 1) : brq->v.len % ed->pkt_size) brq->flags &= ~USBRQ_TERMINATE_SHORT;

	brq->status = brq->v.len;
	if (__unlikely(!brq->v.len)) goto zero_packet;
	next_run:
	RAISE_SPL(SPL_VSPACE);
	vdma = brq->v.vspace->op->vspace_dmalock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &data_dma_unlock);
	LOWER_SPL(SPL_USB);
	if (__unlikely(!vdma.ptr)) {
		r = -EVSPACEFAULT;
		ret_err:
		*next_le = __32CPU2LE(UHCI_ADDR_T);
		if (*(__u32 *)&brq->internal1 != __32CPU2LE(UHCI_ADDR_T)) {
			brq->internal1 = (unsigned long)GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*(__u32 *)&brq->internal1 & ~UHCI_ADDR_VF)), TD, td);
			UHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
		}
		return r;
	}
	data_dma_addr = vdma.ptr;
	brq->v.ptr += vdma.len;
	brq->v.len -= vdma.len;

	td = NULL;	/* warning, go away */

		/* !!! FIXME: change to > to test bounce */
	while (__likely(vdma.len >= ed->pkt_size)) {
		vdma.len -= ed->pkt_size;
		td = UHCI_GET_TD(ed);
		*next_le = __32CPU2LE(td->desc_dma_addr);
		td->td.ptr = __32CPU2LE(vdma.ptr);
		vdma.ptr += ed->pkt_size;
		td->td.flags1 = ed->td_flags_1_le;
		td->td.flags2 = ed->td_flags_2_le;
		ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
		td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
		td->brq = brq;
		next_le = &td->td.link;
	}
	if (__unlikely(vdma.len)) {
		if (__likely(!brq->v.len)) {
			td = UHCI_GET_TD(ed);
			*next_le = __32CPU2LE(td->desc_dma_addr);
			td->td.ptr = __32CPU2LE(vdma.ptr);
			td->td.flags1 = (ed->td_flags_1_le & ~__32CPU2LE(TD_FLAGS1_ACTLEN)) | __32CPU2LE((vdma.len - 1) << __BSF_CONST(TD_FLAGS1_ACTLEN));
			td->td.flags2 = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_MAXLEN)) | __32CPU2LE((vdma.len - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN));
			ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
			td->brq = brq;
			next_le = &td->td.link;
			goto set_dma_unlock;
		} else {
			unsigned long orig_len;
			BOUNCE_BUFFER *bb = UHCI_GET_BB(ed);
			bb->prev_dma_unlock = data_dma_unlock;
			bb->prev_dma_addr = data_dma_addr;
			brq->v.ptr -= vdma.len;
			brq->v.len += vdma.len;
			orig_len = brq->v.len;
			brq->v.len = vdma.len;
			RAISE_SPL(SPL_VSPACE);
			brq->v.vspace->op->vspace_physlock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &bb->vphys1, &bb->physunlock1);
			LOWER_SPL(SPL_USB);
			if (__unlikely(bb->vphys1.len != brq->v.len)) {
				r = __unlikely(bb->vphys1.len != 0) ? -EINVAL : -EVSPACEFAULT;
				bb_ret:
				brq->v.len = orig_len;
				td->data_dma_unlock = bb->prev_dma_unlock;
				td->data_dma_addr = bb->prev_dma_addr;
				UHCI_FREE_BB(ed, bb);
				goto ret_err;
			}
			brq->v.ptr += brq->v.len;
			orig_len -= brq->v.len;
			brq->v.len = orig_len > ed->pkt_size - brq->v.len ? ed->pkt_size - brq->v.len : orig_len;
			RAISE_SPL(SPL_VSPACE);
			brq->v.vspace->op->vspace_physlock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &bb->vphys2, &bb->physunlock2);
			LOWER_SPL(SPL_USB);
			if (__unlikely(bb->vphys2.len != brq->v.len)) {
				RAISE_SPL(SPL_VSPACE);
				bb->physunlock1(bb->vphys1.ptr);
				LOWER_SPL(SPL_USB);
				r = __unlikely(bb->vphys2.len != 0) ? -EINVAL : -EVSPACEFAULT;
				goto bb_ret;
			}
			brq->v.ptr += brq->v.len;
			orig_len -= brq->v.len;
			brq->v.len = orig_len;
			if (__unlikely(!USB_IS_IN(ed->type))) {
				void *p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys1.ptr);
				memcpy(bb + 1, p1, bb->vphys1.len);
				KERNEL$UNMAP_PHYSICAL_BANK(p1);
				p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys2.ptr);
				memcpy((char *)(bb + 1) + bb->vphys1.len, p1, bb->vphys2.len);
				KERNEL$UNMAP_PHYSICAL_BANK(p1);
			}

			td = UHCI_GET_TD(ed);
			*next_le = __32CPU2LE(td->desc_dma_addr);
			td->td.ptr = __32CPU2LE(bb->data_dma_addr);
			td->td.flags1 = (ed->td_flags_1_le & ~__32CPU2LE(TD_FLAGS1_ACTLEN)) | __32CPU2LE((bb->vphys1.len + bb->vphys2.len - 1) << __BSF_CONST(TD_FLAGS1_ACTLEN));
			td->td.flags2 = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_MAXLEN)) | __32CPU2LE((bb->vphys1.len + bb->vphys2.len - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN));
			ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
			td->brq = brq;
			next_le = &td->td.link;
			td->data_dma_unlock = __likely(USB_IS_IN(ed->type)) ? UHCI_UNBOUNCE_COPY : UHCI_UNBOUNCE;
			td->data_dma_addr = bb->data_dma_addr;
		}
	} else {
		set_dma_unlock:
		td->data_dma_unlock = data_dma_unlock;
		td->data_dma_addr = data_dma_addr;
	}
	if (brq->v.len) goto next_run;
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) {
		zero_packet:
		td = UHCI_GET_TD(ed);
		*next_le = __32CPU2LE(td->desc_dma_addr);
		td->td.ptr = __32CPU2LE(0);
		td->td.flags1 = ed->td_flags_1_le | __32CPU2LE(TD_FLAGS1_ACTLEN);
		td->td.flags2 = ed->td_flags_2_le | __32CPU2LE(TD_FLAGS2_MAXLEN);
		ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
		td->brq = brq;
		next_le = &td->td.link;
		td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
	}
	td->td.link = __32CPU2LE(UHCI_ADDR_T);
	brq->v.len = (unsigned long)td;
	if (__likely(!(brq->flags & USBRQ_DELAY_INTERRUPT)))
		td->td.flags1 |= __32CPU2LE(TD_FLAGS1_IOC);
	brq->internal1 = (unsigned long)GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*(__u32 *)&brq->internal1 & ~UHCI_ADDR_VF)), TD, td);
	if (__unlikely(!ed->qh_default.uhci->fsbr_time)) {
		brq->flags |= USBRQ_NO_FSBR;
	}
	if (__unlikely(brq->flags & USBRQ_NO_FSBR) && __likely(USB_IS_BULK(ed->type))) UHCI_SET_VERTICAL(brq);
	return 0;
}

static void UHCI_CHANGE_TOGGLE(USB_REQUEST *brq);
static void UHCI_ACTIVATE_QH(QH *qh, USB_REQUEST *brq);

static void UHCI_POST_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	ED *ed;
	QH *qh;
	TD *first_td;
	HC_CHECK_SPL("UHCI_POST_BULK_INTR_CTRL_REQUEST");
	first_td = (TD *)brq->internal1;
	ed = (ED *)brq->internal2;
	if (__unlikely((first_td->td.flags2 ^ ed->d_le) & __32CPU2LE(TD_FLAGS2_D)) && !USB_IS_CTRL(ed->type)) {
		UHCI_CHANGE_TOGGLE(brq);
		return;
	}
	ed->d_le = (~((TD *)brq->v.len)->td.flags2) & __32CPU2LE(TD_FLAGS2_D);
	qh = ed->qh;
	__barrier();
	*qh->link_last = __32CPU2LE(first_td->desc_dma_addr);
	if (__unlikely(qh->link_last != &qh->qh.element_link)) {
		__LOCK_CMPXCHG32(&qh->qh.element_link, __32CPU2LE(UHCI_ADDR_T), __32CPU2LE(first_td->desc_dma_addr));
	} else {
		qh->td_first = first_td;
	}
	qh->link_last = &((TD *)brq->v.len)->td.link;
	if (__unlikely(qh->state == QHS_INACTIVE)) UHCI_ACTIVATE_QH(qh, brq);
	else if (__likely(!(brq->flags & USBRQ_NO_FSBR)) && __likely(USB_IS_BULK(qh->type))) UHCI_SET_FSBR(qh->uhci);
}

static UHCI *UHCI_PROCESS_SCHEDULE(UHCI *uhci);
static UHCI *UHCI_SPECIAL_IRQ(UHCI *uhci, __u16 sts);

DECL_AST(UHCI_IRQ, SPL_USB, AST)
{
	UHCI *uhci = GET_STRUCT(RQ, UHCI, irq_ast);
	__u16 sts = UHCI_READ_16(uhci, UHCI_USBSTS);
	/*__debug_printf("irq, sts %04x\n", sts);*/
	if (__likely(!sts)) goto eoi;
	UHCI_WRITE_16(uhci, UHCI_USBSTS, sts);
	if (__unlikely(sts & (UHCI_USBSTS_HC_HALTED | UHCI_USBSTS_HC_PROCESS_ERROR | UHCI_USBSTS_HC_SYSTEM_ERROR | UHCI_USBSTS_RESUME_DETECT))) {
		uhci = UHCI_SPECIAL_IRQ(uhci, sts);
		goto eoi;
	}
	uhci = UHCI_PROCESS_SCHEDULE(uhci);
	eoi:
	uhci->irq_ctrl.eoi();
	RETURN;
}

static void UHCI_ENDPOINT_STALL(QH *qh);
static int UHCI_PROCESS_QUEUE_HEAD_NOINLINE(QH *qh);
static void UHCI_PROCESS_UNLINKING_ENTRIES(UHCI *uhci, int force);

static __finline__ int UHCI_PROCESS_QUEUE_HEAD(QH *qh)
{
	if (__unlikely(qh->td_first != NULL)) return UHCI_PROCESS_QUEUE_HEAD_NOINLINE(qh);
	else return 0;
}

static UHCI *UHCI_PROCESS_SCHEDULE(UHCI *uhci)
{
	__u32 addr_le, end_ctrl_le;
	QH *qh;

	UHCI_DISABLE_SOF_IRQ(uhci);

	XLIST_FOR_EACH(qh, &uhci->intr_qh, QH, list) {
		UHCI_PROCESS_QUEUE_HEAD(qh);
	}

	addr_le = UHCI2AREA(uhci)->ctrl_start.head_link;
	end_ctrl_le = UHCI_LOOP_START_Q_LE(uhci);
	while (__unlikely(addr_le != end_ctrl_le)) {
		qh = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
		addr_le = qh->qh.head_link;
		UHCI_PROCESS_QUEUE_HEAD(qh);
	}
	addr_le = UHCI2AREA(uhci)->loop_start.head_link;
	if (!(addr_le & __32CPU2LE(UHCI_ADDR_T))) {
		int fsbr_advance = 0;
		while (1) {
			int brk;
			qh = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
			brk = uhci->last_head_link == &qh->qh.head_link;
			addr_le = qh->qh.head_link;
			fsbr_advance |= UHCI_PROCESS_QUEUE_HEAD(qh);
			if (__unlikely(brk)) break;
		}
		if (__likely(fsbr_advance)) {
			UHCI_SET_FSBR(uhci);
		}
	}

	if (__unlikely(!XLIST_EMPTY(&uhci->unlinking))) UHCI_PROCESS_UNLINKING_ENTRIES(uhci, 0);

	return uhci;
}

static void UHCI_PROCESS_QUEUE_HEAD_NOINLINE_ERROR(QH *qh);

static int UHCI_PROCESS_QUEUE_HEAD_NOINLINE(QH *qh)
{
	int fsbr_advance = 0;
	TD *td = qh->td_first;
	while (td->desc_dma_addr != (__32LE2CPU(qh->qh.element_link) & ~UHCI_ADDR_VF)) {
		USB_REQUEST *brq;
		__u32 next_addr;
		if (__unlikely(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE | TD_FLAGS1_STATUS_STALLED))) {
			if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE)) goto skip_this_queue;
			UHCI_PROCESS_QUEUE_HEAD_NOINLINE_ERROR(qh);
		}
		if (__unlikely(td->data_dma_unlock != KERNEL$NULL_VSPACE_DMAUNLOCK)) {
			RAISE_SPL(SPL_VSPACE);
			td->data_dma_unlock(td->data_dma_addr);
			LOWER_SPL(SPL_USB);
		}
		qh->last_element_link = 0;
		brq = td->brq;
		if (__likely(!(brq->flags & USBRQ_NO_FSBR))) fsbr_advance = 1;
		next_addr = __32LE2CPU(td->td.link);
		UHCI_FREE_TD((ED *)brq->internal2, td);
		if (__unlikely(next_addr & UHCI_ADDR_T)) {
			qh->td_first = NULL;
			qh->link_last = &qh->qh.element_link;
			brq->internal2 = 0;
			CALL_AST(brq);
			goto skip_this_queue;
		}
		td = GET_STRUCT(KERNEL$DMA_2_VIRT(next_addr & ~UHCI_ADDR_VF), TD, td);
		if (__unlikely(td->brq != brq)) {
			brq->internal2 = 0;
			CALL_AST(brq);
		}
		qh->td_first = td;
	}
	if (__unlikely(!(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE)))) {
		/* Intel PIIX errata: bitstuff error may be set without stall */
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_STALLED | TD_FLAGS1_STATUS_BITSTUFF_ERROR) || ((__32LE2CPU(td->td.flags1) >> __BSF_CONST(TD_FLAGS1_ACTLEN)) & (TD_FLAGS1_ACTLEN >> __BSF_CONST(TD_FLAGS1_ACTLEN))) != ((__32LE2CPU(td->td.flags2) >> __BSF_CONST(TD_FLAGS2_MAXLEN)) & (TD_FLAGS2_MAXLEN >> __BSF_CONST(TD_FLAGS2_MAXLEN))))
			UHCI_ENDPOINT_STALL(qh);
	}
	skip_this_queue:;
	return fsbr_advance;
}

static void UHCI_FSBR_TIMER(TIMER *t)
{
	int settime;
	UHCI *uhci;
	__u32 *addr_le;
	LOWER_SPL(SPL_USB);
	uhci = GET_STRUCT(t, UHCI, fsbr_timer);
	addr_le = &UHCI2AREA(uhci)->loop_start.head_link;
	settime = 0;
	while (addr_le != uhci->last_head_link) {
		__u32 el;
		QH *x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
		if (!x->td_first) goto cont;
		if (__unlikely(x->td_first->brq->flags & USBRQ_NO_FSBR)) goto cont;
		el = x->qh.element_link;
		if (__unlikely(el == x->last_element_link)) {
			TD *y;
			if (__unlikely(el & __32CPU2LE(UHCI_ADDR_T))) goto cont;
			y = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(el) & ~UHCI_ADDR_VF), TD, td);
	/* there's a small-probability race here: if the controller updates
	   element_link at this point, we update IOC of inactive entry.
	   However it doesn't hurt --- fsbr will be enabled again one tick later
	*/
			if (__unlikely(!(y->td.flags1 & __32CPU2LE(TD_FLAGS1_IOC)))) {
				__LOCK_OR32(&y->td.flags1, __32CPU2LE(TD_FLAGS1_IOC));
				settime = 1;
			}
			goto cont;
		}
		x->last_element_link = el;
		goto fsbr_en;
		break;
		cont:
		addr_le = &x->qh.head_link;
	}
	*uhci->last_head_link = __32CPU2LE(UHCI_ADDR_T);
	if (settime) KERNEL$SET_TIMER((JIFFIES_PER_SECOND + 499) / 500, &uhci->fsbr_timer);
	else VOID_LIST_ENTRY(&uhci->fsbr_timer.list);
	return;

	fsbr_en:
	*uhci->last_head_link = UHCI2AREA(uhci)->loop_start.head_link;
	KERNEL$SET_TIMER(uhci->fsbr_time, &uhci->fsbr_timer);
}

static void UHCI_REPORT_ERROR(TD *td, int level, __const__ char *msg, int only_stall);
static void UHCI_START_UNLINK_QH(QH *qh);
static void UHCI_POST_STALLED(USB_REQUEST *brq);

static void UHCI_ENDPOINT_STALL(QH *qh)
{
	TD *td = qh->td_first;
	USB_REQUEST *brq = td->brq;
	int r;
	int zap_one = 0;
	__u32 addr;
		/* Intel PIIX errata: bitstuff error may be set without stall */
	if (!(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_STALLED | TD_FLAGS1_STATUS_BITSTUFF_ERROR))) {
		if (__unlikely(!(brq->flags & USBRQ_ALLOW_SHORT))) {
			UHCI_REPORT_ERROR(td, 1, "SHORT PACKET", 0), r = -ENODATA;
		} else if (__unlikely(brq->status < 0)) {
			r = brq->status;
			zap_one = 1;
		} else if (qh->state == QHS_ACTIVE) {
			goto do_unlink;
		} else {
			TD *tdx;
			unsigned pktlen = ((__32LE2CPU(td->td.flags2) + (1 << __BSF_CONST(TD_FLAGS2_MAXLEN))) & TD_FLAGS2_MAXLEN) >> __BSF_CONST(TD_FLAGS2_MAXLEN);
			unsigned xflen = ((__32LE2CPU(td->td.flags1) + (1 << __BSF_CONST(TD_FLAGS1_ACTLEN))) & TD_FLAGS1_ACTLEN) >> __BSF_CONST(TD_FLAGS1_ACTLEN);
			int diff = pktlen - xflen;
			if (__unlikely(diff < 0)) {
				UHCI_REPORT_ERROR(td, 0, "TRANSFERRED LENGTH LARGER THAN PACKET LENGTH", 0), r = -EFTYPE;
				goto do_it;
			}
			tdx = td;
			while (!((addr = __32LE2CPU(tdx->td.link) & ~UHCI_ADDR_VF) & UHCI_ADDR_T)) {
				tdx = GET_STRUCT(KERNEL$DMA_2_VIRT(addr), TD, td);
				if (tdx->brq != brq) break;
				diff += ((__32LE2CPU(tdx->td.flags2) + (1 << __BSF_CONST(TD_FLAGS2_MAXLEN))) & TD_FLAGS2_MAXLEN) >> __BSF_CONST(TD_FLAGS2_MAXLEN);
			}
			if ((unsigned long)diff > (unsigned long)brq->status) {
				UHCI_REPORT_ERROR(td, 0, "LEFT DATA LARGER THAN WHOLE TRANSFER LENGTH", 0), r = -EFTYPE;
				goto do_it;
			}
			r = brq->status - diff;
			zap_one = 1;
		}
	} else {
		r = 0;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BITSTUFF_ERROR))
			UHCI_REPORT_ERROR(td, 2, "BIT STUFFING ERROR", 0), r = -EIO;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_CRC_TIMEOUT))
			UHCI_REPORT_ERROR(td, 2, "CRC ERROR OR DEVICE NOT RESPONDING", 0), r = -EIO;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BABBLE))
			UHCI_REPORT_ERROR(td, 1, "BABBLE", 0), r = -EPROTO;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BUFFER_ERROR))
			UHCI_REPORT_ERROR(td, 2, (__32LE2CPU(td->td.flags2) & TD_FLAGS2_PID) == TD_FLAGS2_PID_IN ? "BUFFER UNDERRUN" : "BUFFER OVERRUN", 0), r = -ENOBUFS;
		if (!r)
			UHCI_REPORT_ERROR(td, 1, "STALL", 1), r = -EPIPE;
	}
	do_it:
	if (USB_IS_CTRL(qh->type) || USB_IS_INTR(qh->type)) zap_one = 1;
	if (brq->status >= 0) brq->status = r;
	if (!zap_one) {
		while (!((addr = __32LE2CPU(td->td.link) & ~UHCI_ADDR_VF) & UHCI_ADDR_T)) {
			td = GET_STRUCT(KERNEL$DMA_2_VIRT(addr), TD, td);
			if (td->brq->status >= 0) td->brq->status = -EPIPE;
		}
		if (qh->state != QHS_ACTIVE) GET_STRUCT(qh, ED, qh_default)->ep.post = UHCI_POST_STALLED;
	}
	if (__likely(qh->state == QHS_ACTIVE)) do_unlink: UHCI_START_UNLINK_QH(qh);
}

static void UHCI_REPORT_ERROR(TD *td, int level, __const__ char *msg, int only_stall)
{
	USB_REQUEST *brq = td->brq;
	ED *ed;
	UHCI *uhci;
	int errorlevel;
	if (__likely(only_stall) && __likely(brq->flags & (USBRQ_QUIET_STALL | USBRQ_QUIET_ERROR))) return;
	if (brq->status < 0) return;
	ed = (ED *)brq->internal2;
	uhci = ed->qh_default.uhci;
	errorlevel = brq->flags & USBRQ_QUIET_ERROR ? 0 : uhci->errorlevel;
	if (__unlikely(level <= errorlevel)) {
		int lclass = level == 2 ? __SYSLOG_NET_WARNING : level == 1 ? __SYSLOG_NET_ERROR : __SYSLOG_HW_BUG;
		KERNEL$SYSLOG(lclass, uhci->dev_name, "DEVICE %u, %s PIPE %u: %s", (unsigned)(__32LE2CPU(td->td.flags2) & TD_FLAGS2_DEVICE_ADDRESS) >> __BSF_CONST(TD_FLAGS2_DEVICE_ADDRESS), USB$ENDPOINT_TYPE_STRING(ed->type), (unsigned)(__32LE2CPU(td->td.flags2) & TD_FLAGS2_ENDPT) >> __BSF_CONST(TD_FLAGS2_ENDPT), msg);
	}
}

static void UHCI_PROCESS_QUEUE_HEAD_NOINLINE_ERROR(QH *qh)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, qh->uhci->dev_name, "STALLED REQUEST RETIRED: FLAGS %08X, FLAGS2: %08X", __32LE2CPU(qh->td_first->td.flags1), __32LE2CPU(qh->td_first->td.flags2));
	qh->td_first->brq->status = -EFTYPE;
}

static void UHCI_PROCESS_UNLINKING_QUEUE_HEAD(QH *qh);

static void UHCI_PROCESS_UNLINKING_ENTRIES(UHCI *uhci, int force)
{
	QH *qh;
	u_jiffies_lo_t jj = KERNEL$GET_JIFFIES_LO();
	__u16 frame = UHCI_READ_FRNUM(uhci);
	XLIST_FOR_EACH(qh, &uhci->unlinking, QH, list) {
		QH *qhh;
		if (__unlikely(qh->cleanup_frame == frame) && !force) {
			if (__unlikely(jj - qh->unlink_start > TIMEOUT_JIFFIES(QH_CLEANUP_TIMEOUT))) {
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, uhci->dev_name, "FRAME NUMBER DIDN'T ADVANCE TO CLEAN UP ENDPOINTS, FORCING CLEANUP");
				goto force_cleanup;
			}
			UHCI_ENABLE_SOF_IRQ(uhci);
			continue;
		}
		force_cleanup:
		qhh = LIST_STRUCT(qh->list.prev, QH, list);
		DEL_FROM_LIST(&qh->list);
		UHCI_PROCESS_UNLINKING_QUEUE_HEAD(qh);
		qh = qhh;
	}
}

static void UHCI_PROCESS_UNLINKING_QUEUE_HEAD(QH *qh)
{
	__u32 *ptr_le;
	TD *td;
	USB_REQUEST *brq;
	__u32 d_le;
	qh->last_element_link = 0;
	UHCI_PROCESS_QUEUE_HEAD(qh);
	td = qh->td_first;
	if (!td) {
		if (__unlikely(qh->state == QHS_UNLINKING_FREEING)) {
			qh->state = QHS_INACTIVE;
			UHCI_FREE_ENDPOINT(&GET_STRUCT(qh, ED, qh_default)->ep);
			return;
		} else {
			qh->state = QHS_INACTIVE;
			return;
		}
	}
	/* process requests that have been correctly finished while unlinking */
	ptr_le = &qh->qh.element_link;
	brq = td->brq;
	d_le = td->td.flags2 & __32CPU2LE(TD_FLAGS2_D);
	if (__unlikely(!(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE))) && brq->status >= 0) {
		if (((__32LE2CPU(td->td.flags1) >> __BSF_CONST(TD_FLAGS1_ACTLEN)) & (TD_FLAGS1_ACTLEN >> __BSF_CONST(TD_FLAGS1_ACTLEN))) != ((__32LE2CPU(td->td.flags2) >> __BSF_CONST(TD_FLAGS2_MAXLEN)) & (TD_FLAGS2_MAXLEN >> __BSF_CONST(TD_FLAGS2_MAXLEN)))) {
			d_le ^= __32CPU2LE(TD_FLAGS2_D);
			goto post_current_td;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, qh->uhci->dev_name, "INACTIVE NON-ERRORNEOUS TD ON UNLINKED QUEUE: FLAGS1: %08X, FLAGS2: %08X, STATUS: %ld", (unsigned)__32LE2CPU(td->td.flags1), (unsigned)__32LE2CPU(td->td.flags2), brq->status);
		brq->status = -EFTYPE;
	}
	next_td:
	brq = td->brq;
	if (__unlikely(brq->status < 0)) {
		while (td->brq == brq) {
			/* for short control transfers, we need to leave status phase descriptor */
			if (__likely(brq->status >= 0) && __unlikely(USB_IS_CTRL(qh->type))) {
				TD *td_next;
				__u32 le = td->td.link;
				if (le & __32CPU2LE(UHCI_ADDR_T)) goto skip_this_td;
				td_next = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(le) & ~UHCI_ADDR_VF), TD, td);
				if (td_next->brq != brq) goto skip_this_td;
			}
			post_current_td:
			RAISE_SPL(SPL_VSPACE);
			td->data_dma_unlock(td->data_dma_addr);
			LOWER_SPL(SPL_USB);
			*ptr_le = td->td.link;
			UHCI_FREE_TD((ED *)brq->internal2, td);
			if (!(*ptr_le & __32CPU2LE(UHCI_ADDR_T))) {
				td = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*ptr_le) & ~UHCI_ADDR_VF), TD, td);
				if (ptr_le == &qh->qh.element_link) qh->td_first = td;
			} else {
				td = NULL;
				if (ptr_le == &qh->qh.element_link) qh->td_first = NULL, qh->link_last = &qh->qh.element_link;
				break;
			}
		}
		brq->internal2 = 0;
		CALL_AST(brq);
		if (td) goto next_td;
	} else {
		skip_this_td:
		if (!USB_IS_CTRL(qh->type)) {
			td->td.flags2 = (td->td.flags2 & ~__32CPU2LE(TD_FLAGS2_D)) | d_le;
			d_le ^= __32CPU2LE(TD_FLAGS2_D);
		}
		ptr_le = &td->td.link;
		if (!(*ptr_le & __32CPU2LE(UHCI_ADDR_T))) {
			td = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*ptr_le) & ~UHCI_ADDR_VF), TD, td);
			goto next_td;
		}
	}
	if (!USB_IS_CTRL(qh->type)) {
		GET_STRUCT(qh, ED, qh_default)->d_le = d_le;
		GET_STRUCT(qh, ED, qh_default)->td_flags_2_le = (GET_STRUCT(qh, ED, qh_default)->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_D)) | d_le;
	}
	qh->state = QHS_INACTIVE;
	if (qh->td_first) UHCI_ACTIVATE_QH(qh, qh->td_first->brq);
}

static void UHCI_START_UNLINK_QH(QH *qh)
{
	__u32 *addr_le;
	if (__likely(USB_IS_BULK(qh->type))) {
		addr_le = &UHCI2AREA(qh->uhci)->loop_start.head_link;
		if (__unlikely(*addr_le & __32CPU2LE(UHCI_ADDR_T)))
			KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST IS EMPTY");
		while ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) != qh->desc_dma_addr) {
			QH *x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			addr_le = &x->qh.head_link;
			if (__unlikely(*addr_le & __32CPU2LE(UHCI_ADDR_T)))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST TERMINATED AND QH NOT FOUND");
			if (__unlikely(addr_le == qh->uhci->last_head_link))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST LOOPS AND QH NOT FOUNT");
		}
		*addr_le = qh->qh.head_link;
		if (__32LE2CPU(UHCI2AREA(qh->uhci)->loop_start.head_link) == (qh->desc_dma_addr | UHCI_ADDR_Q)) {
			UHCI2AREA(qh->uhci)->loop_start.head_link = __32CPU2LE(UHCI_ADDR_T);
			KERNEL$DEL_TIMER(&qh->uhci->fsbr_timer);
			VOID_LIST_ENTRY(&qh->uhci->fsbr_timer.list);
		}
		if (qh->uhci->last_head_link == &qh->qh.head_link) qh->uhci->last_head_link = addr_le;
		if (!(*qh->uhci->last_head_link & __32CPU2LE(UHCI_ADDR_T))) *qh->uhci->last_head_link = UHCI2AREA(qh->uhci)->loop_start.head_link;
	} else if (USB_IS_CTRL(qh->type) && qh != &UHCI2AREA(qh->uhci)->low_ctrl) {
		addr_le = &UHCI2AREA(qh->uhci)->ctrl_start.head_link;
		while ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) != qh->desc_dma_addr) {
			QH *x;
			if (__unlikely(*addr_le == UHCI_LOOP_START_Q_LE(qh->uhci)))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: QH NOT FOUND IN CTRL LIST");
			x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			addr_le = &x->qh.head_link;
		}
		*addr_le = qh->qh.head_link;
	} else {
		int set = 0;
		unsigned j;
		for (j = 0; j < N_FRAME_LIST; j++) {
			addr_le = &UHCI2AREA(qh->uhci)->fl.addr[j];
			while (*addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
				QH *x;
				if (__unlikely(!(*addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
					TD *is;
					is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le)), TD, td);
					addr_le = &is->td.link;
					continue;
				}
				if ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) == qh->desc_dma_addr) {
					*addr_le = qh->qh.head_link;
					set = 1;
					break;
				}
				x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
				if (j >= x->stride) break;
				addr_le = &x->qh.head_link;
			}
		}
		if (__unlikely(!set))
			KERNEL$SUICIDE("UHCI_START_UNLINK_QH: QH NOT FOUND IN INTERRUPT TREE");
		DEL_FROM_LIST(&qh->list);
	}
	__barrier();
	qh->state = QHS_UNLINKING;
	qh->cleanup_frame = UHCI_READ_FRNUM(qh->uhci);
	qh->unlink_start = KERNEL$GET_JIFFIES_LO();
	ADD_TO_XLIST(&qh->uhci->unlinking, &qh->list);
	UHCI_ENABLE_SOF_IRQ(qh->uhci);
}

static void UHCI_ACTIVATE_INTR(QH *qh);

static void UHCI_ACTIVATE_QH(QH *qh, USB_REQUEST *brq)
{
	if (__likely(USB_IS_BULK(qh->type))) {
		__u32 link = UHCI2AREA(qh->uhci)->loop_start.head_link;
		if (link & __32CPU2LE(UHCI_ADDR_T)) {
			qh->uhci->last_head_link = &qh->qh.head_link;
		}
		qh->qh.head_link = link;
		__barrier();
		UHCI2AREA(qh->uhci)->loop_start.head_link = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		if (__likely(!(brq->flags & USBRQ_NO_FSBR)) || !(*qh->uhci->last_head_link & __32CPU2LE(UHCI_ADDR_T))) {
			UHCI_SET_FSBR(qh->uhci);
		}
		qh->state = QHS_ACTIVE;
	} else if (__likely(USB_IS_CTRL(qh->type)) && qh != &UHCI2AREA(qh->uhci)->low_ctrl) {
		qh->qh.head_link = UHCI2AREA(qh->uhci)->ctrl_start.head_link;
		__barrier();
		UHCI2AREA(qh->uhci)->ctrl_start.head_link = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		qh->state = QHS_ACTIVE;
	} else {
		UHCI_ACTIVATE_INTR(qh);
	}
}

static unsigned QH_TIME(QH *qh);

static void UHCI_ACTIVATE_INTR(QH *qh)
{
	unsigned i, j;
	unsigned mintim, mini;
#if __DEBUG >= 1
	if (__unlikely((unsigned)(qh->stride - 1) >= N_FRAME_LIST) || __unlikely(qh->stride & (qh->stride - 1)))
		KERNEL$SUICIDE("UHCI_ACTIVATE_INTR: INVALID STRIDE %u", (unsigned)qh->stride);
#endif
	mintim = MAXINT, mini = 0;
	/*__debug_printf("activate intr: stride %d\n", qh->stride);*/
	if (__unlikely(qh == &UHCI2AREA(qh->uhci)->low_ctrl)) goto no_bw_check;
	/*__debug_printf("check bw\n");*/
	for (i = 0; i < qh->stride; i++) {
		unsigned maxtim = 0;
		for (j = i; j < N_FRAME_LIST; j += qh->stride) {
			__u32 addr_le = UHCI2AREA(qh->uhci)->fl.addr[j];
			unsigned tim = 0;
			while (addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
				QH *x;
				if (__unlikely(!(addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
					TD *is;
					is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le)), TD, td);
					addr_le = is->td.link;
					continue;
				}
				x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
				if (x != &UHCI2AREA(qh->uhci)->low_ctrl)
					tim += QH_TIME(x);
				addr_le = x->qh.head_link;
			}
			if (__likely(tim >= maxtim)) maxtim = tim;
		}
		if (maxtim < mintim) mintim = maxtim, mini = i;
	}
	if (__unlikely(mintim + QH_TIME(qh) > MAX_PERIODIC_BANDWIDTH)) {
		TD *td;
		if (qh->uhci->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, qh->uhci->dev_name, "CAN'T ALLOCATE PERIODIC BANDWIDTH");
		td = qh->td_first;
		if (__likely(td != NULL)) while (1) {
			td->brq->status = -ENOSR;
			if (td->td.link & __32CPU2LE(UHCI_ADDR_T)) break;
			td = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(td->td.link) & ~UHCI_ADDR_VF), TD, td);
		}
		UHCI_PROCESS_UNLINKING_QUEUE_HEAD(qh);
		return;
	}
	no_bw_check:
	for (j = mini; j < N_FRAME_LIST; j += qh->stride) {
		__u32 *addr_le = &UHCI2AREA(qh->uhci)->fl.addr[j];
		while (*addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
			QH *x;
			if (__unlikely(!(*addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
				TD *is;
				is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le)), TD, td);
				addr_le = &is->td.link;
				continue;
			}
			x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			if (x == qh) goto already_have;
			if (x->stride < qh->stride) break;
			addr_le = &x->qh.head_link;
		}
		if (__unlikely(j == mini)) qh->qh.head_link = *addr_le;
		else if (__unlikely(qh->qh.head_link != *addr_le))
			KERNEL$SUICIDE("UHCI_ACTIVATE_INTR: INTERRUPT TREE CORRUPT: %08X != %08X", __32CPU2LE(qh->qh.head_link), __32CPU2LE(*addr_le));
		__barrier();
		*addr_le = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		already_have:;
	}
	qh->state = QHS_ACTIVE;
	ADD_TO_XLIST(&qh->uhci->intr_qh, &qh->list);
}

static unsigned QH_TIME(QH *qh)
{
	ED *ed = GET_STRUCT(qh, ED, qh_default);
	int bytes = USB$HC_ENDPOINT_BANDWIDTH(ed->type, ed->td_flags_1_le & __32CPU2LE(TD_FLAGS1_LS) ? USB_LOW_SPEED : USB_FULL_SPEED, ed->pkt_size);
	if (__unlikely(bytes < 0)) KERNEL$SUICIDE("QH_TIME: COULD NOT DETERMINE TIME FOR ALLOCATED ENDPOINT: %d", bytes);
	return bytes;
}

static void UHCI_CHANGE_TOGGLE(USB_REQUEST *brq)
{
	TD *td = (TD *)brq->internal1;
	ED *ed;
	while (1) {
		__u32 addr;
		td->td.flags2 ^= __32CPU2LE(TD_FLAGS2_D);
		addr = __32LE2CPU(td->td.link) & ~UHCI_ADDR_VF;
		if (__unlikely(addr & UHCI_ADDR_T)) break;
		td = GET_STRUCT(KERNEL$DMA_2_VIRT(addr), TD, td);
	}
	ed = (ED *)brq->internal2;
	ed->td_flags_2_le = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_D)) | ((~((TD *)brq->v.len)->td.flags2) & __32CPU2LE(TD_FLAGS2_D));
	UHCI_POST_BULK_INTR_CTRL_REQUEST(brq);
}

static void UHCI_POST_DISABLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("UHCI_POST_DISABLED");
	UHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -ECONNRESET;
	CALL_AST(brq);
}

static void UHCI_POST_STALLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("UHCI_POST_STALLED");
	UHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -EPIPE;
	CALL_AST(brq);
}

static void UHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	TD *td = (TD *)brq->internal1;
	ED *ed = (ED *)brq->internal2;
	brq->internal2 = 0;
	HC_CHECK_SPL("UHCI_FREE_BULK_INTR_CTRL_REQUEST");
	while (1) {
		__u32 addr;
		RAISE_SPL(SPL_VSPACE);
		td->data_dma_unlock(td->data_dma_addr);
		LOWER_SPL(SPL_USB);
		addr = __32LE2CPU(td->td.link) & ~UHCI_ADDR_VF;
		UHCI_FREE_TD(ed, td);
		if (__unlikely(addr & UHCI_ADDR_T)) break;
		td = GET_STRUCT(KERNEL$DMA_2_VIRT(addr), TD, td);
	}
}

static void UHCI_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	HC_CHECK_SPL("UHCI_CLEAR_STALL");
	ed->d_le = 0;
	ed->td_flags_2_le &= ~__32CPU2LE(TD_FLAGS2_D);
	if (__likely(ed->ep.post == UHCI_POST_STALLED))
		ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
	if (__unlikely(ed->qh != &ed->qh_default)) return;
	if (__unlikely(ed->qh_default.td_first != NULL))
		KERNEL$SUICIDE("UHCI_CLEAR_STALL: CLEAR STALL ON ACTIVE ENDPOINT");
}

static void UHCI_CANCEL_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	ED *ed;
	QH *qh;
	HC_CHECK_SPL("UHCI_CANCEL_BULK_INTR_CTRL_REQUEST");
	if (!brq->internal2) return;
	if (__likely(brq->status >= 0)) brq->status = -EINTR;
	ed = (ED *)brq->internal2;
	qh = ed->qh;
	if (__unlikely(qh->state == QHS_ACTIVE)) UHCI_START_UNLINK_QH(qh);
}

static void UHCI_SET_VERTICAL(USB_REQUEST *brq)
{
	TD *td = (TD *)brq->internal1;
	while (1) {
		if (td == (TD *)brq->v.len) break;
		td->td.link |= __32CPU2LE(UHCI_ADDR_VF);
		td = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(td->td.link) & ~UHCI_ADDR_VF), TD, td);
	}
}

static __NORET_ATTR__ void UHCI_PREPARE_CTRL_SUICIDE_ERR(int r);
static __NORET_ATTR__ void UHCI_PREPARE_CTRL_SUICIDE_PKT(void *p);

static int UHCI_PREPARE_CTRL_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *rq)
{
	int r;
	int was_data;
	unsigned ioc;
	USB_CTRL_REQUEST *brq = (USB_CTRL_REQUEST *)rq;
	ED *ed = GET_STRUCT(endpt, ED, ep);
	USB_REQUEST setup;
	TD *setup_td, *status_td;
	HC_CHECK_SPL("UHCI_PREPARE_CTRL_REQUEST");

	brq->internal2 = (unsigned long)ed;
	brq->status = brq->v.len;
	ioc = !(brq->flags & USBRQ_DELAY_INTERRUPT);
	brq->flags |= USBRQ_DELAY_INTERRUPT;
	ed->td_flags_2_le |= __32CPU2LE(TD_FLAGS2_D);
	was_data = 0;
	if (brq->v.len) {
		was_data = 1;
		if (brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN)) ed->type = 0;
		ed->td_flags_2_le = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_PID)) | (USB_IS_IN(ed->type) ? __32CPU2LE(TD_FLAGS2_PID_IN) : __32CPU2LE(TD_FLAGS2_PID_OUT));
		r = UHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, (USB_REQUEST *)brq);
		ed->type = USB_EP_CTRL;
		if (__unlikely(r)) return r;
	}

	setup.flags = USBRQ_DELAY_INTERRUPT;
	setup.v.ptr = (unsigned long)&brq->setup;
	setup.v.len = sizeof brq->setup;
	setup.v.vspace = &KERNEL$VIRTUAL;
	if (__unlikely(r = UHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, &setup)))
		UHCI_PREPARE_CTRL_SUICIDE_ERR(r);
	setup_td = (TD *)setup.internal1;
	if (__unlikely(!(setup_td->td.link & __32CPU2LE(UHCI_ADDR_T))))
		UHCI_PREPARE_CTRL_SUICIDE_PKT(brq);
	setup_td->brq = (USB_REQUEST *)brq;
	setup_td->td.flags2 = (setup_td->td.flags2 & ~__32CPU2LE(TD_FLAGS2_D | TD_FLAGS2_PID)) | __32CPU2LE(TD_FLAGS2_PID_SETUP);

	status_td = UHCI_GET_TD(ed);
	status_td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
	status_td->brq = (USB_REQUEST *)brq;
	status_td->td.link = __32CPU2LE(UHCI_ADDR_T);
	status_td->td.flags1 = __32CPU2LE(TD_FLAGS1_C_ERR_3 | TD_FLAGS1_STATUS_ACTIVE | TD_FLAGS1_ACTLEN) |
		(ed->td_flags_1_le & __32CPU2LE(TD_FLAGS1_LS)) |
		(ioc * __32CPU2LE(TD_FLAGS1_IOC));
	status_td->td.flags2 = __32CPU2LE(TD_FLAGS2_MAXLEN | TD_FLAGS2_D) |
		(ed->td_flags_2_le & __32CPU2LE(TD_FLAGS2_ENDPT | TD_FLAGS2_DEVICE_ADDRESS)) |
		(brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN) && was_data ? __32CPU2LE(TD_FLAGS2_PID_OUT) : __32CPU2LE(TD_FLAGS2_PID_IN));
	status_td->td.ptr = __32CPU2LE(0);

	if (was_data) {
		TD *first_td, *last_td;
		first_td = (TD *)brq->internal1;
		setup_td->td.link = __32CPU2LE(first_td->desc_dma_addr);
		brq->internal1 = (unsigned long)setup_td;
		last_td = (TD *)brq->v.len;
		last_td->td.link = __32CPU2LE(status_td->desc_dma_addr);
	} else {
		brq->internal1 = (unsigned long)setup_td;
		setup_td->td.link = __32CPU2LE(status_td->desc_dma_addr);
	}
	brq->v.len = (unsigned long)status_td;

	if (__likely(!(setup_td->td.flags1 & __32CPU2LE(TD_FLAGS1_LS)))) UHCI_SET_VERTICAL((USB_REQUEST *)brq);
	return 0;
}

static __NORET_ATTR__ void UHCI_PREPARE_CTRL_SUICIDE_ERR(int r)
{
	KERNEL$SUICIDE("UHCI_PREPARE_CTRL_SUICIDE_ERR: UHCI_PREPARE_BULK_INTR_REQUEST RETURNED ERROR %s", strerror(-r));
}

static __NORET_ATTR__ void UHCI_PREPARE_CTRL_SUICIDE_PKT(void *p)
{
	KERNEL$SUICIDE("UHCI_PREPARE_CTRL_SUICIDE_PKT: SETUP PACKET IS SPLIT, REQEUST %p", p);
}

static ED_PAGE *NEW_ED_PAGE(ED *ed)
{
	ED_PAGE *pg;
	CONTIG_AREA_REQUEST car;
	car.nclusters = 1;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (__unlikely(car.status < 0)) return __ERR_PTR(car.status);
	pg = car.ptr;
	pg->next = ed->pages;
	ed->pages = pg;
	return pg;
}

static USB_ENDPOINT *UHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size)
{
	UHCI *uhci = USB2UHCI(usb);
	unsigned packets, bounce_buffers;
	CONTIG_AREA_REQUEST car;
	ED *ed;
	unsigned long p;
	TD **pp;
	BOUNCE_BUFFER **bb;
	int error = 0;
	VDESC v;
	VDMA vdma;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!n_rq) || __unlikely(!rq_size))
		KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: REQUEST_RESERVATION NOT SET (N_RQ %u, RQ_SIZE %u", n_rq, rq_size);
	if (__unlikely(uhci->flags & FLAG_DEAD)) return __ERR_PTR(-EIO);
	if (__unlikely(addr == USB_ROOT_HUB_ADDR)) {
		switch (pipe) {
			case 0:
				if (type != USB_EP_CTRL) goto invl;
				uhci->root_ctrl.usb = usb;
				return &uhci->root_ctrl;
			case 1:
				if (type != USB_EP_INTR_IN) goto invl;
				uhci->root_intr.usb = usb;
				return &uhci->root_intr;
			default:
			invl:
				KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: INVALID ROOT HUB ENDPOINT: TYPE %d, PIPE %d", type, pipe);
		}
	}
	packets = ((rq_size + pkt_size - 1) / pkt_size + (USB_IS_CTRL(type)) * 2) * n_rq;
	bounce_buffers = (((rq_size + __PAGE_CLUSTER_SIZE - 2) >> __PAGE_CLUSTER_BITS) + USB_IS_CTRL(type)) * n_rq;
	car.nclusters = 1;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (__unlikely(car.status < 0)) return __ERR_PTR(car.status);
	ed = car.ptr;
	ADD_TO_XLIST(&uhci->endpoints, &ed->endpoints_list);
	INIT_XLIST(&ed->dmalock_list);
	ed->pkt_size = pkt_size;
	ed->type = type;
	ed->qh_default.uhci = uhci;
	ed->td_flags_1_le = __32CPU2LE(TD_FLAGS1_SPD | TD_FLAGS1_C_ERR_3 | ((speed == USB_LOW_SPEED) ? TD_FLAGS1_LS : 0) | (USB_IS_ISO(type) ? TD_FLAGS1_ISO : 0) | TD_FLAGS1_STATUS_ACTIVE | ((ed->pkt_size - 1) << __BSF_CONST(TD_FLAGS1_ACTLEN)));
	ed->td_flags_2_le = __32CPU2LE(((pkt_size - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN)) | (pipe << __BSF_CONST(TD_FLAGS2_ENDPT)) | (addr << __BSF_CONST(TD_FLAGS2_DEVICE_ADDRESS)) | (USB_IS_IN(type) ? TD_FLAGS2_PID_IN : TD_FLAGS2_PID_OUT));
	ed->d_le = 0;
	ed->addr = addr;
	ed->pipe = pipe;
	ed->freelist = NULL;
	ed->bounce_freelist = NULL;
	ed->n_tds = 0;
	ed->n_bounces = 0;
	ed->pages = NULL;
	pp = &ed->freelist;
	p = (unsigned long)(ed + 1);
	while (packets--) {
		TD_DMALOCK *tdl;
		if (__unlikely(!(p & __PAGE_CLUSTER_SIZE_MINUS_1))) goto new_page_1;
		align_again_1:
		p = (p + sizeof(HW_TD) - 1) & ~(unsigned long)(sizeof(HW_TD) - 1);
		if (__unlikely((p & __PAGE_CLUSTER_SIZE_MINUS_1) + sizeof(TD) > __PAGE_CLUSTER_SIZE)) {
			ED_PAGE *pg;
			new_page_1:
			pg = NEW_ED_PAGE(ed);
			if (__unlikely(__IS_ERR(pg))) {
				error = __PTR_ERR(pg);
				break;
			}
			p = (unsigned long)(pg + 1);
			goto align_again_1;
		}
#define tdx	((TD *)p)
		tdl = __sync_malloc(sizeof(TD_DMALOCK));
		if (__unlikely(!tdl)) {
			error = -ENOMEM;
			break;
		}
		v.ptr = (unsigned long)&tdx->td;
		v.len = sizeof(HW_TD);
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &tdl->desc_dma_unlock);
		LOWER_SPL(SPL_USB);
		tdl->desc_dma_addr = vdma.ptr;
		tdx->desc_dma_addr = vdma.ptr;
		tdx->brq = NULL;
		ADD_TO_XLIST(&ed->dmalock_list, &tdl->list);
		*pp = tdx;
		pp = (TD **)&tdx->brq;
#undef tdx
		p += sizeof(TD);
		ed->n_tds++;
	}
	bb = &ed->bounce_freelist;
	if (__likely(!error)) while (bounce_buffers--) {
		if (__unlikely(!(p & __PAGE_CLUSTER_SIZE_MINUS_1))) goto new_page_2;
		if (__unlikely((p & __PAGE_CLUSTER_SIZE_MINUS_1) + sizeof(BOUNCE_BUFFER) + pkt_size > __PAGE_CLUSTER_SIZE)) {
			ED_PAGE *pg;
			new_page_2:
			pg = NEW_ED_PAGE(ed);
			if (__unlikely(__IS_ERR(pg))) {
				error = __PTR_ERR(pg);
				break;
			}
			p = (unsigned long)(pg + 1);
		}
#define bbx	((BOUNCE_BUFFER *)p)
		v.ptr = (unsigned long)(bbx + 1);
		v.len = pkt_size;
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &bbx->data_dma_unlock);
		LOWER_SPL(SPL_USB);
		bbx->data_dma_addr = vdma.ptr;
		bbx->list = NULL;
		bbx->ed = ed;
		*bb = bbx;
		bb = &bbx->list;
#undef bbx
		p += sizeof(BOUNCE_BUFFER) + pkt_size;
		ed->n_bounces++;
	}
	if (USB_IS_CTRL(type) && __unlikely(speed == USB_LOW_SPEED)) {
		ed->qh = &UHCI2AREA(uhci)->low_ctrl;
	} else {
		ed->qh = &ed->qh_default;
		ed->qh_default.qh.element_link = __32CPU2LE(UHCI_ADDR_T);
		ed->qh_default.td_first = NULL;
		ed->qh_default.last_element_link = 0;
		ed->qh_default.link_last = &ed->qh_default.qh.element_link;
		/* ed->qh_default.uhci = uhci; */
		ed->qh_default.state = QHS_INACTIVE;
		ed->qh_default.type = type;

		if (__unlikely(interval > N_FRAME_LIST)) interval = N_FRAME_LIST;
		if (__unlikely(interval <= 0)) interval = 1;
		interval = 1 << __BSR(interval);
		ed->qh_default.stride = interval;

		v.ptr = (unsigned long)&ed->qh_default.qh;
		v.len = sizeof(HW_QH);
		v.vspace = &KERNEL$VIRTUAL;
		RAISE_SPL(SPL_VSPACE);
		vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &ed->desc_dma_unlock);
		LOWER_SPL(SPL_USB);
		ed->qh_default.desc_dma_addr = vdma.ptr;
	}

	switch (type) {
		case USB_EP_CTRL:
			ed->ep.prepare = UHCI_PREPARE_CTRL_REQUEST;
			ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = UHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = UHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = UHCI_CLEAR_STALL;
			break;
		case USB_EP_BULK_IN:
		case USB_EP_BULK_OUT:
		case USB_EP_INTR_IN:
		case USB_EP_INTR_OUT:
			ed->ep.prepare = UHCI_PREPARE_BULK_INTR_REQUEST;
			ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = UHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = UHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = UHCI_CLEAR_STALL;
			break;
		case USB_EP_ISO_IN:
		case USB_EP_ISO_OUT:
			/* unimplemented */
			error = -ENOSYS;
			break;
		default:
			KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: TYPE %d", type);
	}
	ed->ep.usb = usb;

	ed->n_bounce_requests = n_rq + (type == USB_EP_CTRL);
	ed->bounce_request_size = rq_size;
	if (__unlikely(error) || __unlikely(error = KERNEL$RESERVE_BOUNCE(uhci->dev_name, ed->n_bounce_requests, ed->bounce_request_size))) {
		ed->n_bounce_requests = 0;
		ed->bounce_request_size = 0;
	}

	if (__unlikely(error)) {
		UHCI_FREE_ENDPOINT(&ed->ep);
		return __ERR_PTR(error);
	}
	return &ed->ep;
}

static void UHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt)
{
	UHCI *uhci;
	ED *ed;
	QH *qh;
	TD *td;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_DISABLE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	uhci = USB2UHCI(endpt->usb);
	if (__unlikely(endpt == &uhci->root_ctrl) || __unlikely(endpt == &uhci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);
	ed->ep.post = UHCI_POST_DISABLED;
	qh = ed->qh;
	td = qh->td_first;
	if (td != NULL) while (1) {
		if (__likely((ED *)td->brq->internal2 == ed)) td->brq->status = -ECONNRESET;
		if (td->td.link & __32CPU2LE(UHCI_ADDR_T)) break;
		td = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(td->td.link) & ~UHCI_ADDR_VF), TD, td);
	}
	if (__likely(qh->state == QHS_ACTIVE)) UHCI_START_UNLINK_QH(qh);
}

static void UHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt)
{
	UHCI *uhci;
	ED *ed;
	unsigned i;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_FREE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	uhci = USB2UHCI(endpt->usb);
	if (__unlikely(endpt == &uhci->root_ctrl) || __unlikely(endpt == &uhci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);
	if (__likely(ed->qh == &ed->qh_default)) {
		if (__unlikely(ed->qh_default.td_first != NULL))
			KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: FREEING %s ENDPOINT %u:%u WHILE SOME REQUESTS ARE PENDING", USB$ENDPOINT_TYPE_STRING(ed->type), ed->addr, ed->pipe);
		if (ed->qh_default.state != QHS_INACTIVE) {
			if (ed->qh_default.state == QHS_ACTIVE) UHCI_START_UNLINK_QH(&ed->qh_default);
			ed->qh_default.state = QHS_UNLINKING_FREEING;
			return;
		}
		RAISE_SPL(SPL_VSPACE);
		ed->desc_dma_unlock(ed->qh_default.desc_dma_addr);
		LOWER_SPL(SPL_USB);
	}
	for (i = 0; i < ed->n_tds; i++) {
		TD_DMALOCK *tdl;
		UHCI_GET_TD(ed);
		if (__unlikely(XLIST_EMPTY(&ed->dmalock_list)))
			KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: TOO FEW ENTRIES ON DMALOCK_LIST");
		tdl = LIST_STRUCT(ed->dmalock_list.next, TD_DMALOCK, list);
		DEL_FROM_LIST(&tdl->list);
		RAISE_SPL(SPL_VSPACE);
		tdl->desc_dma_unlock(tdl->desc_dma_addr);
		LOWER_SPL(SPL_USB);
		free(tdl);
	}
	if (__unlikely(!XLIST_EMPTY(&ed->dmalock_list)))
		KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: TOO MANY ENTRIES DMALOCK_LIST");
	if (__unlikely(ed->freelist != NULL)) KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: N_TDS MISCOUNTED");
	for (i = 0; i < ed->n_bounces; i++) {
		BOUNCE_BUFFER *bb = UHCI_GET_BB(ed);
		RAISE_SPL(SPL_VSPACE);
		bb->data_dma_unlock(bb->data_dma_addr);
		LOWER_SPL(SPL_USB);
	}
	if (__unlikely(ed->bounce_freelist != NULL)) KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: N_BOUNCES MISCOUNTED");
	while (ed->pages != NULL) {
		ED_PAGE *pg = ed->pages->next;
		KERNEL$VM_RELEASE_CONTIG_AREA(ed->pages, 1);
		ed->pages = pg;
	}
	DEL_FROM_LIST(&ed->endpoints_list);
	if (__likely(ed->bounce_request_size != 0)) {
		KERNEL$UNRESERVE_BOUNCE(uhci->dev_name, ed->n_bounce_requests, ed->bounce_request_size);
	}
	KERNEL$VM_RELEASE_CONTIG_AREA(ed, 1);
}

static int UHCI_ROOT_PREPARE_CTRL(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	UHCI *uhci = GET_STRUCT(endpt, UHCI, root_ctrl);
	HC_CHECK_SPL("UHCI_ROOT_PREPARE_CTRL");
	brq->internal1 = (unsigned long)uhci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("UHCI_ROOT_PREPARE_CTRL: INVALID VSPACE");
	}
	return 0;
}

static void UHCI_ROOT_FREE(USB_REQUEST *rq)
{
	HC_CHECK_SPL("UHCI_ROOT_FREE");
}

static void UHCI_ROOT_POST_CTRL(USB_REQUEST *rq)
{
	USB_CTRL_REQUEST *urq = (USB_CTRL_REQUEST *)rq;
	UHCI *uhci = (UHCI *)urq->internal1;
	HC_CHECK_SPL("UHCI_ROOT_POST_CTRL");
	urq->status = 0;
	switch (__16LE2CPU(urq->setup.request)) {
		case USBHUB_REQ_SET_PORT_FEATURE: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= uhci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_SUSPEND:
					UHCI_SET_PORT_BIT(uhci, idx, UHCI_PORTSC_SUSPEND);
					break;
				case USB_PORT_FEAT_RESET:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE);
					UHCI_SET_PORT_BIT(uhci, idx, UHCI_PORTSC_RESET);
					KERNEL$UDELAY(10);
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE_CHANGE);
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_CONNECT_CHANGE);
					KERNEL$DEL_TIMER(&uhci->hub_timer);
					KERNEL$SET_TIMER(UHCI_HUB_RESET_TIME, &uhci->hub_timer);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_CLEAR_PORT_FEATURE: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= uhci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_ENABLE:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE);
					break;
				case USB_PORT_FEAT_C_ENABLE:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE_CHANGE);
					break;
				/* unimplemented 
				case USB_PORT_FEAT_SUSPEND:
				case USB_PORT_FEAT_C_SUSPEND:
				*/
				case USB_PORT_FEAT_C_CONNECTION:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_CONNECT_CHANGE);
					break;
				case USB_PORT_FEAT_C_OVER_CURRENT:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_OC_CHANGE);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_GET_HUB_STATUS: {
			if (__unlikely(urq->v.len != sizeof(__u32))) {
				urq->status = -EINVAL;
				break;
			}
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(0);
			urq->status = sizeof(__u32);
			break;
		}
		case USBHUB_REQ_GET_PORT_STATUS: {
			unsigned idx = __16LE2CPU(urq->setup.w_index) - 1;
			__u32 stat;
			__u16 ps;
			if (__unlikely(idx >= uhci->n_ports)) {
				urq->status = -EINVAL;
				break;
			}
			if (__unlikely(urq->v.len != sizeof(__u32))) {
				urq->status = -EINVAL;
				break;
			}
			stat = 0;
			ps = UHCI_READ_PORT(uhci, idx);
			if (uhci->flags & FLAG_REVERT_OC) ps ^= UHCI_PORTSC_OC;
			if (ps & UHCI_PORTSC_CONNECT_CHANGE) stat |= USB_PORT_STAT_C_CONNECTION;
			if (ps & UHCI_PORTSC_ENABLE_CHANGE) stat |= USB_PORT_STAT_C_ENABLE;
			if (ps & UHCI_PORTSC_OC_CHANGE) stat |= USB_PORT_STAT_C_OVER_CURRENT;
			stat |= USB_PORT_STAT_POWER;
			if (ps & UHCI_PORTSC_CONNECT) stat |= USB_PORT_STAT_CONNECTION;
			if (ps & UHCI_PORTSC_ENABLE) stat |= USB_PORT_STAT_ENABLE;
			if (ps & (UHCI_PORTSC_SUSPEND | UHCI_PORTSC_RESUME_DETECT)) stat |= USB_PORT_STAT_SUSPEND;
			if (ps & UHCI_PORTSC_OC) stat |= USB_PORT_STAT_OVER_CURRENT;
			if (ps & UHCI_PORTSC_RESET) stat |= USB_PORT_STAT_RESET;
			if (ps & UHCI_PORTSC_LOW_SPEED) stat |= USB_PORT_STAT_LOWSPEED;
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(stat);
			/*__debug_printf("PORT STATUS: %04X -> %08X\n", ps, stat);*/
			urq->status = sizeof(__u32);
			break;
		}
		case USBHUB_REQ_GET_HUB_DESCRIPTOR: {
			USB_DESCRIPTOR_HUB *desc;
			if (__unlikely(urq->v.len != USB_DESCRIPTOR_HUB_SIZE)) {
				urq->status = -EINVAL;
				break;
			}
			desc = (USB_DESCRIPTOR_HUB *)(unsigned long)urq->v.ptr;
			memset(desc, 0xff, USB_DESCRIPTOR_HUB_SIZE);
			desc->bDescLength = 7 + 2 * ((uhci->n_ports + 1 + 7) / 8);
			desc->bDescriptorType = USB_DT_HUB;
			desc->bNbrPorts = uhci->n_ports;
			desc->wHubCharacteristics = USB_DESCRIPTOR_HUB_CHAR_OC_INDIVIDUAL | USB_DESCRIPTOR_HUB_CHAR_POWER_NONE;
			desc->wHubCharacteristics2 = 0;
			desc->bPwrOn2PwrGood = 1;
			desc->bHubContrCurrent = 0;
			desc->DeviceRemovable[0] = 0;
			urq->status = desc->bDescLength;
			break;
		}
		default:
		dflt: {
			USB$HC_GENERIC_ROOT_CTRL(UHCI2USB(uhci), urq);
			return;
		}
	}
	CALL_AST(urq);
}

static void UHCI_ROOT_CANCEL_CTRL(USB_REQUEST *rq)
{
	HC_CHECK_SPL("UHCI_ROOT_CANCEL_CTRL");
}

static int UHCI_ROOT_PREPARE_INTR(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	UHCI *uhci = GET_STRUCT(endpt, UHCI, root_intr);
	HC_CHECK_SPL("UHCI_ROOT_PREPARE_INTR");
	brq->internal1 = (unsigned long)uhci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("UHCI_ROOT_PREPARE_INTR: INVALID VSPACE");
	}
	return 0;
}

static void UHCI_ROOT_POST_INTR(USB_REQUEST *urq)
{
	UHCI *uhci = (UHCI *)urq->internal1;
	HC_CHECK_SPL("UHCI_ROOT_POST_INTR");
	if (__unlikely(uhci->rh_intr != NULL))
		KERNEL$SUICIDE("UHCI_ROOT_POST_INTR: ANOTHER REQUEST ALREADY PENDING");
	if (__unlikely(urq->v.len < ((uhci->n_ports + 1 + 7) >> 3))) {
		urq->status = -EINVAL;
		CALL_AST(urq);
		return;
	}
	urq->v.len = (uhci->n_ports + 1 + 7) >> 3;
	memset((void *)(unsigned long)urq->v.ptr, 0, urq->v.len);
	urq->status = RQS_CANCELABLE;
	uhci->rh_intr = urq;
	return;
}

static void UHCI_ROOT_CANCEL_INTR(USB_REQUEST *urq)
{
	UHCI *uhci = (UHCI *)urq->internal1;
	HC_CHECK_SPL("UHCI_ROOT_CANCEL_INTR");
	if (__unlikely(urq->status != RQS_CANCELABLE)) return;
	uhci = (UHCI *)urq->internal1;
	if (__likely(urq == uhci->rh_intr)) {
		uhci->rh_intr = NULL;
		urq->status = -EINTR;
		CALL_AST(urq);
	}
}

static void UHCI_ROOT_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	HC_CHECK_SPL("UHCI_ROOT_CLEAR_STALL");
}

static void UHCI_HUB_TIMER(TIMER *t)
{
	int set;
	unsigned i;
	UHCI *uhci;
	LOWER_SPL(SPL_USB);
	uhci = GET_STRUCT(t, UHCI, hub_timer);
	KERNEL$SET_TIMER(UHCI_HUB_CHECK_TIME, &uhci->hub_timer);
	UHCI_PROCESS_SCHEDULE(uhci);
	set = 0;
	for (i = 0; i < uhci->n_ports; i++) {
		__u16 status = UHCI_READ_PORT(uhci, i);
		/*if (!i) {
			static __u16 old = 0;
			if (status != old)
				__debug_printf("%s: port %d, status %04x\n", uhci->dev_name, i, status);
			old = status;
		}*/
		if (__unlikely(status & UHCI_PORTSC_RESET)) {
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_RESET);
			KERNEL$UDELAY(10);
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_ENABLE_CHANGE);
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_CONNECT_CHANGE);
			UHCI_SET_PORT_BIT(uhci, i, UHCI_PORTSC_ENABLE);
			status = UHCI_READ_PORT(uhci, i);
		}
		if (__unlikely(status & (UHCI_PORTSC_CONNECT_CHANGE | UHCI_PORTSC_ENABLE_CHANGE | UHCI_PORTSC_OC_CHANGE)) && __likely(uhci->rh_intr != NULL)) {
			__u8 *data = (__u8 *)(unsigned long)uhci->rh_intr->v.ptr;
			unsigned off = i + 1;
			data[off / 8] |= 1 << (off & 7);
			set = 1;
		}
	}
	if (__unlikely(set)) {
		USB_REQUEST *intr = uhci->rh_intr;
		intr->status = intr->v.len;
		uhci->rh_intr = NULL;
		CALL_AST(intr);
	}
}

static int UHCI_DETECT_PORTS(UHCI *uhci)
{
	unsigned port;
	for (port = 0; port <= UHCI_MAX_PORTS; port++) {
		__u16 status = UHCI_READ_PORT(uhci, port);
		/*__debug_printf("%d: %x, %d\n", port, status, uhci->flags);*/
		if (!(status & UHCI_PORTSC_RESERVED_ONE) || status == 0xffff) break;
	}
	if (port > UHCI_MAX_PORTS) port = UHCI_DEFAULT_PORTS;
	return port;
}

static void UHCI_TAKEOVER(pci_id_t id, int grab_it)
{
	__u16 legsup;
	__u16 usbsts;
	__u16 usbcmd;
	io_t io;
	if (grab_it) {
		if (__unlikely(PCI$FIND_DEVICE(uhci_pci, id, ~(pci_id_t)0, 0, PCI$TEST_LIST, NULL, NULL, NULL, 0))) return;
		/*__debug_printf("id: %08x\n", id);*/
	}
	io = PCI$READ_IO_RESOURCE(id, 4);
	if (__unlikely(!io)) goto ret;
	usbcmd = io_inw(io + UHCI_USBCMD);
	usbcmd &= ~(UHCI_USBCMD_RS | UHCI_USBCMD_CF);
	io_outw(io + UHCI_USBCMD, usbcmd);
	KERNEL$UDELAY(1500);
	usbsts = io_inw(io + UHCI_USBSTS);
	io_outw(io + UHCI_USBSTS, usbsts);
	legsup = PCI$READ_CONFIG_WORD(id, UHCI_PCI_LEGSUP);
	if (legsup & UHCI_PCI_LEGSUP_HAS_BIOS) {
		PCI$WRITE_CONFIG_WORD(id, UHCI_PCI_LEGSUP, legsup & ~UHCI_PCI_LEGSUP_USBSMIEN);
		/*__debug_printf("takeover...\n");*/
	}
	/*else __debug_printf("no takeover...\n");*/
	PCI$WRITE_CONFIG_WORD(id, UHCI_PCI_LEGSUP, UHCI_PCI_LEGSUP_DEFAULT);
	ret:
	if (grab_it) {
		PCI$FREE_DEVICE(id);
	}
}

static int UHCI_RESET(UHCI *uhci)
{
	unsigned i;
	__u8 sofmod;
	sofmod = UHCI_READ_8(uhci, UHCI_SOFMOD);
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_HCRESET);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (!(UHCI_READ_16(uhci, UHCI_USBCMD) & UHCI_USBCMD_HCRESET)) goto done_reset;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "RESET FAILED");
	return -EIO;
	done_reset:
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBCMD, 0);
	for (i = 0; i < uhci->n_ports; i++) {
		UHCI_WRITE_PORT(uhci, i, 0);
	}
	UHCI_WRITE_8(uhci, UHCI_SOFMOD, sofmod);
	UHCI_WRITE_32(uhci, UHCI_FLBASEADD, uhci->area_dmaaddr + __offsetof(AREA, fl));
	uhci->last_head_link = &UHCI2AREA(uhci)->loop_start.head_link;
	INIT_XLIST(&uhci->intr_qh);
	for (i = 0; i < N_FRAME_LIST; i++) {
		UHCI2AREA(uhci)->fl.addr[i] = UHCI_CTRL_START_Q_LE(uhci);
	}
	UHCI2AREA(uhci)->sof_irq.link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->sof_irq.flags1 = __32CPU2LE(TD_FLAGS1_IOC | TD_FLAGS1_ACTLEN);
	UHCI2AREA(uhci)->sof_irq.flags2 = __32CPU2LE(TD_FLAGS2_MAXLEN | TD_FLAGS2_DEVICE_ADDRESS | TD_FLAGS2_PID_SETUP);
	UHCI2AREA(uhci)->sof_irq.ptr = __32CPU2LE(0);
	UHCI2AREA(uhci)->ctrl_start.head_link = UHCI_LOOP_START_Q_LE(uhci);
	UHCI2AREA(uhci)->ctrl_start.element_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->loop_start.head_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->loop_start.element_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->low_ctrl.qh.element_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->low_ctrl.td_first = NULL;
	UHCI2AREA(uhci)->low_ctrl.last_element_link = 0;
	UHCI2AREA(uhci)->low_ctrl.link_last = &UHCI2AREA(uhci)->low_ctrl.qh.element_link;
	UHCI2AREA(uhci)->low_ctrl.uhci = uhci;
	UHCI2AREA(uhci)->low_ctrl.state = QHS_INACTIVE;
	UHCI2AREA(uhci)->low_ctrl.type = USB_EP_CTRL;
	UHCI2AREA(uhci)->low_ctrl.stride = 1;
	UHCI2AREA(uhci)->low_ctrl.desc_dma_addr = __32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, low_ctrl.qh));
	return 0;
}

static void UHCI_CONTINUE_RESET(UHCI *uhci)
{
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_RS | UHCI_USBCMD_MAXP);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, UHCI_USBINTR_TIMEOUT_CRC | UHCI_USBINTR_RESUME | UHCI_USBINTR_IOC | UHCI_USBINTR_SHORT_PACKET);
}

static void UHCI_HARD_ERROR(UHCI *uhci);

static UHCI *UHCI_SPECIAL_IRQ(UHCI *uhci, __u16 sts)
{
	if (uhci->flags & FLAG_DEAD) goto h_e;
	if (sts == 0xffff) goto h_e;
	if (sts & UHCI_USBSTS_HC_PROCESS_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UHCI PROCESSING ERROR");
	if (sts & UHCI_USBSTS_HC_SYSTEM_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, uhci->dev_name, "UHCI PCI SYSTEM ERROR");
	if (sts & UHCI_USBSTS_RESUME_DETECT)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UNEXPECTED RESUME INTERRUPT");
	if (sts & ~(UHCI_USBSTS_HC_PROCESS_ERROR | UHCI_USBSTS_HC_SYSTEM_ERROR | UHCI_USBSTS_RESUME_DETECT)) {
		if ((sts & ~UHCI_USBSTS_RESERVED) == UHCI_USBSTS_HC_HALTED)
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "CONTROLLER UNEXPECTEDLY HALTED");
		else
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UNKNOWN STATUS REGISTER: %04X", sts);
	}
	h_e:
	UHCI_HARD_ERROR(uhci);
	return uhci;
}

static int UHCI_STOP(UHCI *uhci)
{
	unsigned i;
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBCMD, 0);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (UHCI_READ_16(uhci, UHCI_USBSTS) & UHCI_USBSTS_HC_HALTED) goto done_stop;
	}
	if (uhci->errorlevel >= 2)
		KERNEL$SYSLOG(__SYSLOG_HW_WARNING, uhci->dev_name, "STOP FAILED, RESETTING");
	done_stop:
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_HCRESET);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (!(UHCI_READ_16(uhci, UHCI_USBCMD) & UHCI_USBCMD_HCRESET)) goto done_reset;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "RESET FAILED, TRYING HARD RESET");
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_GRESET);
	return -EIO;
	done_reset:
	return 0;
}

static void UHCI_DIE(UHCI *uhci);

static void UHCI_HARD_ERROR(UHCI *uhci)
{
	ED *ed;
	QH *qh;
	if (__unlikely(UHCI_STOP(uhci))) {
		UHCI_DIE(uhci);
	}
	XLIST_FOR_EACH(ed, &uhci->endpoints, ED, endpoints_list)
		UHCI_DISABLE_ENDPOINT(&ed->ep);
	XLIST_FOR_EACH(qh, &uhci->intr_qh, QH, list)	/* needed only to unlink low-speed control queue */
		UHCI_START_UNLINK_QH(qh);
	UHCI_PROCESS_UNLINKING_ENTRIES(uhci, 1);
	if (__unlikely(!XLIST_EMPTY(&uhci->intr_qh)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: INTERRUPT LIST NOT EMPTY");
	if (__unlikely(UHCI2AREA(uhci)->ctrl_start.head_link != UHCI_LOOP_START_Q_LE(uhci)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: CONTROL LIST NOT EMPTY");
	if (__unlikely(UHCI2AREA(uhci)->loop_start.head_link != __32CPU2LE(UHCI_ADDR_T)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: BULK LIST NOT EMPTY");
	if (__unlikely(UHCI_RESET(uhci)))
		UHCI_DIE(uhci);
}

static void UHCI_DIE(UHCI *uhci)
{
	if (!(uhci->flags & FLAG_DEAD))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "CONTROLLER DIED");
	uhci->flags |= FLAG_DEAD;
}

static void UHCI_INIT_ROOT(HANDLE *h, void *data)
{
	UHCI *uhci = USB2UHCI(data);
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = uhci;
	h->op = &UHCI_HANDLE_OPERATIONS;
}

static void UHCI_WAIT_FOR_LISTS(UHCI *uhci)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS AT SPL %08X", KERNEL$SPL);
	if (UHCI2AREA(uhci)->low_ctrl.state == QHS_ACTIVE) UHCI_START_UNLINK_QH(&UHCI2AREA(uhci)->low_ctrl);
	/* this won't loop indefinitely because of unlink_start */
	while (1) {
		if (__unlikely(!XLIST_EMPTY(&uhci->intr_qh)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED INTERRUPT OR LOW-SPEED CONTROL ENDPOINT");
		if (__unlikely(UHCI2AREA(uhci)->ctrl_start.head_link != UHCI_LOOP_START_Q_LE(uhci)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED CONTROL ENDPOINT");
		if (__unlikely(UHCI2AREA(uhci)->loop_start.head_link != __32CPU2LE(UHCI_ADDR_T)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED BULK ENDPOINT");
		if (__likely(XLIST_EMPTY(&uhci->unlinking))) break;
		KERNEL$SLEEP(1);
		UHCI_PROCESS_SCHEDULE(uhci);	/* in case interrupt was blocked by some other driver */
	}
	if (__unlikely(!XLIST_EMPTY(&uhci->endpoints)))
		KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: ENDPOINTS LEAKED");
}

static int UHCI_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	unsigned fsbr_time = DEFAULT_FSBR_TIME;
	char **arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char dev_name[__MAX_STR_LEN];
	char gstr[__MAX_STR_LEN];
	char *opt, *optend, *str;
	int r, rr, reg = 0;
	unsigned long flags;
	char *chip_name;
	io_t io;
	int irq;
	CONTIG_AREA_REQUEST car;
	DEVICE_REQUEST devrq;
	UHCI *uhci;
	VDESC desc;
	VDMA vdma;
	int n_ports;
	char *e;
	int errorlevel = 0;
	char *plug = USB$HC_DEFAULT_PLUG_SCRIPT;
	struct __param_table params[] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		"PLUG", __PARAM_STRING, 0, __MAX_STR_LEN, NULL,
		"FSBR_TIME", __PARAM_UNSIGNED_INT, 0, (MAXINT - 999) / JIFFIES_PER_SECOND, NULL,
		NULL, 0, 0, 0, NULL,
	};

	params[0].__p = &errorlevel;
	params[1].__p = &errorlevel;
	params[2].__p = &plug;
	params[3].__p = &fsbr_time;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UHCI: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(uhci_pci, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UHCI: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}

	BUST_EHCI();

	_snprintf(dev_name, sizeof(dev_name), "USB$UHCI@" PCI_ID_FORMAT, id);
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);

	io = PCI$READ_IO_RESOURCE(id, 4);
	if (!io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	irq = PCI$READ_INTERRUPT_LINE(id);
	/* Documentation says that function 2 holds the LEGSUP register, however
	   ICH6 will lock up if takeover is not performed on function 0 (and
	   doesn't care about other functions). Rather takeover both. */
	RAISE_SPL(SPL_DEV);
	UHCI_TAKEOVER((id & ~(pci_id_t)PCI_ID_T_FUNCTION) | ((pci_id_t)0 << PCI_ID_T_FUNCTION_SHIFT), 1);
	UHCI_TAKEOVER((id & ~(pci_id_t)PCI_ID_T_FUNCTION) | ((pci_id_t)2 << PCI_ID_T_FUNCTION_SHIFT), 1);
	LOWER_SPL(SPL_ZERO);
	car.nclusters = N_CLUSTERS;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		if (car.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-car.status));
		r = car.status;
		goto ret2;
	}
	memset(car.ptr, 0, sizeof(AREA) + sizeof(UHCI));
	uhci = (UHCI *)((char *)car.ptr + sizeof(AREA));
	strcpy(uhci->dev_name, dev_name);
	uhci->id = id;
	uhci->io = io;
	uhci->flags = flags;
	uhci->errorlevel = errorlevel;
	uhci->fsbr_time = (fsbr_time * JIFFIES_PER_SECOND + 999) / 1000;
	INIT_TIMER(&uhci->fsbr_timer);
	uhci->fsbr_timer.fn = UHCI_FSBR_TIMER;
	VOID_LIST_ENTRY(&uhci->fsbr_timer.list);
	INIT_XLIST(&uhci->unlinking);
	INIT_XLIST(&uhci->endpoints);

	uhci->root_ctrl.prepare = UHCI_ROOT_PREPARE_CTRL;
	uhci->root_ctrl.post = UHCI_ROOT_POST_CTRL;
	uhci->root_ctrl.free = UHCI_ROOT_FREE;
	uhci->root_ctrl.cancel = UHCI_ROOT_CANCEL_CTRL;
	uhci->root_ctrl.clear_stall = UHCI_ROOT_CLEAR_STALL;
	uhci->root_intr.prepare = UHCI_ROOT_PREPARE_INTR;
	uhci->root_intr.post = UHCI_ROOT_POST_INTR;
	uhci->root_intr.free = UHCI_ROOT_FREE;
	uhci->root_intr.cancel = UHCI_ROOT_CANCEL_INTR;
	uhci->root_intr.clear_stall = UHCI_ROOT_CLEAR_STALL;

	desc.ptr = (unsigned long)UHCI2AREA(uhci);
	desc.len = sizeof(AREA);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	vdma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &uhci->area_dmaunlock);
	LOWER_SPL(SPL_ZERO);
	uhci->area_dmaaddr = vdma.ptr;

	n_ports = UHCI_DETECT_PORTS(uhci);
	if (__unlikely(n_ports <= 0)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO ROOT HUB PORTS");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO ROOT HUB PORTS", dev_name);
		r = -ENXIO;
		goto ret3;
	}
	uhci->n_ports = n_ports;
	RAISE_SPL(SPL_USB);
	UHCI_TAKEOVER(id, 0);
	if (__unlikely(r = UHCI_RESET(uhci))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INITIALIZE CONTROLLER", dev_name);
		goto ret4;
	}
	uhci->irq_ast.fn = UHCI_IRQ;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &uhci->irq_ast, &uhci->irq_ctrl, dev_name))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		r = -EINVAL;
		goto ret4;
	}
	UHCI_CONTINUE_RESET(uhci);
	INIT_TIMER(&uhci->hub_timer);
	uhci->hub_timer.fn = UHCI_HUB_TIMER;
	KERNEL$SET_TIMER(UHCI_HUB_CHECK_TIME, &uhci->hub_timer);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r = USB$HC_REGISTER(UHCI2USB(uhci), &UHCI_HC_OPERATIONS, uhci->dev_name, errorlevel, plug))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER USB HOST CONTROLLER: %s", dev_name, strerror(-r));
		goto ret5;
	}
	reg = 1;

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, id));
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", dev_name, io, irq);

	devrq.name = dev_name;
	devrq.driver_name = "UHCI.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = UHCI_INIT_ROOT;
	devrq.dev_ptr = UHCI2USB(uhci);
	devrq.dcall = USB$HC_DCALL;
	devrq.dcall_type = USB$HC_DCALL_TYPE;
	devrq.dctl = USB$HC_DCTL;
	devrq.unload = UHCI_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret5;
	}
	uhci->lnte = devrq.lnte;
	uhci->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret5:
	uhci->irq_ctrl.mask();
	RAISE_SPL(SPL_USB);
	UHCI_WAIT_FOR_LISTS(uhci);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	KERNEL$RELEASE_IRQ(&uhci->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	if (reg) {
		if ((rr = USB$HC_PREPARE_UNREGISTER(UHCI2USB(uhci))))
			KERNEL$SUICIDE("UHCI: FAILED TO UNREGISTER UHCI CONTROLLER: %s", strerror(-rr));
		USB$HC_UNREGISTER(UHCI2USB(uhci));
	}
	RAISE_SPL(SPL_USB);
	KERNEL$DEL_TIMER(&uhci->hub_timer);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	ret4:
	UHCI_STOP(uhci);
	LOWER_SPL(SPL_ZERO);
	ret3:
	RAISE_SPL(SPL_VSPACE);
	uhci->area_dmaunlock(uhci->area_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	KERNEL$VM_RELEASE_CONTIG_AREA(UHCI2AREA(uhci), N_CLUSTERS);
	ret2:
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int UHCI_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	UHCI *uhci = USB2UHCI(p);
	if (argv[1]) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", uhci->dev_name);
		return -EBADSYN;
	}
	if ((r = USB$HC_PREPARE_UNREGISTER(UHCI2USB(uhci)))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: UNABLE TO REMOVE HOST CONTROLLER: %s", uhci->dev_name, strerror(-r));
		return r;
	}
	KERNEL$UNREGISTER_DEVICE(uhci->lnte, 0);
	uhci->irq_ctrl.mask();
	RAISE_SPL(SPL_USB);
	UHCI_WAIT_FOR_LISTS(uhci);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	KERNEL$RELEASE_IRQ(&uhci->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	USB$HC_UNREGISTER(UHCI2USB(uhci));
	RAISE_SPL(SPL_USB);
	KERNEL$DEL_TIMER(&uhci->hub_timer);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	UHCI_STOP(uhci);
	LOWER_SPL(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	uhci->area_dmaunlock(uhci->area_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	PCI$FREE_DEVICE(uhci->id);
	*release = uhci->dlrq;
	KERNEL$VM_RELEASE_CONTIG_AREA(UHCI2AREA(uhci), N_CLUSTERS);
	return 0;
}

