#include <SPAD/SYSLOG.H>

#include "USB.H"

__const__ char *USB$ENDPOINT_TYPE_STRING(int type)
{
	switch (type) {
		case USB_EP_CTRL: return "CONTROL";
		case USB_EP_BULK_IN: return "BULK IN";
		case USB_EP_BULK_OUT: return "BULK OUT";
		case USB_EP_INTR_IN: return "INTERRUPT IN";
		case USB_EP_INTR_OUT: return "INTERRUPT OUT";
		case USB_EP_ISO_IN: return "ISOCHRONOUS IN";
		case USB_EP_ISO_OUT: return "ISOCHRONOUS OUT";
		default: KERNEL$SUICIDE("USB$ENDPOINT_TYPE_STRING: UNKNOWN ENDPOINT TYPE %d", type);
	}
}

__const__ char *USB$SPEED_STRING(int speed)
{
	switch (speed) {
		case USB_LOW_SPEED: return "LOW-SPEED";
		case USB_FULL_SPEED: return "FULL-SPEED";
		case USB_HIGH_SPEED: return "HIGH-SPEED";
		default: KERNEL$SUICIDE("USB$SPEED_STRING: UNKNOWN SPEED %d", speed);
	}
}

int USB$HC_ENDPOINT_BANDWIDTH(int type, int speed, unsigned pkt_size)
{
	unsigned r;
	if (speed == USB_LOW_SPEED && USB_IS_INTR(type)) {
		r = (pkt_size + 19) * (USB_HIGH_SPEED / USB_LOW_SPEED);
	} else
	if (speed == USB_FULL_SPEED && USB_IS_INTR(type)) {
		r = (pkt_size + 13) * (USB_HIGH_SPEED / USB_FULL_SPEED);
	} else
	if (speed == USB_HIGH_SPEED && USB_IS_INTR(type)) {
		r = pkt_size + 55;
	} else
	if (speed == USB_FULL_SPEED && USB_IS_ISO(type)) {
		r = (pkt_size + 9) * (USB_HIGH_SPEED / USB_FULL_SPEED);
	} else
	if (speed == USB_HIGH_SPEED && USB_IS_ISO(type)) {
		r = pkt_size + 38;
	} else
		return -EINVAL;
	return (r * 7 + 5) / 6;	/* bit stuffing */
}

int USB$GET_DEVICE_SPEED(USB_DEV_INTERFACE *iface)
{
	return iface->cfg->dev->speed;
}

USB_DESCRIPTOR_DEVICE *USB$GET_DEVICE_DESCRIPTOR(USB_DEV_INTERFACE *iface)
{
	return &iface->cfg->dev->dev_desc;
}

USB_DESCRIPTOR_CONFIGURATION *USB$GET_CONFIGURATION_DESCRIPTOR(USB_DEV_INTERFACE *iface)
{
	return &iface->cfg->cfg_desc;
}

USB_DESCRIPTOR_INTERFACE *USB$GET_INTERFACE_DESCRIPTOR(USB_DEV_INTERFACE *iface)
{
	return iface->desc;
}

USB_DESCRIPTOR_ENDPOINT *USB$GET_ENDPOINT_DESCRIPTOR(USB_DEV_INTERFACE *iface, int ep_type, unsigned n)
{
	unsigned i;
	for (i = 0; i < iface->n_endpts; i++) {
		USB_DESCRIPTOR_ENDPOINT *desc = iface->endpts[n].desc;
		if (__likely(ep_type != -1) && __unlikely(USB_ENDPOINT_TYPECODE(desc) != ep_type)) continue;
		if (__unlikely(n--)) continue;
		return desc;
	}
	return NULL;
}

USB_ENDPOINT *USB$FIND_ENDPOINT(USB_DEV_INTERFACE *iface, int ep_type, unsigned max_requests, unsigned max_request_size)
{
	unsigned i;
	for (i = 0; i < iface->n_endpts; i++) {
		USB_ENDPOINT *ret;
		int spl = KERNEL$SPL;
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), spl)))
			KERNEL$SUICIDE("USB$FIND_ENDPOINT AT SPL %08X", spl);
		RAISE_SPL(SPL_USB);
		if (iface->endpts[i].used) goto cont;
		if (ep_type != USB_ENDPOINT_TYPECODE(iface->endpts[i].desc)) goto cont;
		ret = USB$GET_ENDPOINT(iface, i, ep_type, max_requests, max_request_size);
		LOWER_SPLX(spl);
		return ret;

		cont:
		LOWER_SPLX(spl);
	}
	if (iface->cfg->dev->usb->errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "DEVICE %s, %s DOES NOT HAVE ENDPOINT TYPE %s", USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), USB$ENDPOINT_TYPE_STRING(ep_type));
	return __ERR_PTR(-ENOENT);
}

int USB$REQUEST_DESCRIPTOR(USB_DEV_INTERFACE *iface, __u8 rqtype, __u8 dt, __u8 idx, __u16 windex, void *ptr, unsigned len, unsigned min_len, int flags)
{
	int r;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), spl)))
		KERNEL$SUICIDE("USB$REQUEST_DESCRIPTOR AT SPL %08X", spl);
	RAISE_SPL(SPL_USB);
	r = USB_GET_DESCRIPTOR(iface->cfg->dev, rqtype, dt, idx, windex, ptr, len, min_len, flags);
	LOWER_SPLX(spl);
	return r;
}

void *USB$FIND_DESCRIPTOR(USB_DEV_INTERFACE *iface, __u8 dt, int skip, unsigned len)
{
	unsigned off = (char *)iface->desc - (char *)iface->cfg->descs;
	while (1) {
		void *desc = USB_FIND_DESCRIPTOR(iface->cfg, &off, dt, len, USB_DT_INTERFACE);
		if (!desc) return NULL;
		if (!skip--) return desc;
	}
}

void USB$FILL_CLEAR_STALL_REQUEST(USB_DEV_INTERFACE *iface, USB_ENDPOINT *ep, USB_CTRL_REQUEST *rq)
{
	unsigned i;
	USB_DEV *dev = iface->cfg->dev;
	for (i = 0; i < USB_MAX_ENDPOINTS; i++) {
		if (dev->out_endpoints[i] == ep) {
			goto have_it;
		}
		if (dev->in_endpoints[i] == ep) {
			i |= USB_ENDPOINT_ADDRESS_IN;
			goto have_it;
		}
	}
	KERNEL$SUICIDE("USB$FILL_CLEAR_STALL_REQUEST: NONEXISTENT ENDPOINT, HOST %s, DEVICE %s", ep->usb->dev_name, USB_DEV_ID(dev));
	have_it:
	rq->flags = 0;
	rq->v.ptr = 0;
	rq->v.len = 0;
	rq->v.vspace = &KERNEL$VIRTUAL;
	rq->setup.request = __16CPU2LE(USB_CTRL_RQTYPE_ENDPOINT_OUT | USB_REQ_CLEAR_FEATURE);
	rq->setup.w_value = __16CPU2LE(USB_ENDPOINT_FEAT_HALT);
	rq->setup.w_index = __16CPU2LE(i);
	rq->setup.w_length = __16CPU2LE(0);
}

u_jiffies_lo_t USB$GET_CTRL_TIMEOUT(USB_DEV_INTERFACE *iface)
{
	return USB_CTRL_TIMEOUT;
}

int USB$GET_CTRL_RETRIES(USB_DEV_INTERFACE *iface)
{
	return USB_CTRL_RETRIES;
}

int USB$GET_INTERFACE_NUMBER(USB_DEV_INTERFACE *iface)
{
	return iface->desc->bInterfaceNumber;
}

int USB$GET_ERRORLEVEL(USB_DEV_INTERFACE *iface)
{
	return iface->cfg->dev->usb->errorlevel;
}

void USB$GET_DEVICE_NAME(char *result, size_t len, USB_DEV_INTERFACE *iface, char *beginning)
{
	_snprintf(result, len, "%s_A%d_I%u@%s", beginning, iface->cfg->dev->addr, USB$GET_INTERFACE_NUMBER(iface), iface->cfg->dev->usb->dev_name);
}

int USB$IS_ATTACHED(USB_DEV_INTERFACE *iface)
{
	return iface->cfg->dev->hub_port != -1;
}
