#include <STDLIB.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/THREAD.H>
#include <SPAD/WQ.H>
#include <SPAD/SYNC.H>

#include <SPAD/USBHUB.H>

#include "USB.H"

#define INTR_DATA_SIZE		((USB_HUB_MAXPORTS + 1 + 7) / 8)

typedef struct __port PORT;
typedef struct __hub HUB;

struct __port {
	char active;
	USB_RESET_RQ *reset;
	THREAD_RQ thr;
	__u32 chg_status;
	HUB *hub;
};

#define PORT_FREE			0
#define PORT_INITIALIZING		1
#define PORT_CANCELING_INITIALIZATION	2
#define PORT_ACTIVE			3
#define PORT_ACTIVE_RESETTING		4
#define PORT_CANCELING_RESETTING	5

struct __hub {
	USB_DEV_INTERFACE *iface;
	int n_posted;
	int error;
	USB_ENDPOINT *ctrl;
	USB_ENDPOINT *intr;
	MTX ctrl_mtx;
	USB_CTRL_REQUEST crq;
	USB_REQUEST irq;
	USB_DESCRIPTOR_HUB hub_desc;
	__u8 intr_data[INTR_DATA_SIZE];
	unsigned port_n; /* currently used port, this is not number of ports */
	__u32 status;
	__u32 c_status;
	PORT ports[1];
};

static __finline__ unsigned PORT_N(PORT *port)
{
	return port - port->hub->ports;
}

static __finline__ unsigned INTR_REQ_BYTES(HUB *hub)
{
	return (hub->hub_desc.bNbrPorts + 1 + 7) >> 3;
}

static void FREE_HUB(HUB *hub)
{
	int i;
	for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
		if (__likely(hub->ports[i].thr.thread != NULL)) KERNEL$FREE_THREAD(hub->ports[i].thr.thread);
	}
	free(hub);
}

int USB$HUB_TEST(USB_DEV_INTERFACE *iface, long tt)
{
	USB_DESCRIPTOR_INTERFACE *desc = USB$GET_INTERFACE_DESCRIPTOR(iface);
	if (desc->bInterfaceClass != USB_CLASS_HUB) return USB_SKIP;
	if (desc->bInterfaceSubClass != USB_HUB_SUBCLASS &&
	    desc->bInterfaceSubClass != USB_HUB_SUBCLASS_NONSTD) return USB_SKIP;
	if (desc->bInterfaceProtocol != tt) return USB_SKIP;
	return USB_ATTACH_CLASS;
}

extern AST_STUB HUB_IRQ_AST;

int USB$HUB_ATTACH(USB_DEV_INTERFACE *iface)
{
	unsigned i;
	int r;
	USB_DESCRIPTOR_HUB hub_desc;
	HUB *hub;
	if (__unlikely((r = USB$REQUEST_DESCRIPTOR(iface, (__u8)USBHUB_REQ_GET_HUB_DESCRIPTOR, USB_DT_HUB, 0, 0, &hub_desc, sizeof hub_desc, USB_DESCRIPTOR_HUB_MIN_SIZE, 0)) < 0)) goto ret0;

	if (__unlikely(!hub_desc.bNbrPorts)) {
		if (iface->cfg->dev->usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "HUB %s HAS NO PORT", USB_DEV_ID(iface->cfg->dev));
		r = -EPROTO;
		goto ret0;
	}
	if (__unlikely(hub_desc.bNbrPorts + 1 > USB_HUB_MAXPORTS + 1)) { /* +1 is there to kill warning */
		if (iface->cfg->dev->usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "HUB %s HAS TOO MANY PORTS (%d)", USB_DEV_ID(iface->cfg->dev), hub_desc.bNbrPorts);
		r = -EPROTONOSUPPORT;
		goto ret0;
	}

	/* !!! FIXME: disallow two consecutive bus-powered hubs */

	hub = __sync_malloc(sizeof(HUB) + hub_desc.bNbrPorts * sizeof(PORT));
	if (__unlikely(!hub)) {
		r = -ENOMEM;
		goto ret0;
	}
	memset(hub, 0, sizeof(HUB) + hub_desc.bNbrPorts * sizeof(PORT));
	iface->hub = hub;
	hub->iface = iface;
	MTX_INIT(&hub->ctrl_mtx, "USB$HUB_CTRL_MTX");
	memcpy(&hub->hub_desc, &hub_desc, sizeof hub_desc);
	for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
		hub->ports[i].hub = hub;
		hub->ports[i].active = PORT_FREE;
		if (__unlikely(__IS_ERR(hub->ports[i].thr.thread = KERNEL$ALLOC_THREAD_SYNC()))) {
			r = __PTR_ERR(hub->ports[i].thr.thread);
			hub->ports[i].thr.thread = NULL;
			goto ret1;
		}
	}
	hub->ctrl = USB$GET_DEFAULT_ENDPOINT(iface);
	hub->intr = USB$FIND_ENDPOINT(iface, USB_EP_INTR_IN, 1, INTR_DATA_SIZE);
	if (__unlikely(__IS_ERR(hub->intr))) {
		r = __PTR_ERR(hub->intr);
		hub->intr = NULL;
		goto ret1;
	}

	if ((hub->hub_desc.wHubCharacteristics & USB_DESCRIPTOR_HUB_CHAR_POWER_MASK) == USB_DESCRIPTOR_HUB_CHAR_POWER_GANGED || (hub->hub_desc.wHubCharacteristics & USB_DESCRIPTOR_HUB_CHAR_POWER_MASK) == USB_DESCRIPTOR_HUB_CHAR_POWER_INDIVIDUAL) {
		for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
			USB$SYNC_CTRL(hub->ctrl, USBHUB_REQ_SET_PORT_FEATURE, USB_PORT_FEAT_POWER, i, 0, NULL, 0, USB_HUB_TIMEOUT, USB_HUB_RETRIES, i == 1 ? iface : NULL);
		}
	}

	if (__unlikely(r = KERNEL$SLEEP_CANCELABLE(hub->hub_desc.bPwrOn2PwrGood * JIFFIES_PER_SECOND / 500 + 1))) goto ret1;

	memset(&hub->intr_data, 0xff, INTR_DATA_SIZE);
	hub->n_posted++;
	hub->irq.fn = HUB_IRQ_AST;
	hub->irq.status = INTR_REQ_BYTES(hub);
	CALL_AST(&hub->irq);

	return 0;

	ret1:
	FREE_HUB(hub);
	ret0:
	USB$DETACH_DRIVER(iface);
	return r;
}

static void PROCESS_PORT_CHANGE(HUB *hub, unsigned i);
extern AST_STUB GOT_PORT_STATUS;
static void PROCESS_HUB_CHANGE(HUB *hub);
extern AST_STUB GOT_HUB_STATUS;
extern AST_STUB HUB_THREAD_AST;
static long HUB_THREAD_MAIN(void *p_);
static void KICK_DEVICE(HUB *hub, int port);

static WQ_DECL(HUB_PORT_RESET_DONE, "USB$HUB_PORT_RESET_DONE");

DECL_AST(HUB_IRQ_AST, SPL_USB, USB_REQUEST)
{
	unsigned i;
	HUB *hub = GET_STRUCT(RQ, HUB, irq);
	hub->n_posted--;
	if (__unlikely(hub->n_posted < 0))
		KERNEL$SUICIDE("HUB_IRQ_AST: N_POSTED UNDERFLOW (%d)", hub->n_posted);
	if (__unlikely(hub->irq.status < 0) || __unlikely(hub->error)) {
		HUB_ERROR(hub, hub->irq.status);
		RETURN;
	}
	for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) if (hub->intr_data[i >> 3] & (1 << (i & 7))) {
		hub->intr_data[i >> 3] &= ~(1 << (i & 7));
		PROCESS_PORT_CHANGE(hub, i);
		RETURN;
	}
	if (hub->intr_data[0] & 1) {
		hub->intr_data[0] &= ~1;
		PROCESS_HUB_CHANGE(hub);
		RETURN;
	}
	hub->n_posted++;
	hub->irq.fn = HUB_IRQ_AST;
	hub->irq.flags = 0;
	hub->irq.v.ptr = (unsigned long)&hub->intr_data;
	hub->irq.v.len = INTR_REQ_BYTES(hub);
	hub->irq.v.vspace = &KERNEL$VIRTUAL;
	hub->intr->prepare(hub->intr, &hub->irq);
	hub->intr->post(&hub->irq);
	RETURN;
}

static void PROCESS_PORT_CHANGE(HUB *hub, unsigned i)
{
	hub->port_n = i;
	hub->c_status = 0;
	hub->n_posted++;
	hub->crq.fn = GOT_PORT_STATUS;
	hub->crq.flags = 0;
	hub->crq.v.ptr = (unsigned long)&hub->status;
	hub->crq.v.len = sizeof hub->status;
	hub->crq.v.vspace = &KERNEL$VIRTUAL;
	hub->crq.setup.request = __16CPU2LE(USBHUB_REQ_GET_PORT_STATUS);
	hub->crq.setup.w_value = __16CPU2LE(0);
	hub->crq.setup.w_index = __16CPU2LE(i);
	hub->crq.setup.w_length = __16CPU2LE(sizeof hub->status);
	hub->ctrl->prepare(hub->ctrl, (USB_REQUEST *)&hub->crq);
	hub->ctrl->post((USB_REQUEST *)&hub->crq);
}

DECL_AST(GOT_PORT_STATUS, SPL_USB, USB_CTRL_REQUEST)
{
	unsigned clr;
	__u32 port_status;
	HUB *hub = GET_STRUCT(RQ, HUB, crq);
	PORT *port = &hub->ports[hub->port_n];
	hub->n_posted--;
	if (__unlikely(hub->n_posted < 0))
		KERNEL$SUICIDE("GOT_PORT_STATUS: N_POSTED UNDERFLOW (%d)", hub->n_posted);
	if (__unlikely(hub->crq.status < 0) || __unlikely(hub->error)) {
		HUB_ERROR(hub, hub->crq.status);
		RETURN;
	}
	port_status = __32LE2CPU(hub->status);
	port->chg_status |= port_status;
	hub->c_status |= hub->status;
	if (port->active == PORT_ACTIVE || port->active == PORT_ACTIVE_RESETTING) {
		if (__unlikely((port_status & (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_OVER_CURRENT | USB_PORT_STAT_C_CONNECTION | (port->active != PORT_ACTIVE_RESETTING ? USB_PORT_STAT_C_ENABLE : 0) | USB_PORT_STAT_C_OVER_CURRENT)) != USB_PORT_STAT_CONNECTION)) {
			if (port_status & USB_PORT_STAT_OVER_CURRENT) {
				if (hub->iface->cfg->dev->usb->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hub->iface->cfg->dev->usb->dev_name, "OVERCURRENT ON HUB %s, PORT %u", USB_DEV_ID(hub->iface->cfg->dev), hub->port_n);
			} else if ((port_status & (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_C_CONNECTION | USB_PORT_STAT_ENABLE | USB_PORT_STAT_C_ENABLE | USB_PORT_FEAT_RESET | USB_PORT_FEAT_C_RESET)) == (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_C_ENABLE)) {
				if (hub->iface->cfg->dev->usb->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hub->iface->cfg->dev->usb->dev_name, "PORT ERROR ON HUB %s, PORT %u: STATUS %08X", USB_DEV_ID(hub->iface->cfg->dev), hub->port_n, port_status);
			}
			KICK_DEVICE(hub, hub->port_n);
		}
	}
	if (port_status & USB_PORT_STAT_C_CONNECTION) clr = USB_PORT_FEAT_C_CONNECTION;
	else if (port_status & USB_PORT_STAT_C_ENABLE) clr = USB_PORT_FEAT_C_ENABLE;
	else if (port_status & USB_PORT_STAT_C_SUSPEND) clr = USB_PORT_FEAT_C_SUSPEND;
	else if (port_status & USB_PORT_STAT_C_OVER_CURRENT) clr = USB_PORT_FEAT_C_OVER_CURRENT;
	else if (port_status & USB_PORT_STAT_C_RESET) clr = USB_PORT_FEAT_C_RESET;
	else goto skip_clear;
	hub->status &= __32CPU2LE(~(1 << clr));
	hub->n_posted++;
	hub->crq.fn = GOT_PORT_STATUS;
	hub->crq.flags = 0;
	hub->crq.v.ptr = 0;
	hub->crq.v.len = 0;
	hub->crq.v.vspace = &KERNEL$VIRTUAL;
	hub->crq.setup.request = __16CPU2LE(USBHUB_REQ_CLEAR_PORT_FEATURE);
	hub->crq.setup.w_value = __16CPU2LE(clr);
	hub->crq.setup.w_index = __16CPU2LE(hub->port_n);
	hub->crq.setup.w_length = __16CPU2LE(0);
	hub->ctrl->prepare(hub->ctrl, (USB_REQUEST *)&hub->crq);
	hub->ctrl->post((USB_REQUEST *)&hub->crq);
	RETURN;
	skip_clear:
	port_status = __32LE2CPU(hub->c_status);
	if (port->active == PORT_FREE) {
		if ((port_status & (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_OVER_CURRENT | USB_PORT_STAT_C_CONNECTION)) == (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_C_CONNECTION)) {
			hub->n_posted++;
			port->active = PORT_INITIALIZING;
			port->thr.fn = HUB_THREAD_AST;
			port->thr.thread_main = HUB_THREAD_MAIN;
			port->thr.p = port;
			port->thr.error = NULL;
			port->thr.cwd = NULL;
			port->thr.std_in = -1;
			port->thr.std_out = -1;
			port->thr.std_err = -1;
			port->thr.dlrq = NULL;
			port->thr.spawned = 0;
			/* port->thr.thread already set */
			CALL_IORQ(&port->thr, KERNEL$THREAD);
		}
	}
	hub->n_posted++;
	RETURN_AST(&hub->irq);
}

static void PROCESS_HUB_CHANGE(HUB *hub)
{
	hub->c_status = 0;
	hub->n_posted++;
	hub->crq.fn = GOT_HUB_STATUS;
	hub->crq.flags = 0;
	hub->crq.v.ptr = (unsigned long)&hub->status;
	hub->crq.v.len = sizeof hub->status;
	hub->crq.v.vspace = &KERNEL$VIRTUAL;
	hub->crq.setup.request = __16CPU2LE(USBHUB_REQ_GET_HUB_STATUS);
	hub->crq.setup.w_value = __16CPU2LE(0);
	hub->crq.setup.w_index = __16CPU2LE(0);
	hub->crq.setup.w_length = __16CPU2LE(sizeof hub->status);
	hub->ctrl->prepare(hub->ctrl, (USB_REQUEST *)&hub->crq);
	hub->ctrl->post((USB_REQUEST *)&hub->crq);
}

DECL_AST(GOT_HUB_STATUS, SPL_USB, USB_CTRL_REQUEST)
{
	unsigned clr;
	__u32 hub_status;
	HUB *hub = GET_STRUCT(RQ, HUB, crq);
	hub->n_posted--;
	if (__unlikely(hub->n_posted < 0))
		KERNEL$SUICIDE("GOT_HUB_STATUS: N_POSTED UNDERFLOW (%d)", hub->n_posted);
	if (__unlikely(hub->crq.status < 0) || __unlikely(hub->error)) {
		HUB_ERROR(hub, hub->crq.status);
		RETURN;
	}
	hub_status = __32LE2CPU(hub->status);
	hub->c_status |= hub->status;
	if (__unlikely(hub_status & (USB_HUB_STAT_OVER_CURRENT | USB_HUB_STAT_C_OVER_CURRENT))) {
		int msg = 0;
		unsigned i;
		for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
			if (hub->ports[i].active != PORT_FREE && hub->ports[i].active != PORT_CANCELING_INITIALIZATION) {
				if (!msg) {
					if (hub->iface->cfg->dev->usb->errorlevel >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, hub->iface->cfg->dev->usb->dev_name, "OVERCURRENT ON HUB %s", USB_DEV_ID(hub->iface->cfg->dev));
					msg = 1;
				}
			}
			KICK_DEVICE(hub, i);
		}
	}
	if (hub_status & USB_HUB_STAT_C_LOCAL_POWER) clr = USB_HUB_FEAT_LOCAL_POWER;
	else if (hub_status & USB_HUB_STAT_C_OVER_CURRENT) clr = USB_HUB_FEAT_C_OVER_CURRENT;
	else goto skip_clear;
	hub->status &= __32CPU2LE(~(1 << clr));
	hub->n_posted++;
	hub->crq.fn = GOT_HUB_STATUS;
	hub->crq.flags = 0;
	hub->crq.v.ptr = 0;
	hub->crq.v.len = 0;
	hub->crq.v.vspace = &KERNEL$VIRTUAL;
	hub->crq.setup.request = __16CPU2LE(USBHUB_REQ_CLEAR_HUB_FEATURE);
	hub->crq.setup.w_value = __16CPU2LE(clr);
	hub->crq.setup.w_index = __16CPU2LE(0);
	hub->crq.setup.w_length = __16CPU2LE(0);
	hub->ctrl->prepare(hub->ctrl, (USB_REQUEST *)&hub->crq);
	hub->ctrl->post((USB_REQUEST *)&hub->crq);
	RETURN;
	skip_clear:
	hub_status = __32LE2CPU(hub->c_status);
	hub->n_posted++;
	RETURN_AST(&hub->irq);
}

DECL_AST(HUB_THREAD_AST, SPL_USB, THREAD_RQ)
{
	PORT *port = GET_STRUCT(RQ, PORT, thr);
	HUB *hub = port->hub;
	hub->n_posted--;
	if (__unlikely(hub->n_posted < 0))
		KERNEL$SUICIDE("HUB_THREAD_AST: N_POSTED UNDERFLOW (%d)", hub->n_posted);
	if (__unlikely(port->active == PORT_CANCELING_INITIALIZATION)) {	/* someone attempted to kick us while we were initializing */
		port->active = PORT_FREE;
		KICK_DEVICE(hub, PORT_N(port));
	} else {
		if (__unlikely(port->active != PORT_INITIALIZING))
			KERNEL$SUICIDE("HUB_THREAD_AST: PORT ACTIVE %d", (int)port->active);
		port->active = !port->thr.status ? PORT_ACTIVE : PORT_FREE;
	}
	if ((__unlikely(port->thr.status < 0) && port->thr.status != -EINTR) || __unlikely(hub->error)) {
		HUB_ERROR(hub, port->thr.status);
	}
	RETURN;
}

static USB_DEV *FIND_DEV_FOR_PORT(USB *usb, int hub_addr, int hub_port)
{
	unsigned i;
	for (i = 1; i < USB_MAX_ADDRESSES; i++) {
		USB_DEV *dev = usb->devices[i];
		if (dev && dev->hub_addr == hub_addr && dev->hub_port == hub_port) return dev;
	}
	return NULL;
}

static PORT *FIND_PORT_FOR_DEV(USB_DEV *dev)
{
	USB_DEV *hub_dev;
	USB_DEV_CFG *hub_cfg;
	HUB *hub;
	unsigned i;
	if (__unlikely(dev->hub_port == -1)) return NULL;
	hub_dev = dev->hub_addr < 0 ? &dev->usb->root_hub : dev->usb->devices[dev->hub_addr];
	if (__unlikely(!hub_dev) || __unlikely(!hub_dev->n_attached)) return NULL;
	hub_cfg = &hub_dev->confs[hub_dev->active_cfg];
	for (i = 0; i < hub_cfg->n_ifaces; i++) if (__likely(hub_cfg->ifaces[i].hub != NULL)) {
		hub = hub_cfg->ifaces[i].hub;
		goto found_hub;
	}
	return NULL;
	found_hub:
	if (__unlikely((unsigned)dev->hub_port > hub->hub_desc.bNbrPorts)) return NULL;
	return &hub->ports[dev->hub_port];
}

static int PORT_STATUS_SYNC(PORT *port, __u32 *status)
{
	int r;
	HUB *hub = port->hub;
	if (__unlikely(r = MTX_LOCK_SYNC_CANCELABLE(&hub->ctrl_mtx))) return r;
	r = USB$SYNC_CTRL(hub->ctrl, USBHUB_REQ_GET_PORT_STATUS, 0, PORT_N(port), 4, status, 0, USB_HUB_TIMEOUT, USB_HUB_RETRIES, NULL);
	MTX_UNLOCK(&hub->ctrl_mtx);
	if (__likely(r >= 0)) r = 0, *status = __32LE2CPU(*status);
	return r;
}

static int STATUS_SPEED(PORT *port, __u32 status)
{
	return status & USB_PORT_STAT_LOWSPEED ? USB_LOW_SPEED : status & USB_PORT_STAT_HIGHSPEED && port->hub->iface->cfg->dev->speed == USB_HIGH_SPEED ? USB_HIGH_SPEED : USB_FULL_SPEED;
}

static int PORT_RESET_SYNC(PORT *port, __u32 *status)
{
	int r;
	unsigned wait_cnt, reset_cnt = 0;
	u_jiffies_lo_t reset_wait = *status & USB_PORT_STAT_LOWSPEED ? USB_HUB_LONG_RESET_WAIT : port->hub->iface->cfg->dev == &port->hub->iface->cfg->dev->usb->root_hub ? USB_HUB_ROOT_RESET_WAIT : USB_HUB_SHORT_RESET_WAIT;
	reset_again:
	if (__unlikely(r = MTX_LOCK_SYNC_CANCELABLE(&port->hub->ctrl_mtx))) return r;
	r = USB$SYNC_CTRL(port->hub->ctrl, USBHUB_REQ_SET_PORT_FEATURE, USB_PORT_FEAT_RESET, PORT_N(port), 0, NULL, 0, USB_HUB_TIMEOUT, USB_HUB_RETRIES, NULL);
	MTX_UNLOCK(&port->hub->ctrl_mtx);
	if (__unlikely(r < 0)) {
		return r;
	}
	wait_cnt = 0;
	wait_again:
	if (__unlikely(r = KERNEL$SLEEP_CANCELABLE(reset_wait))) return r;
	if (__unlikely(r = PORT_STATUS_SYNC(port, status))) return r;
	if (__unlikely((*status & (USB_PORT_STAT_CONNECTION | USB_PORT_STAT_C_CONNECTION)) != USB_PORT_STAT_CONNECTION)) return -ECONNRESET;
	if (__unlikely(port->chg_status & USB_PORT_STAT_C_CONNECTION)) return -ECONNRESET;
	if (__unlikely((*status & (USB_PORT_STAT_ENABLE | USB_PORT_STAT_RESET)) != USB_PORT_STAT_ENABLE)) {
		wait_cnt++;
		if (wait_cnt == 2) reset_wait = USB_HUB_LONG_RESET_WAIT;
		if (wait_cnt < USB_HUB_RESET_N_WAIT) goto wait_again;
		reset_cnt++;
		if (reset_cnt < USB_HUB_RESET_N_TRIES) goto reset_again;
		if (FIND_DEV_FOR_PORT(port->hub->iface->cfg->dev->usb, port->hub->iface->cfg->dev->addr, PORT_N(port))) {
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, port->hub->iface->cfg->dev->usb->dev_name, "FAILED TO RESET HUB %s, PORT %u", USB_DEV_ID(port->hub->iface->cfg->dev), PORT_N(port));
		}
		return -EIO;
	}
	return 0;
}

static void PORT_DISABLE_SYNC(PORT *port)
{
	if (__unlikely(MTX_LOCK_SYNC_CANCELABLE(&port->hub->ctrl_mtx))) return;
	USB$SYNC_CTRL(port->hub->ctrl, USBHUB_REQ_CLEAR_PORT_FEATURE, USB_PORT_FEAT_ENABLE, PORT_N(port), 0, NULL, 0, USB_HUB_TIMEOUT, USB_HUB_RETRIES, NULL);
	MTX_UNLOCK(&port->hub->ctrl_mtx);
}

/*
 * *** WARNING ***
 * There must not be any dynamic memory allocations while holding addr0lock
 * Otherwise VM will deadlock while attempting to reset mass storage device
 */

static long HUB_THREAD_MAIN(void *p_)
{
	PORT *port = p_;
	USB_DEV *dev;
	int a0locked = 0;
	int r;
	int addr;
	__u32 status;
	u_jiffies_lo_t start, stable, j;
	/*__debug_printf("attach..\n");*/
	RAISE_SPL(SPL_USB);
	port->chg_status = ~0;
	j = start = stable = KERNEL$GET_JIFFIES_LO();
	while (j - stable <= USB_HUB_DEBOUNCE_STABLE) {
		if (j - start > USB_HUB_DEBOUNCE_TIMEOUT) {
			if (port->hub->iface->cfg->dev->usb->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_NET_WARNING, port->hub->iface->cfg->dev->usb->dev_name, "CONNECTION UNSTABLE ON HUB %s, PORT %u", USB_DEV_ID(port->hub->iface->cfg->dev), PORT_N(port));
			goto ret_1;
		}
		if (__unlikely(KERNEL$SLEEP_CANCELABLE(USB_HUB_DEBOUNCE_STEP))) goto ret_1;
		j = KERNEL$GET_JIFFIES_LO();
		if (__unlikely(PORT_STATUS_SYNC(port, &status))) goto ret_1;
		if (port->chg_status & USB_PORT_STAT_C_CONNECTION) port->chg_status = 0, stable = j;
		if (__unlikely(!(status & USB_PORT_STAT_CONNECTION))) stable = j;
		if (__unlikely(status & USB_PORT_STAT_C_CONNECTION)) {
			if (__unlikely(MTX_LOCK_SYNC_CANCELABLE(&port->hub->ctrl_mtx))) goto ret_1;
			USB$SYNC_CTRL(port->hub->ctrl, USBHUB_REQ_CLEAR_PORT_FEATURE, USB_PORT_FEAT_C_CONNECTION, PORT_N(port), 0, NULL, 0, USB_HUB_TIMEOUT, USB_HUB_RETRIES, NULL);
			MTX_UNLOCK(&port->hub->ctrl_mtx);
			stable = j;
		}
	}
	dev = __sync_malloc(sizeof(USB_DEV));
	if (__unlikely(!dev)) {
		if (port->hub->iface->cfg->dev->usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, port->hub->iface->cfg->dev->usb->dev_name, "NO MEMORY FOR DEVICE");
		goto ret_1;
	}
	memset(dev, 0, sizeof(USB_DEV));
	dev->usb = port->hub->iface->cfg->dev->usb;
	dev->active_cfg = -1;
	dev->hub_addr = port->hub->iface->cfg->dev->addr;
	dev->hub_port = PORT_N(port);
	dev->n_attached = -1;
	INIT_LIST(&dev->unplug);
	addr = USB_FIND_ADDRESS(dev->usb);
	if (__unlikely(addr < 0)) {
		if (dev->usb->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, dev->usb->dev_name, "NO ADDRESS AVAILABLE FOR DEVICE");
		__slow_free(dev);
		goto ret_1;
	}
	dev->addr = addr;
	dev->usb->devices[addr] = dev;

	if (__unlikely(MTX_LOCK_SYNC_CANCELABLE(&dev->usb->addr0lock))) goto ret_2;
	a0locked = 1;

	if (__unlikely(PORT_RESET_SYNC(port, &status))) goto ret_2;

	dev->speed = STATUS_SPEED(port, status);
	if (__unlikely(r = USB_INIT_DEVICE(dev, &a0locked))) {
		if (r != -EINTR && r != -ECONNRESET && dev->hub_port != -1) {
			if (dev->usb->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_NET_WARNING, dev->usb->dev_name, "FAILED TO INITIALIZE DEVICE: %s", strerror(-r));
		}
		goto ret_2;
	}
	if (__unlikely(a0locked))
		KERNEL$SUICIDE("HUB_THREAD_MAIN: USB_INIT_DEVICE RETURNED SUCCESS BUT DIDN'T UNLOCK ADDRESS 0");
	if (__unlikely(dev->n_attached != -1))
		KERNEL$SUICIDE("HUB_THREAD_MAIN: SOMEBODY UNLOCKED DEVICE WHILE INITIALIZING");
	if (__unlikely(dev->hub_port == -1)) goto ret_2;
	dev->n_attached = 0;
	if (__likely(!dev->usb->shutting_down)) USBPLUG_EVENT(dev);
	/*__debug_printf("success..\n");*/
	LOWER_SPL(SPL_ZERO);
	return 0;
	ret_2:
	PORT_DISABLE_SYNC(port);
	if (__unlikely(dev->n_attached != -1))
		KERNEL$SUICIDE("HUB_THREAD_MAIN: SOMEBODY UNLOCKED DEVICE WHILE INITIALIZING (FAILURE)");
	dev->n_attached = 0;
	if (a0locked) MTX_UNLOCK(&dev->usb->addr0lock);
	USB_FREE_DEVICE(dev);
	ret_1:
	LOWER_SPL(SPL_ZERO);
	return 1;
}

/* !!! FIXME: recursion sux */

static void KICK_DEVICE(HUB *hub, int port)
{
	unsigned i, j;
	if (__likely(hub->ports[port].active == PORT_ACTIVE)) hub->ports[port].active = PORT_FREE;
	if (__likely(hub->ports[port].active == PORT_ACTIVE_RESETTING)) KERNEL$CIO((IORQ *)&hub->ports[port].thr), hub->ports[port].active = PORT_CANCELING_RESETTING;
	if (__likely(hub->ports[port].active == PORT_INITIALIZING)) KERNEL$CIO((IORQ *)&hub->ports[port].thr), hub->ports[port].active = PORT_CANCELING_INITIALIZATION;
	for (i = 1; i < USB_MAX_ADDRESSES; i++) {
		USB_DEV *dev;
		j = 0;
		test_again:
		dev = hub->iface->cfg->dev->usb->devices[i];
		if (dev && dev->hub_port != -1 && (
		    (dev->hub_addr == hub->iface->cfg->dev->addr && dev->hub_port == port) ||
		    (dev->hub_addr > 0 && (!dev->usb->devices[dev->hub_addr] || dev->usb->devices[dev->hub_addr]->hub_port == -1))
		    )) {
			if (dev->active_cfg != -1) {
				USB_DEV_CFG *cfg = &dev->confs[dev->active_cfg];
				for (; j < cfg->n_ifaces; j++) if (cfg->ifaces[j].attached && cfg->ifaces[j].hub) {
					/* this may deallocate the device */
					HUB_ERROR(cfg->ifaces[j].hub, -ECONNRESET);
					j++;
					goto test_again;
				}
			}
			/*__debug_printf("kick...\n");*/
			USB_FREE_DEVICE(dev);
		}
	}
}

void HUB_ERROR(void *hub_, int error)
{
	HUB *hub = hub_;
	unsigned i;
	if (!hub->error) hub->error = error;
	for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
		KICK_DEVICE(hub, i);
	}
	if (hub->n_posted) {
		unsigned i;
		for (i = 1; i <= hub->hub_desc.bNbrPorts; i++) {
			PORT *port = &hub->ports[i];
			if (port->active == PORT_INITIALIZING) KERNEL$CIO((IORQ *)&port->thr), port->active = PORT_CANCELING_INITIALIZATION;
			if (port->active == PORT_ACTIVE_RESETTING) KERNEL$CIO((IORQ *)&port->thr), port->active = PORT_CANCELING_RESETTING;
		}
		hub->ctrl->cancel((USB_REQUEST *)&hub->crq);
		hub->intr->cancel(&hub->irq);
		return;
	}
	USB$DETACH_DRIVER(hub->iface);
	/* !!! FIXME: disable port on upstream hub */
	FREE_HUB(hub);
}

extern AST_STUB HUB_RESET_THREAD_AST;
static long HUB_RESET_THREAD_MAIN(void *p_);

DECL_IOCALL(USB$RESET_PORT, SPL_USB, USB_RESET_RQ)
{
	USB_DEV *dev = RQ->iface->cfg->dev;
	PORT *port;
	if (__unlikely(dev->n_attached >= 2)) {
		RQ->status = -EBUSY;
		RETURN_AST(RQ);
	}
	port = FIND_PORT_FOR_DEV(dev);
	if (__unlikely(!port)) {
		reset:
		RQ->status = -ECONNRESET;
		RETURN_AST(RQ);
	}
	if (__unlikely(port->active != PORT_ACTIVE)) {
		if (port->active == PORT_ACTIVE_RESETTING) {
			RQ->status = -EALREADY;
			RETURN_AST(RQ);
		}
		goto reset;
	}
	port->hub->n_posted++;
	port->active = PORT_ACTIVE_RESETTING;
	port->reset = RQ;
	port->thr.fn = HUB_RESET_THREAD_AST;
	port->thr.thread_main = HUB_RESET_THREAD_MAIN;
	port->thr.p = port;
	port->thr.error = NULL;
	port->thr.cwd = NULL;
	port->thr.std_in = -1;
	port->thr.std_out = -1;
	port->thr.std_err = -1;
	port->thr.dlrq = NULL;
	port->thr.spawned = 0;
	/* port->thr.thread already set */
	RETURN_IORQ_CANCELABLE(&port->thr, KERNEL$THREAD, RQ);
}

DECL_AST(HUB_RESET_THREAD_AST, SPL_USB, THREAD_RQ)
{
	PORT *port = GET_STRUCT(RQ, PORT, thr);
	HUB *hub = port->hub;
	USB_RESET_RQ *rq;
	hub->n_posted--;
	if (__unlikely(hub->n_posted < 0))
		KERNEL$SUICIDE("HUB_RESET_THREAD_AST: N_POSTED UNDERFLOW (%d)", hub->n_posted);
	WQ_WAKE_ALL(&HUB_PORT_RESET_DONE);
	if (__unlikely(port->active == PORT_CANCELING_RESETTING)) {
		port->active = PORT_FREE;
		KICK_DEVICE(hub, PORT_N(port));
	} else {
		if (__unlikely(port->active != PORT_ACTIVE_RESETTING))
			KERNEL$SUICIDE("HUB_RESET_THREAD_AST: PORT ACTIVE %d", (int)port->active);
		port->active = PORT_ACTIVE;
	}
	rq = port->reset;
	rq->status = port->thr.status;
	if (__unlikely(hub->error)) {
		HUB_ERROR(hub, port->thr.status);
	}
	RETURN_AST(rq);
}

static long HUB_RESET_THREAD_MAIN(void *p_)
{
	int a0locked = 0;
	PORT *port = p_;
	__u32 status;
	int r;
	USB_DEV *dev;
	RAISE_SPL(SPL_USB);
	if (__unlikely(r = PORT_STATUS_SYNC(port, &status))) goto ret0;
	dev = FIND_DEV_FOR_PORT(port->hub->iface->cfg->dev->usb, port->hub->iface->cfg->dev->addr, PORT_N(port));
	if (__likely(dev != NULL)) {
		if (__unlikely(dev->speed != STATUS_SPEED(port, status))) {
			if (port->hub->iface->cfg->dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, port->hub->iface->cfg->dev->usb->dev_name, "DEVICE %s SPEED DIFFERENT, SHOULD BE %s, PORT REPORTS %s", USB_DEV_ID(dev), USB$SPEED_STRING(dev->speed), USB$SPEED_STRING(STATUS_SPEED(port, status)));
			r = -EIO;
			goto ret0;
		}
		if (port->hub->iface->cfg->dev->usb->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, port->hub->iface->cfg->dev->usb->dev_name, "RESETTING DEVICE %s", USB_DEV_ID(dev));
	}
	if (__unlikely(r = MTX_LOCK_SYNC_CANCELABLE(&port->hub->iface->cfg->dev->usb->addr0lock))) goto ret0;
	a0locked = 1;
	port->chg_status = 0;
	if (__unlikely(r = PORT_RESET_SYNC(port, &status))) goto ret1;
	dev = FIND_DEV_FOR_PORT(port->hub->iface->cfg->dev->usb, port->hub->iface->cfg->dev->addr, PORT_N(port));
	if (__likely(dev != NULL)) {
		if (__unlikely(dev->speed != STATUS_SPEED(port, status))) {
			if (port->hub->iface->cfg->dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, port->hub->iface->cfg->dev->usb->dev_name, "DEVICE %s SPEED CHANGED DURING RESET FROM %s TO %s", USB_DEV_ID(dev), USB$SPEED_STRING(dev->speed), USB$SPEED_STRING(STATUS_SPEED(port, status)));
			r = -EIO;
			goto ret1;
		}
		if (__unlikely(!dev->n_attached))
			KERNEL$SUICIDE("HUB_RESET_THREAD_MAIN: DRIVER DETACHED WHILE PERFORMING RESET");
		r = USB_REINIT_DEVICE(dev, &a0locked);
		if (__unlikely(r)) goto ret1;
		if (__unlikely(a0locked))
			KERNEL$SUICIDE("HUB_RESET_THREAD_MAIN: USB_REINIT_DEVICE RETURNED SUCCESS BUT DIDN'T UNLOCK ADDRESS 0");
	}
	if (__unlikely(a0locked)) MTX_UNLOCK(&port->hub->iface->cfg->dev->usb->addr0lock);
	LOWER_SPL(SPL_ZERO);
	KERNEL$SLEEP(port->reset->wait_time);
	return 0;
	ret1:
	PORT_DISABLE_SYNC(port);
	if (a0locked) MTX_UNLOCK(&port->hub->iface->cfg->dev->usb->addr0lock);
	KICK_DEVICE(port->hub, PORT_N(port));
	ret0:
	LOWER_SPL(SPL_ZERO);
	return r;
}

int USB_HUB_WAIT_FOR_RESET(USB_DEV *dev)
{
	PORT *port;
	int r;
	if (__unlikely(dev == &dev->usb->root_hub)) return 0;
	test_again:
	port = FIND_PORT_FOR_DEV(dev);
	if (__unlikely(!port)) return -ECONNRESET;
	if (__likely(port->active != PORT_ACTIVE_RESETTING) && __likely(port->active != PORT_CANCELING_RESETTING)) return 0;
	if (__unlikely(r = WQ_WAIT_SYNC_CANCELABLE(&HUB_PORT_RESET_DONE))) return r;
	goto test_again;
}

