#ifndef __SPAD_DEV_KRNL_H
#define __SPAD_DEV_KRNL_H

#include <SPAD/AC.H>
#include <SPAD/PROC.H>
#include <SPAD/DEV.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <ARCH/PAGE.H>
#include <SYS/TYPES.H>

__BEGIN_DECLS

/* needed for dmalock and dmaunlock */
int KERNEL$RESERVE_BOUNCE(char *device, unsigned requests, unsigned long size);
void KERNEL$UNRESERVE_BOUNCE(char *device, unsigned requests, unsigned long size);

typedef struct __ffile FFILE;
typedef struct __fblob_ref FBLOB_REF;
typedef struct __fpipe FPIPE;
typedef struct __handle_operations HANDLE_OPERATIONS;

extern __const__ HANDLE_OPERATIONS KERNEL$HANDLE_NOOP;

/* VSPACE */

#ifdef VBUF
#undef VBUF
#endif

typedef struct {
	void *ptr;
	unsigned long len;
	unsigned spl;
} VBUF;

typedef struct {
	__u32 ptr;
	__u32 len;
} VDMA;

typedef struct {
	__u64 ptr;
	__u32 len;
} VDMA64;

typedef struct {
	__p_addr ptr;
	__u32 len;
} VPHYS;

#define PF_WRITE	1	/* write fault, do not change, must be 1 */
#define PF_READ		2	/* read from vspace to device, must be 2 */
#define PF_RW		(PF_READ | PF_WRITE)
#define PF_PAGEIO	4	/* passed to get_pagein_rq, if request goes from
				   paging (get_page) rather than vspace
				   operations. Driver supporting paging but not
				   supporting vspace operations should return
				   error, if this flag is not present */
#define PF_SWAPPER	8	/* must go to swapper (guaranteed response time)
				   - avoid hang when nonresponding pager */
#define PF_TESTPAGE	16	/* passed to get_page --- returns NULL on
				   nonpresent page, (void *)1 on present swapped
				   out page or pointer to page --- only swapper
				   needs to support it, others should return
				   -EINVAL */
#define PF_COW		32	/* passed to get_page, get uninitialized page
				   (ready for cow). */
#define PF_WAIT_4_ALLOC	64	/* passed to get_pagein_rq, wait for page
				   allocation possible (so that get_page
				   succeeds), but do not allocate any page */
#define PF_SHARE	128	/* for MAP_ANONYMOUS | MAP_SHARED mapping */

/* when GET_PAGEIN_RQ is called, you must set rq->tmp1 as function */

typedef void vspace_unmap_t(void *ptr);
typedef void vspace_dmaunlock_t(__u32 ptr);
typedef void vspace_dma64unlock_t(__u64 ptr);
typedef void vspace_physunlock_t(__p_addr ptr);

vspace_unmap_t KERNEL$NULL_VSPACE_UNMAP;
vspace_dmaunlock_t KERNEL$NULL_VSPACE_DMAUNLOCK;
vspace_dma64unlock_t KERNEL$NULL_VSPACE_DMA64UNLOCK;
vspace_physunlock_t KERNEL$NULL_VSPACE_PHYSUNLOCK;

/*
 possible calling sequences:
 	clone_handle(failure)
 	clone_handle -> (lookup)* -> (instantiate_handle | leave_handle)
 	clone_handle -> (lookup)* -> create -> (instantiate_handle | leave_handle)
 	clone_handle -> (lookup)* -> delete(returns success)
 	clone_handle -> (lookup)* -> delete(returns error) -> leave_handle
	clone_handle -> (lookup)* -> rename -> leave_handle
 	clone_handle -> (lookup)* -> lookup_io -> leave_handle
	open_handle, close_handle (on handle instantiated with instantiate_handle)
	   (these two are not a pair!!, open or close might be called any times,
	    it is not valid that for each open there is a close call)
     detach_handle (deinstantiates handle, no more calls on handle are possible)
*/

	/*
	clone returns:
		__IS_ERR() -->error
		== NULL --> ok
		== (void *)2 --> repost immediatelly
		otherwise --> pointer to WQ --- put request on that queue
	lookup returns:
		__IS_ERR() --> error
		== NULL --> ok
		== (void *)1 --> fault, call lookup_io to handle it
		== (void *)2 --> repost immediatelly
		otherwise --> pointer to WQ -- put request on that queue
	create returns:
		__IS_ERR() --> error
		== NULL --> ok
		== (void *)1 --> fault, call lookup_io to handle it
		== (void *)2 --> repost immediatelly
		otherwise --> pointer to WQ -- put request on that queue
	fdelete returns:
		== NULL --> ok, file deleted (leave won't be called)
		__IS_ERR() --> error (leave will be called)
		(void *)2 --> repost immediatelly (leave will be called)
		(void *)3 --> fault, request queued, let it be (leave will be called)
	instantiate_handle returns:
		== NULL -> ok, leave won't be called, detach will be called
		__IS_ERR() --> error (leave will be called)
		(void *)2 --> repost immediatelly (leave will be called)
		(void *)3 --> request queued by instantiate, let it be (leave will be called)
	close_handle returns:
		< 0 -- error
		0 -- closed
		== 1 --> error, handle is closed in case posted error != EINTR */

struct __handle_operations {
	int spl;

	unsigned long (*vspace_get)(VDESC *desc, __const__ VBUF *buf);
	unsigned long (*vspace_put)(VDESC *desc, __const__ VBUF *buf);
	void *(*vspace_map)(VDESC *desc, int rw, vspace_unmap_t **unmap);
	VDMA (*vspace_dmalock)(VDESC *desc, int rw, vspace_dmaunlock_t **unlock);
	void (*vspace_dma64lock)(VDESC *desc, int rw, VDMA64 *dma, vspace_dma64unlock_t **unlock);
	void (*vspace_physlock)(VDESC *desc, int rw, VPHYS *p, vspace_physunlock_t **unlock);
	IORQ *(*vspace_get_pagein_rq)(VDESC *desc, IORQ *rq, int wr);
	PAGE *(*get_page)(HANDLE *h, __v_off idx, int wr);
	int (*swap_op)(HANDLE *h, int fn, ...);

	void *(*clone_handle)(HANDLE *from, HANDLE *to, int open_flags);
	void *(*lookup)(HANDLE *h, char *str, int open_flags);
	void *(*create)(HANDLE *h, char *str, int open_flags); /* like lookup but create file/directory */
	void *(*fdelete)(HANDLE *h, IORQ *rq, int open_flags, HANDLE *parent);
	void *(*rename)(HANDLE *h, char *str, HANDLE *hf, IORQ *rq);
	void (*lookup_io)(HANDLE *h, char *str, IORQ *rq, int open_flags);
	void *(*instantiate_handle)(HANDLE *h, IORQ *rq, int open_flags);
	void (*leave_handle)(HANDLE *h);
	void (*detach_handle)(HANDLE *h);
	void (*open_handle)(HANDLE *h, int open_flags);
	int (*close_handle)(HANDLE *h, IORQ *rq);

	/* operations */
	IO_STUB *read;
	IO_STUB *write;
	IO_STUB *aread;
	IO_STUB *awrite;
	IO_STUB *ioctl;

	IO_STUB *bio;
	IO_STUB *pktio;
};

#define SWAP_OP_ZAP_PAGES	1	/* swap_op */
#define SWAP_OP_FORK		2

/*
 * DESCRIPTION OF VSPACE FUNCTIONS
 * vspace_dmalock:
 *	RETURNS LOCKED DMA ADDRESS & LEN. LEN CAN BE LESS IF IT'S IMPOSSIBLE TO
 *	MAP IT INTO CONTIG SPACE.
 *	!ptr && !len --- PAGE FAULT OCCURED.
 * vspace_dma64lock:
 *	SAME, BUT RETURN 64-BIT ADDRESS.
 * vspace_physlock:
 *	SAME, BUT RETURNS PHYSICAL ADDRESS (THAT CAN BE LATER MAPPED WITH
 *	KERNEL$MAP_PHYSICAL_BANK). ON MACHINES WITHOUT IOMMU, vspace_physlock IS
 *	EQUIVALENT TO vspace_dma64lock.
 * vspace_get:
 *	COPY min(desc->len, buf->len) BYTES FROM VSPACE TO buf. CAN READ LESS
 *	BECAUSE OF PAGE-ALIGNMENT. 0 - PAGE FAULT.
 *	LOWERS SPL TO buf->spl. buf->spl MUST BE HIGHER OR EQUAL THAN SPL_DEV
 *	AND LOWER OR EQUL THAN SPL_VSPACE.
 * vspace_write:
 *	COPY min(desc->len, buf->len) BYTES FROM buf TO VSPACE. CAN WRITE LESS
 *	BECAUSE OF PAGE-ALIGNMENT. 0 - PAGE FAULT.
 *	LOWERS SPL TO buf->spl. buf->spl MUST BE HIGHER OR EQUAL THAN SPL_DEV
 *	AND LOWER OR EQUL THAN SPL_VSPACE.
 * vspace_map:
 *	MAP AREA FOR READ/WRITE. MUST MAP THE WHOLE AREA. AREA MUST NOT CROSS
 *	PAGE BOUNDARY.
 *	!PTR --- PAGE FAULT
 *	__IS_ERR(PTR) --- AREA CROSSES PAGE BOUNDARY
 * get_page:
 *	CALLED AT SPL OF HANDLE (UNLIKE OTHER VSPACE FUNCTIONS)
 *	wr IS PF_xxx FLAGS.
 *		PF_WRITE --- IT WAS WRITE FAULT (OTHERWISE READ FAULT)
 *		PF_SWAPPER --- ONLY SWAPPER, OTHER PAGERS MUST RETURN ERROR WHEN
 *			THIS FLAG PRESENT. WHEN PAGER RESPONDS TO PF_SWAPPER
 *			REQUESTS, IT MUST GUARANTEE TO COMPLETE REQUEST IN
 *			FINITE TIME. (THIS IS TO AVOID HANGING THE WHOLE SYSTEM
 *			WHEN NETWORK FILESYSTEM CANNOT DO PAGEIN)
 *		PG_TESTPAGE --- RETURN (void *)1 IF PAGE IS PRESENT BUT SWAPPED
 *			OUT. ONLY SWAPPER NEEDS TO SUPPORT THIS.
 *	RETURN: POINTER TO PAGE, __ERR_PTR, NULL IF PAGE NOT PRESENT OR SWAPPED
 *		OUT. (void *)1 IF PAGE PRESENT, SWAPPED OUT AND PF_TESTPAGE WAS
 *		SPECIFIED.
 * zap_pages:
 *	CALLED AT SPL OF HANDLE (UNLIKE OTHER VSPACE FUNCTIONS)
 *	ERASE PAGES IN SPECIFIED RANGE, UNSPECIFIED BEHAVIOUR WHEN RANGE IS
 *	NOT ON PAGE BOUNDARY (MUST NOT CRASH BUT DOESN'T HAVE TO BE CORRECT)
 *	RETURN 0 -- PAGES CLEARED. != 0 -- PAGES NOT CLEARED, IMMEDIATELLY REGET *	HANDLE AND RECALL
 */

/* DO_PAGEIN must load first non-present page from vspace range and optionally do readahead */

/*
 * rq - IORQ
 * fn - function to be called
 * vd - faulting VDESC
 * wr - PF_READ, PF_WRITE, PF_RW
 */

void KERNEL$PAGEIN(IORQ *rq, VDESC *vd, int wr);

#define DO_PAGEIN(rqq, vd, wr)						\
do {									\
	KERNEL$PAGEIN((IORQ *)(rqq), vd, wr);				\
	RETURN;								\
} while (0)

#define DO_PAGEIN_NORET(rqq, vd, wr)					\
do {									\
	KERNEL$PAGEIN((IORQ *)(rqq), vd, wr);				\
} while (0)



struct __handle {
	__const__ HANDLE_OPERATIONS *op;
	int handle_num;
	LIST_ENTRY proc_hash;
	PROC *file_addrspace;
	FFILE *file;
	void *fnode;
	long flags;
	long flags2;
	PROC *name_addrspace;
	FFILE *name;
	XLIST_HEAD child_list;
	LIST_ENTRY child_entry;
	LIST_ENTRY fnode_entry;
};

#define HANDLE_NUM_WRITE	(__INT_SGN_BIT >> 1)
#define HANDLE_NUM_READ		(__INT_SGN_BIT >> 2)
#define HANDLE_NUM_MASK		(MAX_HANDLE - 1)

struct __handle_vsp {
	__const__ HANDLE_OPERATIONS *op;
};

void KERNEL$DETACH_HANDLE(HANDLE *h);

FFILE *KERNEL$MAP_FILE_STRUCT(HANDLE *h, IORQ *rq, vspace_unmap_t **unmap);
FBLOB *KERNEL$FIND_FILE_BLOB(HANDLE *h, IORQ *rq, __u32 type, size_t *size);
HANDLE *KERNEL$PROC_VSPACE(PROC *p);
int KERNEL$PUT_IOCTL_STRUCT(IOCTLRQ *rq, void *p, long l);
int KERNEL$GET_IOCTL_STRUCT(IOCTLRQ *rq, void *p, long l);

extern IO_STUB KERNEL$NO_OPERATION;
extern IO_STUB KERNEL$BIO_EMU;

unsigned long KERNEL$NO_VSPACE_GET(VDESC *desc, __const__ VBUF *buf);
unsigned long KERNEL$NO_VSPACE_PUT(VDESC *desc, __const__ VBUF *buf);
void *KERNEL$NO_VSPACE_MAP(VDESC *desc, int rw, vspace_unmap_t **unmap);
VDMA KERNEL$NO_VSPACE_DMALOCK(VDESC *desc, int rw, vspace_dmaunlock_t **unlock);
void KERNEL$NO_VSPACE_DMA64LOCK(VDESC *desc, int rw, VDMA64 *dma, vspace_dma64unlock_t **unlock);
void KERNEL$NO_VSPACE_PHYSLOCK(VDESC *desc, int rw, VPHYS *p, vspace_physunlock_t **unlock);
IORQ *KERNEL$NO_VSPACE_GET_PAGEIN_RQ(VDESC *desc, IORQ *rq, int wr);
PAGE *KERNEL$NO_VSPACE_GET_PAGE(HANDLE *h, __v_off idx, int wr);
int KERNEL$NO_VSPACE_SWAP_OP(HANDLE *h, int op, ...);

extern IO_STUB KERNEL$WAKE_READ;
extern IO_STUB KERNEL$WAKE_WRITE;
extern IO_STUB KERNEL$WAKE_AREAD;
extern IO_STUB KERNEL$WAKE_AWRITE;
extern IO_STUB KERNEL$WAKE_IOCTL;
extern IO_STUB KERNEL$WAKE_BIO;
extern IO_STUB KERNEL$WAKE_PKTIO;

struct __ffile {
	_u_off_t pos;
	int flags;
	unsigned l_refcount;	/* local user-space refcount -- when to deallocate? */
	__u64 aux;
	XLIST_HEAD blobs;
	char path[1];
};

struct __fblob_ref {
	LIST_ENTRY list;
	FBLOB *blob;
};

struct __fpipe {
	unsigned l_refcount;
	unsigned start;
	unsigned end;
	unsigned readers;
	unsigned writers;
	int read_flags;
	int write_flags;
};

#define PIPE_SIZE	(__PAGE_CLUSTER_SIZE - sizeof(FPIPE))

/* DRQ */

typedef int __DCALL(void *ptr, __const__ char *dcall_type, int cmd, void *param);
typedef int __DCTL(void *ptr, void **unload, char *argv[]);

typedef struct {
	IORQ_HEAD;
	__const__ char *name;
	char *driver_name;
	int flags;	/* LNTE_PUBLIC */
	void (*init_root_handle)(HANDLE *, void *data);
	void *dev_ptr;	/* argument to dcall, dctl and unload */
	__DCALL *dcall;
	__const__ char *dcall_type;
	__DCTL *dctl;
	__DCTL *unload;
	int h;
	void *lnte;
	MALLOC_REQUEST mrq;
} DEVICE_REQUEST;

/* user must fill in: name, flags (LNTE_PUBLIC) and init_root_handle */
extern IO_STUB KERNEL$REGISTER_DEVICE;

/* flags are DELETE_LOGICAL_* */
int KERNEL$UNREGISTER_DEVICE(void *lnte_, int flags);

int KERNEL$DEVICE_UNLOAD(void *lnte_, char **argv);

int KERNEL$DCALL(__const__ char *name, __const__ char *type, int cmd, void *param);

#define OPTIMAL_IO_SIZE		__PAGE_CLUSTER_SIZE

int KERNEL$LOCKUP_WAIT_ENTRY(IORQ *rq, HANDLE *h);
int KERNEL$LOCKUP_WAIT_LOOP(IORQ *rq, HANDLE *h);

#define TEST_LOCKUP_ENTRY(RQ, retn)					\
do {									\
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {\
		if (KERNEL$LOCKUP_WAIT_ENTRY((IORQ *)(RQ), (RQ)->handle)) {\
			retn;						\
		}							\
	}								\
} while (0)

#define TEST_LOCKUP_LOOP(RQ, retn)					\
do {									\
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS) && __unlikely((RQ)->v.len != 0)) {							\
		if (KERNEL$LOCKUP_WAIT_LOOP((IORQ *)(RQ), (RQ)->handle)) {\
			retn;						\
		}							\
	}								\
} while (0)

__END_DECLS

#endif
