#ifndef __SPAD_PKT_H
#define __SPAD_PKT_H

#include <ARCH/TIME.H>
#include <ARCH/MOV.H>
#include <SYS/TYPES.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/LIST.H>
#include <ARCH/SETUP.H>
#include <ARCH/PAGE.H>
#include <ARCH/CPOS.H>
#include <SPAD/DL.H>
#include <SPAD/SLAB.H>
#include <SPAD/LIBC.H>
#include <ENDIAN.H>
#include <ARCH/IPCHECKS.H>
#include <STRING.H>

__BEGIN_DECLS

/*
 * ALLOC_PACKET: allocates packet
 * FREE_PACKET: frees packet
 * CLONE_PACKET: must be called before you modify received packet
 * CLONE_PACKET_FOR_RESEND: must be called (only once) before you modify
 *			    received packet in order to resend it
 * DUP_PACKET: must be called before you store packet for long-term
 * DEFRAG_PACKET: removes packet fragments; packets already arive defragmented
 *		  at protocol-level input
 * CHECKSUM_PACKET: computes checksum; packets already arive checksumed at
 *		    protocol-level input
 *
 * ALLOC_PACKET, CLONE_PACKET, DUP_PACKET, DEFRAG_PACKET fail if there's not
 * enough memory --- NET$OOM or NET$DELAYED_OOM should follow.
 */

#define MAX_ADDR_LEN		6
#define MAX_IFTYPE_LEN		13
#define LL_HEADER		(MAX_ADDR_LEN * 2 + 2)
#define DEFAULT_CHECKSUM_POS	12

typedef struct __packet PACKET;

#define __DEBUG_NET	1	/* !!! FIXME */

#if __DEBUG_NET
#define __ap	void *alloc_ptr;
#else
#define __ap
#endif

#ifdef __32_BIT
#define __length_mod	short
	/* On Pentium 4, packet fits exactly in 2 cachelines */
#else
#define __length_mod
#endif

#define __PACKET_ENTRIES			\
	IORQ_HEAD;				\
	int h;					\
	HANDLE *handle;				\
	unsigned __length_mod length;		\
	unsigned __length_mod data_length;	\
	VDESC v;				\
	int flags;				\
	union {					\
		__u32 u;			\
		__u16 s[2];			\
	} checksum;				\
	unsigned long id;			\
	void *sender_data;			\
	LIST_ENTRY list;			\
	jiffies_lo_t sent_time;			\
	unsigned char addrlen;			\
	__ap					\

struct __packet_for_align {
	__PACKET_ENTRIES
	char a[1];
};

struct __packet {
	__PACKET_ENTRIES
	__u8 align[__CHECKSUM_ALIGN - (((int)&((struct __packet_for_align *)NULL)->a + LL_HEADER + DEFAULT_CHECKSUM_POS) & (__CHECKSUM_ALIGN - 1))];
	__u8 data[LL_HEADER];
};

#define PKT_FREE		1	/* bugcheck for double-free */
#define PKT_PENDING_AST		2	/* bugcheck for double AST-call */
#define PKT_DONT_CHANGE		4	/* packet must not be changed */
#define PKT_OUTPUT_CHECKSUM	8	/* checksum must be calculated on send */
#define PKT_INPUT_CHECKSUM	16
#define PKT_IPHDR_CHECKSUM_OK	32	/* received and ip checksum ok */
#define PKT_TCPUDP_CHECKSUM_OK	64	/* received and tcp/udp/icmp/igmp checksum ok */
#define PKT_LOCAL_ECN		128	/* send and sending protocol is capable of ecn */
#define PKT_LOCAL_ECN_CE	256	/* local congestion experienced */
#define PKT_RESERVEDPOOL	512	/* packet is allocated from reserved pool, must not
					   be held indefinfinitely */
#define PKT_PAGING_ERROR	1024	/* packet's sg data is not paged in */
#define PKT_OUTPUT_CHECKSUM_TCP	2048	/* PKT_OUTPUT_CHECKSUM means tcp sum */
#define PKT_OUTPUT_CHECKSUM_UDP	4096	/* PKT_OUTPUT_CHECKSUM means udp sum */

#define PKT_REUSE_FLAGS		PKT_RESERVEDPOOL

#define PKT_Q_ENTRY(RQ)		((LIST_ENTRY *)&RQ->tmp1)
#define PKT_Q_RQ(LIST_ENT)	LIST_STRUCT(LIST_ENT, PACKET, tmp1)

#define PKT_PROTOCOL(p)		(*(__u16 *)(&(p)->data[LL_HEADER - 2]))
#define PKT_DSTADDR(p, l)	(&(p)->data[LL_HEADER - 2 - 2 * (l)])
#define PKT_SRCADDR(p, l)	(&(p)->data[LL_HEADER - 2 - (l)])

#define CHECKSUM_POS(p)		((p)->checksum.s[0])	/* output checksum */
#define CHECKSUM_VAL(p)		((p)->checksum.s[0])	/* input checksum */
#define CHECKSUM_FROM(p)	(*(__s16 *)((p)->checksum.s + 1)) /* both */
#if defined(__LITTLE_ENDIAN)
#define MKCHECKSUM(from, pos)	(((from) << 16) | (pos))
#define XPOSCHECKSUM(u)		((u) & 0xffff)
#define XFROMCHECKSUM(u)	((u) >> 16)
#elif defined(__BIG_ENDIAN)
#define MKCHECKSUM(from, pos)	(((pos) << 16) | ((from) & 0xFFFF))
#define XPOSCHECKSUM(u)		((u) >> 16)
#define XFROMCHECKSUM(u)	((u) & 0xffff)
#else
no endian !
#endif

#define PKT_PREFETCH_1(p)	__PREFETCHT0(p->data + LL_HEADER - 4)
#define PKT_PREFETCH_2(p)	__PREFETCHT0_IF32(p->data + LL_HEADER - 4 + 32)

#define PKT_VERSION	(__DEBUG_NET)
extern __const__ int NET$PKT_VERSION;
extern __const__ int NET$PKT_SIZE;

#define PKT_CHECK_VERSION						\
{									\
	if (__unlikely(NET$PKT_VERSION != PKT_VERSION) || __unlikely(NET$PKT_SIZE != sizeof(PACKET))) {							\
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NET.DLL PACKET INTERFACE VERSION MISMATCH");						\
		return -EPROTO;						\
	}								\
}

#define MAX_PACKET_FRAGMENTS(size)	((((size) + 2 * (1 << __PG_SIZE_BITS) - 2) >> __PG_SIZE_BITS) + 1)

#define TEST_BROADCAST_PACKET(p_, cmd_)					\
do {									\
	unsigned i_ = (p_)->addrlen;					\
	__u8 *q_;							\
	if (__likely(i_)) {						\
		q_ = PKT_DSTADDR(p_, i_);				\
		do {							\
			if (__likely(*q_++ != 0xff)) goto ok_;		\
		} while (--i_);						\
		cmd_;							\
		ok_:;							\
	}								\
} while (0)

#define SET_BROADCAST_DST(p_)						\
	memset(PKT_DSTADDR((p_), (p_)->addrlen), 0xff, (p_)->addrlen)

#define NET_PROTOCOL_802_3	(htons(0x0001))

#define NET_PROTOCOL_IP		(htons(0x0800))
#define NET_PROTOCOL_ARP	(htons(0x0806))
#define NET_PROTOCOL_EAPOL	(htons(0x888E))

#define NET_PROTOCOL_N		0x10000

#define NET_INPUT_DEVICE	"PKT$LO"

#if __DEBUG_NET
#define _PKT_REDZONE_CHAR	'R'
#define _PKT_REDZONE_FREE	'F'
#define _PKT_REDZONE_BYTES	1
#else
#define _PKT_REDZONE_BYTES	0
#endif

#define _ALLOC_GR_BITS		9
#define _ALLOC_GR		(1 << _ALLOC_GR_BITS)
#define _PACKET_SLOT(length)	(((length) + _ALLOC_GR - 1) >> _ALLOC_GR_BITS)
#define _SLOT_SIZE(slot)	((slot) << _ALLOC_GR_BITS)
#define _PACKET_SLOTS		(_PACKET_SLOT(65535 + _PKT_REDZONE_BYTES) + 1)
#define MAX_PACKET_LENGTH_TOP	(65535)

extern struct __slhead NET$_PACKET_SLABS[_PACKET_SLOTS];
extern unsigned NET$MAX_PACKET_LENGTH;

extern long NET$MEMORY_AVAIL;
extern long NET$MEMORY_LIMIT;

void NET$DELAYED_OOM(void);	/* calls NET$OOM_KILL out of current context */
WQ *NET$OOM(void);		/* return: NULL: retry alloc,
				   __IS_ERR: return error, WQ --- wait on it */
int NET$MEMWAIT_SYNC(void);	/* return: 0: retry, -error: return error */

#if __DEBUG >= 1
#define __checkspl(splv, msg)						\
do {									\
	if (__unlikely(KERNEL$SPL != (splv)))				\
		KERNEL$SUICIDE("%s CALLED AT SPL %08X, EXPECTED %08X", (msg), KERNEL$SPL, (splv));\
} while (0)
#else
#define __checkspl(splv, msg)						\
do {									\
} while (0)
#endif

/* do ... while (0) is not useable, because fail routine may contain break */
#define __pkt_slalloc(p__, length__, splv__, fail__)			\
{									\
	__checkspl(splv__, "__pkt_slalloc");				\
	RAISE_SPL(SPL_PACKET);						\
	if (__unlikely(!((p__) = __slalloc(&NET$_PACKET_SLABS[_PACKET_SLOT((length__) + _PKT_REDZONE_BYTES)])))) {					\
		LOWER_SPLX(splv__);					\
		fail__;							\
	} else {							\
		(p__)->length = (length__);				\
		(p__)->flags = 0;					\
		LOWER_SPLX(splv__);					\
	}								\
}

#define __pkt_slfree(p__, splv__)					\
do {									\
	__checkspl(splv__, "__pkt_slfree");				\
	RAISE_SPL(SPL_PACKET);						\
	__slfree(p__);							\
	LOWER_SPLX(splv__);						\
} while (0)

typedef struct {
	PACKET *freelist;
	unsigned length;
	unsigned n_allocated;
	unsigned n_total;
	WQ freepkt_wait;
} PKTPOOL;

#define INIT_PKTPOOL(pool, len)						\
do {									\
	(pool)->freelist = NULL;					\
	(pool)->length = (len);						\
	(pool)->n_allocated = 0;					\
	(pool)->n_total = 0;						\
	WQ_INIT(&(pool)->freepkt_wait, "NET$FREEPKT_WAIT");		\
} while (0)

extern PKTPOOL NET$PKTPOOL;
extern PKTPOOL NET$PKTPOOL_LONG_TERM;
extern AST_STUB NET$FREE_PACKET;
int NET$PKTPOOL_RESERVE(PKTPOOL *pool);
void NET$PKTPOOL_FREE(PKTPOOL *pool);

#if __DEBUG >= 1
#define __check_length(len)						\
do {									\
		/* + 1 is there to shut up warning */			\
	if (__unlikely((unsigned)(len) + 1 > MAX_PACKET_LENGTH_TOP + 1))\
		KERNEL$SUICIDE("ALLOC_PACKET: TRYING TO ALLOCATE TOO BIG PACKET %d, MAXIMAL PACKET LENGTH %d", (unsigned)(len), (unsigned)MAX_PACKET_LENGTH_TOP);									\
} while (0)
#else
#define __check_length(len) do { } while (0)
#endif

#if __DEBUG_NET
#define __setcpos(p_)	__cpos((p_)->alloc_ptr)
#define __CHECK_PKT(p_, m_)						\
do {									\
	if (__unlikely((p_)->flags & (PKT_FREE | PKT_PENDING_AST)) || __unlikely(!(p_)->fn)) {								\
		unsigned long off_ = 0;					\
		__const__ char *str_ = KERNEL$DL_GET_SYMBOL_NAME((p_)->alloc_ptr, &off_, 0);								\
		unsigned long off2_ = 0;				\
		__const__ char *str2_ = KERNEL$DL_GET_SYMBOL_NAME((p_)->fn, &off2_, 0);									\
		KERNEL$SUICIDE(m_": FREE PACKET, PACKET SIZE %d, DATA SIZE %d, FLAGS %X, ALLOCATED AT %s+%lX, AST %s+%lX", (p_)->length, (p_)->data_length, (p_)->flags, str_ ? str_ : "?", off_, str2_ ? str2_ : "?", off2_);		\
	}								\
	if (__unlikely((p_)->data[LL_HEADER + (p_)->length] != _PKT_REDZONE_CHAR)) {									\
		unsigned long off_ = 0;					\
		__const__ char *str_ = KERNEL$DL_GET_SYMBOL_NAME((p_)->alloc_ptr, &off_, 0);								\
		unsigned long off2_ = 0;				\
		__const__ char *str2_ = KERNEL$DL_GET_SYMBOL_NAME((p_)->fn, &off2_, 0);									\
		KERNEL$SUICIDE(m_": REDZONE DAMAGED (%02X), PACKET SIZE %d, DATA SIZE %d, FLAGS %X, ALLOCATED AT %s+%lX, AST %s+%lX", (p_)->data[LL_HEADER + (p_)->length], (p_)->length, (p_)->data_length, (p_)->flags, str_ ? str_ : "?", off_, str2_ ? str2_ : "?", off2_);						\
	}								\
} while (0)
#define __checkdoublefree(p_)						\
do {									\
	(p_)->fn = (void *)1;						\
	__CHECK_PKT(p_, "FREE_PACKET");					\
	(p_)->data[LL_HEADER + (p_)->length] = _PKT_REDZONE_FREE;	\
	(p_)->flags = PKT_FREE;						\
} while (0)
#define __set_empty(p_)							\
do {									\
	(p_)->fn = NULL;						\
	(p_)->data[LL_HEADER + (p_)->length] = _PKT_REDZONE_CHAR;	\
} while (0)
#define CALL_PKT(p_)							\
do {									\
	__CHECK_PKT(p_, "CALL_PKT");					\
	(p_)->flags |= PKT_PENDING_AST;					\
	CALL_AST(p_);							\
} while (0)
#define RETURN_PKT(p_)							\
do {									\
	__CHECK_PKT(p_, "RETURN_PKT");					\
	(p_)->flags |= PKT_PENDING_AST;					\
	RETURN_AST(p_);							\
} while (0)
#define PKT_AST_ENTER(p_)						\
do {									\
	if (__unlikely(!((p_)->flags & PKT_PENDING_AST))) {		\
		unsigned long off_ = 0;					\
		__const__ char *str_ = KERNEL$DL_GET_SYMBOL_NAME((p_)->alloc_ptr, &off_, 0);								\
		KERNEL$SUICIDE("PKT_AST_ENTER: NOT PENDING AST, PACKET SIZE %d, DATA SIZE %d, FLAGS %X, ALLOCATED AT %s+%lX", (p_)->length, (p_)->data_length, (p_)->flags, str_ ? str_ : "?", off_);					\
	}								\
	(p_)->flags &= ~PKT_PENDING_AST;				\
	__CHECK_PKT(p_, "PKT_AST_ENTER");				\
} while (0)
#else
#define __setcpos(p_) do { } while (0)
#define __CHECK_PKT(p_, m_) do { } while (0)
#define __checkdoublefree(p_) do { } while (0)
#define __set_empty(p_) do { } while (0)
#define CALL_PKT(p_) CALL_AST(p_)
#define RETURN_PKT(p_) RETURN_AST(p_)
#define PKT_AST_ENTER(p_) do { } while (0)
#endif

/* do ... while (0) is not useable, because fail routine may contain break */
#define ALLOC_PACKET(p_, len_, pool_, spl_, fail_)			\
{									\
	__check_length(len_);						\
	__pkt_slalloc(p_, len_, SPL_X(spl_), {				\
		if (__unlikely(!(pool_))				\
		 || __unlikely((len_) > ((PKTPOOL *)(pool_))->length)	\
		 || __unlikely(!((p_) = ((PKTPOOL *)(pool_))->freelist))) {\
			fail_;						\
			__unreachable_code();				\
		}							\
		((PKTPOOL *)(pool_))->freelist = *(PACKET **)(p_);	\
		((PKTPOOL *)(pool_))->n_allocated++;			\
		(p_)->flags = PKT_RESERVEDPOOL;				\
	});								\
	__set_empty(p_);						\
	__setcpos(p_);							\
	(p_)->v.ptr = 0;						\
	(p_)->v.len = 0;						\
	(p_)->v.vspace = &KERNEL$PHYSICAL;				\
}

#define PKT_CLONE_NORESERVED		1
#define PKT_CLONE_RESERVED_LONG_TERM	2

PACKET *NET$CLONE_PACKET(PACKET *p, PKTPOOL *pool);

/* do ... while (0) is not useable, because fail routine may contain break */
#define TRYFREE_PACKET(p_, pool_, fail_)				\
{									\
	if (__likely((pool_) != NULL) &&				\
	    __unlikely(((PKTPOOL *)(pool_))->n_allocated != 0) && 	\
	    __likely((p_)->length >= ((PKTPOOL *)(pool_))->length)) {	\
		__checkdoublefree(p_);					\
		*(PACKET **)(p_) = ((PKTPOOL *)(pool_))->freelist;	\
		((PKTPOOL *)(pool_))->freelist = (p_);			\
		((PKTPOOL *)(pool_))->n_allocated--;			\
		WQ_WAKE_ALL(&((PKTPOOL *)(pool_))->freepkt_wait);	\
	} else {							\
		fail_;							\
	}								\
}

#define FREE_PACKET(p_, pool_, spl_)					\
do {									\
	TRYFREE_PACKET(p_, pool_, {					\
		__pkt_slfree(p_, SPL_X(spl_));				\
	});								\
} while (0)

/* do ... while (0) is not useable, because fail routine may contain break */
#define CLONE_PACKET(p_, fail_)						\
{									\
	PACKET *np_;							\
	__CHECK_PKT(p_, "CLONE_PACKET");				\
	if (__unlikely((p_)->flags & PKT_DONT_CHANGE)) {		\
		if (__unlikely(!(np_ = NET$CLONE_PACKET(p_, &NET$PKTPOOL)))) {\
			fail_;						\
			__unreachable_code();				\
		}							\
		(p_) = np_;						\
	}								\
}

/* do ... while (0) is not useable, because fail routine may contain break */
#define CLONE_PACKET_FOR_RESEND(p_, fail_)				\
{									\
	PACKET *np_;							\
	__CHECK_PKT(p_, "CLONE_PACKET");				\
	if (__unlikely((p_)->flags & PKT_DONT_CHANGE)) {		\
		if (__unlikely(!(np_ = NET$CLONE_PACKET(p_, &NET$PKTPOOL)))) {\
			fail_;						\
			__unreachable_code();				\
		}							\
		(p_) = np_;						\
	} else if ((p_)->fn == NET$FREE_PACKET)				\
		(*(unsigned long *)(p_)->sender_data)--;		\
}

/* do ... while (0) is not useable, because fail routine may contain break */
#define DUP_PACKET(p_, pool_, fail__)					\
{									\
	PACKET *np_;							\
	__CHECK_PKT(p_, "DUP_PACKET");					\
	if (__unlikely((p_)->flags & (PKT_DONT_CHANGE | PKT_RESERVEDPOOL))) {\
		if (__unlikely(!(np_ = NET$CLONE_PACKET(p_, pool_)))) {	\
			fail__;						\
			__unreachable_code();				\
		}							\
		(p_) = np_;						\
	} else if (__likely((p_)->fn != NET$FREE_PACKET)) {		\
		ALLOC_PACKET(np_, (p_)->length, pool_, SPL_NET, {	\
			fail__;						\
			__unreachable_code();				\
		});							\
		np_->fn = (p_)->fn;					\
		np_->sender_data = (p_)->sender_data;			\
		np_->status = 0;					\
		CALL_PKT(np_);						\
		(p_)->fn = NET$FREE_PACKET;				\
		(p_)->sender_data = &KERNEL$LIST_END;			\
	} else {							\
		(*(unsigned long *)(p_)->sender_data)--;		\
		(p_)->sender_data = &KERNEL$LIST_END;			\
	}								\
}

/* do ... while (0) is not useable, because fail routine may contain break */
#define DEFRAG_PACKET(p_, fail_)					\
{									\
	PACKET *np_;							\
	__CHECK_PKT(p_, "DEFRAG_PACKET");				\
	if (__unlikely((p_)->v.len != 0)) {				\
		if (__unlikely(!(np_ = NET$CLONE_PACKET(p_, &NET$PKTPOOL)))) {\
			fail_;						\
			__unreachable_code();				\
		}							\
		(p_) = np_;						\
	}								\
}

/* do ... while (0) is not useable, because fail routine may contain break */
#define CHECKSUM_PACKET(p_, fail_)					\
{									\
	__CHECK_PKT(p_, "CHECKSUM_PACKET");				\
	if ((p_)->flags & PKT_OUTPUT_CHECKSUM) {			\
		int r_;							\
		if (__unlikely(r_ = NET$CHECKSUM_PACKET(p_))) {		\
			fail_;						\
			__unreachable_code();				\
		}							\
	}								\
}

typedef struct __pktqueue PKTQUEUE;

__u8 *NET$MAP_INDIRECT_PACKET(VBUF *ibuf, VDESC *v, vspace_unmap_t **unmap);
int NET$CHECKSUM_PACKET(PACKET *p);

int NETQUE$ALLOC_QUEUE(PKTQUEUE **q);
void NETQUE$FREE_QUEUE(PKTQUEUE *q);
void NETQUE$DISCRAD_PACKETS(PKTQUEUE *q, int err);
void NETQUE$ENQUEUE_PACKET(PKTQUEUE *q, PACKET *rq);
int NETQUE$QUEUE_EMPTY(PKTQUEUE *q);
PACKET *NETQUE$DEQUEUE(PKTQUEUE *q);
void NETQUE$REQUEUE_DEQUEUED_PACKET(PKTQUEUE *q, PACKET *rq);

extern int NET$LOCAL_HANDLE;

typedef void PROTOCOL_HANDLER(PACKET *p);
int NET$REGISTER_PROTOCOL(PROTOCOL_HANDLER *p, unsigned type);
void NET$UNREGISTER_PROTOCOL(unsigned type);

char *NET$PRINT_STRING(char str[__MAX_STR_LEN], __u8 *string, unsigned len);

typedef struct {
	__u32 mtu;
	unsigned long id;
	__u16 arptype;
	__u8 addrlen;
	__u8 addr[MAX_ADDR_LEN];
	__u8 type[MAX_IFTYPE_LEN];
} IF_TYPE;

#define IDTYPE_LO	(0x1 << (sizeof(unsigned long) * 8 - 3))
#define IDTYPE_IOPORT	(0x2 << (sizeof(unsigned long) * 8 - 3))
#define IDTYPE_PCI	(0x3 << (sizeof(unsigned long) * 8 - 3))
#define IDTYPE_MASK	((1 << (sizeof(unsigned long) * 8 - 3)) - 1)

char *NET$PRINT_HWADDR(char str[__MAX_STR_LEN], __u8 *hwaddr, int hwaddrlen);

#define __LINK_STATE_ENTRIES		\
	__u64 speed;			\
	int flags;			\
	char desc[20]

typedef struct {
	__LINK_STATE_ENTRIES;
} LINK_STATE_PHYS;

typedef struct {
	__LINK_STATE_ENTRIES;
	__u64 seq;
} LINK_STATE;

#define LINK_STATE_UP				1
#define LINK_STATE_UNKNOWN			2
#define LINK_STATE_HALF_DUPLEX			4
#define LINK_STATE_FULL_DUPLEX			8
#define LINK_STATE_AUTO_NEGOTIATION		16
#define LINK_STATE_AUTO_NEGOTIATION_FAILED	32
#define LINK_STATE_REMOTE_FAULT			64
#define LINK_STATE_JABBER			128
#define LINK_STATE_MEDIA_AUTOSELECT		256

char *NET$PRINT_LINK_STATE(char buf[__MAX_STR_LEN], LINK_STATE_PHYS *ls);

__END_DECLS

#endif
