#ifndef __SPAD_QUOTA_H
#define __SPAD_QUOTA_H

#include <SYS/TYPES.H>
#include <SPAD/TIMER.H>
#include <ARCH/BSF.H>
#include <ARCH/SETUP.H>
#include <VALUES.H>

#define MAX_ADVANTAGE		10000
#define DEFAULT_ADVANTAGE	100

#define QUOTA_DECAY_BITS	(__BSR_CONST(JIFFIES_PER_SECOND))

#define DISADVANTAGE(adv)	(MAX_ADVANTAGE * MAX_ADVANTAGE / (adv))
#define DEFAULT_DISADVANTAGE	DISADVANTAGE(DEFAULT_ADVANTAGE)

typedef struct {
	long q_limit;
	long q_granted;
	long q_disadv;
	long q_maxlong_div_disadv;
	long q_node_usage;
	long q_subtree_usage;
} QUOTA;

typedef struct {
	long q_limit;
	long q_granted;
	long q_disadv;
	long q_maxlong_div_disadv;
	long q_node_usage;
	long q_subtree_usage;
	long q_tmp_limit;
	u_jiffies_lo_t q_tmp_limit_time;
} __ALIGN_ATTR__(sizeof(long) * 8 > __CPU_CACHELINE_ALIGN ? __CPU_CACHELINE_ALIGN : sizeof(long) * 8) QUOTA_R;

#define QINIT(q_)							\
do {									\
	QUOTA *_q = (q_);						\
	_q->q_limit = MAXLONG;						\
	_q->q_granted = 0;						\
	_q->q_disadv = DEFAULT_DISADVANTAGE;				\
} while (0)


#define QRINIT(q_)							\
do {									\
	QUOTA_R *_q_ = (q_);						\
	QINIT((QUOTA *)_q_);						\
} while (0)

#define QINIT2(q_)							\
do {									\
	QUOTA *_q = (q_);						\
	_q->q_maxlong_div_disadv = __likely(_q->q_disadv == DEFAULT_DISADVANTAGE) ? MAXLONG / DEFAULT_DISADVANTAGE : MAXLONG / _q->q_disadv;		\
	_q->q_node_usage = 0;						\
	_q->q_subtree_usage = 0;					\
} while (0)

#define QRINIT2(q_)							\
do {									\
	QUOTA_R *_q = (q_);						\
	_q->q_maxlong_div_disadv = __likely(_q->q_disadv == DEFAULT_DISADVANTAGE) ? MAXLONG / DEFAULT_DISADVANTAGE : MAXLONG / _q->q_disadv;		\
	_q->q_node_usage = 0;						\
	_q->q_subtree_usage = 0;					\
	_q->q_tmp_limit = MAXLONG;					\
	_q->q_tmp_limit_time = 0;					\
} while (0)


#define QDONE(q_, m_)							\
do {									\
	if (__unlikely(((q_)->q_node_usage | (q_)->q_subtree_usage) != 0))\
		KERNEL$SUICIDE("QUOTA " m_ " LEAK (NODE %ld, SUBTREE %ld)", (q_)->q_node_usage, (q_)->q_subtree_usage);					\
} while (0)

#define QDONE2(q1_, m1_, q2_, m2_)					\
do {									\
	if (__unlikely(((q1_)->q_node_usage | (q1_)->q_subtree_usage |	\
			(q2_)->q_node_usage | (q2_)->q_subtree_usage) != 0))\
		KERNEL$SUICIDE("QUOTA LEAK: "				\
			       m1_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m2_ " (NODE %ld, SUBTREE %ld)",		\
			(q1_)->q_node_usage, (q1_)->q_subtree_usage,	\
			(q2_)->q_node_usage, (q2_)->q_subtree_usage);	\
} while (0)

#define QDONE6(q1_, m1_, q2_, m2_, q3_, m3_, q4_, m4_, q5_, m5_, q6_, m6_)\
do {									\
	if (__unlikely(((q1_)->q_node_usage | (q1_)->q_subtree_usage |	\
			(q2_)->q_node_usage | (q2_)->q_subtree_usage |	\
			(q3_)->q_node_usage | (q3_)->q_subtree_usage |	\
			(q4_)->q_node_usage | (q4_)->q_subtree_usage |	\
			(q5_)->q_node_usage | (q5_)->q_subtree_usage |	\
			(q6_)->q_node_usage | (q6_)->q_subtree_usage) != 0))\
		KERNEL$SUICIDE("QUOTA LEAK: "				\
			       m1_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m2_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m3_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m4_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m5_ " (NODE %ld, SUBTREE %ld) OR "	\
			       m6_ " (NODE %ld, SUBTREE %ld)",		\
			(q1_)->q_node_usage, (q1_)->q_subtree_usage,	\
			(q2_)->q_node_usage, (q2_)->q_subtree_usage,	\
			(q3_)->q_node_usage, (q3_)->q_subtree_usage,	\
			(q4_)->q_node_usage, (q4_)->q_subtree_usage,	\
			(q5_)->q_node_usage, (q5_)->q_subtree_usage,	\
			(q6_)->q_node_usage, (q6_)->q_subtree_usage);	\
} while (0)

#define Q_NULL_CALL(q_)							\
do {									\
} while (0)

#define QALLOC(q_, n_, isroot, parent, sched_fn, zap, failure)		\
do {									\
	QUOTA *_qq;							\
	(q_)->q_node_usage += (n_);					\
	_qq = (q_);							\
	while (1) {							\
		if (__likely(!_qq->q_subtree_usage)) sched_fn(_qq);	\
		_qq->q_subtree_usage += (n_);				\
		if (__unlikely(_qq->q_subtree_usage > _qq->q_limit)) {	\
			(q_)->q_node_usage -= (n_);			\
			_qq = (q_);					\
			while (_qq->q_subtree_usage <= _qq->q_limit) _qq->q_subtree_usage -= (n_), _qq = parent(_qq);					\
			_qq->q_subtree_usage -= (n_);			\
			(zap) = _qq;					\
			{						\
				failure;				\
			}						\
			__unreachable_code();				\
		}							\
		if (isroot(_qq)) break;					\
		_qq = parent(_qq);					\
	}								\
} while (0)

#define QRALLOC(q_, n_, isroot, parent, sched_fn, zap, failure)		\
do {									\
	QUOTA_R *_qq;							\
	(q_)->q_node_usage += (n_);					\
	_qq = (q_);							\
	while (1) {							\
		if (__likely(!_qq->q_subtree_usage)) sched_fn(_qq);	\
		_qq->q_subtree_usage += (n_);				\
		if (__unlikely(_qq->q_tmp_limit_time != 0)) {		\
			unsigned long _nq;				\
			unsigned _sh;					\
			u_jiffies_lo_t _j = KERNEL$GET_JIFFIES_LO();	\
			_sh = (_j - _qq->q_tmp_limit_time) >> QUOTA_DECAY_BITS;\
			_qq->q_tmp_limit_time += 1 << _sh;		\
			_nq = ((unsigned long)_qq->q_tmp_limit << _sh) & ~__LONG_SGN_BIT;								\
			if ((_nq >> _sh) != _qq->q_tmp_limit) {		\
				_qq->q_tmp_limit = MAXLONG;		\
				_qq->q_tmp_limit_time = 0;		\
			} else {					\
				if (__unlikely(_qq->q_subtree_usage > _qq->q_tmp_limit)) goto _over_limit;						\
			}						\
		}							\
		/*if (_qq->q_subtree_usage > (n_)) goto _over_limit;*/  \
		if (__unlikely(_qq->q_subtree_usage > _qq->q_limit)) {	\
			QUOTA_R *_qqq;					\
			_over_limit:					\
			(q_)->q_node_usage -= (n_);			\
			_qqq = (q_);					\
			while (_qqq != _qq) _qqq->q_subtree_usage -= (n_), _qqq = parent(_qqq);								\
			_qq->q_subtree_usage -= (n_);			\
			(zap) = _qq;					\
			{						\
				failure;				\
			}						\
			__unreachable_code();				\
		}							\
		if (isroot(_qq)) break;					\
		_qq = parent(_qq);					\
	}								\
} while (0)


#if __DEBUG >= 1
#define _q_test_node_usage(_q)		\
	if (__unlikely((_q)->q_node_usage < 0)) KERNEL$SUICIDE("QUOTA NODE USAGE UNDERFLOW (%ld,%ld) AT %s:%d", (_q)->q_node_usage, (_q)->q_subtree_usage, __FILE__, __LINE__);
#define _q_test_subtree_usage(_q)	\
	if (__unlikely((_q)->q_subtree_usage < 0)) KERNEL$SUICIDE("QUOTA SUBTREE USAGE UNDERFLOW (%ld, %ld) AT %s:%d", (_q)->q_node_usage, (_q)->q_subtree_usage, __FILE__, __LINE__);
#else
#define _q_test_node_usage(_q)
#define _q_test_subtree_usage(_q)
#endif

#define QFREE_COMMON(type_, q_, n_, isroot, parent, sched_fn)		\
do {									\
	type_ *_q = (q_);						\
	_q->q_node_usage -= (n_);					\
	_q_test_node_usage(_q);						\
	while (1) {							\
		_q->q_subtree_usage -= (n_);				\
		if (__likely(!_q->q_subtree_usage)) sched_fn(_q);	\
		_q_test_subtree_usage(_q);				\
		if (isroot(_q)) break;					\
		_q = parent(_q);					\
	}								\
} while (0)

#define QFREE(q_, n_, isroot, parent, sched_fn)				\
QFREE_COMMON(QUOTA, q_, n_, isroot, parent, sched_fn)

#define QRFREE(q_, n_, isroot, parent, sched_fn)			\
QFREE_COMMON(QUOTA_R, q_, n_, isroot, parent, sched_fn)

#define QXFER_COMMON(type_, q_, toq_, n_, parent)			\
do {									\
	type_ *_q = (q_);						\
	_q->q_node_usage -= (n_);					\
	_q_test_node_usage(_q);						\
	while (_q != (toq_)) {						\
		_q->q_subtree_usage -= (n_);				\
		_q_test_subtree_usage(_q);				\
		_q = parent(_q);					\
	}								\
	_q->q_node_usage += (n_);					\
} while (0)

#define QXFER(q_, toq_, n_, parent)					\
QXFER_COMMON(QUOTA, q_, toq_, n_, parent)

#define QRXFER(q_, toq_, n_, parent)					\
QXFER_COMMON(QUOTA_R, q_, toq_, n_, parent)


#define QZAP_COMMON(type_, q_, for_all_subnodes, for_all_subnodes_tail, zap, zap_nonleaf)								\
do {									\
	type_ *_q = (q_), *_qq, *_maxq;					\
	long _maxa;							\
	int _shift;							\
	subtree:							\
	_maxq = _q;							\
	_shift = 0;							\
	if ((zap_nonleaf) && _q->q_node_usage) {			\
		_maxa = _q->q_node_usage;				\
		while (__unlikely(_maxa >= MAXLONG / DEFAULT_DISADVANTAGE)) {\
			_maxa >>= 1;					\
			_shift++;					\
		}							\
		_maxa *= DEFAULT_DISADVANTAGE;				\
	} else _maxa = MINLONG;						\
	for_all_subnodes(_qq, _q) {					\
		long _x;						\
		if (!_qq->q_subtree_usage) goto _skip_this;		\
		_x = (_qq->q_subtree_usage - _qq->q_granted) >> _shift;	\
		/*__debug_printf("(%p->%p): x %ld, gr %ld, sh %d, disa %ld, _maxa %ld\n", _q, _qq, _x, _qq->q_granted, _shift, _qq->q_disadv, _maxa);*/	\
		while (__unlikely(_x >= _qq->q_maxlong_div_disadv) || __unlikely(-_x >= _qq->q_maxlong_div_disadv)) {					\
			_x >>= 1;					\
			_maxa >>= 1;					\
			_shift++;					\
		}							\
		_x *= _qq->q_disadv;					\
		if (_x > _maxa) _maxa = _x, _maxq = _qq;		\
		_skip_this:;						\
	} for_all_subnodes_tail(_qq, _q);				\
	if (_maxq == _q) {						\
		(zap) = _q;						\
		goto _done;						\
	}								\
	_q = _maxq;							\
	goto subtree;							\
	_done:;								\
} while (0)

#define QZAP(q_, for_all_subnodes, for_all_subnodes_tail, zap, zap_nonleaf)\
QZAP_COMMON(QUOTA, q_, for_all_subnodes, for_all_subnodes_tail, zap, zap_nonleaf)

#define QRZAP(q_, for_all_subnodes, for_all_subnodes_tail, zap, zap_nonleaf)\
QZAP_COMMON(QUOTA_R, q_, for_all_subnodes, for_all_subnodes_tail, zap, zap_nonleaf)

#define QRRESTRICT(q_)							\
do {									\
	if (__likely((q_)->q_subtree_usage < (q_)->q_tmp_limit)) (q_)->q_tmp_limit = (q_)->q_subtree_usage - 1;						\
	if (__unlikely((q_)->q_tmp_limit <= 1)) (q_)->q_tmp_limit = 1;	\
	(q_)->q_tmp_limit_time = KERNEL$GET_JIFFIES_LO();		\
} while (0)

#endif
