#ifndef __SPAD_SCHED_H
#define __SPAD_SCHED_H

#include <SYS/TYPES.H>
#include <ARCH/SCHED.H>
#include <SPAD/PROC.H>

#define PRI_MAX		10000
#define PRI_INV(x)	(ARCH_SCHED_INV / (x))

#define SELF_INVPRI	(PRI_INV(100))

#define MAXIMUM_BONUS_MULT	(16)	/* 1.6 seconds or less */

#define SCHED_ONLIST		1
#define SCHED_READY		2

/* divide entries into classes:
	acct: most important, used each time something is accounted to a process
		(i.e. quick wake/sleep)
	sched: less important, used when scheduling continuously running
		processes
	readylist: used when modifying readylist (i.e. when starting or stopping
		long-running process)
	update_invpri: used once per minute for dynamic priority adjustment
   and scramble them to minimize cache pollution.
   All important entries fit into 64 bytes, so it is recommended to
   align the structure to a cacheline.
*/


typedef struct {
	sched_unsigned last_total_time;		/* acct */
	sched_unsigned accumulated_time;	/* acct */

	sched_unsigned invpri;			/* acct */
	sched_unsigned borrowlatency;		/* acct */
	sched_unsigned borrowlatency100;	/* acct */

	sched_unsigned count_delta;		/* acct */
	sched_unsigned count;			/* acct */
	sched_unsigned ccount_delta;		/* acct */
	sched_unsigned ccount;			/* acct */

	sched_unsigned schedcount;		/* acct */
	sched_unsigned bonus;			/* acct */

	XLIST_HEAD runq;			/* sched */
	LIST_ENTRY rentry;			/* sched */
	int flags;				/* acct */

	sched_unsigned unblock_time;		/* readylist */
	sched_unsigned pct_usage;		/* update_invpri */
	sched_unsigned mininvpri;		/* update_invpri */
	sched_unsigned maxinvpri;		/* update_invpri */
} SCHED;

typedef struct {
	sched_unsigned last_total_time;		/* acct */
	sched_unsigned accumulated_time;	/* acct */

	sched_unsigned invpri;			/* acct */

	sched_unsigned count_delta;		/* acct */
	sched_unsigned count;			/* acct */
	sched_unsigned ccount_delta;		/* acct */
	sched_unsigned ccount;			/* acct */

	sched_unsigned pct_usage;		/* update_invpri */
	sched_unsigned mininvpri;		/* update_invpri */
	sched_unsigned maxinvpri;		/* update_invpri */
} SCHED_D;

#define _SSETLATENCY(p_, ticks_per_minute, borrow_per_minute_)		\
do {									\
	sched_unsigned _z;						\
	(p_)->borrowlatency = 1;					\
	_z = 0;								\
	ARCH_SCHED_ADD((ticks_per_minute) / (borrow_per_minute_), (p_)->invpri, (p_)->borrowlatency, _z);						\
	/*__debug_printf("setlatency: %d / %d * %d -> %d, %d\n", (ticks_per_minute), (borrow_per_minute_), (p_)->invpri, (p_)->borrowlatency, _z);*/\
	(p_)->borrowlatency100 = 1;					\
	_z = 0;								\
	ARCH_SCHED_ADD((ticks_per_minute) / (borrow_per_minute_), SELF_INVPRI, (p_)->borrowlatency100, _z);						\
} while (0)

#define DEFAULT_MININVPRI	PRI_INV(100)
#define DEFAULT_MAXINVPRI	PRI_INV(10)

#define SDINIT(s_)							\
do {									\
	(s_)->accumulated_time = 0;					\
	(s_)->pct_usage = 0;						\
	(s_)->mininvpri = DEFAULT_MININVPRI;				\
	(s_)->maxinvpri = DEFAULT_MAXINVPRI;				\
	(s_)->count = 0;						\
	(s_)->count_delta = 0;						\
	(s_)->ccount = 0;						\
	(s_)->ccount_delta = 0;						\
} while (0)

#define SINIT(s_)							\
do {									\
	SDINIT(s_);							\
	(s_)->schedcount = 0;						\
	(s_)->bonus = 0;						\
	INIT_XLIST(&(s_)->runq);					\
	(s_)->flags = 0;						\
} while (0)

#define SDINIT2(s_, total_time, ticks_per_minute)			\
do {									\
	(s_)->last_total_time = (total_time);				\
	(s_)->invpri = (s_)->mininvpri;					\
} while (0)

#define SINIT2(s_, total_time, ticks_per_minute, borrow_per_minute)	\
do {									\
	(s_)->unblock_time = (total_time);				\
	SDINIT2(s_, total_time, ticks_per_minute);			\
	_SSETLATENCY((s_), (ticks_per_minute), (borrow_per_minute));	\
} while (0)

#define SINIT3(s_, p_)							\
do {									\
	(s_)->count = (p_)->schedcount - 1;				\
} while (0)

#define SDINIT3(s_, p_)							\
do {									\
} while (0)

#define SPREEMPT(s_, isroot, parent, do_preempt_, dont_preempt_)	\
do {									\
	SCHED *_s = (s_);						\
	if (__unlikely((sched_signed)(_s->ccount - _s->schedcount) > (sched_signed)_s->borrowlatency100)) {						\
		if (__likely(XLIST_EMPTY(&_s->runq))) {			\
			_s->schedcount = _s->ccount;			\
		} else {						\
			/*__debug_printf("preempt disabled at top level %d, %s: %d > %d\n", GET_STRUCT(_s, PROC, sch)->depth, GET_STRUCT(_s, PROC, sch)->jobname, _s->ccount - _s->schedcount, _s->borrowlatency100);*/\
			goto dont_preempt_;				\
		}							\
	}								\
	while (!isroot(_s)) {						\
		SCHED *_p = parent(_s);					\
		if (__unlikely((sched_signed)(_s->count - _p->schedcount) > (sched_signed)_s->borrowlatency)) {						\
			if (__likely(XLIST_EMPTY(&_p->runq)) && __likely(!(_p->flags & SCHED_READY))) {							\
				_p->schedcount = _s->count;		\
			} else {					\
				/*__debug_printf("preempt disabled at level %d, %s: %d > %d\n", GET_STRUCT(_s, PROC, sch)->depth, GET_STRUCT(_s, PROC, sch)->jobname, _s->count - _p->schedcount, _s->borrowlatency);*/\
				goto dont_preempt_;			\
			}						\
		}							\
		_s = _p;						\
	}								\
	goto do_preempt_;						\
} while (0)

#define SACCOUNT_ACCUM_(run_time)					\
	if (!((_s)->flags & SCHED_ONLIST)) (_s)->accumulated_time += (run_time);

#define SACCOUNT_NONE_(run_time)

#define SACCOUNT_(type_, accum_, s_, run_time, total_time, ticks_per_minute, isroot, parent, bonus_, use_bonus_)					\
do {									\
	type_ *_s = (s_);						\
	ARCH_SCHED_ADD((run_time), SELF_INVPRI, _s->ccount, _s->ccount_delta);\
	while (!isroot(_s)) {						\
		accum_(run_time)					\
		if (__unlikely((total_time) - _s->last_total_time >= (ticks_per_minute))) update_invpri_##type_(_s, (total_time), (ticks_per_minute));	\
		if (!__is_constant(run_time) || run_time) {		\
			if (use_bonus_) {				\
				sched_unsigned _rt = (run_time) >> 1;	\
				sched_signed _rtt = _s->bonus_ - _rt;	\
				if (_rtt < 0) {				\
					_rt -= _rtt;			\
					_s->bonus_ = 0;			\
				} else {				\
					_s->bonus_ = _rtt;		\
				}					\
				ARCH_SCHED_ADD(_rt, _s->invpri, _s->count, _s->count_delta);								\
			} else {					\
				ARCH_SCHED_ADD((run_time), _s->invpri, _s->count, _s->count_delta);							\
			}						\
		}							\
		_s = parent(_s);					\
	}								\
} while (0)

#define SACCOUNT(s_, run_time, total_time, ticks_per_minute, isroot, parent)\
do {									\
	SACCOUNT_(SCHED, SACCOUNT_ACCUM_, s_, run_time, total_time, ticks_per_minute, isroot, parent, bonus, 1);					\
} while (0)

#define SDACCOUNT(s_, run_time, total_time, ticks_per_minute, isroot, parent)\
do {									\
	SACCOUNT_(SCHED_D, SACCOUNT_NONE_, s_, run_time, total_time, ticks_per_minute, isroot, parent, count /* doesn't matter, just shut up error */, 0);\
} while (0)

#define SSELECT(s_, v_, fn_)						\
do {									\
	SCHED *_s = (s_), *_q, *_b;					\
	sched_unsigned bc = 0; /* warning go away */			\
	sub:								\
	_b = NULL;							\
	if (_s->flags & SCHED_READY) bc = _s->ccount, _b = _s;		\
	XLIST_FOR_EACH(_q, &_s->runq, SCHED, rentry) {			\
		if (!_b || (sched_signed)(bc - _q->count) >= 0) _b = _q, bc = _q->count;								\
	}								\
	if (__likely(_b != NULL)) {					\
		sched_unsigned _q;					\
		if (__likely(_b != _s)) {				\
			if (__unlikely((sched_signed)(_b->count - _s->schedcount) < 0)) {								\
				_b->bonus -= _b->count - _s->schedcount;\
				if (__unlikely(_b->bonus > _s->borrowlatency100 * MAXIMUM_BONUS_MULT)) {						\
					_b->bonus = _s->borrowlatency100 * MAXIMUM_BONUS_MULT;								\
				}					\
				_b->count = _s->schedcount;		\
			} else {					\
				_q = _s->schedcount;			\
				_s->schedcount = _b->count;		\
				if (__unlikely(((_s->schedcount ^ _q) & SCHED_REFRESH_MASK) != 0)) fn_(_s);						\
			}						\
			_s = _b;					\
			goto sub;					\
		} else {						\
			if (__unlikely((sched_signed)(_b->ccount - _b->schedcount) < 0)) {								\
				_b->ccount = _b->schedcount;		\
			} else {					\
				_q = _b->schedcount;			\
				_b->schedcount = _b->ccount;		\
				if (__unlikely(((_b->schedcount ^ _q) & SCHED_REFRESH_MASK) != 0)) fn_(_b);						\
			}						\
		}							\
	}								\
	(v_) = _b;							\
} while (0)

#define SSETLIST(s_, total_time, parent, isroot)			\
do {									\
	SCHED *_s = (s_), *_p;						\
	while (__unlikely(!(_s->flags & SCHED_ONLIST))) {		\
		_s->unblock_time = total_time;				\
		_s->flags |= SCHED_ONLIST;				\
		_p = parent(_s);					\
		ADD_TO_XLIST(&_p->runq, &_s->rentry);			\
		_s = _p;						\
	}								\
} while (0)

#define SSETREADY(s_)							\
do {									\
	SCHED *_s = (s_);						\
	_s->flags |= SCHED_READY;					\
} while (0)

#define SUNSETREADY(s_, total_time, parent, isroot)			\
do {									\
	SCHED *_s = (s_);						\
	_s->flags &= ~SCHED_READY;					\
	while (__unlikely((_s->flags & (SCHED_READY | SCHED_ONLIST)) == SCHED_ONLIST) && __likely(XLIST_EMPTY(&_s->runq))) {				\
		_s->accumulated_time += (total_time) - _s->unblock_time;\
		_s->flags = 0;						\
		DEL_FROM_LIST(&_s->rentry);				\
		_s = parent(_s);					\
		if (isroot(_s)) break;					\
	}								\
} while (0)

#define SDDONE(s_, m_)							\
do {									\
} while (0)

#define SDONE(s_, m_)							\
do {									\
	SDDONE(s_, m_);							\
	if ((s_)->flags & (SCHED_READY | SCHED_ONLIST))			\
		KERNEL$SUICIDE("FREEING ACTIVE SCHEDULER " m_ ", FLAGS %d", (s_)->flags);								\
} while (0)

#define SREFRESH(s_, for_all_subnodes, for_all_subnodes_tail)		\
do {									\
	SCHED *_s = (s_), *_ss;						\
	if (__likely((sched_signed)(_s->ccount - (_s->schedcount - _s->borrowlatency100 * MAXIMUM_BONUS_MULT)) < 0))					\
		_s->ccount = _s->schedcount - _s->borrowlatency100 * MAXIMUM_BONUS_MULT;								\
	for_all_subnodes(_ss, _s) {					\
		if (__likely((sched_signed)(_ss->count - (_s->schedcount - _ss->borrowlatency * MAXIMUM_BONUS_MULT)) < 0))				\
			_ss->count = _s->schedcount - _ss->borrowlatency * MAXIMUM_BONUS_MULT;								\
	} for_all_subnodes_tail(_ss, _s);				\
} while (0)

#define SGEN_UPDATE_ACCUM						\
	if (_s->flags & SCHED_ONLIST) {					\
		_s->accumulated_time += total_time - _s->unblock_time;	\
		_s->unblock_time = total_time;				\
	}

#define SGEN_FIXUP_COUNTS						\
	_s->count -= _s->count >> 2;					\
	_s->ccount -= _s->ccount >> 2;

#define SGEN_UPDATE(type_, accum_, setlat_)				\
static void update_invpri_##type_(type_ *_s, sched_unsigned total_time, sched_unsigned ticks_per_minute)						\
{									\
	accum_								\
	/*__debug_printf("accumulated time: %u\n", _s->accumulated_time);*/\
		/* max 64 iterations */					\
	if (__unlikely((total_time - _s->last_total_time) >> 6 > ticks_per_minute)) {									\
		_s->pct_usage = 0;					\
		_s->last_total_time = total_time;			\
		_s->accumulated_time = 0;				\
	} else do {							\
		sched_unsigned _acct;					\
		sched_unsigned _pct;					\
		if (__likely(_s->accumulated_time <= ticks_per_minute)) {\
			_acct = _s->accumulated_time;			\
			_s->accumulated_time = 0;			\
		} else {						\
			_acct = ticks_per_minute;			\
			_s->accumulated_time -= ticks_per_minute;	\
		}							\
		_pct = _s->pct_usage;					\
		_pct -= _pct >> 5;					\
		_pct += _acct >> 5;					\
		_s->pct_usage = _pct;					\
	} while (__unlikely(total_time - (_s->last_total_time += ticks_per_minute) >= ticks_per_minute));						\
	_s->invpri = _s->mininvpri + (__u32)(((__u64)_s->maxinvpri - (__u64)_s->mininvpri) * _s->pct_usage / ticks_per_minute);				\
	/*__debug_printf("minpri: %u / %u\n", PRI_INV(_s->mininvpri), _s->mininvpri);									\
	__debug_printf("maxpri: %u / %u\n", PRI_INV(_s->maxinvpri), _s->maxinvpri);									\
	__debug_printf("pri: %u / %u\n", PRI_INV(_s->invpri), _s->invpri);\
	__debug_printf("pct_usage: %u, ticks_per_minute: %u\n", _s->pct_usage, (ticks_per_minute));*/							\
	setlat_;							\
}

#define SCHED_GEN_UPDATE(borrow_per_minute_)	SGEN_UPDATE(SCHED, SGEN_UPDATE_ACCUM, _SSETLATENCY(_s, ticks_per_minute, borrow_per_minute_))
#define SCHED_D_GEN_UPDATE	SGEN_UPDATE(SCHED_D, SGEN_FIXUP_COUNTS, )

typedef struct {
	SCHED_D sch;
	PROC *p;
} IO_SCHED;

sched_unsigned KERNEL$GET_SCHED_TICKS(void);
int KERNEL$ALLOC_IOSCHED(void);
void KERNEL$FREE_IOSCHED(int i);
void KERNEL$ACCOUNT_IOSCHED(PROC *p, int sched, sched_unsigned start, sched_unsigned now);
sched_signed KERNEL$COMPARE_IOSCHED(PROC *p1, int s1, PROC *p2, int s2);

#endif
