#ifndef __SPAD_TTY_H
#define __SPAD_TTY_H

#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIBPROC.H>
#include <SYS/TYPES.H>
#include <ENDIAN.H>

__BEGIN_DECLS

#define N_CONTROLS	64	/* if you change this, check all usage of tty->control_count */

struct tty_detach_seq {
	__u64 tty_seq;
	__u64 detach_seq;
};

struct tty_control_state {
	__u64 tty_seq;
	__u64 detach_seq;
	__u64 control_seq;
};

struct tty_window_size {
	__u64 tty_seq;
	__u64 detach_seq;
	unsigned ws_col;
	unsigned ws_row;
	unsigned ws_xpixel;
	unsigned ws_ypixel;
};

struct mouse_state {
	int x, y, z;		/* scaled movement */
	int rx, ry, rz;		/* unscaled movement */
	int wx, wy, wz;		/* wheel movement */
	unsigned buttons;	/* buttons */
};

struct mouse_info {
	int buttons;
	int wheels;
	int axes;
};

struct tty_videomode {
	unsigned x;
	unsigned y;
	unsigned bpp;
	unsigned scanline;
	unsigned memsize;
	unsigned pages;
	unsigned images;
	int text;
	__u64 phys;
};

struct tty_videomode_params {
	unsigned vm;
	unsigned pixel_clock;
	unsigned h;
	unsigned h_sync_start;
	unsigned h_sync_end;
	unsigned h_total;
	unsigned v;
	unsigned v_sync_start;
	unsigned v_sync_end;
	unsigned v_total;
	unsigned flags;
};

struct tty_keyboard_params {
	int repeat_delay;
	int repeat_rate;
	char soft;
};

struct console_mouse_request {
	char drag;
	char update;
	int x, y;
	int start_x, start_y;
	int min_x, max_x;
	int min_y, max_y;
};

struct scroll_request {
	int flags;
	int n;
};

#define SCROLL_HALFPAGES	1
#define SCROLL_DRAGGING		2

/* don't change --- corresponds with VESA 3.0 bits */
#define VIDEO_PARAMS_DOUBLE_SCAN	1
#define VIDEO_PARAMS_INTERLACED		2
#define VIDEO_PARAMS_HSYNC_NEG		4
#define VIDEO_PARAMS_VSYNC_NEG		8

#define CONSOLE_CMD_INSTALL_DRIVER	1
#define CONSOLE_CMD_RELEASE_DRIVER	2
#define CONSOLE_CMD_GET			16

typedef long videomode_call_t(int cmd, int vc, ...);

#define VIDEOMODE_RELEASE		0
#define VIDEOMODE_GET_MODE		1
#define VIDEOMODE_SET_MODE		2
#define VIDEOMODE_GET_PAGE		3
#define VIDEOMODE_GET_PARAM		4
#define VIDEOMODE_WAITACTIVE		5
#define VIDEOMODE_WAITACTIVE_WAKE	6
#define VIDEOMODE_SET_PALETTE		7
#define VIDEOMODE_UNMAP_PAGES		8
#define VIDEOMODE_WAITRETRACE		9
#define VIDEOMODE_WAITRETRACE_SET_DISPLAY_START	10
#define VIDEOMODE_SET_DISPLAY_START	11
#define VIDEOMODE_GET_PIXEL_CLOCK	12
#define VIDEOMODE_AVAIL_ACCEL		13
#define VIDEOMODE_DO_ACCEL		14
#define VIDEOMODE_CAN_MMAP		15
#define VIDEOMODE_GET_FONT_MODE		16
#define VIDEOMODE_SET_FONT		17
#define VIDEOMODE_GET_DEFAULT_CHARSET	18

union accel_param {
	struct {
		int x, y, w, h;
		__u32 c;
	} fillbox;
	struct {
		int x1, y1, x2, y2, w, h;
	} screencopy;
};

struct driver_function {
	char *name;
	void *ptr;
};

struct video_mode {
	unsigned mode;
	unsigned x, y, bpp, scanline;
	unsigned memsize;
	unsigned pages;
	unsigned images;
	int winnum_start, winnum_end;
	int winshift;
	int text;
	int vga_compatible;
	__p_addr phys;
	unsigned total_mem;
};

struct font_blob {
	BLOB_HEAD;
	__u8 charset[32];
	__u8 fontfile[32];
	__u8 xlate[4][256];
	__u8 w, h;
	unsigned long data[1];
	/* variable-size font data, one character this size in longs
	(w * h + 8 * sizeof(unsigned long) - 1) / (8 * sizeof(unsigned long))
	*/
};

#define LONGS_PER_CHAR(w, h)	(((w) * (h) + 8 * sizeof(unsigned long) - 1) / (8 * sizeof(unsigned long)))

#define DEFAULT_KEYBOARD_NAME		"PCKBD"

typedef long kbdmode_call_t(int cmd, int kc, ...);
#define KEYBOARD_CMD_GET		0
typedef void kbdmouse_call_t(struct mouse_state *);
#define KEYBOARD_MOUSE_GET		0x100
#define KEYBOARD_MOUSE_PUT		0x101
#define KEYBOARD_EXTKBD_GET		0x102
#define KEYBOARD_EXTKBD_PUT		0x103
typedef void kbdext_call_t(unsigned keycode, unsigned down);
#define KBD_KEY_UP			0	/* these are hardcoded at some places */
#define KBD_KEY_DOWN			1
#define KBD_KEY_DOWN_NO_UP		2	/* up won't be sent ... */
#define KBD_KEY_DOWN_SOFT_REPEAT	3

#define KBD_RELEASE			0
#define KBD_READ			1
#define KBD_WAIT			2
#define KBD_CLEAR_BUFFER		3
#define KBD_SET_MODE			4
#define KBD_SET_MOUSE_MODE		5
#define KBD_CLEAR_MOUSE_QUEUE		6
#define KBD_GET_MOUSE_STATE		7
#define KBD_PULL_MOUSE_STATE		8
#define KBD_WAIT_MOUSE			9
#define KBD_GET_MOUSE_INFO		10
#define KBD_ERASE_CLIPBOARD		11
#define KBD_SET_KEYBOARD_RATE		12
#define KBD_SET_KEYMAP			13
#define KBD_SWITCH_KEYBOARD		14
#define KBD_SET_LEDS			15
#define KBD_GET_LEDS			16

/* from this point on, data are duplicated in KBD_HEAD.I */

#define MAX_KEYMAP_SEQUENCE		5	/* duplicated in KBD_HEAD.I */

struct keymap_entry {
	__u32 if_mod_set;
	__u32 if_mod_reset;
	__u16 if_key;
	__u8 seq;
	__u8 do_sequence[MAX_KEYMAP_SEQUENCE];
	__u32 do_mod_set;
	__u32 do_mod_reset;
};

#define STRING_NOCHANGE		"\000\000\001\000\000"
#define VAL_NOCHANGE		__32CPU2BE(0x00010000)

#define MOD_CAPS_LOCK	0x00000001
#define MOD_NUM_LOCK	0x00000002
#define MOD_ADD_ESC	0x00000004
#define MOD_ALT_NUM	0x00000008
#define MOD_MOUSE_PASTE	0x00000010
#define MOD_NOREINIT	(~(MOD_CAPS_LOCK | MOD_NUM_LOCK))

#define KEY_CODE_MASK	0x03ff
#define KEY_IS_DOWN	0x0400
#define KEY_IS_UP	0x0800
#define KEY_NONEMPTY_OUTPUT	(KEY_IS_DOWN | KEY_IS_UP)
#define KEY_EMPTY_OUTPUT	(KEY_IS_DOWN | KEY_IS_UP | 1)
#define KEY_DOWN_NO_UP	0x1000
#define KEY_DOWN	0x2000
#define KEY_REPEAT	0x4000
#define KEY_UP		0x8000
#define KEY		(KEY_DOWN | KEY_DOWN_NO_UP | KEY_REPEAT)

#define SEQ_MASK	0x3f
#define SEQ_QUIT_PASS	0x40
#define SEQ_QUIT	0x80

/* end of duplication */

struct keymap_blob {
	BLOB_HEAD;
	__u8 keymap[32];
	/* followed by variable number of struct keymap_entry */
};

#define __MAX_BYTES_IN_CHAR		8

#define __TTY_NOMEM_BUFFER_SIZE		32

#define TTYF_MODE_MASK			0x00001e6f
#define TTYF_KBDMODE			0x00000003	/* coresponds with K_* macros */
#define TTYF_RAW			0x00000004
#define TTYF_NOECHO			0x00000008
#define TTYF_NONBLOCK			0x00000010
#define TTYF_MOUSE_MODE			0x00000060
#define TTYF_MOUSE_COPYPASTE		0x00000000
#define TTYF_MOUSE_CHAR			0x00000020
#define TTYF_MOUSE_RAW			0x00000040
#define TTYF_MOUSESELECT		0x00000080
#define TTYF_O_NL_CRNL			0x00000100
#define TTYF_LEDS			0x00000e00
#define TTYF_DEFAULT_LEDS		0x00001000
#define TTYF_NEW			0x40000000

typedef struct __tty TTY;

struct __tty {
	char *buf;		/* input buffer */
	int buf_size;		/* must be power of 2 */
	int h_root;		/* handle of TTY device */

	void (*writeout)(TTY *tty);
		/* when called, write out characters buffer_to_write,buffer_to_write_len and call back TTY$WRITE_DONE --- TTY$WRITE_DONE must not be called directly from writeout() */
	void (*dtty)(TTY *tty);

	LIST_HEAD handles;

	char *buffer;
	int buffer_len;
	int buffer_ptr;

	char *buffer_to_write;
	int buffer_to_write_len;

	char input_displayed;	/* 1 -- input line is displayed on screen
				   0 -- not displayed (due to write)
				   2 -- erase is in progress */

	TIMER write_timer;
	TIMER update_timer;

	WQ write_wait;
	WQ read_wait;
	WQ in_wait;		/* can be used to wait, if TTY$IN returns zero */

	/* the status of line discipline */
	int buf_start;		/* ptr to start of the buffer */
	int buf_updateline;	/* ptr to start of line to be updated (<= buf_line) */
	int buf_needupdate;	/* screen update is needed from this position */
	int buf_line;		/* ptr to start of last line in buffer */
	int buf_cursor;		/* ptr to cursor position on last line */
	int buf_end;		/* ptr to 1st free char in buffer */
	/* NOTE: all these pointers are shifted in insert_in_buffer and in
	   IOCTL_TTY_PUTBACK. if you add another, you must update it there */

	int mode;	/* TTYF_* */

	int bits;
#define TTY_EOF			0x00000001
#define TTY_PUTBACK		0x00000002
#define TTY_AUTH_POS		0x00000004
#define TTY_AUTH_NEG		0x00000008
#define TTY_PERMANENT_EOF	0x00000010
#define TTY_PERMANENT_RAW	0x00000020
#define TTY_TAB			0x00000040
#define TTY_2TAB		0x00000080
#define TTY_DELAY		0x00000100
#define TTY_IN_WRITEOUT		0x00000200
#define TTY_LOCK_WINDOW_SIZE	0x00000400

	char in_state;	/* 0-normal 1-^[ 2-^[[ 3-^[[[ 4-delayed processing */
	int in_number;	/* if state is 2 || 3 */

	int xsize, ysize;	/* size of terminal */
	int xpixels, ypixels;

	int xstart;		/* initial x position (column is copied here) */
	int xpos, ypos;		/* current position of cursor */
	int max_xpos, max_ypos; /* last max position */
	char text_ok;		/* when 1, text is up to date (but cursor not)
				   when 2, cursor is up to date too */
	
	int out_state_buffer_size;
	char out_state_buffer[__MAX_BYTES_IN_CHAR];

	__u64 control_seq[N_CONTROLS];
	__u64 detach_seq[MAX_PROC_DEPTH];
	__u64 tty_seq;
	char last_char;
	WQ control_wait[N_CONTROLS];
	WQ state_wait;

	videomode_call_t *vmode;
	int vc;
	struct tty_videomode_params current_videomode;
	struct tty_keyboard_params current_keyboard_params;
	PROC *font_proc;
	FBLOB *font_blob;
	PROC *keymap_proc;
	FBLOB *keymap_blob;

	kbdmode_call_t *kmode;
	int kc;

	char nomem_buffer[__TTY_NOMEM_BUFFER_SIZE + 1];/* +1 for red zone */

};

/* before calling TTY$CREATE you must set:
	buf	(you must alloc the buffer yourself)
	buf_size (must be power of 2 !)
	writeout
	dtty	(may be NULL)
  TTY$CREATE must be called at spl <= SPL_DEV
*/
int TTY$CREATE(TTY *tty);
/* after calling TTY$CREATE you must call KERNEL$REGISTER_DEVICE with TTY$INIT_ROOT and tty
*/

void TTY$INIT_ROOT(HANDLE *ttyh, void *tty);

/* keyboard input on terminal
   returns number of chars accepted
   if return value < len, terminal queue has overflowed

   TTY$IN must be calles at spl == SPL_TTY
*/
int TTY$IN(TTY *tty, char *str, int str_len);

void TTY$WRITE_DONE(TTY *tty);

	/* you must unregister device before call */
void TTY$DESTROY(TTY *tty);

__END_DECLS

#endif
