#ifndef __SPAD_USB_H
#define __SPAD_USB_H

#include <SPAD/DEV.H>
#include <SYS/TYPES.H>
#include <SPAD/TIMER.H>

typedef struct __usb USB;
typedef struct __usb_dev USB_DEV;
typedef struct __usb_dev_interface USB_DEV_INTERFACE;
typedef struct __usb_endpoint USB_ENDPOINT;
typedef struct __usb_ctrl_request USB_CTRL_REQUEST;
typedef struct __usb_request USB_REQUEST;
typedef struct __usb_args USB_ARGS;

struct __usb_endpoint {
	int (*prepare)(USB_ENDPOINT *, USB_REQUEST *);
	void (*post)(USB_REQUEST *);
	void (*free)(USB_REQUEST *);
	void (*cancel)(USB_REQUEST *);
	void (*clear_stall)(USB_ENDPOINT *);
	USB *usb;
};

#define USB_EP_CTRL		1
#define USB_EP_BULK_IN		2
#define USB_EP_BULK_OUT		3
#define USB_EP_INTR_IN		4
#define USB_EP_INTR_OUT		5
#define USB_EP_ISO_IN		6
#define USB_EP_ISO_OUT		7

#define USB_IS_CTRL(type)	((type) == USB_EP_CTRL)
#define USB_IS_BULK(type)	(((type) & ~1) == 2)
#define USB_IS_INTR(type)	(((type) & ~1) == 4)
#define USB_IS_ISO(type)	(((type) & ~1) == 6)
#define USB_IS_IN(type)		(!((type) & 1))
#define USB_IS_OUT(type)	((type) & 1)

#define USB_LOW_SPEED		187500
#define USB_FULL_SPEED		1500000
#define USB_HIGH_SPEED		60000000

#define USB_FPS			1000

#define USB_REQUEST_HEAD		\
	AST_HEAD;			\
	long status;			\
	int flags;			\
	VDESC v;			\
	unsigned long internal1;	\
	unsigned long internal2

#define USBRQ_ALLOW_SHORT	1
#define USBRQ_TERMINATE_SHORT	2
#define USBRQ_DELAY_INTERRUPT	4
#define USBRQ_QUIET_ERROR	8
#define USBRQ_QUIET_STALL	16
#define USBRQ_NO_FSBR		32

typedef struct {
	__u16 request;
	__u16 w_value;
	__u16 w_index;
	__u16 w_length;
} USB_CTRL_SETUP;

struct __usb_ctrl_request {
	USB_REQUEST_HEAD;
	USB_CTRL_SETUP setup;
};

struct __usb_request {
	USB_REQUEST_HEAD;
};

#define USB_CTRL_DIR_OUT		0
#define USB_CTRL_DIR_IN			0x80
#define USB_CTRL_TYPE_MASK		(0x3 << 5)
#define USB_CTRL_TYPE_STANDARD		(0x0 << 5)
#define USB_CTRL_TYPE_CLASS		(0x1 << 5)
#define USB_CTRL_TYPE_VENDOR		(0x2 << 5)
#define USB_CTRL_RECIP_MASK		0x1f
#define USB_CTRL_RECIP_DEVICE		0x00
#define USB_CTRL_RECIP_INTERFACE	0x01
#define USB_CTRL_RECIP_ENDPOINT		0x02
#define USB_CTRL_RECIP_OTHER		0x03

#define USB_CTRL_RQTYPE_DEVICE		(USB_CTRL_DIR_IN | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_DEVICE)
#define USB_CTRL_RQTYPE_DEVICE_OUT	(USB_CTRL_DIR_OUT | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_DEVICE)
#define USB_CTRL_RQTYPE_INTERFACE	(USB_CTRL_DIR_IN | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_INTERFACE)
#define USB_CTRL_RQTYPE_INTERFACE_OUT	(USB_CTRL_DIR_OUT | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_INTERFACE)
#define USB_CTRL_RQTYPE_ENDPOINT	(USB_CTRL_DIR_IN | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_ENDPOINT)
#define USB_CTRL_RQTYPE_ENDPOINT_OUT	(USB_CTRL_DIR_OUT | USB_CTRL_TYPE_STANDARD | USB_CTRL_RECIP_ENDPOINT)

#define USB_REQ_GET_STATUS		0x0000
#define USB_REQ_CLEAR_FEATURE		0x0100
#define USB_REQ_SET_FEATURE		0x0300
#define USB_REQ_SET_ADDRESS		0x0500
#define USB_REQ_GET_DESCRIPTOR		0x0600
#define USB_REQ_SET_DESCRIPTOR		0x0700
#define USB_REQ_GET_CONFIGURATION	0x0800
#define USB_REQ_SET_CONFIGURATION	0x0900
#define USB_REQ_GET_INTERFACE		0x0A00
#define USB_REQ_SET_INTERFACE		0x0B00
#define USB_REQ_SYNCH_FRAME		0x0C00

#define USB_DEVICE_FEAT_SELF_POWERED	0
#define USB_DEVICE_FEAT_REMOTE_WAKEUP	1
#define USB_DEVICE_FEAT_TEST_MODE	2
#define USB_DEVICE_FEAT_B_HNP_ENABLE	3
#define USB_DEVICE_FEAT_A_HNP_SUPPORT	4
#define USB_DEVICE_FEAT_A_ALT_HNP_SUPPORT 5
#define USB_DEVICE_FEAT_DEBUG_MODE	6
#define USB_DEVICE_STAT_SELF_POWERED	(1 << USB_DEVICE_FEAT_SELF_POWERED)
#define USB_DEVICE_STAT_REMOTE_WAKEUP	(1 << USB_DEVICE_FEAT_REMOTE_WAKEUP)
#define USB_DEVICE_STAT_TEST_MODE	(1 << USB_DEVICE_FEAT_TEST_MODE)
#define USB_DEVICE_STAT_B_HNP_ENABLE	(1 << USB_DEVICE_FEAT_B_HNP_ENABLE)
#define USB_DEVICE_STAT_A_HNP_SUPPORT	(1 << USB_DEVICE_FEAT_A_HNP_SUPPORT)
#define USB_DEVICE_STAT_A_ALT_HNP_SUPPORT (1 << USB_DEVICE_FEAT_A_ALT_HNP_SUPPORT)
#define USB_DEVICE_STAT_DEBUG_MODE	(1 << USB_DEVICE_FEAT_DEBUG_MODE)

#define USB_ENDPOINT_FEAT_HALT		0
#define USB_ENDPOINT_STAT_HALT		(1 << USB_ENDPOINT_FEAT_HALT)

#define USB_DT_DEVICE			0x01
#define USB_DT_CONFIG			0x02
#define USB_DT_STRING			0x03
#define USB_DT_INTERFACE		0x04
#define USB_DT_ENDPOINT			0x05
#define USB_DT_DEVICE_QUALIFIER		0x06
#define USB_DT_OTHER_SPEED_CONFIG	0x07
#define USB_DT_INTERFACE_POWER		0x08
#define USB_DT_OTG			0x09
#define USB_DT_DEBUG			0x0a
#define USB_DT_INTERFACE_ASSOCIATION	0x0b

#define USB_DT_STRING_LANG_ID			0x00
#define USB_DT_STRING_SERIAL_NUMBER		0x01
#define USB_DT_STRING_PRODUCT_DESCRIPTION	0x02
#define USB_DT_STRING_VENDOR_DESCRIPTION	0x03

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u16 bcdUSB;
	__u8 bDeviceClass;
	__u8 bDeviceSubClass;
	__u8 bDeviceProtocol;
	__u8 bMaxPacketSize0;
	__u16 idVendor;
	__u16 idProduct;
	__u16 bcdDevice;
	__u8 iManufacturer;
	__u8 iProduct;
	__u8 iSerialNumber;
	__u8 bNumConfigurations;
} USB_DESCRIPTOR_DEVICE;

#define USB_DESCRIPTOR_DEVICE_SIZE		18
#define USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0	7

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u16 bcdUSB;
	__u8 bDeviceClass;
	__u8 bDeviceSubClass;
	__u8 bDeviceProtocol;
	__u8 bMaxPacketSize0;
	__u8 bNumConfigurations;
	__u8 bReserved;
} USB_DESCRIPTOR_DEVICE_QUALIFIER;

#define USB_DESCRIPTOR_DEVICE_QUIALIFIER_SIZE	10

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u16 wTotalLength;
	__u8 bNumInterfaces;
	__u8 bConfigurationValue;
	__u8 iConfiguration;
	__u8 bmAttributes;
#define USB_CONFIGURATION_ATTRIBUTE_REMOTE_WAKEUP	0x20
#define USB_CONFIGURATION_ATTRIBUTE_SELF_POWERED	0x40
	__u8 bMaxPower;
} USB_DESCRIPTOR_CONFIGURATION;

#define USB_DESCRIPTOR_CONFIGURATION_SIZE	9

/* interface and endpoint descriptors must not contain 16 or 32-bit values ---
   they are unaligned */

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bInterfaceNumber;
	__u8 bAlternateSetting;
	__u8 bNumEndpoints;
	__u8 bInterfaceClass;
	__u8 bInterfaceSubClass;
	__u8 bInterfaceProtocol;
	__u8 iInterface;
} USB_DESCRIPTOR_INTERFACE;

#define USB_DESCRIPTOR_INTERFACE_SIZE		9

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u8 bEndpointAddress;
#define USB_ENDPOINT_ADDRESS_MASK			0x0f
#define USB_ENDPOINT_ADDRESS_IN				0x80
	__u8 bmAttributes;
#define USB_ENDPOINT_ATTRIBUTE_TRANSFER_MASK		0x03
#define USB_ENDPOINT_ATTRIBUTE_TRANSFER_CONTROL		0x00
#define USB_ENDPOINT_ATTRIBUTE_TRANSFER_ISOCHRONOUS	0x01
#define USB_ENDPOINT_ATTRIBUTE_TRANSFER_BULK		0x02
#define USB_ENDPOINT_ATTRIBUTE_TRANSFER_INTERRUPT	0x03
#define USB_ENDPOINT_ATTRIBUTE_SYNC_MASK		0x0c
#define USB_ENDPOINT_ATTRIBUTE_SYNC_NONE		0x00
#define USB_ENDPOINT_ATTRIBUTE_SYNC_ASYNCHRONOUS	0x04
#define USB_ENDPOINT_ATTRIBUTE_SYNC_ADAPTIVE		0x08
#define USB_ENDPOINT_ATTRIBUTE_SYNC_SYNCHRONOUS		0x0c
#define USB_ENDPOINT_ATTRIBUTE_USAGE_MASK		0x30
#define USB_ENDPOINT_ATTRIBUTE_USAGE_DATA		0x00
#define USB_ENDPOINT_ATTRIBUTE_USAGE_FEEDBACK		0x10
#define USB_ENDPOINT_ATTRIBUTE_USAGE_IMPLICIT_FEEDBACK	0x20
	__u8 wMaxPacketSize_lo;
	__u8 wMaxPacketSize_hi;
	__u8 bInterval;
} USB_DESCRIPTOR_ENDPOINT;

#define USB_DESCRIPTOR_ENDPOINT_SIZE		7

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u16 wLANGID[126];
} USB_DESCRIPTOR_STRING_ZERO;

#define USB_DESCRIPTOR_STRING_ZERO_SIZE		sizeof(USB_DESCRIPTOR_STRING_ZERO)

typedef struct {
	__u8 bLength;
	__u8 bDescriptorType;
	__u16 bString[126];
} USB_DESCRIPTOR_STRING;

#define USB_DESCRIPTOR_STRING_SIZE		sizeof(USB_DESCRIPTOR_STRING)


#define USB_CLASS_PER_INTERFACE			0x00
#define USB_CLASS_AUDIO				0x01
#define USB_CLASS_COMM				0x02
#define USB_CLASS_HID				0x03
#define USB_CLASS_PHYSICAL			0x05
#define USB_CLASS_STILL_IMAGE			0x06
#define USB_CLASS_PRINTER			0x07
#define USB_CLASS_MASS_STORAGE			0x08
#define USB_CLASS_HUB				0x09
#define USB_CLASS_CDC_DATA			0x0a
#define USB_CLASS_CSCID				0x0b
#define USB_CLASS_CONTENT_SEC			0x0d
#define USB_CLASS_VIDEO				0x0e
#define USB_CLASS_APP_SPEC			0xfe
#define USB_CLASS_VENDOR_SPEC			0xff


#define USB_MIN_CONFIGURATION		0
#define USB_MAX_CONFIGURATION		256
#define USB_MIN_INTERFACE		0
#define USB_MAX_INTERFACE		256
#define USB_MIN_ALTSETTING		0
#define USB_MAX_ALTSETTING		256

int USB$PARSE_PARAMS(char *opt, char *optend, char *str, USB_ARGS **args);
void USB$FREE_PARAMS(USB_ARGS *args);
USB_DEV_INTERFACE *USB$ATTACH_DRIVER(USB_ARGS *args, int (*test)(USB_DEV_INTERFACE *, long), long data);
void USB$DETACH_DRIVER(USB_DEV_INTERFACE *iface);

/* test function returns one of these */
#define USB_SKIP			0
#define USB_ATTACH_INTERFACE		1
#define USB_ATTACH_CLASS		2
#define USB_ATTACH_DEVICE		3

USB_ENDPOINT *USB$GET_DEFAULT_ENDPOINT(USB_DEV_INTERFACE *iface);
USB_ENDPOINT *USB$GET_ENDPOINT(USB_DEV_INTERFACE *iface, unsigned n, int wanted_type, unsigned max_requests, unsigned max_request_size);
void USB$FREE_ENDPOINT(USB_DEV_INTERFACE *iface, USB_ENDPOINT *endpt);
int USB$SYNC_CTRL(USB_ENDPOINT *endpt, __u16 request, __u16 w_value, __u16 w_index, __u16 w_length, void *ptr, int flags, u_jiffies_lo_t timeout, int retries, USB_DEV_INTERFACE *reset_iface);

typedef struct {
	IORQ_HEAD;
	USB_DEV_INTERFACE *iface;
	jiffies_lo_t wait_time;
} USB_RESET_RQ;
extern IO_STUB USB$RESET_PORT;

int USB$GET_DEVICE_SPEED(USB_DEV_INTERFACE *iface);
USB_DESCRIPTOR_DEVICE *USB$GET_DEVICE_DESCRIPTOR(USB_DEV_INTERFACE *iface);
USB_DESCRIPTOR_CONFIGURATION *USB$GET_CONFIGURATION_DESCRIPTOR(USB_DEV_INTERFACE *iface);
USB_DESCRIPTOR_INTERFACE *USB$GET_INTERFACE_DESCRIPTOR(USB_DEV_INTERFACE *iface);
USB_DESCRIPTOR_ENDPOINT *USB$GET_ENDPOINT_DESCRIPTOR(USB_DEV_INTERFACE *iface, int ep_type, unsigned n);
USB_ENDPOINT *USB$FIND_ENDPOINT(USB_DEV_INTERFACE *iface, int ep_type, unsigned max_requests, unsigned max_request_size);
int USB$REQUEST_DESCRIPTOR(USB_DEV_INTERFACE *iface, __u8 rqtype, __u8 dt, __u8 idx, __u16 windex, void *ptr, unsigned len, unsigned min_len, int flags);
#define USB_DESC_NO_TYPECHECK	1
#define USB_DESC_QUIET_ERROR	2
void *USB$FIND_DESCRIPTOR(USB_DEV_INTERFACE *iface, __u8 dt, int skip, unsigned len);
void USB$FILL_CLEAR_STALL_REQUEST(USB_DEV_INTERFACE *iface, USB_ENDPOINT *ep, USB_CTRL_REQUEST *rq);

u_jiffies_lo_t USB$GET_CTRL_TIMEOUT(USB_DEV_INTERFACE *iface);
int USB$GET_CTRL_RETRIES(USB_DEV_INTERFACE *iface);
int USB$GET_INTERFACE_NUMBER(USB_DEV_INTERFACE *iface);
int USB$GET_ERRORLEVEL(USB_DEV_INTERFACE *iface);
void USB$GET_DEVICE_NAME(char *result, size_t len, USB_DEV_INTERFACE *iface, char *beginning);
int USB$IS_ATTACHED(USB_DEV_INTERFACE *iface);

__const__ char *USB$SPEED_STRING(int speed);
__const__ char *USB$ENDPOINT_TYPE_STRING(int type);

#endif
