#include <SPAD/ALLOC.H>
#include <STDLIB.H>
#include <SYS/STAT.H>
#include <KERNEL/DEV.H>
#include <SPAD/IOCTL.H>
#include <SPAD/LIBC.H>

#include <SPAD/CD.H>

extern AST_STUB CD_OPEN;
extern AST_STUB CD_STAT;
extern AST_STUB CD_ALLOC;

typedef struct {
	CDRQ *rq;
	int h;
	FFILE *f;
	union {
		OPENRQ o;
		struct {
			IOCTLRQ io;
			struct stat stat;
		} s;
		MALLOC_REQUEST m;
	} u;
} CDS;

DECL_IOCALL(KERNEL$CD, SPL_DEV, CDRQ)
{
	CDS *cds;
	if (__unlikely(!RQ->cwd)) {
		if (__unlikely(!(RQ->cwd = malloc(sizeof(CWD))))) {
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$CD, sizeof(CWD));
			RETURN;
		}
		INIT_XLIST(&RQ->cwd->cwds);
	}
	if (__unlikely(!(cds = malloc(sizeof(CDS))))) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$CD, sizeof(CDS));
		RETURN;
	}
	cds->rq = RQ;
	cds->u.o.fn = CD_OPEN;
	cds->u.o.path = RQ->path;
	cds->u.o.cwd = RQ->cwd;
	cds->u.o.flags = _O_NOACCESS | _O_NOPOSIX;
	RETURN_IORQ_CANCELABLE(&cds->u.o, KERNEL$OPEN, RQ);
}

DECL_AST(CD_OPEN, SPL_DEV, OPENRQ)
{
	CDS *cds = GET_STRUCT(RQ, CDS, u.o);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, cds->rq);
	if (__unlikely(cds->u.o.status < 0)) {
		CDRQ *rq = cds->rq;
		rq->status = cds->u.o.status;
		__slow_free(cds);
		RETURN_AST(rq);
	}
	cds->h = cds->u.o.status;
	cds->u.s.io.fn = CD_STAT;
	cds->u.s.io.h = cds->h;
	cds->u.s.io.ioctl = IOCTL_STAT;
	cds->u.s.io.param = 0;
	cds->u.s.io.v.ptr = (unsigned long)&cds->u.s.stat;
	cds->u.s.io.v.len = sizeof(struct stat);
	cds->u.s.io.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&cds->u.s.io, KERNEL$IOCTL, cds->rq);
}

DECL_AST(CD_STAT, SPL_DEV, IOCTLRQ)
{
	CDS *cds = GET_STRUCT(RQ, CDS, u.s.io);
	CDRQ *rq;
	OPENRQ o;
	FFILE *f;
	unsigned s;
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, cds->rq);
	if (__unlikely(cds->u.s.io.status < 0)) {
		KERNEL$FAST_CLOSE(cds->h);
		rq = cds->rq;
		rq->status = cds->u.s.io.status;
		if (rq->status == -ENOOP) rq->status = -ENOTDIR;
		__slow_free(cds);
		RETURN_AST(rq);
	}
	if (__unlikely(!S_ISDIR(cds->u.s.stat.st_mode))) {
		KERNEL$FAST_CLOSE(cds->h);
		rq = cds->rq;
		rq->status = -ENOTDIR;
		__slow_free(cds);
		RETURN_AST(rq);
	}
	rq = cds->rq;
	o.path = rq->path;
	o.cwd = rq->cwd;
	f = open_file_name(&o, &s);
	if (__unlikely(__IS_ERR(f))) {
		if (f == __ERR_PTR(-ENOMEM)) {
			KERNEL$MEMWAIT((IORQ *)rq, KERNEL$CD, s);
			KERNEL$FAST_CLOSE(cds->h);
			__slow_free(cds);
			RETURN;
		}
		KERNEL$FAST_CLOSE(cds->h);
		rq = cds->rq;
		rq->status = __PTR_ERR(f);
		__slow_free(cds);
		RETURN_AST(rq);
	}
	cds->f = f;
	cds->u.m.size = sizeof(CWDE) + strlen(f->path) + 1;
	cds->u.m.fn = CD_ALLOC;
	RETURN_IORQ_CANCELABLE(&cds->u.m, KERNEL$UNIVERSAL_MALLOC, cds->rq);
}

DECL_AST(CD_ALLOC, SPL_DEV, MALLOC_REQUEST)
{
	CDS *cds = GET_STRUCT(RQ, CDS, u.m);
	CDRQ *rq;
	CWDE *cwde, *cwdee;
	char *p;
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, cds->rq);
	if (__unlikely(cds->u.m.status < 0)) {
		__slow_free(cds->f);
		KERNEL$FAST_CLOSE(cds->h);
		rq = cds->rq;
		rq->status = cds->u.m.status;
		__slow_free(cds);
		RETURN_AST(rq);
	}
	cwde = cds->u.m.ptr;
	cwde->name = (char *)(cwde + 1);
	strcpy(cwde->name, cds->f->path);
	__upcase(cwde->name);
	free(cds->f);
	KERNEL$FAST_CLOSE(cds->h);
	p = strchr(cwde->name, ':');
	if (__unlikely(!p)) KERNEL$SUICIDE("CD_ALLOC: FILENAME DOES NOT CONTAIN ':'");
	cwde->lnlen = p - cwde->name;
	cwde->len = strlen(cwde->name);
	rq = cds->rq;
	XLIST_FOR_EACH(cwdee, &rq->cwd->cwds, CWDE, list) {
		if (__likely(cwdee->lnlen == cwde->lnlen) && __likely(!memcmp(cwdee->name, cwde->name, cwdee->lnlen))) {
			DEL_FROM_LIST(&cwdee->list);
			free(cwdee);
			break;
		}
	}
	ADD_TO_XLIST(&rq->cwd->cwds, &cwde->list);
	free(cds);
	rq->status = 0;
	RETURN_AST(rq);
}


char *KERNEL$GET_CWD(CWD *cwd)
{
	CWDE *cwde;
	if (__unlikely(!cwd) || __unlikely(XLIST_EMPTY(&cwd->cwds))) return NULL;
	cwde = LIST_STRUCT(cwd->cwds.next, CWDE, list);
	return cwde->name;
}

void KERNEL$FREE_CWD(CWD *cwd)
{
	if (__likely(cwd != NULL)) {
		while (!XLIST_EMPTY(&cwd->cwds)) {
			CWDE *cwde = LIST_STRUCT(cwd->cwds.next, CWDE, list);
			DEL_FROM_LIST(&cwde->list);
			KERNEL$UNIVERSAL_FREE(cwde);
		}
		free(cwd);
	}
}

