#include <KERNEL/DEV.H>
#include <SPAD/WQ.H>
#include <SPAD/VM.H>
#include <KERNEL/ASM.H>
#include <SPAD/HASH.H>
#include <STRING.H>
#include <ERRNO.H>
#include <KERNEL/PROC.H>
#include <KERNEL/DEV_OFF.H>
#include <SPAD/ALLOC.H>
#include <SPAD/LIBC.H>
#include <STDLIB.H>
#include <SYS/TYPES.H>
#include <FCNTL.H>
#include <SPAD/THREAD.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/BIO.H>
#include <SPAD/PKT.H>
#include <KERNEL/UACCESS.H>
#include <KERNEL/UDATA.H>
#include <SYS/MMAN.H>
#include <KERNEL/VM_ARCH.H>
#include <SPAD/IOCTL.H>
#include <KERNEL/SELECT.H>
#include <SYS/PARAM.H>

#define KERNEL_NO_RESTART_PRIORITY	8
#define MAX_GET_HANDLE_RECURSE	8	/* minimum acceptable value is 3 */
#define LOCKUP_TIMEOUT		TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 1000)

#define SWAPPER_LNM		"SYS$SWAPPER"
#define SWAPPER_LNM_LEN		11
#define BLOCK_LNM		"SYS$BLOCK"

static LNT kernel_lnt;
LNT *LN = &kernel_lnt;

HANDLE **FILES_PTR = NULL;
unsigned N_FILES = 0;

WQ_DECL(handle_block, "KERNEL$HANDLE_BLOCK");

WQ_DECL(KERNEL$LOGICAL_WAIT, "KERNEL$LOGICAL_WAIT");

struct multiopen *multiopen = NULL;

void UNBLOCK_LNT(void)
{
	WQ_WAKE_ALL(&handle_block);
}

static int lookup_user_lnt(PROC *proc, LNT *lnt, char *lnm, int hash, int mode, IORQ *rq, u_jiffies_lo_t j, PROC *orig_proc)
{
	static char lnt_name[__MAX_STR_LEN];
	void *v;
	LNTE *e;
	LIST_ENTRY **le;
	if (__unlikely(!lnt)) return -3;
	if (__unlikely((unsigned long)lnt & (LNT_ALIGN - 1))) {
		edom:
		return -3;
	}
	le = &lnt->hash[hash].next;
	if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
	while (1) {
		rd1:
		if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
		uread_ptr(proc, le, v, {
			if (__likely((unsigned long)rq >= 2)) {
				if (__likely(!VM_FAULT(proc, le, PF_SWAPPER, rq))) goto rd1;
			} else {
				if (!rq) if (__likely(!PROC_OTHER_VM_FAULT(orig_proc, proc, le, PF_SWAPPER))) goto rd1;
			}
			return -1;
		});

		if (v == (void *)UPLACE(UDATA_LIST_END) || __unlikely(v == (void *)&KERNEL$LIST_END)) break;
		if (__unlikely((unsigned long)v & (LNT_ALIGN - 1))) goto edom;

		e = LIST_STRUCT(v, LNTE, hash);
		rd2:
		if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
		uread_str(proc, e->name, lnt_name, __MAX_STR_LEN, 0, v, {
			if (__likely((unsigned long)rq >= 2)) {
				if (__likely(!VM_FAULT(proc, v, PF_SWAPPER, rq))) goto rd2;
			} else {
				if (!rq) if (__likely(!PROC_OTHER_VM_FAULT(orig_proc, proc, v, PF_SWAPPER))) goto rd2;
			}
			return -1;
		});
		if (__unlikely(__IS_ERR(v))) return -2;
		if (__likely(!_strcasecmp(lnm, lnt_name))) {
			int handle;
			rd3:
			if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
			uread_int(proc, &e->handle, handle, {
				if (__likely((unsigned long)rq >= 2)) {
					if (__likely(!VM_FAULT(proc, &e->handle, PF_SWAPPER, rq))) goto rd3;
				} else {
					if (!rq) if (__likely(!PROC_OTHER_VM_FAULT(orig_proc, proc, &e->handle, PF_SWAPPER))) goto rd3;
				}
				return -1;
			});
			if (__unlikely(handle < 0)) return -2;
			if (!(handle & LNTE_PUBLIC) && __unlikely(mode != LN_ALL)) return -2;
			return handle;
		}
		if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > LOCKUP_TIMEOUT) && proc != &KERNEL$PROC_KERNEL) {
			if (__likely((unsigned long)rq >= 2)) {
				rq->status = -EINTR;
				CALL_AST(rq);
			}
			return -1;
		}
		le = &e->hash.next;
	}
	return -3;
}

typedef struct {
	PROC *proc;
	int h;
} PROCH;

/* returns proc != NULL --> translated lnm
	   proc == NULL && h == 1 --> blocked
	   proc == NULL && h < 0 --> error -h
*/
static PROCH translate_lnm(PROC *proc, char *lnm, IORQ *rq, u_jiffies_lo_t j)
{
	PROC *orig_proc = proc;
	char *lnm_orig = lnm;
	PROCH ret;
	char *lnh;
	int hash = 0;
	int mode;
	LNTE *e;
	if (__unlikely(lnm[0] == '@')) {
		lnm++;
		if (__unlikely(lnm[0] == '@')) goto notfound;
	}
	if (__unlikely(!_strcasecmp(lnm, BLOCK_LNM))) {
		ret.proc = proc;
		ret.h = BLOCK_HANDLE;
		goto return_insert_cache;
	}
	lnh = lnm;
	quickcasehash(lnh, *lnh, hash);
	hash &= LNT_HASH_SIZE - 1;
	mode = LN_ALL;
	if (__unlikely(!_strcasecmp(lnm, SWAPPER_LNM))) {
		if (__likely(proc != &KERNEL$PROC_KERNEL)) {
			proc = proc->backptr[1];
			goto pp;
		}
	}

	if (__unlikely(lnm != lnm_orig)) {
		if (__unlikely(proc == &KERNEL$PROC_KERNEL)) goto notfound;
		goto pp;
	}
	while (__likely(proc != &KERNEL$PROC_KERNEL)) {
		int r;
		LNT *lnt;
		rd1:
		if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
		uread_ptr(proc, KUPLACE(UDATA_STRUCT + OFF_UDATA_lnt), lnt, {
			if (__likely((unsigned long)rq >= 2)) {
				if (__likely(!VM_FAULT(proc, (void *)KUPLACE(UDATA_STRUCT + OFF_UDATA_lnt), PF_SWAPPER, rq))) goto rd1;
			} else {
				if (!rq) if (__likely(!PROC_OTHER_VM_FAULT(orig_proc, proc, (void *)KUPLACE(UDATA_STRUCT + OFF_UDATA_lnt), PF_SWAPPER))) goto rd1;
			}
			goto ret_fault;
		});
		r = lookup_user_lnt(proc, lnt, lnm, hash, mode, rq, j, orig_proc);
		if (r >= 0) {
			ret.proc = proc;
			ret.h = r;
			goto return_insert_cache;
		}
		if (__unlikely(r == -1)) goto ret_fault;
		if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
		if (__unlikely(r == -2)) goto notfound;
		pp:
		r = lookup_user_lnt(proc->parent, proc->parent_lnt, lnm, hash, mode, rq, j, orig_proc);
		if (r >= 0) {
			ret.proc = proc->parent;
			ret.h = r;
			goto return_insert_cache;
		}
		if (__unlikely(r == -1)) goto ret_fault;
		if (__likely((unsigned long)rq >= 2)) CHECK_RQ_STATE(rq);
		if (__unlikely(r == -2)) goto notfound;
		mode = proc->ln_mode;
		if (__unlikely(mode == LN_NONE) && _strcasecmp(lnm, SWAPPER_LNM)) goto notfound;
		proc = proc->parent;
	}

	XLIST_FOR_EACH(e, &kernel_lnt.hash[hash], LNTE, hash) if (__likely(!__strcasexcmp(e->name, lnm, lnh))) {
		if (!(e->handle & LNTE_PUBLIC) && mode != LN_ALL) goto notfound;
		ret.proc = &KERNEL$PROC_KERNEL;
		ret.h = e->handle;
		goto return_insert_cache;
	}
	notfound:
	ret.proc = NULL;
	ret.h = -ENOLNM;
	return ret;

	ret_fault:
	ret.proc = NULL;
	ret.h = 1;
	return ret;

	return_insert_cache:
	if (__likely((unsigned long)rq >= 2)) if (__likely(strlen(lnm_orig) < LN_CACHE_NAMELEN)) {
		int i;
		LN_CACHE_ENTRY *lce = &orig_proc->ln_cache[0];
		for (i = 0; i < LN_CACHE_SIZE; i++) {
			orig_proc->ln_cache[i].touch = (orig_proc->ln_cache[i].touch >> 1) | 0x80;
			if (__unlikely(orig_proc->ln_cache[i].touch > lce->touch)) lce = &orig_proc->ln_cache[i];
		}
		strcpy(lce->name, lnm_orig);
		lce->touch = 0xf0;
		lce->h = ret.h;
		lce->proc = ret.proc;
	}
	return ret;
}

/* it is expected that this is not called with zero-length lnm */
static PROCH translate_lnm_cached(PROC *proc, char *lnm, IORQ *rq, u_jiffies_lo_t j)
{
	int i;
	PROCH proch;
	CHECK_RQ_STATE(rq);
	for (i = 0; i < LN_CACHE_SIZE; i++) {
		if (__likely(!_strcasecmp(proc->ln_cache[i].name, lnm))) {
			proc->ln_cache[i].touch <<= 1;
			proch.proc = proc->ln_cache[i].proc;
			proch.h = proc->ln_cache[i].h;
			return proch;
		}
	}
	return translate_lnm(proc, lnm, rq, j);
}

static int write_ln(struct ln_list_internal *ln, char *name, long l)
{
	void *v;
	if (__unlikely(ln->result_left -= l) < 0) return 0;
	wr1:
	uwrite_str(ln->proc, ln->result_addr, name, v, {
		if (ln->success) return 2;
		if (__likely(!PROC_VM_FAULT(ln->proc, v, PF_WRITE))) goto wr1;
		return 1;
	});
	ln->success = 1;
	ln->result_addr += l;
	return 0;
}

/* return:
	< 0 --- error
	0 --- ok
	1 --- page fault, need to wait
	2 --- page fault but some data already read
*/

static int read_table(PROC *pproc, LNT *lnt, int mode, struct ln_list_internal *ln)
{
	char result[__MAX_STR_LEN];
	int r;
	long l;
	int nexta;
	LIST_ENTRY **le;
	char *v;
	LNTE *e;
	int handle;
	PROCH proch;
	next_hash:
	if (__unlikely(ln->hash >= LNT_HASH_SIZE)) {
		ret:
		ln->hash = 0;
		ln->depth++;
		ln->ptr = NULL;
		return 0;
	}
	next_hash_2:
	le = &lnt->hash[ln->hash].next;
	nexta = !ln->ptr;
	next_entry:
	if (__unlikely((unsigned long)le & (LNT_ALIGN - 1))) goto ret;
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - ln->j > LOCKUP_TIMEOUT)) return ln->success + 1;
	rd1:
	uread_ptr(pproc, le, v, {
		if (ln->success) return 2;
		if (__likely(!PROC_OTHER_VM_FAULT(ln->proc, pproc, le, PF_SWAPPER))) goto rd1;
		return 1;
	});
	if (v == (void *)UPLACE(UDATA_LIST_END) || __unlikely(v == (void *)&KERNEL$LIST_END)) {
		if (__likely(nexta)) {
			ln->hash++;
			goto next_hash;
		}
		ln->ptr = NULL;
		goto next_hash_2;
	}
	if (__unlikely((unsigned long)v & (LNT_ALIGN - 1))) goto ret;
	e = LIST_STRUCT(v, LNTE, hash);
	if (!nexta) {
		nexta = ln->ptr == e;
		skip_this:
		le = &e->hash.next;
		goto next_entry;
	}
	rd3:
	uread_int(pproc, &e->handle, handle, {
		if (ln->success) return 2;
		if (__likely(!PROC_OTHER_VM_FAULT(ln->proc, pproc, &e->handle, PF_SWAPPER))) goto rd3;
		return 1;
	});
	if (handle != -1 && !(handle & LNTE_PUBLIC) && mode != LN_ALL) goto skip_this;
	rd2:
	uread_str(pproc, e->name, result, __MAX_STR_LEN, 0, v, {
		if (ln->success) return 2;
		if (__likely(!PROC_OTHER_VM_FAULT(ln->proc, pproc, v, PF_SWAPPER))) goto rd2;
		return 1;
	});
	if (__unlikely(__IS_ERR(v))) return __PTR_ERR(v);
	if (__unlikely(!*result)) return -EINVAL;
	l = v - e->name + 1;
	proch = translate_lnm(ln->proc, result, (void *)(unsigned long)ln->success, ln->j);
	if (__unlikely(!proch.proc)) {
		if (__likely(proch.h < 0)) goto skip_this;
		return ln->success + 1;
	}
	r = write_ln(ln, result, l);
	if (__unlikely(r)) return r;
	if (__unlikely(ln->result_left < 0)) return 0;
	ln->ptr = e;
	goto skip_this;
}

int get_lnm(struct ln_list_internal *ln)
{
	int r, dp, mode;
	LNT *lnt;
	PROC *pproc;
	if (__unlikely(ln->depth == -1)) return 0;
	pproc = ln->proc;
	dp = 0;
	mode = LN_ALL;
	p:
	if (dp == ln->depth) {
		if (__unlikely(r = read_table(pproc->parent, pproc->parent_lnt, mode, ln))) return r;
		if (__unlikely(ln->result_left < 0)) return 0;
	}
	dp++;
	mode = pproc->ln_mode;
	if (__unlikely(mode == LN_NONE)) {
		if (dp == ln->depth) {
			if (__unlikely(r = write_ln(ln, SWAPPER_LNM, SWAPPER_LNM_LEN + 1))) return r;
			if (__unlikely(ln->result_left < 0)) return 0;
		}
		ln->depth = -1;
		return 0;
	}
	pproc = pproc->parent;
	if (dp == ln->depth) {
		if (__likely(pproc != &KERNEL$PROC_KERNEL)) {
			r1:
			uread_ptr(pproc, KUPLACE(UDATA_STRUCT + OFF_UDATA_lnt), lnt, {
				if (ln->success) return 2;
				if (__likely(!PROC_OTHER_VM_FAULT(ln->proc, pproc, (void *)KUPLACE(UDATA_STRUCT + OFF_UDATA_lnt), PF_SWAPPER))) goto r1;
				return 1;
			});
		} else lnt = &kernel_lnt;
		if (__unlikely(r = read_table(pproc, lnt, mode, ln))) return r;
		if (__unlikely(ln->result_left < 0)) return 0;
	}
	if (__unlikely(pproc == &KERNEL$PROC_KERNEL)) {
		ln->depth = -1;
		return 0;
	}
	dp++;
	goto p;
}

/*
   get_handle[_fast]
  	-- called at SPL_DEV (or higher, but raised from SPL_DEV)
  	   returns at SPL of handle it found
   returns: __IS_ERR -- error
	    NULL -- rq is pending on page fault
	    othewise -- HANDLE
 */

static int get_handle_recurse = 0;
static int get_handle_recurse_cut = 0;
static PROC *get_handle_recurse_proc;
static int get_handle_recurse_handle_num;
static int get_handle_recurse_open_flags;

#define UREAD_STR_LEN	256

static char lookup_str[2 * UREAD_STR_LEN];

static int SET_FILE_FLAG(HANDLE *h, IORQ *rq, int flag);

/* reseni rekurze GET_HANDLE->VM_FAULT->GET_HANDLE:
	pokud je pocitadlo == 1, nastavit ho na 2 a ulozit parametry, vratit NULL
	na zacatku zvysit pocitadlo rekurzi
	na konci, pokud se vraci NULL a pocitadlo == 2, obnovit parametry a skocit na zacatek
	podoble, jako u VM_FAULT
*/

HANDLE *GET_HANDLE_TEST(PROC *proc, int handle_num, int incomplete)
{
	HANDLE *h;
	XLIST_FOR_EACH(h, &proc->handles[handle_num & proc->handle_hash_mask], HANDLE, proc_hash) if (__likely((h->handle_num & HANDLE_NUM_MASK) == handle_num)) {
		int s;
		s = KERNEL$SPL;
		if (__unlikely(SPLX_BELOW(s, h->op->spl))) RAISE_SPLX(h->op->spl);
		if (__likely(h->op != &KERNEL$HANDLE_NOOP)) return h;
		LOWER_SPLX(s);
		return incomplete ? h : NULL;
	}
	return NULL;
}

HANDLE *GET_HANDLE_FAST_INTERNAL(PROC *proc, int handle_num)
{
	HANDLE *h;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("GET_HANDLE_FAST_INTERNAL AT SPL %08X", KERNEL$SPL);
#endif
	XLIST_FOR_EACH(h, &proc->handles[handle_num & proc->handle_hash_mask], HANDLE, proc_hash) if (__likely((h->handle_num & HANDLE_NUM_MASK) == handle_num)) {
		if (__unlikely((unsigned long)h - (unsigned long)&proc->reserved_handles >= sizeof(proc->reserved_handles))) {
			RAISE_SPL(SPL_CACHE);
			KERNEL$CACHE_TOUCH_VM_ENTITY(HANDLE_TO_VMENTITY(h), proc);
#ifndef __i386__
			LOWER_SPL(SPL_DEV);
#else
			{
		/* GCC is nuts --- it does common subexpression elimination
		   on constant SPL_X(SPL_DEV), moves it up and eats a register
		   for that. Only "volatile" keyword will prevent it from doing
		   that */
				int x;
				__asm__ volatile ("MOVL %1, %0" : "=a"(x) : "i"(SPL_X(SPL_DEV)));
				LOWER_SPLX(x);
			}
#endif
		}
		RAISE_SPLX(h->op->spl);
		if (__likely(h->op != &KERNEL$HANDLE_NOOP)) return h;
		LOWER_SPL(SPL_DEV);
		break;
	}
	return NULL;
}

HANDLE *GET_HANDLE(PROC *proc, int handle_num, int open_flags, IORQ *rq)
{
#define ghreturn(x)	do { retval = x; goto retrn; } while (0)
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	HANDLE *retval;
	void *q;
	int p;
	char *s;
	HANDLE *h, *hp;
	PROCH proch;
	int noret = 0;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("GET_HANDLE AT SPL %08X", KERNEL$SPL);
#endif

	if (__unlikely(++get_handle_recurse >= MAX_GET_HANDLE_RECURSE)) {
		if (__likely(get_handle_recurse == MAX_GET_HANDLE_RECURSE) && __likely(!get_handle_recurse_cut)) {
			get_handle_recurse_proc = NULL;	/* just catch bugs */
			get_handle_recurse_cut = 1;
		} else {
			KERNEL$SUICIDE("GET_HANDLE: DOUBLE RECURSIVE FAULT (%d/%d). FIRST FAULT WAS (%d/%d). GET_HANDLE_RECURSE %d, GET_HANDLE_RECURSE_CUT %d", proc->depth, handle_num, get_handle_recurse_proc->depth, get_handle_recurse_handle_num, get_handle_recurse, get_handle_recurse_cut);
		}
		ghreturn(NULL);
	}

	again:
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > LOCKUP_TIMEOUT) && proc != &KERNEL$PROC_KERNEL && handle_num != SWAPPER_HANDLE) {
		rq->status = -EINTR;
		CALL_AST(rq);
		ghreturn(NULL);
	}
	CHECK_RQ_STATE(rq);
	SWITCH_PROC_ACCOUNT(proc, SPL_X(SPL_DEV));
#if __DEBUG >= 1
	if (__unlikely(ARCH_PROC_INVALID(proc)))
		KERNEL$SUICIDE("GET_HANDLE: INVALID PROCESS %p (HANDLE %d, FLAGS %d)", proc, handle_num, open_flags);
#endif
	if (__unlikely(handle_num == BLOCK_HANDLE)) {
		if (__unlikely(open_flags & _O_NOWAIT) && __likely(proc == &KERNEL$PROC_KERNEL)) {
			rq->status = -EWOULDBLOCK;
			CALL_AST(rq);
			ghreturn(NULL);
		}
		block:
		WQ_WAIT_F(&handle_block, rq);
		ghreturn(NULL);
	}
	if (__likely(proc != &KERNEL$PROC_KERNEL)) {
		int n;
		FFILE **fbase;
		FFILE *file, *xfile;
		PROC *xproc;
		WQ *wq;

		XLIST_FOR_EACH(h, &proc->handles[handle_num & proc->handle_hash_mask], HANDLE, proc_hash) if (__likely((h->handle_num & HANDLE_NUM_MASK) == handle_num)) {
			if (__likely(h->file_addrspace != NULL)) goto goth;
			goto aaa;
		}
		h = NULL;

		aaa:

		if (__unlikely(handle_num == SWAPPER_HANDLE)) {
			file = NULL;
			goto file_ok;
		}

		rd2:
		uread_int(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, n_handles)), n, {
			if (__likely(!VM_FAULT(proc, (void *)KUPLACE(UDATA_STRUCT + __offsetof(UDATA, n_handles)), PF_SWAPPER, rq))) {
				CHECK_RQ_STATE(rq);
				goto rd2;
			}
			goto retn;
		});

		if (__unlikely(handle_num >= n) || __unlikely((unsigned)handle_num >= MAX_HANDLE))
			ghreturn(__ERR_PTR(-EBADF));

		rd3:
		uread_ptr(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, handles)), fbase, {
			if (__likely(!VM_FAULT(proc, (void *)KUPLACE(UDATA_STRUCT + __offsetof(UDATA, handles)), PF_SWAPPER, rq))) {
				CHECK_RQ_STATE(rq);
				goto rd3;
			}
			goto retn;
		});
		if (__unlikely((unsigned long)fbase & (PTR_ALIGN - 1))) {
			ghreturn(__ERR_PTR(-EDOM));
		}
		rd4:
		uread_ptr(proc, fbase + handle_num, file, {
			if (__likely(!VM_FAULT(proc, fbase + handle_num, PF_SWAPPER, rq))) {
				CHECK_RQ_STATE(rq);
				goto rd4;
			}
			goto retn;
		});

		if (__unlikely(!file)) ghreturn(__ERR_PTR(-EBADF));

		file = (FFILE *)((unsigned long)file & ~1UL);
		if (__unlikely((unsigned long)file & (FILE_ALIGN - 1))) {
			ghreturn(__ERR_PTR(-EDOM));
		}

		file_ok:
		if (!h) {
			/*test_all_handles();*/
			h = PROC_ALLOC_HANDLE(proc, handle_num, &wq);
			if (__unlikely(!h)) {
				WQ_WAIT_F(wq, rq);
				ghreturn(NULL);
			}
			/*test_all_handles();*/
		}
		h1:

		h->name = file;
		xproc = proc;
		xfile = file;

		go_to_parent:
		h->file_addrspace = xproc;
		h->file = xfile;

		SWITCH_PROC_ACCOUNT(proc, SPL_X(SPL_DEV));

		if (__unlikely(handle_num == SWAPPER_HANDLE)) goto goth;

		uread_int(xproc, &xfile->flags, n, {
			h->file_addrspace = NULL;
			if (__likely(!VM_FAULT(xproc, &xfile->flags, PF_SWAPPER, rq))) {
				CHECK_RQ_STATE(rq);
				goto h1;
			}
			goto retn;
		});

		if (__unlikely(n & _O_KRNL_FORKED)) {
			int idx;
			uread_int(xproc, &xfile->pos, idx, {
				h->file_addrspace = NULL;
				if (__likely(!VM_FAULT(xproc, &xfile->pos, PF_SWAPPER, rq))) {
					CHECK_RQ_STATE(rq);
					goto h1;
				}
				goto retn;
			});
			if (__unlikely((unsigned)idx >= xproc->parent_forktable_n)) {
				h->file_addrspace = NULL;
				ghreturn(__ERR_PTR(-EBADF));
			}
			uread_ptr(xproc->parent, &xproc->parent_forktable[idx], xfile, {
				h->file_addrspace = NULL;
				if (__likely(!VM_FAULT(xproc->parent, &xproc->parent_forktable[idx], PF_SWAPPER, rq))) {
					CHECK_RQ_STATE(rq);
					goto h1;
				}
				goto retn;
			});
			if (__unlikely((unsigned long)xfile & (FILE_ALIGN - 1))) {
				h->file_addrspace = NULL;
				ghreturn(__ERR_PTR(-EDOM));
			}
			xproc = xproc->parent;
			goto go_to_parent;
		}
		h->handle_num = (h->handle_num & HANDLE_NUM_MASK) | (((unsigned)n << __BSF_CONST(HANDLE_NUM_READ)) & (HANDLE_NUM_READ | HANDLE_NUM_WRITE));

		goto goth;
	}
	if (__unlikely((unsigned)handle_num >= N_FILES))
		KERNEL$SUICIDE("GET_HANDLE(%d): INVALID HANDLE", handle_num);
	h = FILES_PTR[handle_num];
	if (__unlikely(!h))
		KERNEL$SUICIDE("GET_HANDLE(%d): HANDLE NOT OPEN", handle_num);
	h->file_addrspace = h->name_addrspace;
	h->file = h->name;
	goth:
#if __DEBUG >= 1
	if (__unlikely(!h->op))
		KERNEL$SUICIDE("GET_HANDLE: INVALID HANDLE %d", handle_num);
#endif
	if (__likely(h->op != &KERNEL$HANDLE_NOOP)) {
		RAISE_SPLX(h->op->spl);
		if (__likely(h->op != &KERNEL$HANDLE_NOOP)) goto ret_h;
		LOWER_SPL(SPL_DEV);
	}

	INIT_XLIST(&h->child_list);
	VOID_LIST_ENTRY(&h->child_entry);

	if (__unlikely(!rq)) {
		h->file_addrspace = NULL;
		ghreturn(NULL);
	}

	if (__unlikely(handle_num == SWAPPER_HANDLE)) {
		proch = translate_lnm_cached(proc, SWAPPER_LNM, rq, j);
		s = NULL;	/* against warning */
		goto got_proch;
	}

	rd10:
	uread_str(h->name_addrspace, h->name->path, lookup_str, UREAD_STR_LEN, ':', s, {
		if (__likely(!VM_FAULT(h->name_addrspace, s, PF_SWAPPER, rq))) {
			CHECK_RQ_STATE(rq);
			goto rd10;
		}
		goto fault;
	});
	if (__unlikely(__IS_ERR(s))) goto ret_s;
	if (!*lookup_str) {
		h->file_addrspace = NULL;
		ghreturn(__ERR_PTR(-ENOENT));
	}
	proch = translate_lnm_cached(proc, lookup_str, rq, j);
	got_proch:
	if (__unlikely(!proch.proc)) {
		/*if (proch.h == -ENOLNM && open_flags & _O_LNWAIT) __debug_printf("block (x%x) at %s, %s, %d\n", handle_num, lookup_str, proc->jobname, proc->depth);*/
		goto np;
	}
	CHECK_RQ_STATE(rq);

	if (__unlikely(proch.h & LNTE_BLOCK_IF_NOT_FOUND)) {
		/*__debug_printf("lnwait (x%x,x%x:o%o) at %s, %s, %d\n", handle_num, proch.h, open_flags, lookup_str, proc->jobname, proc->depth);*/
		open_flags |= _O_LNWAIT;
	}
	GET_HANDLE_FAST(hp, proch.proc, proch.h & LNTE_HANDLE_MASK, open_flags & (_O_NOWAIT | _O_LNWAIT), rq, {
		/* warning --- h might have been freed. Must recheck */
		if ((__likely(proc != &KERNEL$PROC_KERNEL) && __unlikely(h != GET_HANDLE_TEST(proc, handle_num, 1))) || __unlikely(h->op != &KERNEL$HANDLE_NOOP) || __unlikely(!h->file_addrspace)) {
			h = NULL;
		}
		if (__unlikely(hp == NULL)) {
			goto fault;
		}
		CHECK_RQ_STATE(rq);
		if (__unlikely(__IS_ERR(hp))) {
			goto nh;
		}
		if (__unlikely(!h)) {
			goto lo_spl_call_io_ret;
		}
	});

	/* set up handle file pointer according to proch.h */
	if (__unlikely((proch.h & LNTE_FILE_SHARE) != 0)) {
		if (__unlikely(SET_FILE_FLAG(h, rq, _O_NOFASTSEEK))) {
			goto fault;
		}
		h->file_addrspace = hp->file_addrspace;
		h->file = hp->file;
	}

	/* ok, we have the parent handle here and we can start translation */

	if (__likely(hp->op->clone_handle != NULL)) {
		q = hp->op->clone_handle(hp, h, open_flags);
		if (__unlikely(q != NULL)) {
			h->op = &KERNEL$HANDLE_NOOP;
			/* leave will be called but it is NULL now */
			goto lookup_err;
		}
	} else {
		h->op = hp->op;
		h->fnode = hp->fnode;
		h->flags = hp->flags;
		h->flags2 = hp->flags2;
	}

	if (__unlikely(handle_num == SWAPPER_HANDLE)) goto done;

	next_dir:
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > LOCKUP_TIMEOUT) && proc != &KERNEL$PROC_KERNEL) {
		rq->status = -EINTR;
		CALL_AST(rq);
		goto fault2;
	}
	SWITCH_PROC_ACCOUNT(proc, h->op->spl);
	uread_str(h->name_addrspace, s, lookup_str, UREAD_STR_LEN, '/', s, {
		ureadflt2:
		if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
		h->op = &KERNEL$HANDLE_NOOP;
		LOWER_SPL(SPL_DEV);
		if (__likely(!VM_FAULT(h->name_addrspace, s, PF_SWAPPER, rq))) {
			CHECK_RQ_STATE(rq);
			CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
		}
		goto fault;
	});
	if (__unlikely(__IS_ERR(s))) goto ret_s_2;
	if (__unlikely(open_flags & _O_RENAME)) {
		char *es;
		uread_str(h->name_addrspace, s, lookup_str + UREAD_STR_LEN, UREAD_STR_LEN, '/', es, {
			s = es;
			goto ureadflt2;
		});
		if (__unlikely(__IS_ERR(es))) {
			q = es;
			goto lookup_err_r;
		}
		if (!lookup_str[UREAD_STR_LEN]) {
			HANDLE *rh;
			if (__unlikely(!h->op->rename)) goto no_lookup;
			if (__unlikely(h->name_addrspace == &KERNEL$PROC_KERNEL)) {
				if (__unlikely((unsigned)((OPENRQ *)rq)->rename_handle >= N_FILES)) {
					KERNEL$SUICIDE("GET_HANDLE: INVALID HANDLE %d AS RENAME SOURCE", ((OPENRQ *)rq)->rename_handle);
				}
				rh = FILES_PTR[((OPENRQ *)rq)->rename_handle];
				if (__unlikely(!rh))
					KERNEL$SUICIDE("GET_HANDLE: RENAME SOURCE HANDLE %d NOT OPEN", ((OPENRQ *)rq)->rename_handle);
				if (__unlikely(SPLX_BELOW(KERNEL$SPL, h->op->spl))) RAISE_SPLX(h->op->spl);
				if (__unlikely(rh->op == &KERNEL$HANDLE_NOOP)) {
					LOWER_SPLX(h->op->spl);
					goto la;
				}
			} else {
				rh = GET_HANDLE_TEST(h->name_addrspace, ((OPENRQ *)rq)->rename_handle, 0);
			}
			if (__unlikely(!rh)) {
				la:
				if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
				h->op = &KERNEL$HANDLE_NOOP;
				h->file_addrspace = NULL;
				LOWER_SPL(SPL_DEV);
				if (__unlikely(get_handle_recurse_cut)) goto retn;
				handle_num = ((OPENRQ *)rq)->rename_handle;
				open_flags = 0;
				noret = 1;
				goto again;
			}
			q = h->op->rename(h, lookup_str, rh, rq);
			if (__likely(!q)) q = __ERR_PTR(-ESRCH);
			goto lookup_err;
		}
	}
	if (!*lookup_str) goto done;
	if (__unlikely(!h->op->lookup)) goto no_lookup;
	q = h->op->lookup(h, lookup_str, open_flags);
	/*__debug_printf("lookup(%s)->%p.", lookup_str, q);*/
	if (__likely(!q)) goto next_dir;

	if (__unlikely(q != __ERR_PTR(-ENOENT))) goto lookup_err;
	if (!(open_flags & O_CREAT)) {
		goto lookup_err_r;
	}

	uread_str(h->name_addrspace, s, lookup_str + UREAD_STR_LEN, UREAD_STR_LEN, '/', s, goto ureadflt2);
	if (__unlikely(__IS_ERR(s))) goto ret_s_2;
	if (lookup_str[UREAD_STR_LEN]) {
		q = __ERR_PTR(-ENOENT);
		goto lookup_err_r;
	}
	if (__likely(h->op->create != NULL)) {
		q = h->op->create(h, lookup_str, open_flags);
		if (__unlikely(q != NULL)) goto lookup_err;
		goto done_inst;
	}
	enotdir:
	q = __ERR_PTR(-ENOTDIR);
	goto lookup_err_r;

	done:
	if (__unlikely((open_flags & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL))) goto excl;
	if (__unlikely((open_flags & _O_DELETE) != 0)) goto fdelete;
	done_inst:
	if (__likely(h->op->instantiate_handle != NULL) && __unlikely((q = h->op->instantiate_handle(h, rq, open_flags)) != NULL)) {
		if (__unlikely(q == (void *)1))
			KERNEL$SUICIDE("GET_HANDLE: instantiate_handle RETURNED 1");
		goto lookup_err;
	}

#if __DEBUG >= 1
	if (__unlikely(h->op == &KERNEL$HANDLE_NOOP))
		KERNEL$SUICIDE("GET_HANDLE: HANDLE DETACHED UNDER US");
	if (__unlikely(!XLIST_EMPTY(&h->child_list)))
		KERNEL$SUICIDE("GET_HANDLE: CHILD LIST OF NEW HANDLE NOT EMPTY");
#endif
	ADD_TO_XLIST(&hp->child_list, &h->child_entry);
	ret_h:
	CHECK_RQ_STATE(rq);
	/*{
		char *p = KERNEL$DL_GET_SYMBOL_NAME(h->op, NULL, 0);
		if (strcmp(p, "SWAP_OPERATIONS")) __debug_printf("got handle %d(%d/%d): %s, iorq %s, ast %s, noret %d\n", h->handle_num, h->name_addrspace->depth, h->file_addrspace->depth, p, KERNEL$DL_GET_SYMBOL_NAME(rq->tmp1, NULL, 0), KERNEL$DL_GET_SYMBOL_NAME(rq->fn, NULL, 0), noret);
	}*/
	if (__unlikely(noret)) {
		lo_spl_call_io_ret:
		LOWER_SPL(SPL_DEV);
		CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
		ghreturn(NULL);
	}
	ghreturn(h);

	excl:
	q = __ERR_PTR(-EEXIST);
	goto lookup_err_r;

	fdelete:
	if (__unlikely(!h->op->fdelete)) goto enotdir;
	q = h->op->fdelete(h, rq, open_flags, hp);
	if (__likely(!q)) {
		q = __ERR_PTR(-ESRCH);
		goto lookup_err_q;
	}
	goto lookup_err;

	no_lookup:
	q = __ERR_PTR(-ENOTDIR);
	goto lookup_err_r;

	lookup_err:
	if (__likely(__IS_ERR(q))) {
		lookup_err_r:
		LOWER_SPLX(h->op->spl);
		if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
		lookup_err_q:
		h->op = &KERNEL$HANDLE_NOOP;
		h->file_addrspace = NULL;
		LOWER_SPL(SPL_DEV);
		CHECK_RQ_STATE(rq);
		ghreturn(q);
	}
	if (__likely(q != (void *)1)) {
		if (__unlikely(q == (void *)3)) {
		} else if (__likely(q != (void *)2)) {
			WQ_WAIT_F((WQ *)q, rq);
			LOWER_SPLX(h->op->spl);
		} else {
			CHECK_RQ_STATE(rq);
			CALL_IORQ_LSTAT_EXPR(rq, (IO_STUB *)rq->tmp1);
		}
		goto fault2;
	}
	CHECK_RQ_STATE(rq);
	if (__unlikely(!h->op->lookup_io)) goto nl;
	p = strlen(lookup_str);
	lookup_str[p] = '/';
	uread_str(h->name_addrspace, s, lookup_str + p + 1, UREAD_STR_LEN, 0, s, goto ureadflt2);
	if (__unlikely(__IS_ERR(s))) lookup_str[p] = 0;
	h->op->lookup_io(h, lookup_str, rq, open_flags);
	/* fall through */

	fault2:
	if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
	h->op = &KERNEL$HANDLE_NOOP;
	fault:
	if (__likely(h != NULL)) h->file_addrspace = NULL;
	LOWER_SPL(SPL_DEV);
	retn:
	if (__unlikely(get_handle_recurse_cut)) {
		if (get_handle_recurse == 1) {
			get_handle_recurse_cut = 0;
			proc = get_handle_recurse_proc;
			handle_num = get_handle_recurse_handle_num;
			open_flags = get_handle_recurse_open_flags;
			noret = 1;
			goto again;
		}
		if (get_handle_recurse == MAX_GET_HANDLE_RECURSE - 1) {
			get_handle_recurse_proc = proc;
			get_handle_recurse_handle_num = handle_num;
			get_handle_recurse_open_flags = open_flags;
		}
	}
	ghreturn(NULL);

	nl:
	if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
	h->op = &KERNEL$HANDLE_NOOP;
	h->file_addrspace = NULL;
	LOWER_SPL(SPL_DEV);
	CHECK_RQ_STATE(rq);
	ghreturn(__ERR_PTR(-ENOENT));

	nh:
	if (__likely(h != NULL)) h->file_addrspace = NULL;
	LOWER_SPL(SPL_DEV);
	CHECK_RQ_STATE(rq);
	ghreturn(hp);

	np:
	if (__unlikely(!__IS_ERR(proch.h))) {
		goto fault;
	}
	h->file_addrspace = NULL;
	LOWER_SPL(SPL_DEV);
	CHECK_RQ_STATE(rq);
	if (__unlikely(proch.h == -ENOLNM) && __unlikely(open_flags & _O_LNWAIT)) goto block;
	ghreturn(__ERR_PTR(proch.h));

	ret_s_2:
	if (__unlikely(h->op->leave_handle != NULL)) h->op->leave_handle(h);
	h->op = &KERNEL$HANDLE_NOOP;
	ret_s:
	h->file_addrspace = NULL;
	LOWER_SPL(SPL_DEV);
	CHECK_RQ_STATE(rq);
	ghreturn((void *)s);
#undef ghreturn

	retrn:
	get_handle_recurse--;
	if (__unlikely(get_handle_recurse_cut) && __unlikely(!get_handle_recurse))
		KERNEL$SUICIDE("GET_HANDLE: GET_HANDLE_RECURSE_CUT LEAKED");
	return retval;
}

static struct {
	union {
		MALLOC_REQUEST mrq;
		CONTIG_AREA_REQUEST car;
	} u;
	WQ wq;
	unsigned new_n;
} alloc;

static char alloc_pending = 0;

DECL_AST(allocated_fd, SPL_DEV, IORQ)
{
	if (__unlikely(!alloc_pending))
		KERNEL$SUICIDE("allocated_fd: NO PENDING ALLOCATION");
	if (__unlikely(RQ->status < 0)) {
		IORQ *q;
		while ((q = WQ_GET_ONE(&alloc.wq))) {
			q->status = RQ->status;
			CALL_AST(q);
		}
	} else {
		HANDLE **ptr, **old_ptr;
		unsigned old_n;
		if (alloc_pending == 1) ptr = alloc.u.mrq.ptr;
		else ptr = alloc.u.car.ptr;
		memset(mempcpy(ptr, FILES_PTR, N_FILES * sizeof(HANDLE *)), 0, (alloc.new_n - N_FILES) * sizeof(HANDLE *));
		__barrier();
		old_ptr = FILES_PTR;
		old_n = N_FILES;
		FILES_PTR = ptr;
		N_FILES = alloc.new_n;
		__barrier();
		if (old_n * sizeof(HANDLE *) <= PAGE_CLUSTER_SIZE) KERNEL$UNIVERSAL_FREE(old_ptr);
		else KERNEL$VM_RELEASE_CONTIG_AREA(old_ptr, (old_n * sizeof(HANDLE *) + PAGE_CLUSTER_SIZE - 1) / PAGE_CLUSTER_SIZE);
		WQ_WAKE_ALL(&alloc.wq);
	}
	alloc_pending = 0;
	RETURN;
}

static int alloc_fd(IORQ *rq, IO_STUB *fn)
{
	static unsigned CURRENT_FILE = 0;	/* must be <= N_FILES */
	unsigned i;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("alloc_fd AT SPL %08X", KERNEL$SPL);
#endif
	for (i = CURRENT_FILE; i < N_FILES; i++)
		if (__likely(!FILES_PTR[i]) && __likely(i != BLOCK_HANDLE) && __likely(i != SWAPPER_HANDLE)) {
			HANDLE *h;
			found_file:
			h = malloc(sizeof(HANDLE));
			if (__unlikely(!h)) {
				KERNEL$MEMWAIT(rq, fn, sizeof(HANDLE));
				return -1;
			}
			FILES_PTR[i] = h;
			CURRENT_FILE = i + 1;
			return i;
		}
	for (i = 0; i < CURRENT_FILE; i++)
		if (!FILES_PTR[i] && __likely(i != BLOCK_HANDLE) && __likely(i != SWAPPER_HANDLE)) goto found_file;
	if (!alloc_pending) {
		WQ_INIT(&alloc.wq, "KERNEL$ALLOC_WQ");
		alloc.new_n = N_FILES << 1;
		if (!alloc.new_n) alloc.new_n = 1;
		if (__unlikely(alloc.new_n > MAX_HANDLE + 1)) {
			rq->status = -EMFILE;
			CALL_AST(rq);
			return -1;
		}
		if (alloc.new_n * sizeof(HANDLE *) <= PAGE_CLUSTER_SIZE) {
			alloc_pending = 1;
			alloc.u.mrq.size = alloc.new_n * sizeof(HANDLE *);
			alloc.u.mrq.fn = allocated_fd;
			CALL_IORQ(&alloc.u.mrq, KERNEL$UNIVERSAL_MALLOC);
		} else {
			alloc_pending = 2;
			alloc.u.car.nclusters = (alloc.new_n * sizeof(HANDLE *) + PAGE_CLUSTER_SIZE - 1) / PAGE_CLUSTER_SIZE;
			alloc.u.car.flags = CARF_DATA;
			alloc.u.car.fn = allocated_fd;
			CALL_IORQ(&alloc.u.car, KERNEL$VM_GRAB_CONTIG_AREA);
		}
	}
	WQ_WAIT(&alloc.wq, rq, fn);
	return -1;
}

static void free_fd(int h)
{
#if __DEBUG >= 1
	FILES_PTR[h]->op = NULL;
#endif
	free(FILES_PTR[h]);
	FILES_PTR[h] = NULL;
}

DECL_IOCALL(KERNEL$OPEN, SPL_DEV, OPENRQ)
{
	HANDLE *nh, *rh;
	FFILE *f;
	int i;
	unsigned s;
	i = alloc_fd((IORQ *)RQ, KERNEL$OPEN);
	if (__unlikely(i < 0)) RETURN;

	f = open_file_name(RQ, &s);
	if (__unlikely(__IS_ERR(f))) {
		free_fd(i);
		if (f == __ERR_PTR(-ENOMEM)) {
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$OPEN, s);
			RETURN;
		}
		RQ->status = __PTR_ERR(f);
		RETURN_AST(RQ);
	}
	if (__unlikely(s) && __likely(!(RQ->flags & _O_NOMULTIOPEN))) {
		free_fd(i);
		MULTIOPEN(RQ, f);
		RETURN;
	}
	nh = FILES_PTR[i];

	f->pos = 0;
	f->flags = RQ->flags + 1;
	f->l_refcount = 1;
	f->aux = 0;
	INIT_XLIST(&f->blobs);
	nh->handle_num = i | (((unsigned)f->flags << __BSF_CONST(HANDLE_NUM_READ)) & (HANDLE_NUM_READ | HANDLE_NUM_WRITE));
	nh->name_addrspace = &KERNEL$PROC_KERNEL;
	nh->name = f;
	nh->op = &KERNEL$HANDLE_NOOP;
	RQ->tmp1 = (unsigned long)&KERNEL$OPEN;
	if (__unlikely(RQ->flags & _O_NOOPEN)) {
		if (__likely(!(RQ->flags & _O_CLOSE))) goto skip_open_1;
		goto ret_0;
	}
	rh = GET_HANDLE(&KERNEL$PROC_KERNEL, i, RQ->flags + 1, (IORQ *)RQ);
	if (__unlikely(__IS_ERR(rh))) goto er;
	if (__unlikely(!rh)) goto ret;
#if __DEBUG >= 1
	if (__unlikely(rh != nh))
		KERNEL$SUICIDE("KERNEL$OPEN: WEIRD HANDLE RETURNED FROM GET_HANDLE: %p != %p", rh, nh);
#endif
	if (__likely(!(RQ->flags & _O_CLOSE))) {
		if (__unlikely(nh->op->open_handle != NULL) && __likely(!(RQ->flags & _O_NOOPEN_CALL))) nh->op->open_handle(nh, RQ->flags + 1);
		skip_open_1:
		RQ->status = i;
		RETURN_AST(RQ);
	} else {
		KERNEL$DETACH_HANDLE(nh);
		goto ret_0;
	}
	er:
	if (__PTR_ERR(rh) == -ESRCH) ret_0: RQ->status = 0;
	else RQ->status = __PTR_ERR(rh);
	CALL_AST(RQ);
	ret:
	free_fd(i);
	__slow_free(f);
	RETURN;
}

int _dup_noposix(int n, IORQ *rq, IO_STUB *fn)
{
	int spl;
	HANDLE *h, *nh;
	int nn;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) KERNEL$SUICIDE("_dup_noposix AT SPL %08X", spl);
	RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)n >= N_FILES)) {
		/* this may be called from /MAP switch, so do not panic */
		badf:
		rq->status = -EBADF;
		CALL_AST(rq);
		lo_ret__1:
		LOWER_SPLX(spl);
		return -1;
	}
	h = FILES_PTR[n];
	if (__unlikely(!h)) goto badf;
	nn = alloc_fd(rq, fn);
	if (__unlikely(nn < 0)) goto lo_ret__1;
	h->name->l_refcount++;
	nh = FILES_PTR[nn];
	nh->handle_num = nn;
	nh->name_addrspace = &KERNEL$PROC_KERNEL;
	nh->name = h->name;
	nh->op = &KERNEL$HANDLE_NOOP;
	LOWER_SPLX(spl);
	return nn;
}

extern IO_STUB CONT_CLOSE;
extern AST_STUB DONE_CLOSE;

DECL_IOCALL(KERNEL$CLOSE, SPL_DEV, CLOSERQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$CLOSE;
	if (__unlikely(CLOSE_SELECT_HANDLE(RQ->h, (IORQ *)RQ))) RETURN;
	RQ->tmp1 = (unsigned long)CONT_CLOSE;
	RQ->real_ast = RQ->fn;
	RQ->fn = DONE_CLOSE;
	RETURN_IORQ_LSTAT(RQ, CONT_CLOSE);
}

DECL_IOCALL(CONT_CLOSE, SPL_DEV, CLOSERQ)
{
	int r;
	HANDLE *h;
	if (__unlikely((unsigned)RQ->h >= N_FILES))
		KERNEL$SUICIDE("CONT_CLOSE(%d): INVALID HANDLE", RQ->h);
	h = FILES_PTR[RQ->h];
	if (__unlikely(!h))
		KERNEL$SUICIDE("CONT_CLOSE(%d): HANDLE NOT OPEN", RQ->h);
	if (__unlikely(h->name->flags & _O_NOOPEN)) goto ret_0;
	h = GET_HANDLE(&KERNEL$PROC_KERNEL, RQ->h, 0, (IORQ *)RQ);
	if (__unlikely(__IS_ERR(h))) {
		RQ->status = __PTR_ERR(h);
		RETURN_AST(RQ);
	}
	if (__unlikely(!h)) RETURN;
	if (h->op->close_handle) {
		r = h->op->close_handle(h, (IORQ *)RQ);
		if (__unlikely(r > 0)) RETURN;
	} else ret_0: r = 0;
	RQ->status = r;
	RETURN_AST(RQ);
}

DECL_AST(DONE_CLOSE, SPL_DEV, CLOSERQ)
{
	KERNEL$FAST_CLOSE(RQ->h);
	RQ->h = -1;
	RQ->fn = RQ->real_ast;
	RETURN_AST(RQ);
}

void KERNEL$FAST_CLOSE(int hn)
{
	HANDLE *h;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("KERNEL$FAST_CLOSE AT SPL %08X", KERNEL$SPL);
#endif
	if (__likely(hn == BLOCK_HANDLE)) return;
	if (__unlikely((unsigned)hn >= N_FILES)) {
		KERNEL$SUICIDE("KERNEL$FAST_CLOSE(%d): INVALID HANDLE", hn);
	}
	RAISE_SPL(SPL_DEV);
	h = FILES_PTR[hn];
	if (__unlikely(!h))
		KERNEL$SUICIDE("KERNEL$FAST_CLOSE(%d): HANDLE NOT OPEN", hn);
	RAISE_SPL(SPL_TOP);
	LOWER_SPLX(h->op->spl);
	KERNEL$DETACH_HANDLE(h);
	LOWER_SPL(SPL_DEV);
	UNREF_FILE(h->name);
	free_fd(hn);
	LOWER_SPLX(spl);
	return;
}

#if 0

void test_handle(HANDLE *h, int depth, HANDLE *p)	/* !!! FIXME: remove */
{
	static HANDLE *q = NULL;
	HANDLE *hh;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (depth) if (q == h) KERNEL$SUICIDE("[%p] loop @ depth %d, %p, %p, %p / %p, %p, %p", NULL, depth, h, h->name_addrspace, h->file_addrspace, p, p ? p->name_addrspace : 0, p ? p->file_addrspace : 0);
	if (!(depth & (depth - 1))) q = h;
	if (h->op == &KERNEL$HANDLE_NOOP) {
		if (depth) KERNEL$SUICIDE("[%p] nop handle, depth %d, handle (%p/%p), parent (%p/%p/%p)", NULL, depth, h, h->name_addrspace, p, p ? p->name_addrspace : 0, p ? p->op : 0);
		else goto ret;
	}
	XLIST_FOR_EACH(hh, &h->child_list, HANDLE, child_entry) test_handle(hh, depth + 1, h);
	ret:
	LOWER_SPLX(spl);
}

static HANDLE *reserved;

int test_reserved(PROC *p)
{
	HANDLE *hh;
	if (p == &KERNEL$PROC_KERNEL) return 0;
	XLIST_FOR_EACH(hh, &p->reserved_handle_list, HANDLE, proc_hash) if (hh == reserved)
		KERNEL$SUICIDE("used handle on reserved list");
	return 0;
}

int test_proc_handles(PROC *p)
{
	int i;
	HANDLE *h;
	for (i = 0; i <= p->handle_hash_mask; i++) XLIST_FOR_EACH(h, &p->handles[i], HANDLE, proc_hash) {
		reserved = h;
		test_reserved(p);
		/*FOR_ALL_PROCS(&KERNEL$PROC_KERNEL, test_reserved);*/
		test_handle(h, 0, NULL);
	}
	return 0;
}

void test_all_handles()
{
	FOR_ALL_PROCS(&KERNEL$PROC_KERNEL, test_proc_handles);
}

#endif

/* This function used to have recursion but it was rewritten because recursion
   can cause stack overflow.
   We store the parent pointer temporarily to child_entry.next to achieve
   O(n) complexity wrt number of handles in tree. */

void KERNEL$DETACH_HANDLE(HANDLE *h)
{
	int spl;
	HANDLE *ph, *xh;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != h->op->spl))
		KERNEL$SUICIDE("KERNEL$DETACH_HANDLE: CALLED AT SPL %08X, EXPECTED %08X", KERNEL$SPL, h->op->spl);
#endif
	/*test_all_handles();*/
	if (__unlikely(h->op == &KERNEL$HANDLE_NOOP)) return;
	ph = NULL;
	recurse:
	DEL_FROM_LIST(&h->child_entry);
	h->child_entry.next = (LIST_ENTRY *)ph;
	processed_child:
	if (__unlikely(!XLIST_EMPTY(&h->child_list))) {
		ph = h;
		h = LIST_STRUCT(h->child_list.next, HANDLE, child_entry);
#if __DEBUG >= 1
		if (__unlikely(h->op == &KERNEL$HANDLE_NOOP))
			KERNEL$SUICIDE("KERNEL$DETACH_HANDLE: ALREADY DETACHED HANDLE IN THE TREE (OP %s)", KERNEL$DL_GET_SYMBOL_NAME(ph ? ph->op : h->op, (unsigned long *)(void *)&KERNEL$LIST_END, 0));
#endif
		goto recurse;
	}
	if (__likely(SPLX_BELOW(h->op->spl, SPL_X(SPL_VSPACE)))) RAISE_SPL(SPL_VSPACE);
	xh = (HANDLE *)h->child_entry.next;
	VOID_LIST_ENTRY(&h->child_entry);
	if (h->op->detach_handle) h->op->detach_handle(h);
	spl = h->op->spl;
	h->op = &KERNEL$HANDLE_NOOP;
	h->file_addrspace = NULL;
	LOWER_SPLX(spl);
	if (__unlikely(xh != NULL)) {
		h = xh;
#if __DEBUG >= 1
		if (__unlikely(h->op == &KERNEL$HANDLE_NOOP))
			KERNEL$SUICIDE("KERNEL$DETACH_HANDLE: ALREADY DETACHED HANDLE IN THE TREE WAS SUDDENLY CREATED");
#endif
		goto processed_child;
	}
	return;
}

HANDLE *KERNEL$GET_HANDLE(int hn)
{
	HANDLE *h;
	int spl = KERNEL$SPL;
	if (__unlikely((unsigned)hn >= N_FILES))
		KERNEL$SUICIDE("KERNEL$GET_HANDLE(%d): INVALID HANDLE", hn);
	if (SPLX_BELOW(spl, SPL_X(SPL_DEV))) RAISE_SPL(SPL_DEV);
	h = FILES_PTR[hn];
	LOWER_SPLX(spl);
	if (__unlikely(!h))
		KERNEL$SUICIDE("KERNEL$GET_HANDLE(%d): HANDLE NOT OPEN", hn);
	return h;
}

char *KERNEL$HANDLE_PATH(int h)
{
	if (__unlikely(h == BLOCK_HANDLE)) return BLOCK_LNM ":/";
	return KERNEL$GET_HANDLE(h)->name->path;
}

int KERNEL$HANDLE_FLAGS(int h)
{
	if (__unlikely(h == BLOCK_HANDLE)) return 0;
	return KERNEL$GET_HANDLE(h)->name->flags;
}

#if __DEBUG >= 1

DECL_IOCALL(INVALID_HANDLE, SPL_TOP, SIORQ)
{
	unsigned long off = 0;
	__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME((void *)RQ->fn, &off, 0);
	KERNEL$SUICIDE("INVALID HANDLE %d USED (LIMIT %u), AST %s+%lX", RQ->h, N_FILES, str ? str : "?", off);
	RETURN;
}

#endif

void set_pipe_buffer(int h, FPIPE *buf)
{
	FILES_PTR[h]->name->pos = (_u_off_t)(unsigned long)buf;
}

extern AST_STUB CHANGE_HANDLE_DONE;

DECL_IOCALL(KERNEL$CHANGE_HANDLE, SPL_DEV, CHHRQ)
{
	int r;
	HANDLE *h, *nh;
	unsigned long l;
	FFILE *f, *nf;
	if (__unlikely((unsigned)RQ->h >= N_FILES))
		KERNEL$SUICIDE("KERNEL$CHANGE_HANDLE(%d): INVALID HANDLE", RQ->h);
	h = FILES_PTR[RQ->h];
	if (__unlikely(!h))
		KERNEL$SUICIDE("KERNEL$CHANGE_HANDLE(%d): HANDLE NOT OPEN", RQ->h);
	f = h->name;
	if (__unlikely(f->flags & _O_PIPE)) {
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	}
	l = sizeof(FFILE) + strlen(f->path) + 1 + (RQ->value ? strlen(RQ->option) + strlen(RQ->value) + 3 : 0);
	if (__unlikely(!(nf = malloc(l)))) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$CHANGE_HANDLE, l);
		RETURN;
	}
	r = substitute_option(f->path, nf->path, RQ->option, RQ->value);
	if (__unlikely(r <= 0)) {
		__slow_free(nf);
		RQ->status = r;
		RETURN_AST(RQ);
	}
	nf->pos = f->pos;
	nf->flags = f->flags;
	nf->l_refcount = 1;
	nf->aux = f->aux;
	INIT_XLIST(&nf->blobs);
	r = alloc_fd((IORQ *)RQ, KERNEL$CHANGE_HANDLE);
	if (__unlikely(r < 0)) {
		__slow_free(nf);
		RETURN;
	}
	nh = FILES_PTR[r];
	nh->handle_num = r;
	nh->name_addrspace = &KERNEL$PROC_KERNEL;
	nh->name = nf;
	nh->op = &KERNEL$HANDLE_NOOP;
	RQ->ioctlrq.fn = &CHANGE_HANDLE_DONE;
	RQ->ioctlrq.h = r;
	RQ->ioctlrq.ioctl = IOCTL_NOP;
	RQ->ioctlrq.param = 0;
	RQ->ioctlrq.v.ptr = 0;
	RQ->ioctlrq.v.len = 0;
	RQ->ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->ioctlrq, KERNEL$IOCTL, RQ);
}

DECL_AST(CHANGE_HANDLE_DONE, SPL_DEV, IOCTLRQ)
{
	CHHRQ *rq = GET_STRUCT(RQ, CHHRQ, ioctlrq);
	int i;
	FFILE *f, *nf;
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, rq);
	nf = FILES_PTR[rq->ioctlrq.h]->name;
	if (__unlikely(rq->ioctlrq.status < 0) && __unlikely(rq->ioctlrq.status != -ENOOP)) {

		KERNEL$FAST_CLOSE(rq->ioctlrq.h);
		rq->status = rq->ioctlrq.status;
		RETURN_AST(rq);
	}
	f = FILES_PTR[rq->h]->name;
	nf->pos = f->pos;
	nf->flags = f->flags;
	nf->aux = f->aux;
	MOVE_XLIST_HEAD(&nf->blobs, &f->blobs);
	__barrier();
	INIT_XLIST(&f->blobs);
	for (i = 0; i < N_FILES; i++) {
		HANDLE *hh = FILES_PTR[i];
		if (__unlikely(hh != NULL)) {
			int d = 0;
			if (__unlikely(hh->file == f)) hh->file = nf, d = 1;
			if (__unlikely(hh->name == f)) hh->name = nf, d = 2;
			if (__unlikely(d)) {
				RAISE_SPL(SPL_TOP);
				LOWER_SPLX(hh->op->spl);
				KERNEL$DETACH_HANDLE(hh);
				LOWER_SPL(SPL_DEV);
				CLOSE_SELECT_HANDLE(i, NULL);
			}
			if (d == 2) {
				nf->l_refcount++;
				if (__likely(UNREF_FILE(f))) break;
			}
		}
	}
	if (__unlikely(nf->l_refcount == 1))
		KERNEL$SUICIDE("CHANGE_HANDLE_DONE: HANDLE %d(%s) DISAPPEARED WHILE CHANGING", rq->h, nf->path);
	KERNEL$FAST_CLOSE(rq->ioctlrq.h);
	rq->status = 0;
	RETURN_AST(rq);
}

DECL_IOCALL(KERNEL$SET_HANDLE_BLOB, SPL_DEV, HBLOBRQ)
{
	HANDLE *h;
	FFILE *f;
	FBLOB_REF *ref;
	void *to_free_1, *to_free_2;
	unsigned i;
	if (__unlikely((unsigned)RQ->h >= N_FILES))
		KERNEL$SUICIDE("KERNEL$SET_HANDLE_BLOB(%d): INVALID HANDLE", RQ->h);
	h = FILES_PTR[RQ->h];
	if (__unlikely(!h))
		KERNEL$SUICIDE("KERNEL$SET_HANDLE_BLOB(%d): HANDLE NOT OPEN", RQ->h);
	f = h->name;
	ref = malloc(sizeof(FBLOB_REF));
	if (__unlikely(!ref)) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$SET_HANDLE_BLOB, sizeof(FBLOB_REF));
		RETURN;
	}
	SET_FILE_BLOB(f, ref, RQ->blob, &to_free_1, &to_free_2);
	for (i = 0; i < N_FILES; i++) {
		HANDLE *hh = FILES_PTR[i];
		if (__unlikely(hh != NULL)) {
			if (__unlikely(hh->name == f)) {
				RAISE_SPL(SPL_TOP);
				LOWER_SPLX(hh->op->spl);
				KERNEL$DETACH_HANDLE(hh);
				LOWER_SPL(SPL_DEV);
				CLOSE_SELECT_HANDLE(i, NULL);
			}
		}
	}
	free(to_free_1);
	free(to_free_2);
	RQ->status = 0;
	RETURN_AST(RQ);
}

static int vm_fault_recurse = 0;
static void *vm_fault_recurse_proc;
static void *vm_fault_recurse_addr;
static int vm_fault_recurse_wr;

int VM_FAULT(PROC *proc, void *addr, int wr, IORQ *rq)
{
	int error;
	WQ *wq;
	VMMAP *vmmap;
	int nvmmaps;
	int top, bottom, mid;
	/*__debug_printf("VM_FAULT(%s, %p,%d, %p)\n", proc->jobname, addr, wr, __builtin_return_address(0));*/
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("VM_FAULT AT SPL %08X", KERNEL$SPL);
	if (__unlikely(ARCH_PROC_INVALID(proc)))
		KERNEL$SUICIDE("VM_FAULT: INVALID PROCESS %p (FAULT %p/%d)", proc, addr, wr);
#endif
	if (__unlikely(proc == &KERNEL$PROC_KERNEL)) {
#if __DEBUG_PAGEFAULTS > 0
		if (__unlikely((unsigned long)addr < KUVMTOP))
#endif
		{
			unsigned long off = 0, offa = 0;
			__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME((void *)rq->tmp1, &off, 0), *stra = KERNEL$DL_GET_SYMBOL_NAME((void *)rq->fn, &offa, 0);
			KERNEL$SUICIDE("VM_FAULT ON KERNEL$PROC_KERNEL, ADDRESS %p, CALLER %s+%lX, AST %s+%lX", addr, str ? str : "?", off, stra ? stra : "?", offa);
		}
		return 0;
	}
	SWITCH_PROC_ACCOUNT(proc, SPL_X(SPL_DEV));
	if (__unlikely(proc->fork_child != NULL)) {
		WQ *wq;
		if (__unlikely((wq = VM_ARCH_READ_ONLY(proc)) != NULL)) {
			WQ_WAIT_F(wq, rq);
			LOWER_SPL(SPL_DEV);
			return 1;
		}
		proc->fork_child->fork_parent = NULL;
		proc->fork_child = NULL;
	}
	if (__unlikely(vm_fault_recurse)) {
		if (__likely(vm_fault_recurse == 1)) {
			vm_fault_recurse_proc = proc;
			vm_fault_recurse_addr = addr;
			vm_fault_recurse_wr = wr;
			vm_fault_recurse = 2;
		} else {
			KERNEL$SUICIDE("VM_FAULT: DOUBLE RECURSIVE FAULT (%p/%d). FIRST FAULT WAS (%p/%d)", addr, wr, vm_fault_recurse_addr, vm_fault_recurse_wr);
		}
		return 1;
	}
	recurse_retry:
	if (__unlikely((unsigned long)addr >= KUVMTOP)) {
		/*__debug_printf("invalid addr: %lx\n", addr);*/
		invalid:
		/*__debug_printf("invalid\n");*/
		error = -EFAULT;
		invalid_err:
		rq->status = error;
		CALL_AST(rq);
		return error;
	}
	uread_ptr(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, vmmaps)), vmmap, {addr = (void *)KUPLACE(UDATA_STRUCT + __offsetof(UDATA, vmmaps)); wr = PF_READ | PF_SWAPPER; goto double_fault;});
	if (__unlikely((unsigned long)vmmap & (VMMAP_ALIGN - 1))) {
		__debug_printf("EINVAL 1\n");
		error = -EINVAL;
		goto invalid_err;
	}
	uread_int(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, n_vmmaps)), nvmmaps, {addr = (void *)KUPLACE(UDATA_STRUCT + __offsetof(UDATA, n_vmmaps)); wr = PF_READ | PF_SWAPPER; goto double_fault;});
	if (__unlikely((unsigned)nvmmaps > MAXINT)) goto invalid;
	uread_int(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, last_vmmap)), mid, {addr = (void *)KUPLACE(UDATA_STRUCT + __offsetof(UDATA, last_vmmap)); wr = PF_READ | PF_SWAPPER; goto double_fault;});
	bottom = -1;
	top = nvmmaps - 1;
	if (__likely((unsigned)mid < nvmmaps)) goto proc_hint;
	hint_failed:
	bottom = 0;
	while (top >= bottom) {
			/* max 32 iterations, no need to break out of a loop */
		void *vmaddr;
		size_t vmsize;
		__v_off vmoffset;
		int vmhandle;
		unsigned vmflags;
		VMMAP *cvmmap;
		HANDLE *handle, *filehandle;
		PAGE *page, *filepage;
		void *v1, *v2;
		mid = ((unsigned)top + (unsigned)bottom) >> 1;
		proc_hint:
		cvmmap = vmmap + mid;
		uread_ptr(proc, &cvmmap->vmaddr, vmaddr, {addr = &cvmmap->vmaddr; wr = PF_READ | PF_SWAPPER; goto double_fault;});
		/*__debug_printf("vm fault: %p, (%p) bottom=%d,top=%d,mid=%d\n", addr, vmaddr, bottom, top, mid);*/
		/*__debug_printf("ADDR: %p. ", vmaddr);*/
		if (addr < vmaddr) {
			if (bottom < 0) goto hint_failed;
			top = mid - 1;
			continue;
		}
		uread_size_t(proc, &cvmmap->vmsize, vmsize, {addr = &cvmmap->vmsize; wr = PF_READ | PF_SWAPPER; goto double_fault;});
		/*__debug_printf("SIZE: %lx. ", vmsize);*/
		if ((char *)addr >= (char *)vmaddr + vmsize) {
			if (bottom < 0) goto hint_failed;
			bottom = mid + 1;
			continue;
		}
		if (bottom >= 0) {
			uwrite_int(proc, KUPLACE(UDATA_STRUCT + __offsetof(UDATA, last_vmmap)), mid, );
		}
		uread_int(proc, &cvmmap->flags, vmflags, {addr = &cvmmap->flags; wr = PF_READ | PF_SWAPPER; goto double_fault;});
		/*__debug_printf("VMFLAGS: %x.", vmflags);*/

		if (__unlikely(!(vmflags & (wr & PF_WRITE ? PROT_WRITE : PROT_READ | PROT_EXEC)))) goto invalid;
		if (__likely(vmflags & MAP_ANONYMOUS)) {
			if (PF_SHARE > MAP_SHARED) {
				wr |= ((unsigned)vmflags & MAP_SHARED) * (PF_SHARE / MAP_SHARED);
			} else {
				wr |= ((unsigned)vmflags & MAP_SHARED) / (PF_SHARE > MAP_SHARED ? 1 /* against div-by-zero warning */ : MAP_SHARED / PF_SHARE);
			}
			goto double_fault;
		}

		uread___voff_t(proc, &cvmmap->offset, vmoffset, {addr = &cvmmap->offset; wr = PF_READ | PF_SWAPPER; goto double_fault;});
		uread_int(proc, &cvmmap->handle, vmhandle, {addr = &cvmmap->handle; wr = PF_READ | PF_SWAPPER; goto double_fault;});
		/*__debug_printf("OFFSET: %Lx, HANDLE %d.", vmoffset, vmhandle);*/

		if ((vmflags & (MAP_SHARED | __PROT_WRITTEN)) == __PROT_WRITTEN) {
			redo_fault:
			vm_fault_recurse = 1;
			GET_HANDLE_FAST(handle, proc, SWAPPER_HANDLE, 0, rq, );
			if (__unlikely(vm_fault_recurse == 2)) {
				do_recurse:
				proc = vm_fault_recurse_proc;
				addr = vm_fault_recurse_addr;
				wr = vm_fault_recurse_wr;
				vm_fault_recurse = 0;
				goto recurse_retry;
			}
			vm_fault_recurse = 0;
			if (__unlikely(!handle)) return 1;
			if (__unlikely(__IS_ERR(handle))) {
				error = __PTR_ERR(handle);
				goto invalid_err;
			}
			page = handle->op->get_page(handle, (unsigned long)addr, wr | PF_TESTPAGE);
		/*__debug_printf("get_page_test1[%p-%p+%Lx] (proc %p, res %p)", addr, vmaddr, vmoffset, proc, page);*/
			if (page != NULL) {
				vmoffset = 0;
				vmaddr = NULL;
				goto found_in_swapper;
			}

			if (__unlikely(wr & PF_SWAPPER)) {
				__debug_printf("EINVAL 2\n");
				error = -EINVAL;
				goto invalid_err;
			}

			if (__likely(!(wr & PF_WRITE))) {
				LOWER_SPL(SPL_DEV);
				goto process_fault;
			}

			/* do COW here */
			filehandle = GET_HANDLE_TEST(proc, vmhandle, 0);
			if (__unlikely(!filehandle)) {
				LOWER_SPL(SPL_DEV);
				vm_fault_recurse = 1;
				filehandle = GET_HANDLE(proc, vmhandle, 0, rq);
				if (__unlikely(vm_fault_recurse == 2)) {
					goto do_recurse;
				}
				vm_fault_recurse = 0;
				if (__unlikely(!filehandle)) return 1;
				if (__IS_ERR(filehandle)) {
					error = __PTR_ERR(filehandle);
					goto invalid_err;
				}
				goto repost_rq_lower;
			}
		/*__debug_printf("get_page_backing[%p-%p+%Lx]", addr, vmaddr, vmoffset);*/
			filepage = filehandle->op->get_page(filehandle, ((unsigned long)addr - (unsigned long)vmaddr) + vmoffset, wr & ~PF_WRITE);
			if (__unlikely(!filepage)) {
				wr &= ~PF_WRITE;
				handle = filehandle;
				goto do_pagein_fn;
			}
			if (__unlikely(__IS_ERR(filepage))) {
				if (__likely(__PTR_ERR(filepage) == -ERANGE)) goto eoor;
				error = __PTR_ERR(filepage);
				goto invalid_err_lower;
			}
			eoor:
		/*__debug_printf("get_page_for_cow[%p-%p+%Lx]", addr, vmaddr, vmoffset);*/
			page = handle->op->get_page(handle, (unsigned long)addr, wr | PF_COW);
			if (__unlikely(__IS_ERR(page))) goto invalid_err_page;
			if (__unlikely(!page)) {
				vmoffset = 0;
				vmaddr = NULL;
				wr |= PF_WAIT_4_ALLOC;
				goto do_pagein_fn;
			}

			v1 = KERNEL$MAP_PHYSICAL_PAGE(page);
			if (__likely(!__IS_ERR(filepage))) {
				if (__likely(IS_PAGE_POINTER(filepage))) {
					v2 = KERNEL$MAP_PHYSICAL_PAGE(filepage);
					memcpy(v1, v2, PAGE_CLUSTER_SIZE);
				} else {
#define spage	((SPAGE *)filepage)
					v2 = (char *)KERNEL$MAP_PHYSICAL_PAGE(spage->page) + spage->offset;
					memset(mempcpy(v1, v2, spage->n_pages << PG_SIZE_BITS), 0, PAGE_CLUSTER_SIZE - (spage->n_pages << PG_SIZE_BITS));
#undef spage
				}
				KERNEL$UNMAP_PHYSICAL_BANK(v2);
			} else {
				memset(v1, 0, PAGE_CLUSTER_SIZE);
			}
			KERNEL$UNMAP_PHYSICAL_BANK(v1);
			goto do_mapping;
		}

		if (__unlikely(wr & PF_SWAPPER)) {
			__debug_printf("EINVAL 3\n");
			error = -EINVAL;
			goto invalid_err;
		}

		process_fault:
		vm_fault_recurse = 1;
		GET_HANDLE_FAST(handle, proc, vmhandle, 0, rq, );
		if (__unlikely(vm_fault_recurse == 2)) {
			goto do_recurse;
		}
		vm_fault_recurse = 0;

		if (__unlikely(!handle)) {
			return 1;
		}
		if (__unlikely(__IS_ERR(handle))) {
			error = __PTR_ERR(handle);
			goto invalid_err;
		}
		page = handle->op->get_page(handle, ((unsigned long)addr - (unsigned long)vmaddr) + vmoffset, wr);
		/*__debug_printf("get_page[%p-%p+%Lx] (proc %p, res %p)", addr, vmaddr, vmoffset, proc, page);*/
		found_in_swapper:
		if (__unlikely(!page) || __unlikely(page == (void *)1)) {
			VDESC vd;
			do_pagein_fn:
			vd.ptr = (((unsigned long)addr - (unsigned long)vmaddr) + vmoffset) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
			vd.len = PAGE_CLUSTER_SIZE;
			vd.vspace = handle;
			DO_PAGEIN_NORET(rq, &vd, wr | PF_PAGEIO);
			LOWER_SPL(SPL_DEV);
			return 1;
		}
		if (__unlikely(__IS_ERR(page))) {
			if (__likely(__PTR_ERR(page) == -ERANGE) && __likely((vmflags & (MAP_SHARED | MAP_ANONYMOUS | __PROT_WRITTEN)) == __PROT_WRITTEN)) {
				LOWER_SPL(SPL_DEV);
				wr |= PF_WRITE;
				goto redo_fault;
			}
			invalid_err_page:
			error = __PTR_ERR(page);
			invalid_err_lower:
			LOWER_SPL(SPL_DEV);
			goto invalid_err;
		}

		do_mapping:
#if __DEBUG >= 1
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), KERNEL$SPL)))
			KERNEL$SUICIDE("VM_FAULT: PAGER ON SPL %08X (MAX ALLOWED %08X)", KERNEL$SPL, SPL_X(SPL_VSPACE));
#endif
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_FS), KERNEL$SPL))) {
			if (__likely(IS_PAGE_POINTER(page))) {
				RAISE_SPL(SPL_VSPACE);
				if (__unlikely((wq = page->lockdown(page, 0)) != NULL)) goto wwq;
				LOWER_SPL(SPL_FS);
				wq = VM_ARCH_MAP_PAGE(proc, (unsigned long)addr, page, wr);
				RAISE_SPL(SPL_VSPACE);
				page->lockdown(page, 1);
			} else {
				RAISE_SPL(SPL_VSPACE);
				if (__unlikely((wq = ((SPAGE *)page)->lockdown((SPAGE *)page, 0)) != NULL)) goto wwq;
				LOWER_SPL(SPL_FS);
				wq = VM_ARCH_MAP_PAGE(proc, (unsigned long)addr, page, wr);
				RAISE_SPL(SPL_VSPACE);
				((SPAGE *)page)->lockdown((SPAGE *)page, 1);
			}
		} else {
			RAISE_SPL(SPL_FS);
			wq = VM_ARCH_MAP_PAGE(proc, (unsigned long)addr, page, wr);
		}
		if (__unlikely(__IS_ERR(wq))) {
			error = __PTR_ERR(wq);
			goto invalid_err_lower;
		}
		if (__unlikely(wq != NULL)) {
			wwq:
			WQ_WAIT_F(wq, rq);
			LOWER_SPL(SPL_DEV);
			return 1;
		}
		LOWER_SPL(SPL_DEV);

#if __DEBUG_WAIT_PAGEFAULTS > 0
		if (__unlikely((random() & 255) < __DEBUG_WAIT_PAGEFAULTS)) {
			CALL_IORQ_LSTAT_EXPR(rq, (IORQ *)rq->tmp1);
			return 1;
		}
#endif
		return 0;
	
		repost_rq_lower:
		LOWER_SPL(SPL_DEV);
		CALL_IORQ_LSTAT_EXPR(rq, (IORQ *)rq->tmp1);
		return 1;

		double_fault:
		/*__debug_printf("double_fault at %p.", addr);*/
		vmflags = MAP_ANONYMOUS | PROT_READ | PROT_WRITE | PROT_EXEC;
		vmhandle = SWAPPER_HANDLE;
		vmoffset = 0;
		vmaddr = NULL;
		vmsize = KUVMTOP;
		goto process_fault;
	}
	goto invalid;
}

void KERNEL$PAGEIN(IORQ *rq, VDESC *vd, int wr)
{
	int spl;
	IORQ *pgin;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), KERNEL$SPL)))
		KERNEL$SUICIDE("KERNEL$PAGEIN AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(rq->fn == bio_emu_endrq)) {
		rq->status = -EVSPACEFAULT;
		CALL_AST(rq);
		return;
	}
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	pgin = vd->vspace->op->vspace_get_pagein_rq(vd, rq, wr);
	LOWER_SPLX(spl);
	if (__likely(pgin != NULL)) CALL_IORQ_EXPR(pgin, (IO_STUB *)pgin->tmp1);
}

static int LOCKUP_WAIT(IORQ *rq, HANDLE *h);

int KERNEL$LOCKUP_WAIT_ENTRY(IORQ *rq, HANDLE *h)
{
	if (__unlikely(KERNEL$LOCKUP_LEVEL < LOCKUP_LEVEL_ALL_IORQS + (h->name_addrspace == &KERNEL$PROC_KERNEL))) return 0;
	return LOCKUP_WAIT(rq, h);
}

int KERNEL$LOCKUP_WAIT_LOOP(IORQ *rq, HANDLE *h)
{
	if (__unlikely(KERNEL$LOCKUP_LEVEL < LOCKUP_LEVEL_ONE_PASS + (h->name_addrspace == &KERNEL$PROC_KERNEL))) return 0;
	return LOCKUP_WAIT(rq, h);
}

static int LOCKUP_WAIT(IORQ *rq, HANDLE *h)
{
	if (__unlikely(h->name_addrspace == &KERNEL$PROC_KERNEL)) {
		if (__likely((rq->status & RQS_PROC_MASK) == RQS_PROC) &&
		    __unlikely((rq->status & RQS_PRIORITY_MASK) >= RQS_ADD_PRIORITY * KERNEL_NO_RESTART_PRIORITY)) {
			return 0;
		}
		WQ_WAIT_F(&KERNEL$LOCKUP_EVENTS, rq);
	} else {
		/*if (io_inb(0x60) == 0x36) KERNEL$STACK_DUMP();*/
		rq->status = -EINTR;
		CALL_AST(rq);
	}
	return 1;
}

extern AST_STUB LNTE_ALLOCATED;

DECL_IOCALL(KERNEL$REGISTER_DEVICE, SPL_DEV, DEVICE_REQUEST)
{
	HANDLE *h;
	WQ *inst;
	int i;
	if (__unlikely(RQ->flags & ~(LNTE_PUBLIC | LNTE_DONT_WAKE_ALL)))
		KERNEL$SUICIDE("KERNEL$REGISTER_DEVICE: %s: INVALID FLAGS %X", RQ->name, RQ->flags);
	if (__unlikely(!RQ->dcall != !RQ->dcall_type))
		KERNEL$SUICIDE("KERNEL$REGISTER_DEVICE: %s: DCALL (%p) AND DCALL_TYPE (%p) INCONSISTENCY", RQ->name, RQ->dcall, RQ->dcall_type);
#if __DEBUG >= 1
	if (RQ->dcall_type) *(volatile __const__ char *)RQ->dcall_type;
	*(volatile __const__ char *)RQ->driver_name;
#endif
	i = alloc_fd((IORQ *)RQ, KERNEL$REGISTER_DEVICE);
	if (__unlikely(i < 0)) RETURN;
	h = FILES_PTR[i];
	memset(h, 0, sizeof(HANDLE));
	h->handle_num = i;
	INIT_XLIST(&h->child_list);
	VOID_LIST_ENTRY(&h->child_entry);
	h->name_addrspace = &KERNEL$PROC_KERNEL;
	if (RQ->init_root_handle) RQ->init_root_handle(h, RQ->dev_ptr);
	else h->op = &empty_handle;
	RAISE_SPLX(h->op->spl);
	if (h->op->instantiate_handle) {
		inst = h->op->instantiate_handle(h, NULL, 0);
		if (__unlikely(inst != NULL)) KERNEL$SUICIDE("KERNEL$REGISTER_DEVICE: %s: INSTANTIATE_HANDLE RETURNED %p", RQ->name, inst);
	}
	LOWER_SPL(SPL_DEV);
	RQ->h = i;
	RQ->mrq.fn = &LNTE_ALLOCATED;
	RQ->mrq.size = sizeof(LNTE) + strlen(RQ->name);
	RETURN_IORQ_CANCELABLE(&RQ->mrq, &KERNEL$UNIVERSAL_MALLOC, RQ);
}

DECL_AST(LNTE_ALLOCATED, SPL_DEV, MALLOC_REQUEST)
{
	char *c;
	int h;
	int r;
	LNTE *e;
	LNTE *lnte;
	DEVICE_REQUEST *drq = LIST_STRUCT(RQ, DEVICE_REQUEST, mrq);
	IO_DISABLE_CHAIN_CANCEL(SPL_DL, drq);
	if (__unlikely(RQ->status < 0)) {
		HANDLE *nh;
		drq->status = RQ->status;
		free_ret:
		nh = FILES_PTR[drq->h];
		RAISE_SPL(SPL_TOP);
		LOWER_SPLX(nh->op->spl);
		KERNEL$DETACH_HANDLE(nh);
		LOWER_SPL(SPL_DEV);
		free_fd(drq->h);
		RETURN_AST(drq);
	}
	drq->lnte = lnte = RQ->ptr;
	lnte->dev_ptr = drq->dev_ptr;
	lnte->dcall = drq->dcall;
	lnte->dcall_type = drq->dcall_type;
	lnte->dctl = drq->dctl;
	lnte->unload = drq->unload;
	lnte->driver_name = drq->driver_name;
	lnte->handle = drq->h | (drq->flags & LNTE_PUBLIC) | LNTE_DEVICE;
	strcpy(lnte->name, drq->name);
	if (__unlikely(__check_logical_name(lnte->name, 1))) {
		free_lnte_ret:
		KERNEL$UNIVERSAL_FREE(lnte);
		drq->status = -EBADSYN;
		goto free_ret;
	}
	c = lnte->name;
	h = 0;
	quickcasehash(c, *c, h);
	h &= LNT_HASH_SIZE - 1;

	XLIST_FOR_EACH(e, &kernel_lnt.hash[h], LNTE, hash) if (__likely(!__strcasexcmp(e->name, lnte->name, c))) {
		if (__unlikely((r = KERNEL$DELETE_LOGICAL(e->name, 0)))) {
			if (r == -EBUSY) r = -EEXIST;
			drq->status = r;
			goto free_lnte_ret;
		}
		break;
	}

	ADD_TO_XLIST(&kernel_lnt.hash[h], &lnte->hash);
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	if (__unlikely(!(drq->flags & LNTE_DONT_WAKE_ALL))) DO_BREAK(&KERNEL$PROC_KERNEL, NULL, 1);
	MULTIOPEN_CLEAR_CACHE();
	CLEAR_SUBPROC_LN_CACHE();
	drq->status = 0;
	RETURN_AST(drq);
}

int KERNEL$UNREGISTER_DEVICE(void *lnte_, int flags)
{
	int spl;
	LNTE *e = lnte_;
	unsigned hn = e->handle & LNTE_HANDLE_MASK;
	HANDLE *h = FILES_PTR[hn];
	if (__unlikely(flags & ~(DELETE_LOGICAL_BLOCK | DELETE_LOGICAL_UNUSED | DELETE_LOGICAL_DONT_WAKE_ALL)))
		KERNEL$SUICIDE("KERNEL$UNREGISTER_DEVICE: INVALID FLAGS %X", flags);
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$UNREGISTER_DEVICE AT SPL %08X", KERNEL$SPL);
	RAISE_SPLX(h->op->spl);
	if (__unlikely(flags & DELETE_LOGICAL_UNUSED) && __unlikely(!XLIST_EMPTY(&h->child_list))) goto err;
	if (flags & DELETE_LOGICAL_BLOCK) {
		e->handle = (e->handle & LNTE_PUBLIC) | BLOCK_HANDLE;
		e->dcall = NULL;
		e->dcall_type = NULL;
		e->dctl = NULL;
		e->unload = NULL;
	} else {
		DEL_FROM_LIST(&e->hash);
		KERNEL$UNIVERSAL_FREE(e);
	}
	
	/* Warning: this function must not block up to this point --- to prevent
	   race dcall & unload */

	LOWER_SPL(SPL_DEV);
	CLEAR_SUBPROC_LN_CACHE();
	MULTIOPEN_CLEAR_CACHE();
	RAISE_SPL(SPL_TOP);
	LOWER_SPLX(h->op->spl);
	KERNEL$DETACH_HANDLE(h);
	LOWER_SPL(SPL_DEV);
	free_fd(hn);
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	if (__unlikely(!(flags & DELETE_LOGICAL_DONT_WAKE_ALL))) DO_BREAK(&KERNEL$PROC_KERNEL, NULL, 1);
	LOWER_SPLX(spl);
	return 0;

	err:
	LOWER_SPLX(spl);
	return -EBUSY;
}

int KERNEL$DEVICE_UNLOAD(void *lnte_, char **argv)
{
	LNTE *lnte = lnte_;
	char **arg = argv;
	int state = 0;
	int flags = 0;
	struct __param_table params[] = {
		"BLOCK", __PARAM_BOOL, DELETE_LOGICAL_BLOCK, DELETE_LOGICAL_BLOCK, NULL,
		"UNUSED", __PARAM_BOOL, DELETE_LOGICAL_UNUSED, DELETE_LOGICAL_UNUSED, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &flags, params[1].__p = &flags;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", lnte->name);
		return -EBADSYN;
	}
	return KERNEL$UNREGISTER_DEVICE(lnte, flags);
}

static __finline__ int can_dcall(__const__ char *list, __const__ char *type)
{
	__const__ char *s;
	char c;
	if (__unlikely(!*type)) return 0;
	/*__debug_printf("ss: %p, %p\n", list, type);*/
	s = strstr(list, type);
	again:
	if (__unlikely(!s)) return 0;
	if (__unlikely(s != list) && __unlikely(s[-1] != ',')) goto fail;
	c = s[strlen(type)];
	if (__unlikely(c) && __unlikely(c != ',')) goto fail;
	return 1;
	fail:
	/*__debug_printf("ss2: %p, %p\n", s + 1, type);*/
	s = strstr(s + 1, type);
	goto again;
}

int KERNEL$DCALL(__const__ char *name, __const__ char *devtype, int cmd, void *param)
{
	int r;
	LNTE *e;
	__const__ char *c;
	int h;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$DCALL AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DEV);
	c = name;
	h = 0;
	quickcasehash(c, *c && *c != ':', h);
	h &= LNT_HASH_SIZE - 1;

	XLIST_FOR_EACH(e, &kernel_lnt.hash[h], LNTE, hash) if (__likely(!__strcasexcmp(e->name, name, c))) {
		if (!(e->handle & LNTE_DEVICE) || !e->dcall || !can_dcall(e->dcall_type, devtype)) {
			r = -ENOOP;
			goto ret;
		}
		r = e->dcall(e->dev_ptr, devtype, cmd, param);
		goto ret;
	}
	r = -ENODEV;
	ret:
	LOWER_SPLX(spl);
	return r;
}

typedef struct {
	THREAD_RQ dthread;
	DCTLRQ *drq;
	__DCTL *dctl;
	void *dctl_ptr;
	LIST_ENTRY list;
	char lnm[1];
} DTHREAD;

static DECL_XLIST(dthreads);
static WQ_DECL(dthread_wait, "KERNEL$DTHREAD_WAIT");

extern AST_STUB DTHREAD_DONE;
static long DTHREAD_MAIN(void *p);

DECL_IOCALL(KERNEL$DCTL, SPL_DEV, DCTLRQ)
{
	DTHREAD *d, *dd;
	LNTE *e;
	__const__ char *c;
	int h;
	RQ->error[0] = 0;
	MTX_LOCK(&DEV_MUTEX, RQ, KERNEL$DCTL, RETURN);
	MTX_UNLOCK(&DEV_MUTEX);

	c = RQ->name;
	h = 0;
	quickcasehash(c, *c && *c != ':', h);
	h &= LNT_HASH_SIZE - 1;
	if (__likely(c[0] == ':') && __unlikely(c[1])) {
		_snprintf(RQ->error, __MAX_STR_LEN, "INVALID DEVICE NAME");
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	}

	d = malloc(sizeof(DTHREAD) + c - RQ->name);
	if (__unlikely(!d)) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$DCTL, sizeof(DTHREAD) + c - RQ->name);
		RETURN;
	}
	*(char *)mempcpy(d->lnm, RQ->name, c - RQ->name) = 0;
	__upcase(d->lnm);
	d->drq = RQ;
	XLIST_FOR_EACH(dd, &dthreads, DTHREAD, list) if (__unlikely(!strcmp(dd->lnm, d->lnm))) {
		WQ_WAIT(&dthread_wait, RQ, KERNEL$DCTL);
		__slow_free(d);
		RETURN;
	}
	ADD_TO_XLIST(&dthreads, &d->list);

	XLIST_FOR_EACH(e, &kernel_lnt.hash[h], LNTE, hash) if (__likely(!strcmp(e->name, d->lnm))) {
		if (!(e->handle & LNTE_DEVICE) || !(d->dctl = (!RQ->unload ? e->dctl : e->unload))) {
			RQ->status = -ENOOP;
			_snprintf(RQ->error, __MAX_STR_LEN, "DEVICE %s: DOES NOT SUPPORT %s", e->name, !RQ->unload ? "DCTL" : "UNLOAD");
			goto ret;
		}
		if (RQ->unload) KERNEL$HEAP_CHECK(RQ->name);
		d->dctl_ptr = e->dev_ptr;
		d->dthread.fn = DTHREAD_DONE;
		d->dthread.thread_main = DTHREAD_MAIN;
		d->dthread.p = d;
		d->dthread.error = RQ->error;
		d->dthread.cwd = RQ->cwd;
		d->dthread.std_in = RQ->std_in;
		d->dthread.std_out = RQ->std_out;
		d->dthread.std_err = RQ->std_err;
		d->dthread.dlrq = NULL;
		d->dthread.thread = NULL;
		d->dthread.spawned = 0;
		RETURN_IORQ_CANCELABLE(&d->dthread, KERNEL$THREAD, RQ);
	}
	RQ->status = -ENODEV;
	_snprintf(RQ->error, __MAX_STR_LEN, "NO SUCH DEVICE (%s)", d->lnm);
	ret:
	DEL_FROM_LIST(&d->list);
	free(d);
	WQ_WAKE_ALL(&dthread_wait);
	RETURN_AST(RQ);
}

static long DTHREAD_MAIN(void *p)
{
	DTHREAD *d = p;
	void *unl = NULL;
	DLRQ unload;
	int r = d->dctl(d->dctl_ptr, &unl, d->drq->argv);
	LOWER_SPL(SPL_ZERO);
	if (unl) {
		memcpy(&unload, unl, sizeof(DLRQ));
		SYNC_IO(&unload, KERNEL$DL_UNLOAD_MODULE);
		free(unl);
	}
	return r;
}

DECL_AST(DTHREAD_DONE, SPL_DEV, THREAD_RQ)
{
	DTHREAD *d = GET_STRUCT(RQ, DTHREAD, dthread);
	DCTLRQ *rq;
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, d->drq);
	if (!d->dthread.spawned) {
		if (d->dthread.status != -EINTR) _snprintf(d->drq->error, __MAX_STR_LEN, "ERROR SPAWNING THREAD: %s", strerror(-d->dthread.status));
	}
	d->drq->status = d->dthread.status;
	rq = d->drq;
	DEL_FROM_LIST(&d->list);
	free(d);
	WQ_WAKE_ALL(&dthread_wait);
	if (rq->unload) KERNEL$HEAP_CHECK(rq->name);
	RETURN_AST(rq);
}

DECL_IOCALL(KERNEL$LIST_LOGICALS, SPL_DEV, LOGICAL_LIST_REQUEST)
{
	int i;
	size_t n;
	size_t l;
	size_t prefix_len = strlen(RQ->prefix);
	LNTE *ln;
	LOGICAL_LIST_ENTRY *e, **a;
	INIT_XLIST(&RQ->freelist);
	l = 0;
	n = 0;
	for (i = 0; i < LNT_HASH_SIZE; i++) XLIST_FOR_EACH_UNLIKELY(ln, &kernel_lnt.hash[i], LNTE, hash) if (!_memcasecmp(ln->name, RQ->prefix, prefix_len)) {
		l += (sizeof(LOGICAL_LIST_ENTRY) + strlen(ln->name) + sizeof(unsigned long) - 1) & ~(sizeof(unsigned long) - 1);
		n++;
	}
	if (__unlikely(!n)) {
		RQ->entries = NULL;
		RQ->n_entries = 0;
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	if (__unlikely(!(e = malloc(l)))) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$LIST_LOGICALS, l);
		RETURN;
	}
	if (__unlikely(!(a = malloc(n * sizeof(LOGICAL_LIST_ENTRY *))))) {
		__slow_free(e);
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$LIST_LOGICALS, n * sizeof(LOGICAL_LIST_ENTRY *));
		RETURN;
	}
	RQ->entries = a;
	RQ->n_entries = n;
	ADD_TO_XLIST(&RQ->freelist, &e->list);
	n = 0;
	for (i = 0; i < LNT_HASH_SIZE; i++) XLIST_FOR_EACH_UNLIKELY(ln, &kernel_lnt.hash[i], LNTE, hash) if (!_memcasecmp(ln->name, RQ->prefix, prefix_len)) {
		l = (sizeof(LOGICAL_LIST_ENTRY) + strlen(ln->name) + sizeof(unsigned long) - 1) & ~(sizeof(unsigned long) - 1);
		strcpy(e->name, ln->name);
		e->flags = ln->handle;
		if (ln->handle & LNTE_DEVICE) e->driver_name = ln->driver_name;
		else e->driver_name = NULL;
		RQ->entries[n++] = e;
		e = (LOGICAL_LIST_ENTRY *)((char *)e + l);
	}
	SORT_LOGICAL_LIST(RQ);
	RQ->status = 0;
	RETURN_AST(RQ);
}

static int SET_FILE_FLAG(HANDLE *h, IORQ *rq, int flag)
{
	int spl;
	FFILE *a;
	PROC *p = h->name_addrspace;
	VDESC vd;
	vspace_unmap_t *unmap;
	vd.ptr = (unsigned long)h->name;
	vd.len = (unsigned long)((FFILE *)NULL)->path;
	vd.vspace = (HANDLE *)&p->vspace;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	a = p->vspace.op->vspace_map(&vd, PF_RW, &unmap);
	if (__likely(a != NULL) && __likely(!__IS_ERR(a))) {
		a->flags |= flag;
		unmap(a);
		LOWER_SPLX(spl);
		return 0;
	}
	if (__likely(!a)) {
		DO_PAGEIN_NORET(rq, &vd, PF_RW);
		goto ret1;
	}
	rq->status = __PTR_ERR(a);
	CALL_AST(rq);
	ret1:
	LOWER_SPLX(spl);
	return 1;
}

FFILE *KERNEL$MAP_FILE_STRUCT(HANDLE *h, IORQ *rq, vspace_unmap_t **unmap)
{
	FFILE *a;
	PROC *p = h->file_addrspace;
	VDESC vd;
	if (__unlikely(!p)) {
		if (rq) CALL_IORQ_EXPR(rq, (IO_STUB *)rq->tmp1);
		return NULL;
	}
	vd.ptr = (unsigned long)h->file;
	vd.len = __offsetof(FFILE, path);
	vd.vspace = (HANDLE *)&p->vspace;
	a = p->vspace.op->vspace_map(&vd, PF_RW, unmap);
	if (__likely(a != NULL) && __likely(!__IS_ERR(a))) return a;
	if (__likely(!a)) {
		if (rq) DO_PAGEIN_NORET(rq, &vd, PF_RW);
		return NULL;
	}
	if (rq) {
		rq->status = __PTR_ERR(a);
		CALL_AST(rq);
	}
	return NULL;
}

static __finline__ FFILE *MAP_NAME_STRUCT(HANDLE *h, IORQ *rq, vspace_unmap_t **unmap)
{
	FFILE *a;
	PROC *p = h->name_addrspace;
	VDESC vd;
	vd.ptr = (unsigned long)h->name;
	vd.len = __offsetof(FFILE, path);
	vd.vspace = (HANDLE *)&p->vspace;
	a = p->vspace.op->vspace_map(&vd, PF_RW, unmap);
	if (__likely(a != NULL) && __likely(!__IS_ERR(a))) return a;
	if (__likely(!a)) {
		DO_PAGEIN_NORET(rq, &vd, PF_RW);
		return NULL;
	}
	rq->status = __PTR_ERR(a);
	CALL_AST(rq);
	return NULL;
}

FBLOB *KERNEL$FIND_FILE_BLOB(HANDLE *h, IORQ *rq, __u32 type, size_t *size)
{
	FBLOB_REF *br_addr, *br;
	FBLOB *bl_addr, *bl;
	VDESC vd;
	vspace_unmap_t *unmap;
	FFILE *f;
	int spl;
	u_jiffies_lo_t j;
	if (__unlikely(!h->file_addrspace)) return (void *)1;
	spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), spl)))
		KERNEL$SUICIDE("KERNEL$FIND_FILE_BLOB AT SPL %08X", spl);
	RAISE_SPL(SPL_VSPACE);
	f = MAP_NAME_STRUCT(h, rq, &unmap);
	if (__unlikely(!f)) {
		LOWER_SPLX(spl);
		return NULL;
	}
	br_addr = LIST_STRUCT(f->blobs.next, FBLOB_REF, list);
	unmap(f);
	j = KERNEL$GET_JIFFIES_LO();
	next:
	if (__likely(br_addr == LIST_STRUCT((void *)UPLACE(UDATA_LIST_END), FBLOB_REF, list)) || __likely(br_addr == LIST_STRUCT((void *)&KERNEL$LIST_END, FBLOB_REF, list))) {
		LOWER_SPLX(spl);
		return (void *)1;
	}
	if (__unlikely((unsigned long)br_addr <= 1)) {
		rq->status = -EINVAL;
		call_ast:
		LOWER_SPLX(spl);
		CALL_AST(rq);
		return NULL;
	}
	vd.ptr = (unsigned long)br_addr;
	vd.len = (unsigned long)sizeof(FBLOB_REF);
	vd.vspace = (HANDLE *)&h->name_addrspace->vspace;
	br = h->name_addrspace->vspace.op->vspace_map(&vd, PF_READ, &unmap);
	if (__unlikely(!br)) {
		dopagein:
		LOWER_SPLX(spl);
		DO_PAGEIN_NORET(rq, &vd, PF_READ);
		return NULL;
	}
	if (__unlikely(__IS_ERR(br))) {
		rq->status = __PTR_ERR(br);
		goto call_ast;
	}
	br_addr = LIST_STRUCT(br->list.next, FBLOB_REF, list);
	bl_addr = br->blob;
	vd.ptr = (unsigned long)bl_addr;
	unmap(br);
	vd.len = (unsigned long)sizeof(FBLOB);
	TEST_SPLX(spl, SPL_X(SPL_VSPACE));
	bl = h->name_addrspace->vspace.op->vspace_map(&vd, PF_READ, &unmap);
	if (__unlikely(!bl)) goto dopagein;
	if (__unlikely(__IS_ERR(bl))) {
		rq->status = __PTR_ERR(bl);
		goto call_ast;
	}
	if (__unlikely(bl->type != type)) {
		unmap(bl);
		TEST_SPLX(spl, SPL_X(SPL_VSPACE));
		if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > LOCKUP_TIMEOUT) && h->name_addrspace != &KERNEL$PROC_KERNEL) {
			rq->status = -EINTR;
			goto call_ast;
		}
		goto next;
	}
	*size = bl->size;
	unmap(bl);
	LOWER_SPLX(spl);
	return bl_addr;
}

HANDLE *KERNEL$PROC_VSPACE(PROC *p)
{
	return (HANDLE *)&p->vspace;
}

#define CHECK_OFFSET(str,val)				\
if ((long)&((str *)NULL)->val != OFF_##str##_##val)	\
	KERNEL$SUICIDE("MISCOMPILED KERNEL: ENTRY OFF_"#str"_"#val" DOES NOT MATCH. FIX STRUCTURE OFFSETS IN DEV_OFF.H OR UDATADEF.H");

#define CHECK_OFFSET2(str,val1,val2)			\
if ((long)&((str *)NULL)->val1.val2 != OFF_##str##_##val1##_##val2)	\
	KERNEL$SUICIDE("MISCOMPILED KERNEL: ENTRY OFF_"#str"_"#val1"_"#val2" DOES NOT MATCH. FIX STRUCTURE OFFSETS IN DEV_OFF.H");

static __finline__ void dev_check_offsets(void)
{
	ARCH_CHECK_OFFSETS;
	CHECK_OFFSET(TIMER, fn);
	CHECK_OFFSET(IORQ, fn);
	CHECK_OFFSET(IORQ, tmp1);
	CHECK_OFFSET(IORQ, tmp2);
	CHECK_OFFSET(IORQ, tmp3);
	CHECK_OFFSET(IORQ, status);
	CHECK_OFFSET(HANDLE, op);
	CHECK_OFFSET(SIORQ, h);
	CHECK_OFFSET2(SIORQ, v, ptr);
	CHECK_OFFSET2(SIORQ, v, len);
	CHECK_OFFSET2(SIORQ, v, vspace);
	CHECK_OFFSET(SIORQ, progress);
	CHECK_OFFSET(SIORQ, handle);
	CHECK_OFFSET(AIORQ, h);
	CHECK_OFFSET2(AIORQ, v, ptr);
	CHECK_OFFSET2(AIORQ, v, len);
	CHECK_OFFSET2(AIORQ, v, vspace);
	CHECK_OFFSET(AIORQ, pos);
	CHECK_OFFSET(AIORQ, progress);
	CHECK_OFFSET(AIORQ, handle);
	CHECK_OFFSET(IOCTLRQ, h);
	CHECK_OFFSET2(IOCTLRQ, v, ptr);
	CHECK_OFFSET2(IOCTLRQ, v, len);
	CHECK_OFFSET2(IOCTLRQ, v, vspace);
	CHECK_OFFSET(IOCTLRQ, ioctl);
	CHECK_OFFSET(IOCTLRQ, param);
	CHECK_OFFSET(IOCTLRQ, handle);
	CHECK_OFFSET(BIORQ, h);
	CHECK_OFFSET(BIORQ, handle);
	CHECK_OFFSET(PACKET, h);
	CHECK_OFFSET(PACKET, handle);
	CHECK_OFFSET(HANDLE_OPERATIONS, read);
	CHECK_OFFSET(HANDLE_OPERATIONS, write);
	CHECK_OFFSET(HANDLE_OPERATIONS, aread);
	CHECK_OFFSET(HANDLE_OPERATIONS, awrite);
	CHECK_OFFSET(HANDLE_OPERATIONS, ioctl);
	CHECK_OFFSET(HANDLE_OPERATIONS, bio);
	CHECK_OFFSET(HANDLE_OPERATIONS, pktio);
	CHECK_OFFSET(PROC, vspace);
	CHECK_OFFSET(UDATA, base);
	CHECK_OFFSET(UDATA, lnt);
	CHECK_OFFSET(UDATA, xcpt_handler);
	CHECK_OFFSET(UDATA, xcpt_available);
	CHECK_OFFSET(UDATA, xcpt_return);
	CHECK_OFFSET(UDATA, xcpt_type);
	CHECK_OFFSET(UDATA, xcpt_address);
	CHECK_OFFSET(UDATA, xcpt_error);
	CHECK_OFFSET(UDATA, timer_handler);
	CHECK_OFFSET(UDATA, timer_available);
	CHECK_OFFSET(UDATA, vspace);
	CHECK_OFFSET(UDATA, environ);
	CHECK_OFFSET(UDATA, environ_callbacks);
	CHECK_OFFSET(UDATA, environ_list);
	CHECK_OFFSET(UDATA, progname);
	CHECK_OFFSET(UDATA, exit_func);
	CHECK_OFFSET(UDATA, errno);
	CHECK_OFFSET(UDATA, repost_ast_spl);
	CHECK_OFFSET(UDATA, repost_ast_fn);
	CHECK_OFFSET(UDATA, current_thread);
	CHECK_OFFSET(UDATA, init_thread);
	CHECK_OFFSET(UDATA, isthreaded);
	CHECK_OFFSET(UDATA, multiopen);
	CHECK_OFFSET(UDATA, subproc_ctl);
	CHECK_OFFSET(UDATA, logical_wait);

	if (sizeof(_ABIORQ) > sizeof(BIORQ) || sizeof(_IBIORQ) > sizeof(BIORQ))
		KERNEL$SUICIDE("MISCOMPILED KERNEL: SIZEOF(BIORQ) == %d, SIZEOF(_ABIORQ) == %d, SIZEOF(_IBIORQ) == %d", (int)sizeof(BIORQ), (int)sizeof(_ABIORQ), (int)sizeof(_IBIORQ));

	if (((sizeof(PROC) + (PROC_ALIGN - 1)) & ~(PROC_ALIGN - 1)) != SIZEOF_PROC_ALIGNED)
		KERNEL$SUICIDE("MISCOMPILED KERNEL: SIZEOF(PROC) == %d, SIZEOF_PROC_ALIGNED == %d, PROC ALIGN == %d", sizeof(PROC), SIZEOF_PROC_ALIGNED, PROC_ALIGN);
}

void DEV_INIT(void)
{
	int i;
	dev_check_offsets();
	for (i = 0; i < LNT_HASH_SIZE; i++) INIT_XLIST(&kernel_lnt.hash[i]);
	KERNEL$PROC_ACCOUNT = &KERNEL$PROC_KERNEL;
}

