#include <KERNEL/DEV.H>
#include <KERNEL/VMDEF.H>
#include <KERNEL/KSPAWN.H>
#include <SPAD/BIO.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>
#include <SPAD/HASH.H>
#include <STDLIB.H>
#include <LIMITS.H>
#include <TIME.H>
#include <SPAD/PKT.H>
#include <UNISTD.H>
#include <SPAD/LINK.H>
#include <KERNEL/LINK.H>

MTX_DECL(DEV_MUTEX, "KERNEL$DEV_MUTEX");

static void PENDING_DELETE(FFILE *f);
static void FREE_BLOB_REF(FBLOB_REF *br);

int UNREF_FILE(FFILE *f)
{
#if __DEBUG >= 1
	if (__unlikely(!f->l_refcount))
		KERNEL$SUICIDE("UNREF_FILE: FILE %p HAS ZERO REFCOUNT", f);
#endif
	if (__likely(!--f->l_refcount)) {
		FREE_PIPE(f);
		while (__unlikely(!XLIST_EMPTY(&f->blobs))) {
			FBLOB_REF *br = LIST_STRUCT(f->blobs.next, FBLOB_REF, list);
			FREE_BLOB_REF(br);
		}
		if (__unlikely(f->flags & _O_PENDING_DELETE)) {
			PENDING_DELETE(f);
		} else {
			free(f);
		}
		return 1;
	}
	return 0;
}

void FREE_PIPE_INTERNAL(FFILE *f)
{
	FPIPE *p = (FPIPE *)(unsigned long)f->pos;
	if (__likely(p != NULL) && !--p->l_refcount) {
		KERNEL$FREE_KERNEL_PAGE(p, VM_TYPE_WIRED_MAPPED);
	}
}

DECL_XLIST(PENDING_DELETE_LIST);
extern AST_STUB PENDING_DELETE_FN;

static void PENDING_DELETE(FFILE *f)
{
	static OPENRQ PENDING_DELETE_RQ;
	if (XLIST_EMPTY(&PENDING_DELETE_LIST)) {
		PENDING_DELETE_RQ.fn = PENDING_DELETE_FN;
		PENDING_DELETE_RQ.status = 1;
		CALL_AST(&PENDING_DELETE_RQ);
	}
	ADD_TO_XLIST(&PENDING_DELETE_LIST, (LIST_ENTRY *)(void *)f);
}

DECL_AST(PENDING_DELETE_FN, SPL_DEV, OPENRQ)
{
	FFILE *f;
	if (RQ->status != 1) {
		f = (FFILE *)PENDING_DELETE_LIST.next;
		DEL_FROM_LIST((LIST_ENTRY *)f);
		free(f);
		if (__likely(XLIST_EMPTY(&PENDING_DELETE_LIST))) RETURN;
	}
	f = (FFILE *)PENDING_DELETE_LIST.next;
	RQ->flags = _O_DELETE | _O_CLOSE | _O_NOPOSIX;
	RQ->path = f->path;
	RQ->cwd = NULL;
	RETURN_IORQ(RQ, KERNEL$OPEN);
}

static void FREE_BLOB_REF(FBLOB_REF *br)
{
	KERNEL$UNREF_BLOB(br->blob);
	DEL_FROM_LIST(&br->list);
	free(br);
}

void KERNEL$UNREF_BLOB(FBLOB *b)
{
	int spl = KERNEL$SPL;
	if (SPLX_BELOW(spl, SPL_X(SPL_DEV))) RAISE_SPL(SPL_DEV);
#if __DEBUG >= 1
	if (__unlikely(!b->l_refcount))
		KERNEL$SUICIDE("KERNEL$UNREF_BLOB: FILE'S BLOB %08X (AT %p) HAS ZERO REFCOUNT", b->type, b);
#endif
	if (__likely(!--b->l_refcount)) {
		free(b);
	}
	LOWER_SPLX(spl);
}

void SET_FILE_BLOB(FFILE *f, FBLOB_REF *br, FBLOB *bl, void **free1, void **free2)
{
	FBLOB_REF *bs;
	*free1 = *free2 = NULL;
	br->blob = bl;
	bl->l_refcount++;
	XLIST_FOR_EACH(bs, &f->blobs, FBLOB_REF, list) if (bs->blob->type == bl->type) {
		FBLOB *orig_blob = bs->blob;
		if (__unlikely(!bl->size)) {
			DEL_FROM_LIST_ORDERED(&bs->list);
			*free2 = bs;
			goto unref_free_br_ret;
		}
		bs->blob = br->blob;
		unref_free_br_ret:
#if __DEBUG >= 1
		if (__unlikely(!orig_blob->l_refcount))
			KERNEL$SUICIDE("SET_FILE_BLOB: FILE'S BLOB %08X (AT %p) HAS ZERO REFCOUNT", orig_blob->type, orig_blob);
#endif
		*free1 = !--orig_blob->l_refcount ? orig_blob : NULL;
		free_br_ret:
		free(br);
		return;
	}
	if (__unlikely(!bl->size)) goto free_br_ret;
	ADD_TO_XLIST_ORDERED(&f->blobs, &br->list);
	return;
}

DECL_IOCALL(KERNEL$NO_OPERATION, SPL_TOP, IORQ)
{
	/* We could do SWITCH_PROC_ACCOUNT here, but it's not needed because
	   this is called directly from process without blocking.
	   We can't call it because we are on SPL_TOP and we don't know which
	   request type we have (SIORQ, AIORQ, IOCTLRQ, BIORQ, PACKET) */
	RQ->status = -ENOOP;
	RETURN_AST(RQ);
}

static IORQ geth;
static VDESC geth_vdesc;
static WQ_DECL(geth_wait, "KERNEL$GETH_WAIT");
static IORQ * volatile geth_caller = NULL;
static int geth_wr;

extern IO_STUB NOOP_CONT_PAGEIN;

static IORQ *NOOP_GET_PAGEIN_RQ(VDESC *desc, IORQ *rq, int wr)
{
	if (__unlikely(!geth_caller)) {
		WQ_WAIT_F(&geth_wait, rq);
		return NULL;
	}
	memcpy(&geth_vdesc, desc, sizeof(VDESC));
	geth_caller = rq;
	geth_wr = wr;
	geth.tmp1 = (unsigned long)NOOP_CONT_PAGEIN;
	return &geth;
}

DECL_IOCALL(NOOP_CONT_PAGEIN, SPL_DEV, IORQ)
{
	IORQ *caller;
	HANDLE *h = geth_vdesc.vspace;
	if (__unlikely(h->op != &KERNEL$HANDLE_NOOP)) goto x;
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
		WQ_WAIT_F(&KERNEL$LOCKUP_EVENTS, geth_caller);
		goto free_geth;
	}
	/* SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV)); will be done either by GET_HANDLE or RETURN_IORQ_LSTAT */
	h = GET_HANDLE(h->name_addrspace, h->handle_num & HANDLE_NUM_MASK, 0, geth_caller);
	if (!h) {
		free_geth:
		RAISE_SPL(SPL_VSPACE);
		geth_caller = NULL;
		WQ_WAKE_ALL(&geth_wait);
		LOWER_SPL(SPL_DEV);
		RETURN;
	}
	if (__unlikely(__IS_ERR(h))) {
		caller = geth_caller;
		caller->status = __PTR_ERR(h);
		RAISE_SPL(SPL_VSPACE);
		geth_caller = NULL;
		WQ_WAKE_ALL(&geth_wait);
		LOWER_SPL(SPL_DEV);
		RETURN_AST(caller);
	}
	caller = geth_caller;
	RAISE_SPL(SPL_VSPACE);
	geth_caller = NULL;
	WQ_WAKE_ALL(&geth_wait);
	LOWER_SPL(SPL_DEV);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	x:
	RAISE_SPL(SPL_VSPACE);
	caller = geth_caller;
	geth_caller = NULL;
	WQ_WAKE_ALL(&geth_wait);
	DO_PAGEIN(caller, &geth_vdesc, geth_wr);
}

#define NOP_OP(name, opp, rq, call)					\
DECL_IOCALL(name, SPL_DEV, rq)						\
{									\
	HANDLE *h = RQ->handle;						\
	if (__unlikely(h->op != &KERNEL$HANDLE_NOOP)) goto x;		\
	RQ->tmp1 = (unsigned long)name;					\
	TEST_LOCKUP_ENTRY(RQ, RETURN);					\
	/* SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV)); will be done either by GET_HANDLE or RETURN_IORQ_LSTAT */				\
	h = GET_HANDLE(h->name_addrspace, h->handle_num & HANDLE_NUM_MASK, 0, (IORQ *)RQ);								\
	if (!h) RETURN;							\
	if (__unlikely(__IS_ERR(h))) {					\
		RQ->status = __PTR_ERR(h);				\
		call(RQ);						\
	}								\
	x:								\
	RETURN_IORQ_LSTAT(RQ, opp);					\
}

NOP_OP(NOOP_READ, KERNEL$WAKE_READ, SIORQ, RETURN_AST)
NOP_OP(NOOP_WRITE, KERNEL$WAKE_WRITE, SIORQ, RETURN_AST)
NOP_OP(NOOP_AREAD, KERNEL$WAKE_AREAD, AIORQ, RETURN_AST)
NOP_OP(NOOP_AWRITE, KERNEL$WAKE_AWRITE, AIORQ, RETURN_AST)
NOP_OP(NOOP_IOCTL, KERNEL$WAKE_IOCTL, IOCTLRQ, RETURN_AST)
NOP_OP(NOOP_BIO, KERNEL$WAKE_BIO, BIORQ, RETURN_AST)
NOP_OP(NOOP_PKTIO, KERNEL$WAKE_PKTIO, PACKET, RETURN_PKT)

__const__ HANDLE_OPERATIONS KERNEL$HANDLE_NOOP = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	NOOP_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	NOOP_READ,
	NOOP_WRITE,
	NOOP_AREAD,
	NOOP_AWRITE,
	NOOP_IOCTL,
	NOOP_BIO,
	NOOP_PKTIO
};

__const__ HANDLE_OPERATIONS empty_handle = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,

	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
};

extern AST_STUB bio_emu_endrq;
extern AST_STUB bio_emu_endflush;

DECL_IOCALL(KERNEL$BIO_EMU, SPL_FS, BIORQ)
{
	_u_off_t o;
	BIODESC *desc;
	SWITCH_PROC_ACCOUNT(RQ->proc, SPL_X(SPL_FS));
#if __DEBUG >= 1
	if (__unlikely(RQ->fault_sec != -1))
		KERNEL$SUICIDE("KERNEL$BIO_EMU: fault_sec NOT SET: %"__sec_t_format"X, AST %s", RQ->fault_sec, KERNEL$DL_GET_SYMBOL_NAME(RQ->fn, (unsigned long *)(void *)&KERNEL$LIST_END, 0));
#endif
	if (__likely(!(RQ->flags & (BIO_WRITE | BIO_FLUSH)))) {
		RQ->tmp1 = (unsigned long)KERNEL$AREAD;
		goto io;
	} else if (__likely(RQ->flags & BIO_WRITE)) {
		RQ->tmp1 = (unsigned long)KERNEL$AWRITE;
		goto io;
	} else if (__likely(RQ->flags & BIO_FLUSH)) {
#define IRQ	(&((_IBIORQ *)RQ)->i)
		IRQ->h = RQ->h;
		__barrier();
		IRQ->fn = bio_emu_endflush;
		IRQ->ioctl = IOCTL_FSYNC;
		IRQ->param = PARAM_FSYNC_ONLY_DATA;
		IRQ->v.ptr = 0;
		IRQ->v.len = 0;
		IRQ->v.vspace = &KERNEL$VIRTUAL;
		RETURN_IORQ_CANCELABLE(IRQ, KERNEL$IOCTL, RQ);
#undef IRQ
	} else KERNEL$SUICIDE("KERNEL$BIO_EMU: INVALID FLAGS %X", RQ->flags);
	io:
	/*__debug_printf("BIO EMU: %p->(%d)%p\n", RQ, __offsetof(BIORQ, proc), RQ->proc);*/
	if (__unlikely(RQ->sec > (_u_off_t)-1 >> BIO_SECTOR_SIZE_BITS)) {
		RQ->status = -ERANGE;
		RETURN_AST(RQ);
	}
	o = (_u_off_t)RQ->sec << BIO_SECTOR_SIZE_BITS;
#define ARQ	(&((_ABIORQ *)RQ)->a)
	desc = RQ->desc;
	/* handling of page-faults is disabled in KERNEL$PAGEIN if AST is bio_emu_endrq */
	RQ->tmp3 = (unsigned long)desc->next;
	ARQ->h = RQ->h;
	__barrier();
	ARQ->fn = bio_emu_endrq;
	ARQ->v.ptr = desc->v.ptr;
	ARQ->v.len = desc->v.len;
	ARQ->v.vspace = desc->v.vspace;
	ARQ->pos = o;
	ARQ->progress = 0;
	RETURN_IORQ_CANCELABLE(ARQ, (IO_STUB *)RQ->tmp1, RQ);
#undef ARQ
}

DECL_AST(bio_emu_endrq, SPL_FS, AIORQ)
{
	_ABIORQ *a = GET_STRUCT(RQ, _ABIORQ, a);
	BIODESC *desc;
#define b	 ((BIORQ *)a)
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, a);
	/*__debug_printf("BIO EMU ENDRQ: %p->(%d)%p\n", RQ, __offsetof(BIORQ, proc), RQ->proc);*/
	if (__unlikely(a->a.status <= 0)) {
		int h;
		b->status = __likely(a->a.status != 0) ? a->a.status : -ERANGE;
		h = a->a.h;
		b->fault_sec = a->a.pos >> BIO_SECTOR_SIZE_BITS;
		__barrier();
		b->h = h;
		RETURN_AST(b);
	}
	if (__unlikely(a->a.v.len != 0)) {
		a->a.progress = 0;
		RETURN_IORQ_CANCELABLE(&a->a, (IO_STUB *)b->tmp1, b);
	}
	desc = (BIODESC *)a->tmp3;
	if (__likely(!desc)) {
		b->status = 0;
		b->h = a->a.h;
		RETURN_AST(b);
	}
	b->tmp3 = (unsigned long)desc->next;
	a->a.v.ptr = desc->v.ptr;
	a->a.v.len = desc->v.len;
	a->a.v.vspace = desc->v.vspace;
	a->a.progress = 0;
	RETURN_IORQ_CANCELABLE(&a->a, (IO_STUB *)b->tmp1, b);
#undef b
}

DECL_AST(bio_emu_endflush, SPL_FS, IOCTLRQ)
{
	BIORQ *b = (BIORQ *)GET_STRUCT(RQ, _IBIORQ, i);
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, b);
	b->status = ((_IBIORQ *)b)->i.status;
	__barrier();
	b->fault_sec = -1;
	RETURN_AST(b);
}

int substitute_option(char *src, char *dest, char *option, char *value)
{
	char *p;
	p = strchr(src, ':');
	if (__unlikely(!p)) return -EBADSYN;
	p++;
	if (__unlikely(!*p)) {
		e:
		if (value) {
			strcpy(dest, src);
			if (dest[strlen(dest) - 1] == '/') strcat(dest, "^");
			else strcat(dest, "/^");
			strcat(dest, option);
			if (*value) {
				strcat(dest, "=");
				strcat(dest, value);
			}
			return 1;
		} else return 0;
	}
	f:
	if (*p == '^') {
		char *o = option, *op = p + 1;
		while (*o && __upcasechr(*o) == __upcasechr(*op)) o++, op++;
		if (*o || (*op && *op != '=' && *op != '/')) {
			p = op;
			goto d;
		}
		if (value) {
			char *oo, *oop;
			if (!*op || *op == '/') {
				if (!*value) return 0;
				memcpy(dest, src, op - src);
				strcpy(dest + (op - src), "=");
				strcat(dest + (op - src), value);
				strcat(dest + (op - src), op);
				return 1;
			}
			op++;
			oo = value, oop = op;
			while (*oo && __upcasechr(*oo) == __upcasechr(*oop)) oo++, oop++;
			if (!*oo && (!*oop || *oop == '/')) return 0;
			while (*oop && *oop != '/') oop++;
			if (!*value) op--;
			memcpy(dest, src, op - src);
			strcpy(dest + (op - src), value);
			strcat(dest + (op - src), oop);
			return 1;
		} else {
			memcpy(dest, src, p - src);
			while (*op && *op != '/') op++;
			if (dest[p - src - 1] == '/' && *op == '/') op++;
			strcpy(dest + (p - src), op);
			return 1;
		}
	}
	d:
	p = strchr(p, '/');
	if (!p) goto e;
	p++;
	goto f;
}

#define split	put_ioctl_split
#define fn	KERNEL$PUT_IOCTL_STRUCT
#define put	1

#include "GETPUTIO.I"

#define split	get_ioctl_split
#define fn	KERNEL$GET_IOCTL_STRUCT
#define put	0

#include "GETPUTIO.I"

extern AST_STUB SETLOG_CHANGE_HANDLE_BLOB_DONE;

DECL_IOCALL(KERNEL$SET_LOGICAL, SPL_DEV, LOGICAL_REQUEST)
{
	int h;
	LNTE *e;
	__const__ char *c;
	char *d;
	int newh;

#if __DEBUG >= 1
	if (__unlikely(RQ->flags & ~(LNTE_PUBLIC | LNTE_BLOCK_IF_NOT_FOUND | LNTE_FILE_SHARE | LNTE_DONT_WAKE_ALL)) || __unlikely(RQ->flags_chg_mask & ~(LNTE_PUBLIC | LNTE_BLOCK_IF_NOT_FOUND | LNTE_FILE_SHARE)))
		KERNEL$SUICIDE("KERNEL$SET_LOGICAL: INVALID FLAGS: %X/%X (LNM %s->%s)", RQ->flags, RQ->flags_chg_mask, RQ->alias, RQ->name);
#endif
	c = RQ->name;
	if (__check_logical_name((char *)c, 0)) {
		RQ->status = -EBADSYN;
		RETURN_AST(RQ);
	}
	h = 0;
	quickcasehash(c, *c && *c != ':', h);
	h &= LNT_HASH_SIZE - 1;

	XLIST_FOR_EACH(e, &LN->hash[h], LNTE, hash) if (__likely(!__strcasexcmp(e->name, RQ->name, c))) goto found;
	if (__unlikely(KERNEL$KERNEL)) {
		RQ->status = -ENOLNM;
		RETURN_AST(RQ);
	}
	RQ->alias = malloc(strlen(RQ->name) + 3);
	if (__unlikely(!RQ->alias)) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$SET_LOGICAL, strlen(RQ->name) + 3);
		RETURN;
	}
	((char *)RQ->alias)[0] = '@';
	d = stpcpy((char *)RQ->alias + 1, RQ->name);
	if (d[-1] != ':') {
		d[0] = ':';
		d[1] = 0;
	}
	__upcase((char *)RQ->alias);
	RQ->cwd = (void *)1;
		/* !!! TODO: query public state, if not chaging it */
	RETURN_IORQ_LSTAT(RQ, KERNEL$CREATE_LOGICAL);

	found:
	if (__unlikely(e->handle & LNTE_DEVICE) && __unlikely(((unsigned long)RQ->chh_option | (unsigned long)RQ->blob) != 0)) {
		RQ->status = -EOPNOTSUPP;
		RETURN_AST(RQ);
	}
	newh = (e->handle & (~RQ->flags_chg_mask | LNTE_HANDLE_MASK | LNTE_DEVICE)) | (RQ->flags & (LNTE_PUBLIC | LNTE_BLOCK_IF_NOT_FOUND | LNTE_FILE_SHARE));
	if ((unsigned long)RQ->chh_option | (unsigned long)RQ->blob) {
		newh = (newh & ~LNTE_PUBLIC) | (e->handle & newh & LNTE_PUBLIC);
	}
	if (newh != e->handle) {
		e->handle = newh;
		WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
		/*DO_BREAK(&KERNEL$PROC_KERNEL, NULL, 1); done in SWAP_ALL_SUBPROCESSES*/
		MULTIOPEN_CLEAR_CACHE();
		SWAP_ALL_SUBPROCESSES();
		CLEAR_SUBPROC_LN_CACHE();
	}
	if (RQ->chh_option) {
		RQ->u.chh.fn = SETLOG_CHANGE_HANDLE_BLOB_DONE;
		RQ->u.chh.h = newh & LNTE_HANDLE_MASK;
		RQ->u.chh.option = RQ->chh_option;
		RQ->u.chh.value = RQ->chh_value;
		RQ->chh_option = NULL;
		RETURN_IORQ_CANCELABLE(&RQ->u.chh, KERNEL$CHANGE_HANDLE, RQ);
	}
	if (__unlikely(RQ->blob != NULL)) {
		RQ->u.chh.fn = SETLOG_CHANGE_HANDLE_BLOB_DONE;
		RQ->u.brq.h = newh & LNTE_HANDLE_MASK;
		RQ->u.brq.blob = RQ->blob;
		RQ->blob = NULL;
		RETURN_IORQ_CANCELABLE(&RQ->u.brq, KERNEL$SET_HANDLE_BLOB, RQ);
	}
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_AST(SETLOG_CHANGE_HANDLE_BLOB_DONE, SPL_DEV, IORQ)
{
	LOGICAL_REQUEST *lrq = GET_STRUCT(RQ, LOGICAL_REQUEST, u.chh);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, lrq);
	if (__unlikely(lrq->u.chh.status < 0)) {
		lrq->status = lrq->u.chh.status;
		RETURN_AST(lrq);
	}
	RETURN_IORQ_LSTAT(lrq, KERNEL$SET_LOGICAL);
}

int KERNEL$DELETE_LOGICAL(char *lnm, int flags)
{
	int r;
	int spl;
	int h;
	LNTE *e;
	char *c;
	int handle_to_close;
#if __DEBUG >= 1
	if (__unlikely(flags & ~(DELETE_LOGICAL_BLOCK | DELETE_LOGICAL_DONT_WAKE_ALL)))
		KERNEL$SUICIDE("KERNEL$DELETE_LOGICAL: INVALID FLAGS: %X (LNM %s)", flags, lnm);
#endif
	if (SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL)) KERNEL$SUICIDE("KERNEL$DELETE_LOGICAL AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DEV);

	if (__check_logical_name(lnm, 0)) {
		r = -EBADSYN;
		goto ret;
	}
	c = lnm;
	h = 0;
	quickcasehash(c, *c && *c != ':', h);
	h &= LNT_HASH_SIZE - 1;

	XLIST_FOR_EACH(e, &LN->hash[h], LNTE, hash) if (__likely(!__strcasexcmp(e->name, lnm, c))) goto found;
	r = -ENOLNM;
	goto ret;

	found:
	if (e->handle & LNTE_DEVICE) {
		r = -EBUSY;
		goto ret;
	}
	if (flags & DELETE_LOGICAL_BLOCK) {
		e->handle = (e->handle & (LNTE_PUBLIC | LNTE_FILE_SHARE)) | BLOCK_HANDLE;
	} else {
		DEL_FROM_LIST_ORDERED(&e->hash);
		KERNEL$UNIVERSAL_FREE(e);
	}
	MULTIOPEN_CLEAR_CACHE();
	if (__likely(!(flags & DELETE_LOGICAL_DONT_WAKE_ALL)))
		SWAP_ALL_SUBPROCESSES(); /* page mappings are not bound on handles --- need to erase them */
	CLEAR_SUBPROC_LN_CACHE();
	handle_to_close = e->handle & LNTE_HANDLE_MASK;
	if (__likely(handle_to_close != BLOCK_HANDLE)) KERNEL$FAST_CLOSE(handle_to_close);
	else UNBLOCK_LNT();
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	/* DO_BREAK(&KERNEL$PROC_KERNEL, NULL, 1); done in SWAP_ALL_SUBPROCESSES */
	r = 0;

	ret:
	LOWER_SPLX(spl);
	return r;
}

extern AST_STUB LOG_OPEN;
static void LOG_LNTE_ALLOC(LOGICAL_REQUEST *lrq);
extern AST_STUB LOG_CHANGE_HANDLE_BLOB_DONE;
extern AST_STUB LOG_LNTE_ALLOCATED;

DECL_IOCALL(KERNEL$CREATE_LOGICAL, SPL_DEV, LOGICAL_REQUEST)
{
#if __DEBUG >= 1
	if (__unlikely(RQ->flags & ~(LNTE_PUBLIC | LNTE_BLOCK_IF_NOT_FOUND | LNTE_FILE_SHARE | LNTE_DONT_WAKE_ALL)))
		KERNEL$SUICIDE("KERNEL$CREATE_LOGICAL: INVALID FLAGS: %X (LNM %s->%s)", RQ->flags, RQ->alias, RQ->name);
#endif
	RQ->u.open.fn = LOG_OPEN;
	RQ->u.open.flags = (__unlikely(RQ->flags & LNTE_FILE_SHARE) ? O_RDWR : _O_NOACCESS) | _O_NOPOSIX;
	RQ->u.open.path = RQ->alias;
	RQ->u.open.cwd = RQ->cwd != (void *)1 ? RQ->cwd : NULL;
	RETURN_IORQ_CANCELABLE(&RQ->u.open, KERNEL$OPEN, RQ);
}

DECL_AST(LOG_OPEN, SPL_DEV, OPENRQ)
{
	LOGICAL_REQUEST *lrq = GET_STRUCT(RQ, LOGICAL_REQUEST, u.open);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, lrq);
	if (__unlikely(lrq->u.open.status < 0)) {
		if (__likely((lrq->u.open.flags & 7) != _O_NOACCESS)) {
			if (__likely((lrq->u.open.flags & 7) != O_RDONLY)) lrq->u.open.flags--;
			else lrq->u.open.flags = (lrq->u.open.flags & ~7) | _O_NOACCESS;
			RETURN_IORQ_CANCELABLE(&lrq->u.open, KERNEL$OPEN, lrq);
		}
		if (__unlikely(lrq->cwd == (void *)1)) free((char *)lrq->alias);
		lrq->status = lrq->u.open.status;
		RETURN_AST(lrq);
	}
	lrq->flags_chg_mask = lrq->u.open.status;
	LOG_LNTE_ALLOC(lrq);
	RETURN;
}

static void LOG_LNTE_ALLOC(LOGICAL_REQUEST *lrq)
{
	if (__unlikely(lrq->chh_option != NULL)) {
		lrq->u.chh.fn = LOG_CHANGE_HANDLE_BLOB_DONE;
		lrq->u.chh.h = lrq->flags_chg_mask;
		lrq->u.chh.option = lrq->chh_option;
		lrq->u.chh.value = lrq->chh_value;
		lrq->chh_option = NULL;
		CALL_IORQ_CANCELABLE(&lrq->u.chh, KERNEL$CHANGE_HANDLE, lrq);
		return;
	}
	if (__unlikely(lrq->blob != NULL)) {
		lrq->u.chh.fn = LOG_CHANGE_HANDLE_BLOB_DONE;
		lrq->u.brq.h = lrq->flags_chg_mask;
		lrq->u.brq.blob = lrq->blob;
		lrq->blob = NULL;
		CALL_IORQ_CANCELABLE(&lrq->u.brq, KERNEL$SET_HANDLE_BLOB, lrq);
		return;
	}
	lrq->u.mrq.fn = LOG_LNTE_ALLOCATED;
	lrq->u.mrq.size = sizeof(LNTE) + strlen(lrq->name);
	CALL_IORQ_CANCELABLE(&lrq->u.mrq, KERNEL$UNIVERSAL_MALLOC, lrq);
}

DECL_AST(LOG_CHANGE_HANDLE_BLOB_DONE, SPL_DEV, IORQ)
{
	LOGICAL_REQUEST *lrq = GET_STRUCT(RQ, LOGICAL_REQUEST, u.chh);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, lrq);
	if (__unlikely(lrq->u.chh.status < 0)) {
		KERNEL$FAST_CLOSE(lrq->flags_chg_mask);
		if (__unlikely(lrq->cwd == (void *)1)) free((char *)lrq->alias);
		lrq->status = lrq->u.chh.status;
		RETURN_AST(lrq);
	}
	LOG_LNTE_ALLOC(lrq);
	RETURN;
}

DECL_AST(LOG_LNTE_ALLOCATED, SPL_DEV, MALLOC_REQUEST)
{
	LNTE *lnte, *e;
	int h, r;
	char *c;
	LOGICAL_REQUEST *lrq = GET_STRUCT(RQ, LOGICAL_REQUEST, u.mrq);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, lrq);
	if (__unlikely(lrq->u.mrq.status < 0)) {
		KERNEL$FAST_CLOSE(lrq->flags_chg_mask);
		if (__unlikely(lrq->cwd == (void *)1)) free((char *)lrq->alias);
		lrq->status = lrq->u.mrq.status;
		RETURN_AST(lrq);
	}
	lnte = RQ->ptr;
	lnte->handle = lrq->flags_chg_mask | (lrq->flags & (LNTE_PUBLIC | LNTE_BLOCK_IF_NOT_FOUND | LNTE_FILE_SHARE));
	lnte->dcall = NULL;
	lnte->dcall_type = NULL;
	lnte->dctl = NULL;
	lnte->unload = NULL;
	strcpy(lnte->name, lrq->name);
	if (__unlikely(__check_logical_name(lnte->name, 1))) {
		KERNEL$FAST_CLOSE(lrq->flags_chg_mask);
		KERNEL$UNIVERSAL_FREE(lnte);
		if (__unlikely(lrq->cwd == (void *)1)) free((char *)lrq->alias);
		lrq->status = -EINVAL;
		RETURN_AST(lrq);
	}
	c = lnte->name;
	h = 0;
	quickcasehash(c, *c, h);
	h &= LNT_HASH_SIZE - 1;

	XLIST_FOR_EACH(e, &LN->hash[h], LNTE, hash) if (__likely(!__strcasexcmp(e->name, lnte->name, c))) {
		char *h1, *h2;
		if (__unlikely(e->handle & LNTE_DEVICE)) goto do_delete;
		h1 = KERNEL$HANDLE_PATH(lnte->handle & LNTE_HANDLE_MASK);
		h2 = KERNEL$HANDLE_PATH(e->handle & LNTE_HANDLE_MASK);
		if (__unlikely(!h1) || __unlikely(!h2)) goto do_delete;
		if (strcmp(h1, h2)) goto do_delete;
		if (__unlikely((lnte->handle & ~LNTE_HANDLE_MASK) != (e->handle & ~LNTE_HANDLE_MASK))) goto do_delete;
		r = 0;
		goto free_this;
		do_delete:
		if (__unlikely(r = KERNEL$DELETE_LOGICAL(e->name, 0))) {
			if (r == -EBUSY) r = -EEXIST;
			free_this:
			KERNEL$FAST_CLOSE(lrq->flags_chg_mask);
			KERNEL$UNIVERSAL_FREE(lnte);
			if (__unlikely(lrq->cwd == (void *)1)) free((char *)lrq->alias);
			lrq->status = r;
			RETURN_AST(lrq);
		}
		break;
	}

	ADD_TO_XLIST_ORDERED(&LN->hash[h], &lnte->hash);
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	MULTIOPEN_CLEAR_CACHE();
	UNBLOCK_LNT();
	if (__likely(!(lrq->flags & LNTE_DONT_WAKE_ALL))) {
		if (__likely(!KERNEL$KERNEL)) SWAP_ALL_SUBPROCESSES();
		else DO_BREAK(&KERNEL$PROC_KERNEL, NULL, 1);
	}
	CLEAR_SUBPROC_LN_CACHE();
	if (__unlikely(lrq->cwd == (void *)1)) {
		USR_SWAP_ALL_HANDLES();
		free((char *)lrq->alias);
	}
	lrq->status = 0;
	RETURN_AST(lrq);
}

static int CMP_LOGICAL_ENTRY(__const__ void *v1, __const__ void *v2)
{
	__const__ LOGICAL_LIST_ENTRY *e1 = *(LOGICAL_LIST_ENTRY **)v1;
	__const__ LOGICAL_LIST_ENTRY *e2 = *(LOGICAL_LIST_ENTRY **)v2;
	return strcmp(e1->name, e2->name);
}

void SORT_LOGICAL_LIST(LOGICAL_LIST_REQUEST *r)
{
	int i, j, sub;
	qsort(r->entries, r->n_entries, sizeof(LOGICAL_LIST_ENTRY *), CMP_LOGICAL_ENTRY);
	for (i = r->n_entries - 1; i > 0; i--) {
		if (__unlikely(!strcmp(r->entries[i - 1]->name, r->entries[i]->name))) goto shift;
	}
	shift_done:
	j = 0;
	for (i = 0; i < r->n_entries; i++) {
		/*if (__likely(r->entries[i]->flags != LNTE_INVALID))*/ r->entries[j++] = r->entries[i];
	}
	r->n_entries = j;
	return;
	shift:
	sub = 0;
	j = 1;
	for (i = 1; i < r->n_entries; i++) {
		if (__likely(strcmp(r->entries[j - 1]->name, r->entries[i]->name))) {
			r->entries[j++] = r->entries[i];
		} else {
			if (r->entries[j - 1]->flags == LNTE_PARENT /*&& r->entries[i]->flags != LNTE_PARENT*/) r->entries[j - 1] = r->entries[i];
			sub++;
		}
	}
	r->n_entries -= sub;
	goto shift_done;
}

void KERNEL$FREE_LOGICAL_LIST(LOGICAL_LIST_REQUEST *r)
{
	while (!XLIST_EMPTY(&r->freelist)) {
		LOGICAL_LIST_ENTRY *e = LIST_STRUCT(r->freelist.next, LOGICAL_LIST_ENTRY, list);
		DEL_FROM_LIST(&e->list);
		free(e);
	}
	free(r->entries);
}

struct multiopen {
	OPENRQ *rq;
	FFILE *f;
	char *ne;
	int llr_valid;
	time_t llr_time;
	int idx;
	int error;
	int error_pri;
	LOGICAL_LIST_REQUEST llr;
	OPENRQ o;
	MALLOC_REQUEST mrq;
};

extern struct multiopen *multiopen;

extern AST_STUB MULTIOPEN_GOT_LIST;
extern IO_STUB MULTIOPEN_TRY_OPEN;
extern AST_STUB MULTIOPEN_NAME_ALLOCATED;
extern AST_STUB MULTIOPEN_OPEN;

void MULTIOPEN_CLEAR_CACHE(void)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("MULTIOPEN_CLEAR_CACHE AT SPL %08X", KERNEL$SPL);
	if (__likely(multiopen != NULL)) multiopen->llr_time = -1;
}

int MULTIOPEN_ERROR_PRI(int error)
{
#if defined(__GNUC__) && __GNUC__ >= 3
	__builtin_expect(error, -ENOENT);
#endif
	switch (error) {
		case -ENOLNM:	return 1;
		case -ENOENT:	return 2;
		case -EACCES:
		case -EPERM:
		case -EROFS:	return 3;
		case -EISDIR:
		case -ENOTDIR:	return 4;
		default:	return 5;
	}
}

static void MULTIOPEN_FREE(struct multiopen *mo)
{
	free(mo->f);
	if (__likely(!multiopen)) {
		multiopen = mo;
		return;
	}
	if (__likely(mo->llr_valid) && (__unlikely(!multiopen->llr_valid) || __unlikely((unsigned)multiopen->llr_time < (unsigned)mo->llr_time))) {
		struct multiopen *mo_x = multiopen;
		multiopen = mo;
		mo = mo_x;
	}
	if (__likely(mo->llr_valid)) KERNEL$FREE_LOGICAL_LIST(&mo->llr);
	free(mo);
}

void MULTIOPEN(OPENRQ *RQ, FFILE *f)
{
	struct multiopen *mo;
	if (__unlikely(!multiopen)) {
		if (__unlikely(!(mo = malloc(sizeof(struct multiopen))))) {
			__slow_free(f);
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$OPEN, sizeof(struct multiopen));
			return;
		}
		mo->llr_valid = 0;
	} else {
		mo = multiopen;
		multiopen = NULL;
	}
	mo->rq = RQ;
	mo->f = f;
	mo->ne = strchr(f->path, ':');
	if (__unlikely(!mo->ne) || __unlikely(mo->ne == f->path)) KERNEL$SUICIDE("MULTIOPEN: PATH DOESN'T CONTAIN COLON: \"%s\" -> \"%s\"", RQ->path, f->path);
	mo->ne--;
	if (__unlikely(mo->ne[0] != '.')) {
		RQ->status = -EINVAL;
		MULTIOPEN_FREE(mo);
		CALL_AST(RQ);
		return;
	}
	mo->idx = -1;
	if (__unlikely(!mo->llr_valid)) {
		list:
		mo->llr.prefix = "";	/* !!! TODO: use prefixes ? */
		mo->llr.fn = MULTIOPEN_GOT_LIST;
		CALL_IORQ_CANCELABLE(&mo->llr, KERNEL$LIST_LOGICALS, RQ);
		return;
	}
	if (__unlikely(mo->llr_time != time(NULL))) {
		KERNEL$FREE_LOGICAL_LIST(&mo->llr);
		mo->llr_valid = 0;
		goto list;
	}
	CALL_IORQ(&mo->o, MULTIOPEN_TRY_OPEN);
	return;
}

DECL_AST(MULTIOPEN_GOT_LIST, SPL_DEV, LOGICAL_LIST_REQUEST)
{
	OPENRQ *rq;
	struct multiopen *mo = LIST_STRUCT(RQ, struct multiopen, llr);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, mo->rq);
	if (__unlikely(RQ->status < 0)) {
		rq = mo->rq;
		rq->status = RQ->status;
		MULTIOPEN_FREE(mo);
		RETURN_AST(rq);
	}
	mo->llr_valid = 1;
	time(&mo->llr_time);
	RETURN_IORQ(&mo->o, MULTIOPEN_TRY_OPEN);
}

DECL_IOCALL(MULTIOPEN_TRY_OPEN, SPL_DEV, OPENRQ)
{
	OPENRQ *rq;
	struct multiopen *mo = LIST_STRUCT(RQ, struct multiopen, o);
	if (mo->idx == -1) {
		int i;
		mo->error_pri = MULTIOPEN_ERROR_PRI(mo->error = -ENOLNM);
		for (i = 0; i < mo->llr.n_entries; i++) if (__unlikely(!__memcasexcmp(mo->llr.entries[i]->name, mo->f->path, mo->ne))) {
			mo->idx = i;
			goto try_it;
		}
		err:
		rq = mo->rq;
		rq->status = mo->error;
		MULTIOPEN_FREE(mo);
		RETURN_AST(rq);
	}
	if (__unlikely(mo->idx >= mo->llr.n_entries) || __memcasexcmp(mo->llr.entries[mo->idx]->name, mo->f->path, mo->ne)) {
		goto err;
	}
	try_it:
	mo->mrq.size = strlen(mo->llr.entries[mo->idx]->name) + strlen(mo->ne);
	mo->mrq.fn = MULTIOPEN_NAME_ALLOCATED;
	RETURN_IORQ_CANCELABLE(&mo->mrq, KERNEL$UNIVERSAL_MALLOC, mo->rq);
}

DECL_AST(MULTIOPEN_NAME_ALLOCATED, SPL_DEV, MALLOC_REQUEST)
{
	char *p;
	OPENRQ *rq;
	struct multiopen *mo = LIST_STRUCT(RQ, struct multiopen, mrq);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, mo->rq);
	if (__unlikely(RQ->status < 0)) {
		rq = mo->rq;
		rq->status = RQ->status;
		MULTIOPEN_FREE(mo);
		RETURN_AST(rq);
	}
	p = RQ->ptr;
	stpcpy(stpcpy(p, mo->llr.entries[mo->idx]->name), mo->ne + 1);
	mo->o.flags = mo->rq->flags;
	if (__unlikely(mo->o.flags & _O_NOOPEN)) {
		mo->o.flags = (mo->o.flags & ~_O_NOOPEN) | _O_NOOPEN_CALL;
	}
	mo->o.path = p;
	mo->o.rename_handle = mo->rq->rename_handle;
	mo->o.cwd = NULL;
	mo->o.fn = MULTIOPEN_OPEN;
	RETURN_IORQ_CANCELABLE(&mo->o, KERNEL$OPEN, mo->rq);
}

DECL_AST(MULTIOPEN_OPEN, SPL_DEV, OPENRQ)
{
	OPENRQ *rq;
	struct multiopen *mo = LIST_STRUCT(RQ, struct multiopen, o);
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, mo->rq);
	KERNEL$UNIVERSAL_FREE(mo->mrq.ptr);
	if (RQ->status < 0) {
		int pri;
		if (__unlikely(RQ->status == -EINTR)) {
			got_it:
			rq = mo->rq;
			rq->status = RQ->status;
			MULTIOPEN_FREE(mo);
			RETURN_AST(rq);
		}
		pri = MULTIOPEN_ERROR_PRI(RQ->status);
		if (pri > mo->error_pri) {
			mo->error = RQ->status;
			mo->error_pri = pri;
		}
		mo->idx++;
		RETURN_IORQ_LSTAT(&mo->o, MULTIOPEN_TRY_OPEN);
	}
	goto got_it;
}

int is_exec(__const__ char *orig_p)
{
	char *p = (char *)orig_p;
	int r, i;
	struct link_header lh;
#define lhc ((char *)&lh)
	int h;
	int rcount = 0;
	reopen:
	h = open(p, O_RDONLY | _O_NOPOSIX);
	if (__unlikely(h == -1)) {
		if (p != orig_p) errno = ENOEXEC;
		return 0;
	}
	r = read(h, &lh, sizeof lh);
	if (__likely(r > 2) && __unlikely(lhc[0] == '#') && __unlikely(lhc[1] == '!') && __likely(p == orig_p)) {
		close(h);
		return 1;
	}
	if (__unlikely(r != sizeof lh)) {
		if (__unlikely(r == -1)) {
			close(h);
			return 0;
		}
		if (__unlikely(r < 2) || __unlikely(memchr(lhc, '\n', r) != &lhc[r - 1]) || __unlikely(memchr(lhc, 0, r) != 0)) goto c_enoexec;
		if (__unlikely(++rcount > LINK_MAX_RCOUNT)) {
			close(h);
			errno = ELOOP;
			return 0;
		}
		r--;
		lhc[r] = 0;
		i = _is_absolute(lhc);
		if (__unlikely(i == _ABS_LOGREL) || __unlikely(i == _ABS_CURROOT)) goto c_enoexec;
		if (__unlikely(i == _ABS_TOTAL)) {
			close(h);
			p = lhc;
			goto reopen;
		} else {
			char *f = KERNEL$HANDLE_PATH(h);
			char *ff = strrchr(f, '/');
			int sl;
			if (__unlikely(!ff)) KERNEL$SUICIDE("is_exec: NO SLASH IN \"%s\"", f);
			sl = ff - f;
			p = alloca(sl + r + 2);
			memcpy(p, f, sl + 1);
			memcpy(p + sl + 1, lhc, r + 1);
			close(h);
			goto reopen;
		}
		c_enoexec:
		close(h);
		enoexec:
		errno = ENOEXEC;
		return 0;
	}
	close(h);
	if (__unlikely(LINK_CHECK_HEADER(&lh, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, NULL))) {
		goto enoexec;
	}
	return 1;
#undef lhc
}

int is_cmd(__const__ char *p)
{
	int l;
	if (__unlikely((l = strlen(p)) < 4)) {
		errno = ENOEXEC;
		return 0;
	}
	if (p[l - 4] != '.' || __upcasechr(p[l - 3]) != 'C' || __upcasechr(p[l - 2]) != 'M' || __upcasechr(p[l - 1]) != 'D') {
		errno = ENOEXEC;
		return 0;
	}
	return !open(p, O_RDONLY | _O_NOPOSIX | _O_CLOSE);
}

