#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <KERNEL/LINK.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SLAB.H>
#include <SPAD/THREAD.H>
#include <SPAD/SYNC.H>
#include <KERNEL/FDE.H>
#include <UNISTD.H>
#include <KERNEL/VM_ARCH.H>
#include <ARCH/BSF.H>
#include <KERNEL/FIXUP.H>

#include <SPAD/DL.H>
#include <DLFCN.H>

#define LIB_DIR_1	"LIB.:/"
#define LIB_DIR_2	"MFS:/"

#define MAX_RECURSION	8

#define MAX_HEAP_SIZE	(PAGE_CLUSTER_SIZE / 4)
#define PAGEUP(u)	(((u) + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1))
#define HEAPUP(u)	(2 << __BSR(((u) - 1) | 3))
#define LDUP(u)		(((u) + __MALLOC_ALIGN - 1) & ~(unsigned long)(__MALLOC_ALIGN - 1))

typedef struct {
	IORQ_HEAD;
	struct library_description *ld;
	int do_uninit;
	int force_status;
	IORQ *dlrq;
} UNLOAD_REQ;

struct library_description {
	LIST_ENTRY list;
	void *code;
	unsigned long code_size;
	unsigned code_clusters;
	char type_code;
	void *data;
	unsigned long data_size;
	unsigned data_clusters;
	char type_data;
	char ld_allocated;
	int refcount;
	int type;
	char filename[__MAX_STR_LEN];
	char modname[__MAX_STR_LEN];
	union {
		DLINITRQ uninit;
		UNLOAD_REQ unload;
	} u;
	UNLOAD_REQ *unload_rq;
	unsigned n_dependencies;
	struct library_description *dependencies[1];
};

/* type_code, type_data */
#define NONE	0
#define HEAP	1
#define LONG	2

/* other DLT_ flags are in INCLUDE/SPAD/DL.H */
#define DLT_SHARED	0x10000

static DECL_XLIST(DL_MODULES);

static struct library_description kernel;
static DLRQ kernel_rq;

static int dl_initialized = 0;

void __DL_INIT(void *code, void *code_top, void *data, void *data_top)
{
	/*__debug_printf("DL_INIT: %p, %p, %p, %p\n", code, code_top, data, data_top);*/
	memset(&kernel, 0, sizeof(kernel));
	kernel.code = code;
	kernel.code_size = (char *)code_top - (char *)code;
	kernel.code_clusters = PAGEUP(kernel.code_size) / PAGE_CLUSTER_SIZE;
	kernel.type_code = LONG;
	kernel.data = data;
	kernel.data_size = (char *)data_top - (char *)data;
	kernel.data_clusters = PAGEUP(kernel.data_size) / PAGE_CLUSTER_SIZE;
	kernel.type_data = LONG;
	kernel.ld_allocated = 0;
	kernel.refcount = 1;
	kernel.type = DLT_EXPORT | DLT_SHARED;
	strlcpy(kernel.filename, "KERNEL.DLL", __MAX_STR_LEN);
	strlcpy(kernel.modname, "KERNEL", __MAX_STR_LEN);
	kernel.n_dependencies = 1;
	kernel.dependencies[0] = NULL;
	ADD_TO_XLIST(&DL_MODULES, &kernel.list);
	__barrier();
	dl_initialized = 1;
}

void *__DL_GET_KERNEL_HANDLE(void)
{
	memset(&kernel_rq, 0, sizeof(kernel_rq));
	kernel_rq.ld_handle = &kernel;
	return &kernel_rq;
}

static struct library_description *find_ld_for_address(void *addr)
{
	struct library_description *ld;
	XLIST_FOR_EACH(ld, &DL_MODULES, struct library_description, list) {
		if ((char *)addr >= (char *)ld->code && (char *)addr < (char *)ld->code + ld->code_size) return ld;
		if ((char *)addr >= (char *)ld->data && (char *)addr < (char *)ld->data + ld->data_size) return ld;
	}
	return NULL;
}

static void split_memset(char *p, __u8 val, size_t s, int syn)
{
	do {
		unsigned l = s > 4096 ? 4096 : s;
		memset(p, val, l);
		p += l;
		s -= l;
		if (syn) TEST_LOCKUP_SYNC;
		else if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_KERNEL_IORQS)) return;
	} while (s);
}

static MTX_DECL(DL_MUTEX, "KERNEL$DL_MUTEX");

static void PREPARE_LD_THREAD(THREAD_RQ *ld_thread, DLRQ *dlrq);
static int DL_TEST_FIRST(DLRQ *rq);
static long DL_LOAD_SYNC(void *p_);
static int TEST_CHANGE(int handle, __u8 link_hash_1[LINK_HASH_LENGTH], AIORQ *a);
extern AST_STUB DL_LOAD_MODULE_DONE;
static void FREE_LD(struct library_description *ld);

extern IO_STUB DL_UNLOAD_INTERNAL;
extern AST_STUB DL_UNLOAD_DONE;

DECL_IOCALL(KERNEL$DL_LOAD_MODULE, SPL_DL, DLRQ)
{
	static THREAD_RQ ld_thread;
	RQ->error[0] = 0;
	if (__likely(!RQ->filename[0]) && __unlikely(!RQ->modname[0])) {
		_snprintf(RQ->error, __MAX_STR_LEN, "NO FILE NAME OR MODULE NAME");
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	}
	MTX_LOCK(&DL_MUTEX, RQ, KERNEL$DL_LOAD_MODULE, RETURN);
	if (__likely(DL_TEST_FIRST(RQ))) {
		MTX_UNLOCK(&DL_MUTEX);
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	PREPARE_LD_THREAD(&ld_thread, RQ);
	ld_thread.fn = DL_LOAD_MODULE_DONE;
	RETURN_IORQ_CANCELABLE(&ld_thread, KERNEL$THREAD, RQ);
}

static void PREPARE_LD_THREAD(THREAD_RQ *ld_thread, DLRQ *dlrq)
{
	ld_thread->thread_main = DL_LOAD_SYNC;
	ld_thread->p = dlrq;
	ld_thread->error = dlrq->error;
	ld_thread->cwd = NULL;
	ld_thread->std_in = -1;
	ld_thread->std_out = -1;
	ld_thread->std_err = -1;
	ld_thread->dlrq = NULL;
	ld_thread->spawned = 0;
}

static int DL_TEST_FIRST(DLRQ *rq)
{
	struct library_description *lentry;
	XLIST_FOR_EACH(lentry, &DL_MODULES, struct library_description, list) {
		if ((!rq->filename[0] || !_strcasecmp(lentry->filename, rq->filename)) && (__unlikely(!rq->modname[0]) || !_strcasecmp(lentry->modname, rq->modname)) && lentry->type & DLT_SHARED && !(!(lentry->type & DLT_EXPORT) && rq->type & DLT_EXPORT)) {
			lentry->refcount++;
			rq->ld_handle = lentry;
			return 1;
		}
	}
	if (!rq->filename[0]) {
		strlcpy(rq->filename, rq->modname, __MAX_STR_LEN);
		strlcat(rq->filename, ".DLL", __MAX_STR_LEN);
	}
	return 0;
}

/* In order to save kernel stack space, most these variables are static.
   Variables that need to be valid across last recursive call must not be
   static. Requests also must not be static (because the request dispatcher
   needs to locate thread's stack from request address). */

static long DL_LOAD_SYNC(void *p_)
{
	static int recursion = 0;
	static struct link_header lh;
	static __u8 link_hash_1[LINK_HASH_LENGTH];
	static char file[__MAX_STR_LEN];
	static int handle;
	static unsigned rcount;
	static void *header;
	static unsigned hsize;
	static unsigned n_dependencies;
	static char *export;
	static int sh;
	static unsigned compression_method;
	static unsigned long compressed_file_size;
	static unsigned long code_alloc, rodata_size, data_size, bss_size;
	static unsigned long data_alloc;
	static char type_code, type_data;
	static unsigned long ld_struct_size;
	static unsigned long alloc;
	static char internal_ld;
	static void *code_ptr;
	static void *data_ptr;
	static unsigned code_clusters;
	static unsigned data_clusters;
	static int r1;
	static __f_off l_l;
	static __f_off l_l_2;

	struct library_description *ld;
	DLRQ *dlrq;
	DLRQ *deprq;
	unsigned i;
	int r;
	union {
		OPENRQ o;
		AIORQ a;
		MALLOC_REQUEST m;
		CONTIG_AREA_REQUEST c;
		DLINITRQ init;
		UNLOAD_REQ u;
	} u;

	dlrq = p_;
	rcount = 0;

	RAISE_SPL(SPL_DL);

	if (__unlikely(strpbrk(dlrq->filename, ":/") != NULL)) {
		strlcpy(file, dlrq->filename, __MAX_STR_LEN);
	} else {
		_snprintf(file, __MAX_STR_LEN, LIB_DIR_1 "%s", dlrq->filename);
	}
	do_open:
	u.o.flags = O_RDONLY | _O_NOWAIT;
	u.o.path = file;
	u.o.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.o, KERNEL$OPEN);
	handle = u.o.status;
	if (__unlikely(u.o.status < 0)) {
		if (__likely(file != dlrq->filename) && __likely(!memcmp(file, LIB_DIR_1, strlen(LIB_DIR_1))) && __likely(u.o.status != -EINTR)) {
			_snprintf(file, __MAX_STR_LEN, LIB_DIR_2 "%s", dlrq->filename);
			u.o.flags = O_RDONLY | _O_NOWAIT;
			u.o.path = file;
			u.o.cwd = KERNEL$CWD();
			SYNC_IO_CANCELABLE(&u.o, KERNEL$OPEN);
			if (__likely(u.o.status >= 0)) {
				handle = u.o.status;
				goto open_ok;
			}
		}
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "OPEN: %s (%s)", strerror(-handle), file);
		return handle;
	}
	open_ok:
	restart__changed:
	u.a.h = handle;
	u.a.v.ptr = (unsigned long)link_hash_1;
	u.a.v.len = LINK_HASH_LENGTH;
	u.a.v.vspace = &KERNEL$VIRTUAL;
	u.a.pos = __offsetof(struct link_header, hash);
	u.a.progress = 0;
	SYNC_IO_CANCELABLE(&u.a, KERNEL$AREAD);
	if (__unlikely(u.a.status < 0)) {
		ioerror:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "READ: %s (%s)", strerror(-u.a.status), KERNEL$HANDLE_PATH(handle));
		KERNEL$FAST_CLOSE(handle);
		return u.a.status;
	}
	r1 = u.a.status;
	u.a.h = handle;
	u.a.v.ptr = (unsigned long)&lh;
	u.a.v.len = sizeof lh;
	u.a.v.vspace = &KERNEL$VIRTUAL;
	u.a.pos = 0;
	u.a.progress = 0;
	SYNC_IO_CANCELABLE(&u.a, KERNEL$AREAD);
	if (__unlikely(u.a.status < 0)) goto ioerror;
	if (__unlikely(u.a.status < sizeof(struct link_header))) {
#define lhc	((char *)&lh)
		r = u.a.status;
		if (__unlikely(r < 2) || __unlikely(memchr(lhc, '\n', r) != &lhc[r - 1]) || __unlikely(memchr(lhc, 0, r) != 0)) {
			badf:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NOT AN EXECUTABLE FILE");
			KERNEL$FAST_CLOSE(handle);
			return -ENOEXEC;
		}
		r--;
		lhc[r] = 0;
		i = _is_absolute(lhc);
		if (__unlikely(i == _ABS_LOGREL) || __unlikely(i == _ABS_CURROOT)) goto badf;
		if (__unlikely(++rcount > LINK_MAX_RCOUNT)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "TOO MANY REFERENCES");
			KERNEL$FAST_CLOSE(handle);
			return -ELOOP;
		}
		if (__unlikely(i == _ABS_TOTAL)) {
			KERNEL$FAST_CLOSE(handle);
			strlcpy(file, lhc, __MAX_STR_LEN);
			goto do_open;
		} else {
			char *f = KERNEL$HANDLE_PATH(handle);
			char *ff = strrchr(f, '/');
			if (__unlikely(!ff)) KERNEL$SUICIDE("DL_LOAD_SYNC: NO SLASH IN \"%s\"", f);
			_snprintf(file, __MAX_STR_LEN, "%.*s%s", (int)(ff + 1 - f), f, lhc);
			KERNEL$FAST_CLOSE(handle);
			goto do_open;
		}
#undef lhc
	}
	if (__unlikely(r1 != LINK_HASH_LENGTH)) {
		trunc:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "TRUNCATED FILE");
		KERNEL$FAST_CLOSE(handle);
		return -EFTYPE;
	}
	if (__unlikely(LINK_CHECK_HEADER(&lh, &hsize, &n_dependencies, (void *)&KERNEL$LIST_END, &compression_method, &compressed_file_size, KERNEL$ERROR_MSG()))) {
		KERNEL$FAST_CLOSE(handle);
		return -ENOEXEC;
	}
	u.m.size = hsize;
	SYNC_IO_CANCELABLE(&u.m, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.m.status < 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR HEADER: %s", strerror(-u.m.status));
		KERNEL$FAST_CLOSE(handle);
		return u.m.status;
	}
	header = u.m.ptr;
	u.a.h = handle;
	u.a.v.ptr = (unsigned long)header;
	u.a.v.len = hsize;
	u.a.v.vspace = &KERNEL$VIRTUAL;
	u.a.pos = 0;
	u.a.progress = 0;
	SYNC_IO_CANCELABLE(&u.a, KERNEL$AREAD);
	if (__unlikely(u.a.status != hsize)) {
		short_read:
		KERNEL$UNIVERSAL_FREE(header);
		if (u.a.status < 0) goto ioerror;
		goto trunc;
	}
	if (__unlikely(r = TEST_CHANGE(handle, link_hash_1, &u.a))) {
		if (r < 0) goto short_read;
		KERNEL$UNIVERSAL_FREE(header);
		goto restart__changed;
	}
	if (__unlikely(LINK_GET_MODULE_EXPORT_NAME(header, &export, &sh, KERNEL$ERROR_MSG()))) {
		free_close_ret_eftype:
		KERNEL$UNIVERSAL_FREE(header);
		KERNEL$FAST_CLOSE(handle);
		return -EFTYPE;
	}
	if (dlrq->modname[0] && __unlikely(_strcasecmp(export, dlrq->modname))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "DLL INTERFACE NOT MATCHING (WANTED %s, GOT %s)", dlrq->modname, export);
		KERNEL$UNIVERSAL_FREE(header);
		KERNEL$FAST_CLOSE(handle);
		return -EINVAL;
	}
	if (__unlikely(compression_method)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COMPRESSION NOT SUPPORTED");
		KERNEL$UNIVERSAL_FREE(header);
		KERNEL$FAST_CLOSE(handle);
		return -ENOSYS;
	}
	if (__unlikely(LINK_GET_SECTIONS_SIZE(header, &code_alloc, &rodata_size, &data_size, &bss_size, KERNEL$ERROR_MSG()))) {
		goto free_close_ret_eftype;
	}
	data_alloc = rodata_size + data_size + bss_size;
	KERNEL$UNIVERSAL_FREE(header);

	ld_struct_size = sizeof(struct library_description) + (n_dependencies - !!n_dependencies) * sizeof(struct library_description *);

	if (__unlikely(FOLD_DL)) {
		if (code_alloc + data_alloc <= MAX_HEAP_SIZE && HEAPUP(code_alloc + data_alloc) <= HEAPUP(code_alloc) + HEAPUP(data_alloc)) {
			type_code = HEAP;
			type_data = NONE;
			goto do_alloc;
		}
		if (code_alloc <= MAX_HEAP_SIZE && data_alloc <= MAX_HEAP_SIZE) {
			type_code = HEAP;
			type_data = HEAP;
			goto do_alloc;
		}
		if (code_alloc <= MAX_HEAP_SIZE && PAGEUP(code_alloc + data_alloc) > PAGEUP(data_alloc)) {
			type_code = HEAP;
			type_data = LONG;
			goto do_alloc;
		}
	}
	if (data_alloc <= MAX_HEAP_SIZE && (!FOLD_DL || PAGEUP(code_alloc + data_alloc) > PAGEUP(code_alloc))) {
		type_code = LONG;
		type_data = HEAP;
		goto do_alloc;
	}
	if (__unlikely(FOLD_DL)) {
		if (code_alloc + data_alloc <= (1 << PG_SIZE_BITS << PG_BANK_BITS) && PAGEUP(code_alloc + data_alloc) < PAGEUP(code_alloc) + PAGEUP(data_alloc)) {
			type_code = LONG;
			type_data = NONE;
			goto do_alloc;
		}
	}
	type_code = LONG;
	type_data = LONG;
	do_alloc:

	ld = NULL;
	code_clusters = 0;
	data_clusters = 0;

	alloc = code_alloc;
	if (type_data == NONE) alloc += data_alloc;
	if (type_code == HEAP) {
		internal_ld = 0;
		if (FOLD_DL && (HEAPUP(LDUP(alloc) + ld_struct_size) == HEAPUP(alloc))) {
			alloc = LDUP(alloc) + ld_struct_size;
			internal_ld = 1;
		}
		u.m.size = alloc;
		SYNC_IO_CANCELABLE(&u.m, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(u.m.status < 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR CODE: %s", strerror(-u.m.status));
			KERNEL$FAST_CLOSE(handle);
			return u.m.status;
		}
		if (internal_ld) ld = (struct library_description *)((char *)u.m.ptr + alloc - ld_struct_size);
		code_ptr = u.m.ptr;
	} else if (type_code == LONG) {
		internal_ld = 0;
		if (FOLD_DL && (PAGEUP(LDUP(alloc) + ld_struct_size) == PAGEUP(alloc))) {
			alloc = LDUP(alloc) + ld_struct_size;
			internal_ld = 1;
		}
		u.c.align = 0;
		u.c.flags = CARF_CODE;
		code_clusters = u.c.nclusters = PAGEUP(alloc) / PAGE_CLUSTER_SIZE;
		SYNC_IO_CANCELABLE(&u.c, KERNEL$VM_GRAB_CONTIG_AREA);
		if (__unlikely(u.c.status < 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR CODE: %s", strerror(-u.c.status));
			KERNEL$FAST_CLOSE(handle);
			return u.c.status;
		}
		if (internal_ld) ld = (struct library_description *)((char *)u.c.ptr + alloc - ld_struct_size);
		code_ptr = u.c.ptr;
	} else KERNEL$SUICIDE("DL_LOAD_SYNC: BAD TYPE CODE %d", type_code);

	alloc = data_alloc;
	if (type_data == HEAP) {
		internal_ld = 0;
		if (!ld && (HEAPUP(LDUP(alloc) + ld_struct_size) == HEAPUP(alloc))) {
			alloc = LDUP(alloc) + ld_struct_size;
			internal_ld = 1;
		}
		u.m.size = alloc;
		SYNC_IO_CANCELABLE(&u.m, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(u.m.status < 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR DATA: %s", strerror(-u.m.status));
			if (type_code == HEAP) KERNEL$UNIVERSAL_FREE(code_ptr);
			else KERNEL$VM_RELEASE_CONTIG_AREA(code_ptr, code_clusters);
			KERNEL$FAST_CLOSE(handle);
			return u.m.status;
		}
		if (internal_ld) ld = (struct library_description *)((char *)u.m.ptr + alloc - ld_struct_size);
		data_ptr = u.m.ptr;
	} else if (type_data == LONG) {
		internal_ld = 0;
		if (!ld && (PAGEUP(LDUP(alloc) + ld_struct_size) == PAGEUP(alloc))) {
			alloc = LDUP(alloc) + ld_struct_size;
			internal_ld = 1;
		}
		u.c.align = 0;
		u.c.flags = CARF_DATA;
		data_clusters = u.c.nclusters = PAGEUP(alloc) / PAGE_CLUSTER_SIZE;
		SYNC_IO_CANCELABLE(&u.c, KERNEL$VM_GRAB_CONTIG_AREA);
		if (__unlikely(u.c.status < 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR DATA: %s", strerror(-u.c.status));
			if (type_code == HEAP) KERNEL$UNIVERSAL_FREE(code_ptr);
			else KERNEL$VM_RELEASE_CONTIG_AREA(code_ptr, code_clusters);
			KERNEL$FAST_CLOSE(handle);
			return u.c.status;
		}
		if (internal_ld) ld = (struct library_description *)((char *)u.c.ptr + alloc - ld_struct_size);
		data_ptr = u.c.ptr;
	} else if (type_data == NONE) {
		data_ptr = (char *)code_ptr + code_alloc;
	} else KERNEL$SUICIDE("DL_LOAD_SYNC: BAD TYPE DATA %d", type_data);

	if (!ld) {
		u.m.size = ld_struct_size;
		SYNC_IO_CANCELABLE(&u.m, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(u.m.status < 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR LIBRARY DESCRIPTION: %s", strerror(-u.m.status));
			if (type_code == HEAP) KERNEL$UNIVERSAL_FREE(code_ptr);
			else KERNEL$VM_RELEASE_CONTIG_AREA(code_ptr, code_clusters);
			if (type_data == HEAP) KERNEL$UNIVERSAL_FREE(data_ptr);
			else if (type_data == LONG) KERNEL$VM_RELEASE_CONTIG_AREA(data_ptr, data_clusters);
			KERNEL$FAST_CLOSE(handle);
			return u.m.status;
		}
		ld = u.m.ptr;
		ld->ld_allocated = 1;
	} else {
		ld->ld_allocated = 0;
	}

	VOID_LIST_ENTRY(&ld->list);
	ld->code = code_ptr;
	ld->code_size = code_alloc;
	ld->type_code = type_code;
	ld->code_clusters = code_clusters;
	ld->data = data_ptr;
	ld->data_size = data_alloc;
	ld->type_data = type_data;
	ld->data_clusters = data_clusters;
	ld->refcount = 1;
	ld->type = dlrq->type | (sh ? DLT_SHARED : 0);
	strlcpy(ld->filename, dlrq->filename, __MAX_STR_LEN);
	strlcpy(ld->modname, export, __MAX_STR_LEN);
	ld->n_dependencies = n_dependencies;
	for (i = 0; i < ld->n_dependencies; i++) {
		ld->dependencies[i] = NULL;
	}

	u.a.h = handle;
	u.a.v.ptr = (unsigned long)code_ptr;
	u.a.v.len = code_alloc;
	u.a.v.vspace = &KERNEL$VIRTUAL;
	u.a.pos = 0;
	u.a.progress = 0;
	SYNC_IO_CANCELABLE(&u.a, KERNEL$AREAD);
	if (__unlikely(u.a.status != code_alloc)) {
		code_error:
		if (u.a.status < 0) {
			code_ioerror:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "READ: %s (%s)", strerror(-u.a.status), KERNEL$HANDLE_PATH(handle));
			KERNEL$FAST_CLOSE(handle);
			FREE_LD(ld);
			return u.a.status;
		}
		if (TEST_CHANGE(handle, link_hash_1, &u.a) > 0) {
			free_ld_restart:
			FREE_LD(ld);
			goto restart__changed;
		}
		if (u.a.status < 0) goto code_ioerror;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "FILE TRUNCATED");
		KERNEL$FAST_CLOSE(handle);
		FREE_LD(ld);
		return -EFTYPE;
	}
	u.a.h = handle;
	u.a.v.ptr = (unsigned long)data_ptr;
	u.a.v.len = rodata_size + data_size;
	u.a.v.vspace = &KERNEL$VIRTUAL;
	u.a.pos = code_alloc;
	u.a.progress = 0;
	SYNC_IO_CANCELABLE(&u.a, KERNEL$AREAD);
	if (__unlikely(u.a.status != rodata_size + data_size)) {
		goto code_error;
	}

	split_memset((char *)data_ptr + rodata_size + data_size, 0, bss_size, 1);

	if (__unlikely(r = TEST_CHANGE(handle, link_hash_1, &u.a))) {
		if (r < 0) goto code_error;
		goto free_ld_restart;
	}
	KERNEL$FAST_CLOSE(handle);

	if (__unlikely(LINK_RELOC_MODULE(ld->code, ld->data))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RELOC FAILED");
		FREE_LD(ld);
		return -EFTYPE;
	}
	u.m.size = sizeof(DLRQ);
	SYNC_IO_CANCELABLE(&u.m, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.m.status < 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY FOR DEPENDENCY REQUEST: %s", strerror(-u.m.status));
		FREE_LD(ld);
		return u.m.status;
	}
	deprq = u.m.ptr;
	for (i = 0; i < ld->n_dependencies; i++) {
		static unsigned flags;
		char *depname;
		deprq->type = DLT_EXPORT;
		deprq->filename[0] = 0;
		if (__unlikely(LINK_GET_DEPENDENCY(ld->code, i, &depname, &flags, KERNEL$ERROR_MSG()))) {
			r = -EFTYPE;
			goto free_deps;
		}
		strlcpy(deprq->modname, depname, __MAX_STR_LEN);
		if (__unlikely(flags & LIB_LAZY)) continue;
			/* recursion */
		if (DL_TEST_FIRST(deprq)) {
			r = 0;
		} else {
			if (__unlikely(recursion == MAX_RECURSION - 1)) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "TOO DEEP DEPENDENCE RECURSION");
				r = -ELOOP;
				goto err_name_free_deps;
			}
			recursion++;
			r = DL_LOAD_SYNC(deprq);
			recursion--;
		}
		if (__unlikely(r)) {
			err_name_free_deps:
			if (__likely(strlen(depname) + strlen(KERNEL$ERROR_MSG()) <= __MAX_STR_LEN - 3)) {
				memmove(KERNEL$ERROR_MSG() + strlen(depname) + 2, KERNEL$ERROR_MSG(), strlen(KERNEL$ERROR_MSG()) + 1);
				memcpy(stpcpy(KERNEL$ERROR_MSG(), depname), ": ", 2);
			}
			goto free_deps;
		}
		ld->dependencies[i] = deprq->ld_handle;
		if (__unlikely(LINK_FIXUP_DLL_DEPENDENCY(ld->code, i, ld->dependencies[i]->code, KERNEL$ERROR_MSG()))) {
			r = -EFTYPE;
			goto err_name_free_deps;
		}
	}
	KERNEL$UNIVERSAL_FREE(deprq);

	DEL_FROM_LIST(&ld->list);
	ADD_TO_XLIST(&DL_MODULES, &ld->list);

	if (__unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$HANDLE", SYM_SPECIAL, &l_l))) {
		*(void **)l_l = ld;
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$FIXUP", SYM_SPECIAL, &l_l)) && __unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$FIXUP_END", SYM_SPECIAL, &l_l_2))) {
		FIXUP_FEATURES((struct feature_fixup *)l_l, (struct feature_fixup *)l_l_2);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$LOAD", SYM_SPECIAL, &l_l))) {
		u.init.error = KERNEL$ERROR_MSG();
		SYNC_IO(&u.init, (IO_STUB *)l_l);
		if (__unlikely(u.init.status < 0)) {
			r = u.init.status;
			goto free_deps;
		}
	}
#if 0 /* thread couldn't be created on unload */
	if (__unlikely(!LINK_GET_SYMBOL(ld->code, "_init", SYM_SPECIAL, &l_l))) {
		LOWER_SPL(SPL_ZERO);
		((void (*)(void))l_l)();
		RAISE_SPL(SPL_DL);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$CTORS", SYM_SPECIAL, &l_l)) && __unlikely(!LINK_GET_SYMBOL(ld->code, "DLL$CTORS_END", SYM_SPECIAL, &l_l_2))) {
		LOWER_SPL(SPL_ZERO);
		while (l_l < l_l_2) {
			(*(void (**)(void))l_l)();
			l_l += sizeof(void (*)(void));
		}
		RAISE_SPL(SPL_DL);
	}
#endif
	dlrq->ld_handle = ld;
	return 0;

	free_deps:
	u.u.ld = ld;
	u.u.do_uninit = 0;
	SYNC_IO(&u.u, DL_UNLOAD_INTERNAL);
	return r;
}

static int TEST_CHANGE(int handle, __u8 link_hash_1[LINK_HASH_LENGTH], AIORQ *a)
{
	static __u8 link_hash_2[LINK_HASH_LENGTH];
	a->h = handle;
	a->v.ptr = (unsigned long)link_hash_2;
	a->v.len = LINK_HASH_LENGTH;
	a->v.vspace = &KERNEL$VIRTUAL;
	a->pos = __offsetof(struct link_header, hash);
	a->progress = 0;
	SYNC_IO_CANCELABLE(a, KERNEL$AREAD);
	if (__unlikely(a->status != LINK_HASH_LENGTH)) return -1;
	return !!memcmp(link_hash_1, link_hash_2, LINK_HASH_LENGTH);
}

DECL_AST(DL_LOAD_MODULE_DONE, SPL_DL, THREAD_RQ)
{
	DLRQ *dlrq = RQ->p;
	IO_DISABLE_CHAIN_CANCEL(SPL_DL, dlrq);
	if (__unlikely(!RQ->spawned)) _snprintf(dlrq->error, __MAX_STR_LEN, "CAN'T SPAWN THREAD: %s", strerror(-RQ->status));
	dlrq->status = RQ->status;
	MTX_UNLOCK(&DL_MUTEX);
	RETURN_AST(dlrq);
}

static void FREE_LD(struct library_description *ld)
{
	void *code_ptr, *data_ptr;
	char type_code, type_data;
	unsigned code_clusters, data_clusters;
	DEL_FROM_LIST_ORDERED(&ld->list);
	/* warning, order is important. List may be read in __dlsym from interrupt */
#if __DEBUG >= 1
	split_memset(ld->code, 0xAA, ld->code_size, 0);
	split_memset(ld->data, 0xAA, ld->data_size, 0);
#endif
	code_ptr = ld->code;
	type_code = ld->type_code;
	code_clusters = ld->code_clusters;
	data_ptr = ld->data;
	type_data = ld->type_data;
	data_clusters = ld->data_clusters;
	if (!ld->ld_allocated) ld = NULL;
	if (type_code == HEAP) KERNEL$UNIVERSAL_FREE(code_ptr);
	else KERNEL$VM_RELEASE_CONTIG_AREA(code_ptr, code_clusters);
	if (type_data == HEAP) KERNEL$UNIVERSAL_FREE(data_ptr);
	else if (type_data == LONG) KERNEL$VM_RELEASE_CONTIG_AREA(data_ptr, data_clusters);
	KERNEL$UNIVERSAL_FREE(ld);
}

extern AST_STUB DL_LOAD_LAZY_MODULE_DONE;
static DLLZRQ *lazy_rq;
static struct library_description *lazy_ld;
static unsigned lazy_i;

DECL_IOCALL(KERNEL$DL_LOAD_LAZY, SPL_DL, DLLZRQ)
{
	static DLRQ dlrq;
	static THREAD_RQ ld_thread;
	unsigned i;
	struct library_description *ld;
	if (RQ->caller) {
		ld = find_ld_for_address(RQ->caller);
		if (__unlikely(!ld)) {
			RQ->status = -EINVAL;
			_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T FIND CALLER'S MODULE");
			RETURN_AST(RQ);
		}
	} else {
		ld = RQ->handle->ld_handle;
	}
	RQ->error[0] = 0;

	MTX_LOCK(&DL_MUTEX, RQ, KERNEL$DL_LOAD_LAZY, RETURN);
	lazy_rq = RQ;
	lazy_ld = ld;
	for (i = 0; i < ld->n_dependencies; i++) {
		unsigned flags;
		char *depname;
		if (__unlikely(LINK_GET_DEPENDENCY(ld->code, i, &depname, &flags, RQ->error))) {
			RQ->status = -EFTYPE;
			goto unlock_ret;
		}
		if (__likely(!(flags & LIB_LAZY)) && __unlikely(!ld->dependencies[i])) {
			KERNEL$SUICIDE("KERNEL$DL_LOAD_LAZY: NON-LAZY DEPENDENCY NOT LOADED, MODULE %s (%s), INDEX %u", ld->modname, ld->filename, i);
		}
		if (!_strcasecmp(depname, RQ->modname)) {
			if (__unlikely(ld->dependencies[i] != NULL)) {
				RQ->status = 0;
				goto unlock_ret;
			}
			lazy_i = i;
			dlrq.type = DLT_EXPORT;
			dlrq.filename[0] = 0;
			strlcpy(dlrq.modname, depname, __MAX_STR_LEN);
			PREPARE_LD_THREAD(&ld_thread, &dlrq);
			ld_thread.fn = DL_LOAD_LAZY_MODULE_DONE;
			ld_thread.error = RQ->error;
			if (__likely(DL_TEST_FIRST(&dlrq))) {
				ld_thread.status = 0;
				RETURN_AST(&ld_thread);
			}
			RETURN_IORQ_CANCELABLE(&ld_thread, KERNEL$THREAD, RQ);
		}
	}
	RQ->status = -ENOENT;
	_snprintf(RQ->error, __MAX_STR_LEN, "NO SUCH MODULE");
	unlock_ret:
	MTX_UNLOCK(&DL_MUTEX);
	RETURN_AST(RQ);
}

DECL_AST(DL_LOAD_LAZY_MODULE_DONE, SPL_DL, THREAD_RQ)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_DL, lazy_rq);
	if (__unlikely(!RQ->spawned)) _snprintf(lazy_rq->error, __MAX_STR_LEN, "CAN'T SPAWN THREAD: %s", strerror(-RQ->status));
	if (__likely(!RQ->status)) {
		DLRQ *dlrq = RQ->p;
		if (__unlikely(lazy_ld->dependencies[lazy_i] != NULL))
			KERNEL$SUICIDE("DL_LOAD_LAZY_MODULE_DONE: LAZY DEPENDENCY SOMEHOW CREATED FOR MODULE %s (%s), INDEX %u", lazy_ld->modname, lazy_ld->filename, lazy_i);
		lazy_ld->dependencies[lazy_i] = dlrq->ld_handle;
		if (__unlikely(LINK_FIXUP_DLL_DEPENDENCY(lazy_ld->code, lazy_i, lazy_ld->dependencies[lazy_i]->code, lazy_rq->error))) {
			static UNLOAD_REQ u;
			lazy_ld->dependencies[lazy_i] = NULL;
			u.fn = DL_UNLOAD_DONE;
			u.ld = dlrq->ld_handle;
			u.do_uninit = 1;
			u.force_status = -EFTYPE;
			u.dlrq = (IORQ *)lazy_rq;
			RETURN_IORQ(&u, DL_UNLOAD_INTERNAL);
		}
	}
	lazy_rq->status = RQ->status;
	MTX_UNLOCK(&DL_MUTEX);
	RETURN_AST(lazy_rq);
}

DECL_IOCALL(KERNEL$DL_UNLOAD_MODULE, SPL_DL, DLRQ)
{
	static UNLOAD_REQ u;
	struct library_description *ld = RQ->ld_handle;
	MTX_LOCK(&DL_MUTEX, RQ, KERNEL$DL_UNLOAD_MODULE, RETURN);
	RQ->error[0] = 0;
	u.fn = DL_UNLOAD_DONE;
	u.ld = ld;
	u.do_uninit = 1;
	u.force_status = 0;
	u.dlrq = (IORQ *)RQ;
	RETURN_IORQ(&u, DL_UNLOAD_INTERNAL);
}

DECL_AST(DL_UNLOAD_DONE, SPL_DL, UNLOAD_REQ)
{
	IORQ *dlrq = RQ->dlrq;
	if (RQ->force_status) dlrq->status = RQ->force_status;
	else dlrq->status = RQ->status;
	MTX_UNLOCK(&DL_MUTEX);
	RETURN_AST(dlrq);
}

extern AST_STUB DL_UNLOADING;

DECL_IOCALL(DL_UNLOAD_INTERNAL, SPL_DL, UNLOAD_REQ)
{
	static char error_sink[__MAX_STR_LEN];
	struct library_description *ld = RQ->ld;
	__f_off ul_l;
	if (__unlikely(!ld)) KERNEL$SUICIDE("DL_UNLOAD_INTERNAL: UNLOADING NULL MODULE");
	if (__unlikely(ld->refcount <= 0)) KERNEL$SUICIDE("DL_UNLOAD_INTERNAL: REFCOUNT %d", ld->refcount);
	if (--ld->refcount) {
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	error_sink[0] = 0;
	ld->unload_rq = RQ;
	ld->u.uninit.fn = DL_UNLOADING;
	ld->u.uninit.error = error_sink;
	if (RQ->do_uninit && !LINK_GET_SYMBOL(ld->code, "DLL$UNLOAD", SYM_SPECIAL, &ul_l)) {
		RETURN_IORQ(&ld->u.uninit, (IO_STUB *)ul_l);
	} else {
		ld->u.uninit.status = 0;
		RETURN_AST(&ld->u.uninit);
	}
}

DECL_AST(DL_UNLOADING, SPL_DL, IORQ)
{
	UNLOAD_REQ *urq;
	struct library_description *ld = GET_STRUCT(RQ, struct library_description, u);
	while (ld->n_dependencies--) {
		struct library_description *ldx = ld->dependencies[ld->n_dependencies];
		if (__unlikely(!ldx)) continue;
		ld->u.unload.fn = DL_UNLOADING;
		ld->u.unload.ld = ldx;
		ld->u.unload.do_uninit = 1;
		RETURN_IORQ(&ld->u.unload, DL_UNLOAD_INTERNAL);
	}
	urq = ld->unload_rq;
	FREE_LD(ld);
	urq->status = 0;
	RETURN_AST(urq);
}


int KERNEL$DL_GET_SYMBOL(DLRQ *d, char *name, unsigned long *val)
{
	struct library_description *ld = d->ld_handle;
	__f_off v;
	int r = LINK_GET_SYMBOL(ld->code, name, SYM_EXPORTED, &v);
	*val = v;
	return r;
}

__const__ char *KERNEL$DL_GET_SYMBOL_NAME(__const__ void *sym, unsigned long *offset, int codeonly)
{
	struct library_description *ld;
	unsigned long off;
	if (__unlikely(!offset)) offset = &off;
	if (!dl_initialized) return NULL;
	dl_initialized = 0;
	XLIST_FOR_EACH(ld, &DL_MODULES, struct library_description, list) {
		char *name = LINK_GET_SYMBOL_NAME(ld->code, sym, offset, codeonly);
		if (name) {
			dl_initialized = 1;
			return name;
		}
	}
	*offset = (unsigned long)sym;
	dl_initialized = 1;
	return NULL;
}

static char dl_error[__MAX_STR_LEN];

void *dlopen(__const__ char *filename, int flag)
{
	DLRQ dl;
	dl.type = flag & RTLD_GLOBAL ? DLT_EXPORT : 0;
	strlcpy(dl.filename, filename, __MAX_STR_LEN);
	dl.modname[0] = 0;
	dl.error[0] = 0;
	SYNC_IO_CANCELABLE(&dl, KERNEL$DL_LOAD_MODULE);
	if (__unlikely(dl.status < 0)) {
		strlcpy(dl_error, dl.error, __MAX_STR_LEN);
		if (__unlikely(!dl_error[0])) _snprintf(dl_error, sizeof(dl_error), strerror(-dl.status));
		return NULL;
	}
	return dl.ld_handle;
}

int dlclose(void *handle)
{
	DLRQ dl;
	memset(&dl, 0, sizeof dl);
	dl.ld_handle = handle;
	SYNC_IO_CANCELABLE(&dl, KERNEL$DL_UNLOAD_MODULE);
	return 0;
}

__const__ char *dlerror(void)
{
	if (__likely(dl_error[0])) {
		memmove(dl_error + 1, dl_error, __MAX_STR_LEN - 2);
		dl_error[__MAX_STR_LEN - 1] = 0;
		dl_error[0] = 0;
		return dl_error + 1;
	}
	return NULL;
}

void *__dlsym(void *handle, __const__ char *symbol, void *addr);

void *__dlsym(void *handle, __const__ char *symbol, void *addr)
{
	struct library_description *ld = handle;
	__f_off val;
	int spl;
	if (__likely(SPLX_BELOW(spl = KERNEL$SPL, SPL_X(SPL_DL)))) RAISE_SPL(SPL_DL);
	dl_error[0] = 0;
	if (__unlikely(!ld)) {
		ld = find_ld_for_address(addr);
		if (__unlikely(!ld)) {
			_snprintf(dl_error, __MAX_STR_LEN, "UNKNOWN CALLING ADDRESS %p", addr);
			LOWER_SPLX(spl);
			return NULL;
		}
	} else if (__unlikely(ld == RTLD_DEFAULT)) {
		XLIST_FOR_EACH(ld, &DL_MODULES, struct library_description, list) {
			if (!LINK_GET_SYMBOL(ld->code, symbol, SYM_INTERNAL, &val)) goto found;
		}
		goto not_found;
	}
	if (__unlikely(LINK_GET_SYMBOL(ld->code, symbol, SYM_INTERNAL, &val))) {
		not_found:
		_snprintf(dl_error, __MAX_STR_LEN, "SYMBOL %s NOT FOUND", symbol);
		LOWER_SPLX(spl);
		return NULL;
	}
	found:
	LOWER_SPLX(spl);
	return (void *)val;
}

int dladdr(__const__ void *addr, Dl_info *info)
{
	struct library_description *ld;
	unsigned long offset;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DL), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("dladdr AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DL);
	XLIST_FOR_EACH(ld, &DL_MODULES, struct library_description, list) {
		char *name = LINK_GET_SYMBOL_NAME(ld->code, addr, &offset, 0);
		if (name) {
			info->dli_fname = ld->filename;
			info->dli_fbase = ld->code;
			info->dli_sname = name;
			info->dli_saddr = (char *)addr - offset;
			LOWER_SPLX(spl);
			return 1;
		}
	}
	LOWER_SPLX(spl);
	return 0;
}

struct dwarf_fde *_Unwind_Find_FDE(void *pc, struct dwarf_eh_bases *bases);

struct dwarf_fde *_Unwind_Find_FDE(void *pc, struct dwarf_eh_bases *bases)
{
	struct library_description *ld;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DL), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("_Unwind_Find_FDE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DL);
	XLIST_FOR_EACH(ld, &DL_MODULES, struct library_description, list) {
		if (((char *)pc >= (char *)ld->code && (char *)pc < (char *)ld->code + ld->code_size) || ((char *)pc >= (char *)ld->data && (char *)pc < (char *)ld->data + ld->data_size)) {
			__f_off start, end;
			LOWER_SPLX(spl);
			if (__unlikely(LINK_GET_SYMBOL(ld->code, "DLL$EFS", SYM_SPECIAL, &start))) return NULL;
			if (__unlikely(LINK_GET_SYMBOL(ld->code, "DLL$EFE", SYM_SPECIAL, &end))) return NULL;
			return FIND_FDE(pc, bases, start, end, (__f_off)ld->code, (__f_off)ld->data);
		}
	}
	LOWER_SPLX(spl);
	return NULL;
}

void KERNEL$DL_CALL_DTORS(void)
{
	KERNEL$SUICIDE("KERNEL$DL_CALL_DTORS CALLED");
}
