#include <SPAD/LIST.H>
#include <SYS/TYPES.H>
#include <STRING.H>
#include <SPAD/SLAB.H>
#include <SPAD/AC.H>
#include <SPAD/LIBC.H>
#include <ARCH/BARRIER.H>

#include <STDLIB.H>
#include <SPAD/ENV.H>
#include <KERNEL/ENV.H>

#include <KERNEL/ENVA.I>

struct env_entry {
	LIST_ENTRY list;
	int free;
	int namelen;
	char string[1];
};

static char *find_env(__const__ char *string, int *ppos);

char *getenv(__const__ char *name)
{
	char *f;
	int spl = KERNEL$SPL;
	if (__likely(SPLX_BELOW(spl, SPL_X(SPL_DEV)))) RAISE_SPL(SPL_DEV);
	f = find_env(name, (int *)(void *)&KERNEL$LIST_END);
	LOWER_SPLX(spl);
	return f;
}

static char *find_env(__const__ char *string, int *ppos)
{
	char *e;
	int pos;
	if (__unlikely(!environ)) {
		*ppos = 0;
		return NULL;
	}
	for (pos = 0; (e = environ[pos]); pos++) {
		__const__ char *str = string;
		while (*e && *str == *e) str++, e++;
		if (__likely((unsigned char)((unsigned char)*e - '=') | (unsigned char)*str)) continue;
		*ppos = pos;
		return e + 1;
	}
	*ppos = pos;
	return NULL;
}

static char *alloc_entry(__const__ char *name, int size)
{
	struct env_entry *ee;
	int nl = strlen(name);
	XLIST_FOR_EACH(ee, &environ_list, struct env_entry, list) if (__unlikely(ee->free) && __unlikely(nl == ee->namelen) && __likely(__alloc_size(ee) >= size) && !memcmp(name, ee->string, nl)) {
		ee->free = 0;
		return ee->string;
	}
	ee = malloc(size);
	if (__unlikely(!ee)) return NULL;
	ADD_TO_XLIST(&environ_list, &ee->list);
	ee->free = 0;
	ee->namelen = nl;
	return ee->string;
}

static int env_on_list(__const__ char *str)
{
	struct env_entry *ee;
	XLIST_FOR_EACH(ee, &environ_list, struct env_entry, list) if (__unlikely(ee->string == str)) return 1;
	return 0;
}

static void environ_callback(void)
{
	LIST_ENTRY *c;
	c = environ_callbacks.next;
	while (c != &KERNEL$LIST_END) {
		ENV_CALLBACK *e = LIST_STRUCT(c, ENV_CALLBACK, list);
		c = c->next;
		e->fn(e);
	}
}

int setenv(__const__ char *name, __const__ char *value, int rewrite)
{
	return _setenv(name, value, rewrite, (size_t *)(void *)&KERNEL$LIST_END);
}

int _setenv(__const__ char *name, __const__ char *value, int rewrite, size_t *failsize)
{
	int pos;
	char *new;
	int fr;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("_setenv AT SPL %08X", spl);
#endif
	RAISE_SPL(SPL_DEV);
	if ((new = find_env(name, &pos))) {
		struct env_entry *ee;
		int sv, as;
		if (__unlikely(!rewrite)) {
			goto ls_0;
		}
		sv = strlen(value);
		if (__likely(strlen(new) >= sv)) {
			cpy:
			RAISE_SPL(SPL_TOP);
			strcpy(new, value);
			LOWER_SPL(SPL_DEV);
			goto ls_0_callback;
		}
		as = sizeof(struct env_entry) + strlen(name) + sv + 2;
		if (env_on_list(environ[pos])) {
			ee = LIST_STRUCT(environ[pos], struct env_entry, string);
			if (__alloc_size(ee) >= as) goto cpy;
		} else ee = NULL;
		new = alloc_entry(name, as);
		if (__unlikely(!new)) {
			*failsize = as;
			ls_1:
			LOWER_SPLX(spl);
			return -1;
		}
		if (__likely(ee != NULL)) ee->free = 1;
		copyname:
		stpcpy(stpcpy(stpcpy(new, name), "="), value);
		__barrier();
		environ[pos] = new;
		ls_0_callback:
		environ_callback();
		ls_0:
		LOWER_SPLX(spl);
		return 0;
	}
	if (__unlikely(!(fr = env_freeable(environ))) || __unlikely(__alloc_size(environ) < sizeof(char *) * (pos + 2))) {
		char **newe = malloc(sizeof(char *) * (pos + 2));
		char **olde;
		if (__unlikely(!newe)) {
			*failsize = sizeof(char *) * (pos + 2);
			goto ls_1;
		}
		newe[0] = NULL;
		if (__likely(environ != NULL)) memcpy(newe, environ, sizeof(char *) * (pos + 1));
		olde = environ;
		__barrier();
		environ = newe;
		__barrier();
		if (__likely(fr)) __slow_free(olde);
	}
	new = alloc_entry(name, sizeof(struct env_entry) + strlen(name) + strlen(value) + 2);
	if (__unlikely(!new)) {
		environ[pos] = NULL;
		*failsize = sizeof(struct env_entry) + strlen(name) + strlen(value) + 2;
		goto ls_1;
	}
	environ[pos + 1] = NULL;
	goto copyname;
}

void unsetenv(__const__ char *name)
{
	int pos;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("unsetenv AT SPL %08X", spl);
#endif
	RAISE_SPL(SPL_DEV);
	if (__unlikely(!find_env(name, &pos))) goto ls;
	if (__likely(env_on_list(environ[pos]))) {
		struct env_entry *ee = LIST_STRUCT(environ[pos], struct env_entry, string);
		ee->free = 1;
	}
	while ((environ[pos] = environ[pos + 1])) pos++;
	environ_callback();
	ls:
	LOWER_SPLX(spl);
}

int putenv(__const__ char *name_value)
{
	return _putenv(name_value, (size_t *)(void *)&KERNEL$LIST_END);
}

int _putenv(__const__ char *name_value, size_t *failsize)
{
	char *nv, *e;
	int r;
	if (__unlikely(!(e = strchr(name_value, '=')))) {
		unsetenv(name_value);
		return 0;
	}
	if (__unlikely(!(nv = malloc(strlen(name_value) + 1)))) {
		*failsize = strlen(name_value) + 1;
		return -1;
	}
	strcpy(nv, name_value);
	e = nv + (e - name_value);
	*e = 0;
	r = _setenv(nv, e + 1, 1, failsize);
	free(nv);
	return r;
}

void KERNEL$REGISTER_ENV_CALLBACK(ENV_CALLBACK *c)
{
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("KERNEL$REGISTER_ENV_CALLBACK AT SPL %08X", spl);
#endif
	RAISE_SPL(SPL_DEV);
#if __DEBUG >= 2
	{
		ENV_CALLBACK *t;
		XLIST_FOR_EACH(t, &environ_callbacks, ENV_CALLBACK, list) if (__unlikely(t == c))
			KERNEL$SUICIDE("KERNEL$REGISTER_ENV_CALLBACK: CALLBACK ALREADY REGISTERED");
	}
#endif
	ADD_TO_XLIST(&environ_callbacks, &c->list);
	LOWER_SPLX(spl);
}

void KERNEL$UNREGISTER_ENV_CALLBACK(ENV_CALLBACK *c)
{
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl)))
		KERNEL$SUICIDE("KERNEL$UNREGISTER_ENV_CALLBACK AT SPL %08X", spl);
#endif
	RAISE_SPL(SPL_DEV);
#if __DEBUG >= 2
	{
		ENV_CALLBACK *t;
		XLIST_FOR_EACH(t, &environ_callbacks, ENV_CALLBACK, list) if (t == c) goto got_it;
		KERNEL$SUICIDE("KERNEL$UNREGISTER_ENV_CALLBACK: CALLBACK NOT REGISTERED");
		got_it:;
	}
#endif
	DEL_FROM_LIST(&c->list);
	LOWER_SPLX(spl);
}

void ENV_INIT(void)
{
	INIT_XLIST(&environ_list);
	INIT_XLIST(&environ_callbacks);
}
