#define __FROM_SLAB_C

#include <SPAD/LIST.H>
#include <SYS/TYPES.H>
#include <SPAD/SLAB.H>

#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <KERNEL/VM.H>
#include <ARCH/SETUP.H>
#include <KERNEL/UDATA.H>
#include <KERNEL/VMDEF.H>
#include <KERNEL/VM_ARCH.H>
#include <SYS/MMAN.H>

static DECL_LIST(all_slabs);
static int n_slabs = 0;

static struct __slpage dummy_page = { NULL, NULL, INIT_VAL_LIST(dummy_page.list), 0 };

#define current_page __current_page
#define objsize __objsize
#define objalign __objalign
#define n_pages __n_pages
#define n_free_pages __n_free_pages
#define page_limit __page_limit
#define alloc_type __alloc_type
#define ctor __ctor
#define g __g
#define n_reserved_pages __n_reserved_pages
#define name __name

#define freelist __freelist
#define head __head
#define n_alloc __n_alloc

#define nexte __nexte

static void RELEASE_FREE_SLPAGE(struct __slpage *p, int spl);
static void INIT_SLPAGE(struct __slhead *h, struct __slpage *p);

static __finline__ struct __slentry *FIRST_SLENTRY(void *p, unsigned objalign)
{
	struct __slentry *sl = (struct __slentry *)((char *)p + sizeof(struct __slpage));
	sl = (struct __slentry *)(((unsigned long)sl + objalign) & ~(unsigned long)objalign);
	return sl;
}

static __finline__ struct __slentry *NEXT_SLENTRY(struct __slentry *sl, unsigned objsize, unsigned objalign)
{
	sl = (struct __slentry *)((char *)sl + objsize);
	sl = (struct __slentry *)(((unsigned long)sl + objalign) & ~(unsigned long)objalign);
	return sl;
}

int KERNEL$SLAB_ENTRIES_PER_PAGE(unsigned size, unsigned align)
{
	unsigned space;
	if (__unlikely(!size) || __unlikely(align & (align - 1))) KERNEL$SUICIDE("KERNEL$SLAB_ENTRIES_PER_PAGE: INVALID ARGUMENTS, SIZE %u, ALIGN %u", size, align);
	if (align < __SLAB_SIZE_ALIGN) align = __SLAB_SIZE_ALIGN;
	size = (size + __SLAB_SIZE_ALIGN - 1) & ~(__SLAB_SIZE_ALIGN - 1);
	space = (unsigned)(unsigned long)FIRST_SLENTRY(NULL, align - 1);
	if (__unlikely(space > PAGE_CLUSTER_SIZE)) return 0;
	space = PAGE_CLUSTER_SIZE - space;
	size = (size + align - 1) & ~(align - 1);
	return space / size;
}

void KERNEL$SLAB_INIT(struct __slhead *h, unsigned size, unsigned align, int alloc_type, void (*ctor)(void *g, void *o), void *g, long *limit, char *name)
{
	int spl;
	if (__unlikely(!name) || __unlikely(!*name))
		KERNEL$SUICIDE("KERNEL$SLAB_INIT: SLAB HAS NO NAME, SIZE %u, ALIGN %u", size, align);
	if (__unlikely(KERNEL$SLAB_ENTRIES_PER_PAGE(size, align) <= 0))
		KERNEL$SUICIDE("KERNEL$SLAB_INIT: SLAB %s HAS TOO HIGH SIZE (%u) AND ALIGN (%u)", name, size, align);
	INIT_LIST(&h->pages);
	INIT_LIST(&h->free_pages);
	INIT_LIST(&h->full_pages);
	h->current_page = &dummy_page;
	if (align < __SLAB_SIZE_ALIGN) align = __SLAB_SIZE_ALIGN;
	size = (size + __SLAB_SIZE_ALIGN - 1) & ~(__SLAB_SIZE_ALIGN - 1);
	h->objsize = size;
	h->objalign = align - 1;
	h->n_pages = 0;
	h->n_free_pages = 0;
	h->n_reserved_pages = 0;
	h->page_limit = limit ? limit : (void *)&KERNEL$LIST_END;
	h->alloc_type = alloc_type;
	h->ctor = ctor;
	h->g = g;
	h->name = name;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	ADD_TO_LIST(&all_slabs, &h->global_list);
	n_slabs++;
	LOWER_SPLX(spl);
}

void KERNEL$SLAB_DESTROY(struct __slhead *h)
{
	int spl;
	/*critical_printf("(%p, %p), (%p, %p), %p, %d\n", h->pages.next, h->pages.prev, h->free_pages.next, h->free_pages.prev, h->current_page, h->objsize);*/
	if (__unlikely(!LIST_EMPTY(&h->pages)) || __unlikely(!LIST_EMPTY(&h->full_pages))) KERNEL$SUICIDE("TRYING TO FREE NONEMPTY SLAB %s (%p)", h->name, h);
#if __DEBUG >= 1
	KERNEL$SLAB_CHECK(h, "KERNEL$SLAB_DESTROY");
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	DEL_FROM_LIST(&h->global_list);
	n_slabs--;
	while (!LIST_EMPTY(&h->free_pages)) {
		RELEASE_FREE_SLPAGE(LIST_STRUCT(h->free_pages.next, struct __slpage, list), spl);
		RAISE_SPL(SPL_TOP);
	}
	LOWER_SPLX(spl);
	if (__unlikely(h->n_pages | h->n_free_pages))
		KERNEL$SUICIDE("KERNEL$SLAB_DESTROY: PAGES LEAKED, n_pages %d, n_free_pages %d, SLAB %s", h->n_pages, h->n_free_pages, h->name);
}

static void SLAB_ADD_RESERVED_PAGE(struct __slhead *h, void *page)
{
	int spl;
	INIT_SLPAGE(h, page);
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	(*h->page_limit)--;
	h->n_pages++;
	h->n_free_pages++;
	ADD_TO_LIST(&h->free_pages, &((struct __slpage *)page)->list);
	LOWER_SPLX(spl);
	h->n_reserved_pages++;
}

int KERNEL$SLAB_RESERVE(struct __slhead *h, unsigned n)
{
	unsigned per_page = KERNEL$SLAB_ENTRIES_PER_PAGE(h->objsize, h->objalign + 1);
	unsigned pages = (n + per_page - 1) / per_page;
	while (pages--) {
		void *p;
		if (__unlikely(h->n_pages == MAXINT)) {
			return -ENOMEM;
		}
		while (__unlikely(!(p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED)))) {
			int r;
			if (__unlikely(r = KERNEL$MEMWAIT_SYNC(__PAGE_CLUSTER_SIZE))) {
				return r;
			}
		}
		SLAB_ADD_RESERVED_PAGE(h, p);
	}
	return 0;
}

int KERNEL$SLAB_EMPTY(struct __slhead *h)
{
	return LIST_EMPTY(&h->pages) && LIST_EMPTY(&h->full_pages);
}

/* called at SPL_TOP, lowers to spl */
static void RELEASE_FREE_SLPAGE(struct __slpage *p, int spl)
{
	int type;
	struct __slhead *h;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TOP)))
		KERNEL$SUICIDE("RELEASE_FREE_SLPAGE AT SPL %08X", KERNEL$SPL);
	if (__unlikely(p->n_alloc != 0))
		KERNEL$SUICIDE("RELEASE_FREE_SLPAGE: SLAB %s, PAGE %p, %d ENTRIES ALLOCATED", p->head->name, p, p->n_alloc);
#endif
	h = p->head;
	if (h->n_pages > h->n_reserved_pages) {
		type = VM_TYPE_CACHED_MAPPED;
	} else {
		(*h->page_limit)++;
		type = VM_TYPE_WIRED_MAPPED;
	}
	h->n_pages--;
	h->n_free_pages--;
	DEL_FROM_LIST(&p->list);
	LOWER_SPLX(spl);
	KERNEL$FREE_KERNEL_PAGE(p, type);
}

static void INIT_SLPAGE(struct __slhead *h, struct __slpage *p)
{
	struct __slentry *sl, **lsl;
	p->head = h;
	p->n_alloc = 0;
	sl = FIRST_SLENTRY(p, h->objalign);
	lsl = &p->freelist;
	do {
		if (h->ctor) h->ctor(h->g, sl);
		*lsl = sl;
		lsl = &sl->nexte;
		sl = NEXT_SLENTRY(sl, h->objsize, h->objalign);
	} while ((char *)sl + h->objsize <= ((char *)p + PAGE_CLUSTER_SIZE));
	*lsl = NULL;
}

struct __slentry *KERNEL$__SLAB_ALLOC(struct __slhead *h)
{
	struct __slpage *p;
	struct __slentry *sl;
	int spl;
#if __DEBUG >= 3
	KERNEL$SLAB_CHECK(h, "KERNEL$SLAB_ALLOC");
#endif
	/*if ((sl = p->freelist)) {
		p->freelist = sl->nexte;
		p->n_alloc++;
		return sl;
	}*/
	while (__likely(!LIST_EMPTY(&h->pages))) {
		p = LIST_STRUCT(h->pages.next, struct __slpage, list);
		if (__likely(p->freelist != NULL)) {
			struct __slentry *sl = p->freelist;
			p->n_alloc++;
			p->freelist = sl->nexte;
			h->current_page = p;
			return sl;
		}
		h->current_page = &dummy_page;
		DEL_FROM_LIST(&p->list);
		ADD_TO_LIST(&h->full_pages, &p->list);
	}
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (h->n_free_pages) {
		if (__likely(h->n_free_pages <= h->n_pages - h->n_reserved_pages)) {
			if (__unlikely(h->page_limit != (void *)&KERNEL$LIST_END)) {
				if (__unlikely(--*h->page_limit < 0)) {
					goto enomem_plim_lspl;
				}
			}
			KERNEL_TRANSFER_VM_STATE(VM_TYPE_CACHED_MAPPED, h->alloc_type);
		}
		h->n_free_pages--;
		p = LIST_STRUCT(h->free_pages.next, struct __slpage, list);
		DEL_FROM_LIST(&p->list);
		LOWER_SPLX(spl);
		fl:
		ADD_TO_LIST(&h->pages, &p->list);
		h->current_page = p;
		sl = p->freelist;
		p->freelist = sl->nexte;
		p->n_alloc = 1;
		return sl;
	}
	if (__unlikely(h->page_limit != (void *)&KERNEL$LIST_END)) {
		if (__unlikely(--*h->page_limit < 0)) {
			goto enomem_plim_lspl;
		}
	}
	if (__unlikely(h->n_pages == MAXINT)) {
		goto enomem_plim_lspl;
	}
	h->n_pages++;
	LOWER_SPLX(spl);
	if (__unlikely(!(p = KERNEL$ALLOC_KERNEL_PAGE(h->alloc_type)))) {
		spl = KERNEL$SPL;
		RAISE_SPL(SPL_TOP);
		h->n_pages--;
		enomem_plim_lspl:
		(*h->page_limit)++;
		LOWER_SPLX(spl);
		if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = ENOMEM;
		return NULL;
	}
	INIT_SLPAGE(h, p);
	goto fl;
}

void KERNEL$__SLAB_FREE_PAGE(struct __slpage *p)
{
	int spl;
	struct __slhead *h = p->head;
	DEL_FROM_LIST(&p->list);
	if (__unlikely(h->current_page == p)) {
		h->current_page = LIST_STRUCT(h->pages.next, struct __slpage, list);
		if (__unlikely(LIST_EMPTY(&h->pages)))
			h->current_page = &dummy_page;
	}
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	ADD_TO_LIST(&h->free_pages, &p->list);
	h->n_free_pages++;
	if (__likely(h->n_free_pages <= h->n_pages - h->n_reserved_pages)) {
		KERNEL_TRANSFER_VM_STATE(h->alloc_type, VM_TYPE_CACHED_MAPPED);
		(*h->page_limit)++;
	}
	LOWER_SPLX(spl);
#if __DEBUG >= 3
	KERNEL$SLAB_CHECK(h, "KERNEL$SLAB_FREE_PAGE");
#endif
}

void KERNEL$__SLAB_PARTIAL_PAGE(struct __slpage *p)
{
	DEL_FROM_LIST(&p->list);
	ADD_TO_LIST_END(&p->head->pages, &p->list);
#if __DEBUG >= 3
	KERNEL$SLAB_CHECK(p->head, "KERNEL$SLAB_PARTIAL_PAGE");
#endif
}

int KERNEL$SLAB_REAP(void)
{
	struct __slhead *s;
	int ret = 0;
	int spl = KERNEL$SPL;
	int i;
	for (i = 0; i < n_slabs; i++) {
		a:
		RAISE_SPL(SPL_TOP);
		if (__unlikely(!n_slabs)) goto x;
		s = LIST_STRUCT(all_slabs.prev, struct __slhead, global_list);
		if (s->n_pages > s->n_reserved_pages && s->n_free_pages) {
			RELEASE_FREE_SLPAGE(LIST_STRUCT(s->free_pages.prev, struct __slpage, list), spl);
			/* lowers SPL */
			ret = 1;
			goto a;
		}
		DEL_FROM_LIST(&s->global_list);
		ADD_TO_LIST(&all_slabs, &s->global_list);
		x:
		LOWER_SPLX(spl);
	}
	return ret;
}

__const__ static struct __slhead dummy_bucket =
	{ { NULL, NULL }, { NULL, NULL }, { NULL, NULL}, &dummy_page };

#define db	((struct __slhead *)&dummy_bucket)

struct __slhead *KERNEL$__MALLOC_BUCKETS[__BSR_MAX] = {
	db, db, db, db, db, db, db, db, db, db, db, db, db, db, db, db,
	db, db, db, db, db, db, db, db, db, db, db, db, db, db, db, db,
};

#undef db

static struct __slhead malloc_buckets[__BSR_MAX];

#define bucket_active(b)	((b)->pages.next != NULL)
#define small_heap		(!bucket_active(&malloc_buckets[__BSR_CONST(__MIN_SLAB_SIZE - 1) + 1]))

/*
void dump_heap(int x)
{
	HEAP_FREELIST_T **tp = (void *)HEAP_FREELIST, *t, *nt;
	__debug_printf("heap(%d): ", x);
	while (__likely((t = *tp) != NULL)) __debug_printf("(f: %p, %ld, %p)", t, t->size, t->next), tp = (HEAP_FREELIST_T **)&t->next;
	__debug_printf("done\n");
}
*/

#define HEAP_FREELIST	((UDATA *)UPLACE(UDATA_STRUCT))->heap_freelist

/*#include <SPAD/IOCTL.H>
#include <SPAD/SWAPPER.H>
#include <SPAD/SYNC.H>
#include <UNISTD.H>*/

void *KERNEL$__MALLOC(size_t s, int spl, int synch)
{
	size_t size;
	void *p;
	malloc_again:
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_MALLOC), spl)))
		KERNEL$SUICIDE("malloc AT SPL %08X", spl);
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_MALLOC)))
		KERNEL$SUICIDE("KERNEL$__MALLOC AT SPL %08X", KERNEL$SPL);
#endif
#if __DEBUG >= 3
	KERNEL$HEAP_CHECK("KERNEL$__MALLOC");
#endif
	if (__unlikely(!s)) return KERNEL$__STDHEAP_TOP;
	if (__likely(s < PAGE_CLUSTER_SIZE)) {
		struct __slhead *b;
		if (__unlikely(small_heap)) {
			HEAP_FREELIST_T **tp, *t, *nt;
			size = MALLOC_ALIGN(s);
			tp = (HEAP_FREELIST_T **)&HEAP_FREELIST;
			while (__likely((t = *tp) != NULL)) {
				if (t->size == size) {
					p = (char *)t + sizeof(size_t);
					if (__unlikely(((int)(unsigned long)p & (PAGE_CLUSTER_SIZE - 1)) + size - sizeof(size_t) > PAGE_CLUSTER_SIZE)) {
						split_block:
						nt = (HEAP_FREELIST_T *)((((unsigned long)t + PAGE_CLUSTER_SIZE) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1)) - sizeof(size_t));
						nt->size = t->size - ((char *)nt - (char *)t);
						t->size = (char *)nt - (char *)t;
						nt->next = t->next;
						t->next = nt;
						goto cont;
					}
					*tp = t->next;
					goto ret;
				}
				if (__likely(t->size > size)) {
					p = (char *)t + sizeof(size_t);
					if (__unlikely(((int)(unsigned long)p & (PAGE_CLUSTER_SIZE - 1)) + size - sizeof(size_t) > PAGE_CLUSTER_SIZE)) goto split_block;
					nt = (HEAP_FREELIST_T *)((char *)t + size);
					nt->size = t->size - size;
					nt->next = t->next;
					*tp = nt;
					t->size = size;
					goto ret;
				}
				cont:
				tp = (HEAP_FREELIST_T **)&t->next;
			}
			__SLAB_INIT_MALLOC_ARENA();
		}
		b = KERNEL$__MALLOC_BUCKETS[__BSR((s - 1) | (sizeof(void *) - 1))];
		if (__unlikely(!bucket_active(b))) goto long_alloc;
		p = __slalloc(b);
		if (__unlikely(!p)) goto enomem;
		goto ret;
	} else {
		long_alloc:
		size = (s + PAGE_CLUSTER_SIZE - 1) & ~(size_t)(PAGE_CLUSTER_SIZE - 1);
		if (__unlikely(!size)) {
			erange:
			if (spl == SPL_X(SPL_ZERO)) errno = ERANGE;
			return NULL;
		}
		if (__unlikely(KERNEL$KERNEL)) {
			if (__unlikely(size != PAGE_CLUSTER_SIZE)) {
				goto erange;
			}
			if (__unlikely(!(p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED)))) {
				enomem:
				if (__likely(synch)) {
					LOWER_SPLX(spl);
					if (__unlikely(KERNEL$MEMWAIT_SYNC(s))) return NULL;
					RAISE_SPL(SPL_MALLOC);
					goto malloc_again;
				}
				if (spl == SPL_X(SPL_ZERO) || synch) errno = ENOMEM;
				return NULL;
			}
			return p;
		}
		if (size > PAGE_CLUSTER_SIZE && __unlikely(!USER_SIZE_DESCS)) {
			if (__IS_ERR(p = MAKE_USER_SIZES(size))) {
				ret_err_p:
				if (spl == SPL_X(SPL_ZERO)) errno = -__PTR_ERR(p);
				return NULL;
			}
			goto ok;
		}
		p = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS | _MAP_IMMEDIATE, -1, 0);
		if (__unlikely(!p)) {
			munmap(p, size);
			return NULL;
		}
		if (__unlikely(__IS_ERR(p))) goto ret_err_p;
		ok:
		if (__likely(USER_SIZE_DESCS != NULL)) USER_SIZE_DESCS[(unsigned long)p / PAGE_CLUSTER_SIZE].size = size - PAGE_CLUSTER_SIZE;
		/*if (KERNEL$BPL == SPL_X(SPL_ZERO) && spl == SPL_X(SPL_ZERO) && ((UDATA *)UPLACE(UDATA_STRUCT))->init_thread) {
			LD_PREPARE s;
			IOCTLRQ io;
			size_t l;
			s.from = (unsigned long)p;
			s.to = (unsigned long)p + size;
			io.v.ptr = (unsigned long)&s;
			io.v.len = sizeof s;
			io.v.vspace = &KERNEL$VIRTUAL;
			io.ioctl = IOCTL_SWAPPER_LDCACHE_PREPARE;
			io.param = 0;
			io.h = SWAPPER_HANDLE;
			LOWER_SPL(SPL_ZERO);
			SYNC_IO(&io, KERNEL$IOCTL);
			RAISE_SPL(SPL_MALLOC);
			if (__unlikely(io.status < 0)) {
				__critical_printf("swapper io failed: %s\n", strerror(-io.status));
				_exit(127);
			}
			*(char *)p = 0;
			for (l = 0; l < size; l++) {
				if (__unlikely(((unsigned char *)p)[l])) {
					__critical_printf("non zero @ %ld: %d\n", (unsigned long)l, ((unsigned char *)p)[l]);
					_exit(127);
				}
			}
		}*/
	}
	ret:
	return p;
}

void KERNEL$__FREE(void *s)
{
	size_t size;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_MALLOC)))
		KERNEL$SUICIDE("KERNEL$__FREE AT SPL %08X", KERNEL$SPL);
#endif
#if __DEBUG >= 3
	KERNEL$HEAP_CHECK("KERNEL$__FREE");
#endif
	if (__unlikely(!s) || __unlikely(s == KERNEL$__STDHEAP_TOP)) return;
	if (__likely(__nonstdheap(s))) {
		HEAP_FREELIST_T **tp = (HEAP_FREELIST_T **)&HEAP_FREELIST, *t, *lt = NULL, *st = (HEAP_FREELIST_T *)((char *)s - sizeof(size_t));
		while ((t = *tp) && t < st) lt = t, tp = (HEAP_FREELIST_T **)&t->next;
#if __DEBUG >= 1
		if (__unlikely(t == st)) KERNEL$SUICIDE("KERNEL$__FREE: FREE FREE BLOCK %p(%ld)", st, (long)st->size);
		if (lt && __unlikely((char *)lt + lt->size > (char *)st)) KERNEL$SUICIDE("KERNEL$__FREE: BLOCK %p(%ld) OVERLAPS WITH PREVIOUS FREE BLOCK %p(%ld)", st, (long)st->size, lt, (long)lt->size);
		if (__likely(t != 0) && __unlikely((char *)st + st->size > (char *)t)) KERNEL$SUICIDE("KERNEL$__FREE: BLOCK %p(%ld) OVERLAPS WITH NEXT FREE BLOCK %p(%ld)", st, (long)st->size, t, (long)t->size);
#endif
		if (lt && (char *)lt + lt->size == (char *)st) {
			lt->size += st->size;
			if ((char *)lt + lt->size == (char *)t) {
				lt->size += t->size;
				lt->next = t->next;
			}
			goto ret;
		}
		if ((char *)st + st->size == (char *)t) {
			*tp = st;
			st->size += t->size;
			st->next = t->next;
			goto ret;
		}
		st->next = t;
		*tp = st;
		goto ret;
	}
	size = 0;
	if (__likely(USER_SIZE_DESCS != NULL)) size = USER_SIZE_DESCS[(unsigned long)s / PAGE_CLUSTER_SIZE].size;
	size += PAGE_CLUSTER_SIZE;
	if (__unlikely(KERNEL$KERNEL)) {
		if (__unlikely(size != PAGE_CLUSTER_SIZE))
			KERNEL$SUICIDE("KERNEL$__FREE: FREE OF NON-MALLOCATED BLOCK %p", s);
		KERNEL$FREE_KERNEL_PAGE(s, VM_TYPE_WIRED_MAPPED);
	} else {
		munmap(s, size);
	}
	ret:;
}

void *memalign(size_t align, size_t size)
{
#if __DEBUG >= 3
	KERNEL$HEAP_CHECK("memalign");
#endif
	if (__unlikely(align & (align - 1))) {
		align = 2 << __BSR(align - 1);
		if (__unlikely(!align)) return NULL;
	}
	if (__unlikely(align > PAGE_CLUSTER_SIZE)) {
		void *p, *rp;
		size_t asize;
		if (__unlikely(KERNEL$KERNEL)) {
			if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = ERANGE;
			return NULL;
		}
		size = (size + PAGE_CLUSTER_SIZE - 1) & ~(size_t)(PAGE_CLUSTER_SIZE - 1);
		align = (align + PAGE_CLUSTER_SIZE - 1) & ~(size_t)(PAGE_CLUSTER_SIZE - 1);
		asize = size + align - PAGE_CLUSTER_SIZE;
		if (asize > PAGE_CLUSTER_SIZE && __unlikely(!USER_SIZE_DESCS)) {
			if (__IS_ERR(p = MAKE_USER_SIZES(size))) {
				if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = -__PTR_ERR(p);
			}
			goto ok;
		}
		p = mmap(NULL, asize, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS | _MAP_IMMEDIATE, -1, 0);
		if (__unlikely(!p)) {
			munmap(p, asize);
			return NULL;
		}
		if (__unlikely(__IS_ERR(p))) {
			if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = -__PTR_ERR(p);
			return NULL;
		}
		ok:
		rp = (void *)(((unsigned long)p + align - PAGE_CLUSTER_SIZE) & -align);
		munmap(p, (char *)rp - (char *)p);
		munmap((char *)rp + size, ((char *)p + asize) - ((char *)rp + size));
		if (__likely(USER_SIZE_DESCS != NULL)) USER_SIZE_DESCS[(unsigned long)rp / PAGE_CLUSTER_SIZE].size = size - PAGE_CLUSTER_SIZE;
		return rp;
	}
	if (__unlikely(small_heap)) __SLAB_INIT_MALLOC_ARENA();
	if (__unlikely(align > size)) size = align;
	return malloc(size);
}

void *KERNEL$__REALLOC(void *s, size_t size)
{
	void *t;
	size_t lsize;
#if __DEBUG >= 3
	KERNEL$HEAP_CHECK("KERNEL$__REALLOC");
#endif
	if (__unlikely(!s) || __unlikely(s == KERNEL$__STDHEAP_TOP)) return malloc(size);
	lsize = __alloc_size(s);
	if (lsize >= size) return s;
	if (__likely((t = malloc(size)) != NULL)) memcpy(t, s, lsize > size ? size : lsize), free(s);
	return t;
}

void *KERNEL$__CALLOC(size_t s)
{
	void *p = malloc(s);
	if (__likely(p != NULL) && (__likely(s < PAGE_CLUSTER_SIZE) || __unlikely(KERNEL$KERNEL))) memset(p, 0, s);
	return p;
}

size_t KERNEL$__ALLOC_SIZE(void *s)
{
	if (__unlikely(!s) || __unlikely(s == KERNEL$__STDHEAP_TOP)) return 0;
	if (__likely(__nonstdheap(s))) return *((size_t *)s - 1) - sizeof(size_t);
	if (__unlikely(!USER_SIZE_DESCS)) return PAGE_CLUSTER_SIZE;
	return (size_t)USER_SIZE_DESCS[(unsigned long)s / PAGE_CLUSTER_SIZE].size + PAGE_CLUSTER_SIZE;
}

void KERNEL$__MALLOC_ERROR(void)
{
	if (__likely(KERNEL$SPL == SPL_X(SPL_ZERO))) errno = ENOMEM;
}

void __SLAB_INIT_MALLOC_ARENA(void)
{
	int i;
	struct __slhead *lslab = NULL;		/* warning, go away */
	for (i = __BSR_MAX - 1; i >= 0; i--) {
		if (((2UL << i) >> i) != 2UL) continue;
		if ((2UL << i) < __MIN_SLAB_SIZE) KERNEL$__MALLOC_BUCKETS[i] = lslab;
		else if (KERNEL$SLAB_ENTRIES_PER_PAGE(2UL << i, 2UL << i) >= 2) {
			KERNEL$SLAB_INIT(KERNEL$__MALLOC_BUCKETS[i] = lslab = &malloc_buckets[i], 2UL << i, 2UL << i, VM_TYPE_WIRED_MAPPED, NULL, NULL, NULL,
				!i ? "KERNEL$MALLOC-2" :
				i == 1 ? "KERNEL$MALLOC-4" :
				i == 2 ? "KERNEL$MALLOC-8" :
				i == 3 ? "KERNEL$MALLOC-16" :
				i == 4 ? "KERNEL$MALLOC-32" :
				i == 5 ? "KERNEL$MALLOC-64" :
				i == 6 ? "KERNEL$MALLOC-128" :
				i == 7 ? "KERNEL$MALLOC-256" :
				i == 8 ? "KERNEL$MALLOC-512" :
				i == 9 ? "KERNEL$MALLOC-1024" :
				i == 10 ? "KERNEL$MALLOC-2048" :
				i == 11 ? "KERNEL$MALLOC-4096" :
				i == 12 ? "KERNEL$MALLOC-8192" :
				i == 13 ? "KERNEL$MALLOC-16384" :
				i == 14 ? "KERNEL$MALLOC-32768" :
				i == 15 ? "KERNEL$MALLOC-65536" :
				i == 16 ? "KERNEL$MALLOC-131072" :
				i == 17 ? "KERNEL$MALLOC-262144" :
				i == 18 ? "KERNEL$MALLOC-524288" :
				i == 19 ? "KERNEL$MALLOC-1048576" :
				i == 20 ? "KERNEL$MALLOC-2097152" :
				i == 21 ? "KERNEL$MALLOC-4194304" :
				i == 22 ? "KERNEL$MALLOC-8388608" :
				i == 23 ? "KERNEL$MALLOC-16777216" :
				"KERNEL$MALLOC");
		}
	}
}

void KERNEL$SLABS_DUMP(void)
{
	struct __slhead *s;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	LIST_FOR_EACH_BACK(s, &all_slabs, struct __slhead, global_list) {
		int res = s->n_reserved_pages;
		int tot = s->n_pages;
		int fre = s->n_free_pages;
		if (res + fre > tot) fre = tot - res;
		__critical_printf("%24s (%d): %d / %d / %d\n", s->name, s->alloc_type, res, tot - res - fre, fre);
	}
	LOWER_SPLX(spl);
}

static void CHECK_SLAB_PAGE(struct __slhead *sl, struct __slpage *p, int free, __const__ char *msg)
{
	struct __slentry *e;
	int cnt;
	if (__unlikely(!p) || __unlikely((int)(unsigned long)p & (PAGE_CLUSTER_SIZE - 1)))
		KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): SLAB HAS INVALID PAGE %p ON %s", sl->name, msg, p, free == 1 ? "FREELIST" : free == -1 ? "FULLLIST" : "LIST");
	if (__unlikely(p->head != sl))
		KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): PAGE %p(%d) HAS SLAB POINTER %p BUT IS IN SLAB %p", sl->name, msg, p, free, p->head, sl);
	if (__unlikely(p->list.prev->next != &p->list))
		KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): PAGE %p(%d) HAS CORRUPTED PREVIOUS LIST POINTER", sl->name, msg, p, free);
	if (__unlikely(p->list.next->prev != &p->list))
		KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): PAGE %p(%d) HAS CORRUPTED NEXT LIST POINTER", sl->name, msg, p, free);
	e = p->freelist;
	cnt = 0;
	while (e) {
		if (__unlikely((char *)e <= (char *)p) || __unlikely((char *)e >= (char *)p + PAGE_CLUSTER_SIZE))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): ENTRY %p OUT OF PAGE, PAGE %p(%d)", sl->name, msg, e, p, free);
		if (__unlikely((int)(unsigned long)e & sl->objalign))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): UNALIGNED ENTRY IN FREELIST: %p, PAGE %p(%d)", sl->name, msg, e, p, free);
		e = e->nexte;
		if (__unlikely(cnt++ >= PAGE_CLUSTER_SIZE))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): LOOP IN FREELIST, PAGE %p(%d)", sl->name, msg, p, free);
	}
	if (free == 1) {
		if (__unlikely(p->n_alloc))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): USED PAGE ON FREELIST, PAGE %p, COUNT %d", sl->name, msg, p, p->n_alloc);
	} else if (free == -1) {
		if (__unlikely(p->freelist != NULL))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): PARTIALLY FREE PAGE ON FULLLIST, PAGE %p, COUNT %d", sl->name, msg, p, p->n_alloc);
	} else {
		if (__unlikely(!p->n_alloc))
			KERNEL$SUICIDE("CHECK_SLAB_PAGE(%s, %s): FREE PAGE ON LIST, PAGE %p", sl->name, msg, p);
	}
}

void KERNEL$SLAB_CHECK(struct __slhead *sl, __const__ char *msg)
{
	int spl;
	int x, i;
	struct __slpage *p;
	x = 0;
	LIST_FOR_EACH(p, &sl->pages, struct __slpage, list) {
		CHECK_SLAB_PAGE(sl, p, 0, msg);
		if (p == sl->current_page) x = 1;
	}
	if (!x && __unlikely(sl->current_page != &dummy_page))
		KERNEL$SUICIDE("KERNEL$SLAB_CHECK(%s, %s): SLAB HAS INVALID CURRENT PAGE %p", sl->name, msg, sl->current_page);
	LIST_FOR_EACH(p, &sl->full_pages, struct __slpage, list) {
		CHECK_SLAB_PAGE(sl, p, -1, msg);
	}
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	i = 0;
	LIST_FOR_EACH(p, &sl->free_pages, struct __slpage, list) {
		CHECK_SLAB_PAGE(sl, p, 1, msg);
		i++;
	}
	if (__unlikely(i != sl->n_free_pages))
		KERNEL$SUICIDE("KERNEL$SLAB_CHECK(%s, %s): BAD NUMBER OF FREE PAGES: %d != %d (COUNTED)", sl->name, msg, sl->n_free_pages, i);
	LOWER_SPLX(spl);
}

void KERNEL$HEAP_CHECK(__const__ char *msg)
{
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_MALLOC), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s) AT SPL %08X", msg, KERNEL$SPL);
	RAISE_SPL(SPL_MALLOC);
	if (__likely(!KERNEL$KERNEL)) {
		HEAP_FREELIST_T *t = (HEAP_FREELIST_T *)HEAP_FREELIST;
		HEAP_FREELIST_T *tt;
		if (t) while (1) {
			if (__unlikely(((int)(unsigned long)t + sizeof(size_t)) & (__MALLOC_ALIGN - 1)))
				KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): BAD POINTER IN SMALL HEAP FREELIST: %p", msg, t);
			if (__unlikely((int)t->size & (__MALLOC_ALIGN - 1)) || __unlikely((char *)t + t->size < (char *)t))
				KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): BAD SIZE %ld AT FREE BLOCK %p", msg, (unsigned long)t->size, t);
			if (__unlikely(!t->next)) break;
			if (__unlikely((HEAP_FREELIST_T *)t->next <= t))
				KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): BLOCK %p POINTS BACKWARDS AT %p", msg, t, t->next);
			if (__unlikely((char *)t->next < (char *)t + t->size))
				KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): BLOCK %p POINTS TO ITS OWN FREE SPACE AT %p (SIZE %ld)", msg, t, t->next,(unsigned long)t->size);
			tt = (HEAP_FREELIST_T *)((char *)t + t->size);
			t = t->next;
			while (t != tt) {
				if (__unlikely(tt > t))
					KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): ALLOCATED BLOCK %p OVERLAPS WITH FREE BLOCK %p", msg, tt, t);
				if (__unlikely((int)tt->size & (__MALLOC_ALIGN - 1)) || __unlikely((char *)tt + tt->size < (char *)tt))
					KERNEL$SUICIDE("KERNEL$HEAP_CHECK(%s): BAD SIZE %ld AT ALLOCATED BLOCK %p", msg, (unsigned long)tt->size, tt);
				tt = (HEAP_FREELIST_T *)((char *)tt + tt->size);
			}
		}
	}
	LOWER_SPLX(spl);
	if (__likely(!small_heap)) {
		int i;
		for (i = 0; i < __BSR_MAX; i++) {
			RAISE_SPL(SPL_MALLOC);
			if (bucket_active(KERNEL$__MALLOC_BUCKETS[i]))
				KERNEL$SLAB_CHECK(KERNEL$__MALLOC_BUCKETS[i], msg);
			LOWER_SPLX(spl);
		}
	}
	VM_CHECK_MAGICS();
}

