#include <SPAD/READDIR.H>
#include <DIRENT.H>
#include <SPAD/SYNC.H>
#include <SPAD/CD.H>
#include <STDLIB.H>
#include <UNISTD.H>

struct __dir {
	READDIR_RQ r;
	unsigned long pos;
	int fd;
	char *name;
	char *name_to_free;
};

DIR *opendir(__const__ char *name)
{
	READDIR_RQ r;
	DIR *d = __sync_malloc(sizeof(DIR));
	int spl;
	if (__unlikely(!d)) return NULL;
	spl = KERNEL$SPL;
	if (__likely(SPLX_BELOW(spl, SPL_X(SPL_BOTTOM)))) RAISE_SPL(SPL_BOTTOM);
	d->name = d->name_to_free = _join_paths(KERNEL$GET_CWD(KERNEL$CWD()), name);
	if (__unlikely(!d->name)) {
		LOWER_SPLX(spl);
		goto free_d_ret;
	}
	LOWER_SPLX(spl);
	d->fd = -1;
	r.cwd = NULL;
	r.flags = 0;
	r.path = d->name;
	SYNC_IO(&r, KERNEL$READDIR);
	if (__unlikely(r.status != 0)) {
		errno = -r.status;
		__slow_free(d->name_to_free);
		free_d_ret:
		__slow_free(d);
		return NULL;
	}
	memcpy(&d->r, &r, sizeof(READDIR_RQ));
	d->pos = 0;
	return d;
}

DIR *fdopendir(int fd)
{
	READDIR_RQ r;
	DIR *d = __sync_malloc(sizeof(DIR));
	if (__unlikely(!d)) return NULL;
	d->name_to_free = NULL;
	d->name = KERNEL$HANDLE_PATH(fd);
	if (__unlikely(!d->name)) {
		errno = EBADF;
		goto free_d_ret;
	}
	d->fd = fd;
	r.cwd = NULL;
	r.flags = 0;
	r.path = d->name;
	SYNC_IO(&r, KERNEL$READDIR);
	if (__unlikely(r.status != 0)) {
		errno = -r.status;
		free_d_ret:
		__slow_free(d);
		return NULL;
	}
	memcpy(&d->r, &r, sizeof(READDIR_RQ));
	d->pos = 0;
	return d;
}

struct dirent *readdir(DIR *d)
{
	if (__unlikely(d->pos >= d->r.n_entries)) return NULL;
	return d->r.entries[d->pos++];
}

void rewinddir(DIR *d)
{
	READDIR_RQ r;
	r.cwd = NULL;
	r.flags = 0;
	r.path = d->name;
	SYNC_IO(&r, KERNEL$READDIR);
	KERNEL$FREE_READDIR(&d->r);
	if (__likely(!r.status)) {
		memcpy(&d->r, &r, sizeof(READDIR_RQ));
	}
	d->pos = 0;
}

int closedir(DIR *d)
{
	if (__unlikely(d->fd != -1)) KERNEL$FAST_CLOSE(d->fd);
	KERNEL$FREE_READDIR(&d->r);
	free(d->name_to_free);
	free(d);
	return 0;
}

off_t telldir(DIR *d)
{
	return d->pos;
}

void seekdir(DIR *d, off_t offset)
{
	d->pos = offset;
}

int dirfd(__const__ DIR *d_)
{
	DIR *d = (DIR *)d_;	/* lame but makes prototype ok */
	if (d->fd == -1) {
		int fd = open(d->name, O_RDONLY | _O_NOOPEN);
		if (__unlikely(fd < 0)) return fd;
			/* !!! SMPFIX: lock */
		if (__unlikely(__CMPXCHGI(&d->fd, -1, fd))) KERNEL$FAST_CLOSE(fd);
	}
	return d->fd;
}
