#include <SPAD/LIBC.H>
#include <SPAD/TIMER.H>
#include <ARCH/BARRIER.H>
#include <KERNEL/TIME.H>
#include <KERNEL/VMDEF.H>
#include <VALUES.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/SYNC.H>

#include <UNISTD.H>
#include <TIME.H>
#include <SYS/TIME.H>
#include <SYS/TIMES.H>
#include <SYS/TIMEB.H>
#include <SIGNAL.H>

int subtract_time(struct timeval *t1, struct timeval *t2);

int subtract_time(struct timeval *t1, struct timeval *t2)
{
	/*__debug_printf("subtract: %Ld.%06d - %Ld.%06d\n", t1->tv_sec, t1->tv_usec, t2->tv_sec, t2->tv_usec);*/
	if ((int)(t1->tv_usec -= t2->tv_usec) < 0) {
		t1->tv_usec += 1000000;
		if (!t1->tv_sec) goto z;
		t1->tv_sec--;
	}
	if (t1->tv_sec < t2->tv_sec) {
		t1->tv_sec = 0;
		z:
		t1->tv_usec = 0;
		return -1;
	}
	t1->tv_sec -= t2->tv_sec;
	if (__unlikely(!t1->tv_usec) && __unlikely(!t1->tv_sec)) return 0;
	/*__debug_printf("result: %Ld.%06d\n", t1->tv_sec, t1->tv_usec);*/
	return 1;
}

time_t time(time_t *p)
{
	time_t s;
	s = GET_TIME_SEC();
	if (__unlikely(p != NULL)) *p = s;
	return s;
}

time_t __time32_2_time_t(__u32 time32)
{
	time_t t = time32 | (GET_TIME_SEC() & ((time_t)0xFFFFFFFF << 31 << 1));
	if (__likely(time32 < 0x80000000)) {
		if (__unlikely((__u32)t >= time32 + 0x80000000)) t -= (time_t)1 << 31 << 1;
	} else {
		if (__unlikely((__u32)t < time32 - 0x80000000)) t -= (time_t)1 << 31 << 1;
	}
	return t;
}

int stime(__const__ time_t *t)
{
	struct timeval tv;
	if (__unlikely(!t)) {
		errno = EINVAL;
		return -1;
	}
	tv.tv_sec = *t;
	tv.tv_usec = 0;
	return settimeofday(&tv, NULL);
}

int gettimeofday(struct timeval *tv, struct timezone *tz)
{
	if (__unlikely(tz != NULL)) {
		tz->tz_minuteswest = TIME_MINUTESWEST;
		tz->tz_dsttime = TIME_DST;
	}
	if (__likely(tv != NULL)) {
		return GET_TIME_OF_DAY(tv);
	}
	return 0;
}

int settimeofday(__const__ struct timeval *tv, __const__ struct timezone *tz)
{
	int spl;
	if (__unlikely(__tv_invalid(tv))) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(!KERNEL$KERNEL)) {
		errno = EPERM;
		return -1;
	}
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__likely(tv != NULL)) TIME_SEC = tv->tv_sec;
	if (__unlikely(tz != NULL)) TIME_MINUTESWEST = tz->tz_minuteswest, TIME_DST = tz->tz_dsttime;
	LOWER_SPLX(spl);
	return 0;
}

clock_t clock(void)	/* !!! TODO: should return process time only :-/ */
{
	static clock_t first_time = -1;
	int r;
	clock_t clk;
	struct timeval tv;
	if (__unlikely(r = gettimeofday(&tv, NULL))) return r;
	clk = (__u64)tv.tv_sec * CLK_TCK + (unsigned)tv.tv_usec / (1000000 / CLK_TCK);
	if (__unlikely(first_time == -1))
		first_time = clk;
	return clk - first_time;
}

clock_t times(struct tms *t)
{
	int r;
	struct timeval tv;
	if (__unlikely(r = gettimeofday(&tv, NULL))) return r;
	if (__likely(t != NULL)) {
			/* !!! TODO: really measure it */
		t->tms_utime = clock();
		t->tms_cutime = 0;
		t->tms_stime = t->tms_cstime = 0;
	}
	return (__u64)tv.tv_sec * CLK_TCK + (unsigned)tv.tv_usec / (1000000 / CLK_TCK);
}

int ftime(struct timeb *t)
{
	int r;
	struct timeval tv;
	struct timezone tz;
	if (__unlikely(!t)) return 0;
	if (__unlikely(r = gettimeofday(&tv, &tz))) return r;
	t->time = tv.tv_sec;
	t->millitm = (unsigned)tv.tv_usec / 1000;
	t->timezone = tz.tz_minuteswest;
	t->dstflag = tz.tz_dsttime;
	return 0;
}

unsigned sleep(unsigned seconds)
{
	time_t s, ss;
	if (__unlikely(!seconds)) {
		KERNEL$THREAD_YIELD();
		return 0;
	}
	while (__unlikely(seconds > MAXUINT / JIFFIES_PER_SECOND)) {
		unsigned left;
		if (__likely(seconds == -1)) {
			while (!sleep(MAXUINT / JIFFIES_PER_SECOND)) ;
			return -1;
		}
		seconds -= MAXUINT / JIFFIES_PER_SECOND;
		if ((left = sleep(MAXUINT / JIFFIES_PER_SECOND))) return left + seconds;
	}
	s = GET_TIME_SEC();
	__cancel_start();
	if (__likely(!KERNEL$SLEEP_CANCELABLE(seconds * JIFFIES_PER_SECOND))) {
		__cancel_end();
		return 0;
	}
	__cancel_end();
	ss = GET_TIME_SEC();
	ss -= s;
	if (__unlikely(ss >= seconds)) return 1;
	return seconds - ss;
}

int nanosleep(__const__ struct timespec *req, struct timespec *rem)
{
	int r;
	struct timeval tv, tv2;
	if (__unlikely(__ts_invalid(req))) {
		errno = EINVAL;
		return -1;
	}
	if (rem && __unlikely(r = gettimeofday(&tv2, NULL))) return r;
	__cancel_start();
	if (__likely(!KERNEL$SLEEP_CANCELABLE((__u64)req->tv_sec * JIFFIES_PER_SECOND + (unsigned)req->tv_nsec / (1000000000 / JIFFIES_PER_SECOND)))) {
		if (rem) rem->tv_sec = 0, rem->tv_nsec = 0;
		__cancel_end();
		return 0;
	} else {
		__cancel_end();
		if (rem) {
			if (__unlikely(r = gettimeofday(&tv, NULL))) return r;
			subtract_time(&tv, &tv2);
			memcpy(rem, req, sizeof(struct timespec));
			rem->tv_nsec = (unsigned)rem->tv_nsec / 1000;
			subtract_time((struct timeval *)(void *)rem, &tv);
			rem->tv_nsec = (unsigned)rem->tv_nsec * 1000;
		}
		errno = EINTR;
		return -1;
	}
}

int usleep(unsigned long usec)
{
	u_jiffies_lo_t j;
	if (__unlikely(!usec)) return 0;
	USEC_2_JIFFIES_LO(usec, j);
	if (__likely(!KERNEL$SLEEP_CANCELABLE(j))) return 0;
	errno = EINTR;
	return -1;
}

size_t getpagesize(void)
{
	return PAGE_CLUSTER_SIZE;
}

pid_t getpid(void)
{
	return 3;
}

pid_t getppid(void)
{
	return 2;
}

uid_t getuid(void)
{
	return 1;
}

int setuid(uid_t u)
{
	if (u == 1) return 0;
	errno = EPERM;
	return -1;
}

gid_t getgid(void)
{
	return 1;
}

int setgid(gid_t u)
{
	if (u == 1) return 0;
	errno = EPERM;
	return -1;
}

uid_t geteuid(void)
{
	return 1;
}

int seteuid(uid_t u)
{
	if (u == 1) return 0;
	errno = EPERM;
	return -1;
}

gid_t getegid(void)
{
	return 1;
}

int setegid(gid_t u)
{
	if (u == 1) return 0;
	errno = EPERM;
	return -1;
}

int setreuid(uid_t r, uid_t e)
{
	if (r == 1 && e == 1) return 0;
	errno = EPERM;
	return -1;
}

int setregid(gid_t r, gid_t e)
{
	if (r == 1 && e == 1) return 0;
	errno = EPERM;
	return -1;
}

int getgroups(int size, gid_t *list)
{
	if (__likely(size)) {
		list[0] = 1;
		return 1;
	}
	return 0;
}

int setgroups(size_t size, const gid_t *list)
{
	errno = EPERM;
	return -1;
}

int setpgid(pid_t pid, pid_t pgid)
{
	return 0;
}

pid_t getpgid(pid_t pid)
{
	return getpid();
}

int setpgrp(pid_t __pid, pid_t __pgid)
{
	return setpgid(__pid, __pgid);
}

pid_t getpgrp(void)
{
	return getpgid(0);
}

pid_t setsid(void)
{
	return getpid();
}

int issetugid(void)
{
	return 0;
}

int pause(void)
{
	sigset_t m;
	sigemptyset(&m);
	sigsuspend(&m);
	errno = EINTR;
	return -1;
}

int sigblock(int mask)
{
	sigset_t m = mask;
	sigset_t old;
	sigemptyset(&old);
	sigprocmask(SIG_BLOCK, &m, &old);
	return old;
}

int siggetmask(void)
{
	sigset_t o;
	sigprocmask(SIG_BLOCK, NULL, &o);
	return o;
}

int sigsetmask(int mask)
{
	sigset_t m = mask;
	sigset_t old;
	sigemptyset(&old);
	sigprocmask(SIG_SETMASK, &m, &old);
	return old;
}

int sigpause(int mask)
{
	sigset_t m = mask;
	return sigsuspend(&m);
}
