#include <ERRNO.H>
#include <STDARG.H>
#include <STRING.H>
#include <TERMIOS.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SIGNAL.H>
#include <LIB/KERNEL/USIGNAL.H>
#include <SYS/STAT.H>
#include <UNISTD.H>
#include <VALUES.H>
#include <SYS/TYPES.H>
#include <ARCH/BSF.H>
#include <SPAD/TTY.H>

#include <SYS/IOCTL.H>
#include <SYS/SOUNDCARD.H>
#include <SYS/KD.H>
#include <LINUX/FD.H>

static int get_mode_data(int h, unsigned ioc, unsigned long param, void *data, unsigned long datalen, int tty);

static int get_mode(int h, unsigned ioc, unsigned long param, int tty)
{
	return get_mode_data(h, ioc, param, NULL, 0, tty);
}

static int get_mode_data(int h, unsigned ioc, unsigned long param, void *data, unsigned long datalen, int tty)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = ioc;
	io.param = param;
	io.v.vspace = &KERNEL$VIRTUAL;
	io.v.ptr = (unsigned long)data;
	io.v.len = datalen;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) io.status = tty ? -ENOTTY : -EINVAL;
		if (io.status == -ENOLNM && tty && (unsigned)io.h <= 2) {
			raise_sighup();
			io.status = -EIO;
		}
		errno = -io.status;
		return -1;
	}
	return io.status;
}

static int set_mode_int(int h, char *str, int val)
{
	char valstr[12];
	CHHRQ ch;
	if (__unlikely(_snprintf(valstr, sizeof(valstr), "%d", val) < 0)) {
		errno = ERANGE;
		return -1;
	}
	ch.h = h;
	ch.option = str;
	ch.value = valstr;
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	if (__unlikely(ch.status < 0)) {
		if (ch.status == -EBADMOD || ch.status == -ENOENT || ch.status == -ENOOP) ch.status = -ENOTTY;
		if (ch.status == -ENOLNM && (unsigned)ch.h <= 2) {
			raise_sighup();
			ch.status = -EIO;
		}
		errno = -ch.status;
		return -1;
	}
	return 0;
}

static int kd_get_mode(int h);

unsigned char intr_key = 3;
unsigned char quit_key = 28;
/*unsigned char stop_key = 26;*/

static unsigned char set_intr_key = 3;
static unsigned char set_quit_key = 28;
static unsigned char set_stop_key = 26;

static int get_tty(int h, struct termios *t)
{
	int r;
	if (__unlikely((r = get_mode(h, IOCTL_TTY_GET_TERM_FLAGS, 0, 1)) < 0)) return r;
	memset(t, 0, sizeof(struct termios));
	t->c_iflag = IGNBRK | (r & TTYF_RAW ? 0 : ICRNL);
	t->c_oflag = r & TTYF_O_NL_CRNL ? OPOST | ONLCR : 0;
	t->c_cflag = CREAD | CS8;
	t->c_lflag = (r & TTYF_RAW ? 0 : ICANON) | (r & TTYF_NOECHO ? 0 : ECHOCTL | ECHO | ECHOK | ECHOE | ECHOKE) | ISIG;
	t->c_cc[VEOF] = 4;
	t->c_cc[VERASE] = 127;
	t->c_cc[VKILL] = 21;
	t->c_cc[VERASE2] = 8;
	t->c_cc[VLNEXT] = 22;
	t->c_cc[VMIN] = 1;
	t->c_ispeed = 38400;
	t->c_ospeed = 38400;
	t->c_cc[VINTR] = set_intr_key;
	t->c_cc[VQUIT] = set_quit_key;
	t->c_cc[VSUSP] = set_stop_key;
	if (__unlikely(set_intr_key != intr_key) || __unlikely(set_quit_key != quit_key)) t->c_lflag &= ~ISIG;
	return 0;
}

static int set_tty(int h, struct termios *t)
{
	int r;
	unsigned qk;
	int kd = kd_get_mode(h);
	int ignore_signal = kd >= 0 && kd != K_XLATE && kd != K_UNICODE;
	set_intr_key = t->c_cc[VINTR];
	qk = !ignore_signal && t->c_lflag & ISIG ? set_intr_key : 0;
	if (__unlikely(intr_key != qk)) {
		INSTALL_INTR_KEY(0, intr_key, 0);
		INSTALL_INTR_KEY(0, intr_key = qk, SIGINT);
	}
	set_quit_key = t->c_cc[VQUIT];
	qk = !ignore_signal && t->c_lflag & ISIG ? set_quit_key : 0;
	if (__unlikely(quit_key != qk)) {
		INSTALL_INTR_KEY(0, quit_key, 0);
		INSTALL_INTR_KEY(0, quit_key = qk, SIGQUIT);
	}
	set_stop_key = t->c_cc[VSUSP];
/*
	qk = !ignore_signal && t->c_lflag & ISIG ? set_stop_key : 0;
	if (__unlikely(stop_key != qk)) {
		INSTALL_INTR_KEY(0, stop_key, 0);
		INSTALL_INTR_KEY(0, stop_key = qk, SIGTSTP);
	}
*/
	if (__unlikely(r = set_mode_int(h, "ECHO", ((unsigned)t->c_lflag / ECHO) & 1))) return r;
	if (__unlikely(r = set_mode_int(h, "ICANON", ((unsigned)t->c_lflag / ICANON) & 1))) return r;
	if (__unlikely(r = set_mode_int(h, "ONLCR", (t->c_oflag & (OPOST | ONLCR)) == (OPOST | ONLCR)))) return r;
	read(h, NULL, 0);
	return 0;
}

static int set_tty_x(int h, struct termios *t)
{
	int er, r;
	r = set_tty(h, t);
	if (__unlikely((unsigned)h >= 3) || __unlikely(r)) return r;
	er = errno;
	if (h != 0) set_tty(0, t);
	if (h != 1) set_tty(1, t);
	if (h != 2) set_tty(2, t);
	if (!r) errno = er;
	return 0;
}

static __finline__ int drain_tty(int h)
{
	return get_mode(h, IOCTL_FSYNC, 0, 1);
}

static __finline__ int flush_tty(int h, int type)
{
	return get_mode(h, IOCTL_TTY_FLUSH, type, 1);
}

static int tty_get_winsize(int h, struct winsize *w)
{
	int r;
	struct tty_window_size ws;
	if (__unlikely((r = get_mode_data(h, IOCTL_TTY_GET_WINSIZE, 0, &ws, sizeof ws, 1)) < 0)) return r;
	w->ws_col = ws.ws_col;
	w->ws_row = ws.ws_row;
	w->ws_xpixel = ws.ws_xpixel;
	w->ws_ypixel = ws.ws_ypixel;
	return 0;
}

static int tty_set_winsize(int h, struct winsize *w)
{
	char valstr[48];
	CHHRQ ch;
	if (__unlikely(_snprintf(valstr, sizeof(valstr), "%d,%d,%d,%d", w->ws_col, w->ws_row, w->ws_xpixel, w->ws_ypixel) < 0)) {
		errno = ERANGE;
		return -1;
	}
	ch.h = h;
	ch.option = "WINSIZE";
	ch.value = valstr;
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	if (__unlikely(ch.status < 0)) {
		if (ch.status == -EBADMOD || ch.status == -ENOENT || ch.status == -ENOOP) ch.status = -ENOTTY;
		if (ch.status == -ENOLNM && (unsigned)ch.h <= 2) {
			raise_sighup();
			ch.status = -EIO;
		}
		errno = -ch.status;
		return -1;
	}
	return 0;
}

static int nread(int h, int *n)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_NREAD;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status == -ENOOP)) {
		struct stat st;
		off_t s;
		if (fstat(h, &st)) return -1;
		s = lseek(h, 0, SEEK_CUR);
		if (__unlikely(s == (off_t)-1)) return -1;
		if (__unlikely(s > st.st_size)) *n = 0;
		else if (__unlikely(st.st_size - s > MAXINT)) *n = MAXINT;
		else *n = st.st_size - s;
		return 0;
	}
	if (__unlikely(io.status < 0)) {
		errno = -io.status;
		return -1;
	}
	if (__unlikely(io.status >= MAXINT)) *n = MAXINT;
	else *n = io.status;
	return 0;
}

static __finline__ int snd_sync(int h, int mode)
{
	return get_mode(h, IOCTL_SND_SYNC, mode, 0);
}

static int snd_speed(int h, int *speed)
{
	int r;
	if (__likely(*speed > 0)) {
		if (__unlikely(r = set_mode_int(h, "RATE", *speed))) return r;
	}
	if (__unlikely((r = get_mode(h, IOCTL_SND_GETRATE, 0, 0)) < 0)) return r;
	*speed = r;
	return 0;
}

static int snd_channels(int h, int *chan)
{
	int r;
	if (__likely(*chan > 0)) {
		if (__unlikely(r = set_mode_int(h, "CHANNELS", *chan))) return r;
	}
	if (__unlikely((r = get_mode(h, IOCTL_SND_GETCHANNELS, 0, 0)) < 0)) return r;
	*chan = r;
	return 0;
}

static int snd_fmt(int h, int *fmt)
{
	int r;
	if (__likely(*fmt > AFMT_QUERY)) {
		if (__unlikely(*fmt & (*fmt - 1))) goto skip_set;
		if (__unlikely(r = set_mode_int(h, "FORMAT", __BSR(*fmt)))) return r;
	}
	skip_set:
	if (__unlikely((r = get_mode(h, IOCTL_SND_GETFORMAT, 0, 0)) < 0)) return r;
	*fmt = 1 << r;
	return 0;
}

static __finline__ int snd_subdivide(int h, int *div)
{
	return set_mode_int(h, "SUVDIVIDE", *div);
}

static int snd_setfragment(int h, int *frg)
{
	int r;
	if (__unlikely(r = set_mode_int(h, "FRAGSIZE", *frg & 0xffff))) return r;
	if (__unlikely(r = set_mode_int(h, "FRAGS", (*frg >> 16) & 0x7fff))) return r;
	return 0;
}

static __finline__ int snd_getfmts(int h)
{
	return get_mode(h, IOCTL_SND_GETFORMATMASK, 0, 0);
}

static __finline__ int snd_getspace(int h, audio_buf_info *info, int rec)
{
	return get_mode_data(h, IOCTL_SND_GETSPACE, rec, info, sizeof(audio_buf_info), 0);
}

static __finline__ int snd_getcaps(int h)
{
	return get_mode(h, IOCTL_SND_GETCAPS, 0, 0);
}

static __finline__ int snd_gettrigger(int h)
{
	return get_mode(h, IOCTL_SND_GETTRIGGER, 0, 0);
}

static __finline__ int snd_settrigger(int h, int trig)
{
	return get_mode(h, IOCTL_SND_SETTRIGGER, trig, 0);
}

static __finline__ int snd_getptr(int h, count_info *ci, int rec)
{
	return get_mode_data(h, IOCTL_SND_GETPTR, rec, ci, sizeof(count_info), 0);
}

static __finline__ int snd_getodelay(int h)
{
	return get_mode(h, IOCTL_SND_GETODELAY, 0, 0);
}

static int snd_geterror(int h, audio_errinfo *ae)
{
	return get_mode_data(h, IOCTL_SND_GETERROR, 0, ae, sizeof(audio_errinfo), 0);
}

static int snd_mixread(int h, int mix)
{
	if (__unlikely(mix > PARAM_MIX_IDX_MASK)) {
		errno = EINVAL;
		return -1;
	}
	return get_mode(h, IOCTL_SND_MIXREAD, mix, 0);
}

static int snd_mixwrite(int h, int mix, int *val)
{
	int r;
	if (__unlikely(mix > PARAM_MIX_IDX_MASK) ||
	    __unlikely(*val > (MAXINT >> PARAM_MIX_DATA_SHIFT))) {
		errno = EINVAL;
		return -1;
	}
	r = get_mode(h, IOCTL_SND_MIXWRITE, (*val << PARAM_MIX_DATA_SHIFT) | mix, 0);
	if (__unlikely(r < 0)) return r;
	*val = r;
	return 0;
}

static int kd_get_mode(int h)
{
	return get_mode(h, IOCTL_TTY_GETKBDMODE, 0, 1);
}

static int kd_set_mode(int h, int mode)
{
	struct termios t;
	int r = set_mode_int(h, "KBDRAW", mode);
	if (__unlikely(r)) return r;
	r = get_tty(h, &t);
	if (__unlikely(r)) {
		read_ret:
		read(h, NULL, 0);
		return r;
	}
	r = set_tty(h, &t);
	if (__unlikely(r)) goto read_ret;
	return 0;
}

static __finline__ int kd_get_leds(int h)
{
	return get_mode(h, IOCTL_TTY_GETLED, 0, 1);
}

static int kd_set_leds(int h, int leds)
{
	int r = set_mode_int(h, "KBDLED", leds);
	if (__unlikely(r)) return r;
	read(h, NULL, 0);
	return 0;
}

static __finline__ int fd_getprm(int h, struct floppy_struct *f)
{
	return get_mode_data(h, IOCTL_BIO_FDGETPRM, 0, f, sizeof(struct floppy_struct), 0);
}

int ioctl(int h, unsigned long io, ...)
{
	va_list va;
	/*__debug_printf("ioctl(%d, %lx)\n", h, io);*/
	switch (io) {
		case TIOCGETA: {
			struct termios *t;
			va_start(va, io);
			t = va_arg(va, struct termios *);
			va_end(va);
			return get_tty(h, t);
		}
		case TIOCSETA: {
			struct termios *t;
			va_start(va, io);
			t = va_arg(va, struct termios *);
			va_end(va);
			return set_tty_x(h, t);
		}
		case TIOCSETAW: {
			struct termios *t;
			drain_tty(h);
			va_start(va, io);
			t = va_arg(va, struct termios *);
			va_end(va);
			return set_tty_x(h, t);
		}
		case TIOCSETAF: {
			struct termios *t;
			drain_tty(h);
			flush_tty(h, FREAD);
			va_start(va, io);
			t = va_arg(va, struct termios *);
			va_end(va);
			return set_tty_x(h, t);
		}
		case TIOCSPGRP: {
			return 0;
		}
		case TIOCGPGRP: {
			pid_t *i;
			va_start(va, io);
			i = va_arg(va, pid_t *);
			va_end(va);
			*i = getpgrp();
			return 0;
		}
		case TIOCSBRK: {
			return 0;
		}
		case TIOCDRAIN: {
			return drain_tty(h);
		}
		case TIOCFLUSH: {
			int c;
			va_start(va, io);
			c = va_arg(va, int);
			va_end(va);
			return flush_tty(h, c);
		}
		case TIOCSTOP: {
			return 0;
		}
		case TIOCSTART: {
			return 0;
		}
		case TIOCGWINSZ: {
			struct winsize *w;
			va_start(va, io);
			w = va_arg(va, struct winsize *);
			va_end(va);
			return tty_get_winsize(h, w);
		}
		case TIOCSWINSZ: {
			struct winsize *w;
			va_start(va, io);
			w = va_arg(va, struct winsize *);
			va_end(va);
			return tty_set_winsize(h, w);
		}
		case FIOCLEX: {
			return fcntl(h, F_SETFD, FD_CLOEXEC);
		}
		case FIONCLEX: {
			return fcntl(h, F_SETFD, 0);
		}
		case FIONBIO: {
			int *c;
			int fl;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			if (__unlikely((fl = fcntl(h, F_GETFL)) == -1)) return -1;
			return fcntl(h, F_SETFL, (fl & ~O_NONBLOCK) | (*c ? O_NONBLOCK : 0));
		}
		case FIONREAD: {
			int *n;
			va_start(va, io);
			n = va_arg(va, int *);
			va_end(va);
			return nread(h, n);
		}
		case SNDCTL_DSP_RESET: {
			return snd_sync(h, PARAM_SND_SYNC_RESET);
		}
		case SNDCTL_DSP_SYNC: {
			return snd_sync(h, PARAM_SND_SYNC_SYNC);
		}
		case SNDCTL_DSP_SPEED: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			return snd_speed(h, c);
		}
		case SNDCTL_DSP_STEREO: {
			int *c, d, r;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			if (__unlikely(*c < 0)) d = 0;
			else if (__unlikely(!*c)) d = 1;
			else d = 2;
			r = snd_channels(h, &d);
			if (__likely(!r)) *c = d - 1;
			return r;
		}
		case SNDCTL_DSP_GETBLKSIZE: {
			int *c, r;
			audio_buf_info info;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			r = snd_getspace(h, &info, PARAM_SND_GETSPACE_CURRENT);
			if (__likely(!r)) *c = info.fragsize;
			return r;
		}
		case SNDCTL_DSP_SETFMT: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			return snd_fmt(h, c);
		}
		case SNDCTL_DSP_CHANNELS: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			return snd_channels(h, c);
		}
		case SNDCTL_DSP_POST: {
			return snd_sync(h, PARAM_SND_SYNC_POST);
		}
		case SNDCTL_DSP_SUBDIVIDE: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			return snd_subdivide(h, c);
		}
		case SNDCTL_DSP_SETFRAGMENT: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			return snd_setfragment(h, c);
		}
		case SNDCTL_DSP_GETFMTS: {
			int c = snd_getfmts(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, int *) = c;
			va_end(va);
			return 0;
		}
		case SNDCTL_DSP_GETOSPACE: {
			audio_buf_info *info;
			va_start(va, io);
			info = va_arg(va, audio_buf_info *);
			va_end(va);
			return snd_getspace(h, info, PARAM_SND_GETSPACE_PLAYBACK);
		}
		case SNDCTL_DSP_GETISPACE: {
			audio_buf_info *info;
			va_start(va, io);
			info = va_arg(va, audio_buf_info *);
			va_end(va);
			return snd_getspace(h, info, PARAM_SND_GETSPACE_RECORD);
		}
		case SNDCTL_DSP_NONBLOCK: {
			int fl;
			if (__unlikely((fl = fcntl(h, F_GETFL)) == -1)) return -1;
			return fcntl(h, F_SETFL, fl | O_NONBLOCK);
		}
		case SNDCTL_DSP_GETCAPS: {
			int c = snd_getcaps(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, int *) = c;
			va_end(va);
			return 0;
		}
		case SNDCTL_DSP_GETTRIGGER: {
			int c = snd_gettrigger(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, int *) = c;
			va_end(va);
			return 0;
		}
		case SNDCTL_DSP_SETTRIGGER: {
			int c;
			va_start(va, io);
			c = *va_arg(va, int *);
			va_end(va);
			return snd_settrigger(h, c);
		}
		case SNDCTL_DSP_GETIPTR: {
			count_info *ci;
			va_start(va, io);
			ci = va_arg(va, count_info *);
			va_end(va);
			return snd_getptr(h, ci, PARAM_SND_GETPTR_RECORD);
		}
		case SNDCTL_DSP_GETOPTR: {
			count_info *ci;
			va_start(va, io);
			ci = va_arg(va, count_info *);
			va_end(va);
			return snd_getptr(h, ci, PARAM_SND_GETPTR_PLAYBACK);
		}
		case SNDCTL_DSP_MAPINBUF:
		case SNDCTL_DSP_MAPOUTBUF:
		case SNDCTL_DSP_SETSYNCRO: {
			errno = EINVAL;
			return -1;
		}
		case SNDCTL_DSP_SETDUPLEX: {
			int d;
			if (__unlikely((d = snd_getcaps(h)) < 0)) return d;
			if (__unlikely(!(d & DSP_CAP_DUPLEX))) {
				errno = EINVAL;
				return -1;
			}
			return 0;
		}
		case SNDCTL_DSP_GETODELAY: {
			int c = snd_getodelay(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, int *) = c;
			va_end(va);
			return 0;
		}
		case SNDCTL_DSP_GETERROR: {
			audio_errinfo *e;
			va_start(va, io);
			e = va_arg(va, audio_errinfo *);
			va_end(va);
			return snd_geterror(h, e);
		}
		case SOUND_PCM_READ_RATE: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			*c = 0;
			return snd_speed(h, c);
		}
		case SOUND_PCM_READ_CHANNELS: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			*c = 0;
			return snd_channels(h, c);
		}
		case SOUND_PCM_READ_BITS: {
			int *c;
			va_start(va, io);
			c = va_arg(va, int *);
			va_end(va);
			*c = AFMT_QUERY;
			return snd_fmt(h, c);
		}
		case SOUND_PCM_READ_FILTER:
		case SOUND_PCM_WRITE_FILTER: {
			errno = EINVAL;
			return -1;
		}
		case KDGKBMODE: {
			int c = kd_get_mode(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, int *) = c;
			va_end(va);
			return 0;
		}
		case KDSKBMODE: {
			int c;
			va_start(va, io);
			c = va_arg(va, int);
			va_end(va);
			return kd_set_mode(h, c);
		}
		case KDGETLED: {
			int c = kd_get_leds(h);
			if (__unlikely(c < 0)) return c;
			va_start(va, io);
			*va_arg(va, char *) = c;
			va_end(va);
			return 0;
		}
		case KDSETLED: {
			int c;
			va_start(va, io);
			c = va_arg(va, int);
			va_end(va);
			return kd_set_leds(h, c);
		}
		case FDGETPRM: {
			struct floppy_struct *f;
			va_start(va, io);
			f = va_arg(va, struct floppy_struct *);
			va_end(va);
			return fd_getprm(h, f);
		}
		default: {
			if (io >= _SOUND_MIXER_READ && io < _SOUND_MIXER_READ + _SOUND_MIXER_RW_SPACE) {
				int c = snd_mixread(h, io - _SOUND_MIXER_READ);
				if (__unlikely(c < 0)) return c;
				va_start(va, io);
				*va_arg(va, int *) = c;
				va_end(va);
				return 0;
			}
			if (io >= _SOUND_MIXER_WRITE && io < _SOUND_MIXER_WRITE + _SOUND_MIXER_RW_SPACE) {
				int *c;
				va_start(va, io);
				c = va_arg(va, int *);
				va_end(va);
				return snd_mixwrite(h, io - _SOUND_MIXER_WRITE, c);
			}
			errno = EINVAL;
			return -1;
		}
	}
}
