#include <STRING.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <SIGNAL.H>
#include <SYS/STAT.H>
#include <SYS/TYPES.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SOCKET.H>
#include <SYS/UIO.H>
#include <KERNEL/VMDEF.H>

#include <NETINET/IN.H>
#include <SYS/SOCKET.H>
#include <KERNEL/DEV.H>
#include <NETINET/DHCP.H>

static char last_sock[4] = "\001\001\001\001";

#define SOCKNAMELEN	17	/* net$inet:/@sockTP */

int socket(int domain, int type, int protocol)
{
	OPENRQ o;
	char sockname[SOCKNAMELEN + 1];
	__u32 dom = domain;
	again:
	memcpy(sockname, "NET$", 4);
	memcpy(sockname + 4, &dom, 4);
	sockname[8] = ':';
	sockname[9] = '/';
	sockname[10] = '@';
	memcpy(sockname + 11, last_sock, 4);
	sockname[15] = type + 'A';
	sockname[16] = protocol + 'A';
	sockname[17] = 0;
	inc_csock(last_sock);
	o.flags = O_RDWR | O_CREAT | O_EXCL | _O_SOCKET;
	o.path = sockname;
	o.cwd = NULL;
	SYNC_IO(&o, KERNEL$OPEN);
	if (__unlikely(o.status < 0)) {
		if (__likely(o.status == -EEXIST)) {
			unsigned r = rand();
			last_sock[0] = fixup_sock_char(last_sock[0] ^ r);
			last_sock[1] = fixup_sock_char(last_sock[1] ^ (r >> 8));
			last_sock[2] = fixup_sock_char(last_sock[2] ^ (r >> 16));
			last_sock[3] = fixup_sock_char(last_sock[3] ^ (r >> 24));
			goto again;
		}
		if (o.status == -ENOLNM) errno = EPFNOSUPPORT;
		else errno = -o.status;
		return -1;
	}
	return o.status;
}

int socketpair(int domain, int type, int protocol, int *fd)
{
	int s1, s2, s11;
	socklen_t ssl;
	if (__likely(domain == PF_INET)) {
		struct sockaddr_in sin;
		s1 = socket(domain, type, protocol);
		if (__unlikely(s1 == -1)) return -1;
		s2 = socket(domain, type, protocol);
		if (__unlikely(s2 == -1)) {
			close(s1);
			return -1;
		}
		memset(&sin, 0, sizeof sin);
		sin.sin_family = AF_INET;
		sin.sin_port = IPPORT_ANY;
		sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
		if (__unlikely(bind(s1, (struct sockaddr *)&sin, sizeof sin))) goto cls;
		ssl = sizeof sin;
		if (__unlikely(getsockname(s1, (struct sockaddr *)&sin, &ssl))) goto cls;
		if (__unlikely(listen(s1, 1))) goto cls;
		if (__unlikely(connect(s2, (struct sockaddr *)&sin, sizeof sin))) goto cls;
		if (__unlikely((s11 = accept(s1, NULL, NULL)) == -1)) goto cls;
		close(s1);
		fd[0] = s11;
		fd[1] = s2;
		return 0;
	}
	errno = EPFNOSUPPORT;
	return -1;
	cls:
	close(s1);
	close(s2);
	return -1;
}

int accept(int h, struct sockaddr *addr, socklen_t *addrlen)
{
	IOCTLRQ io;
	OPENRQ o;
	char *sockname;
	char *psname;
	int l;
	io.h = h;
	io.ioctl = IOCTL_ACCEPT;
	io.param = 0;
	io.v.ptr = (unsigned long)addr;
	io.v.len = addr && __likely(addrlen != NULL) ? *addrlen : 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	if (addr && __likely(addrlen != NULL)) *addrlen = addr->sa_len;
	psname = KERNEL$HANDLE_PATH(io.h);
	if (__unlikely(!psname)) {
		errno = EBADF;
		return -1;
	}
	l = strlen(psname);
	if (__unlikely(l < SOCKNAMELEN)) {
		errno = ENOTSOCK;
		return -1;
	}
	sockname = alloca(strlen(psname) + 1);
	strcpy(sockname, psname);
	sockname[11] = io.status;
	sockname[12] = io.status >> 8;
	sockname[13] = io.status >> 16;
	sockname[14] = io.status >> 24;
	if (strstr(sockname, "^NONBLOCK")) o.flags = O_RDWR | _O_SOCKET | O_NONBLOCK;
	o.flags = O_RDWR | _O_SOCKET;
	o.path = sockname;
	o.cwd = NULL;
	SYNC_IO(&o, KERNEL$OPEN);
	if (__unlikely(o.status < 0)) {
		errno = -o.status;
		return -1;
	}
	return o.status;
}

int bind(int h, const struct sockaddr *addr, socklen_t addrlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_BIND;
	io.param = 0;
	io.v.ptr = (unsigned long)addr;
	io.v.len = __likely(addr != NULL) ? addrlen : 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	return 0;
}

int connect(int h, const struct sockaddr *addr, socklen_t addrlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_CONNECT;
	io.param = 0;
	io.v.ptr = (unsigned long)addr;
	io.v.len = addrlen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	return 0;
}

int getpeername(int h, struct sockaddr *addr, socklen_t *addrlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_GETPEERNAME;
	io.param = 0;
	io.v.ptr = (unsigned long)addr;
	io.v.len = *addrlen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	*addrlen = addr->sa_len;
	return 0;
}

int getsockname(int h, struct sockaddr *addr, socklen_t *addrlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_GETSOCKNAME;
	io.param = 0;
	io.v.ptr = (unsigned long)addr;
	io.v.len = *addrlen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	*addrlen = addr->sa_len;
	return 0;
}

int listen(int h, int backlog)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_LISTEN;
	io.param = backlog;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	return 0;
}

int getsockopt(int h, int level, int opt, void *optval, socklen_t *optlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_GETSOCKOPT;
	io.param = __SO_MAKEPARAM(level, opt);
	io.v.ptr = (unsigned long)optval;
	io.v.len = *optlen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	*optlen = io.status;
	return 0;
}

int setsockopt(int h, int level, int opt, const void *optval, socklen_t optlen)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_SETSOCKOPT;
	io.param = __SO_MAKEPARAM(level, opt);
	io.v.ptr = (unsigned long)optval;
	io.v.len = optlen;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	return 0;
}

int shutdown(int h, int type)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_SHUTDOWN;
	io.param = type;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = ENOTSOCK;
		else errno = -io.status;
		return -1;
	}
	return 0;
}

static ssize_t _recvfrom(int h, void *buf, size_t size, int *flags, struct sockaddr *from, socklen_t *from_len)
{
	struct _msghdr *msg;
	AIORQ a;
	socklen_t zero = 0;
	int f = KERNEL$HANDLE_FLAGS(h);
	if (__unlikely(f == -1)) return -1;
	if (__unlikely(!(f & _O_SOCKET))) {
		errno = ENOTSOCK;
		return -1;
	}
	if (!from_len) from_len = &zero;
	if (!flags && !*from_len) return read(h, buf, size);
	msg = alloca(sizeof(struct _msghdr) + *from_len);
	if (__unlikely(((unsigned long)msg & (PAGE_CLUSTER_SIZE - 1)) + sizeof(struct _msghdr) + *from_len > PAGE_CLUSTER_SIZE)) msg = alloca(sizeof(struct _msghdr) + *from_len);
	msg->msg_flags = *flags;
	msg->msg_addr.sa_len = *from_len;
	a.h = h;
	a.v.ptr = (unsigned long)buf;
	a.v.len = size;
	a.v.vspace = &KERNEL$VIRTUAL;
	a.pos = (unsigned long)msg;
	a.progress = 0;
	__cancel_in(SYNC_IO_SIGINTERRUPT(&a, KERNEL$AREAD));
	if (__likely(a.status >= 0)) {
		c:
		*flags = msg->msg_flags;
		memcpy(from, &msg->msg_addr, *from_len = msg->msg_addr.sa_len);
		return a.status;
	}
	if (__unlikely(a.progress != 0)) {
		a.status = a.progress;
		goto c;
	}
	errno = -a.status;
	return -1;
}

ssize_t recv(int h, void *buf, size_t size, int flags)
{
	if (!flags) return read(h, buf, size);
	return _recvfrom(h, buf, size, &flags, NULL, NULL);
}

ssize_t recvfrom(int h, void *buf, size_t size, int flags, struct sockaddr *from, socklen_t *from_len)
{
	return _recvfrom(h, buf, size, &flags, from, from_len);
}

ssize_t recvmsg(int h, struct msghdr *msg, int flags)
{
	int i;
	char *p;
	size_t s = 0, ss, sss;
	for (i = 0; i < msg->msg_iovlen; i++) s += msg->msg_iov[i].iov_len;
	if (__unlikely(!(p = __sync_malloc(s)))) return -1;
	ss = _recvfrom(h, p, s, &msg->msg_flags, msg->msg_name, &msg->msg_namelen);
	sss = ss;
	if (__likely(ss != (size_t)-1)) {
		char *pp = p;
		for (i = 0; i < msg->msg_iovlen; i++) {
			size_t s = msg->msg_iov[i].iov_len;
			if (s >= ss) s = ss;
			memcpy(pp, msg->msg_iov[i].iov_base, s);
			if (!(ss -= s)) break;
			pp += s;
		}
	}
	free(p);
	return sss;
}

ssize_t send(int h, const void *buf, size_t size, int flags)
{
	if (!flags) return write(h, buf, size);
	return sendto(h, buf, size, flags, NULL, 0);
}

ssize_t sendto(int h, const void *buf, size_t size, int flags, const struct sockaddr *from, socklen_t from_len)
{
	struct _msghdr *msg;
	AIORQ a;
	int f = KERNEL$HANDLE_FLAGS(h);
	if (__unlikely(f == -1)) return -1;
	if (__unlikely(!(f & _O_SOCKET))) {
		errno = ENOTSOCK;
		return -1;
	}
	if (!flags && !from_len) return write(h, buf, size);
	msg = alloca(sizeof(struct _msghdr) + from_len);
	if (__unlikely(((unsigned long)msg & (PAGE_CLUSTER_SIZE - 1)) + sizeof(struct _msghdr) + from_len > PAGE_CLUSTER_SIZE)) msg = alloca(sizeof(struct _msghdr) + from_len);
	msg->msg_flags = flags;
	memcpy(&msg->msg_addr, from, from_len);
	msg->msg_addr.sa_len = from_len;
	a.h = h;
	a.v.ptr = (unsigned long)buf;
	a.v.len = size;
	a.v.vspace = &KERNEL$VIRTUAL;
	a.pos = (unsigned long)msg;
	a.progress = 0;
	__cancel_in(SYNC_IO_SIGINTERRUPT(&a, KERNEL$AWRITE));
	if (__unlikely(flags & MSG_EOF)) shutdown(h, SHUT_WR);
	if (__likely(a.status >= 0)) return a.status;
	if (__unlikely(a.progress != 0)) return a.progress;
	if (__unlikely(a.status == -EPIPE) && __likely(!(flags & MSG_NOSIGNAL))) raise(SIGPIPE);
	errno = -a.status;
	return -1;
}

ssize_t sendmsg(int h, const struct msghdr *msg, int flags)
{
	int i;
	char *p;
	size_t s = 0, ss;
	for (i = 0; i < msg->msg_iovlen; i++) s += msg->msg_iov[i].iov_len;
	if (__unlikely(!(p = __sync_malloc(s)))) return -1;
	s = 0;
	for (i = 0; i < msg->msg_iovlen; i++) ss = s, memcpy(p + ss, msg->msg_iov[i].iov_base, s += msg->msg_iov[i].iov_len);
	ss = sendto(h, p, s, msg->msg_flags, msg->msg_name, msg->msg_namelen);
	free(p);
	return ss;
}

static ssize_t sendfile_copy(int h, int in_h, off_t *offset, size_t count)
{
	char buffer[2048];
	size_t t = 0;
	while (count) {
		int w, p;
		int r = pread(in_h, buffer, count > sizeof(buffer) ? sizeof(buffer) : count, *offset);
		if (__unlikely(r <= 0)) {
			if (t) return t;
			return r;
		}
		p = 0;
		do {
			w = write(h, buffer + p, r);
			if (__unlikely(w <= 0)) {
				if (t) return t;
				return w;
			}
			t += w;
			*offset += w;
			p += w;
		} while (r -= w);
		count -= p;
	}
	return t;
}

static size_t sendfile_remaining(int in_h, off_t pos)
{
	struct stat st;
	if (__unlikely(fstat(in_h, &st))) return 0;
	if (__unlikely((_u_off_t)st.st_size - (_u_off_t)pos > 0xfffffffeU)) return 0;
	return st.st_size - pos;
}

ssize_t sendfile(int h, int in_h, off_t *offset, size_t count)
{
	SIORQ s;
	if (__unlikely(count == -1)) {
		count = sendfile_remaining(in_h, *offset);
		if (__unlikely(!count)) {
			count = -1;
			goto c;
		}
	}
	if (__unlikely(!offset)) {
		offset = (off_t *)(void *)&KERNEL$LIST_END;
		s.v.ptr = 0;
	} else {
		s.v.ptr = *offset;
	}
	s.h = h;
	s.v.len = count;
	s.v.vspace = KERNEL$GET_HANDLE(in_h);
	s.progress = 0;
	__cancel_in(SYNC_IO_SIGINTERRUPT(&s, KERNEL$WRITE));
	*offset = s.v.ptr;
	if (__likely(s.status >= 0)) return s.status;
	if (__unlikely(s.progress != 0)) return s.progress;
	if (__unlikely(s.status == -ENOVSPACE)) c:return sendfile_copy(h, in_h, offset, count);
	if (__unlikely(s.status == -EPIPE)) _thread_raise(SIGPIPE);
	errno = -s.status;
	return -1;
}

int dhcp_option(int domain, int option, char *buf, size_t size)
{
	__u32 dom = domain;
	char netname[11];
	int h;
	IOCTLRQ io;
	memcpy(netname, "NET$", 4);
	memcpy(netname + 4, &dom, 4);
	netname[8] = ':';
	netname[9] = '/';
	netname[10] = 0;
	h = open(netname, O_RDONLY | _O_NOPOSIX);
	if (__unlikely(h == -1)) return -1;
	io.h = h;
	io.ioctl = IOCTL_DHCP_OPTION;
	io.param = option;
	io.v.ptr = (unsigned long)buf;
	io.v.len = size;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);
	close(h);
	if (__unlikely(io.status < 0)) {
		errno = -io.status;
		return -1;
	}
	return io.status;
}

void SOCKET_CLEANUP(void)
{
	int i;
	LOGICAL_LIST_REQUEST llr;
	llr.prefix = "NET$";
	SYNC_IO(&llr, KERNEL$LIST_LOGICALS);
	if (__unlikely(llr.status < 0)) return;
	for (i = 0; i < llr.n_entries; i++) {
		int h;
		union {
			IOCTLRQ io;
			char name[__MAX_STR_LEN - 1 + 3];
		} u;
		char *n = llr.entries[i]->name;
		stpcpy(stpcpy(u.name, n), ":/");
		h = open(u.name, O_RDONLY | _O_NOPOSIX | O_TRUNC);
		if (__likely(h != -1)) {
			u.io.h = h;
			u.io.ioctl = IOCTL_SOCKET_LINGER;
			u.io.param = 0;
			u.io.v.ptr = 0;
			u.io.v.len = 0;
			u.io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&u.io, KERNEL$IOCTL);
			close(h);
		}
	}
	KERNEL$FREE_LOGICAL_LIST(&llr);
}


