#include <SYS/TYPES.H>
#include <ERRNO.H>
#include <SPAD/AC.H>
#include <SPAD/LIBC.H>
#include <SPAD/DL.H>
#include <SPAD/WQ.H>
#include <SPAD/SYNC.H>
#include <KERNEL/IRQ.H>
#include <KERNEL/DEV.H>
#include <ARCH/SETUP.H>

#if __DEBUG >= 1
static __NORET_ATTR__ void RQ_STATE_ERROR(IORQ *io, char *msg)
{
	unsigned long off = 0, offa = 0;
	__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME((void *)io->tmp1, &off, 0), *stra = KERNEL$DL_GET_SYMBOL_NAME((void *)io->fn, &offa, 0);
	KERNEL$SUICIDE("RQ_STATE_ERROR (%s): REQUEST IN STATE %08lX, CALLER %s+%lX, AST %s+%lX", msg, io->status, str ? str : "?", off, stra ? stra : "?", offa);
}

void CHECK_RQ_STATE(IORQ *io)
{
	if (__unlikely(io->status == RQS_WANTCANCEL)) return;
	if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_PROCESSING)) {
		RQ_STATE_ERROR(io, "CHECK_RQ_STATE");
	}
}
#endif

#if __DEBUG >= 2 && defined(__WQ_DEBUG)
#define REGISTER_WQS
#endif

#ifdef REGISTER_WQS
static void DO_WQ_DUMP(void);
static DECL_XLIST(ALL_WQS);
__finline__ void CHECK_WQ_EMPTY(WQ *wq)
{
	if (__unlikely(!wq->list.next) || __unlikely(!wq->list.prev))
		KERNEL$SUICIDE("CHECK_WQ_EMPTY: WQ LIST CORRUPTED, WQ %s", wq->name);
	if (WQ_EMPTY(wq)) {	
		DEL_FROM_LIST(&wq->list);
		wq->list.next = wq->list.prev = NULL;
	}
}
#else
#define CHECK_WQ_EMPTY(x)
#endif

static void WAIT_SPL(WQ *wq, IORQ *io, int spl);

void KERNEL$WQ_WAIT(WQ *wq, IORQ *io)
{
	WAIT_SPL(wq, io, KERNEL$SPL);
}

static void WAIT_SPL(WQ *wq, IORQ *io, int spl)
{
	WQ *q, *p, *wi;
	if (__unlikely(io == &KERNEL$DUMMY_IORQ)) return;
	RAISE_SPL(SPL_TOP);
	wi = __IORQ2WQ(io);
#define io __WQ2IORQ(wi)
	if (__unlikely(io->status == RQS_WANTCANCEL)) goto azmb;
#if __DEBUG >= 1
	if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_PROCESSING)) {
		RQ_STATE_ERROR(io, "WAIT_SPL");
	}
#endif
#ifdef REGISTER_WQS
	if (__likely(wq->next == wq)) {
		if (__unlikely((unsigned long)wq->list.next | (unsigned long)wq->list.prev)) KERNEL$SUICIDE("WAIT_SPL: WQ LIST CORRUPTED (EMPTY WQ, NONEMPTY LIST), WQ %s", wq->name);
		ADD_TO_XLIST(&ALL_WQS, &wq->list);
	}
#endif
#ifdef REGISTER_WQS
	if (__unlikely(!wq->list.next) || __unlikely(!wq->list.prev))
		KERNEL$SUICIDE("CHECK_WQ_EMPTY: WQ LIST CORRUPTED (NONEMPTY WQ, EMPTY LIST), WQ %s", wq->name);
#endif
	for (q = wq->prev; q != wq && __unlikely((__WQ2IORQ(q)->status | RQS_THRESHOLD_MASK) < io->status); q = q->prev) if (__unlikely(SPLX_BUSY(spl))) {
		if (__WQ2IORQ(wq->next)->status < io->status) q = wq;
		break;
	}
	p = q->next;
	(p->prev = wi)->next = p;
	(q->next = wi)->prev = q;
	io->status = ((io->status + RQS_ADD_PRIORITY) & RQS_PRIORITY_MASK) | RQS_WQCANCELABLE;
	azml:
	LOWER_SPLX(spl);
	return;
	azmb:
	io->status = -EINTR;
	CALL_AST(io);
	goto azml;
#undef io
}

IORQ *KERNEL$WQ_GET_ONE(WQ *wq)
{
	IORQ *io;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__likely(!WQ_EMPTY(wq))) {
		io = __WQ2IORQ(wq->next);
		(wq->next = wq->next->next)->prev = wq;
#if __DEBUG >= 1
		if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_WQCANCELABLE)) RQ_STATE_ERROR(io, "KERNEL$WQ_GET_ONE");
#endif
		io->status = (io->status & RQS_PRIORITY_MASK) | RQS_PROCESSING;
		CHECK_WQ_EMPTY(wq);
	} else io = NULL;
	LOWER_SPLX(spl);
	return io;
}

void KERNEL$WQ_WAKE_ONE(WQ *wq)
{
	IORQ *io;
	int spl;
#ifdef REGISTER_WQS
	if (__unlikely(!strcmp(wq->name, __WQ_DUMP_MAGIC)))
		DO_WQ_DUMP();
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__likely(!WQ_EMPTY(wq))) {
		io = __WQ2IORQ(wq->next);
		(wq->next = wq->next->next)->prev = wq;
#if __DEBUG >= 1
		if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_WQCANCELABLE)) RQ_STATE_ERROR(io, "KERNEL$WQ_WAKE_ONE");
#endif
		io->status = (io->status & RQS_PRIORITY_MASK) | RQS_PROCESSING;
		CALL_IORQ_LSTAT_EXPR(io, (IO_STUB *)io->tmp1);
		CHECK_WQ_EMPTY(wq);
	}
	LOWER_SPLX(spl);
}

void KERNEL$WQ_WAKE_ALL(WQ *wq)
{
	int spl = KERNEL$SPL;
#ifndef REGISTER_WQS
	WQ wqq;
	unsigned long count;
	regain:
	count = INTR_COUNT;
#endif
	RAISE_SPL(SPL_TOP);
	while (!WQ_EMPTY(wq)) {
		IORQ *io;
		io = __WQ2IORQ(wq->next);
		(wq->next = wq->next->next)->prev = wq;
#if __DEBUG >= 1
		if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_WQCANCELABLE)) RQ_STATE_ERROR(io, "KERNEL$WQ_WAKE_ALL");
#endif
		io->status = (io->status & RQS_PRIORITY_MASK) | RQS_PROCESSING;
		CALL_IORQ_LSTAT_EXPR(io, (IO_STUB *)io->tmp1);
		CHECK_WQ_EMPTY(wq);
#ifndef REGISTER_WQS
		if (__unlikely(count != INTR_COUNT)) goto drop;
#endif
	}
	LOWER_SPLX(spl);
	return;

#ifndef REGISTER_WQS
	drop:
	if (__likely(wq != &wqq)) {
		WQ *wq1 = wq->next, *wq2 = wq->prev;
		wq1->prev = &wqq;
		wq2->next = &wqq;
		wqq.next = wq->next;
		wqq.prev = wq->prev;
		wq->next = wq->prev = wq;
		wq = &wqq;
	}
	LOWER_SPLX(spl);
	goto regain;
#endif
}

#ifdef REGISTER_WQS

static void DO_WQ_DUMP(void)
{
	WQ *wq, *wqq;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	XLIST_FOR_EACH(wq, &ALL_WQS, WQ, list) {
		if (__likely(wq != LIST_STRUCT(ALL_WQS.next, WQ, list))) __critical_printf("  ");
		__critical_printf("*WQ %s: ", wq->name);
		for (wqq = wq->next; wqq != wq; wqq = wqq->next) {
			IORQ *io = __WQ2IORQ(wqq);
			unsigned long off = 0, offa = 0;
			__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME((void *)io->tmp1, &off, 0), *stra = KERNEL$DL_GET_SYMBOL_NAME((void *)io->fn, &offa, 0);
			if (wqq != wq->next) __critical_printf(", ");
			__critical_printf("(IO %s+%lX, AST %s+%lX)", str ? str : "?", off, stra ? stra : "?", offa);
			if (__unlikely(SPLX_BUSY(spl))) {
				WQ *xwq, *xwqq;
				WQ *wq_p = wq->prev, *wqq_p = wqq->prev;
				WQ *wq_n = wq->next, *wqq_n = wqq->next;
				LOWER_SPLX(spl);
				RAISE_SPL(SPL_TOP);
				XLIST_FOR_EACH(xwq, &ALL_WQS, WQ, list) if (xwq == wq) goto wq_ok;
				XLIST_FOR_EACH(xwq, &ALL_WQS, WQ, list) if (xwq == wq_p) {
					wq = xwq;
					goto cont_wq;
				}
				XLIST_FOR_EACH(xwq, &ALL_WQS, WQ, list) if (xwq == wq_n) {
					wq = xwq->prev;
					goto cont_wq;
				}
				__critical_printf("LISTING INTERRUPTED");
				goto brk_wq;
				wq_ok:
				for (xwqq = wq->next; xwqq != wq; xwqq = xwqq->next) if (xwqq == wqq) goto wqq_ok;
				for (xwqq = wq->next; xwqq != wq; xwqq = xwqq->next) if (xwqq == wqq_p) {
					wqq = xwqq;
					goto wqq_ok;
				}
				for (xwqq = wq->next; xwqq != wq; xwqq = xwqq->next) if (xwqq == wqq_n) {
					wqq = xwqq->prev;
					goto wqq_ok;
				}
				__critical_printf("QUEUE LIST INTERRUPTED");
				goto cont_wq;
				wqq_ok:;
			}
		}
		cont_wq:;
	}
	brk_wq:
	LOWER_SPLX(spl);
	__critical_printf("\n");
}

#endif

int KERNEL$MTX_LOCK(MTX *mtx, IORQ *io, int dontlock)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__unlikely(io->status == RQS_WANTCANCEL)) goto azmb;
#if __DEBUG >= 1
	if (__unlikely((io->status & RQS_ACTION_MASK) != RQS_PROCESSING)) {
		RQ_STATE_ERROR(io, "KERNEL$MTX_LOCK");
	}
#endif
	if (__unlikely(!mtx->lock)) {
		if (__likely(dontlock)) {
			CALL_IORQ_LSTAT_EXPR(io, (IO_STUB *)io->tmp1);
		} else {
			mtx->lock = 1;
		}
		LOWER_SPLX(spl);
		return 0;
	}
	WAIT_SPL(&mtx->wq, io, spl);
	return 1;

	azmb:
	io->status = -EINTR;
	CALL_AST(io);
	LOWER_SPLX(spl);
	return 1;
}

void KERNEL$MTX_UNLOCK(MTX *mtx)
{
	__barrier();
	mtx->lock = 0;
	__barrier();
	WQ_WAKE_ALL_PL(&mtx->wq);
}

typedef struct {
	IORQ_HEAD;
	MTX *mtx;
} LIORQ;

DECL_IOCALL(DO_LOCK, SPL_THREAD, LIORQ)
{
	MTX_LOCK(RQ->mtx, RQ, DO_LOCK, RETURN);
	RQ->status = 0;
	RETURN_AST(RQ);
}

void KERNEL$MTX_LOCK_SYNC(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO(&liorq, DO_LOCK);
}

int KERNEL$MTX_LOCK_SYNC_CANCELABLE(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO_CANCELABLE(&liorq, DO_LOCK);
	return liorq.status;
}

int KERNEL$MTX_LOCK_SYNC_SIGINTERRUPT(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO_SIGINTERRUPT(&liorq, DO_LOCK);
	return liorq.status;
}

DECL_IOCALL(DO_WAIT, SPL_THREAD, LIORQ)
{
	MTX_WAIT(RQ->mtx, RQ, KERNEL$SUCCESS);
	RETURN;
}

void KERNEL$MTX_WAIT_SYNC(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO(&liorq, DO_WAIT);
}

int KERNEL$MTX_WAIT_SYNC_CANCELABLE(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO_CANCELABLE(&liorq, DO_WAIT);
	return liorq.status;
}

int KERNEL$MTX_WAIT_SYNC_SIGINTERRUPT(MTX *mtx)
{
	LIORQ liorq;
	liorq.mtx = mtx;
	SYNC_IO_SIGINTERRUPT(&liorq, DO_WAIT);
	return liorq.status;
}

void KERNEL$MTX_SET(MTX *mtx, unsigned long val)
{
	if (!val) {
		KERNEL$MTX_UNLOCK(mtx);
		return;
	}
	mtx->lock = val;
}
