#include <STRING.H>
#include <STDLIB.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <SPAD/CHARSET.H>
#include <SPAD/WQ.H>
#include <ICONV.H>

#include "CHRCACHE.H"
#include "CHRICONV.H"

#define SPLIT_CHAR	'|'

typedef struct {
	CHRCACHE_HEAD;
	__const__ struct iconv_charset *from;
	void *from_ex;
	__const__ struct iconv_charset *to;
	void *to_ex;
	FALLBACK *fb;
	unsigned char flags;
} ICONV;

typedef struct __iconv_t {
	ICONV *ic;
	void *from_state;
	void *to_state;
} ICONV_STATE;

static void DESTROY_ICONV(CHRCACHE *chr)
{
	ICONV *ic = (ICONV *)chr;
	if (ic->from->free_charset) ic->from->free_charset(ic->from_ex);
	if (ic->to->free_charset) ic->to->free_charset(ic->to_ex);
	CHARSET$CLOSE_FALLBACK(ic->fb, 1);
}

#define SPECIAL_8B(c)	((c) == 9 || (c) == 10 || (c) == 13 || (c) == 27)

static ssize_t get_char_8b(__const__ char *buf, size_t len, __s32 *uni, void *ex)
{
	if (__unlikely(SPECIAL_8B(*buf))) {
		*uni = *buf;
		return 1;
	}
	if (__unlikely((*uni = CHARSET$8B2UNI(ex, *buf)) == -1)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	return 1;
}

static ssize_t put_char_8b(char *buf, size_t len, __s32 uni, void *ex)
{
	int ch;
	if (__unlikely(!len)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	if (__unlikely(SPECIAL_8B(uni))) {
		*buf = uni;
		return 1;
	}
	ch = CHARSET$UNI28B(ex, uni);
	if (__likely(ch >= 0)) {
		*buf = ch;
		return 1;
	}
	return ICONV_FALLBACK_NEEDED;
}

static void free_charset_8b(void *ex)
{
	CHARSET$CLOSE_8B(ex, 1);
}

static void *init_state_8b(void *ex)
{
	return ex;
}

static __const__ struct iconv_charset charset_8b = {
	get_char_8b,
	put_char_8b,
	free_charset_8b,
	init_state_8b,
	NULL,
};

static __const__ struct iconv_charset *GET_CHARSET(__const__ char *name, void **ex)
{
	CHARSET_8B *ch;
	__const__ struct iconv_charset *ret = ICONV_GET_CHARSET(name, ex);
	if (ret) return ret;
	ch = CHARSET$OPEN_8B(name);
	if (__unlikely(!ch)) return NULL;
	*ex = ch;
	return &charset_8b;
}

static int ICONV_INIT(CHRCACHE *chr, unsigned long str)
{
	ICONV *ic = (ICONV *)chr;
	char *to, *from;
	char *to1, *from1, *opt1;
	from1 = (char *)str;
	to1 = strchr(from1, SPLIT_CHAR);
	*to1++ = 0;
	opt1 = strchr(to1, SPLIT_CHAR);
	*opt1++ = 0;
	if (__unlikely(opt1[1])) {
		errno = EINVAL;
		return -1;
	}
	ic->flags = opt1[0] - 0x80;
	from = CHARSET$RESOLVE_ALIAS(from1);
	if (__unlikely(!from)) return -1;
	to = CHARSET$RESOLVE_ALIAS(to1);
	if (__unlikely(!to)) {
		__slow_free(from);
		return -1;
	}
	ic->from = GET_CHARSET(from, &ic->from_ex);
	free(from);
	if (__unlikely(!ic->from)) {
		__slow_free(to);
		return -1;
	}
	ic->to = GET_CHARSET(to, &ic->to_ex);
	free(to);
	if (__unlikely(!ic->to)) {
		if (ic->from->free_charset) ic->from->free_charset(ic->from_ex);
		return -1;
	}
	if (__unlikely(ic->flags & (ICONV_ATTR_NO_FALLBACK | ICONV_ATTR_DELAYED_LOADING))) ic->fb = NULL;
	else ic->fb = CHARSET$OPEN_FALLBACK(NULL);
	ic->destructor = DESTROY_ICONV;
	return 0;
}

/*
#include <fcntl.h>
#include <unistd.h>
*/

ICONV_STATE *CHARSET$ICONV_OPEN(__const__ char *to, __const__ char *from, int flags)
{
	ICONV *ic;
	ICONV_STATE *ic_st;
	char *chs, *ptr;
	/*{
		int h = open("tmp:/ic", O_WRONLY | O_CREAT | O_APPEND, 0644);
		write(h, from, strlen(from));
		write(h, "-", 1);
		write(h, to, strlen(to));
		write(h, "\n", 1);
		close(h);
	}*/
	chs = alloca(strlen(from) + strlen(to) + 3 + 3);
	ptr = stpcpy(chs, from);
	*ptr++ = SPLIT_CHAR;
	ptr = stpcpy(ptr, to);
	*ptr++ = SPLIT_CHAR;
	*ptr++ = flags + 0x80;
	*ptr = 0;
	__upcase(chs);
	ic = (ICONV *)CHRCACHE_OPEN(chs, TYPE_ICONV, NULL, sizeof(ICONV), ICONV_INIT);
	if (__unlikely(!ic)) return NULL;
	if (__unlikely(!(ic_st = __sync_malloc(sizeof(ICONV_STATE))))) {
		ret_1:
		CHRCACHE_CLOSE((CHRCACHE *)ic, 0, TYPE_ICONV);
		return NULL;
	}
	ic_st->ic = ic;
	if (ic->from->init_state) {
		if (!(ic_st->from_state = ic->from->init_state(ic->from_ex))) {
			ret_2:
			__slow_free(ic_st);
			goto ret_1;
		}
	} else {
		ic_st->from_state = NULL;
	}
	if (ic->to->init_state) {
		if (!(ic_st->to_state = ic->to->init_state(ic->to_ex))) {
			if (ic->from->free_state) ic->from->free_state(ic_st->from_state);
			goto ret_2;
		}
	} else {
		ic_st->to_state = NULL;
	}
	if (flags & ICONV_ATTR_LEAVE_AT_UNLOAD) {
		ic->leave = 1;
	}
	return ic_st;
}

void CHARSET$ICONV_CLOSE(ICONV_STATE *ic_st)
{
	ICONV *ic = ic_st->ic;
	if (ic->from->free_state) ic->from->free_state(ic_st->from_state);
	if (ic->to->free_state) ic->to->free_state(ic_st->to_state);
	free(ic_st);
	CHRCACHE_CLOSE((CHRCACHE *)ic, 1, TYPE_ICONV);
}

static ssize_t ICONV_FALLBACK_OUTPUT(char *outbuf, size_t outlen, __s32 uni, ICONV *ic);

size_t CHARSET$ICONV(ICONV_STATE *ic_st, char **inbuf, size_t *inlen, char **outbuf, size_t *outlen)
{
	ICONV *ic = ic_st->ic;
	size_t n;
	if (__unlikely(!inbuf) || __unlikely(!*inbuf)) return 0;
	n = 0;
	while (__likely(*inlen != 0)) {
		ssize_t inbytes, outbytes;
		__s32 uni;
		if (__unlikely(ic->flags & ICONV_ATTR_PASS_CONTROL) && (__unlikely((__u8)**inbuf < 0x20) || __unlikely((__u8)**inbuf == 0x7f))) {
			if (__unlikely(!*outlen)) {
				errno = E2BIG;
				return -1;
			}
			**outbuf = **inbuf;
			inbytes = 1;
			outbytes = 1;
			goto shift_it;
		}
		if (__unlikely((inbytes = ic->from->get_char(*inbuf, *inlen, &uni, ic_st->from_state)) < 0)) {
			if (__unlikely(errno == EILSEQ) && __likely(ic->flags & ICONV_ATTR_IGNORE_INVALID_INPUT)) {
				*inbuf += 1;
				*inlen -= 1;
				goto cont;
			}
			return -1;
		}
		if (__unlikely(uni == -1)) {
			outbytes = 0;
		} else if (__unlikely((outbytes = ic->to->put_char(*outbuf, *outlen, uni, ic_st->to_state)) < 0)) {
			if (__likely(outbytes == ICONV_FALLBACK_NEEDED)) {
				if (__unlikely((outbytes = ICONV_FALLBACK_OUTPUT(*outbuf, *outlen, uni, ic)) < 0)) return -1;
			} else {
				return -1;
			}
			n++;
		}
		shift_it:
		*outbuf += outbytes;
		*outlen -= outbytes;
		*inbuf += inbytes;
		*inlen -= inbytes;
		cont:;
	}
	return n;
}

static ssize_t ICONV_FALLBACK_OUTPUT(char *outbuf, size_t outlen, __s32 uni, ICONV *ic)
{
	size_t out_pos;
	__s32 *str;
	unsigned order = 0;
	if (__unlikely(!ic->fb)) {
		static MTX_DECL(ICONV_MTX, "CHRARSET$ICONV_MTX");
		if (__unlikely((ic->flags & (ICONV_ATTR_NO_FALLBACK | ICONV_ATTR_DELAYED_LOADING)) != ICONV_ATTR_DELAYED_LOADING)) {
			goto no_fallback;
		}
		MTX_LOCK_SYNC(&ICONV_MTX);
		if (__unlikely(!ic->fb)) {
			ic->fb = CHARSET$OPEN_FALLBACK(NULL);
		}
		MTX_UNLOCK(&ICONV_MTX);
	}
	next_order:
	str = CHARSET$GET_FALLBACK(ic->fb, uni, order);
	if (__unlikely(!str)) {
		no_fallback:
		if (__likely(ic->flags & ICONV_ATTR_IGNORE_IMPOSSIBLE_OUTPUT)) {
			return 0;
		} else {
			errno = EILSEQ;
			return ICONV_ERROR;
		}
	}
	out_pos = 0;
	while (*str != -1) {
		ssize_t outbytes = ic->to->put_char(outbuf + out_pos, outlen - out_pos, *str, ic->to_ex);
		if (__unlikely(outbytes < 0)) {
			if (outbytes == ICONV_FALLBACK_NEEDED) {
				order++;
				goto next_order;
			}
			return outbytes;
		}
		out_pos += outbytes;
		str++;
	}
	return out_pos;
}

