#include <STRING.H>
#include <ERRNO.H>
#include <ARCH/BSF.H>
#include <STDLIB.H>
#include <SPAD/CHARSET.H>

#include "CHRICONV.H"

/*********/
/* UTF-8 */
/*********/

static ssize_t get_char_utf8_long(__const__ unsigned char *buf, size_t len, __s32 *uni);

static ssize_t get_char_utf8(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	if (__likely(*(unsigned char *)buf < 0x80)) {
		*uni = *(unsigned char *)buf;
		return 1;
	}
	return get_char_utf8_long(buf, len, uni);
}

static ssize_t get_char_utf8_long(__const__ unsigned char *buf, size_t len, __s32 *uni)
{
	unsigned c = *buf;
	unsigned bit, nb;
	__const__ unsigned char *endptr;
	__u32 val;
	if (__unlikely(c < 0xc0) || __unlikely(c >= 0xfe)) {
		eilseq:
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	bit = __BSR(c ^ 0xff);
	nb = 7 - bit;
	if (__unlikely(len < nb)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	endptr = buf++ + nb;
	val = c & ((1 << bit) - 1);
	do {
		c = *buf ^ 0x80;
		if (__unlikely(c >= 0x40)) goto eilseq;
		val = (val << 6) | c;
	} while (__unlikely(++buf < endptr));
	*uni = val;
	return nb;
}

static ssize_t put_char_utf8_long(unsigned char *buf, size_t len, __s32 uni);

static ssize_t put_char_utf8(char *buf, size_t len, __s32 uni, void *state)
{
	if (__likely((__u32)uni < 0x80)) {
		if (__unlikely(!len)) {
			errno = E2BIG;
			return ICONV_ERROR;
		}
		*buf = uni;
		return 1;
	}
	return put_char_utf8_long(buf, len, uni);
}

static ssize_t put_char_utf8_long(unsigned char *buf, size_t len, __s32 uni)
{
	unsigned char *ptr;
	unsigned bit, xby;
	if (__unlikely((__s32)uni < 0)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	bit = __BSR(uni);
	xby = (bit - 1) / 5;
	if (__unlikely(xby >= len)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	ptr = buf + xby;
	do {
		*ptr = (uni & 0x3f) ^ 0x80;
		uni >>= 6;
	} while (__unlikely(--ptr > buf));
	*ptr = uni | (-0x80 >> xby);
	return xby + 1;
}

static __const__ struct iconv_charset charset_utf8 = {
	get_char_utf8,
	put_char_utf8,
	NULL,
	NULL,
	NULL,
};


/***********/
/* UCS2-LE */
/***********/

static ssize_t get_char_ucs2le(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	if (__unlikely(len < 2)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	*uni = (unsigned char)buf[0] + ((unsigned char)buf[1] << 8);
	return 2;
}

static ssize_t put_char_ucs2le(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely(len < 2)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	if (__unlikely((__u32)uni >= 0x10000)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	buf[0] = uni;
	buf[1] = uni >> 8;
	return 2;
}

static __const__ struct iconv_charset charset_ucs2le = {
	get_char_ucs2le,
	put_char_ucs2le,
	NULL,
	NULL,
	NULL,
};


/***********/
/* UCS2-BE */
/***********/

static ssize_t get_char_ucs2be(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	if (__unlikely(len < 2)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	*uni = ((unsigned char)buf[0] << 8) + (unsigned char)buf[1];
	return 2;
}

static ssize_t put_char_ucs2be(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely(len < 2)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	if (__unlikely((__u32)uni >= 0x10000)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	buf[1] = uni;
	buf[0] = uni >> 8;
	return 2;
}

static __const__ struct iconv_charset charset_ucs2be = {
	get_char_ucs2be,
	put_char_ucs2be,
	NULL,
	NULL,
	NULL,
};


/********/
/* UCS2 */
/********/

struct ucs2_state {
	char le;
};

static ssize_t get_char_ucs2(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	struct ucs2_state *st = state;
	__s32 u;
	if (__unlikely(len < 2)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	if (__unlikely(st->le)) u = (unsigned char)buf[0] + ((unsigned char)buf[1] << 8);
	else u = ((unsigned char)buf[0] << 8) + (unsigned char)buf[1];
	if (__unlikely(u == 0xfeff)) u = -1;
	else if (__unlikely(u == 0xfffe)) u = -1, st->le ^= 1;
	*uni = u;
	return 2;
}

static ssize_t put_char_ucs2(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely(uni == 0xfeff) || __unlikely(uni == 0xfffe)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	return put_char_ucs2be(buf, len, uni, state);
}

static void *init_state_ucs2(void *ex)
{
	struct ucs2_state *st = __sync_malloc(sizeof(struct ucs2_state));
	if (__unlikely(!st)) return NULL;
	st->le = 0;
	return st;
}

static void free_state_ucs2(void *state)
{
	free(state);
}

static __const__ struct iconv_charset charset_ucs2 = {
	get_char_ucs2,
	put_char_ucs2,
	NULL,
	init_state_ucs2,
	free_state_ucs2,
};


/***********/
/* UCS4-LE */
/***********/

static ssize_t get_char_ucs4le(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	if (__unlikely(len < 4)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	*uni = (unsigned char)buf[0] + ((unsigned char)buf[1] << 8) + ((unsigned char)buf[2] << 16) + ((unsigned char)buf[3] << 24);
	return 4;
}

static ssize_t put_char_ucs4le(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely(len < 4)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	buf[0] = uni;
	buf[1] = uni >> 8;
	buf[2] = uni >> 16;
	buf[3] = uni >> 24;
	return 4;
}

static __const__ struct iconv_charset charset_ucs4le = {
	get_char_ucs4le,
	put_char_ucs4le,
	NULL,
	NULL,
	NULL,
};

/***********/
/* UCS4-BE */
/***********/

static ssize_t get_char_ucs4be(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	if (__unlikely(len < 4)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	*uni = ((unsigned char)buf[0] << 24) + ((unsigned char)buf[1] << 16) + ((unsigned char)buf[2] << 8) + (unsigned char)buf[3];
	return 4;
}

static ssize_t put_char_ucs4be(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely(len < 4)) {
		errno = E2BIG;
		return ICONV_ERROR;
	}
	buf[3] = uni;
	buf[2] = uni >> 8;
	buf[1] = uni >> 16;
	buf[0] = uni >> 24;
	return 4;
}

static __const__ struct iconv_charset charset_ucs4be = {
	get_char_ucs4be,
	put_char_ucs4be,
	NULL,
	NULL,
	NULL,
};

/********/
/* UCS4 */
/********/

struct ucs4_state {
	char le;
};

static ssize_t get_char_ucs4(__const__ char *buf, size_t len, __s32 *uni, void *state)
{
	struct ucs4_state *st = state;
	__s32 u;
	if (__unlikely(len < 4)) {
		errno = EINVAL;
		return ICONV_ERROR;
	}
	if (__unlikely(st->le)) u = (unsigned char)buf[0] + ((unsigned char)buf[1] << 8) + ((unsigned char)buf[2] << 16) + ((unsigned char)buf[3] << 24);
	else u = ((unsigned char)buf[0] << 24) + ((unsigned char)buf[1] << 16) + ((unsigned char)buf[2] << 8) + (unsigned char)buf[3];
	if (__unlikely((__u32)u == 0xfeffU)) u = -1;
	else if (__unlikely((__u32)u == 0xfffe0000U)) u = -1, st->le ^= 1;
	else if (__unlikely((__s32)u < 0)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	*uni = u;
	return 4;
}

static ssize_t put_char_ucs4(char *buf, size_t len, __s32 uni, void *state)
{
	if (__unlikely((__u32)uni == 0xfeffU) || __unlikely((__s32)uni < 0)) {
		errno = EILSEQ;
		return ICONV_ERROR;
	}
	return put_char_ucs4be(buf, len, uni, state);
}

static void *init_state_ucs4(void *ex)
{
	struct ucs4_state *st = __sync_malloc(sizeof(struct ucs4_state));
	if (__unlikely(!st)) return NULL;
	st->le = 0;
	return st;
}

static void free_state_ucs4(void *state)
{
	free(state);
}

static __const__ struct iconv_charset charset_ucs4 = {
	get_char_ucs4,
	put_char_ucs4,
	NULL,
	init_state_ucs4,
	free_state_ucs4,
};



__const__ struct iconv_charset *ICONV_GET_CHARSET(__const__ char *name, void **ex)
{
	if (!strcmp(name, "UTF-8")) return &charset_utf8;
	if (!strcmp(name, "UCS-2LE")) return &charset_ucs2le;
	if (!strcmp(name, "UCS-2BE")) return &charset_ucs2be;
	if (!strcmp(name, "UCS-2")) return &charset_ucs2;
	if (!strcmp(name, "UCS-4LE")) return &charset_ucs4le;
	if (!strcmp(name, "UCS-4BE")) return &charset_ucs4be;
	if (!strcmp(name, "UCS-4")) return &charset_ucs4;
	return NULL;
}

void CHARSET$LIST_SPECIAL(void (*process)(const char *name, void *cookie), void *cookie)
{
	process("UTF-8", cookie);
	process("UCS-2LE", cookie);
	process("UCS-2BE", cookie);
	process("UCS-2", cookie);
	process("UCS-4LE", cookie);
	process("UCS-4BE", cookie);
	process("UCS-4", cookie);
}
