#include <SYS/TYPES.H>
#include <SYS/STAT.H>
#include <UNISTD.H>
#include <STDLIB.H>
#include <FCNTL.H>
#include <TIME.H>
#include <VALUES.H>
#include <ERRNO.H>
#include <SYS/MMAN.H>
#include <ARCH/PAGE.H>
#include <KERNEL/VM.H>
#include <KERNEL/VMDEF.H>

#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>

#include <SYS/IPC.H>
#include <SYS/SHM.H>
#include <LIB/KERNEL/IPC.H>

static DECL_XLIST(allocated_shms);

struct shm {
	LIST_ENTRY list;
	char name[1];
};

/*
	SYS$IPC:/SHM_6789ABCD
*/

int shmget(key_t key, size_t size, int flags)
{
	char string[30];
	int oflags, h;
	struct stat st;
	if (key == IPC_PRIVATE) {
		int r;
		again:
		key = arc4random() & MAXINT;
		if (__unlikely(key == IPC_PRIVATE)) {
			goto again;
		}
		if (__likely((r = shmget(key, size, IPC_CREAT | IPC_EXCL | flags)) != -1)) return r;
		if (__unlikely(errno == EEXIST)) goto again;
		return -1;
	}
	if (__unlikely(!size)) {
		errno = EINVAL;
		return -1;
	}
	_snprintf(string, sizeof string, "SYS$IPC:/SHM_%08X", (int)key);
	oflags = O_RDWR;
	if (flags & IPC_CREAT) {
		oflags |= O_CREAT;
		if (flags & IPC_EXCL) oflags |= O_EXCL;
	}
	h = open(string, oflags, flags & 0777);
	if (__unlikely(h == -1)) {
		if (errno == EACCES) {
			if (__likely((h = open(string, O_RDONLY)) != -1)) goto op;
		}
		return -1;
	}
		/* !!! TODO: flock file (when it'll be implemented ...) */
	op:
	if (__unlikely(fstat(h, &st))) {
		close(h);
		return -1;
	}
	if (st.st_size < size) {
		struct shm *shm;
		if (__unlikely(st.st_size != 0)) {
			close(h);
			errno = EINVAL;
			return -1;
		}
		if (__unlikely(!(shm = malloc(sizeof(struct shm) + sizeof string + 1)))) {
			close(h);
			unlink(string);
			return -1;
		}
		strcpy(shm->name, string);
		ADD_TO_XLIST(&allocated_shms, &shm->list);
		if (__unlikely(ftruncate(h, size))) {
			close(h);
			unlink(string);
			return -1;
		}
	}
	close(h);
	return key;
}

void *shmat(int id, const void *addr, int flags)
{
	char string[30];
	int h;
	struct stat st;
	void *p;
	int spl;
	if (flags & SHM_RND) addr = (const void *)((unsigned long)addr & ~(unsigned long)(SHMLBA - 1));
	_snprintf(string, sizeof string, "SYS$IPC:/SHM_%08X", id);
	if (__unlikely((h = open(string, O_RDWR)) == -1)) {
		if (errno == ENOENT) errno = EINVAL;
		return (void *)-1L;
	}
	if (__unlikely(fstat(h, &st))) {
		close(h);
		return (void *)-1L;
	}
	if (__unlikely(!st.st_size)) {
		errno = EINVAL;
		close(h);
		return (void *)-1L;
	}
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_MALLOC), KERNEL$SPL)))
		KERNEL$SUICIDE("shmat AT SPL %08X", KERNEL$SPL);
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_MALLOC);
	if (__unlikely(!USER_SIZE_DESCS) && __unlikely(__IS_ERR(p = MAKE_USER_SIZES(0)))) {
		LOWER_SPLX(spl);
		close(h);
		errno = -__PTR_ERR(p);
		return (void *)-1L;
	}
	LOWER_SPLX(spl);
	if (__unlikely((p = mmap((void *)addr, st.st_size, __unlikely(flags & SHM_RDONLY) ? PROT_READ : PROT_READ | PROT_WRITE, __likely(!addr) ? MAP_SHARED | _MAP_NOFAULT : MAP_SHARED | _MAP_NOFAULT | MAP_FIXED | _MAP_NOUNMAP, h, 0)) == MAP_FAILED)) {
		close(h);
		return (void *)-1L;
	}
	close(h);
	USER_SIZE_DESCS[(unsigned long)p / PAGE_CLUSTER_SIZE].size = st.st_size;
	return p;
}

int shmdt(const void *addr)
{
	size_t size;
	if (__unlikely((int)(unsigned long)addr & (PAGE_CLUSTER_SIZE - 1))) {
		einval:
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(!USER_SIZE_DESCS)) goto einval;
	size = USER_SIZE_DESCS[(unsigned long)addr / PAGE_CLUSTER_SIZE].size;
	if (__unlikely(!size)) goto einval;
	return munmap((void *)addr, size);
}

int shmctl(int id, int cmd, struct shmid_ds *buf)
{
	struct stat st;
	char string[30];
	_snprintf(string, sizeof string, "SYS$IPC:/SHM_%08X", id);
	if (__unlikely(stat(string, &st))) {
		if (errno == ENOENT) errno = EINVAL;
		return -1;
	}
	switch (cmd) {
		case IPC_STAT:
			buf->shm_perm.cuid = buf->shm_perm.uid = getuid();
			buf->shm_perm.cgid = buf->shm_perm.gid = getgid();
			buf->shm_perm.mode = st.st_mode & 0777;
			buf->shm_segsz = st.st_size;
			buf->shm_cpid = buf->shm_lpid = getpid();
			buf->shm_nattch = 2;
			buf->shm_ctime = buf->shm_dtime = time(&buf->shm_atime);
			return 0;
		case IPC_SET:
			return 0;
		case IPC_RMID:
			return 0;
		default:
			errno = EINVAL;
			return -1;
	}
}

void SHM_FORK(void)
{
	while (__unlikely(!XLIST_EMPTY(&allocated_shms))) {
		struct shm *shm = LIST_STRUCT(allocated_shms.next, struct shm, list);
		DEL_FROM_LIST(&shm->list);
		free(shm);
	}
}

void SHM_CLEANUP(void)
{
	while (__unlikely(!XLIST_EMPTY(&allocated_shms))) {
		struct shm *shm = LIST_STRUCT(allocated_shms.next, struct shm, list);
		unlink(shm->name);
		DEL_FROM_LIST(&shm->list);
		free(shm);
	}
}
