#include <LIB/KERNEL/UASM.H>
#include <SYS/MMAN.H>
#include <SPAD/WQ.H>
#include <STDLIB.H>
#include <FCNTL.H>
#include <UNISTD.H>
#include <KERNEL/LINK.H>
#include <SPAD/SYNC.H>
#include <KERNEL/SYSCALL.H>
#include <LIB/KERNEL/UTHREAD.H>
#include <KERNEL/FDE.H>
#include <KERNEL/FIXUP.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SWAPPER.H>

#include <SPAD/LINK.H>

#include <SPAD/DL.H>
#include <DLFCN.H>

#define LIB_DIR		"LIB.:/"

/* Use no more than 1/4 of VM space for linking */
#define LINK_VM_SPACE_PORTION	4
#define DIRECT_IO_THRESHOLD	(PAGE_CLUSTER_SIZE)

struct library_description {
	LIST_ENTRY list;
	void *address;
	unsigned long code_size;
	unsigned long code_rodata_size;
	unsigned long code_rodata_data_size;
	unsigned long size;
	int refcount;
	int type;
	unsigned n_dependencies;
	struct library_description **dependencies;
	__u8 export_hash[REL_DIGEST_LENGTH];
	char *filename;
	char modname[1];
};

/* other DLT_ flags are in INCLUDE/SPAD/DL.H */
#define DLT_SHARED		0x10000
#define DLT_NOLIBPATH		0x20000

#define DL_LOCK_SPL	do { if (__unlikely(SPLX_BELOW(SPL_X(SPL_DL), spl = KERNEL$SPL))) KERNEL$SUICIDE("DL_LOCK_SPL (%d) AT SPL %08X", __LINE__, KERNEL$SPL); RAISE_SPL(SPL_DL); MTX_LOCK_SYNC(&udata.dl_mtx); } while (0)
#define DL_UNLOCK_SPL	do { MTX_UNLOCK(&udata.dl_mtx); LOWER_SPLX(spl); } while (0)

#define DL_LOCK		do { if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO))) KERNEL$SUICIDE("DL_LOCK (%d) AT SPL %08X", __LINE__, KERNEL$SPL); RAISE_SPL(SPL_DL); MTX_LOCK_SYNC(&udata.dl_mtx); } while (0)
#define DL_UNLOCK	do { MTX_UNLOCK(&udata.dl_mtx); LOWER_SPL(SPL_ZERO); } while (0)

void __DL_USER_INIT(int mprot)
{
	struct library_description *kernel;
	struct section *sec = (struct section *)((char *)&kernel_dll_lh + kernel_dll_lh.sections);
	unsigned long rs, ds, be;
	if (__unlikely(sec[SECTION_DATA].len > sizeof(udata.kernel_dll_data))) {
		while (1) SYSCALL3(SYS_EXIT, -EINVAL, (unsigned long)"TOO LONG DATA SECTION");
	}
	if (mprot == 1) {
		memcpy(udata.kernel_dll_data, (void *)sec[SECTION_DATA].ptr, sec[SECTION_DATA].len);
	} else {
		memcpy((void *)sec[SECTION_DATA].ptr, udata.kernel_dll_data, sec[SECTION_DATA].len);
		memset((void *)sec[SECTION_BSS].ptr, 0, sec[SECTION_BSS].len);
	}
	rs = (unsigned long)sec[SECTION_RODATA].ptr & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	ds = (unsigned long)sec[SECTION_DATA].ptr & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	/*de = (unsigned long)sec[SECTION_BSS].ptr & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);*/
	be = ((unsigned long)sec[SECTION_BSS].ptr + sec[SECTION_BSS].len + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	INIT_XLIST(&udata.dl_modules);
	kernel = malloc(sizeof(struct library_description) + strlen("KERNEL"));
	if (__unlikely(!kernel)) KERNEL$SUICIDE("__DL_USER_INIT: CAN'T ALLOCATE KERNEL LIBRARY DESCRIPTION");
	kernel->address = &kernel_dll_lh;
	kernel->code_size = ds - (unsigned long)&kernel_dll_lh;
	kernel->code_rodata_size = rs - (unsigned long)&kernel_dll_lh;
	kernel->size = be - (unsigned long)&kernel_dll_lh;
	kernel->refcount = 1;
	kernel->type = DLT_EXPORT | DLT_SHARED;
	kernel->n_dependencies = 0;
	kernel->dependencies = NULL;
	memcpy(kernel->export_hash, udata.rel_digest, REL_DIGEST_LENGTH);
	kernel->filename = "";
	strcpy(kernel->modname, "KERNEL");
	ADD_TO_XLIST(&udata.dl_modules, &kernel->list);
	MTX_INIT(&udata.dl_mtx, "KERNEL$DL_MTX");
}

static struct library_description *find_ld_for_address(void *addr)
{
	struct library_description *ld;
	int spl;
	DL_LOCK_SPL;
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		if ((char *)addr >= (char *)ld->address && (char *)addr < (char *)ld->address + ld->size) {
			ret_ld:
			DL_UNLOCK_SPL;
			return ld;
		}
	}
	ld = NULL;
	goto ret_ld;
}

static int test_change(int h, __u8 link_hash_1[LINK_HASH_LENGTH]);
static void unload_module(struct library_description *ld, int deinit);
static void free_dependencies(struct library_description **dependencies, unsigned n_dependencies);
static struct library_description *load_module_continue(void *head, struct library_description *ld, int h, __u8 link_hash_1[LINK_HASH_LENGTH], char error[__MAX_STR_LEN]);
static void ld_make_error(char error[__MAX_STR_LEN], IOCTLRQ *io, char *msg);

static struct library_description *load_module(int type, __const__ char *filename, __const__ char *modname, char *error)
{
	char *file;
	int h, r, r1;
	unsigned i;
	unsigned rcount = 0;
	struct library_description *ld;
	unsigned hsize;
	void *head;
	unsigned head_allocated = 0;
	unsigned n_dependencies;
	struct library_description **dependencies;
	char *export;
	int sh;
	union {
		struct link_header lh;
		char c[0x1c0];
		struct {
			__f_off l_l;
			__f_off l_l_2;
		} s;
	} u;
	LD_QUERY ldq;
	IOCTLRQ io;
	head = &u;
	head_allocated = sizeof u;
	if (__unlikely(error[0])) error[0] = 0;
	if (__likely(!filename) && __unlikely(!modname)) {
		errno = EINVAL;
		_snprintf(error, __MAX_STR_LEN, "NO FILE NAME OR MODULE NAME");
		return NULL;
	}
	if (__likely(!filename)) {
		file = alloca(strlen(LIB_DIR) + strlen(modname) + 4 + 1);
		memcpy(stpcpy(mempcpy(file, LIB_DIR, strlen(LIB_DIR)), modname), ".DLL", 5);
	} else if (!(type & DLT_NOLIBPATH) && !strpbrk(filename, ":/")) {
		file = alloca(strlen(LIB_DIR) + strlen(filename) + 1);
		stpcpy(mempcpy(file, LIB_DIR, strlen(LIB_DIR)), filename);
	} else {
		file = (char *)filename;
	}
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		if ((__likely(!filename) || !_strcasecmp(ld->filename, file)) && (__unlikely(!modname) || !_strcasecmp(ld->modname, modname)) && ld->type & DLT_SHARED && !(!(ld->type & DLT_EXPORT) && type & DLT_EXPORT)) {
			ld->refcount++;
			return ld;
		}
	}
	do_open:
	h = open(file, O_RDONLY | _O_NOPOSIX);
	if (__unlikely(h < 0)) {
		_snprintf(error, __MAX_STR_LEN, "OPEN: %s (%s)", strerror(errno), file);
		return NULL;
	}
	restart__changed:
	r1 = pread(h, &ldq.h.hash, LINK_HASH_LENGTH, __offsetof(struct link_header, hash));
	if (__unlikely(r1 < 0)) goto ioerror;
	if (__likely((r = pread(h, &u, sizeof u, 0)) < sizeof(struct link_header))) {
		if (__unlikely(r < 0)) {
			ioerror:
			_snprintf(error, __MAX_STR_LEN, "READ: %s (%s)", strerror(errno), KERNEL$HANDLE_PATH(h));
			cl_ret:
			close(h);
			return NULL;
		}
		if (__unlikely(r < 2) || __unlikely(memchr(u.c, '\n', r) != &u.c[r - 1]) || __unlikely(memchr(u.c, 0, r) != 0)) {
			badf:
			_snprintf(error, __MAX_STR_LEN, "NOT AN EXECUTABLE FILE");
			errno = ENOEXEC;
			goto cl_ret;
		}
		r--;
		u.c[r] = 0;
		i = _is_absolute(u.c);
		if (__unlikely(i == _ABS_LOGREL) || __unlikely(i == _ABS_CURROOT)) goto badf;
		if (__unlikely(++rcount > LINK_MAX_RCOUNT)) {
			_snprintf(error, __MAX_STR_LEN, "TOO MANY REFERENCES");
			errno = ELOOP;
			goto cl_ret;
		}
		if (__unlikely(i == _ABS_TOTAL)) {
			close(h);
			file = alloca(r + 1);
			memcpy(file, u.c, r + 1);
			goto do_open;
		} else {
			char *f = KERNEL$HANDLE_PATH(h);
			char *ff = strrchr(f, '/');
			int sl;
			if (__unlikely(!ff)) KERNEL$SUICIDE("load_module: NO SLASH IN \"%s\"", f);
			sl = ff - f;
			file = alloca(sl + r + 2);
			memcpy(mempcpy(file, f, sl + 1), u.c, r + 1);
			close(h);
			goto do_open;
		}
	}
	if (__unlikely(r1 != LINK_HASH_LENGTH)) {
		trunc:
		_snprintf(error, __MAX_STR_LEN, "TRUNCATED FILE");
		errno = EFTYPE;
		goto cl_ret;
	}
	if (__unlikely(LINK_CHECK_HEADER(&u.lh, &hsize, &n_dependencies, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, error))) {
		errno = ENOEXEC;
		goto cl_ret;
	}
	if (__likely(hsize <= sizeof(u))) {
		if (__unlikely(head != &u)) memcpy(head, &u, sizeof u);
	} else {
		if (__likely(hsize > head_allocated)) head = alloca(head_allocated = hsize);
		memcpy(head, &u, sizeof u);
		if (__unlikely((r = pread(h, (char *)head + sizeof u, hsize - sizeof(u), sizeof(u))) != hsize - sizeof(u))) {
			if (r < 0) goto ioerror;
			goto trunc;
		}
	}
	if (__unlikely((r = test_change(h, ldq.h.hash)))) {
		if (r < 0) goto ioerror;
		if (r == 2) goto trunc;
		goto restart__changed;
	}
	dependencies = calloc(n_dependencies, sizeof(struct library_description *));
	if (__unlikely(!dependencies)) {
		_snprintf(error, __MAX_STR_LEN, "OUT OF MEMORY FOR LIBRARY DEPENDENCIES");
		goto cl_ret;
	}
	for (i = 0; i < n_dependencies; i++) {
		char *depname;
		unsigned depflags;
		if (__unlikely(LINK_GET_DEPENDENCY(head, i, &depname, &depflags, error))) {
			errno = EFTYPE;
			free_dep_cl_ret:
			free_dependencies(dependencies, n_dependencies);
			goto cl_ret;
		}
		if (__likely(!(depflags & LIB_LAZY)) && __unlikely(!(dependencies[i] = load_module(DLT_EXPORT, NULL, depname, error)))) {
			if (__likely(strlen(depname) + strlen(error) <= __MAX_STR_LEN - 3)) {
				memmove(error + strlen(depname) + 2, error, strlen(error) + 1);
				memcpy(stpcpy(error, depname), ": ", 2);
			}
			goto free_dep_cl_ret;
		}
	}
	
	if (__unlikely(LINK_GET_MODULE_EXPORT_NAME(head, &export, &sh, error))) {
		errno = EFTYPE;
		goto free_dep_cl_ret;
	}
	if (modname && __unlikely(_strcasecmp(export, modname))) {
		_snprintf(error, __MAX_STR_LEN, "DLL INTERFACE NOT MATCHING (WANTED %s, GOT %s)", modname, export);
		errno = EINVAL;
		goto free_dep_cl_ret;
	}
	ld = malloc(sizeof(struct library_description) + strlen(export) + strlen(file) + 1);
	if (__unlikely(!ld)) {
		_snprintf(error, __MAX_STR_LEN, "OUT OF MEMORY FOR LIBRARY DESCRIPTION");
		goto free_dep_cl_ret;
	}
	ld->dependencies = dependencies;
	ld->n_dependencies = n_dependencies;
	ld->refcount = 1;
	ld->type = type | (sh ? DLT_SHARED : 0);
	ld->filename = stpcpy(ld->modname, export) + 1;
	strcpy(ld->filename, file);

	if (__unlikely(LINK_GET_SECTIONS_SIZE(head, &ld->code_size, &ld->code_rodata_size, &ld->code_rodata_data_size, &ld->size, error))) {
		errno = EFTYPE;
		free_ld_dep_cl_ret:
		__slow_free(ld);
		goto free_dep_cl_ret;
	}
	ld->code_rodata_size += ld->code_size;
	ld->code_rodata_data_size += ld->code_rodata_size;
	ld->size += ld->code_rodata_data_size;

	memset(ldq.h.hash + LINK_HASH_LENGTH, 0, REL_DIGEST_LENGTH);
	for (i = 0; i < ld->n_dependencies; i++) {
		unsigned j;
		struct library_description *ldd = ld->dependencies[i];
		if (__unlikely(!ldd)) continue;
			/*__debug_printf("xor(%d): ", i);*/
		for (j = 0; j < REL_DIGEST_LENGTH; j++) {
			ldq.h.hash[LINK_HASH_LENGTH + j] ^= ldd->export_hash[j];
			/*__debug_printf("%02x", ldq.h.hash[LINK_HASH_LENGTH + j]);*/
			/*__debug_printf("%02x", ldd->export_hash[j]);*/
		}
			/*__debug_printf("\n");*/
	}

	ldq.addr = 0;
	io.h = SWAPPER_HANDLE;
	io.v.ptr = (unsigned long)&ldq;
	io.v.len = sizeof ldq;
	io.v.vspace = &KERNEL$VIRTUAL;
	io.ioctl = IOCTL_SWAPPER_LDCACHE_QUERY;
	io.param = 0;
	next_addr:
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__likely(!io.status)) {
		ld->address = mmap((void *)ldq.addr, ld->size, PROT_READ | PROT_WRITE | PROT_EXEC, MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);
		if (__unlikely(ld->address == MAP_FAILED)) {
			goto free_ld_dep_cl_ret;
		}
		if (__unlikely((unsigned long)ld->address != ldq.addr)) {
			munmap(ld->address, ld->size);
			if (__unlikely((unsigned long)ld->address < ldq.addr)) {
				goto skip_attach;
			}
			ldq.addr = (unsigned long)ld->address;
			goto next_addr;
		}
		if (__unlikely((unsigned long)ld->address + ld->size > (unsigned long)udata.base / LINK_VM_SPACE_PORTION)) {
				/* test if there is space below */
			void *p = mmap(NULL, ld->size, PROT_READ | PROT_WRITE | PROT_EXEC, MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);
			if (__likely(p != MAP_FAILED)) {
				munmap(p, ld->size);
				if ((unsigned long)p + ld->size <= (unsigned long)udata.base / LINK_VM_SPACE_PORTION) {
					/* if there is, don't share */
					munmap(ld->address, ld->size);
					goto skip_attach;
				}
			}
		}
		io.ioctl = IOCTL_SWAPPER_LDCACHE_ATTACH;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__unlikely(io.status != 0)) {
			munmap(ld->address, ld->size);
			/*__debug_printf("not attached: %d\n", io.status);*/
			goto skip_attach;
		}
		memcpy(ld->export_hash, ldq.h.hash + LINK_HASH_LENGTH, REL_DIGEST_LENGTH);
		/*__debug_printf("attached: %s\n", ld->modname);
		{
			unsigned i;
			for (i = 0; i < sizeof ldq.h.hash; i++)
				__debug_printf("%02x", ldq.h.hash[i]);
			__debug_printf("\n");
		}*/
		close(h);
		goto attached;
	}
	/*__debug_printf("notfound: %s\n", ld->modname);*/

	skip_attach:

	ld = load_module_continue(head, ld, h, ldq.h.hash, error);
	if (__unlikely((void *)ld <= (void *)1)) {
		if (!ld) return NULL;
		goto restart__changed;
	}

	attached:

	ADD_TO_XLIST(&udata.dl_modules, &ld->list);
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$HANDLE", SYM_SPECIAL, &u.s.l_l))) {
		*(void **)u.s.l_l = ld;
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$FIXUP", SYM_SPECIAL, &u.s.l_l)) && __unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$FIXUP_END", SYM_SPECIAL, &u.s.l_l_2))) {
		FIXUP_FEATURES((struct feature_fixup *)u.s.l_l, (struct feature_fixup *)u.s.l_l_2);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$LOAD", SYM_SPECIAL, &u.s.l_l))) {
		DLINITRQ *init = alloca(sizeof(DLINITRQ));
		init->error = error;
		SYNC_IO(init, (IO_STUB *)u.s.l_l);
		if (__unlikely(init->status < 0)) {
			unload_module(ld, 0);
			errno = -init->status;
			return NULL;
		}
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "_init", SYM_SPECIAL, &u.s.l_l))) {
		LOWER_SPL(SPL_ZERO);
		((void (*)(void))u.s.l_l)();
		RAISE_SPL(SPL_DL);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$CTORS", SYM_SPECIAL, &u.s.l_l)) && __unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$CTORS_END", SYM_SPECIAL, &u.s.l_l_2))) {
	 	LOWER_SPL(SPL_ZERO);
		while (u.s.l_l < u.s.l_l_2) {
			(*(void (**)(void))u.s.l_l)();
			u.s.l_l += sizeof(void (*)(void));
		}
		RAISE_SPL(SPL_DL);
	}

	return ld;
}

static int test_change(int h, __u8 link_hash_1[LINK_HASH_LENGTH])
{
	int r;
	static __u8 link_hash_2[LINK_HASH_LENGTH];
	if (__unlikely((r = pread(h, link_hash_2, LINK_HASH_LENGTH, __offsetof(struct link_header, hash))) != LINK_HASH_LENGTH)) {
		if (r >= 0) return 2;
		return -1;
	}
	return !!memcmp(link_hash_1, link_hash_2, LINK_HASH_LENGTH);
}

/* return: NULL --- error
	   (void *)1 --- file changed
	   other --- valid link
   leaves h open only if it returns (void *)1
*/

static struct library_description *load_module_continue(void *head, struct library_description *ld, int h, __u8 link_hash_1[LINK_HASH_LENGTH], char error[__MAX_STR_LEN])
{
	unsigned i;
	unsigned long compressed_file_size;
	unsigned compression_method;
	unsigned hash_method;
	ssize_t rs;
	int r;
	struct library_description *ldd;
	IOCTLRQ io;
	union {
		LD_PREPARE ldp;
		LD_SET_IFACE ldsi;
	} u;
	LD_SET_SYMBOLS *sympg;
	if (__unlikely(LINK_CHECK_HEADER(head, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, &hash_method, &compression_method, &compressed_file_size, error))) {
		errno = EFTYPE;
		goto cl_ret;
	}
	if (__unlikely(compression_method)) {
		_snprintf(error, __MAX_STR_LEN, "COMPRESSION NOT SUPPORTED");
		errno = ENOSYS;
		goto cl_ret;
	}
	ld->address = mmap(NULL, ld->size, PROT_READ | PROT_WRITE | PROT_EXEC, MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);
	if (__unlikely(ld->address == MAP_FAILED)) {
		_snprintf(error, __MAX_STR_LEN, "CAN'T ALLOCATE MEMORY: %s", strerror(errno));
		cl_ret:
		close(h);
		free_dependencies(ld->dependencies, ld->n_dependencies);
		__slow_free(ld);
		return NULL;
	}
	if (__unlikely(ld->code_rodata_data_size >= DIRECT_IO_THRESHOLD)) {
		fcntl(h, F_SETFL, O_DIRECT);
	}
	if (__unlikely((rs = pread(h, ld->address, ld->code_rodata_data_size, 0)) != ld->code_rodata_data_size)) {
		if (rs >= 0) {
			r = test_change(h, link_hash_1);
			if (r == 1) {
				restart__change:
				munmap(ld->address, ld->size);
				free_dependencies(ld->dependencies, ld->n_dependencies);
				__slow_free(ld);
				return (void *)1;
			}
			if (r >= 0) {
				trunc:
				_snprintf(error, __MAX_STR_LEN, "TRUNCATED FILE");
				eftype_unmap_cl_ret:
				errno = EFTYPE;
				goto unmap_cl_ret;
			}
		}
		ioerror:
		_snprintf(error, __MAX_STR_LEN, "READ: %s (%s)", strerror(errno), KERNEL$HANDLE_PATH(h));
		unmap_cl_ret:
		munmap(ld->address, ld->size);
		goto cl_ret;
	}
	if (__unlikely((r = test_change(h, link_hash_1)))) {
		if (r < 0) goto ioerror;
		if (r == 2) goto trunc;
		goto restart__change;
	}
	if (__unlikely(rs != ld->code_rodata_data_size)) {
		_snprintf(error, __MAX_STR_LEN, "TRUNCATED FILE");
		goto eftype_unmap_cl_ret;
	}

	if (__unlikely(hash_method == LINK_HASH_NONE)) goto ldcache_skip;

	/* Attempt to link using kernel */

	io.h = SWAPPER_HANDLE;
	io.v.ptr = (unsigned long)&u.ldp;
	io.v.len = sizeof u.ldp;
	io.v.vspace = &KERNEL$VIRTUAL;
	io.ioctl = IOCTL_SWAPPER_LDCACHE_PREPARE;
	io.param = 0;
	u.ldp.from = (unsigned long)ld->address;
	u.ldp.to = (u.ldp.from + ld->code_rodata_data_size + PAGE_CLUSTER_SIZE - 1) & ~(unsigned long)(PAGE_CLUSTER_SIZE - 1);
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (SWAPPER_STATUS_SKIP(io.status)) goto ldcache_skip;
		_snprintf(error, __MAX_STR_LEN, "CREATING LDCACHE FAILED: %s", strerror(-io.status));
		iostatus_unmap_cl_ret:
		errno = -io.status;
		goto unmap_cl_ret;
	}
	io.v.ptr = 0;
	io.v.len = 0;
	io.ioctl = IOCTL_SWAPPER_LDCACHE_CHECK;
	io.param = (unsigned long)ld->address;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (SWAPPER_STATUS_SKIP(io.status)) goto ldcache_skip;
		ld_make_error(error, &io, "VERIFYING LDCACHE FAILED");
		goto iostatus_unmap_cl_ret;
	}
	io.ioctl = IOCTL_SWAPPER_LDCACHE_RELOC;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (SWAPPER_STATUS_SKIP(io.status)) goto restart__change;
		ld_make_error(error, &io, "RELOCATING FAILED");
		goto iostatus_unmap_cl_ret;
	}

	sympg = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
	if (__unlikely(!sympg)) {
		errno = ENOMEM;
		goto unmap_cl_ret;
	}
	for (i = 0; i < ld->n_dependencies; i++) if (__likely((ldd = ld->dependencies[i]) != NULL)) {
		unsigned n_exported_syms;
		unsigned j;
		LD_SET_SYMBOLS_RECORD *lssr;
		io.v.ptr = (unsigned long)&u.ldsi;
		io.v.len = sizeof u.ldsi;
		io.ioctl = IOCTL_SWAPPER_LDCACHE_SET_IFACE;
		u.ldsi.iface = i;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__unlikely(io.status < 0)) {
			KERNEL$FREE_KERNEL_PAGE(sympg, VM_TYPE_WIRED_MAPPED);
			if (SWAPPER_STATUS_SKIP(io.status)) goto reloc_skip;
			ld_make_error(error, &io, "SETTING INTERFACE FAILED");
			goto iostatus_unmap_cl_ret;
		}
		if (__unlikely(LINK_GET_SYMBOL_COUNTS(ldd->address, &n_exported_syms, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, error))) {
			freepage_eftype_unmap_cl_ret:
			KERNEL$FREE_KERNEL_PAGE(sympg, VM_TYPE_WIRED_MAPPED);
			goto eftype_unmap_cl_ret;
		}
		io.v.ptr = (unsigned long)sympg;
		io.v.len = PAGE_CLUSTER_SIZE;
		io.ioctl = IOCTL_SWAPPER_LDCACHE_RELOC_SYM;
		sympg->start = __offsetof(LD_SET_SYMBOLS, data);
		for (j = 0; j < n_exported_syms; j++) {
			char *n;
			unsigned len;
			__f_off value;
			if (__unlikely(!(n = LINK_GET_SYMBOL_BY_INDEX(ldd->address, j, &value, error)))) {
				goto freepage_eftype_unmap_cl_ret;
			}
			len = strlen(n);
			if (__unlikely(len >= LINK_DYNAMIC_SYMBOL_NAME_LEN)) {
				_snprintf(error, __MAX_STR_LEN, "TOO LONG EXPORTED SYMBOL");
				goto freepage_eftype_unmap_cl_ret;
			}
			lssr = (LD_SET_SYMBOLS_RECORD *)((char *)sympg + sympg->start);
			if (__unlikely(sympg->start + LD_SET_SYMBOLS_RECORD_LEN(len) > PAGE_CLUSTER_SIZE - LD_SET_SYMBOLS_RECORD_LEN(0))) {
				lssr->namelen = 0;
				sympg->start = __offsetof(LD_SET_SYMBOLS, data);
				SYNC_IO(&io, KERNEL$IOCTL);
				if (__unlikely(io.status < 0)) {
					fixdep_fail:
					KERNEL$FREE_KERNEL_PAGE(sympg, VM_TYPE_WIRED_MAPPED);
					if (SWAPPER_STATUS_SKIP(io.status)) goto restart__change;
					ld_make_error(error, &io, "FIXING UP DEPENDENCIES FAILED");
					goto iostatus_unmap_cl_ret;
				}
				sympg->start = __offsetof(LD_SET_SYMBOLS, data);
				lssr = (LD_SET_SYMBOLS_RECORD *)((char *)sympg->data);
			}
			lssr->value = value;
			memcpy(lssr->name, n, len);
			lssr->namelen = len;
			sympg->start += LD_SET_SYMBOLS_RECORD_LEN(len);
		}
		if (__likely(sympg->start != __offsetof(LD_SET_SYMBOLS, data))) {
			lssr = (LD_SET_SYMBOLS_RECORD *)((char *)sympg + sympg->start);
			lssr->namelen = 0;
			sympg->start = __offsetof(LD_SET_SYMBOLS, data);
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status < 0)) goto fixdep_fail;
		}
	}
	KERNEL$FREE_KERNEL_PAGE(sympg, VM_TYPE_WIRED_MAPPED);
	io.v.ptr = (unsigned long)ld->export_hash;
	io.v.len = REL_DIGEST_LENGTH;
	io.ioctl = IOCTL_SWAPPER_LDCACHE_FINALIZE;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (SWAPPER_STATUS_SKIP(io.status)) goto restart__change;
		ld_make_error(error, &io, "FINALIZING LDCACHE FAILED");
		goto iostatus_unmap_cl_ret;
	}

	goto link_done;

	/* Linking using kernel failed ... */

	ldcache_skip:
	if (__unlikely(LINK_RELOC_MODULE(ld->address, NULL))) {
		_snprintf(error, __MAX_STR_LEN, "RELOC FAILED");
		errno = EFTYPE;
		goto unmap_cl_ret;
	}
	reloc_skip:

	for (i = 0; i < ld->n_dependencies; i++) if (__likely((ldd = ld->dependencies[i]) != NULL)) {
		if (__unlikely(LINK_FIXUP_DLL_DEPENDENCY(ld->address, i, ldd->address, error))) {
			errno = EINVAL;
			goto unmap_cl_ret;
		}
	}

	link_done:

	close(h);

	return ld;
}

static void ld_make_error(char error[__MAX_STR_LEN], IOCTLRQ *io, char *msg)
{
	LD_GET_ERROR ldg;
	int err = io->status;
	io->v.ptr = (unsigned long)&ldg;
	io->v.len = sizeof ldg;
	io->ioctl = IOCTL_SWAPPER_LDCACHE_GET_ERROR;
	SYNC_IO(io, KERNEL$IOCTL);
	if (io->status >= 0 && (err == -EIO || err == -ENOENT)) {
		strlcpy(error, ldg.error, __MAX_STR_LEN);
	} else {
		_snprintf(error, __MAX_STR_LEN, "%s: %s", msg, io->status >= 0 ? ldg.error : strerror(-err));
	}
	io->status = err;
}

static void free_dependencies(struct library_description **dependencies, unsigned n_dependencies)
{
	unsigned i;
	int e;
	e = errno;
	for (i = 0; i < n_dependencies; i++) {
		if (dependencies[i]) unload_module(dependencies[i], 1);
	}
	free(dependencies);
	errno = e;
}


static void call_deinit(struct library_description *ld)
{
	__f_off l_l, l_l_2;
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$DTORS", SYM_SPECIAL, &l_l)) && __unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$DTORS_END", SYM_SPECIAL, &l_l_2))) {
		LOWER_SPL(SPL_ZERO);
		while (l_l_2 > l_l) {
			l_l_2 -= sizeof(void (*)(void));
			(*(void (**)(void))l_l_2)();
		}
		RAISE_SPL(SPL_DL);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "_fini", SYM_SPECIAL, &l_l))) {
		LOWER_SPL(SPL_ZERO);
		((void (*)(void))l_l)();
		RAISE_SPL(SPL_DL);
	}
	if (__unlikely(!LINK_GET_SYMBOL(ld->address, "DLL$UNLOAD", SYM_SPECIAL, &l_l))) {
		char error[__MAX_STR_LEN];
		DLINITRQ init;
		init.error = error;
		SYNC_IO(&init, (IO_STUB *)l_l);
	}
}

static void unload_module(struct library_description *ld, int deinit)
{
	int i;
	if (__unlikely(--ld->refcount)) {
		if (__unlikely(ld->refcount < 0) || __unlikely(!deinit)) KERNEL$SUICIDE("unload_module: MODULE %s, REFCOUNT %d", ld->modname, ld->refcount);
		return;
	}
	if (__likely(deinit)) {
		call_deinit(ld);
	}
	for (i = 0; i < ld->n_dependencies; i++) {
		struct library_description *d;
		if (__likely((d = ld->dependencies[i]) != NULL)) {
			ld->dependencies[i] = NULL;
			unload_module(d, 1);
		}
	}
	/*__debug_printf("unmap: %p, %x, %x + %x + %x\n", ld, ld->address, ld->code_size, ld->data_size, ld->bss_size);
	sleep(5);*/
	munmap(ld->address, ld->size);
	DEL_FROM_LIST(&ld->list);
	free(ld);
}

void KERNEL$DL_CALL_DTORS(void)
{
	struct library_description *ld;
	DL_LOCK;
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		call_deinit(ld);
	}
	DL_UNLOCK;
}

static char dl_error[__MAX_STR_LEN + 1] = { 0 };

void *dlopen(__const__ char *filename, int flag)
{
	char error[__MAX_STR_LEN];
	struct library_description *ld;
	DL_LOCK;
	if (__unlikely(!filename)) {
		ld = udata.dl_exe;
		if (__unlikely(!ld)) strcpy(dl_error, "NO EXECUTABLE LOADED YET");
	} else {
		ld = load_module(flag & RTLD_GLOBAL ? DLT_EXPORT : 0, filename, NULL, error);
		if (__unlikely(!ld)) strcpy(dl_error, error);
	}
	DL_UNLOCK;
	return ld;
}

int dlclose(void *handle)
{
	DL_LOCK;
	unload_module(handle, 1);
	DL_UNLOCK;
	return 0;
}

__const__ char *dlerror(void)
{
	if (__likely(dl_error[0])) {
		memmove(dl_error + 1, dl_error, __MAX_STR_LEN - 1);
		dl_error[__MAX_STR_LEN] = 0;
		dl_error[0] = 0;
		return dl_error + 1;
	}
	return NULL;
}

void *__dlsym(void *handle, __const__ char *symbol, void *addr);

void *__dlsym(void *handle, __const__ char *symbol, void *addr)
{
	struct library_description *ld = handle;
	__f_off val;
	int spl;
	DL_LOCK_SPL;
	dl_error[0] = 0;
	if (__unlikely(!ld)) {
		ld = find_ld_for_address(addr);
		if (!ld) {
			_snprintf(dl_error, __MAX_STR_LEN, "UNKNOWN CALLING ADDRESS %p", addr);
			unlock_null:
			DL_UNLOCK_SPL;
			return NULL;
		}
	} else if (__unlikely(ld == RTLD_DEFAULT)) {
		XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
			if (!LINK_GET_SYMBOL(ld->address, symbol, SYM_INTERNAL, &val)) goto found;
		}
		goto not_found;
	}
	if (__unlikely(LINK_GET_SYMBOL(ld->address, symbol, SYM_INTERNAL, &val))) {
		not_found:
		_snprintf(dl_error, __MAX_STR_LEN, "SYMBOL %s NOT FOUND", symbol);
		goto unlock_null;
	}
	found:
	DL_UNLOCK_SPL;
	return (void *)val;
}

int dladdr(__const__ void *addr, Dl_info *info)
{
	struct library_description *ld;
	unsigned long offset;
	int spl;
	DL_LOCK_SPL;
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		char *name = LINK_GET_SYMBOL_NAME(ld->address, addr, &offset, 0);
		if (name) {
			info->dli_fname = ld->filename;
			info->dli_fbase = ld->address;
			info->dli_sname = name;
			info->dli_saddr = (char *)addr - offset;
			DL_UNLOCK_SPL;
			return 1;
		}
	}
	DL_UNLOCK_SPL;
	return 0;
}

static long do_dl(void *RQ_)
{
	struct library_description *ld;
	DLRQ *RQ = RQ_;
	DL_LOCK;
	errno = 0;
	ld = load_module(RQ->type, *RQ->filename ? RQ->filename : NULL, *RQ->modname ? RQ->modname : NULL, RQ->error);
	if (!ld) {
		RQ->status = -errno;
		if (__unlikely(!RQ->status))
			KERNEL$SUICIDE("do_dl: NO ERROR AND NULL RETURNED");
	} else {
		RQ->status = 0;
	}
	RQ->ld_handle = ld;
	DL_UNLOCK;
	return 0;
}

static long do_udl(void *RQ_)
{
	DLRQ *RQ = RQ_;
	DL_LOCK;
	unload_module(RQ->ld_handle, 1);
	DL_UNLOCK;
	RQ->status = 0;
	return 0;
}

int KERNEL$DL_GET_SYMBOL(DLRQ *d, char *name, unsigned long *val)
{
	__const__ char *e;
	void *s = dlsym(d->ld_handle, name);
	if (__unlikely((e = dlerror()) != NULL)) return -1;
	*val = (unsigned long)s;
	return 0;
}

__const__ char *KERNEL$DL_GET_SYMBOL_NAME(__const__ void *sym, unsigned long *offset, int codeonly)
{
	struct library_description *ld;
	unsigned long off;
	if (__unlikely(!offset)) offset = &off;
	if (__unlikely(!udata.dl_modules.next)) goto ret_null;
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		char *name = LINK_GET_SYMBOL_NAME(ld->address, sym, offset, codeonly);
		if (name) return name;
	}
	ret_null:
	*offset = (unsigned long)sym;
	return NULL;
}

static long do_dll(void *RQ_)
{
	struct library_description *ld;
	int i;
	DLLZRQ *RQ = RQ_;
	if (RQ->caller) {
		ld = find_ld_for_address(RQ->caller);
		if (__unlikely(!ld)) {
			RQ->status = -EINVAL;
			_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T FIND CALLER'S MODULE");
			return 0;
		}
	} else {
		if (RQ->handle) ld = RQ->handle->ld_handle;
		else ld = udata.dl_exe;
	}
	DL_LOCK;
	for (i = 0; i < ld->n_dependencies; i++) {
		char *dep;
		if (__unlikely(LINK_GET_DEPENDENCY(ld->address, i, &dep, (unsigned *)(void *)&KERNEL$LIST_END, RQ->error))) {
			RQ->status = -EFTYPE;
			goto done;
		}
		if (!_strcasecmp(dep, RQ->modname)) {
			if (ld->dependencies[i]) {
				RQ->status = 0;
				goto done;
			}
			if (__unlikely(!(ld->dependencies[i] = load_module(DLT_EXPORT, NULL, dep, RQ->error)))) {
				RQ->status = -errno;
				goto done;
			}
			if (__unlikely(LINK_FIXUP_DLL_DEPENDENCY(ld->address, i, ld->dependencies[i]->address, RQ->error))) {
				RQ->status = -errno;
				unload_module(ld->dependencies[i], 1);
				ld->dependencies[i] = NULL;
				goto done;
			}
			RQ->status = 0;
			goto done;
		}
	}
	RQ->status = -ENOENT;
	_snprintf(RQ->error, __MAX_STR_LEN, "NO SUCH MODULE");
	done:
	DL_UNLOCK;
	return 0;
}

static void dl_sync(IORQ *rq, long (*fn)(void *));
extern AST_STUB dl_sync_done;

DECL_IOCALL(KERNEL$DL_LOAD_MODULE, SPL_BOTTOM, DLRQ)
{
	dl_sync((IORQ *)RQ, do_dl);
	RETURN;
}

DECL_IOCALL(KERNEL$DL_UNLOAD_MODULE, SPL_BOTTOM, DLRQ)
{
	if (__unlikely(!RQ->ld_handle)) KERNEL$SUICIDE("KERNEL$DL_UNLOAD_MODULE: UNLOADING EMPTY MODULE %s (%s)", RQ->modname, RQ->filename);
	dl_sync((IORQ *)RQ, do_udl);
	RETURN;
}

DECL_IOCALL(KERNEL$DL_LOAD_LAZY, SPL_BOTTOM, DLLZRQ)
{
	dl_sync((IORQ *)RQ, do_dll);
	RETURN;
}

static void dl_sync(IORQ *rq, long (*fn)(void *))
{
	THREAD_RQ *t = malloc(sizeof(THREAD_RQ));
	if (__unlikely(!t)) {
		rq->status = -ENOMEM;
		CALL_AST(rq);
		return;
	}
	t->fn = dl_sync_done;
	t->thread_main = fn;
	t->p = rq;
	t->error = NULL;
	t->cwd = NULL;
	t->std_in = 0;
	t->std_out = 1;
	t->std_err = 2;
	t->dlrq = NULL;
	t->thread = NULL;
	t->spawned = 0;
	CALL_IORQ_CANCELABLE(t, KERNEL$THREAD, rq);
}

DECL_AST(dl_sync_done, SPL_BOTTOM, THREAD_RQ)
{
	IORQ *p;
	IO_DISABLE_CHAIN_CANCEL(SPL_BOTTOM, (IORQ *)RQ->p);
	if (__unlikely(!RQ->spawned)) {
		((IORQ *)RQ->p)->status = RQ->status;
	}
	p = RQ->p;
	free(RQ);
	RETURN_AST(p);
}

#define EXEC_WHITECHAR(c)	(__unlikely((c) == ' ') || __unlikely((c) == '\t'))

static char error[__MAX_STR_LEN];

static void check_shell(__const__ char *filename, START_ARG *start_arg)
{
	int r, s, c, i, n;
	char *cmdline, *cmd;
	char **args;
	int n_args;
	int h = open(filename, O_RDONLY | _O_NOPOSIX);
	if (__unlikely(h == -1)) return;
	cmdline = alloca(__MAX_STR_LEN);
	r = read(h, cmdline, __MAX_STR_LEN - 1);
	if (__unlikely(r < 4)) {
		close(h);
		return;
	}
	close(h);
	cmdline[r] = 0;
	r = 0;
	while (cmdline[r] != '\n' && cmdline[r] != '\r' && cmdline[r]) r++;
	cmdline[r] = 0;
	if (__unlikely(cmdline[0] != '#') || __unlikely(cmdline[1] != '!')) return;

	for (r = 2; __unlikely(EXEC_WHITECHAR(cmdline[r])); r++) ;
	for (s = r; !EXEC_WHITECHAR(cmdline[s]) && cmdline[s]; s++) if (cmdline[s] == '/') r = s + 1;
	if (__unlikely(s == r)) return;
	cmd = alloca(s - r + (1 + 7));
	memcpy(cmd, "PATH.:/", 7);
	memcpy(cmd + 7, cmdline + r, s - r);
	cmd[s - r + 7] = 0;

	n_args = 3;
	for (c = s; cmdline[c]; c++)
		if (EXEC_WHITECHAR(cmdline[c - 1]) && !EXEC_WHITECHAR(cmdline[c])) n_args++;
	if (start_arg->argv[0]) for (i = 1; start_arg->argv[i]; i++) n_args++;
	args = alloca(n_args * sizeof(char *));
	args[0] = cmd;
	n = 1;
	for (c = s; cmdline[c]; c++) {
		if ((EXEC_WHITECHAR(cmdline[c - 1]) || !cmdline[c - 1]) && !EXEC_WHITECHAR(cmdline[c])) args[n++] = cmdline + c;
		if (EXEC_WHITECHAR(cmdline[c])) cmdline[c] = 0;
	}
	args[n++] = (char *)filename;
	if (start_arg->argv[0]) for (i = 1; start_arg->argv[i]; i++) args[n++] = start_arg->argv[i];
	args[n++] = NULL;
	if (__unlikely(n > n_args)) KERNEL$SUICIDE("check_shell: ARGUMENT OVERFLOW: %d > %d", n, n_args);
	LOWER_SPL(SPL_ZERO);
	execve(cmd, args, start_arg->env);
	RAISE_SPL(SPL_DL);
	_snprintf(error, __MAX_STR_LEN, "CAN'T EXEC INTERPRETER %s: %s", cmd, strerror(errno));
}

long __DL_USER_LOAD_EXE(__const__ char *filename, START_ARG *start_arg, void **handle)
{
	long r;
	struct library_description *ld;
	__f_off val;
	char *start;
	DL_LOCK;
	ld = load_module(DLT_NOLIBPATH, filename, "IMG", error);
	if (__unlikely(!ld)) {
		check_shell(filename, start_arg);
		goto err_0;
	}
	*handle = ld;
	start = "_START";
	if (__unlikely(LINK_GET_SYMBOL(ld->address, start, SYM_EXPORTED, &val))) {
		_snprintf(error, __MAX_STR_LEN, "SYMBOL %s NOT FOUND", start);
		err:
		unload_module(ld, 1);
		err_0:
		DL_UNLOCK;
		__critical_printf("CANNOT LOAD %s: %s\n", filename, error);
		return 255;
	}
	if (__unlikely(!(void *)val)) {
		_snprintf(error, __MAX_STR_LEN, "SYMBOL %s IS ZERO", start);
		goto err;
	}
	DL_UNLOCK;
	r = ((long (*)(START_ARG *))val)(start_arg);
	LOWER_SPL(SPL_ZERO);
	return r;
}

struct dwarf_fde *_Unwind_Find_FDE(void *pc, struct dwarf_eh_bases *bases);

struct dwarf_fde *_Unwind_Find_FDE(void *pc, struct dwarf_eh_bases *bases)
{
	int spl;
	struct library_description *ld;
	DL_LOCK_SPL;
	/*__debug_printf("except: %x\n", pc);*/
	XLIST_FOR_EACH(ld, &udata.dl_modules, struct library_description, list) {
		/*__debug_printf("look at %p %x %x %x (%s)\n", ld->address, ld->code_size, ld->data_size, ld->bss_size, ld->modname);*/
		if ((char *)pc >= (char *)ld->address && (char *)pc < (char *)ld->address + ld->size) {
			__f_off start, end;
			DL_UNLOCK_SPL;
			if (__unlikely(LINK_GET_SYMBOL(ld->address, "DLL$EFS", SYM_SPECIAL, &start))) return NULL;
			if (__unlikely(LINK_GET_SYMBOL(ld->address, "DLL$EFE", SYM_SPECIAL, &end))) return NULL;
			return FIND_FDE(pc, bases, start, end, (__f_off)ld->address, (__f_off)ld->address + ld->code_size);
		}
	}
	DL_UNLOCK_SPL;
	return NULL;
}

