#include <SPAD/AC.H>
#include <LIB/KERNEL/UASM.H>
#include <ARCH/BITOPS.H>
#include <UNISTD.H>
#include <SPAD/LIBC.H>
#include <ARCH/BARRIER.H>
#include <ARCH/BSF.H>
#include <LIB/KERNEL/UTHREAD.H>
#include <SPAD/SYNC.H>
#include <SPAD/IOCTL.H>
#include <STDLIB.H>
#include <STRING.H>
#include <KERNEL/SELECT.H>
#include <LIB/KERNEL/UIO.H>
#include <LIB/KERNEL/SIGSTACK.H>
#include <SPAD/TTY.H>

#include <LIB/KERNEL/USIGNAL.H>
#include <SIGNAL.H>

extern unsigned char intr_key;
extern unsigned char quit_key;
/*extern unsigned char stop_key;*/

__finline__ void PROCESS_PENDING_SIGNALS(void)
{
	if (!((udata.sigpending | udata.thread_sigpending) & ~udata.sigblock)) return;
	if (__unlikely(__CMPXCHGC(&udata.signal_available, 1, 0) != 0)) return;
	CALL_AST(&udata.signal_ast);
}

int sigaction(int signum, __const__ struct sigaction *act, struct sigaction *oldact)
{
	if (__unlikely(signum < 1) || __unlikely(signum >= NSIG)) {
		inval:
		errno = EINVAL;
		return -1;
	}
	if (oldact) memcpy(oldact, &udata.sigaction[signum], sizeof(struct sigaction));
	if (!act) return 0;
	if (__unlikely((1 << signum) & UNHANDLABLE_SIGNALS)) goto inval;
	if (__unlikely(act->sa_handler == SIG_ERR)) goto inval;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO))) KERNEL$SUICIDE("sigaction AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_BOTTOM);
	memcpy(&udata.sigaction[signum], act, sizeof(struct sigaction));
	LOWER_SPL(SPL_ZERO);
	if (signum == SIGINT) INSTALL_INTR_KEY(0, intr_key, SIGINT);
	if (signum == SIGQUIT) INSTALL_INTR_KEY(0, quit_key, SIGQUIT);
	if (signum == SIGWINCH) INSTALL_INTR_KEY(1, -1, SIGWINCH);
	return 0;
}

int SIGCHLD_FOR_STOPPED(void)
{
	if (__likely(udata.sigaction[SIGCHLD].sa_handler == SIG_DFL) || __unlikely(udata.sigaction[SIGCHLD].sa_handler == SIG_IGN)) return 0;
	if (__unlikely(udata.sigaction[SIGCHLD].sa_flags & SA_NOCLDSTOP)) return 0;
	return 1;
}

sigset_t sigintr_set = 0;

int siginterrupt(int signum, int flag)
{
	if (__unlikely(signum < 1) || __unlikely(signum >= NSIG)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO))) KERNEL$SUICIDE("siginterrupt AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_BOTTOM);
	if (!flag) {
		udata.sigaction[signum].sa_flags |= SA_RESTART;
		sigdelset(&sigintr_set, signum);
	} else {
		udata.sigaction[signum].sa_flags &= ~SA_RESTART;
		sigaddset(&sigintr_set, signum);
	}
	LOWER_SPL(SPL_ZERO);
	return 0;
}

__sighandler_t signal(int signum, __sighandler_t handler)
{
	struct sigaction act, oact;
	act.sa_handler = handler;
	/*act.sa_flags = SA_RESETHAND | SA_NODEFER | SA_INTERRUPT;*/
	act.sa_flags = (sigismember(&sigintr_set, signum) ? 0 : SA_RESTART);
	if (__unlikely(sigaction(signum, &act, &oact) < 0)) return SIG_ERR;
	return oact.sa_handler;
}

int sigprocmask(int how, __const__ sigset_t *set, sigset_t *oldset)
{
	sigset_t s, os;
	again:
	os = udata.sigblock;
	if (oldset) *oldset = os;
	if (__unlikely(!set)) return 0;
	switch (how) {
		case SIG_BLOCK:
			s = os | *set;
			break;
		case SIG_UNBLOCK:
			s = os & ~*set;
			break;
		case SIG_SETMASK:
			s = *set;
			break;
		default:
			errno = EINVAL;
			return -1;
	}
	s &= ~UNHANDLABLE_SIGNALS;
	if (__unlikely(__CMPXCHGL(&udata.sigblock, os, s) != 0)) goto again;
	PROCESS_PENDING_SIGNALS();
	return 0;
}

int sigpending(sigset_t *set)
{
	*set = udata.sigpending | udata.thread_sigpending;
	return 0;
}

int raise(int sig)
{
	if (__unlikely(!sig)) return 0;
	if (__unlikely((unsigned)sig >= NSIG)) {
		errno = EINVAL;
		return -1;
	}
	__ORL(&udata.sigpending, (1L << sig));
	PROCESS_PENDING_SIGNALS();
	return 0;
}

int _thread_raise(int sig)
{
	if (__unlikely(!sig)) return 0;
	if (__unlikely((unsigned)sig >= NSIG)) {
		einval:
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(sig == SIGKILL) || __unlikely(sig == SIGSTOP)) {
		goto einval;
	}
	__ORL(&udata.thread_sigpending, (1L << sig));
	PROCESS_PENDING_SIGNALS();
	return 0;
}

void raise_sighup(void)
{
	static char raised = 0;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (!__CMPXCHGC(&raised, 0, 1)) raise(SIGHUP);
	LOWER_SPLX(spl);
}

extern IO_STUB sigsuspend_wait;

int sigsuspend(__const__ sigset_t *mask)
{
	sigset_t o;
	IORQ rq;
	int spl;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("sigsuspend AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	sigprocmask(SIG_SETMASK, mask, &o);
	SYNC_IO_SIGINTERRUPT(&rq, sigsuspend_wait);
	sigprocmask(SIG_SETMASK, &o, NULL);
	errno = EINTR;
	LOWER_SPLX(spl);
	return -1;
}

static WQ_DECL(sigsuspend_wq, "KERNEL$SIGSUSPEND_WQ");	/* never waken up */

DECL_IOCALL(sigsuspend_wait, SPL_BOTTOM, IORQ)
{
	WQ_WAIT(&sigsuspend_wq, RQ, sigsuspend_wait);
	RETURN;
}

static __const__ int sigpriority[];

int sigwait(__const__ sigset_t *set, int *sig)
{
	sigset_t o;
	int spl;
	int i;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("sigwait AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	sigprocmask(SIG_BLOCK, set, &o);
	__cancel_in(
		while (!((udata.sigpending | udata.thread_sigpending) & *set)) {
			THREAD_SIGWAIT();
		}
	);
	for (i = 0; sigpriority[i]; i++) {
		if ((udata.sigpending | udata.thread_sigpending) & *set & (1 << sigpriority[i])) {
			i = sigpriority[i];
			goto io;
		}
	}
	i = __BSF((udata.sigpending | udata.thread_sigpending) & *set);
	io:
	udata.sigpending &= ~(1 << i);
	udata.thread_sigpending &= ~(1 << i);
	sigprocmask(SIG_SETMASK, &o, NULL);
	LOWER_SPLX(spl);
	if (__likely(sig != NULL)) *sig = i;
	return 0;
}

void _thread_sigwait(__const__ sigset_t *set)
{
	sigset_t o;
	int spl;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("_thread_sigwait AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	sigprocmask(SIG_BLOCK, set, &o);
	__cancel_in(
		while (!(udata.thread_sigpending & *set)) {
			__cancel_in(THREAD_SIGWAIT());
		}
	);
	sigprocmask(SIG_SETMASK, &o, NULL);
	LOWER_SPLX(spl);
}

int sigaltstack(__const__ stack_t *ss, stack_t *oss)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO))) KERNEL$SUICIDE("sigaltstack AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_THREAD);
	if (oss) {
		get_altstack(oss);
	}
	if (ss) {
		if (__unlikely(on_altstack())) {
			errno = EPERM;
			err:
			LOWER_SPL(SPL_ZERO);
			return -1;
		}
		if (__unlikely(ss->ss_size < MINSIGSTKSZ)) {
			errno = ENOMEM;
			goto err;
		}
		if (__unlikely((unsigned long)ss->ss_sp + ss->ss_size > KUVMTOP) || __unlikely((unsigned long)ss->ss_sp + ss->ss_size < (unsigned long)ss->ss_sp)) {
			errno = EINVAL;
			goto err;
		}
		set_altstack(ss);
	}
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static void sig_term(int sig)
{
	char msg[__MAX_STR_LEN];
	if (__unlikely(KERNEL$SUICIDE_RESTORE_VIDEOMODE != NULL)) KERNEL$SUICIDE_RESTORE_VIDEOMODE();
	udata.sigblock = -1;
	raise(sig);	/* this is needed because of threads --- if there
		are more threads, they will all terminate on this signal */
	_snprintf(msg, __MAX_STR_LEN, "SIGNAL %s RECEIVED", strsignal(sig));
	_exit_msg(__EXIT_SIGNAL(sig), msg);
}

static void sig_core(int sig)
{
	KERNEL$STACK_DUMP();
	sig_term(sig);
}

static void sig_stop(int sig)
{
	_stop(NULL);
}

static __const__ __sighandler_t default_action[] = {
	sig_core,
	sig_core, sig_core, sig_core, sig_core,
	sig_core, sig_core, sig_term, sig_core,
	sig_term, sig_term, sig_stop, sig_stop,
	sig_stop, SIG_IGN, sig_term, SIG_IGN,
	SIG_IGN, SIG_IGN, sig_term, SIG_IGN,
	SIG_IGN, SIG_IGN, SIG_IGN, SIG_IGN,
	NULL };

static __const__ int sigpriority[] = {
	SIGMCE,
	SIGSEGV, SIGBUS, SIGILL, SIGFPE,
	SIGTRAP, SIGABRT, SIGTERM, SIGQUIT,
	SIGINT, SIGHUP, SIGTSTP, SIGTTIN,
	SIGTTOU, SIGCONT, SIGPIPE, SIGURG,
	SIGIO, SIGCHLD, SIGALRM, SIGVTALRM,
	SIGPROF, SIGUSR1, SIGUSR2, SIGWINCH,
	0 };

void call_sig(void *h, unsigned n);

emit_sigaltstack(call_sig_switch, call_sig);

void call_sig(void *h, unsigned n)
{
	int saved_errno;
	int bpl;
	/* FPU save/restore moved to KERNEL_THREAD_RECURSIVE_BLOCK */
	bpl = KERNEL$BPL;
	KERNEL$BPL = SPL_X(SPL_ZERO);
	LOWER_SPL(SPL_ZERO);
	saved_errno = errno;
	((__sighandler_t)h)(n);
	errno = saved_errno;
	RAISE_SPL(SPL_BOTTOM);
	KERNEL$BPL = bpl;
}

DECL_AST(sig_dispatcher, SPL_BOTTOM, AST)
{
	int i, j;
	sigset_t set;
	udata.signal_available = 1;
	again:
	__barrier();
	if (__unlikely(sigismember(&udata.sigpending, SIGKILL))) sig_term(SIGKILL);
	if (__unlikely(sigismember(&udata.sigpending, SIGSTOP))) {
		__ANDL(&udata.sigpending, ~(1L << SIGSTOP));
		_stop(NULL);
		goto again;
	}
	set = (udata.sigpending | udata.thread_sigpending) & ~udata.sigblock;
	if (__unlikely(!set)) {
		RETURN;
	}
	for (j = 0; (i = sigpriority[j]); j++) if (__unlikely(sigismember(&set, i))) {
		__sighandler_t h;
		sigset_t omask, mask;
		int nointr, alt;
		RAISE_SPL(SPL_THREAD);
		THREAD_SIGWAKE();
		LOWER_SPL(SPL_BOTTOM);
		h = udata.sigaction[i].sa_handler;
		if (h == SIG_DFL) h = default_action[j];
		if (__unlikely(h == SIG_IGN)) {
			__ANDL(&udata.sigpending, ~(1L << i));
			__ANDL(&udata.thread_sigpending, ~(1L << i));
			goto again;
		}
		mask = udata.sigaction[i].sa_mask;
		if (!(udata.sigaction[i].sa_flags & SA_NODEFER)) sigaddset(&mask, i);
		nointr = udata.sigaction[i].sa_flags & SA_RESTART;
		alt = __unlikely(udata.sigaction[i].sa_flags & SA_ONSTACK) && !on_altstack() && __likely(has_altstack() != 0);
		if (udata.sigaction[i].sa_flags & SA_RESETHAND) {
			udata.sigaction[i].sa_handler = SIG_DFL;
			udata.sigaction[i].sa_mask = 0;
			udata.sigaction[i].sa_flags = 0;
		}
		sigprocmask(SIG_BLOCK, &mask, &omask);
		__ANDL(&udata.sigpending, ~(1L << i));
		__ANDL(&udata.thread_sigpending, ~(1L << i));
#if SPL_ZERO + 1 != SPL_BOTTOM
	error: SPL_BOTTOM is not one level above SPL_ZERO
#endif
		SIGNAL_CANCEL_SELECT();
		SIGNAL_CANCEL_IORQ(nointr);
		KERNEL_THREAD_RECURSIVE_BLOCK(__likely(!alt) ? call_sig : call_sig_switch, h, (unsigned)i);
		sigprocmask(SIG_SETMASK, &omask, NULL);
		goto again;
	}
	__ANDL(&udata.sigpending, ~set);
	__ANDL(&udata.thread_sigpending, ~set);
	goto again;
}

void KERNEL$INTR_SYSCALL(void)
{
	SIGNAL_CANCEL_SELECT();
	SIGNAL_CANCEL_IORQ(0);
}

DECL_AST(user_exception, SPL_TOP, AST)
{
	int sig;
	if (__likely(udata.xcpt_type == XCPT_RPF) || __likely(udata.xcpt_type == XCPT_WPF)) {
		int a = UVM_SAVE_FAULT(udata.xcpt_address, udata.xcpt_error);
		if (__likely(a == 1)) goto ok;
		if (__unlikely(a == -1)) {
			sig = SIGBUS;
			goto do_sig;
		}
	}
	sig = __exception2signal(udata.xcpt_type);
	do_sig:
		
	__critical_printf("EXCEPTION: %s.\n\r", __exceptionmsg(udata.xcpt_type, udata.xcpt_return, udata.xcpt_address, udata.xcpt_error, 1));
	KERNEL$STACK_DUMP();

	if (__unlikely(!sig) || __likely(udata.sigaction[sig].sa_handler == SIG_DFL) || __unlikely(((udata.thread_sigpending | udata.sigblock) & (1L << sig)) != 0)) {
		_exit_msg(__EXIT_SIGNAL(sig), __exceptionmsg(udata.xcpt_type, udata.xcpt_return, udata.xcpt_address, udata.xcpt_error, 1));
	}
	
	__ORL(&udata.thread_sigpending, (1L << sig));
	ok:
	udata.xcpt_available = 1;
	__barrier();
	PROCESS_PENDING_SIGNALS();
	RETURN;
}

typedef struct {
	LIST_ENTRY list;
	int key;
	int signal;
	IOCTLRQ io;
	union {
		struct tty_control_state tcs;
		struct tty_window_size tws;
	} u;
	WQ cleanup;
} INTR_KEY;

static void FREE_IK(INTR_KEY *ik)
{
	WQ_WAKE_ALL(&ik->cleanup);
	DEL_FROM_LIST(&ik->list);
	free(ik);
}

extern AST_STUB INTR_CONTROL_GOT;

DECL_AST(INTR_KEY_PRESSED, SPL_DEV, IOCTLRQ)
{
	INTR_KEY *ik = GET_STRUCT(RQ, INTR_KEY, io);
	if (__unlikely(!ik->signal)) {
		free_ik:
		FREE_IK(ik);
		RETURN;
	}
	if (__unlikely(RQ->status == -EINTR)) {
		RETURN_IORQ(&ik->io, KERNEL$IOCTL);
	}
	if (__unlikely(RQ->status == -ETTYCHG)) {
		if (ik->key >= 0) goto ttych;
		else goto rais;
	}
	if (__unlikely(RQ->status < 0)) {
		goto free_ik;
	}
	rais:
	raise(ik->signal);
	ttych:
	ik->io.fn = INTR_CONTROL_GOT;
	ik->io.ioctl = __likely(ik->key >= 0) ? IOCTL_TTY_GET_CONTROL : IOCTL_TTY_GET_WINSIZE;
	RETURN_IORQ(&ik->io, KERNEL$IOCTL);
}

DECL_AST(INTR_CONTROL_GOT, SPL_DEV, IOCTLRQ)
{
	INTR_KEY *ik = GET_STRUCT(RQ, INTR_KEY, io);
	if (__unlikely(!ik->signal)) {
		free_ik:
		FREE_IK(ik);
		RETURN;
	}
	if (__unlikely(RQ->status == -EINTR)) {
		RETURN_IORQ(&ik->io, KERNEL$IOCTL);
	}
	if (__unlikely(RQ->status < 0)) {
		goto free_ik;
	}
	ik->io.fn = INTR_KEY_PRESSED;
	ik->io.ioctl = __likely(ik->key >= 0) ? IOCTL_TTY_WAIT_CONTROL : IOCTL_TTY_WAIT_WINSIZE;;
	RETURN_IORQ(&ik->io, KERNEL$IOCTL);
}

void INSTALL_INTR_KEY(int h, int key, int sig)
{
	INTR_KEY *ik;
	int spl = 0;	/* squash warning */
/*__debug_printf("iik(%d-%d)\n", key, sig);*/
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), KERNEL$SPL)))
		KERNEL$SUICIDE("INSTALL_INTR_KEY AT SPL %08X", KERNEL$SPL);
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_DEV);
	if (__unlikely(udata.sigaction[sig].sa_handler == SIG_IGN)) sig = 0;
	if (__unlikely(key >= N_CONTROLS) || __unlikely(!key))
		goto ret;
	XLIST_FOR_EACH(ik, &udata.intr_keys, INTR_KEY, list) if (__unlikely(ik->io.h == h) && __unlikely(ik->key == key)) {
		if (__likely(sig) && __likely(ik->signal == sig)) goto ret;
		ik->signal = 0;
		KERNEL$CIO((IORQ *)&ik->io);
	}
	if (__unlikely(!sig)) goto ret;
	if (__unlikely(!(ik = malloc(sizeof(INTR_KEY))))) goto ret;
	ik->key = key;
	ik->signal = sig;
	WQ_INIT(&ik->cleanup, "KERNEL$INTR_KEY_CLEANUP");
	ADD_TO_XLIST(&udata.intr_keys, &ik->list);
	ik->io.fn = INTR_CONTROL_GOT;
	ik->io.h = h;
	if (__likely(key >= 0)) {
		ik->io.ioctl = IOCTL_TTY_GET_CONTROL;
		ik->io.param = key;
		ik->io.v.ptr = (unsigned long)&ik->u.tcs;
		ik->io.v.len = sizeof ik->u.tcs;
	} else {
		ik->io.ioctl = IOCTL_TTY_GET_WINSIZE;
		ik->io.param = 0;
		ik->io.v.ptr = (unsigned long)&ik->u.tws;
		ik->io.v.len = sizeof ik->u.tws;
	}
	ik->io.v.vspace = &KERNEL$VIRTUAL;
	CALL_IORQ(&ik->io, KERNEL$IOCTL);
	ret:
	LOWER_SPLX(spl);
}

void SIGNAL_INIT(void)
{
	if (__unlikely(sizeof(default_action) / sizeof(__sighandler_t) != sizeof(sigpriority) / sizeof(int))) KERNEL$SUICIDE("SIGNAL_INIT: MISCOMPILED KERNEL.DLL: DEFAULT ACTIONS (%d) != SIGNAL PRIORITIES (%d)", (int)(sizeof(default_action) / sizeof(__sighandler_t)), (int)(sizeof(sigpriority) / sizeof(int)));
	INIT_XLIST(&udata.intr_keys);
	udata.xcpt_handler.tmp3 = AST_STUB_SPL(user_exception, -1);
	udata.xcpt_handler.fn = user_exception;
	__barrier();
	udata.xcpt_available = 1;
	udata.signal_ast.fn = sig_dispatcher;
	__barrier();
	udata.signal_available = 1;
	PROCESS_PENDING_SIGNALS();
	INSTALL_INTR_KEY(0, intr_key, SIGINT);
	INSTALL_INTR_KEY(0, quit_key, SIGQUIT);
	/*INSTALL_INTR_KEY(stop_key, SIGTSTP);*/
}

void SIGNAL_FORK(void)
{
	udata.sigpending = 0;
	udata.thread_sigpending = 0;
	while (!XLIST_EMPTY(&udata.intr_keys)) FREE_IK(LIST_STRUCT(udata.intr_keys.next, INTR_KEY, list));
}

void SIGNAL_FORK_INTR_KEYS(void)
{
	INSTALL_INTR_KEY(0, intr_key, SIGINT);
	INSTALL_INTR_KEY(0, quit_key, SIGQUIT);
}

int CANCEL_INTR_KEYS(int h, IORQ *rq)
{
	INTR_KEY *ik;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("CANCEL_INTR_KEYS AT SPL %08X", KERNEL$SPL);
	XLIST_FOR_EACH(ik, &udata.intr_keys, INTR_KEY, list) if (ik->io.h == h) {
		ik->signal = 0;
		KERNEL$CIO((IORQ *)&ik->io);
		if (rq != NULL) {
			WQ_WAIT_F(&ik->cleanup, rq);
			rq = NULL;
		}
	}
	return !rq;
}

void REFRESH_INTR_KEYS(int h)
{
	INTR_KEY *ik;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("REFRESH_INTR_KEYS AT SPL %08X", KERNEL$SPL);
	XLIST_FOR_EACH(ik, &udata.intr_keys, INTR_KEY, list) if (ik->io.h == h) {
		KERNEL$CIO((IORQ *)&ik->io);
	}
}
