#include <ARCH/SETUP.H>
#include <LIB/KERNEL/UASM.H>
#include <SPAD/TIMER.H>
#include <KERNEL/SYSCALL.H>
#include <LIB/KERNEL/UTHREAD.H>
#include <LIB/KERNEL/UTHREADA.H>
#include <LIB/KERNEL/UIO.H>
#include <LIB/KERNEL/SIGSTACK.H>
#include <LIB/KERNEL/USIGNAL.H>
#include <SPAD/CD.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <STRING.H>
#include <SETJMP.H>
#include <KERNEL/THREAD.H>

#include <SPAD/THREAD.H>
#include <SPAD/SYNC.H>
#include <KERNEL/SELECT.H>
#include <SCHED.H>

#define PREEMPT_TIME	(JIFFIES_PER_SECOND / 10)

typedef struct {
	TIMER tm;
	THREAD *t;
} TTIMER;


#define THREAD_ENTRIES_1		\
	LIST_ENTRY run_list;

#define THREAD_ENTRIES_2		\
	AST_STUB *unblock_fn;		\
	char sleep;			\
	int errno;			\
	sigset_t sigblock;		\
	sigset_t thread_sigpending;	\
	THREAD_RQ *rq;			\
	IORQ * volatile io;		\
	IORQ * volatile cio;		\
	TTIMER * volatile ctm;		\
	struct signal_chain *sch;	\
	struct select_context *sctx;	\
	LIST_ENTRY sigwait_list;	\
	LIST_ENTRY all_threads;

#define S_RUNNING	0
#define S_READY		1
#define S_BLOCKED	2
#define S_DEAD		3

struct signal_chain {
	IORQ *io;
	IORQ *cio;
	TTIMER *ctm;
	struct signal_chain *next;
};

extern int __isthreaded;

static void START_THREAD(void);

#include <LIB/KERNEL/THREADMD.I>

extern IO_STUB THREAD_WAKE_UP;

static void SET_READY(THREAD *t);
static void TIME_WAKE_UP(TIMER *tm);
static void KERNEL_THREAD_YIELD_INTERNAL(void *x, unsigned to_end);

void *KERNEL$ALLOC_THREAD_SYNC(void)
{
	return ALLOC_THREAD();
}

void KERNEL$FREE_THREAD(void *t)
{
	FREE_THREAD(t);
}

DECL_IOCALL(KERNEL$THREAD, SPL_THREAD, THREAD_RQ)
{
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(RQ->spawned)) KERNEL$SUICIDE("KERNEL$THREAD: SPAWNED FIELD NOT ZEROED (%d)", RQ->spawned);
#endif
	t = RQ->thread;
	if (__likely(!t) && __unlikely(__IS_ERR(t = ALLOC_THREAD()))) {
		RQ->status = __PTR_ERR(t);
		RETURN_AST(RQ);
	}
	RQ->thread_internal = t;
	SET_READY(t);
	VOID_LIST_ENTRY(&t->sigwait_list);
	t->sigblock = -1;
	t->rq = RQ;
	__isthreaded = 1;
	RQ->spawned = 1;
	RETURN;
}

static void (*call_in_fn)(void *, unsigned) = NULL;
static void *call_in_data = NULL;
static unsigned call_in_data2 = NULL;

static void BLOCK_THREAD(void)
{
	THREAD *t;
	test_again:
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_THREAD)))
		KERNEL$SUICIDE("BLOCK_THREAD AT SPL %08X", KERNEL$SPL);
	if (__unlikely(udata.current_thread->sleep == S_RUNNING))
		KERNEL$SUICIDE("BLOCK_THREAD: RUNNING THREAD BLOCKED");
#endif
	if (__likely(LIST_EMPTY(&udata.thread_ready))) {
		while (1) {
			udata.blocked = 1;
			LOWER_SPL(SPL_ZERO);
			SYSCALL3(SYS_BLOCK, (unsigned long)KERNEL$STACKPAGE + UDATA_STRUCT + (unsigned long)&((UDATA *)NULL)->thread_ready, (unsigned long)&udata.thread_ready);
			RAISE_SPL(SPL_THREAD);
			udata.blocked = 0;
			if (__likely(!LIST_EMPTY(&udata.thread_ready))) break;
		}
	}
	t = LIST_STRUCT(udata.thread_ready.next, THREAD, run_list);
#if __DEBUG >= 1
	if (__unlikely(t->sleep != S_READY) && __likely(!udata.exiting))
		KERNEL$SUICIDE("BLOCK_THREAD: NON-READY THREAD ON QUEUE");
#endif
	t->sleep = S_RUNNING;
	DEL_FROM_LIST(&t->run_list);
	if (__unlikely(t != udata.current_thread)) {
		udata.current_thread->errno = errno;
		udata.current_thread->sigblock = udata.sigblock;
		udata.current_thread->thread_sigpending = udata.thread_sigpending;
		udata.sigblock = -1;
		udata.thread_sigpending = 0;
		errno = t->errno;
		if (__unlikely(SWITCH_THREAD(t))) {
			errno = udata.current_thread->errno;
			udata.sigblock = udata.current_thread->sigblock;
			udata.thread_sigpending = udata.current_thread->thread_sigpending;
			PROCESS_PENDING_SIGNALS();
			goto test_again;
		}
		udata.sigblock = udata.current_thread->sigblock;
		udata.thread_sigpending = udata.current_thread->thread_sigpending;
		PROCESS_PENDING_SIGNALS();
	}
	if (__unlikely(call_in_fn != NULL)) {
		void (*cf)(void *, unsigned) = call_in_fn;
		void *cd = call_in_data;
		unsigned cd2 = call_in_data2;
		call_in_fn = NULL;
		call_in_data = NULL;
		LOWER_SPL(SPL_BOTTOM);
		KERNEL_THREAD_RECURSIVE_BLOCK(cf, cd, cd2);
		RAISE_SPL(SPL_THREAD);
	}
}

static void START_THREAD(void)
{
	long r;
	THREAD *t = udata.current_thread;
	udata.sigblock = t->sigblock;
	LOWER_SPL(SPL_ZERO);
	r = t->rq->thread_main(t->rq->p);
	if (__unlikely((r & RQS_PROC_MASK) == RQS_PROC)) r = 0;
	t->rq->thread_internal = NULL;
	t->rq->status = r;
	/* CALL_AST(t->rq); done in END_THREAD */
	LOWER_SPL(SPL_ZERO);
	RAISE_SPL(SPL_TIMER);
	FREE_THREAD_SELECT_STORAGE(&t->sctx);
	RAISE_SPL(SPL_THREAD);
	DEL_FROM_LIST(&t->sigwait_list);
	END_THREAD(t, t->rq, t->rq->thread);
}

static DECL_XLIST(SIGWAIT_THREADS);

void THREAD_SIGWAIT(void)
{
#if __DEBUG >= 1
	if (__unlikely(SPL_X(SPL_THREAD) != KERNEL$SPL)) KERNEL$SUICIDE("THREAD_SIGWAIT CALLED AT SPL %08X", KERNEL$SPL);
#endif
	DEL_FROM_LIST(&udata.current_thread->sigwait_list);
	ADD_TO_XLIST(&SIGWAIT_THREADS, &udata.current_thread->sigwait_list);
	udata.current_thread->sleep = S_BLOCKED;
	BLOCK_THREAD();
}

void THREAD_SIGWAKE(void)
{
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(SPL_X(SPL_THREAD) != KERNEL$SPL)) KERNEL$SUICIDE("THREAD_SIGWAKE CALLED AT SPL %08X", KERNEL$SPL);
#endif
	XLIST_FOR_EACH_UNLIKELY(t, &SIGWAIT_THREADS, THREAD, sigwait_list) {
		THREAD *tt = LIST_STRUCT(t->sigwait_list.prev, THREAD, sigwait_list);
		DEL_FROM_LIST(&t->sigwait_list);
		VOID_LIST_ENTRY(&t->sigwait_list);
		if (t->sleep == S_BLOCKED) {
			SET_READY(t);
		}
		t = tt;
	}
}

IORQ *KERNEL$PREPARE_BLOCK(void)
{
	THREAD *t = udata.current_thread;
	if (__unlikely(t->sleep != S_RUNNING)) KERNEL$SUICIDE("KERNEL$PREPARE_BLOCK: THREAD NOT RUNNING, STATE %d", t->sleep);
	return (IORQ *)t->rq;
}

void KERNEL$BLOCK(IORQ *rq)
{
	while (rq->fn != NULL) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__unlikely(!rq->fn)) {
			LOWER_SPLX(s);
			break;
		}
		udata.current_thread->sleep = S_BLOCKED;
		BLOCK_THREAD();
		LOWER_SPLX(s);
	}
}

void KERNEL$SYNC_IO_SIGINTERRUPT(IORQ *rq, IO_STUB *call)
{
	THREAD *t = udata.current_thread;
	/* warning: must not block before posting IORQ */
#if __DEBUG >= 2
	if (__likely(!udata.exiting)) {
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SYNC_IO_SIGINTERRUPT CALLED AT SPL %08X", KERNEL$SPL);
		if (__unlikely(t->sleep != S_RUNNING)) KERNEL$SUICIDE("KERNEL$SYNC_IO_SIGINTERRUPT: THREAD NOT RUNNING, STATE %d", t->sleep);
		if (__unlikely(t->io != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_SIGINTERRUPT: IO PENGING ON CURRENT THREAD");
		if (__unlikely(t->cio != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_SIGINTERRUPT: CIO PENGING ON CURRENT THREAD");
		if (__unlikely(t->ctm != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_SIGINTERRUPT: TIMER PENGING ON CURRENT THREAD");
	}
#endif
	rq->fn = t->unblock_fn;
	CALL_IORQ_EXPR(rq, (t->cio = rq, call));
	while (__unlikely(rq->fn != NULL)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__unlikely(!rq->fn)) {
			goto brk;
		}
		udata.current_thread->sleep = S_BLOCKED;
		BLOCK_THREAD();
		brk:
		LOWER_SPLX(s);
	}
	udata.current_thread->cio = NULL;
}

void KERNEL$SYNC_IO_CANCELABLE(IORQ *rq, IO_STUB *call)
{
	/* warning: must not block before posting IORQ */
	THREAD *t = udata.current_thread;
#if __DEBUG >= 2
	if (__likely(!udata.exiting)) {
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE CALLED AT SPL %08X", KERNEL$SPL);
		if (__unlikely(t->sleep != S_RUNNING)) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: THREAD NOT RUNNING, STATE %d", t->sleep);
		if (__unlikely(t->io != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: IO PENGING ON CURRENT THREAD");
		if (__unlikely(t->cio != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: CIO PENGING ON CURRENT THREAD");
		if (__unlikely(t->ctm != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: TIMER PENGING ON CURRENT THREAD");
	}
#endif
	rq->fn = t->unblock_fn;
	CALL_IORQ_CANCELABLE_EXPR(rq, (t->io = rq, call), t->rq);
	while (__unlikely(rq->fn != NULL)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__unlikely(!rq->fn)) {
			goto brk;
		}
		udata.current_thread->sleep = S_BLOCKED;
		BLOCK_THREAD();
		brk:
		LOWER_SPLX(s);
	}
	udata.current_thread->io = NULL;
}

void KERNEL$SYNC_IO(IORQ *rq, IO_STUB *call)
{
	THREAD *t = udata.current_thread;
	/* warning: must not block before posting IORQ */
#if __DEBUG >= 2
	if (__likely(!udata.exiting)) {
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SYNC_IO CALLED AT SPL %08X", KERNEL$SPL);
		if (__unlikely(t->sleep != S_RUNNING)) KERNEL$SUICIDE("KERNEL$SYNC_IO: THREAD NOT RUNNING, STATE %d", t->sleep);
		if (__unlikely(t->io != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO: IO PENGING ON CURRENT THREAD");
		if (__unlikely(t->cio != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO: CIO PENGING ON CURRENT THREAD");
		if (__unlikely(t->ctm != NULL)) KERNEL$SUICIDE("KERNEL$SYNC_IO: TIMER PENGING ON CURRENT THREAD");
	}
#endif
	rq->fn = t->unblock_fn;
	CALL_IORQ_EXPR(rq, (t->io = rq, call));
	while (__unlikely(rq->fn != NULL)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__unlikely(!rq->fn)) {
			goto brk;
		}
		udata.current_thread->sleep = S_BLOCKED;
		BLOCK_THREAD();
		brk:
		LOWER_SPLX(s);
	}
	udata.current_thread->io = NULL;
}

/* this should not be called directy. Only from assembler */
DECL_AST(__UNBLOCK, SPL_THREAD, IORQ)
{
	THREAD *t = (THREAD *)RQ->tmp2;
	IO_DISABLE_CHAIN_CANCEL(SPL_THREAD, t->rq);
	if (__unlikely(t->sleep == S_BLOCKED)) {
		SET_READY(t);
	}
	RQ->fn = NULL;
	RETURN;
}

extern AST_STUB PREEMPT_AST;

static void SET_READY(THREAD *t)
{
	if (!LIST_EMPTY(&udata.thread_ready) || !udata.blocked) {
		if (__likely(!udata.preempt_ast_pending)) {
			udata.preempt_ast_pending = 1;
			CALL_AST(&udata.preempt_ast);
		}
	}
	ADD_TO_LIST(&udata.thread_ready, &t->run_list);
	t->sleep = S_READY;
}

static void PREEMPT_TIMER(TIMER *t)
{
	LOWER_SPL(SPL_THREAD);
	udata.preempt_timer_pending = 0;
	if (__likely(!udata.preempt_ast_pending)) {
		CALL_AST(&udata.preempt_ast);
	}
	udata.preempt_ast_pending = 2;
}

DECL_AST(PREEMPT_AST, SPL_BOTTOM, AST)
{
	char preempt_type;
	int bpl = KERNEL$BPL;
	KERNEL$BPL = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_THREAD);
	preempt_type = udata.preempt_ast_pending;
#if __DEBUG >= 1
	if (__unlikely(!preempt_type))
		KERNEL$SUICIDE("PREEMPT_AST: CALLED WITH ZERO PREEMPT_AST_PENDING");
#endif
	udata.preempt_ast_pending = 0;
	if (!udata.blocked) {
		if (__likely(!LIST_EMPTY(&udata.thread_ready))) {
			if (!udata.preempt_timer_pending) udata.preempt_timer_pending = 1, KERNEL$SET_TIMER(PREEMPT_TIME, &udata.preempt_timer);
			LOWER_SPL(SPL_BOTTOM);
			KERNEL_THREAD_RECURSIVE_BLOCK(KERNEL_THREAD_YIELD_INTERNAL, NULL, preempt_type - 1);
		}
	} else {
		if (__unlikely(udata.thread_ready.next != udata.thread_ready.prev)) {
			if (!udata.preempt_timer_pending) udata.preempt_timer_pending = 1, KERNEL$SET_TIMER(PREEMPT_TIME, &udata.preempt_timer);
		}
	}
	/* here we can be on SPL_THREAD or SPL_BOTTOM */
	KERNEL$BPL = bpl;
	RETURN;
}

void KERNEL$THREAD_YIELD(void)
{
	int spl;
	if (LIST_EMPTY(&udata.thread_ready)) {
		spl = KERNEL$SPL;
		TEST_SPLX(SPL_X(SPL_ZERO), spl);
		return;
	}
	KERNEL_THREAD_YIELD_INTERNAL(NULL, 1);
}

static void KERNEL_THREAD_YIELD_INTERNAL(void *x, unsigned to_end)
{
	int spl;
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL_THREAD_YIELD_INTERNAL CALLED AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	t = udata.current_thread;
	if (to_end) ADD_TO_LIST_END(&udata.thread_ready, &t->run_list);
	else ADD_TO_LIST_AFTER(udata.thread_ready.next, &t->run_list);	/* !!! TODO: this works badly when more threads are woken up simultaneously */
	t->sleep = S_READY;
	BLOCK_THREAD();
	LOWER_SPLX(spl);
}

AST_STUB *KERNEL$UNBLOCK(void)
{
	return udata.current_thread->unblock_fn;
}

void KERNEL$CALL_IN_THREAD(THREAD_RQ *rq, void (*fn)(void *, unsigned), void *data, unsigned data2)
{
	int spl;
	THREAD *t, *tt;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_BOTTOM), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$CALL_ON_THREAD CALLED AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	if (__unlikely(!(tt = rq->thread_internal))) goto lo;
	t = udata.current_thread;
	if (__unlikely(tt == t)) {
		LOWER_SPL(SPL_BOTTOM);
		KERNEL_THREAD_RECURSIVE_BLOCK(fn, data, data2);
		goto lo;
	}
	call_in_fn = fn;
	call_in_data = data;
	call_in_data2 = data2;
	SET_READY(t);
	if (__likely(tt->sleep == S_BLOCKED)) {
		SET_READY(tt);
	} else if (tt->sleep == S_READY) {
		DEL_FROM_LIST(&tt->run_list);
		ADD_TO_LIST(&udata.thread_ready, &tt->run_list);
	} else KERNEL$SUICIDE("KERNEL$CALL_IN_THREAD: TARGET THREAD STATE %d", tt->sleep);
	BLOCK_THREAD();
	lo:
	LOWER_SPLX(spl);
}

DECL_IOCALL(WAIT, SPL_THREAD + 1, IORQ)
{
	WQ_WAIT((WQ *)RQ->tmp3, RQ, KERNEL$SUCCESS);
	RETURN;
}

void KERNEL$WQ_WAIT_SYNC(WQ *wq)
{
	IORQ io;
	io.tmp3 = (unsigned long)wq;
	KERNEL$SYNC_IO(&io, WAIT);
}

long KERNEL$WQ_WAIT_SYNC_SIGINTERRUPT(WQ *wq)
{
	IORQ io;
	io.tmp3 = (unsigned long)wq;
	KERNEL$SYNC_IO_SIGINTERRUPT(&io, WAIT);
	return io.status;
}

long KERNEL$WQ_WAIT_SYNC_CANCELABLE(WQ *wq)
{
	IORQ io;
	io.tmp3 = (unsigned long)wq;
	KERNEL$SYNC_IO_CANCELABLE(&io, WAIT);
	return io.status;
}

void KERNEL$SLEEP(jiffies_t j)
{
	u_jiffies_t jjj;
	u_jiffies_t jj = KERNEL$GET_JIFFIES();
	again:
	if (__likely(!KERNEL$SLEEP_CANCELABLE(j))) return;
	jjj = KERNEL$GET_JIFFIES();
	j -= jjj - jj;
	jj = jjj;
	if (j < 0) return;
	goto again;
}

int KERNEL$SLEEP_CANCELABLE(jiffies_t j)
{
	TTIMER tm;
	int spl;
#if __DEBUG >= 1
	if (__likely(!udata.exiting)) {
		if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SLEEP_CANCELABLE CALLED AT SPL %08X", KERNEL$SPL);
	}
#endif
	INIT_TIMER(&tm.tm);
	tm.tm.fn = TIME_WAKE_UP;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	tm.t = udata.current_thread;
	udata.current_thread->ctm = &tm;
	KERNEL$SET_TIMER(j, &tm.tm);
	while (tm.tm.fn != NULL && __likely(tm.tm.fn != (void *)1)) {
		udata.current_thread->sleep = S_BLOCKED;
		BLOCK_THREAD();
	}
	udata.current_thread->ctm = NULL;
	LOWER_SPLX(spl);
	return __likely(!tm.tm.fn) ? 0 : -EINTR;
}

static void TIME_WAKE_UP(TIMER *tm)
{
	THREAD *t;
	LOWER_SPL(SPL_THREAD);
	t = LIST_STRUCT(tm, TTIMER, tm)->t;
	if (__likely(t->sleep == S_BLOCKED)) {
		SET_READY(t);
	}
	tm->fn = NULL;
}

void KERNEL_THREAD_RECURSIVE_BLOCK(void (*f)(void *, unsigned), void *data, unsigned data2)
{
	THREAD *t;
	int s;
	struct signal_chain prq;
	def_save_fpu
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_BOTTOM))) KERNEL$SUICIDE("KERNEL_THREAD_RECURSIVE_BLOCK AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_THREAD);
	t = udata.current_thread;
	if ((s = t->sleep) == S_READY) DEL_FROM_LIST(&t->run_list);
	t->sleep = S_RUNNING;
	prq.io = t->io;
	prq.cio = t->cio;
	prq.ctm = t->ctm;
	prq.next = t->sch;
	t->io = NULL;
	t->cio = NULL;
	t->ctm = NULL;
	t->sch = &prq;
	LOWER_SPL(SPL_BOTTOM);
	save_fpu;
	f(data, data2);
	restore_fpu;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_BOTTOM))) KERNEL$SUICIDE("KERNEL_THREAD_RECURSIVE_BLOCK: FUNCTION RETURNED AT SPL %08X", KERNEL$SPL);
	if (__unlikely(udata.current_thread->sleep != S_RUNNING)) KERNEL$SUICIDE("KERNEL_THREAD_RECURSIVE_BLOCK: THREAD NOT RUNNING, STATE %d", udata.current_thread->sleep);
#endif
	RAISE_SPL(SPL_THREAD);
	if (s != S_RUNNING) {
		t = udata.current_thread;
		ADD_TO_LIST_END(&udata.thread_ready, &t->run_list);
		t->sleep = S_READY;
	}
#if __DEBUG >= 1
	if (__unlikely(t->sch != &prq))
		KERNEL$SUICIDE("KERNEL_THREAD_RECURSIVE_BLOCK: SIGNAL CHAIN CORRUPTED");
#endif
	t->io = prq.io;
	t->cio = prq.cio;
	t->ctm = prq.ctm;
	t->sch = prq.next;
	LOWER_SPL(SPL_BOTTOM);
}

void KERNEL$THREAD_WAIT_FOR_LONGJMP(struct __jmp_buf *j)
{
	struct signal_chain *s;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO))) KERNEL$SUICIDE("KERNEL_THREAD_WAIT_FOR_LONGJMP AT SPL %08X", KERNEL$SPL);
	if (__unlikely(udata.current_thread->io != NULL)) KERNEL$SUICIDE("KERNEL_THREAD_WAIT_FOR_LONGJMP: IO PENGING ON CURRENT THREAD");
	if (__unlikely(udata.current_thread->cio != NULL)) KERNEL$SUICIDE("KERNEL_THREAD_WAIT_FOR_LONGJMP: CIO PENGING ON CURRENT THREAD");
	if (__unlikely(udata.current_thread->ctm != NULL)) KERNEL$SUICIDE("KERNEL_THREAD_WAIT_FOR_LONGJMP: TIMER PENGING ON CURRENT THREAD");
#endif
	RAISE_SPL(SPL_THREAD);
	again:
	while (__unlikely((s = udata.current_thread->sch) != NULL)) {
		if (s->io && __PTR_BELOW_SP(s->io, j) && s->io->fn) {
			KERNEL$CIO(s->io);
			KERNEL$SLEEP(1);
			goto again;
		}
		if (s->cio && __PTR_BELOW_SP(s->cio, j) && s->cio->fn) {
			KERNEL$CIO(s->cio);
			KERNEL$SLEEP(1);
			goto again;
		}
		if (s->ctm && __PTR_BELOW_SP(s->ctm, j) && s->ctm->tm.fn && s->ctm->tm.fn != (void *)1) {
			KERNEL$DEL_TIMER(&s->ctm->tm);
			s->ctm->tm.fn = (void *)1;
		}
		if (!__PTR_BELOW_SP(s, j)) break;
		udata.current_thread->sch = s->next;
	}
	LOWER_SPL(SPL_ZERO);
}

void SIGNAL_CANCEL_IORQ(int nointr)
{
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_BOTTOM))) KERNEL$SUICIDE("SIGNAL_CANCEL_IORQ AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_THREAD);
	t = udata.current_thread;
	if (!nointr && t->cio) KERNEL$CIO(t->cio);
	if (__unlikely(t->ctm != NULL)) {
		if (__likely(t->ctm->tm.fn != NULL) && __likely(t->ctm->tm.fn != (void *)1)) {
			KERNEL$DEL_TIMER(&t->ctm->tm);
			t->ctm->tm.fn = (void *)1;
		}
	}
	LOWER_SPL(SPL_BOTTOM);
}

char *KERNEL$ERROR_MSG(void)
{
	THREAD *t = udata.current_thread;
#if __DEBUG >= 1
	if (__unlikely(!t->rq->error)) KERNEL$SUICIDE("KERNEL$ERROR_MSG: NO ERROR PLACE");
#endif
	return t->rq->error;
}

void *KERNEL$TSR_IMAGE(void)
{
	THREAD *t = udata.current_thread;
	t->rq->tsr = 1;
	return KERNEL$DLRQ();
}

CWD *KERNEL$CWD(void)
{
	THREAD *t = udata.current_thread;
	return t->rq->cwd;
}

void SET_THREAD_CWD(CWD *cwd)
{
	THREAD *t;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), spl)))
		KERNEL$SUICIDE("SET_THREAD_CWD AT SPL %08X", spl);
	RAISE_SPL(SPL_THREAD);
	t = udata.current_thread;
	if (__likely(!t->rq->cwd)) t->rq->cwd = cwd;
	else KERNEL$FREE_CWD(cwd);
	LOWER_SPLX(spl);
}

int KERNEL$STDIN(void)
{
	THREAD *t = udata.current_thread;
	return t->rq->std_in;
}

int KERNEL$STDOUT(void)
{
	THREAD *t = udata.current_thread;
	return t->rq->std_out;
}

int KERNEL$STDERR(void)
{
	THREAD *t = udata.current_thread;
	return t->rq->std_err;
}

void *KERNEL$DLRQ(void)
{
	THREAD *t = udata.current_thread;
	return t->rq->dlrq;
}

THREAD_RQ *KERNEL$THREAD_RQ(void)
{
	THREAD *t = udata.current_thread;
	return t->rq;
}

int KERNEL$IS_INIT_THREAD(void)
{
	return udata.current_thread == udata.init_thread;
}

struct select_context **GET_THREAD_SELECT_STORAGE(void)
{
	THREAD *t = udata.current_thread;
	return &t->sctx;
}

void THREAD_INIT(void)
{
	THREAD *t;
	THREAD_RQ *r;
	char *e;
	udata.preempt_ast.fn = PREEMPT_AST;
	udata.blocked = 0;
	udata.preempt_ast_pending = 0;
	udata.preempt_timer_pending = 0;
	INIT_TIMER(&udata.preempt_timer);
	udata.preempt_timer.fn = PREEMPT_TIMER;
	t = malloc(sizeof(THREAD) + sizeof(THREAD_RQ) + __MAX_STR_LEN);
	if (__unlikely(!t)) KERNEL$SUICIDE("THREAD_INIT: CAN'T ALLOCATE MAIN THREAD");
	r = (THREAD_RQ *)(t + 1);
	e = (char *)(r + 1);
	r->error = e;
	e[0] = 0;
	r->cwd = NULL;
	r->std_in = 0;
	r->std_out = 1;
	r->std_err = 2;
	r->dlrq = NULL;
	r->status = RQS_PROCESSING;
	r->spawned = 1;
	memset(t, 0, sizeof(THREAD));
	t->rq = r;
	t->sleep = S_RUNNING;
	VOID_LIST_ENTRY(&t->sigwait_list);
	udata.current_thread = t;
	udata.init_thread = t;
	INIT_LIST(&udata.thread_ready);
	MD_THREAD_INIT(t);
	if (__likely(udata.cwd != NULL)) {
		CDRQ c;
		c.cwd = NULL;
		c.path = udata.cwd;
		/*__debug_printf("cwd: %s(%p) - %d\n", XCPT_ADDRESS, XCPT_ADDRESS, __alloc_size(XCPT_ADDRESS));*/
		SYNC_IO(&c, KERNEL$CD);
		t->rq->cwd = c.cwd;
		free(udata.cwd);
		udata.cwd = NULL;
	}
}

void _exit(int status)
{
	_exit_msg(status, udata.init_thread->rq->error[0] ? udata.init_thread->rq->error : NULL);
}

int sched_yield(void)
{
	if (LIST_EMPTY(&udata.thread_ready)) {
		KERNEL$SLEEP(0);
	} else {
		KERNEL$THREAD_YIELD();
	}
	return 0;
}
