#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/AC.H>
#include <SPAD/DL.H>
#include <SPAD/VM.H>
#include <ARCH/IO.H>

#include <SPAD/PCI.H>

io_t PCI$READ_IO_RESOURCE(pci_id_t id, unsigned res)
{
	__u32 r;
	char str[__MAX_STR_LEN];
	if (__unlikely(res >= 6)) KERNEL$SUICIDE("PCI$READ_IO_RESOURCE: INVALID RESOURCE %u", res);
	r = PCI$READ_CONFIG_DWORD(id, PCI_BASE_ADDRESS_0 + (res << 2));
	if (__unlikely(!r)) return 0;
	if (__unlikely((r & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_IO)) return 0;
	r &= PCI_BASE_ADDRESS_IO_MASK;
	if (__unlikely(r > IO_SPACE_LIMIT)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "PCI", "%s: INVALID I/O PORT %08X AT SLOT %u", PCI$ID(str, id), (unsigned)r, res);
		return 0;
	}
	return r;
}

__u64 PCI$READ_MEM_RESOURCE(pci_id_t id, unsigned res, unsigned *flags)
{
	__u32 r;
	unsigned mem_type;
	char str[__MAX_STR_LEN];
	if (__unlikely(res >= 6)) KERNEL$SUICIDE("PCI$READ_MEM_RESOURCE: INVALID RESOURCE %u", res);
	r = PCI$READ_CONFIG_DWORD(id, PCI_BASE_ADDRESS_0 + (res << 2));
	if (__unlikely(!r)) return 0;
	if (__unlikely((r & PCI_BASE_ADDRESS_SPACE) != PCI_BASE_ADDRESS_SPACE_MEMORY)) return 0;
	if (flags) *flags = r & PCI_BASE_ADDRESS_MEM_FLAGS;
	mem_type = r & PCI_BASE_ADDRESS_MEM_TYPE_MASK;
	switch (mem_type) {
		case PCI_BASE_ADDRESS_MEM_TYPE_32:
		case PCI_BASE_ADDRESS_MEM_TYPE_1M:
			return r & PCI_BASE_ADDRESS_MEM_MASK;
		case PCI_BASE_ADDRESS_MEM_TYPE_64:
			if (__unlikely(res == 5)) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, "PCI", "%s: 64-BIT RESOURCE AT LAST POSITION", PCI$ID(str, id));
				return 0;
			}
			return __make64(r & PCI_BASE_ADDRESS_MEM_MASK, PCI$READ_CONFIG_DWORD(id, PCI_BASE_ADDRESS_0 + ((res + 1) << 2)));
	}
	return 0;
}

unsigned PCI$READ_INTERRUPT_LINE(pci_id_t id)
{
	return PCI$READ_CONFIG_BYTE(id, PCI_INTERRUPT_LINE);
}

void *PCI$MAP_MEM_RESOURCE(pci_id_t id, unsigned res, int length)
{
	void *v;
	unsigned flags;
	__u64 paddr = PCI$READ_MEM_RESOURCE(id, res, &flags);
	if (!paddr) return NULL;
	v = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(paddr, length, flags & PCI_BASE_ADDRESS_MEM_PREFETCH ? PAT_WB : PAT_UC);
	if (__unlikely(__IS_ERR(v))) return NULL;
	__ADDL(&pci_info->mapped_amount, length);
	return v;
}

void PCI$UNMAP_MEM_RESOURCE(pci_id_t id, void *vaddr, int length)
{
	__SUBL(&pci_info->mapped_amount, length);
	if (__unlikely(pci_info->mapped_amount < 0))
		KERNEL$SUICIDE("PCI$UNMAP_MEM_RESOURCE: MAP UNDERFLOW %ld, UNMAPPING %p, %d", pci_info->mapped_amount, vaddr, length);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM((void *)vaddr, length);
}

char *PCI$TEST_LIST(__const__ void *table, pci_id_t id, unsigned long *driver_data)
{
	__const__ struct pci_id_s *list;
	__u16 vendor, device, subvendor, subdevice;
	__u32 dclass;
	vendor = PCI$READ_CONFIG_WORD(id, PCI_VENDOR_ID);
	for (list = table; list->vendor; list++) if (list->vendor == PCI_ANY_ID || list->vendor == vendor) goto found;
	return NULL;
	found:
	device = PCI$READ_CONFIG_WORD(id, PCI_DEVICE_ID);
	subvendor = PCI$READ_CONFIG_WORD(id, PCI_SUBSYSTEM_VENDOR_ID);
	subdevice = PCI$READ_CONFIG_WORD(id, PCI_SUBSYSTEM_ID);
	dclass = PCI$READ_CONFIG_DWORD(id, PCI_REVISION_ID);
	/*if (device != 0xFFFF)
		__debug_printf("PCI DEVICE %04X (%04X %04X %04X %04X %04X)  ", id, vendor, device, subvendor, subdevice, dclass);*/
	for (list = table; list->vendor; list++) {
		if ((list->vendor == PCI_ANY_ID || list->vendor == vendor) &&
		    (list->device == PCI_ANY_ID || list->device == device) &&
		    (list->subvendor == PCI_ANY_ID || list->subvendor == subvendor) &&
		    (list->subdevice == PCI_ANY_ID || list->subdevice == subdevice) &&
		    (list->dclass == PCI_ANY_ID || list->dclass == (dclass & list->dclass_mask))) {
			/*__debug_printf("PCI DEVICE %04X (%04X %04X %04X %04X %04X) MATCH\n", id, vendor, device, subvendor, subdevice, dclass);*/
			if (driver_data) *driver_data = list->driver_data;
			return list->name ? list->name : "";
		}
	}
	return NULL;
}

int PCI$FIND_DEVICE(__const__ void *table, pci_id_t rq_id, pci_id_t rq_mask, int skip, char *(*test)(__const__ void *table, pci_id_t id, unsigned long *driver_data), pci_id_t *id, char **name, unsigned long *driver_data, int nolock)
{
	char *nm;
	pci_id_t i = 0;
	int spl = KERNEL$SPL;
	if (!nolock) if (SPLX_BELOW(SPL_X(SPL_DEV), spl)) KERNEL$SUICIDE("PCI$FIND_DEVICE AT SPL %08lX", (unsigned long)KERNEL$SPL);
	while (1) {
		__u16 vi;
		/*SPIN_LOCK_RAISE_SPL(&pci_info->lock, SPL_DEV);*/
		if (!nolock) RAISE_SPL(SPL_DEV);
		if (i >= pci_info->n_buses * 256) {
			/*SPIN_UNLOCK_LOWER_SPLX(&pci_info->lock, spl);*/
			if (!nolock) LOWER_SPLX(spl);
			break;
		}
		vi = PCI$READ_CONFIG_WORD(i, PCI_VENDOR_ID);
		if (__unlikely(vi == 0xffff)) {
			if (!(i & 7)) goto skip_this_dev;
			goto skip_this_fn;
		}
		if ((i & rq_mask) == (rq_id & rq_mask) && (nolock || !PCI_DEVICE_IS_USED(i)) && (nm = test(table, i, driver_data)) && (skip >= 0 && !skip--)) {
			if (!nolock) {
				PCI_DEVICE_SET_USED(i);
			}
			if (id) *id = i;
			if (name) *name = nm;
			/*SPIN_UNLOCK_LOWER_SPLX(&pci_info->lock, spl);*/
			if (!nolock) LOWER_SPLX(spl);
			return 0;
		}
		if (!(i & 7)) {
			__u8 h = PCI$READ_CONFIG_BYTE(i, PCI_HEADER_TYPE);
			if (!(h & PCI_HEADER_TYPE_MULTIFUNC)) {
				skip_this_dev:
				i += 8;
			} else {
				skip_this_fn:
				i++;
			}
		} else {
			goto skip_this_fn;
		}
		/*SPIN_UNLOCK_LOWER_SPLX(&pci_info->lock, spl);*/
		if (!nolock) LOWER_SPLX(spl);
		if (!i) break;
	}
	return -1;
}

void PCI$FREE_DEVICE(pci_id_t id)
{
	char str[__MAX_STR_LEN];
	int spl = KERNEL$SPL;
	if (SPLX_BELOW(SPL_X(SPL_DEV), spl)) KERNEL$SUICIDE("PCI$FREE_DEVICE AT SPL %08lX", (unsigned long)KERNEL$SPL);
	/*SPIN_LOCK_RAISE_SPL(&pci_info->lock, SPL_DEV);*/
	RAISE_SPL(SPL_DEV);
	if (!PCI_DEVICE_IS_USED(id)) KERNEL$SUICIDE("PCI$FREE_DEVICE: FREEING ALREADY FREE DEVICE: %s", PCI$ID(str, id));
	PCI_DEVICE_SET_FREE(id);
	/*SPIN_UNLOCK_LOWER_SPLX(&pci_info->lock, spl);*/
	LOWER_SPLX(spl);
}

#define CMD_ENA	(PCI_COMMAND_IO | PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER | PCI_COMMAND_SPECIAL | PCI_COMMAND_INVALIDATE | PCI_COMMAND_VGA_PALETTE | PCI_COMMAND_PARITY | PCI_COMMAND_WAIT | PCI_COMMAND_SERR | PCI_COMMAND_FAST_BACK)
#define CMD_DIS	(PCI_COMMAND_INT_DIS)

void PCI$ENABLE_DEVICE(pci_id_t id, unsigned command_mask)
{
	unsigned cmd = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
	if (__unlikely(command_mask & ~cmd & CMD_ENA) || __unlikely(command_mask & cmd & CMD_DIS)) {
		cmd |= command_mask & CMD_ENA;
		cmd &= ~(command_mask & CMD_DIS);
		PCI$WRITE_CONFIG_WORD(id, PCI_COMMAND, cmd);
	}
}

int PCI$FIND_CAPABILITY(pci_id_t id, int start, __u8 cap)
{
	int count = 0;
	__u8 pos;
	if (start < 0) {
		__u16 status = PCI$READ_CONFIG_WORD(id, PCI_STATUS);
		if (__unlikely(!(status & PCI_STATUS_CAP_LIST))) return -1;
		start = PCI$READ_CONFIG_BYTE(id, PCI_CAPABILITY_LIST);
	}
	pos = start;
	while (pos) {
		char str[__MAX_STR_LEN];
		__u16 id_next;
		pos &= ~3;
		id_next = PCI$READ_CONFIG_WORD(id, pos);
		if ((id_next & 0xff) == cap) return pos;
		if (pos > 0 && pos < 0x40) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "PCI", "%s: CAPABILITIES IN RESERVED AREA (%02X)", PCI$ID(str, id), pos);
			return -1;
		}
		if (++count > 0xc0 / 4) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "PCI", "%s: LOOP IN CAPABILITIES AT %02X, START %02X", PCI$ID(str, id), pos, start);
			return -1;
		}
	}
	return -1;
}

/*static SMPRQ s;*/
static DLINITRQ *dlrq;

extern AST_STUB LOCK_ALLOCATED;
extern AST_STUB LOCK_FREED;

DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	dlrq = RQ;
/*
	s.fn = &LOCK_ALLOCATED;
	s.size = sizeof(struct pci_info);
	s.name = "PCI.DLL$PCI_SHARED";
	s.ctor = &pci_ctor;
	s.dtor = NULL;
	RETURN_IORQ(&s, &KERNEL$GRAB_SHARED_AREA);
}


DECL_AST(LOCK_ALLOCATED, SPL_DEV, SMPRQ)
{
	if (s.status < 0) {
		_snprintf(dlrq->error, __MAX_STR_LEN, "FAILED TO ALLOCATE SHARED LOCK");
		dlrq->status = RQ->status;
		RETURN_AST(dlrq);
	}
	pci_info = s.ptr;*/
	pci_ctor(pci_info = &global_pci_info);
	/*SPIN_LOCK_RAISE_SPL(&pci_info->lock, SPL_DEV);*/
	RAISE_SPL(SPL_DEV);
	if (pci_info->bios_entry == -1UL) {
		struct bios_regs r;
		if (BIOS32$ENTRY(BIOS32_PCI_SERVICE, &pci_info->bios_entry) || pci_info->bios_entry == -1UL) {
			_snprintf(dlrq->error, __MAX_STR_LEN, "CAN'T FIND PCI BIOS");
			err:
			/*SPIN_UNLOCK_LOWER_SPL(&pci_info->lock, SPL_DEV);*/
			dlrq->status = -ENODEV;
			/*
			s.fn = &LOCK_FREED_CANCEL;
			RETURN_IORQ(&s, &KERNEL$RELEASE_SHARED_AREA);*/
			RETURN_AST(dlrq);
		}
		memset(&r, 0, sizeof r);
		r.eax = PCIBIOS_PCI_BIOS_PRESENT;
		if (pci_call_bios(&r)) {
			_snprintf(dlrq->error, __MAX_STR_LEN, "PCI BIOS INSTALLATION CHECK FAILED");
			goto err;
		}
		if (r.edx != PCI_SIGNATURE) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, "PCI", "PCI BIOS HAS BAD SIGNATURE: %08X", (int)r.edx);
			_snprintf(dlrq->error, __MAX_STR_LEN, "PCI BIOS HAS BAD SIGNATURE");
			goto err;
		}
		pci_info->bios_version = r.ebx & 0xffff;
		pci_info->n_buses = (r.ecx & 0xff) + 1;
	}
	/*SPIN_UNLOCK_LOWER_SPL(&pci_info->lock, SPL_DEV);*/
	LOWER_SPL(SPL_DEV);
	dlrq->status = 0;
	RETURN_AST(dlrq);
}

DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	pci_id_t i = 0;
	char str[__MAX_STR_LEN];
	while (1) {
		if (i >= pci_info->n_buses * 256) break;
		if (PCI_DEVICE_IS_USED(i)) KERNEL$SUICIDE("PCI: PCI DEVICE WAS NOT RELEASED BY THE DRIVER: %s", PCI$ID(str, i));
		i++;
		if (!i) break;
	}
	if (pci_info->mapped_amount != 0)
		KERNEL$SUICIDE("PCI: MAPPED MEMORY LEAKED: %ld BYTES", pci_info->mapped_amount);
	dlrq = RQ;
/*
	s.fn = &LOCK_FREED;
	RETURN_IORQ(&s, &KERNEL$RELEASE_SHARED_AREA);
}

DECL_AST(LOCK_FREED, SPL_DEV, SMPRQ)
{*/
	dlrq->status = 0;
	RETURN_AST(dlrq);
}

/*
DECL_AST(LOCK_FREED_CANCEL, SPL_DEV, SMPRQ)
{
	RETURN_AST(dlrq);
}
*/


