#include <SPAD/THREAD.H>
#include <UNISTD.H>
#include <ARCH/BARRIER.H>
#include <ERRNO.H>

#include <PTHREAD.H>
#include "PTHREAD.H"

static __finline__ void set_cancelability(pthread_t p)
{
	p->cancel_enable = ~p->cancelstate & p->canceltype;
}

static void do_cancel(void *p_, unsigned n_)
{
	pthread_t p = p_;
	p->cancel_pending = 1;
	if (!p->cancel_enable) return;
	LOWER_SPL(SPL_ZERO);
	pthread_exit(PTHREAD_CANCELED);
}

int __pthread_cancel(pthread_t p)
{
	KERNEL$CALL_IN_THREAD(&p->rq, do_cancel, p, 0);
	return 0;
}

int pthread_setcancelstate(int state, int *oldstate)
{
	pthread_t p;
	if (__unlikely((unsigned)state > PTHREAD_CANCEL_DISABLE)) return EINVAL;
	p = pthread_self();
	if (oldstate) *oldstate = p->cancelstate;
	p->cancelstate = state;
	set_cancelability(p);
	return 0;
}

int pthread_setcanceltype(int type, int *oldtype)
{
	pthread_t p;
	if (__unlikely((unsigned)type > PTHREAD_CANCEL_ASYNCHRONOUS)) return EINVAL;
	p = pthread_self();
	if (oldtype) *oldtype = p->canceltype;
	p->canceltype = type;
	set_cancelability(p);
	return 0;
}

void pthread_testcancel(void)
{
	__cancel_test();
}

static void cancel_start(void)
{
	pthread_t p = pthread_self();
	if (__unlikely(!p)) return;	/* called from non-pthread thread */
	if (__unlikely(p->cancelstate == PTHREAD_CANCEL_DISABLE)) return;
	p->cancel_enable = 1;
	if (__unlikely(p->cancel_pending)) {
		LOWER_SPL(SPL_ZERO);
		pthread_exit(PTHREAD_CANCELED);
	}
}

static void cancel_end(void)
{
	pthread_t p = pthread_self();
	if (__unlikely(!p)) return;	/* called from non-pthread thread */
	set_cancelability(p);
}

static void cancel_test(void)
{
	pthread_t p = pthread_self();
	if (__unlikely(!p)) return;	/* called from non-pthread thread */
	if (__unlikely(p->cancelstate == PTHREAD_CANCEL_DISABLE)) return;
	if (__unlikely(p->cancel_pending)) {
		LOWER_SPL(SPL_ZERO);
		pthread_exit(PTHREAD_CANCELED);
	}
}

static int need_cancel = 0;

void setup_cancel(void)
{
	if (__unlikely(need_cancel) && __likely(!KERNEL$__NEEDCANCEL)) {
		KERNEL$__CANCEL_START_FN = cancel_start;
		KERNEL$__CANCEL_END_FN = cancel_end;
		KERNEL$__CANCEL_TEST_FN = cancel_test;
		__barrier();
		KERNEL$__NEEDCANCEL = 1;
	}
}

void __pthread_setup_cancel(void)
{
	need_cancel = 1;
}
