#include <STDLIB.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <ARCH/BITOPS.H>
#include <ARCH/BARRIER.H>
#include <SYS/TIME.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYNC.H>
#include <UNISTD.H>

#include "PTHREAD.H"
#include <PTHREAD.H>

int pthread_condattr_init(pthread_condattr_t *a)
{
	struct __pthread_condattr_s *as;
	if (__unlikely(!(as = malloc(sizeof(struct __pthread_condattr_s))))) return ENOMEM;
	a->__p = as;
	as->pshared = PTHREAD_PROCESS_PRIVATE;
	return 0;
}

int pthread_condattr_destroy(pthread_condattr_t *a)
{
	if (__unlikely(!a->__p)) KERNEL$SUICIDE("pthread_condattr_destroy: uninitialized attribute");
	free(a->__p);
	a->__p = NULL;
	return 0;
}

int pthread_condattr_getpshared(const pthread_condattr_t *a, int *s)
{
	*s = a->__p->pshared;
	return 0;
}

int pthread_condattr_setpshared(pthread_condattr_t *a, int s)
{
	if (__unlikely((unsigned)s > PTHREAD_PROCESS_SHARED)) return EINVAL;
	a->__p->pshared = s;
	return 0;
}


int pthread_cond_init(pthread_cond_t *c, const pthread_condattr_t *a)
{
	if (__likely(!a) || __unlikely(a->__p->pshared == PTHREAD_PROCESS_PRIVATE)) {
		struct __pthread_cond_s *as;
		if (__unlikely(!(as = malloc(sizeof(struct __pthread_cond_s))))) return errno;
		c->__p = as;
		WQ_INIT(&as->wq, "PTHREAD$COND_WQ");
		__barrier();
		c->__pshared = PTHREAD_PROCESS_PRIVATE;
	} else {
		c->__shared_cnt = 0;
		__barrier();
		c->__pshared = PTHREAD_PROCESS_SHARED;
	}
	return 0;
}

static void cond_init(pthread_cond_t *c)
{
	if (__likely(!__CMPXCHGI(&c->__pshared, _PTHREAD_PROCESS_CREATE, _PTHREAD_PROCESS_CREATING))) {
		pthread_cond_init(c, NULL);
		return;
	}
	while (__unlikely(c->__pshared == _PTHREAD_PROCESS_CREATING)) KERNEL$SLEEP(1);
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE) || __likely(c->__pshared == PTHREAD_PROCESS_SHARED)) return;
	KERNEL$SUICIDE("cond_init: uninitialized conditional variable");
}

int pthread_cond_destroy(pthread_cond_t *c)
{
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(!c->__p)) goto uni;
		WQ_WAKE_ALL(&c->__p->wq);
		free(c->__p);
	} else if (__likely(c->__pshared == PTHREAD_PROCESS_SHARED) || __likely(c->__pshared == _PTHREAD_PROCESS_CREATE)) {
	} else uni: KERNEL$SUICIDE("pthread_cond_destroy: uninitialized condition variable");
	c->__pshared = -256;
	return 0;
}

int pthread_cond_broadcast(pthread_cond_t *c)
{
	again:
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (WQ_WAKE_ALL(&c->__p->wq)) if (__likely(KERNEL$SPL == SPL_X(SPL_ZERO))) KERNEL$THREAD_YIELD();
	} else if (__likely(c->__pshared == PTHREAD_PROCESS_SHARED)) {
		long d = c->__shared_cnt;
		__CMPXCHGI((int *)&c->__shared_cnt, d, d + 1);
	} else {
		cond_init(c);
		goto again;
	}
	return 0;
}

int pthread_cond_signal(pthread_cond_t *c)
{
	again:
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (WQ_WAKE_ONE(&c->__p->wq)) if (__likely(KERNEL$SPL == SPL_X(SPL_ZERO))) KERNEL$THREAD_YIELD();
	} else if (__likely(c->__pshared == PTHREAD_PROCESS_SHARED)) {
		long d = c->__shared_cnt;
		__CMPXCHGI((int *)&c->__shared_cnt, d, d + 1);
	} else {
		cond_init(c);
		goto again;
	}
	return 0;
}

__u64 gettime(void)
{
	struct timeval tv;
	if (__unlikely(gettimeofday(&tv, NULL))) return 0;
	return (__u64)tv.tv_sec * 1000000 + (unsigned long)tv.tv_usec;
}

int pthread_cond_wait(pthread_cond_t *c, pthread_mutex_t *m)
{
	again:
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		RAISE_SPL(SPL_DEV);
		if (m) pthread_mutex_unlock(m);
		__cancel_in(WQ_WAIT_SYNC(&c->__p->wq));
		LOWER_SPL(SPL_ZERO);
	} else if (__likely(c->__pshared == PTHREAD_PROCESS_SHARED)) {
		long d = c->__shared_cnt;
		if (m) pthread_mutex_unlock(m);
		__cancel_in(while (c->__shared_cnt == d) KERNEL$SLEEP(1));
	} else {
		cond_init(c);
		goto again;
	}
	if (m) pthread_mutex_lock(m);
	return 0;
}

struct wait_struct {
	IORQ io;
	TIMER timer;
	int timed;
};

DECL_IOCALL(wq_wait, SPL_TIMER + 1, IORQ)
{
	WQ_WAIT((WQ *)RQ->tmp3, RQ, KERNEL$SUCCESS);
	RETURN;
}

static void wq_timer(TIMER *t)
{
	struct wait_struct *w;
	LOWER_SPL(SPL_TIMER);
	w = GET_STRUCT(t, struct wait_struct, timer);
	w->timed = 1;
	KERNEL$CIO(&w->io);
}

static void wq_clean(void *w_)
{
	struct wait_struct *w = w_;
	RAISE_SPL(SPL_TIMER);
	if (!w->timed) KERNEL$DEL_TIMER(&w->timer);
	LOWER_SPL(SPL_ZERO);
	/* we don't need to take care about w.io, because longjmp already killed that */
}

int pthread_cond_timedwait(pthread_cond_t *c, pthread_mutex_t *m, const struct timespec *t)
{
	if (__unlikely(__ts_invalid(t))) return EINVAL;
	again:
	if (__likely(c->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		struct wait_struct w;
		INIT_TIMER(&w.timer);
		w.timer.fn = wq_timer;
		w.timed = 0;
		RAISE_SPL(SPL_TIMER);
		if (m) pthread_mutex_unlock(m);
		KERNEL$SET_TIMER((__u64)t->tv_sec * JIFFIES_PER_SECOND + (unsigned)t->tv_nsec / (1000000000 / JIFFIES_PER_SECOND), &w.timer);
		w.io.tmp3 = (unsigned long)&c->__p->wq;
		pthread_cleanup_push(wq_clean, &w);
		__cancel_in(SYNC_IO(&w.io, wq_wait));
		pthread_cleanup_pop(1);
		if (w.io.status) {
			LOWER_SPL(SPL_ZERO);
			if (m) pthread_mutex_lock(m);
			return ETIMEDOUT;
		}
	} else if (__likely(c->__pshared == PTHREAD_PROCESS_SHARED)) {
		long d = c->__shared_cnt;
		__u64 tm, rm;
		if (m) pthread_mutex_unlock(m);
		tm = gettime();
		rm = (__u64)t->tv_sec * 1000000 + (unsigned)t->tv_nsec / 1000;
		if (__likely(c->__shared_cnt == d)) while (1) {
			__cancel_in(KERNEL$SLEEP(1));
			if (c->__shared_cnt != d) break;
			if (__unlikely(gettime() - tm >= rm)) {
				if (m) pthread_mutex_lock(m);
				return ETIMEDOUT;
			}
		}
	} else {
		cond_init(c);
		goto again;
	}
	if (m) pthread_mutex_lock(m);
	return 0;
}

