#include <STDLIB.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <SPAD/THREAD.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <ARCH/SETUP.H>
#include <ARCH/BARRIER.H>
#include <UNISTD.H>

#include <PTHREAD.H>
#include "PTHREAD.H"

static size_t default_stack_size = 0;
static THREAD_RQ *init_thread_1;
static struct __pthread_s init_thread;
static int init_thread_initialized = 0;

size_t __finline__ get_default_stack_size(void)
{
	if (__unlikely(!default_stack_size)) default_stack_size = KERNEL$GET_THREAD_STACK_SIZE();
	return default_stack_size;
}

static void initialize_init_thread(void)
{
	init_thread_1 = KERNEL$THREAD_RQ();
	memset(&init_thread, 0, sizeof(struct __pthread_s));
	init_thread.cancel_enable = 0;
	init_thread.cancelstate = PTHREAD_CANCEL_ENABLE;
	init_thread.canceltype = PTHREAD_CANCEL_DEFERRED;
	init_thread.detached = PTHREAD_CREATE_JOINABLE;
	WQ_INIT(&init_thread.joinwait, "PTHREAD$JOINWAIT(init)");
	init_thread_initialized = 1;
	setup_cancel();
}

static long thread_main(void *p);
extern AST_STUB thread_finished;

int pthread_create(pthread_t *tp, const pthread_attr_t *a, void *(*fn)(void *), void *data)
{
	static char error_dummy[__MAX_STR_LEN];

	struct __pthread_s *p;
	if (__unlikely(!init_thread_initialized)) initialize_init_thread();
	if (__unlikely(!(p = calloc(1, sizeof(struct __pthread_s))))) return ENOMEM;
	if (__likely(tp != NULL)) *tp = p;
	p->fn = fn;
	p->data = data;
	WQ_INIT(&p->joinwait, "PTHREAD$JOINWAIT");
	p->rq.fn = thread_finished;
	p->rq.thread_main = thread_main;
	/*p->rq.p = NULL;*/
	p->rq.error = error_dummy;
	p->rq.cwd = KERNEL$CWD();
	/*p->rq.std_in = 0;*/
	p->rq.std_out = 1;
	p->rq.std_err = 2;
	/*p->cancel_enable = 0;*/
	p->cancelstate = PTHREAD_CANCEL_ENABLE;
	p->canceltype = PTHREAD_CANCEL_DEFERRED;
	sigprocmask(SIG_BLOCK, NULL, &p->init_sigmask);
	RAISE_SPL(SPL_DEV);
	if (__unlikely(a != NULL)) {
		p->detached = a->__p->detachstate;
		KERNEL$SET_THREAD_STACK_SIZE(a->__p->stacksize);
	} else {
		p->detached = PTHREAD_CREATE_JOINABLE;
		KERNEL$SET_THREAD_STACK_SIZE(get_default_stack_size());
	}
	CALL_IORQ(&p->rq, KERNEL$THREAD);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static long thread_main(void *dummy)
{
	pthread_t p = pthread_self();
	sigprocmask(SIG_SETMASK, &p->init_sigmask, NULL);
	if (!setjmp(p->terminate)) {
		p->exit_value = p->fn(p->data);
	}
	p = pthread_self();
	while (__unlikely(p->cleanup_stack != NULL)) pthread_cleanup_pop(1);
	if (__unlikely(p->n_specific)) destroy_specific(p);
	return 0;
}

DECL_AST(thread_finished, SPL_DEV, THREAD_RQ)
{
	pthread_t p = GET_STRUCT(RQ, struct __pthread_s, rq);
	p->finished = 1;
	WQ_WAKE_ALL(&p->joinwait);
	if (p->detached != PTHREAD_CREATE_JOINABLE) free(p);
	RETURN;
}

void pthread_exit(void *v)
{
	pthread_t p;
	sigset_t m;
	sigfillset(&m);
	sigprocmask(SIG_SETMASK, &m, NULL);
	p = pthread_self();
	p->exit_value = v;
	if (__unlikely(p == &init_thread)) {
		init_thread.finished = 1;
		WQ_WAKE_ALL(&init_thread.joinwait);
		while (1) {
			static WQ_DECL(wq, "PTHREAD$INFINITE_WAIT");
			WQ_WAIT_SYNC(&wq);
		}
	}
	longjmp(p->terminate, 1);
}

int pthread_join(pthread_t p, void **value_ptr)
{
	if (__unlikely(p == pthread_self())) return EDEADLK;
#if __DEBUG >= 1
	if (__unlikely(p->detached != PTHREAD_CREATE_JOINABLE))
		KERNEL$SUICIDE("pthread_join: invalid thread id or thread not joinable");
#endif
	RAISE_SPL(SPL_DEV);
	__cancel_in(while (__unlikely(!p->finished)) WQ_WAIT_SYNC(&p->joinwait));
#if __DEBUG >= 1
	if (__unlikely(p->detached != PTHREAD_CREATE_JOINABLE))
		KERNEL$SUICIDE("pthread_join: thread already detached");
	p->detached = PTHREAD_CREATE_DETACHED;
#endif
	if (value_ptr) *value_ptr = p->exit_value;
	LOWER_SPL(SPL_ZERO);
	if (__likely(p != &init_thread)) free(p);
	return 0;
}

pthread_t pthread_self(void)
{
	THREAD_RQ *rq;
	pthread_t p;
	if (__unlikely(!init_thread_initialized)) initialize_init_thread();
	rq = KERNEL$THREAD_RQ();
	if (__unlikely(rq == init_thread_1)) return &init_thread;
	if (__unlikely(rq->thread_main != thread_main)) {
		return NULL;
	}
	p = GET_STRUCT(rq, struct __pthread_s, rq);
	return p;
}

int pthread_detach(pthread_t p)
{
	p->detached = PTHREAD_CREATE_DETACHED;
	return 0;
}

int pthread_equal(pthread_t p1, pthread_t p2)
{
	return p1 == p2;
}

void pthread_cleanup_push(void (*fn)(void *), void *data)
{
	struct cleanup_entry *e;
	pthread_t p;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_MALLOC), spl)))
		KERNEL$SUICIDE("pthread_cleanup_push AT SPL %08X", spl);
#endif
	RAISE_SPLX(SPLX_MAX(spl, SPL_X(SPL_BOTTOM)));
	if (__unlikely(!(e = malloc(sizeof(struct cleanup_entry))))) {
		LOWER_SPLX(spl);
		return;
	}
	e->fn = fn;
	e->data = data;
	p = pthread_self();
	e->next = p->cleanup_stack;
	p->cleanup_stack = e;
	LOWER_SPLX(spl);
}

void pthread_cleanup_pop(int exe)
{
	pthread_t p;
	struct cleanup_entry *e;
	void (*fn)(void *);
	void *data;
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_MALLOC), spl)))
		KERNEL$SUICIDE("pthread_cleanup_pop AT SPL %08X", spl);
#endif
	RAISE_SPLX(SPLX_MAX(spl, SPL_X(SPL_BOTTOM)));
	p = pthread_self();
	e = p->cleanup_stack;
#if __DEBUG >= 1
	if (__unlikely(!e)) KERNEL$SUICIDE("pthread_cleanup_pop: stack is empty");
#endif
	p->cleanup_stack = e->next;
	fn = e->fn;
	data = e->data;
	free(e);
	LOWER_SPLX(spl);
	if (exe) fn(data);
}

