#include <SPAD/LIBC.H>
#include <SPAD/DL.H>
#include <SPAD/LIST.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/HASH.H>
#include <SPAD/DEV.H>
#include <SPAD/READDIR.H>
#include <DIRENT.H>
#include <SYS/STAT.H>
#include <SPAD/IOCTL.H>
#include <SYS/IOCTL.H>
#include <STDLIB.H>
#include <VALUES.H>
#include <PATHS.H>

#include <SPAD/SHELLM.H>
#include "SHELL.H"

static size_t DO_LOCAL(CTX *ctx, char *name);

extern AST_STUB SHELL_IMAGE_EXITED;

DECL_AST(SHELL_IMAGE_COMMAND, SPL_SHELL, CTX)
{
	int device = RQ->argv[0][0] == 'D' || RQ->argv[0][0] == 'd';
	if (__unlikely(!RQ->argv[1])) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%s: SYNTAX ERROR", !device ? "IMAGE" : "DEVICE");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(strlen(RQ->argv[1]) >= __MAX_STR_LEN)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%s: TOO LONG FILE NAME", !device ? "IMAGE" : "DEVICE");
		RQ->return_val = -ENAMETOOLONG;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.imgrq.argv = RQ->argv + 1;
	RQ->u.imgrq.cwd = RQ->cwd;
	RQ->u.imgrq.std_in = RQ->ctty_in;
	RQ->u.imgrq.std_out = RQ->ctty_out;
	RQ->u.imgrq.std_err = RQ->ctty_err;
	RQ->u.imgrq.device = device;
	RQ->u.imgrq.fn = &SHELL_IMAGE_EXITED;
	strcpy(RQ->u.imgrq.filename, RQ->argv[1]);
	RETURN_IORQ_CANCELABLE(&RQ->u.imgrq, KERNEL$DL_LOAD_IMAGE, RQ->rq);
}

DECL_AST(SHELL_IMAGE_EXITED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	RQ->return_val = RQ->u.imgrq.status;
	strcpy(RQ->return_msg, RQ->u.imgrq.error);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_DCTL_EXITED;

DECL_AST(SHELL_DCTL_COMMAND, SPL_SHELL, CTX)
{
	int unload = RQ->argv[0][0] == 'U' || RQ->argv[0][0] == 'u';
	if (__unlikely(!RQ->argv[1])) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%s: SYNTAX ERROR", !unload ? "DCTL" : "UNLOAD");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(strlen(RQ->argv[1]) >= __MAX_STR_LEN)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%s: TOO LONG FILE NAME", !unload ? "DCTL" : "UNLOAD");
		RQ->return_val = -ENAMETOOLONG;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.dctlrq.unload = unload;
	RQ->u.dctlrq.name = RQ->argv[1];
	RQ->u.dctlrq.argv = RQ->argv + 1;
	RQ->u.dctlrq.cwd = RQ->cwd;
	RQ->u.dctlrq.std_in = RQ->ctty_in;
	RQ->u.dctlrq.std_out = RQ->ctty_out;
	RQ->u.dctlrq.std_err = RQ->ctty_err;
	RQ->u.imgrq.fn = &SHELL_DCTL_EXITED;
	RETURN_IORQ_CANCELABLE(&RQ->u.imgrq, KERNEL$DCTL, RQ->rq);
}

DECL_AST(SHELL_DCTL_EXITED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	RQ->return_val = RQ->u.dctlrq.status;
	strcpy(RQ->return_msg, RQ->u.dctlrq.error);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_CTTY_PARSE;
extern AST_STUB SHELL_CTTY_OPEN;
extern AST_STUB SHELL_CTTY_GOTSTATUS;
extern AST_STUB SHELL_OPEN_CTTY;
extern AST_STUB SHELL_CTTY_CLOSED;

DECL_AST(SHELL_CTTY_COMMAND, SPL_SHELL, CTX)
{
	RQ->itmp2 = 0;
	CALL_SHELL_FN(RQ, SHELL_CTTY_PARSE);
}

DECL_AST(SHELL_CTTY_PARSE, SPL_SHELL, CTX)
{
	char *in = NULL, *out = NULL, *err = NULL, *all = NULL;
	int outov = 0, errov = 0;
	char **argv = RQ->argv;
	int state = 0;
	struct __param_table params[] = {
		"IN", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"OUT", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"OUTOVER", __PARAM_BOOL_X, ~0, 1, NULL,
		"OUTOVER", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"ERR", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"ERROVER", __PARAM_BOOL_X, ~0, 1, NULL,
		"ERROVER", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		NULL, 0, 0, 0, NULL,
	};
	if (__unlikely(++RQ->itmp2 == 4)) RETURN_TO_SHELL(RQ);
	params[0].__p = &in;
	params[1].__p = &out;
	params[2].__p = &outov;
	params[3].__p = &out;
	params[4].__p = &err;
	params[5].__p = &errov;
	params[6].__p = &err;
	params[7].__p = &all;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL))) {
		badsyn:
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CTTY: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (all) {
		if (__unlikely(in != NULL) || __unlikely(out != NULL) || __unlikely(err != NULL)) goto badsyn;
	} else {
		if (__unlikely(!in) && __unlikely(!out) && __unlikely(!err)) goto badsyn;
		if (RQ->itmp2 <= 1 && in) RQ->itmp2 = 1, all = in;
		else if (RQ->itmp2 <= 2 && out) RQ->itmp2 = 2, all = out;
		else if (RQ->itmp2 <= 3 && err) RQ->itmp2 = 3, all = err;
		else RETURN_TO_SHELL(RQ);
	}
	if (RQ->itmp2 == 1) {
		RQ->u.openrq.flags = O_RDONLY;
	} else {
		RQ->u.openrq.flags = O_WRONLY | O_CREAT | O_APPEND;
		if (__likely(RQ->itmp2 == 2)) {
			if (__unlikely(outov)) RQ->u.openrq.flags |= O_TRUNC;
		} else {
			if (__unlikely(errov)) RQ->u.openrq.flags |= O_TRUNC;
		}
	}
	RQ->u.openrq.cwd = RQ->cwd;
	RQ->u.openrq.path = all;
	RQ->u.openrq.fn = &SHELL_CTTY_OPEN;
	if (__unlikely(!*all)) RETURN_AST(&RQ->u.openrq);
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, &KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_CTTY_OPEN, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(!*RQ->u.openrq.path)) RQ->u.openrq.status = -1;
	else if (__unlikely(RQ->u.openrq.status < 0)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CTTY: OPEN: %s (%s)", strerror(-RQ->u.openrq.status), RQ->u.openrq.path);
		RQ->return_val = RQ->u.openrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp1 = RQ->u.openrq.status;
	RQ->u.ioctlrq.fn = &SHELL_CTTY_GOTSTATUS;
	RQ->u.ioctlrq.h = RQ->itmp1;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_GET_TERM_FLAGS;
	RQ->u.ioctlrq.param = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_CTTY_GOTSTATUS, SPL_SHELL, CTX)
{
	int *ctty, *cttyo;
	char *cttyt;
	char ctty_t;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	/*__debug_printf("got flags (%x:%s): %x\n", RQ->u.ioctlrq.h, KERNEL$HANDLE_PATH(RQ->u.ioctlrq.h), RQ->u.ioctlrq.status);*/
	ctty_t = RQ->u.ioctlrq.status >= 0;
	ctty = RQ->itmp2 == 1 ? &RQ->ctty_in : RQ->itmp2 == 2 ? &RQ->ctty_out : &RQ->ctty_err;
	cttyo = RQ->itmp2 == 1 ? &RQ->ctty_in_orig : RQ->itmp2 == 2 ? &RQ->ctty_out_orig : &RQ->ctty_err_orig;
	cttyt = RQ->itmp2 == 1 ? &RQ->ctty_in_t : RQ->itmp2 == 2 ? &RQ->ctty_out_t : &RQ->ctty_err_t;
	if (*ctty == -1 || *ctty == *cttyo) {
		*ctty = RQ->itmp1;
		*cttyo = -1;
		*cttyt = ctty_t;
		CALL_SHELL_FN(RQ, SHELL_CTTY_PARSE);
	}
	RQ->u.closerq.fn = &SHELL_CTTY_CLOSED;
	RQ->u.closerq.h = *ctty;
	*ctty = RQ->itmp1;
	*cttyo = -1;
	*cttyt = ctty_t;
	RETURN_IORQ_CANCELABLE(&RQ->u.closerq, &KERNEL$CLOSE, RQ->rq);
}

DECL_AST(SHELL_CTTY_CLOSED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.closerq.h != -1)) KERNEL$FAST_CLOSE(RQ->u.closerq.h);
	CALL_SHELL_FN(RQ, SHELL_CTTY_PARSE);
}


extern AST_STUB SHELL_DTTY_OPEN;
extern AST_STUB SHELL_DTTY_DONE;

DECL_AST(SHELL_DTTY_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || (__unlikely(RQ->argv[2] != NULL))) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DTTY: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.flags = O_RDWR;
	RQ->u.openrq.cwd = RQ->cwd;
	RQ->u.openrq.fn = &SHELL_DTTY_OPEN;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_DTTY_OPEN, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.openrq.status < 0)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DTTY: OPEN: %s (%s)", strerror(-RQ->u.openrq.status), RQ->u.openrq.path);
		RQ->return_val = RQ->u.openrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_DTTY_DONE;
	RQ->u.ioctlrq.h = RQ->u.openrq.status;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_DTTY;
	RQ->u.ioctlrq.param = 0;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_DTTY_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	KERNEL$FAST_CLOSE(RQ->u.ioctlrq.h);
	RETURN_TO_SHELL(RQ);
}


extern AST_STUB SHELL_ASSIGN_COMPLETED;

DECL_AST(SHELL_ASSIGN_COMMAND, SPL_SHELL, CTX)
{
	char **argv = RQ->argv;
	int state = 0;
	int flags = 0;
	char *name = NULL;
	char *alias = NULL;
	struct __param_table params[] = {
		"PUBLIC", __PARAM_BOOL, LNTE_PUBLIC, LNTE_PUBLIC, NULL,
		"PRIVATE", __PARAM_BOOL, LNTE_PUBLIC, 0, NULL,
		"BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, LNTE_BLOCK_IF_NOT_FOUND, NULL,
		"DONT_BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, 0, NULL,
		"FILE_SHARE", __PARAM_BOOL, LNTE_FILE_SHARE, LNTE_FILE_SHARE, NULL,
		"NO_FILE_SHARE", __PARAM_BOOL, LNTE_FILE_SHARE, 0, NULL,
		"", __PARAM_STRING, 1, -1, NULL,
		"", __PARAM_STRING, 1, -1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &flags;
	params[1].__p = &flags;
	params[2].__p = &flags;
	params[3].__p = &flags;
	params[4].__p = &flags;
	params[5].__p = &flags;
	params[6].__p = &alias;
	params[7].__p = &name;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL)) || __unlikely(!name) || __unlikely(!alias)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "ASSIGN: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.lrq.fn = SHELL_ASSIGN_COMPLETED;
	RQ->u.lrq.name = name;
	RQ->u.lrq.alias = alias;
	RQ->u.lrq.flags = flags;
	RQ->u.lrq.cwd = RQ->cwd;
	RQ->u.lrq.chh_option = NULL;
	RQ->u.lrq.blob = NULL;
	RETURN_IORQ_CANCELABLE(&RQ->u.lrq, KERNEL$CREATE_LOGICAL, RQ->rq);
}

DECL_AST(SHELL_ASSIGN_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->return_val = RQ->u.lrq.status))
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "ASSIGN: %s (%s <- %s)", strerror(-RQ->return_val), RQ->u.lrq.alias, RQ->u.lrq.name);
	else *RQ->return_msg = 0;
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_DEASSIGN_COMMAND, SPL_SHELL, CTX)
{
	int r;
	char **argv = RQ->argv;
	int state = 0;
	int block = 0;
	char *name = NULL;
	struct __param_table params[] = {
		"BLOCK", __PARAM_BOOL, ~0, DELETE_LOGICAL_BLOCK, NULL,
		"", __PARAM_STRING, 1, -1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &block;
	params[1].__p = &name;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL)) || __unlikely(!name)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DEASSIGN: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	r = KERNEL$DELETE_LOGICAL(name, block);
	if (__unlikely(r)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DEASSIGN: %s (%s)", strerror(-r), name);
		RQ->return_val = r;
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_SETLOGICAL_COMPLETED;

DECL_AST(SHELL_SETLOGICAL_COMMAND, SPL_SHELL, CTX)
{
	char **argv = RQ->argv;
	int state = 0;
	int flags = 0;
	int flags_chg_mask = 0;
	char *name = NULL;
	struct __param_table params[] = {
		"PUBLIC", __PARAM_BOOL, LNTE_PUBLIC, LNTE_PUBLIC, NULL,
		"PUBLIC", __PARAM_BOOL, LNTE_PUBLIC, LNTE_PUBLIC, NULL,
		"PRIVATE", __PARAM_BOOL, LNTE_PUBLIC, 0, NULL,
		"PRIVATE", __PARAM_BOOL, LNTE_PUBLIC, LNTE_PUBLIC, NULL,
		"BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, LNTE_BLOCK_IF_NOT_FOUND, NULL,
		"BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, LNTE_BLOCK_IF_NOT_FOUND, NULL,
		"DONT_BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, 0, NULL,
		"DONT_BLOCK_IF_NOT_FOUND", __PARAM_BOOL, LNTE_BLOCK_IF_NOT_FOUND, LNTE_BLOCK_IF_NOT_FOUND, NULL,
		"", __PARAM_STRING, 1, -1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &flags;
	params[1].__p = &flags_chg_mask;
	params[2].__p = &flags;
	params[3].__p = &flags_chg_mask;
	params[4].__p = &flags;
	params[5].__p = &flags_chg_mask;
	params[6].__p = &flags;
	params[7].__p = &flags_chg_mask;
	params[8].__p = &name;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL)) || __unlikely(!name)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "SETLOGICAL: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.lrq.fn = SHELL_SETLOGICAL_COMPLETED;
	RQ->u.lrq.name = name;
	RQ->u.lrq.alias = NULL;
	RQ->u.lrq.flags = flags;
	RQ->u.lrq.flags_chg_mask = flags_chg_mask;
	RQ->u.lrq.cwd = NULL;
	RQ->u.lrq.chh_option = NULL;
	RQ->u.lrq.blob = NULL;
	RETURN_IORQ_CANCELABLE(&RQ->u.lrq, KERNEL$SET_LOGICAL, RQ->rq);
}

DECL_AST(SHELL_SETLOGICAL_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->return_val = RQ->u.lrq.status))
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "SETLOGICAL: %s (%s)", strerror(-RQ->return_val), RQ->u.lrq.name);
	else *RQ->return_msg = 0;
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_LOGICAL_DONE;
extern AST_STUB SHELL_LOGICAL_WROTE;

DECL_AST(SHELL_LOGICAL_COMMAND, SPL_SHELL, CTX)
{
	char **argv = RQ->argv;
	int state = 0;
	struct __param_table params[] = {
		"NAME", __PARAM_BOOL, ~0, 0, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &RQ->itmp2;
	RQ->itmp2 = 1;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL))) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "LOGICAL: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u2.llr.prefix = "";
	RQ->u2.llr.fn = SHELL_LOGICAL_DONE;
	RETURN_IORQ_CANCELABLE(&RQ->u2.llr, KERNEL$LIST_LOGICALS, RQ->rq);
}

DECL_AST(SHELL_LOGICAL_DONE, SPL_SHELL, LOGICAL_LIST_REQUEST)
{
	CTX *ctx = GET_STRUCT(RQ, CTX, u2.llr);
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, ctx->rq);
	if (__unlikely(RQ->status < 0)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "LOGICAL: %s", strerror(-RQ->status));
		ctx->return_val = RQ->status;
		RETURN_TO_SHELL(ctx);
	}
	ctx->itmp1 = 0;
	CALL_SHELL_FN(ctx, SHELL_LOGICAL_WROTE);
}

DECL_AST(SHELL_LOGICAL_WROTE, SPL_SHELL, CTX)
{
	LOGICAL_LIST_ENTRY *e;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (RQ->itmp1) {
		if (__unlikely(RQ->u.siorq.status <= 0)) {
			RQ->return_val = RQ->u.siorq.status ? RQ->u.siorq.status : -EEOF;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "LOGICAL: %s (%s)", RQ->u.siorq.status ? strerror(-RQ->u.siorq.status) : "CAN'T WRITE", KERNEL$HANDLE_PATH(RQ->u.siorq.h));
			goto end;
		}
		if (__unlikely(RQ->u.siorq.v.len != 0)) {
			RQ->u.siorq.progress = 0;
			RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
		}
	}
	again:
	if (__unlikely(RQ->itmp1 == RQ->u2.llr.n_entries)) {
		end:
		KERNEL$FREE_LOGICAL_LIST(&RQ->u2.llr);
		RETURN_TO_SHELL(RQ);
	}
	e = RQ->u2.llr.entries[RQ->itmp1++];
	if (!RQ->itmp2) {
		_snprintf(RQ->scratch_1, SCRATCH_1_LEN, "%s", e->name);
	} else {
		if (e->flags >= 0) {
			if (e->flags & LNTE_DEVICE)
				_snprintf(RQ->scratch_1, SCRATCH_1_LEN, "%s (%s)", e->name, e->driver_name);
			else
				_snprintf(RQ->scratch_1, SCRATCH_1_LEN, "%s -> %s", e->name, KERNEL$HANDLE_PATH(e->flags & LNTE_HANDLE_MASK));
			if (e->flags & LNTE_PUBLIC)
				strlcat(RQ->scratch_1, " /PUBLIC", SCRATCH_1_LEN);
			if (e->flags & LNTE_FILE_SHARE)
				strlcat(RQ->scratch_1, " /FILE_SHARE", SCRATCH_1_LEN);
		} else {
			if (e->flags == LNTE_PARENT)
				_snprintf(RQ->scratch_1, SCRATCH_1_LEN, "%s (PARENT)", e->name);
			else
				_snprintf(RQ->scratch_1, SCRATCH_1_LEN, "%s (UNKNOWN, %d)", e->name, e->flags);
		}
	}
	strlcat(RQ->scratch_1, "\n", SCRATCH_1_LEN);
	RQ->u.siorq.fn = SHELL_LOGICAL_WROTE;
	RQ->u.siorq.h = RQ->ctty_out;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->scratch_1;
	RQ->u.siorq.v.len = strlen(RQ->scratch_1);
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	if (__unlikely(RQ->ctty_out == -1)) {
		__critical_printf("%s", RQ->scratch_1);
		goto again;
	}
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, &KERNEL$WRITE, RQ->rq);
}

extern AST_STUB SHELL_MKFILE_COMPLETED;

DECL_AST(SHELL_MKFILE_COMMAND, SPL_SHELL, CTX)
{
	int flags = O_CREAT | _O_CLOSE;
	char *file = NULL;
	struct __param_table params[] = {
		"EXCL", __PARAM_BOOL, O_EXCL, O_EXCL, NULL,
		"TRUNC", __PARAM_BOOL, O_TRUNC, O_TRUNC, NULL,
		"", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **argv = RQ->argv;
	int state = 0;
	params[0].__p = &flags;
	params[1].__p = &flags;
	params[2].__p = &file;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL)) || __unlikely(!file)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MKFILE: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (flags & O_TRUNC) flags |= O_WRONLY;
	else flags |= _O_NOACCESS;
	RQ->u.openrq.fn = SHELL_MKFILE_COMPLETED;
	RQ->u.openrq.flags = flags;
	RQ->u.openrq.path = file;
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_MKFILE_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->return_val = RQ->u.openrq.status))
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MKFILE: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
	else *RQ->return_msg = 0;
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_MKDIR_COMPLETED;

DECL_AST(SHELL_MKDIR_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MKDIR: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_MKDIR_COMPLETED;
	RQ->u.openrq.flags = O_CREAT | O_EXCL | _O_MKDIR | _O_CLOSE;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_MKDIR_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->return_val = RQ->u.openrq.status))
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MKDIR: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
	else *RQ->return_msg = 0;
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_RMDIR_COMPLETED;

DECL_AST(SHELL_RMDIR_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "RMDIR: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_RMDIR_COMPLETED;
	RQ->u.openrq.flags = _O_DELETE | _O_MKDIR | _O_CLOSE;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_RMDIR_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->return_val = RQ->u.openrq.status))
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "RMDIR: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
	else *RQ->return_msg = 0;
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_DEL_COMPLETED;

DECL_AST(SHELL_DEL_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1])) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DEL: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_DEL_COMPLETED;
	RQ->u.openrq.flags = _O_DELETE | _O_CLOSE;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_DEL_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely((RQ->return_val = RQ->u.openrq.status) != 0)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DEL: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
	} else if (RQ->argv[2]) {
		RQ->argv++;
		CALL_SHELL_FN(RQ, SHELL_DEL_COMMAND);
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_FSYNC_DO_OPEN;
extern AST_STUB SHELL_FSYNC_OPENED;
extern AST_STUB SHELL_FSYNC_COMPLETED;

__const__ static struct {
	char *name;
	int flag;
} fsync_flags[] = {
	{ "/ASYNC", PARAM_FSYNC_ASYNC },
	{ "/NO_FLASH", PARAM_FSYNC_NO_FLASH },
	{ NULL, 0 }
};

DECL_AST(SHELL_FSYNC_COMMAND, SPL_SHELL, CTX)
{
	int i, j, k;
	RQ->itmp2 = PARAM_FSYNC_TREE;
	for (i = 1, j = 1; (RQ->argv[j] = RQ->argv[i]); i++, j++) {
		for (k = 0; fsync_flags[k].name; k++) if (!_strcasecmp(fsync_flags[k].name, RQ->argv[i])) {
			RQ->itmp2 |= fsync_flags[k].flag;
			j--;
			break;
		}
	}
	if (__unlikely(!RQ->argv[1])) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "FSYNC: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	CALL_SHELL_FN(RQ, SHELL_FSYNC_DO_OPEN);
}

DECL_AST(SHELL_FSYNC_DO_OPEN, SPL_SHELL, CTX)
{
	RQ->u.openrq.fn = SHELL_FSYNC_OPENED;
	RQ->u.openrq.flags = _O_NOACCESS | _O_NOOPEN;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_FSYNC_OPENED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.openrq.status < 0)) {
		RQ->return_val = RQ->u.openrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "FSYNC: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp1 = RQ->u.openrq.status;
	RQ->u.ioctlrq.fn = SHELL_FSYNC_COMPLETED;
	RQ->u.ioctlrq.h = RQ->itmp1;
	RQ->u.ioctlrq.ioctl = IOCTL_FSYNC;
	RQ->u.ioctlrq.param = RQ->itmp2;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_FSYNC_COMPLETED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0) && RQ->u.ioctlrq.status != -ENOOP) {
		RQ->return_val = RQ->u.ioctlrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "FSYNC: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->itmp1));
		KERNEL$FAST_CLOSE(RQ->itmp1);
		RETURN_TO_SHELL(RQ);
	}
	KERNEL$FAST_CLOSE(RQ->itmp1);
	if (RQ->argv[2]) {
		RQ->argv++;
		CALL_SHELL_FN(RQ, SHELL_FSYNC_DO_OPEN);
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_COPY_OPEN_1;
extern AST_STUB SHELL_COPY_OPEN_2;
extern AST_STUB SHELL_COPY_READ;
extern AST_STUB SHELL_COPY_WRITE;
extern AST_STUB SHELL_COPY_CLOSE;

DECL_AST(SHELL_COPY_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || __unlikely(!RQ->argv[2]) || __unlikely(RQ->argv[3] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "COPY: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_COPY_OPEN_1;
	RQ->u.openrq.flags = O_RDONLY;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_COPY_OPEN_1, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely((RQ->itmp1 = RQ->u.openrq.status) < 0)) {
		RQ->return_val = RQ->u.openrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "COPY: %s: %s", RQ->u.openrq.path, strerror(-RQ->return_val));
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_COPY_OPEN_2;
	RQ->u.openrq.flags = O_WRONLY | O_CREAT | O_TRUNC | O_APPEND;
	RQ->u.openrq.path = RQ->argv[2];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_COPY_OPEN_2, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely((RQ->itmp2 = RQ->u.openrq.status) < 0)) {
		RQ->return_val = RQ->u.openrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "COPY: %s: %s", RQ->u.openrq.path, strerror(-RQ->return_val));
		KERNEL$FAST_CLOSE(RQ->itmp1);
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp3 = __PAGE_CLUSTER_SIZE;
	if (!(RQ->vtmp1 = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED))) RQ->vtmp1 = RQ->scratch_1, RQ->itmp3 = SCRATCH_1_LEN;
	RQ->u.siorq.fn = &SHELL_COPY_READ;
	RQ->u.siorq.h = RQ->itmp1;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->vtmp1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.v.len = RQ->itmp3;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, &KERNEL$READ, RQ->rq);
}

DECL_AST(SHELL_COPY_READ, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status <= 0)) {
		RQ->return_val = RQ->u.siorq.status;
		if (RQ->u.siorq.status)
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "COPY: READ: %s", strerror(-RQ->return_val));
		else *RQ->return_msg = 0;
		KERNEL$FAST_CLOSE(RQ->itmp1);
		if (RQ->vtmp1 != RQ->scratch_1) KERNEL$FREE_KERNEL_PAGE(RQ->vtmp1, VM_TYPE_WIRED_MAPPED);
		RQ->u.closerq.fn = &SHELL_COPY_CLOSE;
		RQ->u.closerq.h = RQ->itmp2;
		RETURN_IORQ_CANCELABLE(&RQ->u.closerq, &KERNEL$CLOSE, RQ->rq);
	}
	RQ->u.siorq.fn = &SHELL_COPY_WRITE;
	RQ->u.siorq.h = RQ->itmp2;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->vtmp1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.v.len = RQ->u.siorq.status;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, &KERNEL$WRITE, RQ->rq);
}

DECL_AST(SHELL_COPY_WRITE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status <= 0)) {
		RQ->return_val = RQ->u.siorq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "COPY: WRITE: %s", RQ->return_val ? strerror(-RQ->return_val) : "EOF");
		KERNEL$FAST_CLOSE(RQ->itmp1);
		if (RQ->vtmp1 != RQ->scratch_1) KERNEL$FREE_KERNEL_PAGE(RQ->vtmp1, VM_TYPE_WIRED_MAPPED);
		RQ->u.closerq.fn = &SHELL_COPY_CLOSE;
		RQ->u.closerq.h = RQ->itmp2;
		RETURN_IORQ_CANCELABLE(&RQ->u.closerq, &KERNEL$CLOSE, RQ->rq);
	}
	if (__unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->u.siorq.progress = 0;
		RETURN_IORQ_CANCELABLE(&RQ->u.siorq, &KERNEL$WRITE, RQ->rq);
	}
	RQ->u.siorq.fn = &SHELL_COPY_READ;
	RQ->u.siorq.h = RQ->itmp1;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->vtmp1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.v.len = RQ->itmp3;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, &KERNEL$READ, RQ->rq);
}

DECL_AST(SHELL_COPY_CLOSE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.closerq.h != -1)) KERNEL$FAST_CLOSE(RQ->u.closerq.h);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_MOVE_OPEN_1;
extern AST_STUB SHELL_MOVE_OPEN_2;

DECL_AST(SHELL_MOVE_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || __unlikely(!RQ->argv[2]) || __unlikely(RQ->argv[3] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MOVE: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_MOVE_OPEN_1;
	RQ->u.openrq.flags = _O_NOACCESS;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_MOVE_OPEN_1, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely((RQ->itmp1 = RQ->u.openrq.status) < 0)) {
		RQ->return_val = RQ->u.openrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MOVE: %s (%s)", strerror(-RQ->return_val), RQ->u.openrq.path);
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_MOVE_OPEN_2;
	RQ->u.openrq.flags = _O_NOACCESS | _O_RENAME | _O_CLOSE;
	RQ->u.openrq.path = RQ->argv[2];
	RQ->u.openrq.cwd = RQ->cwd;
	RQ->u.openrq.rename_handle = RQ->itmp1;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_MOVE_OPEN_2, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.openrq.status < 0)) {
		RQ->return_val = RQ->u.openrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "MOVE: %s (%s -> %s)", strerror(-RQ->return_val), RQ->argv[1], RQ->u.openrq.path);
		KERNEL$FAST_CLOSE(RQ->itmp1);
		RETURN_TO_SHELL(RQ);
	}
	KERNEL$FAST_CLOSE(RQ->itmp1);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_CD_DONE;

DECL_AST(SHELL_CD_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CD: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.cdrq.fn = SHELL_CD_DONE;
	RQ->u.cdrq.cwd = RQ->cwd;
	RQ->u.cdrq.path = RQ->argv[1];
	RETURN_IORQ_CANCELABLE(&RQ->u.cdrq, KERNEL$CD, RQ->rq);
}

DECL_AST(SHELL_CD_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.cdrq.status < 0)) {
		RQ->return_val = RQ->u.cdrq.status;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CD: %s (%s)", strerror(-RQ->u.cdrq.status), RQ->u.cdrq.path);
		RETURN_TO_SHELL(RQ);
	}
	RQ->cwd = RQ->u.cdrq.cwd;
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_SET_WROTE_1;
extern AST_STUB SHELL_SET_WROTE_2;

DECL_AST(SHELL_SET_COMMAND, SPL_SHELL, CTX)
{
	char *aa;
	struct var *v;
	size_t sz;
	if (__unlikely(!RQ->argv[1])) {
		RQ->itmp1 = 0;
		RQ->u.siorq.fn = SHELL_SET_WROTE_1;
		RETURN_AST(&RQ->u.siorq);
	}
	if (__unlikely(RQ->argv[2] != NULL) || __unlikely(!RQ->argv[1][0]) || __unlikely(RQ->argv[1][0] == '=')) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	aa = strchr(RQ->argv[1], '=');
	if (__unlikely(!aa)) aa = RQ->argv[1] + strlen(RQ->argv[1]);
	if (__unlikely((v = find_var(RQ, RQ->argv[1], aa, 1)) != NULL)) del_var(v);
	if (__unlikely(_putenv(RQ->argv[1], &sz) == -1)) {
		int r;
		if (__unlikely(r = KERNEL$OOM_ERR(sz))) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: CAN'T ALLOCATE: %s", strerror(-r));
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
	}
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_SET_WROTE_1, SPL_SHELL, CTX)
{
	int i;
	char *c;
	if (RQ->itmp1) {
		IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
		if (__unlikely(RQ->u.siorq.status < 0)) {
			RQ->return_ignore_err = 0;
			RQ->return_val = RQ->u.siorq.status;
			if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: WRITE: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
			else RQ->return_msg[0] = 0;
			RETURN_TO_SHELL(RQ);
		}
		if (__unlikely(!RQ->u.siorq.status)) {
			RQ->return_ignore_err = 0;
			RQ->return_val = -EEOF;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: CAN'T WRITE");
			RETURN_TO_SHELL(RQ);
		}
	}
	again:
	if (__unlikely(!environ)) RETURN_TO_SHELL(RQ);
	RAISE_SPL(SPL_DEV);
	for (i = 0; i <= RQ->itmp1; i++) if (!environ[i]) {
		LOWER_SPL(SPL_SHELL);
		RETURN_TO_SHELL(RQ);
	}
	i--;
	c = environ[i];
	RQ->itmp1++;
	LOWER_SPL(SPL_SHELL);
	if (__unlikely(RQ->ctty_out == -1)) {
		__critical_printf("%s\n", c);
		goto again;
	}
	RQ->u.siorq.fn = SHELL_SET_WROTE_2;
	RQ->u.siorq.h = RQ->ctty_out;
	RQ->u.siorq.v.ptr = (unsigned long)c;
	RQ->u.siorq.v.len = strlen(c);
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
}

DECL_AST(SHELL_SET_WROTE_2, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status < 0)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = RQ->u.siorq.status;
		if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: WRITE: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
		else RQ->return_msg[0] = 0;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!RQ->u.siorq.status)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = -EEOF;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "SET: CAN'T WRITE");
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->u.siorq.progress = 0;
		RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
	}
	RQ->u.siorq.fn = SHELL_SET_WROTE_1;
	RQ->u.siorq.v.ptr = (unsigned long)"\n";
	RQ->u.siorq.v.len = 1;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
}

extern AST_STUB SHELL_LET_WROTE_1;
extern AST_STUB SHELL_LET_WROTE_2;

DECL_AST(SHELL_LET_COMMAND, SPL_SHELL, CTX)
{
	struct var *v;
	char *aa;
	size_t sz;
	int r;
	if (__unlikely(!RQ->argv[1])) {
		RQ->vtmp1 = NULL;
		RQ->u.siorq.fn = SHELL_LET_WROTE_1;
		RETURN_AST(&RQ->u.siorq);
	}
	if (__unlikely(RQ->argv[2] != NULL)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	aa = strchr(RQ->argv[1], '=');
	if (__unlikely(!aa)) aa = RQ->argv[1] + strlen(RQ->argv[1]);
	if (__unlikely(r = check_var_name(RQ->argv[1], aa))) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: INVALID VARIABLE");
		RQ->return_val = r;
		RETURN_TO_SHELL(RQ);
	}
	memcpy(RQ->scratch_1, RQ->argv[1], aa - RQ->argv[1]);
	RQ->scratch_1[aa - RQ->argv[1]] = 0;
	v = find_var(RQ, RQ->argv[1], aa, 1);
	if (__likely(v != NULL)) {
		if (__likely(__alloc_size(v) >= sizeof(struct var) + strlen(RQ->argv[1])) && __likely(*aa) && __likely(var_reusable(v))) {
			strcpy(v->name + (aa - RQ->argv[1]), aa);
			RETURN_TO_SHELL(RQ);
		}
		del_var(v);
	}
	if (__unlikely(!*aa)) RETURN_TO_SHELL(RQ);
	if (__unlikely(!(v = malloc(sizeof(struct var) + strlen(RQ->argv[1]))))) {
		int r;
		sz = sizeof(struct var) + strlen(RQ->argv[1]);
		if (__unlikely(r = KERNEL$OOM_ERR(sz))) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: CAN'T ALLOCATE: %s", strerror(-r));
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
	}
	strcpy(v->name, RQ->argv[1]);
	init_var(v);
	add_var(RQ, v);
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_LET_WROTE_1, SPL_SHELL, CTX)
{
	int i;
	char *c;
	struct var *v, *m;
	if (RQ->vtmp1) {
		IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
		if (__unlikely(RQ->u.siorq.status < 0)) {
			RQ->return_ignore_err = 0;
			RQ->return_val = RQ->u.siorq.status;
			if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: WRITE: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
			else RQ->return_msg[0] = 0;
			RETURN_TO_SHELL(RQ);
		}
		if (__unlikely(!RQ->u.siorq.status)) {
			RQ->return_ignore_err = 0;
			RQ->return_val = -EEOF;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: CAN'T WRITE");
			RETURN_TO_SHELL(RQ);
		}
	}
	again:
	m = NULL;
	for (i = 0; i < VAR_HASH_SIZE; i++) XLIST_FOR_EACH(v, &RQ->var_hash[i], struct var, hash) if ((__unlikely(!RQ->vtmp1) || strcmp(v->name, ((struct var *)RQ->vtmp1)->name) > 0) && (__unlikely(!m) || strcmp(v->name, m->name) < 0)) m = v;
	if (!m) RETURN_TO_SHELL(RQ);
	RQ->vtmp1 = m;
	c = m->name;
	if (__unlikely(RQ->ctty_out == -1)) {
		__critical_printf("%s\n", c);
		goto again;
	}
	RQ->u.siorq.fn = SHELL_LET_WROTE_2;
	RQ->u.siorq.h = RQ->ctty_out;
	RQ->u.siorq.v.ptr = (unsigned long)c;
	RQ->u.siorq.v.len = strlen(c);
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
}

DECL_AST(SHELL_LET_WROTE_2, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
 	if (__unlikely(RQ->u.siorq.status < 0)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = RQ->u.siorq.status;
		if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: WRITE: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
		else RQ->return_msg[0] = 0;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!RQ->u.siorq.status)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = -EEOF;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "LET: CAN'T WRITE");
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->u.siorq.progress = 0;
		RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
	}
	RQ->u.siorq.fn = SHELL_LET_WROTE_1;
	RQ->u.siorq.v.ptr = (unsigned long)"\n";
	RQ->u.siorq.v.len = 1;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
}

extern AST_STUB SHELL_PRINT_WROTE_1;

DECL_AST(SHELL_PRINT_COMMAND, SPL_SHELL, CTX)
{
	char *c = RQ->argv[1];
	if (__unlikely(!c)) c = "\n";
	if (__unlikely(RQ->ctty_out < 0)) {
		__critical_printf("%s", c);
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.siorq.fn = SHELL_PRINT_WROTE_1;
	RQ->u.siorq.h = RQ->ctty_out;
	RQ->u.siorq.v.ptr = (unsigned long)c;
	RQ->u.siorq.v.len = strlen(c);
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
}

DECL_AST(SHELL_PRINT_WROTE_1, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status < 0)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = RQ->u.siorq.status;
		if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "PRINT: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
		else RQ->return_msg[0] = 0;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!RQ->u.siorq.status) && __unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->return_ignore_err = 0;
		RQ->return_val = -EEOF;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "PRINT: CAN'T WRITE");
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->u.siorq.progress = 0;
		RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$WRITE, RQ->rq);
	}
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_COLOR_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1])) {
		RQ->return_val = -EBADSYN;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%s: SYNTAX ERROR", RQ->argv[0]);
	}
	CALL_SHELL_FN(RQ, SHELL_PRINT_COMMAND);
}

extern AST_STUB SHELL_PUTBACK_WROTE_1;
extern AST_STUB SHELL_PUTBACK_WROTE_2;

DECL_AST(SHELL_PUTBACK_COMMAND, SPL_SHELL, CTX)
{
	RQ->itmp1 = 1;
	while (RQ->argv[RQ->itmp1]) RQ->itmp1++;
	RQ->itmp2 = 1;
	CALL_SHELL_FN(RQ, SHELL_PUTBACK_WROTE_1);
}

DECL_AST(SHELL_PUTBACKN_COMMAND, SPL_SHELL, CTX)
{
	RQ->itmp1 = 1;
	while (RQ->argv[RQ->itmp1]) RQ->itmp1++;
	RQ->itmp2 = 0;
	CALL_SHELL_FN(RQ, SHELL_PUTBACK_WROTE_1);
}

DECL_AST(SHELL_PUTBACK_WROTE_1, SPL_SHELL, CTX)
{
	if (RQ->argv[RQ->itmp1]) {
		IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
		if (__unlikely(RQ->u.ioctlrq.status < 0)) {
			RQ->return_val = RQ->u.ioctlrq.status;
			if (RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "PUTBACK: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.ioctlrq.h));
			RETURN_TO_SHELL(RQ);
		}
	} else {
		if (__unlikely(RQ->ctty_in == -1)) {
			RQ->return_val = -EINVAL;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "PUTBACK: NO CONTROL TERMINAL");
			RETURN_TO_SHELL(RQ);
		}
	}
	if (__unlikely(RQ->itmp1 == 1)) RETURN_TO_SHELL(RQ);
	RQ->u.ioctlrq.fn = SHELL_PUTBACK_WROTE_2;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_PUTBACK;
	RQ->u.ioctlrq.param = 0;
	if (RQ->argv[RQ->itmp1]) {
		RQ->u.ioctlrq.v.ptr = (unsigned long)" ";
		RQ->u.ioctlrq.v.len = 1;
	} else {
		RQ->u.ioctlrq.v.ptr = (unsigned long)"\n";
		RQ->u.ioctlrq.v.len = __likely(RQ->itmp2) ? 1 : 0;
	}
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_PUTBACK_WROTE_2, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0)) {
		RQ->return_val = RQ->u.ioctlrq.status;
		if (RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "PUTBACK: %s (%s)", strerror(-RQ->return_val), KERNEL$HANDLE_PATH(RQ->u.ioctlrq.h));
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_PUTBACK_WROTE_1;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_PUTBACK;
	RQ->u.ioctlrq.param = 0;
	RQ->u.ioctlrq.v.ptr = (unsigned long)RQ->argv[--RQ->itmp1];
	RQ->u.ioctlrq.v.len = strlen((char *)(unsigned long)RQ->u.ioctlrq.v.ptr);
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

extern AST_STUB SHELL_INPUT_BYTE;
extern AST_STUB SHELL_INPUT_ALLOCATED;

DECL_AST(SHELL_INPUT_COMMAND, SPL_SHELL, CTX)
{
	int i;
	int r;
	for (i = 1; RQ->argv[i]; i++) {
		struct var *v;
		if ((v = find_var(RQ, RQ->argv[i], RQ->argv[i] + strlen(RQ->argv[i]), 1))) {
			if (__likely(var_reusable(v))) {
				v->name[v->namel + 1] = 0;
				continue;
			} else {
				del_var(v);
			}
		}
		if (__unlikely(r = check_var_name(RQ->argv[i], RQ->argv[i] + strlen(RQ->argv[i])))) {
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "INPUT: INVALID VARIABLE");
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		if (__unlikely(!(v = malloc(sizeof(struct var) + strlen(RQ->argv[i]) + 1)))) {
			int r;
			if (__unlikely(r = KERNEL$OOM_ERR(sizeof(struct var) + strlen(RQ->argv[i]) + 1))) {
				_snprintf(RQ->return_msg, __MAX_STR_LEN, "INPUT: CAN'T ALLOCATE: %s", strerror(-r));
				RQ->return_val = r;
				RETURN_TO_SHELL(RQ);
			}
			RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
		}
		strcpy(stpcpy(v->name, RQ->argv[i]), "=");
		init_var(v);
		add_var(RQ, v);
	}
	if (__unlikely(RQ->ctty_in == -1)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "INPUT: NO CONTROL TERMINAL");
		RQ->return_val = -EINVAL;
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp1 = 1;
	RQ->itmp2 = 0;
	RQ->u.siorq.fn = SHELL_INPUT_BYTE;
	RQ->u.siorq.h = RQ->ctty_in;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->scratch_1;
	RQ->u.siorq.v.len = 1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$READ, RQ->rq);
}

DECL_AST(SHELL_INPUT_BYTE, SPL_SHELL, CTX)
{
	__const__ char *ifs;
	struct var *v;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status <= 0)) {
		if (__unlikely(RQ->u.siorq.status < 0)) {
			if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "INPUT: %s (%s)", strerror(-RQ->u.siorq.status), KERNEL$HANDLE_PATH(RQ->u.siorq.h));
			RQ->return_val = RQ->u.siorq.status;
		}
		if (!RQ->itmp2) RQ->return_val = 1;
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp2 = 1;
	if (__unlikely(RQ->scratch_1[0] == '\n')) RETURN_TO_SHELL(RQ);
	if (__unlikely(RQ->scratch_1[0] == '\r')) goto rd_more;
	if (__unlikely(!RQ->argv[1])) goto rd_more;
	if (__likely(!(ifs = find_var_env(RQ, "IFS")))) ifs = " 	";
	if (RQ->argv[2] && __unlikely(strchr(ifs, RQ->scratch_1[0]) != NULL)) {
		if (__likely(!RQ->itmp1)) RQ->argv++;
		RQ->itmp1 = 1;
		goto rd_more;
	}
	RQ->itmp1 = 0;
	if (__unlikely(!(v = find_var(RQ, RQ->argv[1], RQ->argv[1] + strlen(RQ->argv[1]), 1)))) goto rd_more;
	if (__unlikely(sizeof(struct var) + strlen(v->name) + 1 > __alloc_size(v))) {
		RQ->u.mrq.fn = SHELL_INPUT_ALLOCATED;
		RQ->u.mrq.size = sizeof(struct var) + strlen(v->name) + 1;
		RETURN_IORQ_CANCELABLE(&RQ->u.mrq, KERNEL$UNIVERSAL_MALLOC, RQ->rq);
	}
	RQ->scratch_1[1] = 0;
	strcat(v->name + v->namel, RQ->scratch_1);
	rd_more:
	RQ->u.siorq.fn = SHELL_INPUT_BYTE;
	RQ->u.siorq.h = RQ->ctty_in;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->scratch_1;
	RQ->u.siorq.v.len = 1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$READ, RQ->rq);
}

DECL_AST(SHELL_INPUT_ALLOCATED, SPL_SHELL, CTX)
{
	struct var *nv, *ov;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status < 0)) {
		if (RQ->u.siorq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "INPUT: CAN'T ALLOCATE: %s", strerror(-RQ->u.siorq.status));
		RQ->return_val = RQ->u.siorq.status;
		RETURN_TO_SHELL(RQ);
	}
	nv = RQ->u.mrq.ptr;
	if (__unlikely(!(ov = find_var(RQ, RQ->argv[1], RQ->argv[1] + strlen(RQ->argv[1]), 1)))) {
		KERNEL$UNIVERSAL_FREE(nv);
		goto rd_more;
	}
	if (__unlikely(sizeof(struct var) + strlen(ov->name) + 1 > __alloc_size(nv))) {
		KERNEL$UNIVERSAL_FREE(nv);
		goto rd_more;
	}
	strcpy(nv->name, ov->name);
	RQ->scratch_1[1] = 0;
	strcat(nv->name, RQ->scratch_1);
	del_var(ov);
	init_var(nv);
	add_var(RQ, nv);
	rd_more:
	RQ->u.siorq.fn = SHELL_INPUT_BYTE;
	RQ->u.siorq.h = RQ->ctty_in;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->scratch_1;
	RQ->u.siorq.v.len = 1;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$READ, RQ->rq);
}

extern AST_STUB SHELL_CALL_OPEN;
extern AST_STUB SHELL_CALL_STAT;
extern AST_STUB SHELL_CALL_ALLOCATED;
extern AST_STUB SHELL_CALL_READ;
extern AST_STUB SHELL_CALL_EXIT;

DECL_AST(SHELL_CALL_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(!RQ->argv[1])) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.openrq.fn = SHELL_CALL_OPEN;
	RQ->u.openrq.path = RQ->argv[1];
	RQ->u.openrq.flags = O_RDONLY;
	RQ->u.openrq.cwd = RQ->cwd;
	RETURN_IORQ_CANCELABLE(&RQ->u.openrq, KERNEL$OPEN, RQ->rq);
}

DECL_AST(SHELL_CALL_OPEN, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.openrq.status < 0)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: %s (%s)", strerror(-RQ->u.openrq.status), RQ->argv[1]);
		RQ->return_val = RQ->u.openrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->itmp1 = RQ->u.openrq.status;
	RQ->u.ioctlrq.fn = SHELL_CALL_STAT;
	RQ->u.ioctlrq.h = RQ->u.openrq.status;
	RQ->u.ioctlrq.ioctl = IOCTL_STAT;
	RQ->u.ioctlrq.param = 0;
	RQ->u.ioctlrq.v.ptr = (unsigned long)&RQ->stat;
	RQ->u.ioctlrq.v.len = sizeof(struct stat);
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_CALL_STAT, SPL_SHELL, CTX)
{
	int argsize;
	int arg;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0)) {
		if (RQ->u.ioctlrq.status == -ENOOP) {
			RQ->stat.st_mode = 0;
			not_reg:
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: NOT REGULAR FILE, MODE %o (%s)", RQ->stat.st_mode, RQ->argv[1]);
			RQ->return_val = -EINVAL;
		} else {
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: ERROR GETTING STAT: %s (%s)", strerror(-RQ->u.ioctlrq.status), RQ->argv[1]);
			RQ->return_val = RQ->u.ioctlrq.status;
		}
		cl_ret:
		KERNEL$FAST_CLOSE(RQ->itmp1);
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!S_ISREG(RQ->stat.st_mode))) goto not_reg;
	if (__unlikely(RQ->stat.st_size > MAXINT)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: %s: FILE IS TOO BIG", RQ->argv[1]);
		RQ->return_val = -EFBIG;
		goto cl_ret;
	}
	argsize = sizeof(char *) - 1;
	for (arg = 1; RQ->argv[arg]; arg++) {
		argsize += strlen(RQ->argv[arg]) + 1;
		argsize += sizeof(char *);
	}
	RQ->u.mrq.fn = SHELL_CALL_ALLOCATED;
	RQ->u.mrq.size = sizeof(SHRCRQ) + RQ->stat.st_size + argsize;
	RETURN_IORQ_CANCELABLE(&RQ->u.mrq, KERNEL$UNIVERSAL_MALLOC, RQ->rq);
}

DECL_AST(SHELL_CALL_ALLOCATED, SPL_SHELL, CTX)
{
	SHRCRQ *rc;
	int i;
	char *p;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.mrq.status < 0)) {
		KERNEL$FAST_CLOSE(RQ->itmp1);
		if (RQ->u.mrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: CAN'T ALLOCATE: %s", strerror(-RQ->u.mrq.status));
		RQ->return_val = RQ->u.mrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->vtmp1 = RQ->u.mrq.ptr;
	rc = RQ->vtmp1;
	rc->narg = 0;
	for (i = 1; RQ->argv[i]; i++) rc->narg++;
	rc->arg = (__const__ char **)(((unsigned long)RQ->vtmp1 + sizeof(SHRCRQ) + (unsigned long)RQ->stat.st_size + sizeof(char *) - 1) & ~(unsigned long)(sizeof(char *) - 1));
	p = (char *)(rc->arg + rc->narg);
	for (i = 1; RQ->argv[i]; i++) {
		rc->arg[i - 1] = p;
		strcpy(p, RQ->argv[i]);
		p += strlen(RQ->argv[i]) + 1;
	}
	if (__unlikely(p > (char *)RQ->vtmp1 + RQ->u.mrq.size))
		KERNEL$SUICIDE("SHELL_CALL_ALLOCATED: ARGS OVERFLOW: %p > %p + %ld", p, RQ->vtmp1, (unsigned long)RQ->u.mrq.size);
	RQ->u.siorq.fn = SHELL_CALL_READ;
	RQ->u.siorq.h = RQ->itmp1;
	RQ->u.siorq.v.ptr = (unsigned long)RQ->vtmp1 + sizeof(SHRCRQ);
	RQ->u.siorq.v.len = RQ->stat.st_size;
	RQ->u.siorq.v.vspace = &KERNEL$VIRTUAL;
	RQ->u.siorq.progress = 0;
	RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$READ, RQ->rq);
}

DECL_AST(SHELL_CALL_READ, SPL_SHELL, CTX)
{
	SHRCRQ *rc;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.siorq.status <= -!RQ->u.siorq.v.len)) {
		KERNEL$FAST_CLOSE(RQ->itmp1);
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CALL: CAN'T READ: %s (%s)", !RQ->u.siorq.status ? "EOF" : strerror(-RQ->u.siorq.status), RQ->argv[1]);
		RQ->return_val = RQ->u.siorq.status ? RQ->u.siorq.status : -EEOF;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->u.siorq.v.len != 0)) {
		RQ->u.siorq.progress = 0;
		RETURN_IORQ_CANCELABLE(&RQ->u.siorq, KERNEL$READ, RQ->rq);
	}
	KERNEL$FAST_CLOSE(RQ->itmp1);
	rc = RQ->vtmp1;
	rc->fn = SHELL_CALL_EXIT;
	rc->ctx = RQ;
	rc->ptr = (__const__ char *)(rc + 1);
	rc->len = RQ->stat.st_size;
	RETURN_IORQ_CANCELABLE(rc, SHELL$RUN_COMMANDS, RQ->rq);
}

DECL_AST(SHELL_CALL_EXIT, SPL_SHELL, SHRCRQ)
{
	CTX *ctx = RQ->ctx;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, ctx->rq);
	memmove(ctx->return_msg, RQ->error_msg, __MAX_STR_LEN);
	ctx->return_val = RQ->status;
	KERNEL$UNIVERSAL_FREE(RQ);
	RETURN_TO_SHELL(ctx);
}


extern AST_STUB SHELL_TTYWAIT_GOT_CONTROL;
extern AST_STUB SHELL_TTYWAIT_DO_WAIT;

DECL_AST(SHELL_TTYWAIT_COMMAND, SPL_SHELL, CTX)
{
	unsigned long n;
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		bads:
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "TTYWAIT: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->ctty_in == -1)) {
		RQ->return_val = -EINVAL;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "TTYWAIT: NO CONTROL TERMINAL");
		RETURN_TO_SHELL(RQ);
	}
	if (RQ->argv[1][0] == '^') {
		n = RQ->argv[1][1] - '@';
		if (n >= 32) n -= 32;
		if (__unlikely(n >= 32)) goto bads;
	} else if (RQ->argv[1][0] == '!') {
		n = RQ->argv[1][1] - '@';
		if (__unlikely(n >= 32)) n -= 32;
		if (__unlikely(n >= 32)) goto bads;
		n += 32;
	} else {
		if (__get_number(RQ->argv[1], strchr(RQ->argv[1], 0), 0, (long *)&n)) goto bads;
		if (__unlikely(n >= 64)) {
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "TTYWAIT: NUMBER OUT OF RANGE");
			RQ->return_val = -ERANGE;
			RETURN_TO_SHELL(RQ);
		}
	}
	RQ->itmp1 = n;
	RQ->u.ioctlrq.fn = SHELL_TTYWAIT_GOT_CONTROL;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_GET_CONTROL;
	RQ->u.ioctlrq.param = RQ->itmp1;
	RQ->u.ioctlrq.v.ptr = (unsigned long)&RQ->u2.w.tcs;
	RQ->u.ioctlrq.v.len = sizeof RQ->u2.w.tcs;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_TTYWAIT_GOT_CONTROL, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0)) {
		if (RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "TTYWAIT: NOT A TTY");
		RQ->return_val = RQ->u.ioctlrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_TTYWAIT_DO_WAIT;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_WAIT_CONTROL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_TTYWAIT_DO_WAIT, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0)) {
		if (__likely(RQ->u.ioctlrq.status == -ETTYCHG)) {
			RQ->u.ioctlrq.fn = SHELL_TTYWAIT_GOT_CONTROL;
			RQ->u.ioctlrq.ioctl = IOCTL_TTY_GET_CONTROL;
			RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
		}
		if (RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "TTYWAIT: NOT A TTY");
		RQ->return_val = RQ->u.ioctlrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_AUTH_DONE;
extern AST_STUB SHELL_AUTH_CORE;

DECL_AST(SHELL_AUTH_COMMAND, SPL_SHELL, CTX)
{
	RQ->itmp1 = 0;
	CALL_SHELL_FN(RQ, SHELL_AUTH_CORE);
}

DECL_AST(SHELL_NOTAUTH_COMMAND, SPL_SHELL, CTX)
{
	RQ->itmp1 = 1;
	CALL_SHELL_FN(RQ, SHELL_AUTH_CORE);
}

DECL_AST(SHELL_AUTH_CORE, SPL_SHELL, CTX)
{
	if (__unlikely(RQ->argv[1] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%sAUTH: SYNTAX ERROR", RQ->itmp1 ? "NOT" : "");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->ctty_in == -1)) {
		RQ->return_val = -EINVAL;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "%sAUTH: NO CONTROL TERMINAL", RQ->itmp1 ? "NOT" : "");
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_AUTH_DONE;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_AUTH;
	RQ->u.ioctlrq.param = RQ->itmp1;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_AUTH_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.ioctlrq.status < 0)) {
		RQ->return_val = RQ->u.ioctlrq.status;
		if (RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "%sAUTH: %s (%s)", RQ->itmp1 ? "NOT" : "", strerror(-RQ->u.ioctlrq.status), KERNEL$HANDLE_PATH(RQ->u.ioctlrq.h));
		RETURN_TO_SHELL(RQ);
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_GETAUTH_DONE;

DECL_AST(SHELL_GETAUTH_COMMAND, SPL_SHELL, CTX)
{
	int nbl = 0;
	if (__unlikely(RQ->argv[1] != NULL)) {
		if (__likely(!_strcasecmp(RQ->argv[1], "/NOBLOCK"))) {
			nbl = 1;
			if (__likely(!RQ->argv[2])) goto syn_ok;
		}
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "GETAUTH: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	syn_ok:
	if (__unlikely(RQ->ctty_in == -1)) {
		RQ->return_val = -EINVAL;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "GETAUTH: NO CONTROL TERMINAL");
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_GETAUTH_DONE;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_GETAUTH;
	RQ->u.ioctlrq.param = nbl;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_GETAUTH_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	RQ->return_val = RQ->u.ioctlrq.status;
	if (__likely(RQ->u.ioctlrq.status != -EWOULDBLOCK) && RQ->u.ioctlrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "%sAUTH: %sn(%s)", RQ->itmp1 ? "NOT" : "", strerror(-RQ->u.ioctlrq.status), KERNEL$HANDLE_PATH(RQ->u.ioctlrq.h));
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_FLUSHTTY_DONE;

DECL_AST(SHELL_FLUSHTTY_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(RQ->argv[1] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "FLUSHTTY: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->ctty_in == -1)) {
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_FLUSHTTY_DONE;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_FLUSH;
	RQ->u.ioctlrq.param = FREAD;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_FLUSHTTY_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_CLEARCLIPBOARD_DONE;

DECL_AST(SHELL_CLEARCLIPBOARD_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(RQ->argv[1] != NULL)) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "CLEARCLIPBOARD: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(RQ->ctty_in == -1)) {
		RETURN_TO_SHELL(RQ);
	}
	RQ->u.ioctlrq.fn = SHELL_CLEARCLIPBOARD_DONE;
	RQ->u.ioctlrq.h = RQ->ctty_in;
	RQ->u.ioctlrq.ioctl = IOCTL_TTY_ERASE_CLIPBOARD;
	RQ->u.ioctlrq.param = 0;
	RQ->u.ioctlrq.v.ptr = 0;
	RQ->u.ioctlrq.v.len = 0;
	RQ->u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->u.ioctlrq, KERNEL$IOCTL, RQ->rq);
}

DECL_AST(SHELL_CLEARCLIPBOARD_DONE, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_DETACH_MEM_ALLOCATED;
extern AST_STUB SHELL_DETACH_CTX_ALLOCATED;
extern AST_STUB SHELL_DETACH_EXITED;

DECL_AST(SHELL_DETACH_COMMAND, SPL_SHELL, CTX)
{
	int ac;
	int al = 0;
	for (ac = 1; RQ->argv[ac]; ac++) al += strlen(RQ->argv[ac]) * 3 + 3;
	RQ->u.mrq.fn = SHELL_DETACH_MEM_ALLOCATED;
	RQ->u.mrq.size = sizeof(SHRCRQ) + al;
	RETURN_IORQ_CANCELABLE(&RQ->u.mrq, KERNEL$UNIVERSAL_MALLOC, RQ->rq);
}

DECL_AST(SHELL_DETACH_MEM_ALLOCATED, SPL_SHELL, CTX)
{
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.mrq.status < 0)) {
		if (RQ->u.mrq.status != -EINTR) _snprintf(RQ->return_msg, __MAX_STR_LEN, "DETACH: CAN'T ALLOCATE: %s", strerror(-RQ->u.mrq.status));
		RQ->return_val = RQ->u.mrq.status;
		RETURN_TO_SHELL(RQ);
	}
	RQ->vtmp1 = RQ->u.mrq.ptr;
	RQ->u.shccrq.fn = SHELL_DETACH_CTX_ALLOCATED;
	RETURN_IORQ_CANCELABLE(&RQ->u.shccrq, SHELL$CREATE_CONTEXT, RQ->rq);
}

DECL_AST(SHELL_DETACH_CTX_ALLOCATED, SPL_SHELL, CTX)
{
	int noesc = 0;
	char **a, *c;
	SHRCRQ *shrcrq;
	IO_DISABLE_CHAIN_CANCEL(SPL_SHELL, RQ->rq);
	if (__unlikely(RQ->u.shccrq.status < 0)) {
		KERNEL$UNIVERSAL_FREE(RQ->vtmp1);
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "DETACH: CAN'T CREATE CONTEXT: %s", strerror(-RQ->u.shccrq.status));
		RQ->return_val = RQ->u.shccrq.status;
		RETURN_TO_SHELL(RQ);
	}
	a = &RQ->argv[1];
	shrcrq = RQ->vtmp1;
	c = (char *)(shrcrq + 1);
	if (*a && !_strcasecmp(*a, "/NOESC")) noesc = 1, a++;
	while (*a) {
		if (c != (char *)(shrcrq + 1)) *c++ = ' ';
		if (__likely(!noesc)) SHELL$ESC_COPY(c, *a);
		else strcpy(c, *a);
		c += strlen(c);
		a++;
	}
	shrcrq->fn = SHELL_DETACH_EXITED;
	shrcrq->ctx = RQ->u.shccrq.ctx;
	shrcrq->ptr = (char *)(shrcrq + 1);
	shrcrq->len = c - shrcrq->ptr;
	shrcrq->arg = NULL;
	shrcrq->narg = 0;
	CALL_IORQ(shrcrq, SHELL$RUN_COMMANDS);
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_DETACH_EXITED, SPL_SHELL, SHRCRQ)
{
	SHELL$DESTROY_CONTEXT(RQ->ctx);
	KERNEL$UNIVERSAL_FREE(RQ);
	RETURN;
}

DECL_AST(SHELL_GOTO_COMMAND, SPL_SHELL, CTX)
{
	int sub;
	__const__ char *c, *p, *q;
	if (__unlikely(!RQ->argv[1]) || __unlikely(!RQ->argv[2]) || __unlikely(RQ->argv[3] != NULL)) {
		bad_syn:
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "GO: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__likely(!_strcasecmp(RQ->argv[1], "TO"))) sub = 0;
	else if (__likely(!_strcasecmp(RQ->argv[1], "SUB"))) sub = 1;
	else goto bad_syn;
	c = RQ->argv[2];
	p = *c == '^' ? c + 1 : c;
	if (__unlikely(*p < '0') || __unlikely(*p > '9')) goto invl_name;
	for (p++; *p; p++) if (__unlikely(!is_var_char(*p))) {
		invl_name:
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "GO %s: INVALID LABEL NAME", sub ? "SUB" : "TO");
		RQ->return_val = -EINVAL;
		RETURN_TO_SHELL(RQ);
	}
	if (__likely(*c != '^')) {
		if (__likely((p = FIND_LABEL(1, RQ->pos, RQ->end, c)) != NULL) ||
		    __likely((p = FIND_LABEL(1, RQ->beg, RQ->pos, c)) != NULL)) {
			ok:
			if (__unlikely(sub)) {
				if (__unlikely(RQ->stack[_SHELL_STACK_SIZE - 1] != NULL)) {
					RQ->return_ignore_err = 0;
					_snprintf(RQ->return_msg, __MAX_STR_LEN, "GO SUB: STACK OVERFLOW");
					RQ->return_val = -EOVERFLOW;
					RETURN_TO_SHELL(RQ);
				}
				memmove(RQ->stack + 1, RQ->stack, sizeof RQ->stack - sizeof RQ->stack[0]);
				RQ->stack[0] = RQ->pos;
			}
			RQ->pos = p;
			RETURN_TO_SHELL(RQ);
		}
		no_label:
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "GO %s: LABEL NOT FOUND", sub ? "SUB" : "TO");
		RQ->return_val = -ENOENT;
		RETURN_TO_SHELL(RQ);
	} else {
		c++;
		if (__likely((q = FIND_LABEL(1, RQ->beg, RQ->end, c)) != NULL)) do p = q; while (__unlikely((q = FIND_LABEL(0, p, p < RQ->pos ? RQ->pos : RQ->end, c)) != NULL));
		else goto no_label;
		goto ok;
	}
}

DECL_AST(SHELL_RETURN_COMMAND, SPL_SHELL, CTX)
{
	if (__unlikely(RQ->argv[1] != NULL)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "RETURN: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!RQ->stack[0])) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "RETURN: WITHOUT GO SUB");
		RQ->return_val = -EOVERFLOW;
		RETURN_TO_SHELL(RQ);
	}
	RQ->pos = RQ->stack[0];
	memmove(RQ->stack, RQ->stack + 1, sizeof RQ->stack - sizeof RQ->stack[0]);
	RQ->stack[_SHELL_STACK_SIZE - 1] = NULL;
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_IF_COMMAND, SPL_SHELL, CTX)
{
	long cond;
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "IF: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(__get_number(RQ->argv[1], strchr(RQ->argv[1], 0), 1, &cond))) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "IF: BAD NUMBER");
		RQ->return_val = -EINVAL;
		RETURN_TO_SHELL(RQ);
	}
	if (!cond) {
		RQ->pos = NEXT_LINE(RQ->beg, RQ->pos, RQ->end, 1);
	}
	RETURN_TO_SHELL(RQ);
}

extern AST_STUB SHELL_FOR_IN;

DECL_AST(SHELL_FOR_COMMAND, SPL_SHELL, CTX)
{
	int r;
	size_t sz;
	char *vname;
	if (__unlikely(!RQ->argv[1]) || __unlikely(!RQ->argv[2])) {
		bads:
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (__likely(!_strcasecmp(RQ->argv[2], "IN"))) {
		size_t ms;
		struct var *v;
		if (__unlikely(r = check_var_name(RQ->argv[1], RQ->argv[1] + strlen(RQ->argv[1])))) {
			invl_var:
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: INVALID VARIABLE");
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		if (__unlikely((sz = DO_LOCAL(RQ, RQ->argv[1])) != 0)) goto oom;
		if (__unlikely(!RQ->argv[3])) {
			vname = RQ->argv[1];
			goto jump_to_next;
		}
		v = find_var(RQ, RQ->argv[1], RQ->argv[1] + strlen(RQ->argv[1]), 1);
		if (v) del_var(v);
		RQ->itmp1 = 3;
		ms = 0;
		do {
			size_t s = strlen(RQ->argv[RQ->itmp1]);
			if (s > ms) ms = s;
			RQ->itmp1++;
		} while (RQ->argv[RQ->itmp1]);
		RQ->itmp1--;
		v = malloc(sz = sizeof(struct var) + strlen(RQ->argv[1]) + 1 + ms);
		if (__unlikely(!v)) {
			int r;
			oom:
			if (__unlikely(r = KERNEL$OOM_ERR(sz))) {
				RQ->return_ignore_err = 0;
				_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: CAN'T ALLOCATE: %s", strerror(-r));
				RQ->return_val = r;
				RETURN_TO_SHELL(RQ);
			}
			RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
		}
		stpcpy(stpcpy(stpcpy(v->name, RQ->argv[1]), "="), RQ->argv[3]);
		init_var(v);
		add_var(RQ, v);
		RQ->vtmp1 = v;
		CALL_SHELL_FN(RQ, SHELL_FOR_IN);
	} if (__likely(!_strcasecmp(RQ->argv[2], "TO"))) {
		struct var *v, *ov;
		__s64 from, to, step;
		char *eq = strchr(RQ->argv[1], '=');
		if (__unlikely(!eq)) goto bads;
		if (__unlikely(r = check_var_name(RQ->argv[1], eq))) goto invl_var;
		memcpy(RQ->scratch_1, RQ->argv[1], eq - RQ->argv[1]);
		RQ->scratch_1[eq - RQ->argv[1]] = 0;
		if (__unlikely(__get_64_number(eq + 1, eq + strlen(eq), 1, &from))) {
			badn:
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: BAD NUMBER");
			RQ->return_val = -EINVAL;
			RETURN_TO_SHELL(RQ);
		}
		if (__unlikely(!RQ->argv[3])) goto bads;
		if (__unlikely(__get_64_number(RQ->argv[3], RQ->argv[3] + strlen(RQ->argv[3]), 1, &to))) goto badn;
		step = 1;
		if (__unlikely(RQ->argv[4] != NULL)) {
			if (__unlikely(_strcasecmp(RQ->argv[4], "STEP"))) goto bads;
			if (__unlikely(!RQ->argv[5])) goto bads;
			if (__unlikely(__get_64_number(RQ->argv[5], RQ->argv[5] + strlen(RQ->argv[5]), 1, &step))) goto badn;
			if (__unlikely(RQ->argv[6] != NULL)) goto bads;
		}
		if (__unlikely((sz = DO_LOCAL(RQ, RQ->scratch_1)) != 0)) goto oom;
		vname = RQ->scratch_1;
		if (__likely(step > 0)) {
			if (__unlikely(from > to)) goto jump_to_next;
		} else if (__likely(step < 0)) {
			if (__unlikely(from < to)) goto jump_to_next;
		}
		v = malloc(sz = sizeof(struct var) + (eq - RQ->argv[1]) + 1 + 21);
		if (__unlikely(!v)) goto oom;
		memcpy(v->name, RQ->argv[1], eq - RQ->argv[1]);
		v->name[eq - RQ->argv[1]] = 0;
		ov = find_var(RQ, RQ->argv[1], eq, 1);
		if (ov) del_var(ov);
		v->name[eq - RQ->argv[1]] = '=';
		_snprintf(v->name + (eq - RQ->argv[1]) + 1, 21, "%"__64_format"d", from);
		init_var(v);
		v->for_to = to;
		v->for_step = step;
		v->for_ptr = RQ->pos;
		add_var(RQ, v);
		RETURN_TO_SHELL(RQ);
	}
	goto bads;

	jump_to_next: {
		__const__ char *next;
		next = FIND_COMMAND(1, RQ->pos, RQ->end, "NEXT", vname);
		if (__unlikely(!next)) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: WITHOUT NEXT");
			RQ->return_val = -ENOENT;
			RETURN_TO_SHELL(RQ);
		}
		RQ->pos = NEXT_LINE(next, next, RQ->end, 0);
		RETURN_TO_SHELL(RQ);
	}
}

DECL_AST(SHELL_FOR_IN, SPL_SHELL, CTX)
{
	size_t sz;
	struct for_value *for_;
	struct var *v;
	for_ = malloc(sz = sizeof(struct for_value) + strlen(RQ->argv[RQ->itmp1]));
	if (__unlikely(!for_)) {
		int r;
		if (__unlikely(r = KERNEL$OOM_ERR(sz))) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "FOR: CAN'T ALLOCATE: %s", strerror(-r));
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
	}
	strcpy(for_->value, RQ->argv[RQ->itmp1]);
	v = RQ->vtmp1;
	ADD_TO_XLIST(&v->for_values, &for_->list);
	if (--RQ->itmp1 == 2) {
		v->for_ptr = RQ->pos;
		RETURN_TO_SHELL(RQ);
	}
	CALL_SHELL_FN(RQ, SHELL_FOR_IN);
}

DECL_AST(SHELL_NEXT_COMMAND, SPL_SHELL, CTX)
{
	struct var *v;
	if (__unlikely(!RQ->argv[1]) || __unlikely(RQ->argv[2] != NULL)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "NEXT: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	v = find_var(RQ, RQ->argv[1], RQ->argv[1] + strlen(RQ->argv[1]), 1);
	if (__unlikely(!v)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "NEXT: VARIABLE %s NOT FOUND", RQ->argv[1]);
		RQ->return_val = -ENOENT;
		RETURN_TO_SHELL(RQ);
	}
	if (__unlikely(!v->for_ptr) || __unlikely((unsigned long)v->for_ptr <= (unsigned long)RQ->beg) || __unlikely((unsigned long)v->for_ptr > (unsigned long)RQ->end)) {
		RQ->return_ignore_err = 0;
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "NEXT: WITHOUT FOR");
		RQ->return_val = -ENOENT;
		RETURN_TO_SHELL(RQ);
	}
	if (!XLIST_EMPTY(&v->for_values)) {
		struct for_value *for_;
		for_ = LIST_STRUCT(v->for_values.next, struct for_value, list);
		DEL_FROM_LIST(&for_->list);
		if (__unlikely(XLIST_EMPTY(&v->for_values))) goto fall_through;
		for_ = LIST_STRUCT(v->for_values.next, struct for_value, list);
		if (__unlikely(__offsetof(struct var, name) + v->namel + 1 + strlen(for_->value) + 1 > __alloc_size(v)))
			KERNEL$SUICIDE("SHELL_NEXT_COMMAND: UNDERALLOCATED VARIABLE %s (FIELD %s): %ld > %ld", v->name, for_->value, (unsigned long)__offsetof(struct var, name) + v->namel + 1 + strlen(for_->value) + 1, (unsigned long)__alloc_size(v));
		strcpy(v->name + v->namel + 1, for_->value);
		goto_for:
		RQ->pos = v->for_ptr;
		RETURN_TO_SHELL(RQ);
	} else {
		__s64 val, res;
		if (__unlikely(__get_64_number(v->name + v->namel + 1, v->name + strlen(v->name), 1, &val))) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "NEXT: BAD NUMBER IN VARIABLE");
			RQ->return_val = -EINVAL;
			RETURN_TO_SHELL(RQ);
		}
		res = (__u64)val + (__u64)v->for_step;
		if (__likely(!((unsigned long)((__u64)val >> 63) ^ (unsigned long)((__u64)v->for_step >> 63))) && __unlikely((unsigned long)((__u64)val >> 63) ^ (unsigned long)((__u64)res >> 63))) {
			goto fall_through;
		}
		if (__unlikely(__offsetof(struct var, name) + v->namel + 1 + 21 > __alloc_size(v)))
			KERNEL$SUICIDE("SHELL_NEXT_COMMAND: UNDERALLOCATED VARIABLE %s (VALUE %"__64_format"d): %ld > %ld", v->name, val, (unsigned long)__offsetof(struct var, name) + v->namel + 1 + 21, (unsigned long)__alloc_size(v));
		_snprintf(v->name + v->namel + 1, 21, "%"__64_format"d", res);
		if (__likely(v->for_step > 0)) {
			if (__unlikely(res > v->for_to)) goto fall_through;
		} else if (__likely(v->for_step < 0)) {
			if (__unlikely(res < v->for_to)) goto fall_through;
		}
		goto goto_for;
	}
	fall_through:
	v->for_ptr = NULL;
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_EXIT_COMMAND, SPL_SHELL, CTX)
{
	RQ->return_ignore_err = 0;
	if (__likely(RQ->argv[1] != NULL)) {
		long code;
		if (__unlikely(__get_number(RQ->argv[1], strchr(RQ->argv[1], 0), 1, &code))) {
			badsyn:
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "EXIT: SYNTAX ERROR");
			RQ->return_val = -EBADSYN;
			RETURN_TO_SHELL(RQ);
		}
		RQ->return_val = code;
		if (__unlikely(RQ->argv[2] != NULL)) {
			if (__unlikely(RQ->argv[3] != NULL)) goto badsyn;
			strlcpy(RQ->return_msg, RQ->argv[2], __MAX_STR_LEN);
		} else RQ->return_msg[0] = 0;
	}
	RQ->internal_flags |= SHELL_DO_EXIT;
	RETURN_TO_SHELL(RQ);
}

DECL_AST(SHELL_SHIFT_COMMAND, SPL_SHELL, CTX)
{
	int chars = 0;
	unsigned n = 1;
	char **argv = RQ->argv;
	int state = 0;
	struct __param_table params[] = {
		"CHARS", __PARAM_BOOL, ~0, 1, NULL,
		"", __PARAM_UNSIGNED_INT, 0, MAXUINT, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &chars;
	params[1].__p = &n;
	if (__unlikely(__parse_params(&argv, &state, params, NULL, NULL, NULL))) {
		_snprintf(RQ->return_msg, __MAX_STR_LEN, "SHIFT: SYNTAX ERROR");
		RQ->return_val = -EBADSYN;
		RETURN_TO_SHELL(RQ);
	}
	if (!chars) {
		if (__likely(RQ->rq->narg > n)) RQ->rq->narg -= n, RQ->rq->arg += n;
		else RQ->rq->narg = 0;
	} else {
		if (__likely(RQ->rq->narg > 1)) {
			__const__ char **a = &RQ->rq->arg[1];
			if (__likely(strlen(*a) > n)) *a += n;
			else *a = "";
		}
	}
	RETURN_TO_SHELL(RQ);
}

static size_t DO_LOCAL(CTX *ctx, char *name)
{
	struct var *v, *vv;
	size_t sz;
	if ((v = find_var(ctx, name, name + strlen(name), 1))) {
		DEL_FROM_LIST(&v->hash);
		goto add_it;
	}
	v = malloc(sz = sizeof(struct var) + strlen(name) + 1);
	if (__unlikely(!v)) return sz;
	init_var(v);
	v->namel = strlen(name);
	strcpy(v->name, name);
	__upcase(v->name);
	add_it:
	XLIST_FOR_EACH(vv, &ctx->rq->local, struct var, hash) if (vv->namel == v->namel && __unlikely(!_memcasecmp(vv->name, v->name, vv->namel))) {
		if (__unlikely(getenv(name) != NULL)) goto add_empty;
		free_var(v);
		goto skip;
	}
	ADD_TO_XLIST(&ctx->rq->local, &v->hash);
	if (__unlikely(getenv(name) != NULL)) {
		v = malloc(sz = sizeof(struct var) + strlen(name) + 1);
		if (__unlikely(!v)) return sz;
		add_empty:
		strcpy(stpcpy(v->name, name), "=");
		init_var(v);
		add_var(ctx, v);
	}
	skip:
	return 0;
}

DECL_AST(SHELL_LOCAL_COMMAND, SPL_SHELL, CTX)
{
	size_t sz;
	another:
	if (__unlikely(!RQ->argv[1])) RETURN_TO_SHELL(RQ);
	sz = DO_LOCAL(RQ, RQ->argv[1]);
	if (__unlikely(sz != 0)) {
		int r;
		if (__unlikely(r = KERNEL$OOM_ERR(sz))) {
			RQ->return_ignore_err = 0;
			_snprintf(RQ->return_msg, __MAX_STR_LEN, "LOCAL: CAN'T ALLOCATE: %s", strerror(-r));
			RQ->return_val = r;
			RETURN_TO_SHELL(RQ);
		}
		RETURN_IORQ(&RQ->u.rq, KERNEL$WAIT_ON_FREEMEM);
	}
	RQ->argv++;
	goto another;
}

DECL_AST(SHELL_REM_COMMAND, SPL_SHELL, CTX)
{
	RETURN_TO_SHELL(RQ);
}

/* list.next, list.prev, len (set to 0), name, flags, command */

__const__ static CMD default_commands[] = {
	"IMAGE", 0, &SHELL_IMAGE_COMMAND,
	"DEVICE", 0, &SHELL_IMAGE_COMMAND,
	"DCTL", 0, &SHELL_DCTL_COMMAND,
	"UNLOAD", 0, &SHELL_DCTL_COMMAND,
	"CTTY", 0, &SHELL_CTTY_COMMAND,
	"INT", 0, &SHELL_INT_COMMAND,
	"DTTY", 0, &SHELL_DTTY_COMMAND,
	"ASSIGN", 0, &SHELL_ASSIGN_COMMAND,
	"DEASSIGN", 0, &SHELL_DEASSIGN_COMMAND,
	"SETLOGICAL", 0, &SHELL_SETLOGICAL_COMMAND,
	"LOGICAL", 0, &SHELL_LOGICAL_COMMAND,
	"MKDIR", 0, &SHELL_MKDIR_COMMAND,
	"MKFILE", 0, &SHELL_MKFILE_COMMAND,
	"RMDIR", 0, &SHELL_RMDIR_COMMAND,
	"DEL", 0, &SHELL_DEL_COMMAND,
	"FSYNC", 0, &SHELL_FSYNC_COMMAND,
	"DIR", CMDF_SYNC_COMMAND, (void *)&SHELL_DIR_COMMAND,
	"COPY", 0, &SHELL_COPY_COMMAND,
	"MOVE", 0, &SHELL_MOVE_COMMAND,
	"CD", 0, &SHELL_CD_COMMAND,
	"SPAWN", 0, &SHELL_SPAWN_COMMAND,
	"FG", 0, &SHELL_FG_COMMAND,
	"REMAP", 0, &SHELL_REMAP_COMMAND,
	"JOBS", 0, &SHELL_JOBS_COMMAND,
	"STOP", 0, &SHELL_PROC_CTRL_COMMAND,
	"RESUME", 0, &SHELL_PROC_CTRL_COMMAND,
	"ZAP", 0, &SHELL_PROC_CTRL_COMMAND,
	"SET", CMDF_DONT_CLEAR_STATUS, &SHELL_SET_COMMAND,
	"LET", CMDF_DONT_CLEAR_STATUS, &SHELL_LET_COMMAND,
	"PRINT", CMDF_DONT_CLEAR_STATUS, &SHELL_PRINT_COMMAND,
	"INK", CMDF_DONT_CLEAR_STATUS, &SHELL_COLOR_COMMAND,
	"PAPER", CMDF_DONT_CLEAR_STATUS, &SHELL_COLOR_COMMAND,
	"BRIGHT", CMDF_DONT_CLEAR_STATUS, &SHELL_COLOR_COMMAND,
	"FLASH", CMDF_DONT_CLEAR_STATUS, &SHELL_COLOR_COMMAND,
	"INVERSE", CMDF_DONT_CLEAR_STATUS, &SHELL_COLOR_COMMAND,
	"PUTBACK", 0, &SHELL_PUTBACK_COMMAND,
	"PUTBACKN", 0, &SHELL_PUTBACKN_COMMAND,
	"INPUT", 0, &SHELL_INPUT_COMMAND,
	"CALL", 0, &SHELL_CALL_COMMAND,
	"TTYWAIT", 0, &SHELL_TTYWAIT_COMMAND,
	"AUTH", 0, &SHELL_AUTH_COMMAND,
	"NOTAUTH", 0, &SHELL_NOTAUTH_COMMAND,
	"GETAUTH", 0, &SHELL_GETAUTH_COMMAND,
	"FLUSHTTY", 0, &SHELL_FLUSHTTY_COMMAND,
	"CLEARCLIPBOARD", 0, &SHELL_CLEARCLIPBOARD_COMMAND,
	"DETACH", 0, &SHELL_DETACH_COMMAND,
	"GO", CMDF_DONT_CLEAR_STATUS, &SHELL_GOTO_COMMAND,
	"RETURN", CMDF_DONT_CLEAR_STATUS, &SHELL_RETURN_COMMAND,
	"IF", CMDF_DONT_CLEAR_STATUS, &SHELL_IF_COMMAND,
	"FOR", CMDF_DONT_CLEAR_STATUS, &SHELL_FOR_COMMAND,
	"NEXT", CMDF_DONT_CLEAR_STATUS, &SHELL_NEXT_COMMAND,
	"EXIT", CMDF_DONT_CLEAR_STATUS, &SHELL_EXIT_COMMAND,
	"SHIFT", CMDF_DONT_CLEAR_STATUS, &SHELL_SHIFT_COMMAND,
	"LOCAL", CMDF_DONT_CLEAR_STATUS, &SHELL_LOCAL_COMMAND,
	"REM", CMDF_DONT_CLEAR_STATUS, &SHELL_REM_COMMAND,
	NULL, 0, NULL,
};

static CMD *commands = (CMD *)default_commands;

CMD *FIND_CMD(__const__ char *name)
{
	CMD *f;
	for (f = commands; f->name; f++) if (__unlikely(!_strcasecmp(f->name, name))) return f;
	return NULL;
}

int SHELL$REGISTER_COMMAND(CMD *cmd)
{
	CMD *nc;
	int i;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_SHELL), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("SHELL$REGISTER_COMMAND AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_SHELL);
	if (__unlikely(FIND_CMD(cmd->name) != NULL)) {
		LOWER_SPLX(spl);
		return -EEXIST;
	}
	for (i = 0; commands[i].name; i++) ;
	if (__unlikely(commands == default_commands)) {
		if (__unlikely(!(nc = malloc((i + 2) * sizeof(CMD))))) {
			LOWER_SPLX(spl);
			return 1;
		}
		memcpy(nc, default_commands, i * sizeof(CMD));
	} else {
		if (__unlikely(!(nc = realloc(commands, (i + 2) * sizeof(CMD))))) {
			LOWER_SPLX(spl);
			return 1;
		}
	}
	commands = nc;
	memcpy(&nc[i], cmd, sizeof(CMD));
	memset(&nc[i + 1], 0, sizeof(CMD));
	LOWER_SPLX(spl);
	return 0;
}

