#include <STDLIB.H>
#include <UNISTD.H>
#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <DLFCN.H>
#include <SYS/STAT.H>
#include <VALUES.H>
#include <MD5.H>
#include <SHA.H>
#include <RIPEMD.H>
#include <SHA256.H>

#include <REGEX.H>
#include <GLOB.H>

#include <SPAD/SHELLM.H>
#include "SHELL.H"

#define check_range(ctx)	1

static int get_number(CTX *ctx, char *num, __s64 *p)
{
	if (__unlikely(__get_64_number(num, num + strlen(num), 1, p))) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "BAD NUMBER (%s)", num);
		ctx->return_val = -EINVAL;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return 1;
	}
	return 0;
}

static char *put_number(__s64 n)
{
	char *a;
	if (__unlikely(!(a = __sync_malloc(21)))) return (char *)1;
	_snprintf(a, 21, "%"__64_format"d", n);
	return a;
}

static char *put_logical(int n)
{
	char *a;
	if (__unlikely(!(a = __sync_malloc(2)))) return (char *)1;
	a[0] = '0' + !!n;
	a[1] = 0;
	return a;
}

static char *alloc(char *r)
{
	char *a;
	if (__unlikely(!r)) r = "";
	a = __sync_malloc(strlen(r) + 1);
	if (__unlikely(!a)) return (char *)1;
	return strcpy(a, r);
}

static char *len(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_number(strlen(args[0]));
}

static char *chr$(CTX *ctx, int n_args, char **args, char *fn)
{
	char *a;
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely((__u64)a1 >= 256) && check_range(ctx)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN CHR$ (%"__64_format"d)", a1);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	if (__unlikely(!(a = __sync_malloc(2)))) return (char *)1;
	a[0] = a1;
	a[1] = 0;
	return a;
}

static char *code(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_number(args[0][0]);
}

static char *abs(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely((__u64)a1 == (__u64)1 << 63) && check_range(ctx)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN ABS (%"__64_format"d)", a1);
		ctx->return_val = -EOVERFLOW;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(__unlikely(a1 < 0) ? -(__u64)a1 : (__u64)a1);
}

static char *sgn(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	return put_number(__likely(a1 > 0) ? 1 : __likely(!a1) ? 0 : -1);
}

static char *sqr(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1;
	int i;
	__u64 r;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(a1 < 0)) {
		if (!check_range(ctx)) return put_logical(0);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN SQR (%"__64_format"d)", a1);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	r = 0;
	for (i = sizeof(__s64) * 4 - 1; i >= 0; i--) {
		__u64 rr, or = r;
		r |= (__u64)1 << i;
		rr = r * r;
		if (__likely(rr > a1)) r = or;
		else if (__unlikely(rr == a1)) break;
	}
	return put_number(r);
}

struct val_param {
	CTX *ctx;
	char *arg;
	int fr;
};

static long val_thread(void *val_param_);

static char *val(CTX *ctx, int n_args, char **args, char *fn)
{
	char *r;
	/*int fr = 0;
	r = EVAL_EXPRESSION(ctx, args[0], args[0] + strlen(args[0]), &fr); */
	struct val_param val_param;
	THREAD_RQ tr;
	val_param.ctx = ctx;
	val_param.arg = args[0];
	val_param.fr = 0;
	tr.thread_main = val_thread;
	tr.p = &val_param;
	tr.error = NULL;
	tr.cwd = KERNEL$CWD();
	tr.std_in = KERNEL$STDIN();
	tr.std_out = KERNEL$STDOUT();
	tr.std_err = KERNEL$STDERR();
	tr.dlrq = NULL;
	tr.thread = NULL;
	tr.spawned = 0;
	SYNC_IO_CANCELABLE(&tr, KERNEL$THREAD);
	if (__unlikely(!tr.spawned)) {
		if (tr.status != -EINTR) _snprintf(ctx->return_msg, __MAX_STR_LEN, "CAN'T SPAWN THREAD: %s", strerror(-tr.status));
		ctx->return_val = tr.status;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	r = (char *)tr.status;
	if (__unlikely(ctx->internal_flags & SHELL_THREAD_ERROR)) {
		if (__unlikely(val_param.fr)) __slow_free(r);
		return NULL;
	}
	if (__unlikely(!val_param.fr)) return alloc(r);
	return r;
}

static long val_thread(void *val_param_)
{
	struct val_param *val_param = val_param_;
	return (long)EVAL_EXPRESSION(val_param->ctx, val_param->arg, val_param->arg + strlen(val_param->arg), &val_param->fr);
}

static char *to_base(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 num_s, base_l;
	__u64 num;
	unsigned base;
	char *str, *pstr, *ostr;
	if (__unlikely(get_number(ctx, args[0], &num_s))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &base_l))) return NULL;
	if (__unlikely((__u64)base_l - 2 > 36 - 2)) {
		if (!check_range(ctx)) return put_logical(0);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN TO_BASE (BASE %"__64_format"d)", base_l);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	base = base_l;
	if (__unlikely(!(str = __sync_malloc(66)))) return (char *)1;
	if (__likely(num_s >= 0)) {
		num = num_s;
		pstr = str;
	} else {
		num = -(__u64)num_s;
		str[0] = '-';
		pstr = str + 1;
	}
	ostr = pstr;
	do {
		unsigned digit = num % base;
		if (digit >= 10) digit += 'A' - '0' - 10;
		*pstr++ = digit + '0';
	} while ((num = num / base));
	if (__unlikely(pstr - ostr > 64))
		KERNEL$SUICIDE("to_base: TOO MANY DIGITS IN NUMBER, SHOT TO MEMORY (BASE %u, LENGTH %ld)", base, (long)(pstr - ostr));
	*pstr-- = 0;
	while (pstr > ostr) {
		char c = *ostr;
		*ostr = *pstr;
		*pstr = c;
		ostr++;
		pstr--;
	}
	return str;
}

static char *from_base(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 base_l;
	__u64 num;
	unsigned base;
	char *str;
	if (__unlikely(get_number(ctx, args[2], &base_l))) return NULL;
	if (__unlikely((__u64)base_l - 2 > 36 - 2)) {
		if (!check_range(ctx)) put_0: return put_logical(0);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN FROM_BASE (BASE %"__64_format"d)", base_l);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	base = base_l;
	str = args[0];
	if (*str == '-') str++;
	if (__unlikely(!*str)) {
		if (!check_range(ctx)) goto put_0;
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "EMPTY NUMBER IN FROM_BASE");
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	num = 0;
	do {
		__u64 newnum;
		unsigned digit;
		if (__likely(*str >= '0') && __likely(*str <= '9')) digit = *str - '0';
		else if (__likely(*str >= 'A') && __likely(*str <= 'Z')) digit = *str - 'A' + 10;
		else if (__likely(*str >= 'a') && __likely(*str <= 'z')) digit = *str - 'a' + 10;
		else {
			invl_digit:
			if (!check_range(ctx)) goto put_0;
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "INVALID CHARACTER IN FROM_BASE (BASE %"__64_format"d)", base_l);
			ctx->return_val = -EDOM;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		}
		if (__unlikely(digit >= base)) goto invl_digit;
		newnum = num * base + digit;
		if (check_range(ctx) && __unlikely(newnum / base != num)) {
			overflow:
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN FROM_BASE");
			ctx->return_val = -EOVERFLOW;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		}
		num = newnum;
	} while (*++str);
	if (__likely(args[0][0] != '-')) {
		if (check_range(ctx) && num >= (__u64)1 << 63) goto overflow;
		return put_number(num);
	} else {
		if (check_range(ctx) && num > (__u64)1 << 63) goto overflow;
		return put_number(-num);
	}
}

static char *rnd(CTX *ctx, int n_args, char **args, char *fn)
{
	int i;
	__u64 ar;
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(a1 <= 0)) {
		if (!check_range(ctx)) return put_logical(0);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN RND (%"__64_format"d)", a1);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	again:
	ar = 0;
	for (i = 0; i < sizeof(__u64) / sizeof(uint32_t); i++) {
		ar <<= sizeof(uint32_t) * 8 - 1;
		ar <<= 1;
		ar |= arc4random();
	}
	ar &= ~((__u64)1 << 63);
	if (__unlikely(ar >= ((__u64)1 << 63) - ((__u64)1 << 63) % a1)) goto again;
	return put_number(ar % a1);
}

static char *inkey$(CTX *ctx, int n_args, char **args, char *fn)
{
	int h;
	char *path, *oldpath;
	char c[2];
	if (__unlikely(ctx->ctty_in == -1)) return alloc(NULL);
	oldpath = KERNEL$HANDLE_PATH(ctx->ctty_in);
	if (__unlikely(!oldpath)) return alloc(NULL);
#define inkey_mode "/^ECHO=0/^ICANON=0"
#define inkey_mode_nbl "/^ECHO=0/^ICANON=0/^NONBLOCK"
	if (!_strcasecmp(fn, "INKEY$")) {
		path = alloca(strlen(oldpath) + strlen(inkey_mode_nbl) + 1);
		strcpy(stpcpy(path, oldpath), inkey_mode_nbl);
	} else {
		path = alloca(strlen(oldpath) + strlen(inkey_mode) + 1);
		strcpy(stpcpy(path, oldpath), inkey_mode);
	}
#undef inkey_mode
#undef inkey_mode_nbl
	if (__unlikely((h = open(path, O_RDONLY)) == -1)) h = ctx->ctty_in;
	if (__unlikely(read(h, c, 1) != 1)) {
		if (__likely(h != ctx->ctty_in)) close(h);
		return alloc(NULL);
	}
	if (__likely(h != ctx->ctty_in)) close(h);
	c[1] = 0;
	return alloc(c);
}

static char *hashsum(CTX *ctx, int n_args, char **args, char *fnname, int len)
{
	char *r;
	void *lib;
	char *(*hashdata)(const unsigned char *data, unsigned int len, char *buf);
	if (__unlikely(!(lib = liblink(ctx, "MD.DLL", fnname, &hashdata, NULL)))) return NULL;
	if (__unlikely(!(r = __sync_malloc(len)))) {
		libclose(lib);
		return (char *)1;
	}
	hashdata(args[0], strlen(args[0]), r);
	libclose(lib);
	return r;
}

static char *md5sum(CTX *ctx, int n_args, char **args, char *fn)
{
	return hashsum(ctx, n_args, args, "MD5Data", MD5_HASHBYTES * 2 + 1);
}

static char *sha1sum(CTX *ctx, int n_args, char **args, char *fn)
{
	return hashsum(ctx, n_args, args, "SHA1_Data", SHA_DIGEST_LENGTH * 2 + 1);
}

static char *ripemd160sum(CTX *ctx, int n_args, char **args, char *fn)
{
	return hashsum(ctx, n_args, args, "RIPEMD160_Data", RIPEMD160_DIGEST_LENGTH * 2 + 1);
}

static char *sha256sum(CTX *ctx, int n_args, char **args, char *fn)
{
	return hashsum(ctx, n_args, args, "SHA256_Data", SHA256_DIGEST_LENGTH * 2 + 1);
}

static char *ctty_in(CTX *ctx, int n_args, char **args, char *fn)
{
	if (__unlikely(ctx->ctty_in == -1)) return alloc(NULL);
	else return alloc(KERNEL$HANDLE_PATH(ctx->ctty_in));
}

static char *ctty_out(CTX *ctx, int n_args, char **args, char *fn)
{
	if (__unlikely(ctx->ctty_out == -1)) return alloc(NULL);
	else return alloc(KERNEL$HANDLE_PATH(ctx->ctty_out));
}

static char *ctty_err(CTX *ctx, int n_args, char **args, char *fn)
{
	if (__unlikely(ctx->ctty_err == -1)) return alloc(NULL);
	else return alloc(KERNEL$HANDLE_PATH(ctx->ctty_err));
}

static char *exist(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(!access(args[0], F_OK));
}

static char *existr(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(!access(args[0], R_OK));
}

static char *existw(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(!access(args[0], W_OK));
}

static char *existd(CTX *ctx, int n_args, char **args, char *fn)
{
	struct stat st;
	if (__unlikely(stat(args[0], &st))) return put_logical(0);
	return put_logical(S_ISDIR(st.st_mode));
}

static char *errcode(CTX *ctx, int n_args, char **args, char *fn)
{
	int e = _err_code(args[0]);
	if (__unlikely(e == -1)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "BAD ERROR CODE (%s)", args[0]);
		ctx->return_val = -EINVAL;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(e);
}

static char *plus(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2, res;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	res = (__u64)a1 + (__u64)a2;
	if (check_range(ctx) && __likely(!((unsigned long)((__u64)a1 >> 63) ^ (unsigned long)((__u64)a2 >> 63))) && __unlikely((unsigned long)((__u64)a1 >> 63) ^ (unsigned long)((__u64)res >> 63))) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN %"__64_format"d + %"__64_format"d", a1, a2);
		ctx->return_val = -EOVERFLOW;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(res);
}

static char *minus(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2, res;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	res = (__u64)a1 - (__u64)a2;
	if (check_range(ctx) && __unlikely((unsigned)((__u64)a1 >> 63) ^ (unsigned)((__u64)a2 >> 63)) && __unlikely((unsigned)((__u64)a1 >> 63) ^ (unsigned)((__u64)res >> 63))) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN %"__64_format"d - %"__64_format"d", a1, a2);
		ctx->return_val = -EOVERFLOW;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(res);
}

static char *unary_minus(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely((__u64)a1 == (__u64)1 << 63) && check_range(ctx)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN UNARY - %"__64_format"d", a1);
		ctx->return_val = -EOVERFLOW;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(-(__u64)a1);
}

static int multiply_check_ovf(__s64 a1, __s64 a2, __s64 *res)
{
	*res = (__u64)a1 * (__u64)a2;
	if (__unlikely(!a2)) return 0;
	if (__unlikely(a2 == -1)) return __unlikely((__u64)a1 == (__u64)1 << 63);
	return __unlikely((*res / a2) != a1);
}

static char *multiply(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2, res;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (__unlikely(multiply_check_ovf(a1, a2, &res)) && check_range(ctx)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN %"__64_format"d * %"__64_format"d", a1, a2);
		ctx->return_val = -EOVERFLOW;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	return put_number(res);
}

static char *divide(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (__unlikely(!a2)) {
		if (check_range(ctx)) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN %"__64_format"d / %"__64_format"d", a1, a2);
			ctx->return_val = -EDOM;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		} else {
			return put_logical(0);
		}
	}
	if  (__unlikely(a2 == -1) && __unlikely((__u64)a1 == (__u64)1 << 63)) {
		if (check_range(ctx)) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN %"__64_format"d / %"__64_format"d", a1, a2);
			ctx->return_val = -EOVERFLOW;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		} else {
			return put_number((__u64)1 << 63);
		}
	}
	return put_number(a1 / a2);
}

static char *reminder(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (__unlikely(a2 <= 0)) {
		if (check_range(ctx)) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN %"__64_format"d // %"__64_format"d", a1, a2);
			ctx->return_val = -EDOM;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		} else {
			return put_logical(0);
		}
	}
	return put_number(a1 % a2);
}

static char *power(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	__s64 r;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (__unlikely(a2 < 0)) {
		if (!check_range(ctx)) return put_number(0);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN %"__64_format"d ^ %"__64_format"d", a1, a2);
		ctx->return_val = -EDOM;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	r = 1;
	while (1) {
		if (a2 & 1) if (__unlikely(multiply_check_ovf(r, a1, &r)) && check_range(ctx)) {
			overflow:
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OVERFLOW IN %"__64_format"d ^ %"__64_format"d", a1, a2);
			ctx->return_val = -EOVERFLOW;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		}
		a2 >>= 1;
		if (!a2) break;
		if (__unlikely(multiply_check_ovf(a1, a1, &a1)) && check_range(ctx)) goto overflow;
	}
	return put_number(r);
}

static char *concat(CTX *ctx, int n_args, char **args, char *fn)
{
	char *r;
	if (__unlikely(!(r = __sync_malloc(strlen(args[0]) + strlen(args[2]) + 1))))
		return (char *)1;
	strcpy(stpcpy(r, args[0]), args[2]);
	return r;
}

	/* must take the same time no matter of strings
		(security --- it can be used to compare passwords */
static __finline__ int safe_strcmp(char *s1, char *s2)
{
	int c = 0;
	do {
		c |= *s1 ^ *s2;
	} while (*s1++ && *s2++);
	return c;
}

static __finline__ int safe_strcasecmp(char *s1, char *s2)
{
	int c = 0;
	do {
		char a = *s1;
		char b = *s2;
		if (a >= 'A' && a <= 'Z') a += 'a' - 'A';
		if (b >= 'A' && b <= 'Z') b += 'a' - 'A';
		c |= a ^ b;
	} while (*s1++ && *s2++);
	return c;
}

static char *equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(!safe_strcmp(args[0], args[2]));
}

static char *not_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(safe_strcmp(args[0], args[2]));
}

static char *case_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(!safe_strcasecmp(args[0], args[2]));
}

static char *case_not_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(safe_strcasecmp(args[0], args[2]));
}

static char *less(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	return put_logical(a1 < a2);
}

static char *greater(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	return put_logical(a1 > a2);
}

static char *less_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	return put_logical(a1 <= a2);
}

static char *greater_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	return put_logical(a1 >= a2);
}


static char *str_less(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(strcmp(args[0], args[2]) < 0);
}

static char *str_greater(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(strcmp(args[0], args[2]) > 0);
}

static char *str_less_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(strcmp(args[0], args[2]) <= 0);
}

static char *str_greater_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(strcmp(args[0], args[2]) >= 0);
}

static char *str_case_less(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(_strcasecmp(args[0], args[2]) < 0);
}

static char *str_case_greater(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(_strcasecmp(args[0], args[2]) > 0);
}

static char *str_case_less_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(_strcasecmp(args[0], args[2]) <= 0);
}

static char *str_case_greater_equal(CTX *ctx, int n_args, char **args, char *fn)
{
	return put_logical(_strcasecmp(args[0], args[2]) >= 0);
}

static char *not(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	return put_logical(!a1);
}

static char *and(CTX *ctx, int n_args, char **args, char *fn)
{
	char *a1;
	__s64 a2;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (a2) a1 = args[0];
	else a1 = "0";
	return alloc(a1);
}

static char *and$(CTX *ctx, int n_args, char **args, char *fn)
{
	char *a1;
	__s64 a2;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	if (a2) a1 = args[0];
	else a1 = "";
	return alloc(a1);
}

static char *or(CTX *ctx, int n_args, char **args, char *fn)
{
	__s64 a1, a2;
	if (__unlikely(get_number(ctx, args[0], &a1))) return NULL;
	if (__unlikely(get_number(ctx, args[2], &a2))) return NULL;
	return put_logical(a1 || a2);
}

char *subchr(CTX *ctx, char *str, char *pos)
{
	__s64 a1;
	char *r;
	if (__unlikely(get_number(ctx, pos, &a1))) return NULL;
	if (__unlikely(a1 < 1) || __unlikely(a1 > strlen(str))) return alloc(NULL);
	if (__unlikely(!(r = __sync_malloc(2)))) return (char *)1;
	r[0] = str[a1 - 1];
	r[1] = 0;
	return r;
}

char *substr(CTX *ctx, char *str, char *from, char *to)
{
	size_t l;
	__s64 a1, a2;
	char *r;
	if (__unlikely(get_number(ctx, from, &a1))) return NULL;
	if (__unlikely(get_number(ctx, to, &a2))) return NULL;
	if (__unlikely(a2 < 1) || __unlikely(a1 > strlen(str)) || __unlikely(a1 > a2)) return alloc(NULL);
	if (__unlikely(a1 < 1)) a1 = 1;
	if (__unlikely(a2 > (l = strlen(str)))) a2 = l;
	if (__unlikely(!(r = __sync_malloc(a2 - a1 + 2)))) return (char *)1;
	memcpy(r, str + a1 - 1, a2 - a1 + 1);
	r[a2 - a1 + 1] = 0;
	return r;
}

typedef int (*regcomp_t)(regex_t *, const char *, int);
typedef int (*regexec_t)(const regex_t *, const char *, size_t, regmatch_t[], int);
typedef int (*regfree_t)(regex_t *);

static char *rematch(CTX *ctx, int n_args, char **args, char *fn)
{
	regex_t r;
	int v;
	regcomp_t regcomp;
	regexec_t regexec;
	regfree_t regfree;
	void *libc;
	if (__unlikely(!(libc = liblink(ctx, "LIBC.DLL", "regcomp", &regcomp, "regexec", &regexec, "regfree", &regfree, NULL)))) return NULL;
	if (__unlikely(regcomp(&r, args[2], REG_EXTENDED))) {
		libclose(libc);
		return put_logical(0);
	}
	v = regexec(&r, args[0], 0, NULL, 0);
	regfree(&r);
	libclose(libc);
	return put_logical(!v);
}

static char *resubst(CTX *ctx, int n_args, char **args, char *fn)
{
	regex_t r;
	regcomp_t regcomp;
	regexec_t regexec;
	regfree_t regfree;
	void *libc;
	regmatch_t match[10];
	char *a, *ret, *a4;
	int rp, a4l;
	if (__unlikely(!(libc = liblink(ctx, "LIBC.DLL", "regcomp", &regcomp, "regexec", &regexec, "regfree", &regfree, NULL)))) return NULL;
	if (__unlikely(regcomp(&r, args[2], REG_EXTENDED))) {
		libclose(libc);
		return alloc("");
	}
	a = args[0];
	ret = NULL;
	rp = 0;
	a4l = strlen(args[4]);
	if (__unlikely(strchr(args[4], '\\') != NULL)) a4l = -1;
	again:
	if (regexec(&r, a, 10, match, 0)) {
		finish1:
		ret = __sync_reallocf(ret, rp + strlen(a) + 1);
		if (__unlikely(!ret)) goto do_ret;
		memcpy(ret + rp, a, strlen(a) + 1);
		goto do_ret;
	}
	if (__likely(a4l != -1)) {
		ret = __sync_reallocf(ret, rp + match[0].rm_so + a4l);
		if (__unlikely(!ret)) goto do_ret;
		memcpy(ret + rp, a, match[0].rm_so);
		memcpy(ret + rp + match[0].rm_so, args[4], a4l);
		rp += match[0].rm_so + a4l;
		goto aagain;
	}
	ret = __sync_reallocf(ret, rp + match[0].rm_so);
	if (__unlikely(!ret)) goto do_ret;
	memcpy(ret + rp, a, match[0].rm_so);
	rp += match[0].rm_so;
	for (a4 = args[4]; *a4; a4++) {
		if (__likely(*a4 != '\\')) {
			ret = __sync_realloc(ret, ++rp);
			if (__unlikely(!ret)) goto do_ret;
			ret[rp - 1] = *a4;
		} else {
			unsigned q = a4[1] - '0';
			if (__likely(q < 10)) {
				a4++;
				ret = __sync_realloc(ret, rp + match[q].rm_eo - match[q].rm_so);
				if (__unlikely(!ret)) goto do_ret;
				memcpy(ret + rp, a + match[q].rm_so, match[q].rm_eo - match[q].rm_so);
				rp += match[q].rm_eo - match[q].rm_so;
			}
		}
	}
	aagain:
	a += match[0].rm_eo;
	if (!*a) goto finish1;
	if (match[0].rm_so == match[0].rm_eo) {
		ret = __sync_reallocf(ret, rp + 1);
		if (__unlikely(!ret)) goto do_ret;
		ret[rp++] = *a++;
	}
	if (__likely(!_strcasecmp(fn, "RESUBST1")) || __unlikely(args[2][0] == '^')) goto finish1;
	goto again;
	
	do_ret:
	regfree(&r);
	libclose(libc);
	if (__unlikely(!ret)) ret = (void *)1;
	return ret;
}

static char *glb(CTX *ctx, int n_args, char **args, char *fn)
{
	int i;
	__s64 idx;
	glob_t g;
	size_t len;
	void *libc;
	char *p, *ret;
	int (*pglob)(const char *, int, int(*)(const char *, int), glob_t *);
	void (*pglobfree)(glob_t *);
	if (__unlikely(n_args < 1) || __unlikely(n_args > 2)) {
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "EXPRESSION SYNTAX ERROR");
		ctx->return_val = -EBADSYN;
		return NULL;
	}
	idx = -1;
	if (n_args == 2) {
		if (__unlikely(get_number(ctx, args[2], &idx))) return NULL;
		if (__unlikely(idx < 0)) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "OUT OF RANGE IN GLOB (%"__64_format"d)", idx);
			ctx->return_val = -EDOM;
			ctx->internal_flags |= SHELL_THREAD_ERROR;
			return NULL;
		}
	}
	if (__unlikely(!(libc = liblink(ctx, "LIBC.DLL", "glob", &pglob, "globfree", &pglobfree, NULL)))) return NULL;
	if (__unlikely(pglob(args[0], GLOB_NOESCAPE, NULL, &g))) {
		ret_empty:
		ret = alloc("");
		goto ret_ret;
	}
	if (idx >= 0) {
		if (__unlikely(idx >= g.gl_pathc)) goto ret_empty;
		ret = alloc(g.gl_pathv[idx]);
		goto ret_ret;
	}
	len = 0;
	for (i = 0; i < g.gl_pathc; i++) len += strlen(g.gl_pathv[i]) + 1;
	ret = __sync_malloc(len);
	if (__unlikely(!ret)) {
		ret = (void *)1;
		goto ret_ret;
	}
	p = ret;
	for (i = 0; ; ) {
		p = stpcpy(p, g.gl_pathv[i]);
		if (++i >= g.gl_pathc) break;
		*p++ = ' ';
	}
	ret_ret:
	pglobfree(&g);
	libclose(libc);
	return ret;
}

static char *exec(CTX *ctx, int n_args, char **args, char *fn)
{
	SHRCRQ rc;
	SHCCRQ cc;
	int i;
	char filename[10 + 16 + 1];
	int h = -1;	/* warning go away */
	ssize_t r;
	struct stat st;
	size_t p, pp;
	char *a;
	if (!n_args) return alloc("");
	rc.ptr = args[0];
	rc.len = strlen(args[0]);
	rc.arg = __sync_malloc(n_args * sizeof(char *));
	if (__unlikely(!rc.arg)) return (char *)1;
	for (i = 0; i < n_args; i++) {
		rc.arg[i] = args[i << 1];
	}
	rc.narg = n_args;
	SYNC_IO(&cc, SHELL$CREATE_CONTEXT);
	if (__unlikely(cc.status < 0)) {
		__slow_free(rc.arg);
		return alloc(strerror(-cc.status));
	}
	rc.ctx = cc.ctx;
	cc.ctx->var_chain = ctx;
	if (__likely(_strcasecmp(fn, "EXEC_CODE"))) {
		_snprintf(filename, sizeof(filename), "TMP:/SHELL%08X%08X%08X%08X", (unsigned)arc4random(), (unsigned)arc4random(), (unsigned)arc4random(), (unsigned)arc4random());
		h = open(filename, O_RDWR | O_CREAT | O_EXCL, 0600);
		if (__unlikely(h == -1)) {
			__slow_free(rc.arg);
			SHELL$DESTROY_CONTEXT(cc.ctx);
			return alloc(strerror(errno));
		}
		if (__likely(!_strcasecmp(fn, "EXEC"))) {
			SHELL$SET_TTY(cc.ctx, ctx->ctty_in, h, ctx->ctty_err, ctx->ctty_in_t, 0, ctx->ctty_err_t, 1);
		} else {
			SHELL$SET_TTY(cc.ctx, ctx->ctty_in, ctx->ctty_out, h, ctx->ctty_in_t, ctx->ctty_out_t, 0, 1);
		}
	} else {
		SHELL$SET_TTY(cc.ctx, ctx->ctty_in, ctx->ctty_out, ctx->ctty_err , ctx->ctty_in_t, ctx->ctty_out_t, ctx->ctty_err_t, 1);
	}
	SHELL$SET_CWD(cc.ctx, ctx->cwd, 1);
	SYNC_IO(&rc, SHELL$RUN_COMMANDS);
	free(rc.arg);
	SHELL$DESTROY_CONTEXT(cc.ctx);
	if (__unlikely(!_strcasecmp(fn, "EXEC_CODE"))) {
		return put_number(rc.status);
	}
	/*	... not sure if desired ... probably not, because with this,
		login script exits if it cannot exec uname
	if (__unlikely(rc.status < 0)) {
		close(h);
		unlink(filename);
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "%s", rc.error_msg);
		ctx->return_val = rc.status;
		return NULL;
	}
	*/
	if (__unlikely(fstat(h, &st) == -1)) {
		int er;
		error:
		er = errno;
		close(h);
		unlink(filename);
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "%s", strerror(er));
		ctx->return_val = -EINVAL;
		ctx->internal_flags |= SHELL_THREAD_ERROR;
		return NULL;
	}
	if (__unlikely(st.st_size >= SIZE_T_MAX)) {
		errno = EFBIG;
		goto error;
	}
	a = __sync_malloc(st.st_size + 1);
	if (__unlikely(!a)) {
		close(h);
		unlink(filename);
		return (char *)1;
	}
	lseek(h, 0, SEEK_SET);
	if (__unlikely((r = read(h, a, st.st_size)) != st.st_size)) {
		if (__unlikely(r != -1)) errno = EEOF;
		goto error;
	}
	close(h);
	unlink(filename);
	for (p = 0, pp = 0; p < st.st_size; p++) {
		if (__likely(a[p] != 0)) a[pp++] = a[p];
	}
	a[pp] = 0;
	if (__likely(pp != 0) && __likely(a[pp - 1] == '\n')) a[pp - 1] = 0;
	return a;
}

static FUNCTION default_functions[] = {
	"LEN", PRIO_FUNCTION, 1, len,
	"CHR$", PRIO_FUNCTION, 1, chr$,
	"CODE", PRIO_FUNCTION, 1, code,
	"ABS", PRIO_FUNCTION, 1, abs,
	"SGN", PRIO_FUNCTION, 1, sgn,
	"SQR", PRIO_FUNCTION, 1, sqr,
	"VAL", PRIO_FUNCTION, 1, val,
	"TO_BASE", PRIO_FUNCTION, 2, to_base,
	"FROM_BASE", PRIO_FUNCTION, 2, from_base,
	"RND", PRIO_FUNCTION, 1, rnd,
	"INKEY$", PRIO_FUNCTION, 0, inkey$,
	"INKEYW$", PRIO_FUNCTION, 0, inkey$,
	"MD5", PRIO_FUNCTION, 1, md5sum,
	"SHA1", PRIO_FUNCTION, 1, sha1sum,
	"RIPEMD160", PRIO_FUNCTION, 1, ripemd160sum,
	"SHA256", PRIO_FUNCTION, 1, sha256sum,
	"CTTY_IN", PRIO_FUNCTION, 0, ctty_in,
	"CTTY_OUT", PRIO_FUNCTION, 0, ctty_out,
	"CTTY_ERR", PRIO_FUNCTION, 0, ctty_err,
	"EXIST", PRIO_FUNCTION, 1, exist,
	"EXISTR", PRIO_FUNCTION, 1, existr,
	"EXISTW", PRIO_FUNCTION, 1, existw,
	"EXISTD", PRIO_FUNCTION, 1, existd,
	"ERRCODE", PRIO_FUNCTION, 1, errcode,
	"REMATCH", PRIO_FUNCTION, 2, rematch,
	"RESUBST", PRIO_FUNCTION, 3, resubst,
	"RESUBST1", PRIO_FUNCTION, 3, resubst,
	"GLOB", PRIO_FUNCTION, -1, glb,
	"EXEC", PRIO_FUNCTION, -1, exec,
	"EXEC_STDERR", PRIO_FUNCTION, -1, exec,
	"EXEC_CODE", PRIO_FUNCTION, -1, exec,
	"-", PRIO_PLUS, 1, unary_minus,
	"NOT", PRIO_NOT, 1, not,
	NULL, 0, 0, NULL,
};

static FUNCTION default_operators[] = {
	"^", PRIO_POWER, 2, power,
	"*", PRIO_MULTIPLY, 2, multiply,
	"/", PRIO_MULTIPLY, 2, divide,
	"//", PRIO_MULTIPLY, 2, reminder,
	"+", PRIO_PLUS, 2, plus,
	"-", PRIO_PLUS, 2, minus,
	"++", PRIO_CONCAT, 2, concat,
	"=", PRIO_COMPARE, 2, equal,
	"=~", PRIO_COMPARE, 2, case_equal,
	"<>", PRIO_COMPARE, 2, not_equal,
	"<>~", PRIO_COMPARE, 2, case_not_equal,
	"<", PRIO_COMPARE, 2, less,
	">", PRIO_COMPARE, 2, greater,
	"<=", PRIO_COMPARE, 2, less_equal,
	">=", PRIO_COMPARE, 2, greater_equal,
	"<<", PRIO_COMPARE, 2, str_less,
	">>", PRIO_COMPARE, 2, str_greater,
	"<<=", PRIO_COMPARE, 2, str_less_equal,
	">>=", PRIO_COMPARE, 2, str_greater_equal,
	"<<~", PRIO_COMPARE, 2, str_case_less,
	">>~", PRIO_COMPARE, 2, str_case_greater,
	"<<=~", PRIO_COMPARE, 2, str_case_less_equal,
	">>=~", PRIO_COMPARE, 2, str_case_greater_equal,
	"AND", PRIO_AND, 2, and,
	"AND$", PRIO_AND, 2, and$,
	"OR", PRIO_OR, 2, or,
	NULL, 0, 0, NULL,
};

static FUNCTION *functions = default_functions;
static FUNCTION *operators = default_operators;

FUNCTION *FIND_FUNCTION(__const__ char *name)
{
	FUNCTION *f;
	for (f = functions; f->name; f++) if (__unlikely(!_strcasecmp(f->name, name))) return f;
	return NULL;
}

FUNCTION *FIND_OPERATOR(__const__ char *name)
{
	FUNCTION *f;
	for (f = operators; f->name; f++) if (__unlikely(!_strcasecmp(f->name, name))) return f;
	return NULL;
}

int OPERATOR_LENGTH(__const__ char *s, __const__ char *e)
{
	FUNCTION *f;
	int maxl = 1;
	for (f = operators; f->name; f++) {
		int l;
		if (__likely((l = strlen(f->name)) <= e - s) && __unlikely(__upcasechr(*s) == __upcasechr(*f->name)) && __likely(!__memcasexcmp(f->name, s, s + l)) && l > maxl) maxl = l;
	}
	return maxl;
}

static int REGISTER(FUNCTION *cmd, FUNCTION **f, FUNCTION *def)
{
	FUNCTION *nc;
	int i;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_SHELL), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("REGISTER AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_SHELL);
	if (__unlikely((def == default_functions ? FIND_FUNCTION : FIND_OPERATOR)(cmd->name) != NULL)) {
		LOWER_SPLX(spl);
		return -EEXIST;
	}
	for (i = 0; (*f)[i].name; i++) ;
	if (__unlikely(*f == def)) {
		if (__unlikely(!(nc = __sync_malloc((i + 2) * sizeof(FUNCTION))))) {
			LOWER_SPLX(spl);
			return 1;
		}
		memcpy(nc, def, i * sizeof(FUNCTION));
	} else {
		if (__unlikely(!(nc = __sync_realloc(*f, (i + 2) * sizeof(FUNCTION))))) {
			LOWER_SPLX(spl);
			return 1;
		}
	}
	*f = nc;
	memcpy(&nc[i], cmd, sizeof(FUNCTION));
	memset(&nc[i + 1], 0, sizeof(FUNCTION));
	LOWER_SPLX(spl);
	return 0;
}

int SHELL$REGISTER_FUNCTION(FUNCTION *fn)
{
	return REGISTER(fn, &functions, default_functions);
}

int SHELL$REGISTER_OPERATOR(FUNCTION *op)
{
	return REGISTER(op, &operators, default_operators);
}
