#include <STDIO.H>
#include <STRING.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <FCNTL.H>
#include <SYS/MMAN.H>
#include <SPAD/LIBC.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TTY.H>
#include <TERMIOS.H>
#include <SYS/IOCTL.H>
#include <ARCH/BARRIER.H>
#include <SIGNAL.H>
#include <VALUES.H>

#include <VGA.H>
#include <VGAMOUSE.H>
#include <VGAKEYBOARD.H>

#include "VGA.H"

#define SIG_SVGALIB	SIGUSR1

static __const__ struct vga_mode {
	short unsigned x;
	short unsigned y;
	unsigned char bpp;
} vga_modes[__GLASTMODE + 1] = {
	0, 0, 0,
	320, 200, 4,
	640, 200, 4,
	640, 350, 4,
	640, 480, 4,
	320, 200, 8,
	320, 240, 8,
	320, 400, 8,
	360, 480, 8,
	640, 480, 1,
	640, 480, 8,
	800, 600, 8,
	1024, 768, 8,
	1280, 1024, 8,
	320, 200, 15,
	320, 200, 16,
	320, 200, 24,
	640, 480, 15,
	640, 480, 16,
	640, 480, 24,
	800, 600, 15,
	800, 600, 16,
	800, 600, 24,
	1024, 768, 15,
	1024, 768, 16,
	1024, 768, 24,
	1280, 1024, 15,
	1280, 1024, 16,
	1280, 1024, 24,
	800, 600, 4,
	1024, 768, 4,
	1280, 1024, 4,
	720, 348, 1,
	320, 200, 32,
	640, 480, 32,
	800, 600, 32,
	1024, 768, 32,
	1280, 1024, 32,
	1152, 864, 4,
	1152, 864, 8,
	1152, 864, 15,
	1152, 864, 16,
	1152, 864, 24,
	1152, 864, 32,
	1600, 1200, 4,
	1600, 1200, 8,
	1600, 1200, 15,
	1600, 1200, 16,
	1600, 1200, 24,
	1600, 1200, 32,
	/*320, 240, 8,*/ 0, 0, 0,
	320, 240, 15,
	320, 240, 16,
	320, 240, 24,
	320, 240, 32,
	400, 300, 8,
	400, 300, 15,
	400, 300, 16,
	400, 300, 24,
	400, 300, 32,
	512, 384, 8,
	512, 384, 15,
	512, 384, 16,
	512, 384, 24,
	512, 384, 32,
	960, 720, 8,
	960, 720, 15,
	960, 720, 16,
	960, 720, 24,
	960, 720, 32,
	1920, 1440, 8,
	1920, 1440, 15,
	1920, 1440, 16,
	1920, 1440, 24,
	1920, 1440, 32,
	/*320, 400, 8,*/ 0, 0, 0,
	320, 400, 15,
	320, 400, 16,
	320, 400, 24,
	320, 400, 32,
	640, 400, 8,
	640, 400, 15,
	640, 400, 16,
	640, 400, 24,
	640, 400, 32,
	320, 480, 8,
	320, 480, 15,
	320, 480, 16,
	320, 480, 24,
	320, 480, 32,
	720, 540, 8,
	720, 540, 15,
	720, 540, 16,
	720, 540, 24,
	720, 540, 32,
	848, 480, 8,
	848, 480, 15,
	848, 480, 16,
	848, 480, 24,
	848, 480, 32,
	1072, 600, 8,
	1072, 600, 15,
	1072, 600, 16,
	1072, 600, 24,
	1072, 600, 32,
	1280, 720, 8,
	1280, 720, 15,
	1280, 720, 16,
	1280, 720, 24,
	1280, 720, 32,
	1360, 768, 8,
	1360, 768, 15,
	1360, 768, 16,
	1360, 768, 24,
	1360, 768, 32,
	1800, 1012, 8,
	1800, 1012, 15,
	1800, 1012, 16,
	1800, 1012, 24,
	1800, 1012, 32,
	1920, 1080, 8,
	1920, 1080, 15,
	1920, 1080, 16,
	1920, 1080, 24,
	1920, 1080, 32,
	2048, 1152, 8,
	2048, 1152, 15,
	2048, 1152, 16,
	2048, 1152, 24,
	2048, 1152, 32,
	2048, 1536, 8,
	2048, 1536, 15,
	2048, 1536, 16,
	2048, 1536, 24,
	2048, 1536, 32,
	512, 480, 8,
	512, 480, 15,
	512, 480, 16,
	512, 480, 24,
	512, 480, 32,
	400, 600, 8,
	400, 600, 15,
	400, 600, 16,
	400, 600, 24,
	400, 600, 32,
	/*400, 300, 8,*/ 0, 0, 0,
};

int vga_version = SVGALIB_VER;

int vga_hasmode(int mode)
{
	int zs;
	char *q, *p;
	if (__unlikely(KERNEL$KERNEL)) return 0;
	if (__unlikely((unsigned)mode > __GLASTMODE)) return 0;
	if (__unlikely(mode == TEXT)) return 1;
	p = KERNEL$HANDLE_PATH(1);
	if (__unlikely(!p)) return 0;
	zs = strlen(p) + 16 + 3 * 10;
	q = alloca(zs);
	_snprintf(q, zs, "%s/^VIDEOMODE=G%dX%dX%d", p, vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp);
	return open(q, O_RDWR | _O_CLOSE) != -1;
}

static char *saved_stdout = NULL, *saved_stderr = NULL;

struct tty_videomode mode_param;
unsigned max_y;
int current_mode = TEXT;

unsigned char *graph_mem = NULL;
unsigned graph_mem_len = 0;
unsigned save_restore_memory = 0;
unsigned current_page;
static unsigned current_display_start;
static unsigned *page_0_backbuffer = NULL;
int vt_active = 0;
typedef void (*switch_fn_t)(void);
static switch_fn_t gotoback = NULL;
static switch_fn_t comefromback = NULL;

static void init_accel_state(void);

static __u8 palette[256 * 4];

static __const__ __u8 default_palette[256 * 4] = {
	0, 0, 0, 0,
	42, 0, 0, 0,
	0, 42, 0, 0,
	42, 42, 0, 0,
	0, 0, 42, 0,
	42, 0, 42, 0,
	0, 21, 42, 0,
	42, 42, 42, 0,
	21, 21, 21, 0,
	63, 21, 21, 0,
	21, 63, 21, 0,
	63, 63, 21, 0,
	21, 21, 63, 0,
	63, 21, 63, 0,
	21, 63, 63, 0,
	63, 63, 63, 0,
	0, 0, 0, 0,
	5, 5, 5, 0,
	8, 8, 8, 0,
	11, 11, 11, 0,
	14, 14, 14, 0,
	17, 17, 17, 0,
	20, 20, 20, 0,
	24, 24, 24, 0,
	28, 28, 28, 0,
	32, 32, 32, 0,
	36, 36, 36, 0,
	40, 40, 40, 0,
	45, 45, 45, 0,
	50, 50, 50, 0,
	56, 56, 56, 0,
	63, 63, 63, 0,
	63, 0, 0, 0,
	63, 0, 16, 0,
	63, 0, 31, 0,
	63, 0, 47, 0,
	63, 0, 63, 0,
	47, 0, 63, 0,
	31, 0, 63, 0,
	16, 0, 63, 0,
	0, 0, 63, 0,
	0, 16, 63, 0,
	0, 31, 63, 0,
	0, 47, 63, 0,
	0, 63, 63, 0,
	0, 63, 47, 0,
	0, 63, 31, 0,
	0, 63, 16, 0,
	0, 63, 0, 0,
	16, 63, 0, 0,
	31, 63, 0, 0,
	47, 63, 0, 0,
	63, 63, 0, 0,
	63, 47, 0, 0,
	63, 31, 0, 0,
	63, 16, 0, 0,
	63, 31, 31, 0,
	63, 31, 39, 0,
	63, 31, 47, 0,
	63, 31, 55, 0,
	63, 31, 63, 0,
	55, 31, 63, 0,
	47, 31, 63, 0,
	39, 31, 63, 0,
	31, 31, 63, 0,
	31, 39, 63, 0,
	31, 47, 63, 0,
	31, 55, 63, 0,
	31, 63, 63, 0,
	31, 63, 55, 0,
	31, 63, 47, 0,
	31, 63, 39, 0,
	31, 63, 31, 0,
	39, 63, 31, 0,
	47, 63, 31, 0,
	55, 63, 31, 0,
	63, 63, 31, 0,
	63, 55, 31, 0,
	63, 47, 31, 0,
	63, 39, 31, 0,
	63, 45, 45, 0,
	63, 45, 49, 0,
	63, 45, 54, 0,
	63, 45, 58, 0,
	63, 45, 63, 0,
	58, 45, 63, 0,
	54, 45, 63, 0,
	49, 45, 63, 0,
	45, 45, 63, 0,
	45, 49, 63, 0,
	45, 54, 63, 0,
	45, 58, 63, 0,
	45, 63, 63, 0,
	45, 63, 58, 0,
	45, 63, 54, 0,
	45, 63, 49, 0,
	45, 63, 45, 0,
	49, 63, 45, 0,
	54, 63, 45, 0,
	58, 63, 45, 0,
	63, 63, 45, 0,
	63, 58, 45, 0,
	63, 54, 45, 0,
	63, 49, 45, 0,
	28, 0, 0, 0,
	28, 0, 7, 0,
	28, 0, 14, 0,
	28, 0, 21, 0,
	28, 0, 28, 0,
	21, 0, 28, 0,
	14, 0, 28, 0,
	7, 0, 28, 0,
	0, 0, 28, 0,
	0, 7, 28, 0,
	0, 14, 28, 0,
	0, 21, 28, 0,
	0, 29, 28, 0,
	0, 28, 21, 0,
	0, 28, 14, 0,
	0, 28, 7, 0,
	0, 28, 0, 0,
	7, 28, 0, 0,
	14, 28, 0, 0,
	21, 28, 0, 0,
	28, 28, 0, 0,
	28, 21, 0, 0,
	28, 14, 0, 0,
	28, 7, 0, 0,
	28, 14, 14, 0,
	28, 14, 17, 0,
	28, 14, 21, 0,
	28, 14, 24, 0,
	28, 14, 28, 0,
	24, 14, 28, 0,
	21, 14, 28, 0,
	17, 14, 28, 0,
	14, 14, 28, 0,
	14, 17, 28, 0,
	14, 21, 28, 0,
	14, 24, 28, 0,
	14, 28, 28, 0,
	14, 28, 24, 0,
	14, 28, 21, 0,
	14, 28, 17, 0,
	14, 28, 14, 0,
	17, 28, 14, 0,
	21, 28, 14, 0,
	24, 28, 14, 0,
	28, 28, 14, 0,
	28, 24, 14, 0,
	28, 21, 14, 0,
	28, 17, 14, 0,
	28, 20, 20, 0,
	28, 20, 22, 0,
	28, 20, 24, 0,
	28, 20, 26, 0,
	28, 20, 28, 0,
	26, 20, 28, 0,
	24, 20, 28, 0,
	22, 20, 28, 0,
	20, 20, 28, 0,
	20, 22, 28, 0,
	20, 24, 28, 0,
	20, 26, 28, 0,
	20, 28, 28, 0,
	20, 28, 26, 0,
	20, 28, 24, 0,
	20, 28, 22, 0,
	20, 28, 20, 0,
	22, 28, 20, 0,
	24, 28, 20, 0,
	26, 28, 20, 0,
	28, 28, 20, 0,
	28, 26, 20, 0,
	28, 24, 20, 0,
	28, 22, 20, 0,
	16, 0, 0, 0,
	16, 0, 4, 0,
	16, 0, 8, 0,
	16, 0, 12, 0,
	16, 0, 16, 0,
	12, 0, 16, 0,
	8, 0, 16, 0,
	4, 0, 16, 0,
	0, 0, 16, 0,
	0, 4, 16, 0,
	0, 8, 16, 0,
	0, 12, 16, 0,
	0, 16, 16, 0,
	0, 16, 12, 0,
	0, 16, 8, 0,
	0, 16, 4, 0,
	0, 16, 0, 0,
	4, 16, 0, 0,
	8, 16, 0, 0,
	12, 16, 0, 0,
	16, 16, 0, 0,
	16, 12, 0, 0,
	16, 8, 0, 0,
	16, 4, 0, 0,
	16, 8, 8, 0,
	16, 8, 10, 0,
	16, 8, 12, 0,
	16, 8, 14, 0,
	16, 8, 16, 0,
	14, 8, 16, 0,
	12, 8, 16, 0,
	10, 8, 16, 0,
	8, 8, 16, 0,
	8, 10, 16, 0,
	8, 12, 16, 0,
	8, 14, 16, 0,
	8, 16, 16, 0,
	8, 16, 14, 0,
	8, 16, 12, 0,
	8, 16, 10, 0,
	8, 16, 8, 0,
	10, 16, 8, 0,
	12, 16, 8, 0,
	14, 16, 8, 0,
	16, 16, 8, 0,
	16, 14, 8, 0,
	16, 12, 8, 0,
	16, 10, 8, 0,
	16, 11, 11, 0,
	16, 11, 12, 0,
	16, 11, 13, 0,
	16, 11, 15, 0,
	16, 11, 16, 0,
	15, 11, 16, 0,
	13, 11, 16, 0,
	12, 11, 16, 0,
	11, 11, 16, 0,
	11, 12, 16, 0,
	11, 13, 16, 0,
	11, 15, 16, 0,
	11, 16, 16, 0,
	11, 16, 15, 0,
	11, 16, 13, 0,
	11, 16, 12, 0,
	11, 16, 11, 0,
	12, 16, 11, 0,
	13, 16, 11, 0,
	15, 16, 11, 0,
	16, 16, 11, 0,
	16, 15, 11, 0,
	16, 13, 11, 0,
	16, 12, 11, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
	0, 0, 0, 0,
};

static struct termios saved_termios;
static int saved_termios_valid = 0;

static IOCTLRQ activate_rq;
static struct tty_detach_seq activate_seq;
static int activate_rq_used = 0;
static int switch_first_time = 1;

static int redraw_pipe[2] = { -1, -1 };

static void copy_videomem(int x)
{
	unsigned char *f;
	unsigned p = current_page;
	unsigned memory_size;
	vga_setpage(0);
	if (__unlikely(save_restore_memory > graph_mem_len)) memory_size = graph_mem_len;
	else memory_size = save_restore_memory;
	f = malloc(memory_size);
	if (__likely(f != NULL)) memcpy(f, graph_mem, memory_size);
	mmap(graph_mem, graph_mem_len, PROT_READ | PROT_WRITE, x ? MAP_FIXED | MAP_SHARED | _MAP_NOFAULT | _MAP_IMMEDIATE : MAP_FIXED | MAP_ANONYMOUS | MAP_PRIVATE | _MAP_IMMEDIATE, x ? 1 : -1, 0);
	if (__likely(f != NULL)) memcpy(graph_mem, f, memory_size), free(f);
	vt_active = x;
	vga_setpage(p);
}

static int update_palette(int index, int n)
{
	IOCTLRQ io;
	if (__unlikely(current_mode == TEXT)) return -1;
	io.h = 1;
	io.ioctl = IOCTL_TTY_SETPALETTE;
	io.param = index;
	io.v.ptr = (unsigned long)(palette + (index << 2));
	io.v.len = n << 2;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) return -1;
	return 0;
}

DECL_AST(redraw_rq_done, SPL_DEV, SIORQ)
{
	activate_rq_used--;
	free(RQ);
	RETURN;
}

DECL_AST(vc_switch, SPL_DEV, IOCTLRQ)
{
	keyboard_clearstate();
	if (__unlikely(activate_rq.ioctl == IOCTL_TTY_SETPALETTE)) {
		activate_rq.ioctl = IOCTL_TTY_WAITACTIVE;
		activate_rq.param = PARAM_WAIT_FOR_RELEASE_REQUEST;
		activate_rq.v.ptr = (unsigned long)&activate_seq;
		activate_rq.v.len = sizeof activate_seq;
		goto post;
	}
	if (__unlikely(activate_rq.ioctl == IOCTL_TTY_SETDISPLAYSTART)) {
		goto test_pal;
	}
	if (__unlikely(activate_rq.status < 0) && __unlikely(activate_rq.status != -EBUSY) && __unlikely(activate_rq.status != -ETTYCHG)) {
		if (activate_rq.status != -EINTR) {
			_background("CAN'T ATTACH GRAPHICS PROGRAM TO TEXT-ONLY TERMINAL");
			goto post;
		}
		activate_rq_used--;
		RETURN;
	}
	switch (activate_rq.param) {
		case PARAM_WAIT_FOR_ACTIVE_AND_LOCK: {
			ignore_1_mouse_event = 1;
			if (!switch_first_time || __unlikely(_merror(graph_mem, graph_mem_len))) copy_videomem(1);
			else switch_first_time = 0, vt_active = 1;
			if (__unlikely(current_display_start != 0)) {
				activate_rq.ioctl = IOCTL_TTY_SETDISPLAYSTART;
				activate_rq.param = current_display_start;
				activate_rq.v.ptr = 0;
				activate_rq.v.len = 0;
				break;
			}
			test_pal:
			if (redraw_pipe[1] != -1) {
				SIORQ *siorq = malloc(sizeof(SIORQ));
				if (__likely(siorq != NULL)) {
 					static __const__ char c = 0;
					activate_rq_used++;
					siorq->fn = redraw_rq_done;
					siorq->h = redraw_pipe[1];
					siorq->v.ptr = (unsigned long)&c;
					siorq->v.len = 1;
					siorq->v.vspace = &KERNEL$VIRTUAL;
					siorq->progress = 0;
					CALL_IORQ(siorq, KERNEL$WRITE);
				}
			}
			if (__unlikely(mode_param.bpp <= 8)) {
				activate_rq.ioctl = IOCTL_TTY_SETPALETTE;
				activate_rq.param = 0;
				activate_rq.v.ptr = (unsigned long)palette;
				activate_rq.v.len = 256 * 4;
				break;
			}
			if (gotoback || comefromback) raise(SIG_SVGALIB);
			activate_rq.ioctl = IOCTL_TTY_WAITACTIVE;
			activate_rq.param = PARAM_WAIT_FOR_RELEASE_REQUEST;
			activate_rq.v.ptr = (unsigned long)&activate_seq;
			activate_rq.v.len = sizeof activate_seq;
			break;
		}
		case PARAM_WAIT_FOR_RELEASE_REQUEST: {
			activate_rq.ioctl = IOCTL_TTY_WAITACTIVE;
			activate_rq.param = PARAM_RELEASE;
			activate_rq.v.ptr = (unsigned long)&activate_seq;
			activate_rq.v.len = sizeof activate_seq;
			copy_videomem(0);
			break;
		}
		case PARAM_RELEASE: {
			if (gotoback || comefromback) raise(SIG_SVGALIB);
			activate_rq.ioctl = IOCTL_TTY_WAITACTIVE;
			activate_rq.param = PARAM_WAIT_FOR_ACTIVE_AND_LOCK;
			activate_rq.v.ptr = (unsigned long)&activate_seq;
			activate_rq.v.len = sizeof activate_seq;
			break;
		}
	}
	post:
	RETURN_IORQ(&activate_rq, KERNEL$IOCTL);
}

int vga_setmode(int mode)
{
	if (__unlikely(KERNEL$KERNEL)) return -1;
	if (mode == TEXT) {
		int h1, h2;
		RAISE_SPL(SPL_DEV);
		while (activate_rq_used) {
			KERNEL$CIO((IORQ *)(void *)&activate_rq);
			KERNEL$SLEEP(1);
		}
		if (graph_mem_len > 0) munmap(graph_mem, graph_mem_len);
		graph_mem = NULL;
		graph_mem_len = 0;
		LOWER_SPL(SPL_ZERO);
		if (__unlikely(!saved_stdout) || __unlikely(!saved_stderr)) return 0;
		h1 = open(saved_stdout, O_WRONLY);
		if (__unlikely(dup2(h1, 1) == -1)) return -1;
		close(h1);
		h2 = open(saved_stderr, O_WRONLY);
		dup2(h2, 2);
		close(h2);
		write(1, NULL, 0);
		if (saved_termios_valid) {
			ioctl(0, TIOCSETA, &saved_termios);
			saved_termios_valid = 0;
		}
		current_mode = TEXT;
		mouse_setmode(0, 0, 0);
		if (__likely(redraw_pipe[0] != -1)) {
			close(redraw_pipe[0]);
			close(redraw_pipe[1]);
			redraw_pipe[0] = -1;
			redraw_pipe[1] = -1;
		}
		return 0;
	} else {
		char v[__MAX_STR_LEN];
		CHHRQ ch;
		IOCTLRQ io;
		struct tty_videomode my_mode_param;
		unsigned char *my_graph_mem;
		if (__unlikely((unsigned)mode > __GLASTMODE)) return -1;
		get_timing(vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp);
		if (!saved_stdout) {
			char *p = KERNEL$HANDLE_PATH(1);
			if (__unlikely(!p)) p = "STDOUT:/";
			if (!(saved_stdout = malloc(strlen(p) + 1))) return -1;
			strcpy(saved_stdout, p);
		}
		if (!saved_stderr) {
			char *p = KERNEL$HANDLE_PATH(2);
			if (__unlikely(!p)) p = "STDERR:/";
			if (!(saved_stderr = malloc(strlen(p) + 1))) return -1;
			strcpy(saved_stderr, p);
		}
		_snprintf(v, sizeof v, "G%dX%dX%d", vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp);
		ch.h = 1;
		ch.option = "VIDEOMODE";
		ch.value = v;
		SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
		if (__unlikely(ch.status < 0)) return -1;
		ch.h = 2;
		ch.option = "VIDEOMODE";
		ch.value = v;
		SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);

		if (pixel_clock) {
			_snprintf(v, sizeof v, "%u,%u,%u,%u,%u,%u,%u,%u,%u,%u", pixel_clock, hh, h_sync_start, h_sync_end, h_total, vv, v_sync_start, v_sync_end, v_total, timing_flags);
			ch.h = 1;
			ch.option = "VIDEOTIMING";
			ch.value = v;
			SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
			ch.h = 2;
			ch.option = "VIDEOTIMING";
			ch.value = v;
			SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
		}

		repeat_getmode:
		io.h = 1;
		io.ioctl = IOCTL_TTY_GETVIDEOMODE;
		io.param = 0;
		io.v.ptr = (unsigned long)&my_mode_param;
		io.v.len = sizeof my_mode_param;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__unlikely(io.status < 0)) {
			KERNEL$SLEEP(1);
			goto repeat_getmode;
		}
		repeat_mmap:
		my_graph_mem = mmap(NULL, my_mode_param.memsize, PROT_READ | PROT_WRITE, MAP_SHARED | _MAP_NOFAULT, 1, 0);
		if (__unlikely(my_graph_mem == MAP_FAILED)) {
			if (errno == ENOMEM) {
				vga_setmode(TEXT);
				return -1;
			}
			KERNEL$SLEEP(1);
			goto repeat_mmap;
		}
		RAISE_SPL(SPL_DEV);
		memcpy(&mode_param, &my_mode_param, sizeof mode_param);
		max_y = mode_param.y * mode_param.images;
		if (graph_mem_len > 0) munmap(graph_mem, graph_mem_len);
		graph_mem = my_graph_mem;
		graph_mem_len = mode_param.memsize;
		init_accel_state();
		LOWER_SPL(SPL_ZERO);
		write(1, NULL, 0);
		if (__likely(current_mode == TEXT) && __likely(!saved_termios_valid)) {
			if (__likely(!ioctl(0, TIOCGETA, &saved_termios))) {
				struct termios termios;
				memcpy(&termios, &saved_termios, sizeof(struct termios));
				termios.c_lflag &= ~(ECHO|ECHOE|ECHOK|ECHONL|ICANON|IEXTEN|NOFLSH|TOSTOP|PENDIN);
				termios.c_cc[VMIN] = 1;
				termios.c_cc[VTIME] = 0;
				ioctl(0, TIOCSETA, &termios);
				saved_termios_valid = 1;
			}
		}
		current_mode = mode;
		current_page = 0;
		vga_claimvideomemory(mode_param.scanline * mode_param.y);
		memset(graph_mem, 0, save_restore_memory);
		if (mode_param.bpp <= 8) {
			memcpy(palette, default_palette, 256 * 4);
		}
		mouse_setmode(1, mode_param.x, mode_param.y);

		if (redraw_pipe[0] == -1) {
			if (__likely(!pipe(redraw_pipe))) {
				if (__unlikely(fcntl(redraw_pipe[0], F_SETFL, O_NONBLOCK)) || __unlikely(fcntl(redraw_pipe[1], F_SETFL, O_NONBLOCK))) {
					close(redraw_pipe[0]);
					close(redraw_pipe[1]);
					redraw_pipe[0] = -1;
					redraw_pipe[1] = -1;
				}
			}
		}
		RAISE_SPL(SPL_DEV);
		while (__unlikely(activate_rq_used)) {
			KERNEL$CIO((IORQ *)(void *)&activate_rq);
			KERNEL$SLEEP(1);
		}
		switch_first_time = 1;
		activate_rq.fn = vc_switch;
		activate_rq.h = 1;
		activate_rq.ioctl = IOCTL_TTY_WAITACTIVE;
		activate_rq.param = PARAM_WAIT_FOR_ACTIVE_AND_LOCK;
		activate_rq.v.ptr = (unsigned long)&activate_seq;
		activate_rq.v.len = sizeof activate_seq;
		activate_rq.v.vspace = &KERNEL$VIRTUAL;
		CALL_IORQ(&activate_rq, KERNEL$IOCTL);
		activate_rq_used = 1;
		vt_active = 0;
		LOWER_SPL(SPL_ZERO);
		RAISE_SPL(SPL_DEV);
		switch_first_time = 0;
		if (__unlikely(!vt_active)) {
			mmap(graph_mem, graph_mem_len, PROT_READ | PROT_WRITE, MAP_FIXED | MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);
		}
		LOWER_SPL(SPL_ZERO);
		return 0;
	}
}

int vga_setflipchar(int c)
{
	return 0;
}

static char *flipped_saved = NULL;
static unsigned flipped_saved_size = 0;
static int flipped_mode = TEXT;
static unsigned flipped_display_start = 0;
static __u8 flipped_palette[256 * 4];

int vga_flip(void)
{
	if (current_mode == TEXT) {
		if (__unlikely(flipped_mode == TEXT)) return 0;
		if (vga_setmode(flipped_mode)) return -1;
		memcpy(graph_mem, flipped_saved, flipped_saved_size > graph_mem_len ? graph_mem_len : flipped_saved_size);
		free(flipped_saved); flipped_saved = NULL;
		flipped_saved_size = 0;
		vga_setdisplaystart(flipped_display_start);
		if (mode_param.bpp <= 8) {
			memcpy(palette, flipped_palette, sizeof flipped_palette);
			update_palette(0, 256);
		}
		return 0;
	} else {
		if (__unlikely(current_page)) vga_setpage(0);
		free(flipped_saved), flipped_saved_size = 0;
		flipped_saved = malloc(graph_mem_len);
		if (__unlikely(!flipped_saved)) return -1;
		memcpy(flipped_saved, graph_mem, graph_mem_len);
		flipped_saved_size = graph_mem_len;
		flipped_mode = current_mode;
		flipped_display_start = current_display_start;
		if (mode_param.bpp <= 8) {
			memcpy(flipped_palette, palette, sizeof flipped_palette);
		}
		return vga_setmode(TEXT);
	}
}

int vga_getxdim(void)
{
	if (__unlikely(current_mode == TEXT)) return 0;
	return mode_param.x;
}

int vga_getydim(void)
{
	if (__unlikely(current_mode == TEXT)) return 0;
	return mode_param.y;
}

int vga_getcolors(void)
{
	if (__unlikely(current_mode == TEXT)) return 0;
	return 1 << (mode_param.bpp <= 24 ? mode_param.bpp : 24);
}

int vga_setpalette(int index, int red, int green, int blue)
{
	if (__unlikely((unsigned)index >= 256)) return -1;
	palette[index * 4] = blue;
	palette[index * 4 + 1] = green;
	palette[index * 4 + 2] = red;
	return update_palette(index, 1);
}

int vga_getpalette(int index, int *red, int *green, int *blue)
{
	if (__unlikely((unsigned)index >= 256)) return -1;
	if (__likely(blue != NULL)) *blue = palette[index * 4 + 0];
	if (__likely(green != NULL)) *green = palette[index * 4 + 1];
	if (__likely(red != NULL)) *red = palette[index * 4 + 2];
	return 0;
}

int vga_setpalvec(int start, int num, int *pal)
{
	int i;
	for (i = start; i < start + num; i++) {
		if (__unlikely((unsigned)i >= 256)) return -1;
		palette[i * 4] = pal[2];
		palette[i * 4 + 1] = pal[1];
		palette[i * 4 + 2] = pal[0];
		pal += 3;
	}
	return update_palette(start, num);
}

int vga_getpalvec(int start, int num, int *pal)
{
	int i;
	for (i = start; i < start + num; i++) {
		if (__unlikely((unsigned)i >= 256)) return -1;
		pal[2] = palette[i * 4];
		pal[1] = palette[i * 4 + 1];
		pal[0] = palette[i * 4 + 2];
		pal += 3;
	}
	return 0;
}

int vga_screenoff(void)
{
	return 0;
}

int vga_screenon(void)
{
	return 0;
}

static int vga_do_getch(int nonblock)
{
	int h;
	char c;
	char *q;
	char *p = KERNEL$HANDLE_PATH(0);
	if (__unlikely(!p)) return EOF;
	q = alloca(strlen(p) + 29);
	strcpy(q, p);
	strcat(q, "/^ECHO=0/^ICANON=0");
	if (nonblock) strcat(q, "/^NONBLOCK");
	rp:
	h = open(q, O_RDONLY);
	if (__unlikely(h == -1)) {
		if (nonblock) {
			strcpy(q, p);
			strcat(q, "/^NONBLOCK");
			nonblock = 0;
			goto rp;
		}
		if (__unlikely(read(0, &c, 1) != 1)) return EOF;
	} else {
		if (__unlikely(read(h, &c, 1) != 1)) {
			close(h);
			return EOF;
		}
		close(h);
	}
	return c;
}

int vga_getch(void)
{
	return vga_do_getch(0);
}

int vga_dumpregs(void)
{
	return 0;
}


vga_cardinfo *vga_getcardinfo(void)
{
	vga_cardinfo *cardinfo;
	char *p, *q;
	int zs;
	int mode, h;
	if (__unlikely(KERNEL$KERNEL)) return NULL;
	if (__unlikely(!(cardinfo = malloc(sizeof(vga_cardinfo))))) return NULL;
	cardinfo->version = 0x100;
	cardinfo->size = sizeof(vga_cardinfo);
	cardinfo->chipset = 0;
	memcpy(&cardinfo->chipset, "VESA", sizeof(int) < 4 ? sizeof(int) : 4);
	cardinfo->physmem = 0xa0000;
	p = KERNEL$HANDLE_PATH(1);
	if (__unlikely(!p)) return cardinfo;
	zs = strlen(p) + 16 + 3 * 10;
	q = alloca(zs);
	for (mode = 1; mode <= __GLASTMODE; mode++) {
		_snprintf(q, zs, "%s/^VIDEOMODE=G%dX%dX%d", p, vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp);
		if ((h = open(q, O_RDWR)) != -1) {
			IOCTLRQ io;
			struct tty_videomode mp;
			io.h = h;
			io.ioctl = IOCTL_TTY_GETVIDEOMODE;
			io.param = 0;
			io.v.ptr = (unsigned long)&mp;
			io.v.len = sizeof mode_param;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			close(h);
			if (__unlikely(io.status < 0)) continue;
			if (__likely(mp.pages == 1) && __likely(mp.phys >= 0x100000)) {
				cardinfo->physmem = mp.phys;
				break;
			}
		}
	}
	return cardinfo;
}

static vga_modeinfo vmi;

vga_modeinfo *vga_getmodeinfo(int mode)
{
	int zs;
	char *q, *p;
	int h;
	IOCTLRQ io;
	struct tty_videomode mp;
	if (__unlikely(KERNEL$KERNEL)) return NULL;
	if (__unlikely((unsigned)mode > __GLASTMODE)) return NULL;
	if (__unlikely(mode == TEXT)) {
		memset(&vmi, 0, sizeof vmi);
		vmi.width = 80;
		vmi.height = 25;
		vmi.bytesperpixel = 2;
		vmi.colors = 16;
		vmi.linewidth = 160;
		vmi.maxlogicalwidth = vmi.linewidth;
		vmi.flags = HAVE_EXT_SET;
		return &vmi;
	}
	p = KERNEL$HANDLE_PATH(1);
	if (__unlikely(!p)) return NULL;
	zs = strlen(p) + 16 + 3 * 10;
	q = alloca(zs);
	_snprintf(q, zs, "%s/^VIDEOMODE=G%dX%dX%d", p, vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp);
	if (__unlikely((h = open(q, O_RDWR)) == -1)) {
		memset(&vmi, 0, sizeof vmi);
		vmi.width = vga_modes[mode].x;
		vmi.height = vga_modes[mode].y;
		vmi.bytesperpixel = (vga_modes[mode].bpp + 7) >> 3;
		vmi.colors = 1 << (vga_modes[mode].bpp < 24 ? vga_modes[mode].bpp : 24);
		vmi.linewidth = vmi.width * vmi.bytesperpixel;
		vmi.maxlogicalwidth = vmi.linewidth;
		vmi.maxpixels = vmi.width * vmi.height;
		vmi.haveblit = 0;
		vmi.flags = 0;
		vmi.linewidth_unit = 8;
		return &vmi;
	}
	io.h = h;
	io.ioctl = IOCTL_TTY_GETVIDEOMODE;
	io.param = 0;
	io.v.ptr = (unsigned long)&mp;
	io.v.len = sizeof mode_param;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	close(h);
	if (__unlikely(io.status < 0)) return NULL;
	memset(&vmi, 0, sizeof vmi);
	vmi.width = mp.x;
	vmi.height = mp.y;
	vmi.bytesperpixel = (mp.bpp + 7) >> 3;
	vmi.colors = 1 << (mp.bpp < 24 ? mp.bpp : 24);
	vmi.linewidth = mp.scanline;
	vmi.maxlogicalwidth = vmi.linewidth;
	if (mp.images <= 1) vmi.startaddressrange = 0;
	else vmi.startaddressrange = 0xffffffffu;
	vmi.maxpixels = (mp.scanline / vmi.bytesperpixel) * mp.y * mp.images;
	vmi.haveblit = 0;
	vmi.flags = HAVE_EXT_SET;
	if (__likely(mp.pages == 1)) vmi.flags |= CAPABLE_LINEAR | IS_LINEAR;
	vmi.memory = (vmi.bytesperpixel * mp.x * mp.y * mp.images + 1023) >> 10;
	memcpy(&vmi.chiptype, "VESA", sizeof(int) < 4 ? sizeof(int) : 4);
	vmi.linewidth_unit = 8;
	return &vmi;
}

int vga_getdefaultmode(void)
{
	char *s = getenv("SVGALIB_DEFAULT_MODE");
	if (__unlikely(!s)) return -1;
	return vga_getmodenumber(s);
}

int vga_getcurrentmode(void)
{
	return current_mode;
}

int vga_getcurrentchipset(void)
{
	int c = 0;
	memcpy(&c, "VESA", sizeof(int) < 4 ? sizeof(int) : 4);
	return c;
}

char *vga_getmodename(int mode)
{
	static char modename[64];
	if (__unlikely(mode == TEXT)) return "TEXT";
	if (__unlikely((unsigned)mode > __GLASTMODE)) return NULL;
	_snprintf(modename, sizeof modename, "G%dx%dx%s", vga_modes[mode].x, vga_modes[mode].y, vga_modes[mode].bpp == 1 ? "2" : vga_modes[mode].bpp == 4 ? "16" : vga_modes[mode].bpp == 8 ? "256" : vga_modes[mode].bpp == 15 ? "32K" : vga_modes[mode].bpp == 16 ? "64K" : vga_modes[mode].bpp == 24 ? "16M" : vga_modes[mode].bpp == 32 ? "16M32" : "?");
	return modename;
}

int vga_getmodenumber(char *name)
{
	int i;
	long l;
	for (i = 0; i <= __GLASTMODE; i++) if (__unlikely(!_strcasecmp(vga_getmodename(i), name))) return i;
	if (!__get_number(name, name + strlen(name), 0, &l) && (unsigned long)l <= __GLASTMODE) return l;
	return -1;
}

int vga_lastmodenumber(void)
{
	return __GLASTMODE;
}

int vga_getoptmode(int x, int y, int colors, int bytesperpixel, int c)
{
	int ideal_mode = -1;
	int min_area = MAXINT, min_bpp = MAXINT;
	int i;
	bytesperpixel <<= 3;
	for (i = 1; i <= __GLASTMODE; i++) {
		if (vga_modes[i].x >= x && vga_modes[i].y >= y && vga_modes[i].bpp >= bytesperpixel) {
			int area = vga_modes[i].x * vga_modes[i].y;
			if ((area < min_area || (area == min_area && vga_modes[i].bpp < min_bpp)) && vga_hasmode(i)) {
				min_area = area;
				min_bpp = vga_modes[i].bpp;
				ideal_mode = i;
			}
		}
	}
	return ideal_mode;
}

unsigned char *vga_getgraphmem(void)
{
	return graph_mem;
}

void vga_setpage(int p)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl))) KERNEL$SUICIDE("vga_setpage at spl %08X", spl);
	RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)p >= (graph_mem_len >> 16))) goto ret;
	if (__unlikely(p == current_page)) goto ret;
	if (__unlikely(!vt_active)) {
		if (__likely(current_page)) {
			memcpy(graph_mem + (current_page << 16), graph_mem, 65536);
		} else {
			if (__unlikely(!page_0_backbuffer)) {
				page_0_backbuffer = malloc(65536);
				if (__unlikely(!page_0_backbuffer)) goto ret;
			}
			memcpy(page_0_backbuffer, graph_mem, 65536);
		}
		if (__likely(p)) {
			memcpy(graph_mem, graph_mem + (p << 16), 65536);
		} else {
			if (__unlikely(!page_0_backbuffer)) goto ret;
			memcpy(graph_mem, page_0_backbuffer, 65536);
		}
	} else {
		mmap(graph_mem, 65536, PROT_READ | PROT_WRITE, MAP_FIXED | MAP_SHARED | _MAP_NOFAULT | _MAP_IMMEDIATE, 1, p << 16);
	}
	current_page = p;
	ret:
	LOWER_SPLX(spl);
}

void vga_setreadpage(int p)
{
}

void vga_setwritepage(int p)
{
}

void vga_setlogicalwidth(int w)
{
}

void vga_setdisplaystart(int a)
{
	IOCTLRQ io;
	current_display_start = a;
	io.h = 1;
	io.ioctl = IOCTL_TTY_SETDISPLAYSTART;
	io.param = a;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
}

void vga_waitretrace_setdisplaystart(int a)
{
	IOCTLRQ io;
	current_display_start = a;
	io.h = 1;
	io.ioctl = IOCTL_TTY_WAITRETRACE_SETDISPLAYSTART;
	io.param = a;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
}

void vga_waitretrace(void)
{
	IOCTLRQ io;
	io.h = 1;
	io.ioctl = IOCTL_TTY_WAITRETRACE;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
}

int vga_claimvideomemory(int n)
{
	if (__unlikely((unsigned)n > graph_mem_len)) return -1;
	save_restore_memory = n;
	return 0;
}

void vga_disabledriverreport(void)
{
}

int vga_setmodeX(void)
{
	return -1;
}

int vga_init(void)
{
	mouse_init(NULL, 0, 0);
	return 0;
}

int vga_initf(int flags)
{
	return vga_init();
}

int vga_getmonitortype(void)
{
	return MON1024_72;
}

void vga_lockvc(void)
{
}

void vga_unlockvc(void)
{
}

int vga_getkey(void)
{
	int k = vga_do_getch(1);
	if (k == EOF) k = 0;
	return k;
}

int vga_oktowrite(void)
{
	return 1;
}

void vga_copytoplanar256(unsigned char *virtualp, int pitch, int voffset, int vpitch, int w, int h)
{
}

void vga_copytoplanar16(unsigned char *virtualp, int pitch, int voffset, int vpitch, int w, int h)
{
}

void vga_copytoplane(unsigned char *virtualp, int pitch, int voffset, int vpitch, int w, int h, int plane)
{
}

int vga_setlinearaddressing(void)
{
	vga_setpage(0);
	return graph_mem_len;
}

void vga_safety_fork(void (*shutdown_routine)(void))
{
}

int vga_simple_init(void)
{
	return vga_init();
}

void vga_chipset_saveregs(unsigned char *r)
{
}

void vga_chipset_setregs(unsigned char *r)
{
}

void vga_setchipset(int c)
{
}

void vga_setchipsetandfeatures(int c, int par1, int par2)
{
}

void vga_gettextfont(void *font)
{
}

void vga_puttextfont(void *font)
{
}

void vga_settextmoderegs(void *regs)
{
}

void vga_gettextmoderegs(void *regs)
{
}

int vga_white(void)
{
	switch (mode_param.bpp) {
		case 1:	return 1;
		case 4: return 15;
		case 8: return 255;
		case 15: return 32767;
		case 16: return 65535;
		case 24:
		case 32: return (1 << 24) - 1;
		default: return 0;
	}
}

static const unsigned char ega_red[16] =
{0, 0, 0, 0, 168, 168, 168, 168, 84, 84, 84, 84, 255, 255, 255, 255};
static const unsigned char ega_green[16] =
{0, 0, 168, 168, 0, 0, 84, 168, 84, 84, 255, 255, 84, 84, 255, 255};
static const unsigned char ega_blue[16] =
{0, 168, 0, 168, 0, 168, 0, 168, 84, 255, 84, 255, 84, 255, 84, 255};

int vga_setegacolor(int c)
{
	if (__unlikely(c < 0)) c = 0;
	else if (__unlikely(c >= 16)) c = 15;
	switch (mode_param.bpp) {
		case 1:	return vga_setcolor(!!c);
		case 4:
		case 8: return vga_setcolor(c);
		default: return vga_setrgbcolor(ega_red[c], ega_green[c], ega_blue[c]);
	}
}

int vga_setrgbcolor(int r, int g, int b)
{
	switch (mode_param.bpp) {
		case 15: return vga_setcolor(((r >> 3) << 10) + ((g >> 3) << 5) + (b >> 3));
		case 16: return vga_setcolor(((r >> 3) << 11) + ((g >> 2) << 5) + (b >> 3));
		case 24:
		case 32: return vga_setcolor((r << 16) + (g << 8) + b);
		default: return -1;
	}
}

void vga_bitblt(int srcaddr, int destaddr, int w, int h, int pitch)
{
}

void vga_imageblt(void *srcaddr, int destaddr, int w, int h, int pitch)
{
}

void vga_fillblt(int destaddr, int w, int h, int pitch, int c)
{
}

void vga_hlinelistblt(int ymin, int n, int *xmin, int *xmax, int pitch, int c)
{
}

void vga_blitwait(void)
{
}

int vga_ext_set(unsigned what, ...)
{
	int a;
	va_list args;
	switch (what) {
		case VGA_EXT_AVAILABLE:
			va_start(args, what);
			a = va_arg(args, int);
			va_end(args);
			switch (a) {
				case VGA_AVAIL_SET:
					return 0;
				case VGA_AVAIL_ACCEL: {
					IOCTLRQ io;
					io.h = 1;
					io.ioctl = IOCTL_TTY_AVAIL_ACCEL;
					io.param = 0;
					io.v.ptr = 0;
					io.v.len = 0;
					io.v.vspace = &KERNEL$VIRTUAL;
					SYNC_IO(&io, KERNEL$IOCTL);
					if (__unlikely(io.status < 0)) return 0;
					return io.status;
				}
				case VGA_AVAIL_ROP:
					return 0;
				case VGA_AVAIL_TRANSPARENCY:
					return 0;
				case VGA_AVAIL_TRANSMODES:
					return 0;
				case VGA_AVAIL_FLAGS:
					return 0;
				default:
					return 0;
			}
		case VGA_EXT_SET:
			return 0;
		case VGA_EXT_CLEAR:
			return 0;
		case VGA_EXT_RESET:
			return 0;
		case VGA_EXT_PAGE_OFFSET:
			return 0;
		case VGA_EXT_FONT_SIZE:
			return 8192;
		default:
			return 0;
	}
}

unsigned accel_in_progress = 0;
static unsigned accel_mode;
static unsigned accel_fg_color;

static void init_accel_state(void)
{
	vga_accel(ACCEL_SYNC);
	accel_mode = 0;
	accel_fg_color = 0;
}

int vga_accel(unsigned op, ...)
{
	int p;
	union accel_param ap;
	IOCTLRQ io;
	va_list args;
	if (op == ACCEL_SETFGCOLOR) {
		va_start(args, op);
		accel_fg_color = va_arg(args, int);
		va_end(args);
		return 0;
	}
	io.h = 1;
	io.ioctl = IOCTL_TTY_DO_ACCEL;
	io.param = op | accel_mode;
	io.v.ptr = (unsigned long)&ap;
	io.v.len = sizeof ap;
	io.v.vspace = &KERNEL$VIRTUAL;
	if (__likely(op == ACCEL_FILLBOX)) {
		va_start(args, op);
		ap.fillbox.x = va_arg(args, int);
		ap.fillbox.y = va_arg(args, int);
		if (__unlikely(!(ap.fillbox.w = va_arg(args, int)))) {
			va_end_ret:
			va_end(args);
			return 0;
		}
		if (__unlikely(!(ap.fillbox.h = va_arg(args, int)))) goto va_end_ret;
		ap.fillbox.c = accel_fg_color;
		va_end(args);
	} else if (__likely(op == ACCEL_SCREENCOPY)) {
		va_start(args, op);
		ap.screencopy.x1 = va_arg(args, int);
		ap.screencopy.y1 = va_arg(args, int);
		ap.screencopy.x2 = va_arg(args, int);
		if (__unlikely((ap.screencopy.y2 = va_arg(args, int)) == ap.screencopy.y1) && __unlikely(ap.screencopy.x1 == ap.screencopy.x2)) goto va_end_ret;
		if (__unlikely(!(ap.screencopy.w = va_arg(args, int)))) goto va_end_ret;
		if (__unlikely(!(ap.screencopy.h = va_arg(args, int)))) goto va_end_ret;
		va_end(args);
	} else if (__likely(op == ACCEL_SYNC)) {
		do_sync:
		if (__likely(!accel_in_progress)) return 0;
		accel_in_progress = 0;
		io.v.ptr = 0;
		io.v.len = 0;
	} else if (op == ACCEL_SETMODE) {
		va_start(args, op);
		p = va_arg(args, int);
		va_end(args);
		if (p & BLITS_IN_BACKGROUND) {
			accel_mode |= _ACCEL_ASYNC;
		} else {
			accel_mode &= ~_ACCEL_ASYNC;
			io.param = ACCEL_SYNC;
			goto do_sync;
		}
		return 0;
	} else return -1;
	if (can_omit_draw) return 0;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__likely((io.param & _ACCEL_OP) != ACCEL_SYNC)) {
		if (__unlikely(io.status < 0)) {
			return accel_do_manually(io.param, &ap);
		} else {
			accel_in_progress |= accel_mode;
		}
	}
	return 0;
}

int vga_initcursor(int a)
{
	return -1;
}

void vga_showcursor(int a)
{
}

void vga_setcursorposition(int x, int y)
{
}

void vga_selectcursor(int a)
{
}

void vga_setcursorimage(int a, int b, int c, int d, unsigned char *p)
{
}

int vga_setcrtcregs(unsigned char *r)
{
	return -1;
}

int vga_getcrtcregs(unsigned char *r)
{
	return -1;
}

int vga_addtiming(int pixelClock, int HDisplay, int HSyncStart, int HSyncEnd, int HTotal, int VDisplay, int VSyncStart, int VSyncEnd, int VTotal, int flags)
{
	return -1;
}

int vga_changetiming(int pixelClock, int HDisplay, int HSyncStart, int HSyncEnd, int HTotal, int VDisplay, int VSyncStart, int VSyncEnd, int VTotal, int flags)
{
	return -1;
}

int vga_getcurrenttiming(int *pixelClock, int *HDisplay, int *HSyncStart, int *HSyncEnd, int *HTotal, int *VDisplay, int *VSyncStart, int *VSyncEnd, int *VTotal, int *flags)
{
	return -1;
}

int vga_addmode(int xdim, int ydim, int cols, int xbytes, int bytespp)
{
	return -1;
}

int vga_guesstiming(int x, int y, int clue, int arg)
{
	return -1;
}


void vga_dpms(int mode)
{
}

int vga_waitevent(int which, fd_set *in, fd_set *out, fd_set *except, struct timeval *timeout)
{
	int retval;
	fd_set backup_in;
	if (__likely(which) && __unlikely(!in)) {
		FD_ZERO(&backup_in);
		in = &backup_in;
	}
	if (which & VGA_KEYEVENT) FD_SET(0, in);
	if (__unlikely(mouse_handle == -1)) which &= ~VGA_MOUSEEVENT;
	if (which & VGA_MOUSEEVENT) FD_SET(mouse_handle, in);
	if (which & VGA_REDRAWEVENT) {
		if (redraw_pipe[0] == -1) {
			which &= ~VGA_REDRAWEVENT;
			goto skip_redrawevent;
		}
		if (__unlikely(save_restore_memory)) vga_claimvideomemory(0);
		FD_SET(redraw_pipe[0], in);
	}
	skip_redrawevent:
	if (__unlikely(select(FD_SETSIZE, in, out, except, timeout) < 0)) {
		if (which & VGA_KEYEVENT) FD_CLR(0, in);
		if (which & VGA_MOUSEEVENT) FD_CLR(mouse_handle, in);
		if (which & VGA_REDRAWEVENT) FD_CLR(redraw_pipe[0], in);
		return -1;
	}
	retval = 0;
	if (which & VGA_KEYEVENT) {
		if (FD_ISSET(0, in)) {
			FD_CLR(0, in);
			retval |= VGA_KEYEVENT;
			keyboard_update();
		}
	}
	if (which & VGA_MOUSEEVENT) {
		if (FD_ISSET(mouse_handle, in)) {
			FD_CLR(mouse_handle, in);
			retval |= VGA_MOUSEEVENT;
			mouse_update();
		}
	}
	if (which & VGA_REDRAWEVENT) {
		if (FD_ISSET(redraw_pipe[0], in)) {
			static char c[2];
			FD_CLR(redraw_pipe[0], in);
			retval |= VGA_REDRAWEVENT;
			while (__unlikely(read(redraw_pipe[0], &c, sizeof c) == sizeof c));
		}
	}
	return retval;
}


static void vt_switch_handler(int sig)
{
	static int last_vt_active = 1;
	if (__likely(vt_active != last_vt_active)) {
		if ((last_vt_active = vt_active)) {
			if (__likely(comefromback != NULL)) comefromback();
		} else {
			if (__likely(gotoback != NULL)) gotoback();
		}
	}
}

void vga_runinbackground(int stat, ...)
{
	va_list params;
	va_start(params, stat);
	if (stat == VGA_GOTOBACK || stat == VGA_COMEFROMBACK) {
		struct sigaction sa;
		sa.sa_handler = vt_switch_handler;
		sigemptyset(&sa.sa_mask);
		sa.sa_flags = SA_RESTART;
		if (sigaction(SIG_SVGALIB, &sa, NULL)) goto ret;
		if (stat == VGA_GOTOBACK) gotoback = va_arg(params, switch_fn_t);
		else comefromback = va_arg(params, switch_fn_t);
		raise(SIG_SVGALIB);
	}
	ret:
	va_end(params);
}

int vga_runinbackground_version(void)
{
	return 3;
}

