/* gtf.c  Generate mode timings using the GTF Timing Standard
 *
 * gcc gtf.c -o gtf -lm -Wall
 *
 * Copyright (c) 2001, Andy Ritger  aritger@nvidia.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * o Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer
 *   in the documentation and/or other materials provided with the
 *   distribution.
 * o Neither the name of NVIDIA nor the names of its contributors
 *   may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <STRING.H>
#include <STDLIB.H>
#include <UNISTD.H>
#include <FCNTL.H>
#include <SYS/STAT.H>
#include <SYS/TYPES.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <VALUES.H>

#include "VGA.H"

unsigned pixel_clock;
unsigned hh;
unsigned h_sync_start;
unsigned h_sync_end;
unsigned h_total;
unsigned vv;
unsigned v_sync_start;
unsigned v_sync_end;
unsigned v_total;
unsigned timing_flags;

#define VIDEOTIMING_FILE	"ETC.:/VIDEOTIMING"

/*
1024x768 75.00
1024x768 81.80  1024 1080 1192 1360  768 769 772 802  -HSync +Vsync Doublescan Interlace
*/

static char *getstr(char **ptr)
{
	char *e;
	char *o, *p = *ptr;
	while (*p == ' ' || *p == '\t' || *p == '\r') p++;
	if (__unlikely(*p == '\n')) {
		*ptr = p;
		return NULL;
	}
	o = p;
	while (*p != ' ' && *p != '\t' && *p != '\r' && *p != '\n') p++;
	if (__unlikely(!(e = malloc(p - o + 1)))) return NULL;
	memcpy(e, o, p - o);
	e[p - o] = 0;
	while (*p == ' ' || *p == '\t' || *p == '\r') p++;
	*ptr = p;
	return e;
}

static int getnum(char **ptr, int dot)
{
	char *d;
	char *e = getstr(ptr);
	long d1, d2;
	int i;
	if (__unlikely(!e)) return -1;
	if (__unlikely(dot) && (d = strchr(e, '.'))) {
		if (__unlikely(__get_number(e, d, 0, &d1))) {
			free_ret:
			__slow_free(e);
			return -1;
		}
		if (__unlikely(__get_number(d + 1, d + strlen(d), 0, &d2)))
			goto free_ret;
		if (__unlikely((unsigned long)d2 >= 1000000)) goto free_ret;
		if (__unlikely((i = strlen(d + 1)) > 6)) goto free_ret;
		while (i < 6) i++, d2 *= 10;
		free(e);
		ret_d:
		if (__unlikely((unsigned long)d1 >= MAXINT / 1000000)) goto free_ret;
		return d1 * 1000000 + d2;
	}
	if (__unlikely(__get_number(e, e + strlen(e), 0, &d1))) goto free_ret;
	free(e);
	if (__unlikely(dot)) {
		d2 = 0;
		goto ret_d;
	}
	if (__unlikely((unsigned long)d1 > MAXINT)) return -1;
	return d1;
}

static void do_gtf(int x, int y, int bpp, int freq, int f_timing_flags);

void get_timing(int x, int y, int bpp)
{
	int h;
	struct stat st;
	char *m, *p, *e;
	unsigned sz;
	unsigned lc;
	char *env;
	pixel_clock = 0;
	if (__unlikely((env = getenv("VIDEO$REFRESH")) != NULL)) {
		long freq;
		if (__unlikely(__get_number(env, env + strlen(env), 0, &freq) || (unsigned long)freq > MAXINT / 1000000)) {
			__critical_printf("INVALID VIDEO$REFRESH ENVIRONMENT VARIABLE");
			goto skip_env;
		}
		do_gtf(x, y, bpp, freq * 1000000, VIDEO_PARAMS_HSYNC_NEG);
		return;
	}
	skip_env:
	if (__unlikely((h = open(VIDEOTIMING_FILE, O_RDONLY)) == -1)) return;
	if (__unlikely(fstat(h, &st))) return;
	if (__unlikely(st.st_size >= MAXUINT)) {
		close(h);
		return;
	}
	sz = (unsigned)st.st_size;
	m = malloc(sz + 1);
	if (__unlikely(!m)) {
		close(h);
		return;
	}
	if (__unlikely(read(h, m, sz) != sz)) {
		__slow_free(m);
		close(h);
		return;
	}
	close(h);
	e = m + sz;
	*e = '\n';
	lc = 1;
	for (p = m; p < e; p = strchr(p, '\n') + 1, lc++) {
		char *s, *xx;
		long xres, yres;
		int freq;
		int gtf, h_set, v_set;
		int f_h, f_h_sync_start, f_h_sync_end, f_h_total;
		int f_v, f_v_sync_start, f_v_sync_end, f_v_total;
		int f_timing_flags;
		if (__unlikely(*p == '#')) continue;
		if (__unlikely(!(s = getstr(&p)))) continue;
		if (__unlikely(!(xx = strchr(s, 'x')))) xx = strchr(s, 'X');
		if (__unlikely(!xx)) {
			invl_free:
			__slow_free(s);
			invl:
			__critical_printf("INVALID VIDEOMODE DESCRIPTION IN " VIDEOTIMING_FILE " AT LINE %u\n", lc);
			continue;
		}
		if (__unlikely(__get_number(s, xx, 0, &xres))) goto invl_free;
		if (__unlikely(__get_number(xx + 1, xx + strlen(xx), 0, &yres))) goto invl_free;
		free(s);
		if (__unlikely((freq = getnum(&p, 1)) == -1)) goto invl;
		f_h = f_h_sync_start = f_h_sync_end = f_h_total = 0;	/* warning, go away */
		f_v = f_v_sync_start = f_v_sync_end = f_v_total = 0;
		gtf = 1;
		if (__unlikely(*p != '\n')) {
			char *p1 = p;
			gtf = 0;
			if (__unlikely((f_h = getnum(&p, 0)) == -1)) {
				p = p1;
				gtf = 1;
				goto no_tim;
			}
			if (__unlikely((f_h_sync_start = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_h_sync_end = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_h_total = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_v = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_v_sync_start = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_v_sync_end = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely((f_v_total = getnum(&p, 0)) == -1)) goto invl;
			if (__unlikely(f_h != xres) || __unlikely(f_v != yres)) goto invl;
		}
		no_tim:
		f_timing_flags = 0;
		h_set = v_set = 0;
		while (__unlikely((s = getstr(&p)) != NULL)) {
			if (!_strcasecmp(s, "DOUBLESCAN")) {
				if (__unlikely(f_timing_flags & VIDEO_PARAMS_DOUBLE_SCAN)) goto invl_free;
				f_timing_flags |= VIDEO_PARAMS_DOUBLE_SCAN;
			} else if (__unlikely(!_strcasecmp(s, "INTERLACE"))) {
				if (f_timing_flags & VIDEO_PARAMS_INTERLACED) goto invl_free;
				f_timing_flags |= VIDEO_PARAMS_INTERLACED;
			} else if (!_strcasecmp(s, "+HSYNC")) {
				if (__unlikely(h_set)) goto invl_free;
				h_set = 1;
			} else if (!_strcasecmp(s, "-HSYNC")) {
				if (__unlikely(h_set)) goto invl_free;
				h_set = 1;
				f_timing_flags |= VIDEO_PARAMS_HSYNC_NEG;
			} else if (!_strcasecmp(s, "+VSYNC")) {
				if (__unlikely(v_set)) goto invl_free;
				v_set = 1;
			} else if (!_strcasecmp(s, "-VSYNC")) {
				if (__unlikely(v_set)) goto invl_free;
				v_set = 1;
				f_timing_flags |= VIDEO_PARAMS_VSYNC_NEG;
			} else goto invl_free;
			free(s);
		}
		if (x != xres || __unlikely(y != yres)) continue;
		if (__unlikely(!gtf)) {
			pixel_clock = freq;
			hh = xres;
			h_sync_start = f_h_sync_start;
			h_sync_end = f_h_sync_end;
			h_total = f_h_total;
			vv = yres;
			v_sync_start = f_v_sync_start;
			v_sync_end = f_v_sync_end;
			v_total = f_v_total;
			timing_flags = f_timing_flags;
			if (timing_flags & VIDEO_PARAMS_DOUBLE_SCAN) {
				vv <<= 1;
				v_sync_start <<= 1;
				v_sync_end <<= 1;
				v_total <<= 1;
			}
			if (timing_flags & VIDEO_PARAMS_INTERLACED) {
				vv >>= 1;
				v_sync_start >>= 1;
				v_sync_end >>= 1;
				v_total >>= 1;
			}
		} else {
			if (__likely(!h_set)) f_timing_flags |= VIDEO_PARAMS_HSYNC_NEG;
			do_gtf(x, y, bpp, freq, f_timing_flags);
		}
		break;
	}
	free(m);
}

void vert_refresh (int h_pixels, int v_lines, float freq, int interlaced, int margins);

static void do_gtf(int x, int y, int bpp, int freq, int f_timing_flags)
{
	int h;
	char v[__MAX_STR_LEN];
	char *p;
	CHHRQ ch;
	IOCTLRQ io;
	unsigned rq_pixel_clock;
	unsigned last_pixel_clock;
	int dir, cnt;
	float c_freq = (float)freq / 1000000, d_freq = 1;
	if (__unlikely(y <= 300)) f_timing_flags |= VIDEO_PARAMS_DOUBLE_SCAN;
	if (__unlikely(f_timing_flags & VIDEO_PARAMS_DOUBLE_SCAN)) y <<= 1;
	hh = x;
	vv = y;
	if (__unlikely(f_timing_flags & VIDEO_PARAMS_INTERLACED)) vv >>= 1;
	timing_flags = f_timing_flags;
	vert_refresh(x, y, c_freq, f_timing_flags & VIDEO_PARAMS_INTERLACED, 0);
	p = KERNEL$HANDLE_PATH(1);
	if (__unlikely(!p)) p = "STDOUT:/";
	h = open(p, _O_NOACCESS);
	if (__unlikely(h == -1)) return;
	_snprintf(v, sizeof v, "G%dX%dX%d", x, y, bpp);
	ch.h = h;
	ch.option = "VIDEOMODE";
	ch.value = v;
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	if (__unlikely(ch.status < 0)) {
		close(h);
		return;
	}
	io.h = h;
	io.ioctl = IOCTL_TTY_GET_PIXEL_CLOCK;
	io.param = pixel_clock;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	close(h);
	/*__debug_printf("PIXEL CLOCK: %d -> %ld\n", pixel_clock, io.status);*/
	if (__unlikely(io.status < 0)) return;
	rq_pixel_clock = io.status;
	dir = 0;
	cnt = 0;
	again:
	last_pixel_clock = pixel_clock;
	if (pixel_clock < rq_pixel_clock) {
		if (dir == -1) d_freq *= 0.6;
		c_freq += d_freq;
		vert_refresh(x, y, c_freq, f_timing_flags & VIDEO_PARAMS_INTERLACED, 0);
		dir = 1;
	} else if (pixel_clock > rq_pixel_clock) {
		if (dir == 1) d_freq *= 0.6;
		c_freq -= d_freq;
		vert_refresh(x, y, c_freq, f_timing_flags & VIDEO_PARAMS_INTERLACED, 0);
		dir = -1;
	} else {
		return;
	}
	/*__debug_printf("again: %d,%d  %d,%d\n", pixel_clock, rq_pixel_clock, (int)(c_freq * 1000000), (int)(d_freq * 1000000));*/
	if (__likely(++cnt < 100) && (__likely(d_freq > 0.00001) || last_pixel_clock != pixel_clock)) goto again;
	pixel_clock = rq_pixel_clock;
}

#define MARGIN_PERCENT    1.8   /* % of active vertical image                */
#define CELL_GRAN         8.0   /* assumed character cell granularity        */
#define MIN_PORCH         1     /* minimum front porch                       */
#define V_SYNC_RQD        3     /* width of vsync in lines                   */
#define H_SYNC_PERCENT    8.0   /* width of hsync as % of total line         */
#define MIN_VSYNC_PLUS_BP 550.0 /* min time of vsync + back porch (microsec) */
#define M                 600.0 /* blanking formula gradient                 */
#define C                 40.0  /* blanking formula offset                   */
#define K                 128.0 /* blanking formula scaling factor           */
#define J                 20.0  /* blanking formula scaling factor           */

/* C' and M' are part of the Blanking Duty Cycle computation */

#define C_PRIME           (((C - J) * K/256.0) + J)
#define M_PRIME           (K/256.0 * M)

/*
 * vert_refresh() - as defined by the GTF Timing Standard, compute the
 * Stage 1 Parameters using the vertical refresh frequency.  In other
 * words: input a desired resolution and desired refresh rate, and
 * output the GTF mode timings.
 *
 * XXX All the code is in place to compute interlaced modes, but I don't
 * feel like testing it right now.
 *
 * XXX margin computations are implemented but not tested (nor used by
 * XFree86 of fbset mode descriptions, from what I can tell).
 */

#define print_value(a, b, c)

static double rint(double x)
{
	return (unsigned long)(x + 0.5);
}

void vert_refresh (int h_pixels, int v_lines, float freq, int interlaced, int margins)
{
    float h_pixels_rnd;
    float v_lines_rnd;
    float v_field_rate_rqd;
    float top_margin;
    float bottom_margin;
    float interlace;
    float h_period_est;
    float vsync_plus_bp;
    float v_back_porch;
    float total_v_lines;
    float v_field_rate_est;
    float h_period;
    float v_field_rate;
    float v_frame_rate;
    float left_margin;
    float right_margin;
    float total_active_pixels;
    float ideal_duty_cycle;
    float h_blank;
    float total_pixels;
    float pixel_freq;
    float h_freq;

    float h_sync;
    float h_front_porch;
    float v_odd_front_porch_lines;

    /*  1. In order to give correct results, the number of horizontal
     *  pixels requested is first processed to ensure that it is divisible
     *  by the character size, by rounding it to the nearest character
     *  cell boundary:
     *
     *  [H PIXELS RND] = ((ROUND([H PIXELS]/[CELL GRAN RND],0))*[CELLGRAN RND])
     */
    
    h_pixels_rnd = rint((float) h_pixels / CELL_GRAN) * CELL_GRAN;
    
    print_value(1, "[H PIXELS RND]", h_pixels_rnd);

    
    /*  2. If interlace is requested, the number of vertical lines assumed
     *  by the calculation must be halved, as the computation calculates
     *  the number of vertical lines per field. In either case, the
     *  number of lines is rounded to the nearest integer.
     *   
     *  [V LINES RND] = IF([INT RQD?]="y", ROUND([V LINES]/2,0),
     *                                     ROUND([V LINES],0))
     */

    v_lines_rnd = interlaced ?
            rint((float) v_lines) / 2.0 :
            rint((float) v_lines);
    
    print_value(2, "[V LINES RND]", v_lines_rnd);
    
    
    /*  3. Find the frame rate required:
     *
     *  [V FIELD RATE RQD] = IF([INT RQD?]="y", [I/P FREQ RQD]*2,
     *                                          [I/P FREQ RQD])
     */

    v_field_rate_rqd = interlaced ? (freq * 2.0) : (freq);

    print_value(3, "[V FIELD RATE RQD]", v_field_rate_rqd);
    

    /*  4. Find number of lines in Top margin:
     *
     *  [TOP MARGIN (LINES)] = IF([MARGINS RQD?]="Y",
     *          ROUND(([MARGIN%]/100*[V LINES RND]),0),
     *          0)
     */

    top_margin = margins ? rint(MARGIN_PERCENT / 100.0 * v_lines_rnd) : (0.0);

    print_value(4, "[TOP MARGIN (LINES)]", top_margin);
    

    /*  5. Find number of lines in Bottom margin:
     *
     *  [BOT MARGIN (LINES)] = IF([MARGINS RQD?]="Y",
     *          ROUND(([MARGIN%]/100*[V LINES RND]),0),
     *          0)
     */

    bottom_margin = margins ? rint(MARGIN_PERCENT/100.0 * v_lines_rnd) : (0.0);

    print_value(5, "[BOT MARGIN (LINES)]", bottom_margin);

    
    /*  6. If interlace is required, then set variable [INTERLACE]=0.5:
     *   
     *  [INTERLACE]=(IF([INT RQD?]="y",0.5,0))
     */

    interlace = interlaced ? 0.5 : 0.0;

    print_value(6, "[INTERLACE]", interlace);
    

    /*  7. Estimate the Horizontal period
     *
     *  [H PERIOD EST] = ((1/[V FIELD RATE RQD]) - [MIN VSYNC+BP]/1000000) /
     *                    ([V LINES RND] + (2*[TOP MARGIN (LINES)]) +
     *                     [MIN PORCH RND]+[INTERLACE]) * 1000000
     */

    h_period_est = (((1.0/v_field_rate_rqd) - (MIN_VSYNC_PLUS_BP/1000000.0))
                    / (v_lines_rnd + (2*top_margin) + MIN_PORCH + interlace)
                    * 1000000.0);

    print_value(7, "[H PERIOD EST]", h_period_est);
    

    /*  8. Find the number of lines in V sync + back porch:
     *
     *  [V SYNC+BP] = ROUND(([MIN VSYNC+BP]/[H PERIOD EST]),0)
     */

    vsync_plus_bp = rint(MIN_VSYNC_PLUS_BP/h_period_est);

    print_value(8, "[V SYNC+BP]", vsync_plus_bp);
    
    
    /*  9. Find the number of lines in V back porch alone:
     *
     *  [V BACK PORCH] = [V SYNC+BP] - [V SYNC RND]
     *
     *  XXX is "[V SYNC RND]" a typo? should be [V SYNC RQD]?
     */
    
    v_back_porch = vsync_plus_bp - V_SYNC_RQD;
    
    print_value(9, "[V BACK PORCH]", v_back_porch);
    

    /*  10. Find the total number of lines in Vertical field period:
     *
     *  [TOTAL V LINES] = [V LINES RND] + [TOP MARGIN (LINES)] +
     *                    [BOT MARGIN (LINES)] + [V SYNC+BP] + [INTERLACE] +
     *                    [MIN PORCH RND]
     */

    total_v_lines = v_lines_rnd + top_margin + bottom_margin + vsync_plus_bp +
        interlace + MIN_PORCH;
    
    print_value(10, "[TOTAL V LINES]", total_v_lines);
    

    /*  11. Estimate the Vertical field frequency:
     *
     *  [V FIELD RATE EST] = 1 / [H PERIOD EST] / [TOTAL V LINES] * 1000000
     */

    v_field_rate_est = 1.0 / h_period_est / total_v_lines * 1000000.0;
    
    print_value(11, "[V FIELD RATE EST]", v_field_rate_est);
    

    /*  12. Find the actual horizontal period:
     *
     *  [H PERIOD] = [H PERIOD EST] / ([V FIELD RATE RQD] / [V FIELD RATE EST])
     */

    h_period = h_period_est / (v_field_rate_rqd / v_field_rate_est);
    
    print_value(12, "[H PERIOD]", h_period);
    

    /*  13. Find the actual Vertical field frequency:
     *
     *  [V FIELD RATE] = 1 / [H PERIOD] / [TOTAL V LINES] * 1000000
     */

    v_field_rate = 1.0 / h_period / total_v_lines * 1000000.0;

    print_value(13, "[V FIELD RATE]", v_field_rate);
    

    /*  14. Find the Vertical frame frequency:
     *
     *  [V FRAME RATE] = (IF([INT RQD?]="y", [V FIELD RATE]/2, [V FIELD RATE]))
     */

    v_frame_rate = interlaced ? v_field_rate / 2.0 : v_field_rate;

    print_value(14, "[V FRAME RATE]", v_frame_rate);
    

    /*  15. Find number of pixels in left margin:
     *
     *  [LEFT MARGIN (PIXELS)] = (IF( [MARGINS RQD?]="Y",
     *          (ROUND( ([H PIXELS RND] * [MARGIN%] / 100 /
     *                   [CELL GRAN RND]),0)) * [CELL GRAN RND],
     *          0))
     */

    left_margin = margins ?
        rint(h_pixels_rnd * MARGIN_PERCENT / 100.0 / CELL_GRAN) * CELL_GRAN :
        0.0;
    
    print_value(15, "[LEFT MARGIN (PIXELS)]", left_margin);
    

    /*  16. Find number of pixels in right margin:
     *
     *  [RIGHT MARGIN (PIXELS)] = (IF( [MARGINS RQD?]="Y",
     *          (ROUND( ([H PIXELS RND] * [MARGIN%] / 100 /
     *                   [CELL GRAN RND]),0)) * [CELL GRAN RND],
     *          0))
     */
    
    right_margin = margins ?
        rint(h_pixels_rnd * MARGIN_PERCENT / 100.0 / CELL_GRAN) * CELL_GRAN :
        0.0;
    
    print_value(16, "[RIGHT MARGIN (PIXELS)]", right_margin);
    

    /*  17. Find total number of active pixels in image and left and right
     *  margins:
     *
     *  [TOTAL ACTIVE PIXELS] = [H PIXELS RND] + [LEFT MARGIN (PIXELS)] +
     *                          [RIGHT MARGIN (PIXELS)]
     */

    total_active_pixels = h_pixels_rnd + left_margin + right_margin;
    
    print_value(17, "[TOTAL ACTIVE PIXELS]", total_active_pixels);
    
    
    /*  18. Find the ideal blanking duty cycle from the blanking duty cycle
     *  equation:
     *
     *  [IDEAL DUTY CYCLE] = [C'] - ([M']*[H PERIOD]/1000)
     */

    ideal_duty_cycle = C_PRIME - (M_PRIME * h_period / 1000.0);
    
    print_value(18, "[IDEAL DUTY CYCLE]", ideal_duty_cycle);
    

    /*  19. Find the number of pixels in the blanking time to the nearest
     *  double character cell:
     *
     *  [H BLANK (PIXELS)] = (ROUND(([TOTAL ACTIVE PIXELS] *
     *                               [IDEAL DUTY CYCLE] /
     *                               (100-[IDEAL DUTY CYCLE]) /
     *                               (2*[CELL GRAN RND])), 0))
     *                       * (2*[CELL GRAN RND])
     */

    h_blank = rint(total_active_pixels *
                   ideal_duty_cycle /
                   (100.0 - ideal_duty_cycle) /
                   (2.0 * CELL_GRAN)) * (2.0 * CELL_GRAN);
    
    print_value(19, "[H BLANK (PIXELS)]", h_blank);
    

    /*  20. Find total number of pixels:
     *
     *  [TOTAL PIXELS] = [TOTAL ACTIVE PIXELS] + [H BLANK (PIXELS)]
     */

    total_pixels = total_active_pixels + h_blank;
    
    print_value(20, "[TOTAL PIXELS]", total_pixels);
    

    /*  21. Find pixel clock frequency:
     *
     *  [PIXEL FREQ] = [TOTAL PIXELS] / [H PERIOD]
     */
    
    pixel_freq = total_pixels / h_period;
    
    print_value(21, "[PIXEL FREQ]", pixel_freq);
    

    /*  22. Find horizontal frequency:
     *
     *  [H FREQ] = 1000 / [H PERIOD]
     */

    h_freq = 1000.0 / h_period;
    
    print_value(22, "[H FREQ]", h_freq);
    


    /* Stage 1 computations are now complete; I should really pass
       the results to another function and do the Stage 2
       computations, but I only need a few more values so I'll just
       append the computations here for now */

    

    /*  17. Find the number of pixels in the horizontal sync period:
     *
     *  [H SYNC (PIXELS)] =(ROUND(([H SYNC%] / 100 * [TOTAL PIXELS] /
     *                             [CELL GRAN RND]),0))*[CELL GRAN RND]
     */

    h_sync = rint(H_SYNC_PERCENT/100.0 * total_pixels / CELL_GRAN) * CELL_GRAN;

    print_value(17, "[H SYNC (PIXELS)]", h_sync);
    

    /*  18. Find the number of pixels in the horizontal front porch period:
     *
     *  [H FRONT PORCH (PIXELS)] = ([H BLANK (PIXELS)]/2)-[H SYNC (PIXELS)]
     */

    h_front_porch = (h_blank / 2.0) - h_sync;

    print_value(18, "[H FRONT PORCH (PIXELS)]", h_front_porch);
    
    
    /*  36. Find the number of lines in the odd front porch period:
     *
     *  [V ODD FRONT PORCH(LINES)]=([MIN PORCH RND]+[INTERLACE])
     */
    
    v_odd_front_porch_lines = MIN_PORCH + interlace;
    
    print_value(36, "[V ODD FRONT PORCH(LINES)]", v_odd_front_porch_lines);
    

    /* finally, pack the results in the mode struct */
    
    h_sync_start = (int) (h_pixels_rnd + h_front_porch);
    h_sync_end = (int) (h_pixels_rnd + h_front_porch + h_sync);
    h_total = (int) (total_pixels);

    v_sync_start = (int) (v_lines_rnd + v_odd_front_porch_lines);
    v_sync_end = (int) (int) (v_lines_rnd + v_odd_front_porch_lines + V_SYNC_RQD);
    v_total = (int) (total_v_lines);

    pixel_clock   = pixel_freq * 1000000;
    /*__critical_printf("%u %u %u %u %u %u %u\n", h_sync_start, h_sync_end, h_total, v_sync_start, v_sync_end, v_total, pixel_clock);*/

    
}

