#include <STDLIB.H>
#include <UNISTD.H>
#include <STRING.H>
#include <STDARG.H>
#include <TIME.H>
#include <SYS/TYPES.H>
#include <SPAD/LIST.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIBPROC.H>
#include <SPAD/SHELL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYSLOG.H>

struct terminal_info {
	struct process_info *proc;
	char name[1];
};

struct process_info {
	LIST_ENTRY list;
	char name[9];
};

struct session_state {
	struct terminal_info **ttys;
	int n_ttys;
	XLIST_HEAD procs;
	volatile int wake;
	IORQ wait;
	WQ wq;
	volatile int terminate;
};

static void not_auth(int h)
{
	IOCTLRQ io;
	char buf[128];
	while (read(h, buf, sizeof buf) > 0) ;
	io.h = h;
	io.ioctl = IOCTL_TTY_AUTH;
	io.param = 1;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
}

static int log_tty_entry(char *ttyname, int h)
{
	char *ptr;
	int len;
	union {
		IOCTLRQ io;
		LOGRQ logrq;
	} u;
	int i, i1, namelen, p, r, f;
	char msg[__MAX_STR_LEN];
	len = 1024;
	again:
	ptr = __sync_malloc(len);
	u.io.h = h;
	u.io.ioctl = IOCTL_TTY_PEEK;
	u.io.param = 0;
	u.io.v.ptr = (unsigned long)ptr;
	u.io.v.len = len;
	u.io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.io, KERNEL$IOCTL);
	if (__unlikely(u.io.status < 0)) {
		__slow_free(ptr);
		return u.io.status;
	}
	if (__unlikely(u.io.status == len)) {
		__slow_free(ptr);
		len <<= 1;
		goto again;
	}
	for (i = 0; i < u.io.status; i++) {
		ptr[i] = __upcasechr(ptr[i]);
	}
	for (i = 0; i < u.io.status; i++) {
		if (ptr[i] == '\n') {
			namelen = i;
			i++;
			goto ok;
		}
	}
	free(ptr);
	return -EINVAL;
	ok:
	if (__unlikely((p = _snprintf(msg, sizeof(msg), "AUTH REQUEST FOR \"%.*s\"", namelen, ptr)) == -1)) {
		__slow_free(ptr);
		return -ENAMETOOLONG;
	}
	f = 1;
	i1 = i;
	for (; i < u.io.status; i++) {
		if (__likely(ptr[i] != ' ') && __likely(ptr[i] != '\n')) continue;
		if (i - i1 == 4 && __likely(!memcmp(ptr + i1, "DONE", 4))) goto ok_m;
		if (__unlikely((r = _snprintf(msg + p, sizeof(msg) + p, f ? " USING %.*s" : ", %.*s", i - i1, ptr + i1)) == -1)) {
			__slow_free(ptr);
			return -ENAMETOOLONG;
		}
		f = 0;
		p += r;
		while (i < u.io.status && ptr[i] != '\n') i++;
		i1 = i + 1;
	}
	free(ptr);
	return -EINVAL;
	ok_m:
	u.logrq.lclass = __SYSLOG_AUTH_INFO;
	u.logrq.device = ttyname;
	u.logrq.msg = msg;
	SYNC_IO(&u.logrq, KERNEL$SYNC_SYSLOG);
	free(ptr);
	return u.logrq.status;
}

static void log_tty_delete(char *ttyname)
{
	LOGRQ logrq;
	char msg[__MAX_STR_LEN];
	_snprintf(msg, sizeof(msg), "TTY CLOSED");
	logrq.lclass = __SYSLOG_AUTH_INFO;
	logrq.device = ttyname;
	logrq.msg = msg;
	SYNC_IO(&logrq, KERNEL$SYNC_SYSLOG);
}

static void init_session_state(struct session_state *s)
{
	s->ttys = NULL;
	s->n_ttys = 0;
	INIT_XLIST(&s->procs);
	s->wake = 0;
	WQ_INIT(&s->wq, "SESSION$WQ");
	s->terminate = 0;
}

static int doshellcmd(char *cmd, ...)
{
	char *s, *q, *cc;
	size_t l;
	va_list a;
	CTX *ctx;
	union {
		SHCCRQ shccrq;
		SHRCRQ shrcrq;
	} u;
	SYNC_IO(&u.shccrq, SHELL$CREATE_CONTEXT);
	if (__unlikely(u.shccrq.status < 0)) return 1;
	ctx = u.shccrq.ctx;
	l = strlen(cmd) + 1;
	va_start(a, cmd);
	c1:
	cc = va_arg(a, char *);
	if (cc) {
		l += 1 + strlen(cc);
		goto c1;
	}
	va_end(a);
	s = __sync_malloc(l);
	if (__unlikely(!s)) {
		SHELL$DESTROY_CONTEXT(ctx);
		return 1;
	}
	q = stpcpy(s, cmd);
	va_start(a, cmd);
	c2:
	cc = va_arg(a, char *);
	if (cc) {
		*q++ = ' ';
		q = stpcpy(q, cc);
		goto c2;
	}
	va_end(a);
#if __DEBUG >= 1
	if (__unlikely(q - s + 1 > l))
		KERNEL$SUICIDE("doshellcmd: SHOT OVER, %ld > %ld", (long)(q - s + 1), (long)l);
#endif
	u.shrcrq.ctx = ctx;
	u.shrcrq.ptr = s;
	u.shrcrq.len = q - s;
	u.shrcrq.arg = NULL;
	u.shrcrq.narg = 0;
	/*__debug_printf("exec(%s) ... ", s);*/
	SYNC_IO(&u.shrcrq, SHELL$RUN_COMMANDS);
	SHELL$DESTROY_CONTEXT(ctx);
	free(s);
	/*__debug_printf("%ld\n", u.shrcrq.status);*/
	return u.shrcrq.status;
}

static void del_tty(struct terminal_info *tty)
{
	if (tty->proc) {
		doshellcmd("CALL AUTH.:/UNMAP.CMD", tty->proc->name, tty->name, NULL);
	}
	log_tty_delete(tty->name);
	free(tty);
}

static int add_tty(struct session_state *s, struct terminal_info *tty)
{
	char *filename;
	char jobname[9];
	char c;
	int i;
	int h;
	struct process_info *proc;
	if (__unlikely(!(filename = __sync_malloc(strlen(tty->name) + 3 + 9)))) return 1;
	strcpy(stpcpy(filename, tty->name), ":/^NONBLOCK");
	h = open(filename, O_RDWR | _O_NOPOSIX);
	free(filename);
	if (__unlikely(h == -1)) return 1;
	if (__unlikely(log_tty_entry(tty->name, h))) {
		not_auth(h);
		close(h);
		return 1;
	}
	memset(jobname, 0, 8);
	i = 0;
	nc:
	if (__unlikely(read(h, &c, 1) != 1)) {
		not_auth(h);
		close(h);
		log_tty_delete(tty->name);
		return 1;
	}
	if (__likely(c != '/')) {
		if (__likely(c != '\n')) {
			c = __upcasechr(c);
			if (!((__likely(c >= 'A') && __likely(c <= 'Z')) || (__likely(c >= '0') && __likely(c <= '9')) || __likely(c == '_') || __likely(c == '-') || __likely(c == '@') || __likely(c == '$'))) {
				not_auth(h);
				close(h);
				log_tty_delete(tty->name);
				return 1;
			}
			if (__unlikely(i == 8)) {
				not_auth(h);
				close(h);
				log_tty_delete(tty->name);
				return 1;
			}
			jobname[i++] = c;
			goto nc;
		}
		if (!i) goto for_me;
	} else if (__unlikely(!i)) goto nc;
	jobname[i] = 0;
	XLIST_FOR_EACH(proc, &s->procs, struct process_info, list)
		if (!memcmp(proc->name, jobname, 8)) goto have_proc;
	proc = __sync_malloc(sizeof(struct process_info));
	if (__unlikely(!proc)) {
		not_auth(h);
		close(h);
		log_tty_delete(tty->name);
		return 1;
	}
	if (__unlikely(doshellcmd("CALL AUTH.:/SPAWN.CMD", jobname, NULL))) {
		__slow_free(proc);
		not_auth(h);
		close(h);
		log_tty_delete(tty->name);
		return 1;
	}
	memcpy(proc->name, jobname, 8);
	proc->name[8] = 0;
	ADD_TO_XLIST(&s->procs, &proc->list);
	have_proc:
	if (__unlikely(doshellcmd("CALL AUTH.:/REMAP.CMD", jobname, tty->name, NULL))) {
		not_auth(h);
		close(h);
		log_tty_delete(tty->name);
		return 1;
	}
	tty->proc = proc;
	close(h);
	return 0;

	for_me:
	if (__unlikely(doshellcmd("CALL AUTH.:/AUTH.CMD", tty->name, NULL))) {
		not_auth(h);
		close(h);
		log_tty_delete(tty->name);
		return 1;
	}
	close(h);
	return 0;
}

static void del_proc(struct process_info *proc)
{
	PROC_CTRL_RQ p;
	memcpy(p.jobname, proc->name, sizeof p.jobname);
	p.op = PROC_CTRL_ZAP;
	p.msg = (char *)&KERNEL$LIST_END;
	p.session = NULL;
	SYNC_IO(&p, KERNEL$PROC_CTRL);
	DEL_FROM_LIST(&proc->list);
	free(proc);
}

DECL_IOCALL(logical_changed, SPL_SHELL, IORQ)
{
	struct session_state *s = GET_STRUCT(RQ, struct session_state, wait);
	s->wake = 1;
	WQ_WAKE_ALL_PL(&s->wq);
	if (__likely(!s->terminate)) WQ_WAIT(&KERNEL$LOGICAL_WAIT, &s->wait, logical_changed);
	else s->terminate = 2;
	RETURN;
}

int main(int argc, char *argv[])
{
	char *msg;
	int error = 0;
	int i, j, k;
	char first = 1, term;
	struct terminal_info **new_ttys;
	struct process_info *proc;
	struct session_state s;
	LOGICAL_LIST_REQUEST llr;
	time_t create_time;
	time(&create_time);
	init_session_state(&s);
	if (__unlikely(argc > 1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SESSION: SYNTAX ERROR");
		return -EBADSYN;
	}
	s.wait.status = RQS_PROCESSING;
	WQ_WAIT(&KERNEL$LOGICAL_WAIT, &s.wait, logical_changed);
	if (__likely(!KERNEL$KERNEL)) doshellcmd("CALL AUTH.:/STARTUP.CMD", NULL);
	repeat:
	llr.prefix = KERNEL$KERNEL ? "" : "TTY$";
	SYNC_IO_CANCELABLE(&llr, KERNEL$LIST_LOGICALS);
	if (__unlikely(llr.status < 0)) {
		if (llr.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SESSION: CAN'T LIST LOGICAL NAMES: %s", strerror(-llr.status));
		error = llr.status;
		goto terminate;
	}
	j = 0, k = 0;
	term = KERNEL$KERNEL;
	new_ttys = __sync_malloc(sizeof(struct terminal_info *) * s.n_ttys);
	if (__unlikely(!new_ttys)) {
		error = -errno;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SESSION: CAN'T ALLOCATE TERMINALS: %s", strerror(errno));
		term = 1;
		goto fll;
	}
	for (i = 0; i < llr.n_entries; i++) {
		LOGICAL_LIST_ENTRY *e = llr.entries[i];
		if (!__memcasexcmp("TTY$", e->name, e->name + 4)) {
			int r;
			char *ttystr = __sync_malloc(strlen(e->name) + 2);
			if (__unlikely(!ttystr)) continue;
			stpcpy(stpcpy(ttystr, e->name), ":");
			r = open(ttystr, _O_NOACCESS | _O_CLOSE | _O_NOPOSIX);
			free(ttystr);
			if (__unlikely(r)) {
				KERNEL$DELETE_LOGICAL(e->name, 0);
				continue;
			}
			x:
			if (__unlikely(j == s.n_ttys)) r = -1;
			else r = strcmp(e->name, s.ttys[j]->name);
			if (__likely(!r)) {
				new_ttys[k++] = s.ttys[j++];
			} else if (r < 0) {
				struct terminal_info **n_ttys, *tty;
				tty = __sync_malloc(sizeof(struct terminal_info) + 1 + strlen(e->name));
				if (__unlikely(!tty)) continue;
				tty->proc = NULL;
				n_ttys = __sync_realloc(new_ttys, sizeof(struct terminal_info *) * (s.n_ttys + (k - j + 1)));
				if (__unlikely(!n_ttys)) {
					__slow_free(tty);
					continue;
				}
				new_ttys = n_ttys;
				strcpy(tty->name, e->name);
				new_ttys[k++] = tty;
				if (add_tty(&s, tty)) {
					__slow_free(tty);
					k--;
				}
			} else {
				del_tty(s.ttys[j++]);
				goto x;
			}
		} else if (KERNEL$KERNEL && __unlikely(!_strcasecmp(e->name, "SYS$TTYSRVR"))) {
			term = 0;
		}
	}
	while (j < s.n_ttys) del_tty(s.ttys[j++]);
	s.n_ttys = k;
	free(s.ttys);
	s.ttys = new_ttys;
	msg = __sync_malloc(__MAX_STR_LEN);
	if (__unlikely(!msg)) {
		cant_alloc_msg:
		error = -errno;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SESSION: CAN'T ALLOCATE STATUS BUFFER: %s", strerror(errno));
		term = 1;
		goto fll;
	}
	XLIST_FOR_EACH(proc, &s.procs, struct process_info, list) {
		PROC_CTRL_RQ pcr;
		memcpy(pcr.jobname, proc->name, 9);
		pcr.op = PROC_CTRL_NOWAIT;
		pcr.msg = msg;
		pcr.session = NULL;
		SYNC_IO(&pcr, KERNEL$PROC_CTRL);
		if (__unlikely(pcr.status != PROC_RUNNING)) {
			struct process_info *pproc;
/*__debug_printf("fatal exit (%ld): %s\n", pcr.status, pcr.msg);*/
			for (i = 0; i < s.n_ttys; i++) if (__unlikely(s.ttys[i]->proc == proc)) {
				char *esc_msg;
				char exit_code[21];
				unsigned sl;
				if (pcr.status == -ESTOPPED) {
/* This is a race --- a subsession wants to terminate itself, but we have just
   attached a tty to it --- so resurrect it. */
					pcr.op = PROC_CTRL_RESUME;
					pcr.msg = (char *)&KERNEL$LIST_END;
					pcr.session = NULL;
					SYNC_IO(&pcr, KERNEL$PROC_CTRL);
					goto next_proc;
				}
				sl = strlen(msg) + 1;
				if (__unlikely(__alloc_size(msg) < sl * 4)) {
					msg = __sync_reallocf(msg, sl * 4);
					if (__unlikely(!msg)) goto cant_alloc_msg;
				}
				esc_msg = msg + sl;
				SHELL$ESC_COPY(esc_msg, msg);
				_snprintf(exit_code, sizeof(exit_code), "%ld", pcr.status);
				/* DETACH is needed because EXIT.CMD may block */
				doshellcmd("DETACH CALL AUTH.:/EXIT.CMD", proc->name, s.ttys[i]->name, exit_code, esc_msg, NULL);
				s.ttys[i]->proc = NULL;
			}
			pproc = LIST_STRUCT(proc->list.prev, struct process_info, list);
			del_proc(proc);
			proc = pproc;
		}
		next_proc:;
	}
	free(msg);
	fll:
	KERNEL$FREE_LOGICAL_LIST(&llr);
	if (__unlikely(term)) goto terminate;
	RAISE_SPL(SPL_SHELL);
	if (__likely(!s.wake)) {
		if (!KERNEL$KERNEL && __unlikely(!s.n_ttys)) {
			char jn[9];
			jn[0] = 0;
			KERNEL$PROC_GET_JOB_NAME(jn);
			if (__unlikely(!jn[0])) {
/* just a performance hack to prevent bouncing if tty is attached late */
				if (time(NULL) - create_time <= 1) {
					KERNEL$SLEEP(0);
					goto woken_up;
				}
				_stop(NULL);
				goto woken_up;
			}
		}
		WQ_WAIT_SYNC(&s.wq);
		woken_up:;
	}
	s.wake = 0;
	LOWER_SPL(SPL_ZERO);
	first = 0;
	goto repeat;

	terminate:
	RAISE_SPL(SPL_SHELL);
	s.terminate = 1;
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	while (s.terminate != 2) KERNEL$SLEEP(1);
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < s.n_ttys; i++) del_tty(s.ttys[i]);
	free(s.ttys);
	s.n_ttys = 0;
	while (!XLIST_EMPTY(&s.procs)) {
		proc = LIST_STRUCT(s.procs.next, struct process_info, list);
		del_proc(proc);
	}
	if (error) {
		return error;
	}
	if (first) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "TTYSRVR.SYS NOT LOADED");
		return -ENOENT;
	} else {
		return 0;
	}
}
