#ifndef __SSHD_OPS_H
#define __SSHD_OPS_H

#include <STRING.H>
#include <STDLIB.H>
#include <ENDIAN.H>
#include <SPAD/LIBC.H>
#include <OPENSSL/BN.H>
#include "SSHD.H"

#define MOD(x, d)	(__likely(!((d) & ((d) - 1))) ? (x) & ((d) - 1) : (x) % (d))
#define ROUNDUP(x, d)	(__likely(!((d) & ((d) - 1))) ? ((x) + (d) - 1) & ~((d) - 1) : ((x) + (d) - 1) - ((x) + (d) - 1) % (d))

static __finline__ void init_str(char **str, unsigned *len)
{
	*str = NULL;
	*len = 0;
}

static __finline__ int add_bytes_to_str(struct ssh_connection *c, char **str, unsigned *len, __u8 *data, unsigned datalen)
{
	if (__unlikely(!(*str = reallocf(*str, *len + datalen)))) {
		if (c) {
			debug_fatal(c, "Can't realloc string");
			abort_ssh_connection(c);
		}
		return -1;
	}
	memcpy(*str + *len, data, datalen);
	*len += datalen;
	return 0;
}

static __finline__ int add_to_str(struct ssh_connection *c, char **str, unsigned *len, char *ns)
{
	return add_bytes_to_str(c, str, len, ns, strlen(ns));
}

static __finline__ int add_chr_to_str(struct ssh_connection *c, char **str, unsigned *len, char chr)
{
	return add_bytes_to_str(c, str, len, &chr, 1);
}

static __finline__ int add_int32_to_str(struct ssh_connection *c, char **str, unsigned *len, __u32 ii)
{
	ii = __32CPU2BE(ii);
	return add_bytes_to_str(c, str, len, (__u8 *)&ii, sizeof ii);
}


static __finline__ void ssh_flush_stream(struct ssh_connection *c)
{
	c->in_stream_start = c->in_stream_end;
}

static __finline__ int ssh_get_byte(struct ssh_connection *c, unsigned *ptr, __u8 *byte)
{
	if (__unlikely(c->in_stream_start + *ptr >= c->in_stream_end)) {
		ssh_get_next_packet(c);
		return 1;
	}
	*byte = c->in_stream[c->in_stream_start + (*ptr)++];
	return 0;
}

static __finline__ int ssh_get_int32(struct ssh_connection *c, unsigned *ptr, __u32 *int32)
{
	if (__unlikely(c->in_stream_start + *ptr + 4 > c->in_stream_end)) {
		ssh_get_next_packet(c);
		return 1;
	}
	memcpy(int32, c->in_stream + c->in_stream_start + *ptr, 4);
	*ptr += 4;
	*int32 = __32BE2CPU(*int32);
	return 0;
}

static __finline__ int ssh_get_int64(struct ssh_connection *c, unsigned *ptr, __u64 *int64)
{
	if (__unlikely(c->in_stream_start + *ptr + 8 > c->in_stream_end)) {
		ssh_get_next_packet(c);
		return 1;
	}
	memcpy(int64, c->in_stream + c->in_stream_start + *ptr, 8);
	*ptr += 8;
	*int64 = __64BE2CPU(*int64);
	return 0;
}

static __finline__ int ssh_get_string(struct ssh_connection *c, unsigned *ptr, char **start, __u32 *len)
{
	if (__unlikely(ssh_get_int32(c, ptr, len))) return 1;
	if (__unlikely(c->in_stream_start + *ptr + *len > c->in_stream_end)) {
		ssh_get_next_packet(c);
		return 1;
	}
	*start = c->in_stream + c->in_stream_start + *ptr;
	*ptr += *len;
	return 0;
}

static __finline__ int ssh_get_bignum(struct ssh_connection *c, unsigned *ptr, BIGNUM *bn)
{
	char *start;
	__u32 len;
	if (__unlikely(ssh_get_string(c, ptr, &start, &len))) return 1;
	if (__unlikely(len > 8192)) {
		debug_fatal(c, "Bignum length %d bytes", (int)len);
		abort_ssh_connection(c);
		return 1;
	}
	if (__likely(len) && __unlikely((__u8)start[0] >= 0x80)) {
		debug_fatal(c, "Negative bignum");
		abort_ssh_connection(c);
		return 1;
	}
	if (__unlikely(!BN_bin2bn(start, len, bn))) {
		debug_fatal(c, "Can't make bignum");
		abort_ssh_connection(c);
		return 1;
	}
	return 0;
}

static __finline__ void ssh_got_data(struct ssh_connection *c, unsigned *ptr)
{
	c->in_stream_start += *ptr;
	if (__unlikely(c->in_stream_start > c->in_stream_end)) {
		KERNEL$SUICIDE("ssh_got_data: out of string: %d > %d", c->in_stream_start, c->in_stream_end);
	}
	*ptr = 0;
}


static __finline__ int ssh_add_bytes(struct ssh_connection *c, __u8 *data, unsigned len)
{
	if (__unlikely(!(c->out_stream = reallocf(c->out_stream, c->out_stream_len + len)))) {
		debug_fatal(c, "Can't realloc out_stream");
		abort_ssh_connection(c);
		return -1;
	}
	memcpy(c->out_stream + c->out_stream_len, data, len);
	c->out_stream_len += len;
	return 0;
}

static __finline__ int ssh_add_byte(struct ssh_connection *c, __u8 byte)
{
	return ssh_add_bytes(c, &byte, 1);
}

static __finline__ int ssh_add_int32(struct ssh_connection *c, __u32 int32)
{
	int32 = __32CPU2BE(int32);
	return ssh_add_bytes(c, (__u8 *)&int32, 4);
}

static __finline__ int ssh_add_int64(struct ssh_connection *c, __u64 int64)
{
	int64 = __64CPU2BE(int64);
	return ssh_add_bytes(c, (__u8 *)&int64, 8);
}

static __finline__ int ssh_add_string(struct ssh_connection *c, __u8 *str, unsigned len)
{
	if (__unlikely(ssh_add_int32(c, len))) return -1;
	return ssh_add_bytes(c, str, len);
}

static __finline__ int ssh_add_bignum(struct ssh_connection *c, BIGNUM *bn)
{
	int r;
	__u8 *buf;
	__u32 bytes;
	if (__unlikely(bn_2_string(bn, &buf, &bytes))) {
		debug_fatal(c, "Can't convert bignum to string");
		abort_ssh_connection(c);
		return -1;
	}
	r = ssh_add_string(c, buf, bytes);
	free(buf);
	return r;
}

#endif
