#include <STDIO.H>
#include <ERRNO.H>
#include <STRING.H>
#include <SPAD/SYNC.H>
#include <OPENSSL/SHA.H>
#include <OPENSSL/PEM.H>
#include <OPENSSL/DSA.H>
#include <OPENSSL/BN.H>
#include <OPENSSL/ERR.H>

#include "SSHD.H"
#include "OPS.H"

static int dss_blob(__u8 **blob, unsigned *len);
static int dss_sign(__u8 **signature, unsigned *siglen, __u8 *data, unsigned datalen);

static DSA *dsa_key = NULL;

static struct public_type ssh_dss = {
	"ssh-dss", dss_blob, dss_sign,
};

struct public_type *public_types[] = {
	&ssh_dss,
	NULL
};

static int add_bn_to_str(char **s, unsigned *l, BIGNUM *bn)
{
	__u8 *nn;
	__u32 n;
	if (__unlikely(bn_2_string(bn, &nn, &n))) {__slow_free(s); return -1;}
	if (__unlikely(add_int32_to_str(NULL, s, l, n))) {__slow_free(nn); return -1;}
	if (__unlikely(add_bytes_to_str(NULL, s, l, nn, n))) {__slow_free(nn); return -1;}
	free(nn);
	return 0;
}

static int dss_blob(__u8 **blob, unsigned *len)
{
	char *s;
	unsigned l;
	if (__unlikely(!dsa_key)) return -1;
	init_str(&s, &l);
	if (__unlikely(add_int32_to_str(NULL, &s, &l, strlen(ssh_dss.name)))) return -1;
	if (__unlikely(add_to_str(NULL, &s, &l, ssh_dss.name))) return -1;
	if (__unlikely(add_bn_to_str(&s, &l, dsa_key->p))) return -1;
	if (__unlikely(add_bn_to_str(&s, &l, dsa_key->q))) return -1;
	if (__unlikely(add_bn_to_str(&s, &l, dsa_key->g))) return -1;
	if (__unlikely(add_bn_to_str(&s, &l, dsa_key->pub_key))) return -1;
	*blob = s;
	*len = l;
	return 0;
}

#define INTBLOB_LEN	20
#define SIGBLOB_LEN	(INTBLOB_LEN * 2)

static int dss_sign(__u8 **signature_, unsigned *siglen, __u8 *data, unsigned datalen)
{
	char **signature = (char **)signature_;
	__u8 digest[SHA_DIGEST_LENGTH], sigblob[SIGBLOB_LEN];
	DSA_SIG *sig;
	unsigned rlen, slen;
	if (__unlikely(!dsa_key)) return -1;
	SHA1(data, datalen, digest);
	sig = DSA_do_sign(digest, sizeof digest, dsa_key);
	if (__unlikely(!sig)) return -1;
	rlen = BN_num_bytes(sig->r);
	slen = BN_num_bytes(sig->s);
	if (__unlikely(rlen > INTBLOB_LEN) || __unlikely(slen > INTBLOB_LEN)) {
		fs:
		DSA_SIG_free(sig);
		return -1;
	}
	memset(sigblob, 0, sizeof sigblob);
	if (__unlikely(!BN_bn2bin(sig->r, sigblob + SIGBLOB_LEN - INTBLOB_LEN - rlen))) goto fs;
	if (__unlikely(!BN_bn2bin(sig->s, sigblob + SIGBLOB_LEN - slen))) goto fs;
	DSA_SIG_free(sig);

		/* !!! SSH_BUG_SIGBLOB */
	
	init_str(signature, siglen);
	if (__unlikely(add_int32_to_str(NULL, signature, siglen, strlen(ssh_dss.name)))) return -1;
	if (__unlikely(add_bytes_to_str(NULL, signature, siglen, ssh_dss.name, strlen(ssh_dss.name)))) return -1;
	if (__unlikely(add_int32_to_str(NULL, signature, siglen, SIGBLOB_LEN))) return -1;
	if (__unlikely(add_bytes_to_str(NULL, signature, siglen, sigblob, SIGBLOB_LEN))) return -1;
	return 0;
}

static void *key_load_private(char *filename, int type, char *err)
{
	FILE *fp = fopen(filename, "r");
	EVP_PKEY *pk;
	void *data;
	if (__unlikely(!fp)) {
		_snprintf(err, __MAX_STR_LEN, "CAN'T OPEN %s: %s", filename, strerror(errno));
		return NULL;
	}
	pk = PEM_read_PrivateKey(fp, NULL, NULL, "");
	if (__unlikely(!pk)) {
		_snprintf(err, __MAX_STR_LEN, "INVALID KEY FORMAT: %lX", ERR_get_error());
		fclose(fp);
		errno = EINVAL;
		return NULL;
	}
	if (__unlikely(pk->type != type)) {
		_snprintf(err, __MAX_STR_LEN, "INVALID KEY TYPE");
		EVP_PKEY_free(pk);
		fclose(fp);
		errno = EINVAL;
		return NULL;
	}
	if (__likely(pk->type == EVP_PKEY_DSA)) {
		data = EVP_PKEY_get1_DSA(pk);
	} else {
		KERNEL$SUICIDE("INVALID TYPE %d REQUESTED", pk->type);
	}
	if (__unlikely(!data)) {
		_snprintf(err, __MAX_STR_LEN, "COULD NOT GET KEY DATA");
		EVP_PKEY_free(pk);
		fclose(fp);
		errno = EINVAL;
		return NULL;
	}
	EVP_PKEY_free(pk);
	return data;
}

int sshd_load_keys(void)
{
	char error[__MAX_STR_LEN];
	dsa_key = key_load_private("SSHD:/SSH_HOST_DSA_KEY", EVP_PKEY_DSA, error);
	if (__unlikely(!dsa_key)) {
		strcpy(KERNEL$ERROR_MSG(), error);
		return -errno;
	}
	return 0;
}

